/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;


import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.uml2sd.ui.drawings.IColor;
import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.IImage;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;

/**
 * Lifeline is the UML2 lifeline graphical representation.<br>
 * Each lifeline owns a set of event occurrences. An event occurrence is the base element in UML2 
 * to set an event in a sequence diagram.<br>
 * Event occurence define the drawing order of graph node along a lifeline. 
 * In this lifeline implementation, event occurrences are just integer index. The event occurrences
 * with the same value on different lifelines will correspond the same y corrdinate value.
 * 
 * @author sveyrier
 *
 */
public class Lifeline extends GraphNode{
	
	/**
	 * The execution occurrences list owned by the lifeline
	 */
	protected List executionOccurrences	= null;
	
	/**
	 * The lifeline position in the containing frame 
	 */
	protected int indexInFrame = 0;
	
	/**
	 * The frame where the lifeline is drawn
	 */
	private Frame frame	= null;
	
	/**
	 * The lifeline end, it is a stop graph node 
	 * associated to the lifeline
	 */
	protected Stop stop	= null;
	
	/**
	 * The current event occurrence created in the lifeline
	 */
	protected int eventOccurrence = 0;
	
	/**
	 * Point the first visible execution occurrence
	 * in the execution occurrences list
	 */
	private int	execOccurrenceDrawIndex	= 0;
	
	protected int category = -1;
	
	protected boolean hasTime = false;
	
	public int getX()
	{
		return Metrics.FRAME_H_MARGIN + Metrics.LIFELINE_H_MAGIN + (indexInFrame-1)*Metrics.swimmingLaneWidth();
	}

	public int getY()
	{
		return 2*Metrics.FRAME_NAME_H_MARGIN + Metrics.LIFELINE_VT_MAGIN/2+ Metrics.getFrameFontHeigth() +
				Metrics.getLifelineHeaderFontHeigth()+Metrics.FRAME_V_MARGIN +2*Metrics.LIFELINE_HEARDER_TEXT_V_MARGIN;
	}

	public int getWidth()
	{
		return Metrics.getLifelineWidth();
	}

	public int getHeight()
	{
		//Set room for two text lines
		return Metrics.getLifelineFontHeigth() /** 2*/ + 2  * Metrics.LIFELINE_NAME_H_MARGIN;
	}	
	
	public Lifeline()
	{
		executionOccurrences = new ArrayList();
	}
	
	/**
	 * Set the lifeline categorie for this lifeline.
	 * @param arrayIndex the index of the categorie to use
	 * @see Frame#setLifelineCategories(LifelineCategories[])
	 */
	public void setCategory(int arrayIndex)
	{
		category = arrayIndex;
	}
	
	/**
	 * Returns the tooltip text for the lifeline
	 * It is the combinaison between the categorie name(if any) and the lifeline name
	 * @return the tooltip text
	 */
	public String getToolTipText()
	{
		if (category >=0)
		{
			LifelineCategories[] categories = frame.getLifelineCategories();
			if (category< categories.length)
			{
				return categories[category].getName()+" "+getName(); //$NON-NLS-1$
			}
			else return ""; //$NON-NLS-1$
		}
		else return ""; //$NON-NLS-1$
	}
	
	/**
	 * Returns the index of the first visible Execution Occurrence in the execution occurrence array.<br>
	 * Execution Occurrences are Y ordered in this array
	 * @return the first visible Execution Occurrence
	 */
	public int getExecOccurrenceDrawIndex()
	{
		return execOccurrenceDrawIndex;
	}
	
	/**
	 * Assigns the Stop given in parameter to this lifeline.<br>
	 * This will result in displaying a cross to end the lifeline
	 * at the event occurence define in the Stop
	 * @param theStop the Stop to assign
	 */
	public void setStop(Stop theStop)
	{
		if (theStop == null)
			return;
		stop = theStop;
		stop.setLifeline(this);
	}
	
	/**
	 * Returns the Stop assign to this lifeline if any.<br>
	 * @return the Stop assigned or if null
	 */
	public Stop getStop()
	{
		return stop;
	}
	
	/**
	 * Set the frame on which this lifeline must be drawn
	 * @param parentFrame
	 */
	protected void setFrame(Frame parentFrame)
	{
		frame = parentFrame;
		if (hasTime) {
			frame.setHasTimeInfo(true);
		}
		if (frame.getMaxEventOccurrence()<getEventOccurrence()+1)
			frame.setMaxEventOccurrence(getEventOccurrence()+1);
	}
	
	/**
	 * Returns the frame which this lifeline is drawn
	 * @return the Frame
	 */
	protected Frame getFrame()
	{
		return frame;
	}
	
	/**
	 * Set the lifeline position index in the containing frame
	 * @param index the lifeline X position
	 */
	
	protected void setIndex(int index)
	{
		indexInFrame=index;
	}
	
	/**
	 * Returns the lifeline position in de the containing frame
	 * @return the X position
	 */
	public int getIndex()
	{
		return indexInFrame;
	}
	
	/**
	 * Set the lifeline event occurrence to the value given in parameter
	 * This only change the current event occurrence, greater event 
	 * created on this lifeline are still valid and usable.
	 * This also need to inform the frame of the operation mostly to store in the frame the greater
	 * event found in the diagram (used to determine the frame height)
	 * @param index the new current event occurrence
	 */
	public void setCurrentEventOccurrence(int eventOcc)
	{
		if ((frame != null)&&(frame.getMaxEventOccurrence()<eventOcc))
			frame.setMaxEventOccurrence(eventOcc);
		eventOccurrence = eventOcc;
	}
	
	/**
	 * Returns the last created event occurrence along the lifeline.
	 * @return the current event occurence
	 */
	public int getEventOccurrence()
	{
		return eventOccurrence;
	}
	
	/**
	 * Creates a new event occurrence along the lifeline.
	 * @return the new created event occurence
	 */
	public int getNewEventOccurrence()
	{
		setCurrentEventOccurrence(eventOccurrence+1);
		return eventOccurrence;
	}
	
	/**
	 * Adds the execution occurence given in paramater to the lifeline.<br>
	 * A Execution Occurence is never drawn in the frame intead it is added to a lifeline
	 * @param exec the execution occurence to add
	 */
	public void addExecution (BasicExecutionOccurrence exec)
	{
		exec.setLifeline(this);
		executionOccurrences.add(exec);
		if ((frame != null)&&(frame.getMaxEventOccurrence()<exec.endOccurrence))
			frame.setMaxEventOccurrence(exec.endOccurrence);
	}
	
	protected void setTimeInfo(boolean value)
	{
		hasTime = value;
		if ((frame != null)&&(value==true))
			frame.setHasTimeInfo(value);
	}
	
	/**
	 * @return true if at least one execution occurrence has time info
	 */
	public boolean hasTimeInfo() {
		return hasTime;
	}
	
	/**
	 * Returns the list of execution occurence on this lifeline
	 * @return the execution occurrence list
	 */
	public List getExecutions()
	{
		return executionOccurrences;
	}
	
	/**
	 * Computes the index of the first visible graph node for
	 * each ordered graph node lists (for instance only executions occurrences list)
	 * depending on the visible area given in parameter
	 *
	 * @param x visible area top left corner x coordinate
	 * @param y visible area top left corner y coordinate
	 * @param width visible area width
	 * @param height visible area height
	 */
	protected void updateIndex(int x, int y)
	{
		int directionY = 1;
		if (y==0)
			execOccurrenceDrawIndex = 0;
		if (execOccurrenceDrawIndex<executionOccurrences.size()&&execOccurrenceDrawIndex>=0)
			if (((BasicExecutionOccurrence)executionOccurrences.get(execOccurrenceDrawIndex)).getY()>y)
				directionY = -1;
		for (int i=execOccurrenceDrawIndex;i<executionOccurrences.size()&&i>=0;i=i+directionY)
		{
			execOccurrenceDrawIndex = i;
			if (directionY ==1)
			{
				int execY = ((BasicExecutionOccurrence)executionOccurrences.get(i)).getY();
				if (execY > y || execY + ((BasicExecutionOccurrence)executionOccurrences.
							get(i)).getHeight()>y)
					break;
			}
			else 
			{
				int execY = ((BasicExecutionOccurrence)executionOccurrences.get(i)).getY();
				if (execY<y || execY+((BasicExecutionOccurrence)executionOccurrences.get(i)).
							getHeight()<y)
					break;
			}
		}
	}
	
	public boolean contains(int _x, int _y)
	{
		int x = getX();
		int y = getY();
		int width = getWidth();
		int height = getHeight();

		if (frame == null)
			return false;
		if (Frame.contains(x, y, width, height,_x, _y))
		{
			return true;
		}
		if (Frame.contains (x+Metrics.getLifelineWidth()/2 - Metrics.EXECUTION_OCCURRENCE_WIDTH/2,
					y+height, Metrics.EXECUTION_OCCURRENCE_WIDTH,
					(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * frame.getMaxEventOccurrence() + 
					Metrics.LIFELINE_VB_MAGIN-4,_x, _y))
		{
			return true;
		}
		
		height =Metrics.getLifelineFontHeigth()+2*Metrics.LIFELINE_HEARDER_TEXT_V_MARGIN;
		int hMargin=(Metrics.LIFELINE_VT_MAGIN-height)/2;

		if (hMargin>=2)
		{
			if (frame.getVisibleAreaY()<y-height-hMargin)
			{
				if (Frame.contains (x-Metrics.LIFELINE_SPACING/2+1,y-height-hMargin, 
						Metrics.swimmingLaneWidth()-2, height+1,_x,_y))
					return true;
			}
			else
			{
				if (Frame.contains (x-Metrics.LIFELINE_SPACING/2+1, frame.getVisibleAreaY(), 
							Metrics.swimmingLaneWidth()-2, height,_x,_y))
					return true;
			}
		}
		return false;
	}
	
	/**
	 * Returns the lifeline visibility for the given visible area
	 * @param vx
	 * @param vy
	 * @param vwidth
	 * @param vheight
	 * @return true if visible false otherwise
	 */
	public boolean isVisible(int vx, int vy, int vwidth, int vheight)
	{
		int x = getX();
		int width = getWidth();
		if (((x>=vx)&&(x<=vx+vwidth))||((x+width>=vx)&&(x<=vx)))
			return true;
		return false;
	}
	
	protected void drawName(IGC context)
	{
		int x = getX();
		int y = getY();
		int width = getWidth();
		int height =Metrics.getLifelineHeaderFontHeigth()+2*Metrics.LIFELINE_HEARDER_TEXT_V_MARGIN;
		int hMargin=Metrics.LIFELINE_VT_MAGIN/4;//(Metrics.LIFELINE_NAME_H_MARGIN)/2;
		
		context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_LIFELINE_HEADER));
		context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_LIFELINE_HEADER));
		context.setFont(Frame.getUserPref().getFont(ISDPreferences.PREF_LIFELINE_HEADER));
		if (hMargin>=0)
		{
			if (frame.getVisibleAreaY()<y-height-hMargin)
			{
				context.fillRectangle(x-Metrics.LIFELINE_SPACING/2+1,y-height-hMargin, 
						Metrics.swimmingLaneWidth()-2, height);
				context.drawRectangle(x-Metrics.LIFELINE_SPACING/2+1, y-height-hMargin, 
							Metrics.swimmingLaneWidth()-2, height);
				context.setForeground(Frame.getUserPref().getFontColor(ISDPreferences.PREF_LIFELINE_HEADER));
				context.drawTextTruncatedCentred(getName(), x+ Metrics.LIFELINE_NAME_V_MARGIN-Metrics.LIFELINE_SPACING/2+1,
						y-height-hMargin,Metrics.swimmingLaneWidth()-2*Metrics.LIFELINE_NAME_V_MARGIN-2, 
						height,true);
			}
			else
			{
				context.fillRectangle(x-Metrics.LIFELINE_SPACING/2+1, frame.getVisibleAreaY(), Metrics.swimmingLaneWidth()-2, height);
				context.drawRectangle(x-Metrics.LIFELINE_SPACING/2+1, frame.getVisibleAreaY(), Metrics.swimmingLaneWidth()-2, height);
				context.setForeground(Frame.getUserPref().getFontColor(ISDPreferences.PREF_LIFELINE_HEADER));
				context.drawTextTruncatedCentred(getName(), x-Metrics.LIFELINE_SPACING/2+ Metrics.LIFELINE_NAME_V_MARGIN+1,
				frame.getVisibleAreaY() ,Metrics.swimmingLaneWidth()-2*Metrics.LIFELINE_NAME_V_MARGIN-2, 
						height,true);
			}
		}
	}
	
	
	/**
	 * Force the lifeline to be drawn at the given coordinate
	 * @param context -  the context to draw into
	 * @param x1 - the x coordinate
	 * @param y1 - the y coordinate
	 */
	public void draw (IGC context, int x, int y)
	{
		//Set the draw color depending if the lifeline must be selected or not
		if (isSelected())
		{
			if (Frame.getUserPref().useGradienColor())
			{
				context.setGradientColor(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_LIFELINE));
			}
			context.setBackground(Frame.getUserPref().getBackGroundColorSelection());
			context.setForeground(Frame.getUserPref().getForeGroundColorSelection());
		}
		else 
		{
			if (Frame.getUserPref().useGradienColor())
			{
				context.setGradientColor(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_LIFELINE));
				context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_FRAME));
			}
			else context.setBackground(Frame.getUserPref().getBackGroundColor(ISDPreferences.PREF_LIFELINE));
			context.setForeground(Frame.getUserPref().getForeGroundColor(ISDPreferences.PREF_LIFELINE));
		}
		//Store the lifeline coordinates to save some calls
		int width = getWidth();
		int height = getHeight();
		
		//Draw the rectangle which contain the lifeline name
		if (Frame.getUserPref().useGradienColor())
		{
			context.fillGradientRectangle(x, y, width, height/2-7,true);
			context.fillRectangle(x, y+height/2-8, width, +height/2-5);
			context.fillGradientRectangle(x, y+height, width, -height/2+6,true);
		}
		else context.fillRectangle(x, y, width, height);
		context.drawRectangle(x, y, width, height);
		
		if (category >=0)
		{
			LifelineCategories[] categories = frame.getLifelineCategories();
			if (category< categories.length)
			{
				IImage image = categories[category].getImage();
				if (image != null)
					context.drawImage(image, x, y,width, height);
			}
		}

		//Draw the lifeline label into the rectangle
		//The label is truncated if it cannot fit
		IColor temp=context.getForeground();
		context.setForeground(Frame.getUserPref().getFontColor(ISDPreferences.PREF_LIFELINE));
		context.drawTextTruncatedCentred(getName(), x+ Metrics.LIFELINE_NAME_V_MARGIN,
									y ,Metrics.getLifelineWidth()-2*Metrics.LIFELINE_NAME_V_MARGIN, 
									height,true);

		context.setLineStyle(context.getLineDashStyle());
		context.setForeground(temp);
 		int oldStyle = context.getLineStyle();
 		
		//Now draw the lifeline vertical line
		//this line height depends on a stop assignment
		//if there is no stop the line is drawn to the bottom of the frame
		
		//by default set the height to reach the frame bottom
 		int dashedLineEnd = y+height+(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * frame.getMaxEventOccurrence() + 
							Metrics.LIFELINE_VB_MAGIN;
		if (stop != null)
		{
			dashedLineEnd = stop.getY();
		}
		
 		if (isSelected())
 		{
			context.setForeground(Frame.getUserPref().getBackGroundColorSelection());
			context.setLineWidth(5);
			context.drawLine(x+Metrics.getLifelineWidth()/2,y+height, x+Metrics.getLifelineWidth()/2, 
								dashedLineEnd-4);
			context.setForeground(Frame.getUserPref().getForeGroundColorSelection());
 		}
							
		context.setLineWidth(Metrics.NORMAL_LINE_WIDTH);
		context.drawLine(x+Metrics.getLifelineWidth()/2,y+height, x+Metrics.getLifelineWidth()/2,
							dashedLineEnd-4);
		context.drawLine(x+Metrics.getLifelineWidth()/2,y+height, x+Metrics.getLifelineWidth()/2,
								dashedLineEnd-4);
		context.setLineStyle(oldStyle);

		context.setLineStyle(context.getLineSolidStyle());
		
		//Draw the execution ocurrence owned by this lifeline
		int exeCount=0;
		//execOccurenceDrawIndex = first visible execution occurrence
		if (x==getX()&&y==getY())
		for (int i=execOccurrenceDrawIndex; i<executionOccurrences.size();i++)
		{
			BasicExecutionOccurrence toDraw = (BasicExecutionOccurrence)executionOccurrences.get(i);
			//if we are ouside the visble area we stop right now
			//This works because execution occurrences are ordered along the Y axis
			if (toDraw.getY()>context.getContentsY() + context.getVisibleHeight())
				break;
			toDraw.draw(context);
			exeCount++;
		}
		
		//It is the stop turn
		if (stop == null)
			return; 
		if (stop.getY()<=context.getContentsY() + context.getVisibleHeight())
			stop.draw(context);
	}
	
	/**
	 * Draws the select execution occurence region using the given color
	 * @param context the graphical context
	 * @param startEvent the region start
	 * @param nbEvent the region hight
	 * @param color the color to use
	 */	
	public void highlightExecOccurrenceRegion(IGC context,int startEvent, int nbEvent, IColor color)
	{
		IColor backupColor= context.getBackground();
		context.setBackground(color);	
		int x=getX() + Metrics.getLifelineWidth()/2-Metrics.EXECUTION_OCCURRENCE_WIDTH/2;
		int y=getY() + getHeight() + 
			(Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing()) * startEvent;
		int width = Metrics.EXECUTION_OCCURRENCE_WIDTH;
		int height=((Metrics.getMessageFontHeigth()+Metrics.getMessagesSpacing())) * nbEvent;
		context.fillRectangle(x, y, width, height);
		context.setBackground(backupColor);
	}
	
	public void draw (IGC context)
	{
		draw(context,getX(),getY());
	}
}
