package org.eclipse.hyades.logging.parsers;

/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Locale;

import org.apache.commons.logging.Log;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl;

/**
 * Parser is the abstract superclass for AbstractAccessLogParser and
 * AbstractErrorLogParser.  Subclasses should implement setConfiguration(Hashtable table), preParse()
 * and postParse(), first calling super methods if the functions implemented here are needed.  
 * Subclasses must implement CommonBaseEvent[] parse() where the parsing is performed,
 * generating CommonBaseEvent objects representing the records in the log.  Abstract
 * methods, getName() and getVersion(), must also be implemented by concrete subclasses
 * to identify the type of log associated with the parser and the version of
 * the parser, respectively.  
 */
public abstract class Parser implements IParser {

    /**
     * Constant for String "Unknown Host".
     */
    protected final static String UNKNOWN_HOST = "Unknown Host";
    
    
   /**
     * Final static field to hold the default CommonBaseEvent array size.
     * The intent is to parse blocks of records in a log file, generating a 
     * CommonBaseEvent for each record with the parseNext() method which 
     * returns an array of CommonBaseEvent objects.  The user calls parseNext()
     * continuously until the end of file is reached.
     */
    public final static int defaultArraySize = 50;
    
    /**
     * Field to hold the CommonBaseEvent array size.  Set it originally to the
     * defaule size.
     */
    protected int MessageArraySize = defaultArraySize;

    /**
     * Array of CommonBaseEvent objects to be returned from the parseNext() method
     * or written to an Apache Commons Log.  
     */
    protected CommonBaseEvent[] messages;

    /**
     * Field to hold index into CommonBaseEvent array. 
     */
    protected int arrayIndex;

	/**
	 * Field to hold the number of log records parsed.
	 * 
	 */
	protected int recordCount = 0;

    // Variable to hold values parsed from log records.

    /* The Locale that the log file was created in as specified by the user */
    protected Locale fileLocale = null;

    /* The character set of the log file as specified by the user */
    protected String charset = null;
   
    /**
     * Parsed value of the locale where the log was produced.  This field
     * may be set to the locale of the parsing machine if the actual locale
     * information is not available in the log.
     */
    protected String originLocale;

    // Variables to hold information about the parsing machine.

    /**
     * IP address of the machine parsing the log file. 
     */
    protected String localHostId;

    /**
     * Format of the IP address of the machine parsing the log file.  This
     * field will be set either to the String "IPV4" or the String "IPV6".
     */
    protected String localHostIdFormat;

    /**
     * Fully-qualified host name of the machine parsing the log file. 
     */
    protected String localHostName;

    // Variable associated with the log file to be parsed.

    /**
     * Path and name of log file to be parsed. 
     */
    protected String file_path;

    /**
     * Reader for log file to be parsed. 
     * 
     * NOTE:  A <code>RandomAccessFile</code> is used to circumvent encoding issues on some UNIX platforms.
     *        Some log files (e.g. Apache error log) may contain control characters which cannot be read
     *        using a <code>FileReader</code> and/or <code>BufferedReader</code>. 
     */
    protected RandomAccessFile logFile = null;

    /**
     * String to hold lines read from log file to be parsed. 
     */
    protected String curLine = "";

    /**
     * Factory for creating events.
     */
    protected static EventFactory eventFactory = new SimpleEventFactoryHomeImpl().getEventFactory("org.eclipse.hyades.logging.parsers.Parser");

    /**
     * Log to which CommonBaseEvent objects produced during parsing
     * will be written.  The value for this field is passed to the parser 
     * when the parse(ILogger argLogger) method is called.
     */
    protected Log logger;

    /**
     * Current line within the file.
     */
    protected long curLineNumber = 0;

    /**
     * This function is called to perform the parsing of the log file.  
     * Minimal function is provided by this abstract superclass.  Subclasses 
     * will override this method to provide parsing function.
     *
     * @param argLogger  logger to which CommonBaseEvent objects are written  
     * @exception LogParserException thrown if exception is detected while executing function of this superclass
     */
    public void parse(Log argLogger) throws LogParserException {
        try {
            logger = argLogger;
            
			preParse();
			CommonBaseEvent[] msgArray = parseNext();
			
			while (msgArray != null) {
				for (int i=0; i < msgArray.length; i++) {
					if (msgArray[i] != null) {
						logger.trace(msgArray[i]);
					}
				}
				msgArray = parseNext();
			}
			postParse();
        }
        catch (Throwable e) {
        	postParse();
            LogParserException lpe = new LogParserException(e);
            lpe.fillInStackTrace();
            throw lpe;
        }
    }

	/**
	 * Abstract method used to retrieve the messages parsed by the parser.  Each concrete
	 * subclass will implement this method to return the messages array of CommonBaseEvent
	 * objects generated from the parsed data. 
	 * 
	 *
	 * @return the array of CBE's generated from the parsed data 
	 */
	public abstract CommonBaseEvent[] parseNext() throws LogParserException;

    /**
     * Abstract method used to retrieve the name of the parser.  Each concrete
     * subclass will implement this method to provide information about the type of log 
     * that it parses.
     *
     * @return the name of this parser 
     */
    public abstract String getName();

    /**
     * Abstract method used to retrieve the version of the parser.  Each concrete
     * subclass will implement this method to provide its version. 
     *
     * @return the version of this parser 
     */
    public abstract String getVersion();

	/**
	 * Method used to retrieve the message array.  
	 * 
	 * @return the message array 
	 */
	public CommonBaseEvent[] getMessages() {
		return messages;
	}

	public void preParse() throws LogParserException {

		getLocalHostId();
		originLocale = getLocale();
	
	}

	/**
	 * This function is called to provide user-specified information to the parser.
	 * Currently, this superclass method handles the path and name of the log file
	 * to be parsed.  Subclasses will override this method to handle additional
	 * user-specified information.
	 *
	 * @param table  Hashtable containing keys and values of user-specified information  
	 * @exception LogParserException thrown if user-specified path and name for the log file does not exist on the parsing machine
	 */
	public void setUserInput(Hashtable table) throws LogParserException {
		setConfiguration(table);
	}

    /**
     * This function is called to provide user-specified information to the parser.
     * Currently, this superclass method handles the path and name of the log file
     * to be parsed.  Subclasses will override this method to handle additional
     * user-specified information.
     *
     * @param table  Hashtable containing keys and values of user-specified information  
     * @exception LogParserException thrown if user-specified path and name for the log file does not exist on the parsing machine
     */
    public void setConfiguration(Hashtable table) throws LogParserException {
        file_path = null;
        Integer array_size;

        /** Commented code in this method could be used to allow user specification
         **  of the size of arrays to be written to the ILogger.
         **  String array_size = null; */

        try {
            file_path = ((String) (table.get(ParserConstants.FILE_PATH_KEY)));

            if (file_path != null) {

                //Must substitute platform dependent file separator character in the file_path instance variable for both UNIX and Windows platforms due to cross-platform communication via the Agent Controller:
                if (ParserConstants.FILE_SEPARATOR_CHARACTER == '/') {
                    file_path = file_path.replace('\\', ParserConstants.FILE_SEPARATOR_CHARACTER);
                }
                else {
                    file_path = file_path.replace('/', ParserConstants.FILE_SEPARATOR_CHARACTER);
                }

                /* Check for the validity of the file */
                File file = new File(file_path);

                if (!file.isFile()) {
                    throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_INVALID_FILE_NAME_ERROR_",file_path));
                }
                else if (!file.canRead()) {
                    throw new LogParserException(ParserUtilities.getResourceString("LOG_PARSER_CANNOT_READ_FILE_ERROR_",file_path));
                }

                logFile = new RandomAccessFile(file,"r");
            }

            charset = ((String) (table.get(ParserConstants.FILE_CHARSET_KEY)));
            fileLocale = ((Locale)(table.get(ParserConstants.FILE_LOCALE_KEY)));

            array_size = (Integer)table.get(ParserConstants.MESSAGE_ARRAY_SIZE_KEY);
            if (array_size != null) {
            	MessageArraySize = array_size.intValue();
            }
            
            messages = new CommonBaseEvent[MessageArraySize];
         
            for (int i = 0; i < MessageArraySize; i++) {
                messages[i] = eventFactory.createCommonBaseEvent();
            }
        }
        catch (Throwable t) {
            ParserUtilities.exceptionHandler(t, ParserUtilities.getResourceString("REMOTE_LOG_PARSER_CONFIG_PARAMETER_ERROR_"));
        }

        /** catch(NumberFormatException nfe)
         **{
         **    LogParserException lpe = new LogParserException("array_size value, " + array_size + " ,must be an integer.", nfe);
         **    lpe.fillInStackTrace();
         **    throw lpe;
         **}*/
    }

    /* utility methods */
	/**
	 * Function to increase the message array size by one.
	 * 
	 */    
	protected void increaseMsgArraySize()
	{
		CommonBaseEvent[] tempArray = new CommonBaseEvent[++MessageArraySize];
		System.arraycopy(messages, 0, tempArray, 0, messages.length);
		messages = tempArray;
		messages[MessageArraySize-1] = eventFactory.createCommonBaseEvent();
	}
	    
    /**
     * Function to determine IP address and fully-qualified host name
     * of the machine parsing the log file. 
     */
    protected void getLocalHostId() {
        try {
            localHostId = java.net.InetAddress.getLocalHost().getHostAddress();
            if (localHostId.indexOf(':') != -1)
                localHostIdFormat = ParserConstants.HOST_ID_FORMAT_IPV6;
            else
                localHostIdFormat = ParserConstants.HOST_ID_FORMAT_IPV4;
        }
        catch (Exception e) {
            localHostId = UNKNOWN_HOST;
            localHostIdFormat = ParserConstants.NONE;
        }
        try {
            localHostName = InetAddress.getByName(localHostId).getHostName();
        }
        catch (Exception e) {
            localHostName = null;
        }
    }

    /**
     * Function to determine the locale (language and country)
     * of the machine parsing the log file. 
     */
    protected String getLocale() {
        
    	String originLanguage = null;
    	String originRegion = null;
    	
    	/* If the locale of the file was passed in as a parameter then use it */
    	if (fileLocale != null) {
    		originLanguage = fileLocale.getLanguage();
    		originRegion = fileLocale.getCountry();
    	}
    	/* Else use the locale of the machine parsing the log file */
    	else {
    		originLanguage = System.getProperty("user.language");
    		originRegion = System.getProperty("user.region");
    	}
    	
        if ((originLanguage != null) && (originRegion != null)){
            return (originLanguage.concat("-").concat(originRegion));
		}
        
        return ("");
    }


    /**
     * Checks that the character in the line at the specified position is 
     * a number.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @return true if the character at the specified position in the string is a number
     */
    protected boolean isNum(String line, int pos) {
        try {
            return Character.isDigit(line.charAt(pos));
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Checks that the characters in the line from the specified position 
     * to that position plus (num - 1)  are all numbers.
     * 
     * @param line The string to check
     * @param pos The position at which to start the check
     * @param num The number of characters to check
     * @return True if all the characters checked are numbers
     */
    protected boolean isNum(String line, int pos, int num) {
        for (int i = 0; i < num; i++) {
            if (!isNum(line, pos + i))
                return false;
        }
        return true;
    }

    /**
     * Checks that the character in the line at the specified position is 
     * a letter.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @return true if the character at the specified position in the string is a letter
     */
    protected boolean isLet(String line, int pos) {
        try {
            return Character.isLetter(line.charAt(pos));
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Checks that the characters in the line from the specified position 
     * to that position + (num - 1) are all letters.
     * 
     * @param line The string to check
     * @param pos The position at which to start the check
     * @param num The number of characters to check
     * @return true if all the characters checked are letters
     */
    protected boolean isLet(String line, int pos, int num) {
        for (int i = 0; i < num; i++) {
            if (!isLet(line, pos + i))
                return false;
        }
        return true;
    }

    /**
     * Checks that the character in the line at the specified position is 
     * the specified character.
     * 
     * @param line The string to check
     * @param pos The position in the string to check
     * @param c The character that should be in this position in the string
     * @return true if the character at the specified position in the string is the specified character
     */
    protected boolean isChar(String line, int pos, char c) {
        try {
            return line.charAt(pos) == c;
        }
        catch (Exception e) {
            return false;
        }
    }

    /**
     * Perform cleanup after parsing is complete.
     */
    public void postParse() throws LogParserException {
        try {
            if (logFile != null)
                logFile.close();
        }
        catch (Exception e) {
            // no action required
			ParserUtilities.exceptionHandler(e, "");
        }
    }

    /**
     * Reads in the next line from the log file. 
     * <p>
     * A line of text is terminated by a carriage-return character 
     * ('\r'), a newline character ('\n'), a carriage-return character 
     * immediately followed by a newline character, or the end of the 
     * file. 
     * <p>
     * Line-terminating characters are discarded and are not included as 
     * part of the returned string. 
     * <p>
     * This API will return <code>null</code> if either an 
     * <code>IOException</code> occurs when reading from the 
     * file or the end of the file is reached before any characters
     * are read.
     * <p>
     *
     * @return   The next line in the file, otherwise null.
     */
    protected String readLine() {

        /*
        Implementation Notes
        Author: Paul E. Slauenwhite
        Since: October 25, 2004.
        
        1.  The RandomAccessFile.read() API is used to read each byte from the file until 
        the end of a line since the RandomAccessFile.readLine() API does not support the full 
        Unicode character set (e.g. DBCS).  
        
        2.  A RandomAccessFile is used to circumvent encoding issues on some UNIX platforms.
        Some log files (e.g. Apache error log) may contain control characters which cannot be read
        using a FileReader and/or BufferedReader. 
        
        3. On z/OS, the new-line character ('\n') is 0x0015 and not 0x000A as it is for Windows/UNIX.
        */
        
        try {

            //10 is the new-line character ('\n') on Windows/UNIX:
            int newLineByte = 10;
            
            //21 is the new-line character ('\n') on z/OS:
            if(ParserConstants.IS_ZOS_PLATFORM){
                newLineByte = 21;
            }
            
            //Create a byte buffer with a default size of 1024 bytes (e.g. upper-bound for expected line length) which may need to be resized:
            byte[] buffer = new byte[1024];
            
            //The current size or number of bytes in the byte buffer:
            int bufferSize = 0;

            //A single byte from the file:
            int singleByte = 0;
            
            //Iterate the remaining bytes in the file until the EOF (e.g. -1) or EOL character(s) (e.g. '\n', '\r' or '\r\n'):
            while((singleByte = logFile.read()) != -1) {

                //10 is the new-line character ('\n') on Windows/UNIX and 21 is the new-line character ('\n') on z/OS:
                if(singleByte == newLineByte){
                    break;
                }

                //13 is the carriage-return character ('\r'):
                else if(singleByte == 13){
                    
                    //Check ahead for a potential new-line character:
                    long currentFilePointer = logFile.getFilePointer();
                    
                    //If the next character is not a new-line character, roll-back to the previous byte's position:
                    if(logFile.read() != newLineByte){
                        logFile.seek(currentFilePointer);
                    }
                    
                    break;
                }

                //If the byte buffer has overflowed, resize by another 1024 indices:
                if(bufferSize == buffer.length){
                    
                    byte[] tempBuffer = new byte[buffer.length + 1024];
                    
                    System.arraycopy(buffer,0,tempBuffer,0,bufferSize);
                    
                    buffer = tempBuffer;                                       
                }
                
                //Valid downcast (int --> byte) since the RandomAccessFile.read() API returns an integer in the range 0 to 255 (0x00 - 0x0ff). 
                buffer[bufferSize++] = ((byte)(singleByte));
            }
            
            if (bufferSize > 0) {
           
                curLineNumber++;

                //Convert the byte buffer to a string based on the platform's default charset:
                return (new String(buffer, 0, bufferSize));
           }
        } 
        catch (IOException i) {
            //Ignore and return null.
        }

        return null;
    }

    /**
     * Method createStringEDE.  Creates an object which implements the 
     * IExtendedDataElement interface and fills in the name/value pair.
     * This functions handles values that are single strings.
     * 
     * @param name Name portion of the name/value pair
     * @param value Value portion of the name/value pair
     * @return IExtendedDataElement
     */
    protected static ExtendedDataElement createStringEDE(String name, String value) {

        ExtendedDataElement extendedDataElement = eventFactory.createExtendedDataElement();
        extendedDataElement.setName(name);

        int valuesLength = 0;

        if ((value != null) && ((valuesLength = value.trim().length()) > 1024)) {

            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

            ArrayList valuesParts = new ArrayList();

            for (int counter = 0; counter < valuesLength; counter += 1024) {
                valuesParts.add(value.substring(counter, Math.min((counter + 1024), valuesLength)));
            }

            extendedDataElement.setValues(((String[]) (valuesParts.toArray(new String[valuesParts.size()]))));
        }
        else {

            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElement.setValues(new String[] {value});
        }

        return extendedDataElement;
    }
}
