package org.eclipse.hyades.logging.events.cbe.impl.tests;

import java.text.ParseException;
import java.util.Arrays;
import java.util.List;

import junit.framework.TestCase;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.ValidationException;
import org.eclipse.hyades.logging.events.cbe.impl.ExtendedDataElementImpl;
import org.eclipse.hyades.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * JUnit test case for the <code>ExtendedDataElement</code>
 * class.
 * 
 * @author Scott Brown
 * @author Andrea Chiavegatto
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1
 */
public class ExtendedDataElementImplTest extends TestCase {
    
    private ExtendedDataElement ede = null;

    /**
     * Constructor with a testcase name.
     * 
     * @param testcase       name of the testcase.
     */
    public ExtendedDataElementImplTest(String testcase) {
        super(testcase);
    }

    /**
     * 
     */
    protected void setUp() throws Exception {
        super.setUp();

        ede = EventFactory.eINSTANCE.createExtendedDataElement();
    }

    /**
     * 
     */
    protected void tearDown() throws Exception {
        super.tearDown();
    }

    public void testMultipleChildrenWithSameContents() {

        ExtendedDataElement extendedDataElement = null;

        for (int counter = 0; counter < 5; counter++) {

            extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
            extendedDataElement.setName("name");
            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElement.setValues(new String[] {"values"});

            ede.addChild(extendedDataElement);
        }

        assertTrue(ede.getChildren().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ExtendedDataElement) (ede.getChildren().get(counter))).getName(), "name");
            assertEquals(((ExtendedDataElement) (ede.getChildren().get(counter))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
            assertEquals(((String)(((ExtendedDataElement) (ede.getChildren().get(counter))).getValues().get(0))), "values");
        }
    }

    public void testMultipleChildrenWithSameObjects() {
       
        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        extendedDataElement.setName("name");
        extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
        extendedDataElement.setValues(new String[] {"values"});

        for (int counter = 0; counter < 5; counter++) {
            ede.addChild(extendedDataElement);
        }

        //NOTE:  The resultant EList only contains one element since the same 
        //object instance cannot be duplicated in the same EList more than once since the 
        //lifetime of the contained object is controlled by the container object.
        assertTrue(ede.getChildren().size() == 1);

        assertEquals(((ExtendedDataElement) (ede.getChildren().get(0))).getName(), "name");
        assertEquals(((ExtendedDataElement) (ede.getChildren().get(0))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
        assertEquals(((String)(((ExtendedDataElement) (ede.getChildren().get(0))).getValues().get(0))), "values");
    }

    public void testMultipleChildrenArrayWithSameContents() {
       
        ExtendedDataElement[] extendedDataElements = new ExtendedDataElement[5];
        
        for (int counter = 0; counter < 5; counter++) {

            extendedDataElements[counter] = EventFactory.eINSTANCE.createExtendedDataElement();
            extendedDataElements[counter].setName("name");
            extendedDataElements[counter].setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
            extendedDataElements[counter].setValues(new String[] {"values"});
        }
        
        ede.getChildren().addAll(Arrays.asList(extendedDataElements));

        assertTrue(ede.getChildren().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((ExtendedDataElement) (ede.getChildren().get(counter))).getName(), "name");
            assertEquals(((ExtendedDataElement) (ede.getChildren().get(counter))).getTypeAsInt(), ExtendedDataElement.TYPE_STRING_VALUE);
            assertEquals(((String)(((ExtendedDataElement) (ede.getChildren().get(counter))).getValues().get(0))), "values");
        }
    }    
    
    public void testMultipleValuesWithSameContents() {

        for (int counter = 0; counter < 5; counter++) {
            ede.getValues().add(new String("value"));
        }

        assertTrue(ede.getValues().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (ede.getValues().get(counter))), "value");
        }
    }

    public void testMultipleValuesWithSameObjects() {
       
        String value = new String("value");

        for (int counter = 0; counter < 5; counter++) {
            ede.getValues().add(value);
        }

        assertTrue(ede.getValues().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (ede.getValues().get(counter))), "value");
        }
    }

    public void testMultipleValuesArrayWithSameContents() {
        
        String[] values = new String[5];
        
        for (int counter = 0; counter < 5; counter++) {
            values[counter] = new String("value");
        }

        ede.getValues().addAll(Arrays.asList(values));
        
        assertTrue(ede.getValues().size() == 5);

        for (int counter = 0; counter < 5; counter++) {
            assertEquals(((String) (ede.getValues().get(counter))), "value");
        }
     }

    /**
     * Tests the serialization for <code>AvailableSituation</code>.
     */
    public void testSerialization() {
        Util.testEObjectSerialization(ede);
    }

    public void testEMF() {
        ede.getValues().add(null);
        //System.out.println("AAA" + ede.getValues().size());
    }

    /**
     * 
     */
    public void testGetChildrenByName() {
        int samples = 10;
        for (int i = 0; i < samples; i++) {
            for (int j = 0; j < i; j++) {
                ExtendedDataElement e1 = ede.addChild("name" + i);
                e1.getValues().add("value" + i + " " + j);
            }
        }

        for (int i = 0; i < samples; i++) {
            List elements = ede.getChildren("name" + i);
            assertEquals(i, elements.size());
            for (int j = 0; j < elements.size(); j++) {
                ExtendedDataElement e1 = (ExtendedDataElement) elements.get(j);
                assertEquals("value" + i + " " + j, e1.getValues().get(0));
            }
        }
    }

    /**
     * Test for both the getters and setters of hexValue.
     * <P>
     * Never mind that the strings are invalid (not hex) ,
     * there is another unit test for checking whether hex
     * values are accepted
     */
    public void testGetSetHexValue() {
        assertNull(ede.getHexValue()); // test default value

        String input = "Testing setHexValue()";
        ede.setHexValue(input);
        assertEquals(input, ede.getHexValue());

        input = "setting another string";
        ede.setHexValue(input);
        assertEquals(input, ede.getHexValue());

        ede.setHexValue(null);
        assertNull(ede.getHexValue());
    }

    /**
     * Tests whether the <code>getHexValueAsByteArray</code>
     * throws exceptions for ill-formed hexValue strings.
     */
    public void testInvalidHexValue() {
        try {
            ede.setHexValue("ZZZ123"); // event length, bad chars
            ede.setType(ExtendedDataElement.TYPE_HEX_BINARY);
            ede.getHexValueAsByteArray();
            fail("String is not a valid hexValue");
        }
        catch (IllegalStateException e) {
            // expected
        }
        StringBuffer hexString = new StringBuffer("A");
        for (int i = 0; i < 100; i++) {
            try {
                ede.setHexValue(hexString.toString()); // odd length
                ede.getHexValueAsByteArray();
                fail("String (" + hexString.length() + ") is not a valid hexValue");
            }
            catch (IllegalStateException e) {
                // expected
            }
        }
    }

    public void testGetName() {
        assertTrue(ede.getName() == null);

        String input = new String("Testing getName()");
        ede.setName(input);
        assertTrue(ede.getName().equals(input));
    }

    public void testSetName() {
        String input = new String("Testing setName()");
        ede.setName(input);
        assertTrue(ede.getName().equals(input));

        input = new String("setting another name");
        ede.setName(input);
        assertTrue(ede.getName().equals(input));
    }

    public void testGetType() {
        assertTrue(ede.getType() == null);

        ede.setType(ExtendedDataElement.TYPE_LONG);
        assertTrue(ede.getType().equals(ExtendedDataElement.TYPE_LONG));
    }

    public void testSetType() {
        ede.setType(ExtendedDataElement.TYPE_STRING_ARRAY);
        assertTrue(ede.getType().equals(ExtendedDataElement.TYPE_STRING_ARRAY));

        String type = "MyType";
        ede.setType(type);
        assertTrue(ede.getType().equals(type));
    }

    public void testGetTypeAsInt() {
        assertTrue(ede.getTypeAsInt() == ExtendedDataElement.TYPE_NO_VALUE_VALUE);

        ede.setTypeAsInt(ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
        assertTrue(ede.getTypeAsInt() == ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
    }

    public void testSetTypeAsInt() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        assertTrue(ede.getTypeAsInt() == ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

        ede.setTypeAsInt(123);
        assertTrue(ede.getTypeAsInt() == ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    }

    public void testGetValues() {
        assertTrue(ede.getValues().isEmpty());

        ede.setValuesAsString("testing getValues()");
        assertTrue(ede.getValues().size() == 1);
    }

    public void testGetChildren() {
        assertTrue(ede.getValues().isEmpty());

        ede.addChild("child 1");
        ede.addChild("child 2");
        assertTrue(ede.getChildren().size() == 2);
    }

    public void testGetValuesAsBoolean() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BOOLEAN_VALUE);
        try {
            ede.getValuesAsBoolean();
            fail("cannot get int value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        boolean inValue = false;
        ede.setValuesAsBoolean(inValue);
        boolean outValue = ede.getValuesAsBoolean();
        assertTrue(outValue == inValue);

        // test get int from different type object (String)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsBoolean();
            fail("cannot get boolean value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsBooleans() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE);
        ede.getValuesAsBooleanArray();

        // test get & set
        boolean[] inValue = { false, true, true };
        ede.setValuesAsBooleanArray(inValue);
        boolean[] outValue = ede.getValuesAsBooleanArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsBooleanArray();
            fail("cannot get boolean[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsByte() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BYTE_VALUE);
        try {
            ede.getValuesAsByte();
            fail("cannot get byte value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        byte inValue = Byte.MAX_VALUE;
        ede.setValuesAsByte(inValue);
        byte outValue = ede.getValuesAsByte();
        assertTrue(outValue == inValue);

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsByte();
            fail("cannot get byte value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsBytes() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BYTE_ARRAY_VALUE);
        ede.getValuesAsByteArray();

        // test get & set
        byte[] inValue = { 0x23, 0x54, 0x78 };
        ede.setValuesAsByteArray(inValue);
        byte[] outValue = ede.getValuesAsByteArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsByteArray();
            fail("cannot get byte[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDate() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);
        try {
            ede.getValuesAsDate();
            fail("cannot get Date value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        String inValue = "2003-02-18T14:30:09.000Z";
        ede.setValuesAsDate(inValue);
        String outValue = ede.getValuesAsDate();
        assertTrue(inValue.equals(outValue));

        // test get Date from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDate();
            fail("cannot get Date value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDates() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        ede.getValuesAsDateArray();

        // test get & set
        String[] inValue = { "1997-01-01T12:00:00.000Z", "2003-02-18T14:30:09.000Z" };
        ede.setValuesAsDateArray(inValue);
        String[] outValue = ede.getValuesAsDateArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDateArray();
            fail("cannot get Date[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDateAsLong() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);
        try {
            ede.getValuesAsDateAsLong();
            fail("cannot get Date value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        long inValue = System.currentTimeMillis();
        String dateStr = EventHelpers.longToDate(inValue);
        ede.setValuesAsDateAsLong(inValue);
        long outValue = ede.getValuesAsDateAsLong();
        assertTrue(inValue == outValue);

        // test get Date from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDateAsLong();
            fail("cannot get DateAsLong value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    /**
     *
     */
    public void testGetValuesAsDatesAsLongWithBadDate() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        ede.setValuesAsDateArray(new String[] { "awinga" });

        // test get byte from different type object (int)
        try {
            ede.getValuesAsDateAsLongArray();
            fail("The date is not a valid xsd:dateTime");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDatesAsLong() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        ede.getValuesAsDateAsLongArray();

        // test get & set
        long[] inValue = { System.currentTimeMillis(), System.currentTimeMillis() + 10000 };
        ede.setValuesAsDateAsLongArray(inValue);
        long[] outValue = ede.getValuesAsDateAsLongArray();
        assertEquals(inValue.length, outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertEquals("Index: " + i, inValue[i], outValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDateAsLongArray();
            fail("cannot get DateAsLong[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDouble() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DOUBLE_VALUE);
        try {
            ede.getValuesAsDouble();
            fail("cannot get double value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        double inValue = Double.MIN_VALUE;
        ede.setValuesAsDouble(inValue);
        double outValue = ede.getValuesAsDouble();
        assertTrue(outValue == inValue);

        // test get double from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDouble();
            fail("cannot get double value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsDoubles() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE);
        ede.getValuesAsDoubleArray();

        // test get & set
        double[] inValue = { 1.25, 1.33 };
        ede.setValuesAsDoubleArray(inValue);
        double[] outValue = ede.getValuesAsDoubleArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsDoubleArray();
            fail("cannot get double[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsFloat() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_FLOAT_VALUE);
        try {
            ede.getValuesAsFloat();
            fail("cannot get float value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        float inValue = Float.MAX_VALUE;
        ede.setValuesAsFloat(inValue);
        float outValue = ede.getValuesAsFloat();
        assertTrue(outValue == inValue);

        // test get float from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsFloat();
            fail("cannot get float value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsFloats() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE);
        ede.getValuesAsFloatArray();

        // test get & set
        float[] inValue = { 1000, 2000, 3000, 1000 };
        ede.setValuesAsFloatArray(inValue);
        float[] outValue = ede.getValuesAsFloatArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsFloatArray();
            fail("cannot get float[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsInt() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);
        try {
            ede.getValuesAsInt();
            fail("cannot get int value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        int inValue = Integer.MIN_VALUE;
        ede.setValuesAsInt(inValue);
        int outValue = ede.getValuesAsInt();
        assertTrue(outValue == inValue);

        // test get int from different type object (String)
        try {
            ede.setType(ExtendedDataElement.TYPE_STRING);
            ede.getValuesAsInt();
            fail("cannot get int value from String type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsInts() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_INT_ARRAY_VALUE);
        ede.getValuesAsIntArray();

        // test get & set
        int[] inValue = { 9, 7, 5 };
        ede.setValuesAsIntArray(inValue);
        int[] outValue = ede.getValuesAsIntArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsIntArray();
            fail("cannot get int[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsLong() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_LONG_VALUE);
        try {
            ede.getValuesAsLong();
            fail("cannot get long value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        long inValue = Long.MAX_VALUE;
        ede.setValuesAsLong(inValue);
        long outValue = ede.getValuesAsLong();
        assertTrue(outValue == inValue);

        // test get long from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsLong();
            fail("cannot get long value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsLongs() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_LONG_ARRAY_VALUE);
        ede.getValuesAsLongArray();

        // test get & set
        long[] inValue = { 9999999, 7777777, 5555555 };
        ede.setValuesAsLongArray(inValue);
        long[] outValue = ede.getValuesAsLongArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_LONG);
            ede.getValuesAsLongArray();
            fail("cannot get long[] value from long type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsShort() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_SHORT_VALUE);
        try {
            ede.getValuesAsShort();
            fail("cannot get short value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        short inValue = Short.MIN_VALUE;
        ede.setValuesAsShort(inValue);
        short outValue = ede.getValuesAsShort();
        assertTrue(outValue == inValue);

        // test get short from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsShort();
            fail("cannot get short value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsShorts() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_SHORT_ARRAY_VALUE);
        ede.getValuesAsShortArray();

        // test get & set
        short[] inValue = { 0, 1, 2 };
        ede.setValuesAsShortArray(inValue);
        short[] outValue = ede.getValuesAsShortArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsShortArray();
            fail("cannot get short[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsString() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
        try {
            ede.getValuesAsString();
            fail("cannot get String value from uninitialized object");
        }
        catch (IllegalStateException e) {
        }

        // test get & set
        String inValue = "test1";
        ede.setValuesAsString(inValue);
        String outValue = ede.getValuesAsString();
        assertTrue(outValue.equals(inValue));

        // test get String from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsString();
            fail("cannot get String value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testGetValuesAsStrings() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
        ede.getValuesAsStringArray();

        // test get & set
        String[] inValue = { "test1", "test2", "test3", "test4" };
        ede.setValuesAsStringArray(inValue);
        String[] outValue = ede.getValuesAsStringArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getValuesAsStringArray();
            fail("cannot get String[] value from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testSetValuesAsBoolean() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BOOLEAN_VALUE);

        // test get & set
        boolean inValue = true;
        ede.setValuesAsBoolean(inValue);
        boolean outValue = ede.getValuesAsBoolean();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = false;
        ede.setValuesAsBoolean(inValue);
        outValue = ede.getValuesAsBoolean();
        assertTrue(outValue == inValue);

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsBoolean(inValue);
    }

    public void testSetValuesAsBooleans() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE);

        // test get & set
        boolean[] inValue = { true, false };
        ede.setValuesAsBooleanArray(inValue);
        boolean[] outValue = ede.getValuesAsBooleanArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        boolean[] inValue2 = { false };
        ede.setValuesAsBooleanArray(inValue2);
        outValue = ede.getValuesAsBooleanArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsBooleanArray(inValue);
    }

    public void testSetValuesAsByte() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BYTE_VALUE);

        // test get & set
        byte inValue = Byte.MIN_VALUE;
        ede.setValuesAsByte(inValue);
        byte outValue = ede.getValuesAsByte();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Byte.MAX_VALUE;
        ede.setValuesAsByte(inValue);
        outValue = ede.getValuesAsByte();
        assertTrue(outValue == inValue);

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsByte(inValue);
    }

    public void testSetValuesAsBytes() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_BYTE_ARRAY_VALUE);

        // test get & set
        byte[] inValue = { 0x23, 0x54, 0x78 };
        ede.setValuesAsByteArray(inValue);
        byte[] outValue = ede.getValuesAsByteArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        byte[] inValue2 = { 0x61, 0x09 };
        ede.setValuesAsByteArray(inValue2);
        outValue = ede.getValuesAsByteArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsByteArray(inValue);
    }

    public void testSetValuesAsDate() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);

        // test get & set
        String inValue = "2000-12-25T09:45:05.000Z";
        ede.setValuesAsDate(inValue);
        String outValue = ede.getValuesAsDate();
        assertTrue(outValue.equals(inValue));

        // do it again to see if the value stored in the ede object is replaced
        inValue = "2001-02-05T19:5:35.000Z";
        ede.setValuesAsDate(inValue);
        outValue = ede.getValuesAsDate();
        assertTrue(outValue.equals(inValue));

        // test set double to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDate(inValue);
    }

    public void testSetValuesAsDates() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

        // test get & set
        String[] inValue = { "2000-12-25T09:45:05.000Z", "2001-12-31T11:59:59.000Z" };
        ede.setValuesAsDateArray(inValue);
        String[] outValue = ede.getValuesAsDateArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // do it again to see if the value stored in the ede object is replaced
        String[] inValue2 = { "1998-02-15T19:35:55.000Z", "1999-12-31T11:00:00.000Z", "1996-10-11T08:30:00.000Z" };
        ede.setValuesAsDateArray(inValue);
        outValue = ede.getValuesAsDateArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDateArray(inValue);
    }

    /**
     *
     */
    public void testGetValuesAsDateAsLongWithBadDate() {
        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);
        ede.setValuesAsDate("awinga");

        // test get byte from different type object (int)
        try {
            ede.getValuesAsDateAsLong();
            fail("The date is not a valid xsd:dateTime");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testSetValuesAsDateAsLong() throws ParseException {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_VALUE);

        // test get & set
        String dateStr = "1993-12-10T13:55:19.000Z";
        long inValue = EventHelpers.dateToLong(dateStr);
        ede.setValuesAsDateAsLong(inValue);
        long outValue = ede.getValuesAsDateAsLong();
        assertTrue(outValue == inValue);
        String outDate = ede.getValuesAsDate();
        assertTrue(dateStr.equals(outDate));

        // do it again to see if the value stored in the ede object is replaced
        inValue = System.currentTimeMillis() + 10000;
        ede.setValuesAsDateAsLong(inValue);
        outValue = ede.getValuesAsDateAsLong();
        assertTrue(outValue == inValue);

        // test set double to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDateAsLong(inValue);
    }

    public void testSetValuesAsDatesAsLong() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);

        // test get & set
        long[] inValue = { System.currentTimeMillis(), System.currentTimeMillis() + 10000 };
        ede.setValuesAsDateAsLongArray(inValue);
        long[] outValue = ede.getValuesAsDateAsLongArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        long[] inValue2 = { System.currentTimeMillis() + 20000, System.currentTimeMillis() + 30000, System.currentTimeMillis() + 40000, };
        ede.setValuesAsDateAsLongArray(inValue2);
        outValue = ede.getValuesAsDateAsLongArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDateAsLongArray(inValue);
    }

    public void testSetValuesAsDouble() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DOUBLE_VALUE);

        // test get & set
        double inValue = Double.MIN_VALUE;
        ede.setValuesAsDouble(inValue);
        double outValue = ede.getValuesAsDouble();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Double.MAX_VALUE;
        ede.setValuesAsDouble(inValue);
        outValue = ede.getValuesAsDouble();
        assertTrue(outValue == inValue);

        // test set double to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDouble(inValue);
    }

    public void testSetValuesAsDoubles() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE);

        // test get & set
        double[] inValue = { 2.6, 1.4, 9.8 };
        ede.setValuesAsDoubleArray(inValue);
        double[] outValue = ede.getValuesAsDoubleArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        double[] inValue2 = { 3.7, 0.5 };
        ede.setValuesAsDoubleArray(inValue2);
        outValue = ede.getValuesAsDoubleArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsDoubleArray(inValue);
    }

    public void testSetValuesAsFloat() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_FLOAT_VALUE);

        // test get & set
        float inValue = Float.MIN_VALUE;
        ede.setValuesAsFloat(inValue);
        float outValue = ede.getValuesAsFloat();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Float.MAX_VALUE;
        ede.setValuesAsFloat(inValue);
        outValue = ede.getValuesAsFloat();
        assertTrue(outValue == inValue);

        // test set float to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsFloat(inValue);
    }

    public void testSetValuesAsFloats() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE);

        // test get & set
        float[] inValue = { 1000, 2000, 3000 };
        ede.setValuesAsFloatArray(inValue);
        float[] outValue = ede.getValuesAsFloatArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        float[] inValue2 = { 1, 22, 333, 4444, 55555 };
        ede.setValuesAsFloatArray(inValue2);
        outValue = ede.getValuesAsFloatArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsFloatArray(inValue);
    }

    public void testSetValuesAsInt() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_INT_VALUE);

        // test get & set
        int inValue = Integer.MIN_VALUE;
        ede.setValuesAsInt(inValue);
        int outValue = ede.getValuesAsInt();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Integer.MAX_VALUE;
        ede.setValuesAsInt(inValue);
        outValue = ede.getValuesAsInt();
        assertTrue(outValue == inValue);

        // test set int to different type object (String)
        ede.setType(ExtendedDataElement.TYPE_STRING);
        ede.setValuesAsInt(inValue);
    }

    public void testSetValuesAsInts() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_INT_ARRAY_VALUE);

        // test get & set
        int[] inValue = { 12, 25, 37, 41, 59, 68 };
        ede.setValuesAsIntArray(inValue);
        int[] outValue = ede.getValuesAsIntArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        int[] inValue2 = { 9, 21 };
        ede.setValuesAsIntArray(inValue2);
        outValue = ede.getValuesAsIntArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsIntArray(inValue);
    }

    public void testSetValuesAsLong() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_LONG_VALUE);

        // test get & set
        long inValue = Long.MIN_VALUE;
        ede.setValuesAsLong(inValue);
        long outValue = ede.getValuesAsLong();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Long.MAX_VALUE;
        ede.setValuesAsLong(inValue);
        outValue = ede.getValuesAsLong();
        assertTrue(outValue == inValue);

        // test set long to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsLong(inValue);
    }

    public void testSetValuesAsLongs() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_LONG_ARRAY_VALUE);

        // test get & set
        long[] inValue = { 1234567890, 234567890, 345678900 };
        ede.setValuesAsLongArray(inValue);
        long[] outValue = ede.getValuesAsLongArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        long[] inValue2 = { 12345, 23456 };
        ede.setValuesAsLongArray(inValue2);
        outValue = ede.getValuesAsLongArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsLongArray(inValue);
    }

    public void testSetValuesAsShort() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_SHORT_VALUE);

        // test get & set
        short inValue = Short.MIN_VALUE;
        ede.setValuesAsShort(inValue);
        short outValue = ede.getValuesAsShort();
        assertTrue(outValue == inValue);

        // do it again to see if the value stored in the ede object is replaced
        inValue = Short.MAX_VALUE;
        ede.setValuesAsShort(inValue);
        outValue = ede.getValuesAsShort();
        assertTrue(outValue == inValue);

        // test set short to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsShort(inValue);
    }

    public void testSetValuesAsShorts() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_SHORT_ARRAY_VALUE);

        // test get & set
        short[] inValue = { 1, 2, 3 };
        ede.setValuesAsShortArray(inValue);
        short[] outValue = ede.getValuesAsShortArray();
        assertTrue(inValue.length == outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertTrue(outValue[i] == inValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        short[] inValue2 = { 4, 5, 6 };
        ede.setValuesAsShortArray(inValue2);
        outValue = ede.getValuesAsShortArray();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertTrue(outValue[i] == inValue2[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsShortArray(inValue);
    }

    public void testSetValuesAsString() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_STRING_VALUE);
        ;

        // test get & set
        String inValue = "test1";
        ede.setValuesAsString(inValue);
        String outValue = ede.getValuesAsString();
        assertTrue(inValue.equals(outValue));

        // do it again to see if the value stored in the ede object is replaced
        inValue = "test2";
        ede.setValuesAsString(inValue);
        outValue = ede.getValuesAsString();
        assertTrue(inValue.equals(outValue));

        // test set String to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsString(inValue);
    }

    public void testSetValuesAsStrings() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

        // test get & set
        String[] inValue = { "test1", "test2", "test3" };
        ede.setValuesAsStringArray(inValue);
        String[] outValue = ede.getValuesAsStringArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // do it again to see if the value stored in the ede object is replaced
        String[] inValue2 = { "test4", "test5" };
        ede.setValuesAsStringArray(inValue);
        outValue = ede.getValuesAsStringArray();
        assertTrue(Arrays.asList(outValue).containsAll(Arrays.asList(inValue)));

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setValuesAsStringArray(inValue);
    }

    /**
     * Tests whether the hex value conversion is independent from
     * the textcase. 
     */
    public void testHexValueStringCase() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_HEX_BINARY_VALUE);

        ede.setHexValue("abcdef");
        byte[] values1 = ede.getHexValueAsByteArray();

        ede.setHexValue("ABCDEF");
        byte[] values2 = ede.getHexValueAsByteArray();

        assertEquals(3, values1.length);
        assertEquals(values1.length, values2.length);

        for (int i = 0; i < values1.length; i++) {
            assertEquals("Mismatch on " + i, values1[i], values2[i]);
        }

    }

    /**
     * Tests whether the object can properly convert byte streams
     * back and forth. 
     */
    public void testHexValueStringConversionWithBadHex() throws ValidationException {
        ede.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        ede.setName("Name");
        ede.setHexValue("abcZ1234");
        try {
            ede.getValuesAsByteArray();
            fail("The hex value is not valid");
        }
        catch (IllegalStateException e) {
            // expected
        }
    }

    /**
     * Tests whether the object can properly convert byte streams
     * back and forth. 
     */
    public void testHexValueStringConversion() {
        final int arraySize = 600000;

        // test uninitialized object
        ede.setTypeAsInt(ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
        assertNull(ede.getHexValueAsByteArray());

        // Creating the basis for comparison. mold is a string
        // of the form "000102030405...FCFDFEFF000102..."
        StringBuffer mold = new StringBuffer();
        byte[] inValue = new byte[arraySize];
        int unicodeBase0Value = Character.getNumericValue('0');
        int unicodeBaseAValue = Character.getNumericValue('A');
        for (int i = 0; i < arraySize; i++) {
            inValue[i] = (byte) i;
            int hiOrder = (i & 0xF0) >> 4;
            int hiValue = (hiOrder < 10) ? unicodeBase0Value + hiOrder : unicodeBaseAValue + hiOrder - 10;
            mold.append(Character.forDigit(hiValue, 16));
            int loOrder = i & 0x0F;
            int loValue = (loOrder < 10) ? unicodeBase0Value + loOrder : unicodeBaseAValue + loOrder - 10;
            mold.append(Character.forDigit(loValue, 16));
        }
        String moldStr = mold.toString().toUpperCase();

        // checking whether the string representation is correct.
        ede.setHexValueAsByteArray(inValue);
        assertEquals(moldStr, ede.getHexValue());

        // checking whether the byte array is returned properly
        byte[] outValue = ede.getHexValueAsByteArray();
        assertEquals("Wrong length", inValue.length, outValue.length);
        for (int i = 0; i < inValue.length; i++) {
            assertEquals("Mismatch on index: " + i, inValue[i], outValue[i]);
        }

        // test get byte from different type object (int)
        try {
            ede.setType(ExtendedDataElement.TYPE_INT);
            ede.getHexValueAsByteArray();
            fail("cannot get byte[] hexValue from int type");
        }
        catch (IllegalStateException e) {
        }
    }

    public void testSetHexValueAsBytes() {
        ede.setTypeAsInt(ExtendedDataElement.TYPE_HEX_BINARY_VALUE);

        // test get & set
        byte[] inValue = { 0x23, 0x54, 0x78 };
        ede.setHexValueAsByteArray(inValue);
        byte[] outValue = ede.getHexValueAsByteArray();
        String outValue2 = ede.getHexValue();
        assertEquals(inValue.length, outValue.length);

        for (int i = 0; i < inValue.length; i++) {
            assertEquals("Wrong value", inValue[i], outValue[i]);
        }

        // do it again to see if the value stored in the ede object is replaced
        byte[] inValue2 = { 0x61, 0x09 };
        ede.setHexValueAsByteArray(inValue2);
        outValue = ede.getHexValueAsByteArray();
        outValue2 = ede.getHexValue();
        assertTrue(inValue2.length == outValue.length);

        for (int i = 0; i < inValue2.length; i++) {
            assertEquals("Wrong value", inValue2[i], outValue[i]);
        }

        // test set byte to different type object (int)
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.setHexValueAsByteArray(inValue);
    }

    public void testAddChild(String name) {
        ExtendedDataElement child1 = ede.addChild("child 1");
        ExtendedDataElement child2 = ede.addChild("child 2");
        EList children = ede.getChildren();
        assertTrue(children.size() == 2);
        assertTrue(children.contains(child2));
        assertTrue(((ExtendedDataElement) children.get(1)).getName().equals("child 2"));
        assertTrue(children.contains(child1));
    }

    public void testAddChild1() {
        ExtendedDataElement child = ede.addChild("child 1", ExtendedDataElement.TYPE_BOOLEAN_VALUE, "true");
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);
        assertTrue(children.contains(child));

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_BOOLEAN_VALUE);
        assertTrue(tmpEde.getValuesAsBoolean() == true);
    }

    public void testAddChild2() {
        String[] values = { "test1", "test2" };
        ede.addChild("child 1", ExtendedDataElement.TYPE_STRING_ARRAY_VALUE, values);
        ede.addChild("child 2", ExtendedDataElement.TYPE_LONG_VALUE, "123");
        EList children = ede.getChildren();
        assertTrue(children.size() == 2);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
        assertTrue(Arrays.asList(tmpEde.getValuesAsStringArray()).containsAll(Arrays.asList(values)));

        tmpEde = (ExtendedDataElement) children.get(1);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_LONG_VALUE);
        assertTrue(tmpEde.getValuesAsLong() == 123);
    }

    public void testAddChild3() {
        String value = "test 1";
        ede.addChild("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_STRING_VALUE);
        assertTrue(tmpEde.getValuesAsString().equals(value));
    }

    public void testAddChild4() {
        String[] values = { "test 1", "test 2" };
        ede.addChild("child 1", values);

        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);
        assertTrue(Arrays.asList(tmpEde.getValuesAsStringArray()).containsAll(Arrays.asList(values)));
    }

    public void testAddChildWithBooleanrrayValue() {
        boolean[] value = { false, true };
        ede.addChildWithBooleanArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE);
        boolean[] outValue = tmpEde.getValuesAsBooleanArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithBooleanValue() {
        boolean value = true;
        ede.addChildWithBooleanValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_BOOLEAN_VALUE);
        assertTrue(tmpEde.getValuesAsBoolean() == value);
    }

    public void testAddChildWithByteArrayValue() {
        byte[] value = { 'a', 'b' };
        ede.addChildWithByteArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_BYTE_ARRAY_VALUE);
        byte[] outValue = tmpEde.getValuesAsByteArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithByteValue() {
        byte value = 'a';
        ede.addChildWithByteValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_BYTE_VALUE);
        assertTrue(tmpEde.getValuesAsByte() == value);
    }

    public void testAddChildWithDateArrayValue() {
        String[] value = { "2000-12-31T11:00:00.000Z", "2001-12-31T11:00:00.000Z", "2002-12-31T11:00:00.000Z" };
        ede.addChildWithDateArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        assertTrue(Arrays.asList(tmpEde.getValuesAsDateArray()).containsAll(Arrays.asList(value)));
    }

    public void testAddChildWithDateAsLongValue() {
        long value = System.currentTimeMillis();
        ede.addChildWithDateAsLongValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DATE_TIME_VALUE);
        assertTrue(tmpEde.getValuesAsDateAsLong() == value);
    }

    public void testAddChildWithDatesAsLongValue() {
        long[] value = { System.currentTimeMillis(), System.currentTimeMillis() + 10000 };
        ede.addChildWithDatesAsLongValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE);
        long[] outValue = tmpEde.getValuesAsDateAsLongArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithDateValue() {
        String value = "2003-11-10T12:00:00.000Z";
        ede.addChildWithDateValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DATE_TIME_VALUE);
        assertTrue(tmpEde.getValuesAsDate().equals(value));
    }

    public void testAddChildWithDoubleArrayValue() {
        double[] value = { 34.7, 65.8 };
        ede.addChildWithDoubleArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE);
        double[] outValue = tmpEde.getValuesAsDoubleArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithDoubleValue() {
        double value = 12.5;
        ede.addChildWithDoubleValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_DOUBLE_VALUE);
        assertTrue(tmpEde.getValuesAsDouble() == value);
    }

    public void testAddChildWithFloatArrayValue() {
        float[] value = { 213213123, 123124144 };
        ede.addChildWithFloatArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE);
        float[] outValue = tmpEde.getValuesAsFloatArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithFloatValue() {
        float value = 'a';
        ede.addChildWithFloatValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_FLOAT_VALUE);
        assertTrue(tmpEde.getValuesAsFloat() == value);
    }

    public void testAddChildWithIntArrayValue() {
        int[] value = { 10, 12 };
        ede.addChildWithIntArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_INT_ARRAY_VALUE);
        int[] outValue = tmpEde.getValuesAsIntArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithIntValue() {
        int value = 123;
        ede.addChildWithIntValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_INT_VALUE);
        assertTrue(tmpEde.getValuesAsInt() == value);
    }

    public void testAddChildWithLongArrayValue() {
        long[] value = { 10121416, 12100806 };
        ede.addChildWithLongArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_LONG_ARRAY_VALUE);
        long[] outValue = tmpEde.getValuesAsLongArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithLongValue() {
        long value = 123456789;
        ede.addChildWithLongValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_LONG_VALUE);
        assertTrue(tmpEde.getValuesAsLong() == value);
    }

    public void testAddChildWithShortArrayValue() {
        short[] value = { 10, 12 };
        ede.addChildWithShortArrayValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_SHORT_ARRAY_VALUE);
        short[] outValue = tmpEde.getValuesAsShortArray();
        assertTrue(value.length == outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertTrue(outValue[i] == value[i]);
        }
    }

    public void testAddChildWithShortValue() {
        short value = 4;
        ede.addChildWithShortValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_SHORT_VALUE);
        assertTrue(tmpEde.getValuesAsShort() == value);
    }

    /**
     * Test for ExtendedDataElement addExtendedDataElement(String, int, String)
     */
    public void testAddExtendedDataElementGenericWithHexBinary() throws ValidationException {
        String hexValue = "abc123";
        ExtendedDataElement child = ede.addChild("AA", ExtendedDataElement.TYPE_HEX_BINARY_VALUE, hexValue);
        assertEquals(1, ede.getChildren().size());

        ExtendedDataElement child1 = (ExtendedDataElement) ede.getChildren().get(0);
        assertEquals("hexBinary", child1.getType());
        assertEquals(child, child1);

        assertEquals(hexValue, child.getHexValue());
    }

    public void testAddChildWithHexValue() {
        String value = "hexValue";
        ede.addChildWithHexValue("child 1", value);
        EList children = ede.getChildren();
        assertTrue(children.size() == 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertTrue(tmpEde.getTypeAsInt() == ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
        assertTrue(tmpEde.getHexValue().equals(value));
    }

    public void testAddChildWithHexValueAsByteArray() {
        byte[] value = { 98, 84 };
        ede.addChildWithHexValueAsByteArray("child 1", value);
        EList children = ede.getChildren();
        assertEquals("Wrong size", children.size(), 1);

        ExtendedDataElement tmpEde = (ExtendedDataElement) children.get(0);
        assertEquals(tmpEde.getTypeAsInt(), ExtendedDataElement.TYPE_HEX_BINARY_VALUE);
        byte[] outValue = tmpEde.getHexValueAsByteArray();
        assertEquals("Wrong length", value.length, outValue.length);

        for (int i = 0; i < value.length; i++) {
            assertEquals("Wrong value", value[i], outValue[i]);
        }
    }

    /**
     * Tests whether the model has been generated in a way that 
     * allows duplicate values inside a list.
     */
    public void testDuplicateValues() {
        EventFactory eFx = EventFactory.eINSTANCE;
        ExtendedDataElement ede = eFx.createExtendedDataElement();
        ede.getValues().add("1");
        ede.getValues().add("1");
        assertEquals(ede.getValues().size(), 2);
    }

    /**
     * Tests to make sure no one breaks the array of ExtendedDataElement type values 
     */
    public void testEDE_TYPE_VALUES() {
        assertEquals(ExtendedDataElement.TYPE_NO_VALUE, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_NO_VALUE_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_BYTE, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_BYTE_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_SHORT, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_SHORT_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_INT, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_INT_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_LONG, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_LONG_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_FLOAT, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_FLOAT_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_DOUBLE, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_DOUBLE_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_STRING, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_STRING_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_DATE_TIME_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_BOOLEAN, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_BOOLEAN_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_BYTE_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_BYTE_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_SHORT_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_SHORT_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_INT_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_INT_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_LONG_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_LONG_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_FLOAT_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_FLOAT_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_DOUBLE_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_DOUBLE_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_STRING_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_STRING_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_DATE_TIME_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_BOOLEAN_ARRAY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_BOOLEAN_ARRAY_VALUE]));
        assertEquals(ExtendedDataElement.TYPE_HEX_BINARY, ((String) ExtendedDataElementImpl.EDE_TYPES[ExtendedDataElement.TYPE_HEX_BINARY_VALUE]));

        assertEquals(ExtendedDataElement.TYPE_NO_VALUE, "noValue");
        assertEquals(ExtendedDataElement.TYPE_BYTE, "byte");
        assertEquals(ExtendedDataElement.TYPE_SHORT, "short");
        assertEquals(ExtendedDataElement.TYPE_INT, "int");
        assertEquals(ExtendedDataElement.TYPE_LONG, "long");
        assertEquals(ExtendedDataElement.TYPE_FLOAT, "float");
        assertEquals(ExtendedDataElement.TYPE_DOUBLE, "double");
        assertEquals(ExtendedDataElement.TYPE_STRING, "string");
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME, "dateTime");
        assertEquals(ExtendedDataElement.TYPE_BOOLEAN, "boolean");
        assertEquals(ExtendedDataElement.TYPE_BYTE_ARRAY, "byteArray");
        assertEquals(ExtendedDataElement.TYPE_SHORT_ARRAY, "shortArray");
        assertEquals(ExtendedDataElement.TYPE_INT_ARRAY, "intArray");
        assertEquals(ExtendedDataElement.TYPE_LONG_ARRAY, "longArray");
        assertEquals(ExtendedDataElement.TYPE_FLOAT_ARRAY, "floatArray");
        assertEquals(ExtendedDataElement.TYPE_DOUBLE_ARRAY, "doubleArray");
        assertEquals(ExtendedDataElement.TYPE_STRING_ARRAY, "stringArray");
        assertEquals(ExtendedDataElement.TYPE_DATE_TIME_ARRAY, "dateTimeArray");
        assertEquals(ExtendedDataElement.TYPE_BOOLEAN_ARRAY, "booleanArray");
        assertEquals(ExtendedDataElement.TYPE_HEX_BINARY, "hexBinary");
    }

    public void testToString() {
        assertNotNull("ede with soma data filled in", ede.toString());
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        assertNotNull("empty ede", ede.toString());
    }

    public void testEDETypeConversion() throws ValidationException {
        // ExtendedDataElement.TYPE_NO_VALUE
        ede.setName("test me");
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        ede.validate();

        // ExtendedDataElement.TYPE_BYTE 
        byte inbyte = 'c';
        ede.setValuesAsByte(inbyte);
        ede.setType(ExtendedDataElement.TYPE_BYTE);
        ede.validate();
        ede.setValuesAsString("c");
        ede.setType(ExtendedDataElement.TYPE_BYTE);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        int[] myByteArray = { 'c', 'd', 'e' };
        ede.setValuesAsIntArray(myByteArray);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        // first validate length
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }
        // then validate the value/type conversion
        ede.setValuesAsDouble(5.5);
        ede.setType(ExtendedDataElement.TYPE_INT);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_SHORT
        ede.setValuesAsString("123");
        ede.setType(ExtendedDataElement.TYPE_SHORT);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_SHORT);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        ede.setValuesAsString("987654321");
        ede.setType(ExtendedDataElement.TYPE_SHORT);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        short[] myShortArray = { 567, 123, 3 };
        ede.setValuesAsShortArray(myShortArray);
        ede.setType(ExtendedDataElement.TYPE_SHORT);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_INT        
        ede.setValuesAsString("123");
        ede.setType(ExtendedDataElement.TYPE_INT);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_INT);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        int[] myIntArray = { 567, 123, 3 };
        ede.setValuesAsIntArray(myIntArray);
        ede.setType(ExtendedDataElement.TYPE_INT);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_LONG
        long inLong = (new Long("1067528539452")).longValue();
        String inLongStr = Long.toString(inLong);
        ede.setValuesAsString(inLongStr);
        ede.setType(ExtendedDataElement.TYPE_LONG);
        ede.validate();
        ede.setValuesAsLong(inLong);
        ede.setType(ExtendedDataElement.TYPE_LONG);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_LONG);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        long[] myLongArray = { 123, 456 };
        ede.setValuesAsLongArray(myLongArray);
        ede.setType(ExtendedDataElement.TYPE_LONG);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_FLOAT
        float inFloat = Float.intBitsToFloat(0x7f7fffff);
        String inFloatStr = Float.toString(inFloat);
        ede.setValuesAsString(inFloatStr);
        ede.setType(ExtendedDataElement.TYPE_FLOAT);
        ede.validate();
        ede.setValuesAsFloat(inFloat);
        ede.setType(ExtendedDataElement.TYPE_FLOAT);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_FLOAT);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        float[] myFloatArray = { 567, 123 };
        ede.setValuesAsFloatArray(myFloatArray);
        ede.setType(ExtendedDataElement.TYPE_FLOAT);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_DOUBLE
        double inDouble = Double.longBitsToDouble(0x7fefffffffffffffL);
        String inDoubleStr = Double.toString(inDouble);
        ede.setValuesAsString(inDoubleStr);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE);
        ede.validate();
        ede.setValuesAsDouble(inDouble);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_DOUBLE);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
        double[] myDoubleArray = { 567, 123 };
        ede.setValuesAsDoubleArray(myDoubleArray);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_STRING
        ede.setValuesAsString("my string");
        ede.setType(ExtendedDataElement.TYPE_STRING);
        ede.validate();
        String instr = Util.getPaddedString(1025);
        ede.setValuesAsString(instr);
        ede.setType(ExtendedDataElement.TYPE_STRING);
        try {
            ede.validate();
            fail(ede.getValues() + " elements exceed the maximum length");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0201E", ve);
        }
        ede.setValuesAsString(null);
        ede.setType(ExtendedDataElement.TYPE_STRING);
        try {
            ede.validate();
            fail(ede.getType() + " cannot has a null value");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0194E", ve);
        }
        String[] myStringArray = { "aaa", "bbb" };
        ede.setValuesAsStringArray(myStringArray);
        ede.setType(ExtendedDataElement.TYPE_STRING);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_DATE_TIME
        long inDate = (new Long("1067528539452")).longValue();
        ede.setValuesAsDateAsLong(inDate);
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME);
        ede.validate();
        ede.setValuesAsString("2003-10-30T10:42:19.452-05:00");
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME);
        ede.validate();
        ede.setValuesAsString("xyz");
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0196E", ve);
        }
        long[] myDateArray = { 987654321, 998877665 };
        ede.setValuesAsDateAsLongArray(myDateArray);
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_BOOLEAN
        boolean inBool = true;
        ede.setValuesAsBoolean(inBool);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsBoolean(false);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsString("true");
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsString("false");
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsString("0");
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsString("1");
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.validate();
        ede.setValuesAsString("TRUE");
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0195E", ve);
        }
        boolean[] myBoolArray = { true, false };
        ede.setValuesAsBooleanArray(myBoolArray);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        try {
            ede.validate();
            fail(ede.getType() + " requires a single value : " + ede.getValues());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0191E", ve);
        }

        // ExtendedDataElement.TYPE_BYTE_ARRAY
        byte[] inByteArray = { 1, 'c', 3 };
        String[] inByteArrayStr = { "1", "2", "3" };
        String[] inByteArrayStrBad = { "1", "2", "z" };
        ede.setValuesAsByteArray(inByteArray);
        ede.setType(ExtendedDataElement.TYPE_BYTE_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inByteArrayStr);
        ede.setType(ExtendedDataElement.TYPE_BYTE_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inByteArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_BYTE_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_SHORT_ARRAY
        short[] inShortArray = { 1, 2, 3 };
        String[] inShortArrayStr = { "1", "2", "3" };
        String[] inShortArrayStrBad = { "z", "2", "3" };
        ede.setValuesAsShortArray(inShortArray);
        ede.setType(ExtendedDataElement.TYPE_SHORT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inShortArrayStr);
        ede.setType(ExtendedDataElement.TYPE_SHORT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inShortArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_SHORT_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_INT_ARRAY
        int[] inIntArray = { 1, 2, 3 };
        String[] inIntArrayStr = { "1", "2", "3" };
        String[] inIntArrayStrBad = { "1", "2", "z" };
        ede.setValuesAsIntArray(inIntArray);
        ede.setType(ExtendedDataElement.TYPE_INT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inIntArrayStr);
        ede.setType(ExtendedDataElement.TYPE_INT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inIntArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_INT_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_LONG_ARRAY
        long[] inLongArray = { 1, 2, 3 };
        String[] inLongArrayStr = { "1", "2", "3" };
        String[] inLongArrayStrBad = { "1", "2", "z" };
        ede.setValuesAsLongArray(inLongArray);
        ede.setType(ExtendedDataElement.TYPE_LONG_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inLongArrayStr);
        ede.setType(ExtendedDataElement.TYPE_LONG_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inLongArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_LONG_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_FLOAT_ARRAY
        float[] inFloatArray = { 1, 2, 3 };
        String[] inFloatArrayStr = { "1", "2", "3" };
        String[] inFloatArrayStrBad = { "1", "2", "z" };
        ede.setValuesAsFloatArray(inFloatArray);
        ede.setType(ExtendedDataElement.TYPE_FLOAT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inFloatArrayStr);
        ede.setType(ExtendedDataElement.TYPE_FLOAT_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inFloatArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_FLOAT_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_DOUBLE_ARRAY
        double[] inDoubleArray = { 1, 2, 3 };
        String[] inDoubleArrayStr = { "1", "2", "3" };
        String[] inDoubleArrayStrBad = { "1", "2", "z" };
        ede.setValuesAsDoubleArray(inDoubleArray);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inDoubleArrayStr);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inDoubleArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_DOUBLE_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // ExtendedDataElement.TYPE_STRING_ARRAY
        String[] inStrArray = { "1", "2", "3" };
        ede.setValuesAsStringArray(inStrArray);
        ede.setType(ExtendedDataElement.TYPE_STRING_ARRAY);
        ede.validate();
        String myAlphaStr = Util.getAlphaString(1025);
        String[] inStrArrayBad = { "1", myAlphaStr, "3" };
        ede.setValuesAsStringArray(inStrArrayBad);
        ede.setType(ExtendedDataElement.TYPE_STRING_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " elements exceed the maximum length");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0201E", ve);
        }

        // ExtendedDataElement.TYPE_DATE_TIME_ARRAY
        long[] inDateArray = { 987654321, 12345678, 33333333 };
        String[] inDateArrayStr = { "2003-10-30T10:42:19.452-05:00", "2003-10-30T10:42:19.452-05:00" };
        String[] inDateArrayStrBad = { "2003-10-30T10:42:19.452-05:00", "xxx" };
        ede.setValuesAsDateAsLongArray(inDateArray);
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inDateArrayStr);
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inDateArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0196E", ve);
        }

        // ExtendedDataElement.TYPE_BOOLEAN_ARRAY
        boolean[] inBoolArray = { true, false, false };
        String[] inBoolArrayStr = { "1", "0", "false" };
        String[] inBoolArrayStrBad = { "true", "x", "false" };
        ede.setValuesAsBooleanArray(inBoolArray);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inBoolArrayStr);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN_ARRAY);
        ede.validate();
        ede.setValuesAsStringArray(inBoolArrayStrBad);
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN_ARRAY);
        try {
            ede.validate();
            fail(ede.getValues() + " is not a valid " + ede.getType());
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0195E", ve);
        }

        // ExtendedDataElement.TYPE_HEX_BINARY 
    }

    /**
     * Tests the <code>equals</code> operator.
     */
    public void testEquals() {
        ExtendedDataElement ede2 = (ExtendedDataElement) Util.clone(ede);
        assertEquals(ede, ede2);

        ede2.setType("difference");
        assertFalse(ede.equals(ede2));
    }

    public void testEquals1() {
        ExtendedDataElement ede2 = (ExtendedDataElement) Util.clone(ede);
        assertEquals(ede, ede2);

        ede.addChild("Megan");
        ede2.addChild("Alex");
        assertFalse(ede.equals(ede2));
    }

    public void testEquals2() {
        ExtendedDataElement ede2 = (ExtendedDataElement) Util.clone(ede);

        ExtendedDataElement c1 = ede.addChild("Megan");
        ExtendedDataElement c2 = (ExtendedDataElement) Util.clone(c1);
        c1.addChild("Wendy");
        c2.addChild("Alex");
        assertFalse(ede.equals(ede2));
    }

    /**
     * Tests for missing required attributes.
     */
    public void testValidationMissingAttributes() throws ValidationException {
        ede = EventFactory.eINSTANCE.createExtendedDataElement();

        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        ede.validate();

        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        Util.assertMissingAttributes(ede, "name");

        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        Util.assertMissingAttributes(ede, "type");

        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_STRING);
        Util.assertMissingAttributes(ede, "value");

        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        Util.assertMissingAttributes(ede, "hexValue");

        // test have value but type is set to hexBinary
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        boolean inBool = false;
        ede.setValuesAsBoolean(inBool);
        ede.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        Util.assertMissingAttributes(ede, "hexValue");

        // have hexValue but type is not set to hexBinary
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        byte[] inHex = { 0x23, 0x54, 0x78 };
        ede.setHexValueAsByteArray(inHex);
        ede.setType(ExtendedDataElement.TYPE_INT);
        Util.assertMissingAttributes(ede, "value");

        // make sure we can validate a hexBinary
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        byte[] inHex1 = { 0x23, 0x54, 0x78 };
        ede.setHexValueAsByteArray(inHex1);
        ede.validate();

        // make sure we can validate a value
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        boolean inBool1 = false;
        ede.setValuesAsBoolean(inBool1);
        ede.validate();
    }

    /**
     * Tests for invalid length attributes.
     */
    public void testValidationInvalidLengths() throws ValidationException {
        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        ede.validate();

        ede.setName(Util.getPaddedString(65));
        Util.assertInvalidBoundaries(ede, "name");
        ede.setName(Util.getPaddedString(64));
        Util.assertValidBoundaries(ede);

        ede.setType(ExtendedDataElement.TYPE_STRING);
        String str1 = Util.getPaddedString(1024);
        String str2 = Util.getPaddedString(1024);
        String str3 = Util.getPaddedString(1024);
        String[] inValue = { str1, str2, str3 };
        ede.setValuesAsStringArray(inValue);
        Util.assertValidBoundaries(ede);
        String str4 = Util.getPaddedString(1025);
        String[] inValue1 = { str1, str4, str3 };
        ede.setValuesAsStringArray(inValue1);
        try {
            ede.validate();
            fail("fourth value too long");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0201E", ve);
        }

        // There is no way to test that type has a max length of 64 since
        // only 20 specific types are allowed.

        // Long hexValue values are tested here: testHexValueStringConversion()
    }

    /**
     * Tests type , values and hexValues attributes.
     */
    public void testValidationTypeAndValues() throws ValidationException {
        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        ede.validate();

        ede.setTypeAsInt(123);
        ede.validate();
        assertEquals(ede.getType(), ExtendedDataElement.TYPE_NO_VALUE);

        ede.setType("My Type");
        try {
            ede.validate();
            fail("invalid type");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0193E", exc);
        }

        ede.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        ede.setHexValue("1A234");
        try {
            ede.validate();
            fail("invalid hexValue - odd number of characters");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0197E", exc);
        }

        ede.setHexValue("ABCDEFGH");
        try {
            ede.validate();
            fail("invalid hexValue - invalid hexadecimal character");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0198E", exc);
        }

        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        try {
            ede.validate();
            fail("missing value attribute for type boolean ");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0203E", exc);
        }

        ede.eUnset(EventPackage.eINSTANCE.getExtendedDataElement_HexValue());
        ede.setValuesAsStringArray(new String[] { "abc", "def" });
        ede.setType(ExtendedDataElement.TYPE_STRING);
        try {
            ede.validate();
            fail("value mismatch for type string");
        }
        catch (ValidationException exc) {
            Util.assertExceptionId("IWAT0191E", exc);
        }

    }

    /**
     * Tests for mutually exclusive attributes.
     */
    public void testValidationMutuallyExclusive() throws ValidationException {
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        byte[] inHex = { 0x23, 0x54, 0x78 };
        boolean inBool = false;
        ede.setValuesAsBoolean(inBool);
        ede.setHexValueAsByteArray(inHex);
        try {
            ede.validate();
            fail("can't have both value and hexValue");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0204E", ve);
        }
    }

    /**
     * 
     */
    public void testValidateGoodDateTime() throws ValidationException {
        ede.setType(ExtendedDataElement.TYPE_DATE_TIME);
        ede.setName("name");
        ede.setValuesAsDateAsLong(System.currentTimeMillis());
        ede.validate();
    }

    /**
     * 
     */
    public void testValidateGoodBoolean() throws ValidationException {
        ede.setType(ExtendedDataElement.TYPE_BOOLEAN);
        ede.setName("name");
        ede.setValuesAsBoolean(false);
        ede.validate();
    }

    /**
     * Validate having lots of children.
     */
    public void testValidationChildren() throws ValidationException {
        ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(Util.getPaddedString(64));
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);
        ede.validate();

        ExtendedDataElement ede1 = ede.addChild("Megan");
        ede1.setType(ExtendedDataElement.TYPE_NO_VALUE);
        String[] values = { "test1", "test2" };
        ExtendedDataElement ede2 = ede.addChild("Alex", ExtendedDataElement.TYPE_STRING_ARRAY_VALUE, values);
        ExtendedDataElement ede3 = ede.addChild("Wendy", ExtendedDataElement.TYPE_LONG_VALUE, "123");
        ede.validate();

        // Make one of the children invalid
        ede3.setValuesAsString("xyz");
        ede3.setType(ExtendedDataElement.TYPE_INT);
        try {
            ede.validate();
            fail("child 3 should be invalid");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }

        // fix child 3 and add two children to child 2
        ede3.setValuesAsInt(123);
        ExtendedDataElement ede2_1 = ede2.addChild("my first grandchild", ExtendedDataElement.TYPE_FLOAT_VALUE, "123");
        ExtendedDataElement ede2_2 = ede2.addChild("my second grandchild", ExtendedDataElement.TYPE_BOOLEAN_VALUE, "false");
        ede.validate();

        // make a grandchild invalid
        ede2_2.setValuesAsString("xyz");
        ede2_2.setType(ExtendedDataElement.TYPE_FLOAT);
        try {
            ede.validate();
            fail("grandchild should be invalid");
        }
        catch (ValidationException ve) {
            Util.assertExceptionId("IWAT0192E", ve);
        }
    }

    /**
     * 
     */
    public void testAdapter() {
        Util.assertAdapterNotification(ede);
    }

    /**
     * 
     */
    public void testUnsupportedFeature() {
        Util.assertUnsupportedFeature(ede);
    }
}
