package org.eclipse.hyades.logging.events.cbe.util;

import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.logging.events.cbe.MissingPropertyException;
import org.eclipse.hyades.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2004 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Helper methods for the several <code>validate</code> methods spread across
 * the EMF generated classes.
 * 
 * @author Andrea Chiavegatto
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @author Scott Brown
 * @version 1.0.1
 * @since 1.0.1
 */
public class EventValidation {

    /**
     * Returns the complete name of a component in the EMF model.
     * 
     * @param component      component for the name retrieval
     * @param name           feature name
     * 
     * @return the complete name of the feature name passed as a parameter.
     */
    public static String getComponentName(EObject component, String name) {
        String componentName = new String(name);
        EObject parent = component;
        while (parent != null) {
            componentName = parent.eClass().getName() + "." + componentName;
            parent = parent.eContainer();
        }
        return componentName;
    }

    /**
     * Determines whether one of the required attributes for a given CBE
     * element is not set to <code>null</code>.
     *  
     * @param elementInstance
     *                       target element for verification.
     * 
     * @throws ValidationException
     *                       if one of the required attributes is <code>null</code>.
     */
    public static void validateRequiredProperties(String className, EObject elementInstance) throws ValidationException {
    
        EClass eClass = elementInstance.eClass();
        List attrs = eClass.getEAllAttributes();
        List containments = eClass.getEAllContainments();

        for (int i = 0; i < attrs.size(); i++) {

            EAttribute attr = (EAttribute) attrs.get(i);
            
            if ((attr.isRequired()) && (!elementInstance.eIsSet(attr))) {
                throw new MissingPropertyException(EventHelpers.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_",EventValidation.getComponentName(elementInstance, attr.getName())));
            }
        } 

        for (int i = 0; i < containments.size(); i++) {
        
            EReference containement = (EReference) containments.get(i);

            if ((containement.isRequired()) && (!elementInstance.eIsSet(containement))) {
                throw new MissingPropertyException(EventHelpers.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_",EventValidation.getComponentName(elementInstance, containement.getName())));
            }
        }
    }

    /**
     * Checks whether a string conforms to the ID type
     */
    public static void validateId(String classname, EObject o, EStructuralFeature att, String id) throws ValidationException {

        if (id != null) {
        
            if ((id.length() < 32) || (id.length() > 64)) {
                throw new ValidationException(EventHelpers.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_LENGTH_EXC_",EventValidation.getComponentName(o, att.getName()), id));
            }

            // a GlobalInstanceId must start with a-z, A-Z
            char[] gidchar = id.toCharArray();
            
            if (!(((gidchar[0] >= 'a') && (gidchar[0] <= 'z')) || ((gidchar[0] >= 'A') && (gidchar[0] <= 'Z')))) {
                throw new ValidationException(EventHelpers.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_EXC_",EventValidation.getComponentName(o, att.getName()), id));
            }
        }
    }

    /**
     * Checks whether a string conforms to the ID type
     */
    public static void validateId(String classname, EObject o, EStructuralFeature att) throws ValidationException {
        validateId(classname, o, att, (String) o.eGet(att));
    }

    /**
     * Checks whether the elements of the list conforms to the ID type
     */
    public static void validateStringIdList(String classname, EObject o, EStructuralFeature att) throws ValidationException {
        StringTokenizer idSt = new StringTokenizer((String) o.eGet(att), " ");
        while (idSt.hasMoreTokens()) {
            validateId(classname, o, att, idSt.nextToken());
        }
    }

    /**
     * Checks whether a string conforms to the maximum length
     */
    public static void validateLength(String classname, EObject o, EStructuralFeature att, int minLength, int maxLength) throws ValidationException {

        String str = (String) o.eGet(att);
        
        if ((str != null) && ((str.length() < minLength) || (str.length() > maxLength))) {

            if (minLength == 0) {
                throw new ValidationException(EventHelpers.getString("LOG_INVALID_LENGTH_EXC_",EventValidation.getComponentName(o, att.getName()), str, new Integer(str.length()), new Integer(maxLength)));
            }
            else {
                throw new ValidationException(EventHelpers.getString("LOG_INVALID_LENGTH_RANGE_EXC_",EventValidation.getComponentName(o, att.getName()), str, new Integer(str.length()), (minLength + " - " + maxLength)));
            }
        }
    }

    /**
     * Checks whether a string conforms to the maximum length
     */
    public static void validateLength(String classname, EObject o, EStructuralFeature att, int maxLength) throws ValidationException {
        validateLength(classname, o, att, 0, maxLength);
    }

    /**
    * Checks whether two attributes are mutually exclusive
    * @param att1 is a basic object
    * @param att2 is a basic object
    */
    public static void validateMutuallyExclusiveAttributes(String classname, EObject o, EStructuralFeature att1, EStructuralFeature att2) throws ValidationException {
        
        if (((o.eGet(att1) == null) && (o.eGet(att2) == null)) || ((o.eGet(att1) != null) && (o.eGet(att2) != null))) {
            throw new ValidationException(EventHelpers.getString("LOG_MUTUALLY_EXCLUSIVE_ATTRIBUTES_EXC_", EventValidation.getComponentName(o, att1.getName()), EventValidation.getComponentName(o, att2.getName())));
        }
    }

    /**
     * We support version 1.0.1 or null
     */
    public static void validateVersion(String classname, EObject o, EStructuralFeature att) throws ValidationException {
        
        String myStr = (String) o.eGet(att);

        if ((myStr != null) && (!myStr.equals("1.0.1"))) {
            throw new ValidationException(EventHelpers.getString("LOG_INVALID_VERSION_EXC_", EventValidation.getComponentName(o, att.getName()), myStr));
        }
    }

    /**
     * Logs a message and throws a validation exception.
     * 
     * @param className      class name of the method from which the exception 
     *                       was thrown.
     * @param resourceKey    key in the resource bundle.
     * @param params         parameters for the message.
     * 
     * @throws ValidationException
     *                       containing the message associated with the 
     *                       validation failure.
     * @deprecated As of 3.1, use instantiate a new ValidationException with a localized message using EventHelpers#getString(String, Object[]).
     */
    public static void throwValidationException(String className, String resourceKey, Object[] params) throws ValidationException {
        throw new ValidationException(EventHelpers.getString(resourceKey, params));
    }

    /**
     * Logs a message and throws a validation exception.
     * 
     * @param className      class name of the method from which the exception 
     *                       was thrown.
     * @param resourceKey    key in the resource bundle.
     * @param params         parameters for the message.
     * 
     * @throws ValidationException
     *                       containing the message associated with the 
     *                       validation failure.
     * @deprecated As of 3.1, use instantiate a new MissingPropertyException with a localized message using EventHelpers#getString(String, Object[]).
     */
    public static void throwMissingPropertyException(String className, String resourceKey, Object[] params) throws ValidationException {
        throw new MissingPropertyException(EventHelpers.getString(resourceKey, params));
    }
}
