/**********************************************************************
 * Copyright (c) 2003 Hyades project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v0.5
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v05.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;
import java.io.IOException;
import java.io.InputStream;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
/**
 * @author slavescu
 */
/**
 * This class wraps an XMLLoader with a SAX DefaultHandler.
 */
public class SAXFragmentHandler extends DefaultHandler implements XMLFragmentHandler {
   //~ Instance fields
   // ----------------------------------------------------------------------------
   protected InputSource inputSource = null;
   protected ParserPipedInputStream xmlStream = null;
   protected SAXParser parser;
   protected Thread parseThread;
   protected IXMLLoader handler;
   protected int depth;
   protected static final int WAIT_TIME = 500;
   protected byte[] rootTag;
   protected volatile boolean rootTagRequired;
   public void setXMLLoader(IXMLLoader loader)
   {
   	handler=loader;
   	init();
   }
   public void init() {
      try {
         initRestart();
         depth = -1;
         rootTagRequired = false;
         parser = makeParser();
         xmlStream = new ParserPipedInputStream(this);
         parseThread = new Thread(Thread.currentThread().getThreadGroup(), "xmlParserThread") {
            public void run() {
                  do {
                     if (xmlStream == null) {
                        return;
                     }
                     //LoadersUtils.log("parse ++++++");
                     //LoadersUtils.log(new Exception());
                     parse();
                  } while (restart);
                  xmlStream = null;
                  //LoadersUtils.log("restart=" + restart);
                  //LoadersUtils.log("rootTagRequired=" + rootTagRequired);
                  //LoadersUtils.log("parse ------");
            }
         };
         parseThread.start();
         //LoadersUtils.log("init");
      } catch (Exception e) {
         LoadersUtils.log(e);
      }
      //      this.handler = handler;
   }
   //~ Methods
   // ------------------------------------------------------------------------------------
   public void setDocumentLocator(Locator locator) {
      // empty method
      //LoadersUtils.log("setDocumentLocator(Locator)");
   }
   public void characters(char[] ch, int start, int length) throws SAXException {
      //LoadersUtils.log("characters=\n" + new String(ch, start, length));
      handler.characters(ch, start, length);
   }
   public void endDocument() throws SAXException {
//      LoadersUtils.log("endDocument");
      restart = false;
      if (depth == 0 && xmlStream != null)
      {
         xmlStream.makeClosed();
         xmlStream = null;
      }
      depth=-1;
      handler.endDocument(null, 0);
   }
   public void endElement(String uri, String localName, String qName) throws SAXException {
//      LoadersUtils.log("endElement=" + qName);
      handler.endElement(qName, 0);
      depth--;
      if (depth == 0)
      {
      	 if(xmlStream != null)
      	 {
	         //LoadersUtils.log("endElement-makeClosed=" + qName);
	         xmlStream.makeClosed();
	         xmlStream = null;
      	 }
         restart = false;
      }
      
   }
   public void endPrefixMapping(String prefix) throws SAXException {
      //      // handler.endPrefixMapping(prefix);
      //LoadersUtils.log("endPrefixMapping(String)");
   }
   public void error(SAXParseException e) throws SAXException {
      //        terminateParser();
      //LoadersUtils.log("error(SAXParseException)");
      //LoadersUtils.log(e);
      throw e;
   }
   public void fatalError(SAXParseException e) throws SAXException {
      //        terminateParser();
      //LoadersUtils.log("depth=" + depth + " rootTagRequired=" + rootTagRequired);
      synchronized(this)
      {
         if (restart && depth == 1 ) {
            depth = -1;
            rootTagRequired = true;
            try {
//                LoadersUtils.log("depth=" + depth + " rootTagRequired=" + rootTagRequired);
               //            rootTagRequired = true;
               //LoadersUtils.log("rootTagRequired=" + rootTagRequired);
               //               terminateParser();
               parser = makeParser();
               if(xmlStream==null)
               		xmlStream = new ParserPipedInputStream(this);
               xmlStream.makeOpened();
            } catch (ParserConfigurationException e1) {
            	LoadersUtils.log(e1);
            	throw new InvalidXMLException(e1);
            } catch (SAXException e1) {
            	LoadersUtils.log(e1);
            	throw new InvalidXMLException(e1);
            }
            return;
         }
      }
      //LoadersUtils.log("fatalError(SAXParseException)");
      //LoadersUtils.log(e);
      throw e;
   }
   public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
      // // handler.ignorableWhitespace(ch, start, length);
      //LoadersUtils.log("ignorableWhitespace(char[], int, int)");
   }
   public void notationDecl(String name, String publicId, String systemId) throws SAXException {
      // // handler.notationDecl(name, publicId, systemId);
      //LoadersUtils.log("notationDecl(String, String, String)");
   }
   public void processingInstruction(String target, String data) throws SAXException {
      // // handler.processingInstruction(target, data);
      //LoadersUtils.log("processingInstruction(String, String)");
   }
   public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
      // // handler.resolveEntity(publicId, systemId);
      //LoadersUtils.log("resolveEntity(String, String)");
      return null;
   }
   public void scanContent(byte[] fragment, int offset, int length) throws InvalidXMLException {
      //LoadersUtils.log("scanContent:\n"+ LoadersUtils.makeString(fragment, offset, length));
   	  //LoadersUtils.log("scanContent(length)"+ length);
      if (xmlStream != null) {
         if (restart && rootTagRequired && rootTag!=null) {
            //LoadersUtils.log("scanContent1 - rootTagRequired=" + rootTagRequired);
            xmlStream.writeBuf(rootTag, 0, rootTag.length);
            rootTagRequired=false;
//            LoadersUtils.log("scanContent2 - rootTagRequired=" + rootTagRequired);
         }
         xmlStream.writeBuf(fragment, offset, length);
      }
   }
   public void skippedEntity(String name) throws SAXException {
      // // handler.skippedEntity(name);
      //LoadersUtils.log("skippedEntity(String)");
   }
   public void startDocument() throws SAXException {
      depth = 0;
      //LoadersUtils.log("startDocument");
      handler.startDocument();
   }
   public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
      depth++;
      if (restart && depth == 1) {
         if(rootTag==null)
         {
            rootTag = ("<"+qName+">").getBytes();
         }
      }
//      LoadersUtils.log("startElement=" + qName + " depth="+depth);
      handler.startElement(qName, false, attributes.getLength() == 0);
      for (int i = 0; i < attributes.getLength(); i++) {
         handler.attributeName(attributes.getQName(i));
         handler.attributeValueCharacters(attributes.getValue(i));
      }
   }
   public void startPrefixMapping(String prefix, String uri) throws SAXException {
      //      // handler.startPrefixMapping(prefix, uri);
      //LoadersUtils.log("startPrefixMapping(String, String)");
   }
   public void terminateParser() {
   	  restart = false;
   	  depth=-1;
      if (xmlStream != null) {
         xmlStream.makeClosed();
         xmlStream = null;
      }
      synchronized(this)  {
      	notifyAll();
        parseThread.interrupt();
	  }
   }
   public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName) throws SAXException {
      // // handler.unparsedEntityDecl(name, publicId, systemId,
      // notationName);
      //LoadersUtils.log("unparsedEntityDecl(String, String, String, String)");
   }
   public void warning(SAXParseException e) throws SAXException {
      //      // handler.warning(xmi);
      //LoadersUtils.log("warning(SAXParseException)");
      //LoadersUtils.log(e);
   }
   /**
    * Make either a validating or non-validating parser; throw an if one could
    * not be made.
    */
   protected SAXParser makeParser() throws ParserConfigurationException, SAXException {
      //LoadersUtils.log("makeParser()");
      SAXParserFactory factory = SAXParserFactory.newInstance();
//      depth = -1;
//      rootTagRequired = false;
//      SAXParserFactoryImpl factory = new SAXParserFactoryImpl();
//      factory.setNamespaceAware(true);
      factory.setFeature("http://xml.org/sax/features/string-interning", true); //$NON-NLS-1$ 
      factory.setValidating(false);
      return factory.newSAXParser();
   }
   protected synchronized void parse() throws InvalidXMLException {
      try {
         inputSource = new InputSource(xmlStream);
         parser.parse(inputSource, this);
      } catch (Exception e) {
      	 //LoadersUtils.log(e);
      	 if(restart)
      	 {
      	 	return;
      	 }
         handler.error(new InvalidXMLException(e));
      }
   }
   //~ Inner Classes
   // ------------------------------------------------------------------------------
   public static class ParserPipedInputStream extends InputStream {
      private byte[] inBuf;
      private boolean closed = false;
      private int inCount;
      private long inTotalLength;
      private long inPartLength;
      private int inPos;
      private SAXFragmentHandler handler;
      /**
       * Constructor MyPipedInputStream.
       * 
       * @param buf
       */
      public ParserPipedInputStream(SAXFragmentHandler handler) {
         this.handler = handler;
         inTotalLength = 0;
         inPartLength = 0;
         inCount = 0;
         inPos = 0;
      }
      /**
       *  
       */
      protected synchronized void makeOpened() {
         inPartLength = 0;
         inBuf = null;
         inCount = 0;
         inPos = 0;
         closed = false;
         notifyAll();
      }
      /**
       * @see java.io.InputStream#available()
       */
      public synchronized int available() throws IOException {
         waitForNewData();
         return inCount;
      }
      /*
       * (non-Javadoc)
       * 
       * @see java.io.InputStream#close()
       */
      public void close() throws IOException {
         super.close();
         closed = true;
         //            notifyClose();
      }
      public synchronized boolean hasEmptyBuffer() {
         return inCount == 0;
      }
      /*
       * (non-Javadoc)
       * 
       * @see java.io.InputStream#mark(int)
       */
      public synchronized void mark(int arg0) {
         super.mark(arg0);
      }
      /**
       * @see java.io.InputStream#markSupported()
       */
      public boolean markSupported() {
         return false;
      }
      /**
       * @see java.io.InputStream#read()
       */
      public synchronized int read() throws IOException {
         if (available() == 0) {
            return -1;
         } else {
            inCount--;
            return inBuf[inPos++];
         }
      }
      /**
       * @see java.io.InputStream#read(byte[], int, int)
       */
      public synchronized int read(byte[] outBuf, int offset, int length) throws IOException {
         if (available() == 0) {
            return -1;
         }
         int readBytes = Math.min(inCount, length);
         System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);
         inPos = inPos + readBytes;
         inCount = inCount - readBytes;
         //LoadersUtils.log("read outBuf: " + LoadersUtils.makeString(outBuf, offset, readBytes));
         return readBytes;
      }
      /**
       * @see java.io.InputStream#read(byte[])
       */
      public int read(byte[] arg0) throws IOException {
         return read(arg0, 0, arg0.length);
      }
      public synchronized void reset() throws IOException {
         super.reset();
      }
      /**
       * @see java.io.InputStream#skip(long)
       */
      public synchronized long skip(long arg0) throws IOException {
         arg0 = arg0 + inPos;
         if (arg0 < inTotalLength) {
            inPos = (int) arg0;
            return arg0;
         } else {
            return -1;
         }
      }
      /**
       * Sets the inBuf.
       * 
       * @param inBuf
       *            The inBuf to set
       */
      public synchronized void writeBuf(byte[] buf, int offset, int length) {
         if ((buf == null) || (length == 0)) {
            return;
         }
         inBuf = buf;
         inPos = offset;
         inTotalLength = inTotalLength+length;
         inPartLength = inPartLength + length;
         inCount = length;
         
         //LoadersUtils.log("writeBuf(byte[], int, int)++++++\n" + LoadersUtils.makeString(buf, offset, length));
         //LoadersUtils.log("writeBuf(byte[], int, int)------");
//         LoadersUtils.log("writeBuf(length)" + length);

         notifyAll();
         waitForEmptyBuffer();
//         LoadersUtils.log("writeBuf(length)" + length);
      }
      /**
       *  
       */
      protected synchronized void makeClosed() {
         inPartLength = 0;
         inBuf = null;
         inCount = 0;
         inPos = 0;
         closed = true;
         notifyAll();
      }
      /**
       *  
       */
      private void waitForEmptyBuffer() {
         while (!hasEmptyBuffer() && !closed) {
            try {
               wait(WAIT_TIME);
               //LoadersUtils.log("waitForEmptyBuffer()" + LoadersUtils.makeString(inBuf, inPos, inCount));
               //LoadersUtils.log("waitForEmptyBuffer() inCount="+inCount);
            } catch (InterruptedException e) {
               Thread.currentThread().interrupt();
               if (hasEmptyBuffer() || closed) {
                  return;
               }
            }
         }
      }
      /**
       *  
       */
      private synchronized void waitForNewData() {
         boolean notify = true;
         while (hasEmptyBuffer()) {
            if (notify) {
               notifyAll();
               notify = false;
            }
            if (closed) {
               return;
            }
            try {
               wait(WAIT_TIME);
//               LoadersUtils.log("waitForNewData() handler.restart="+handler.restart+" hasEmptyBuffer()="+hasEmptyBuffer()+" inPartLength="+inPartLength+" inCount="+inCount);
               if (handler.restart && hasEmptyBuffer() && inPartLength>0) {
                  try {
                     if (handler.depth == 0) {
                        //LoadersUtils.log("close input stream");
                        makeClosed();
                        handler.rootTagRequired = true;
                        return;
                     }
                  } catch (Exception e) {
                     //LoadersUtils.log(e);
                  }
                  return;
               }
            } catch (InterruptedException e) {
               Thread.currentThread().interrupt();
               if (closed) {
                  return;
               }
            }
         }
      }
   }
   private void initRestart() {
      restart = false;
      try {
        parser = makeParser();
         if(parser.getClass().getName().equals("org.apache.crimson.jaxp.SAXParserImpl"))
            restart = true;
      } catch (ParserConfigurationException e1) {
         // TODO Auto-generated catch block
//         e1.printStackTrace();
      } catch (SAXException e1) {
         // TODO Auto-generated catch block
//         e1.printStackTrace();
      }
      //System.out.println("initRestart()-restart="+restart);
   }
   protected boolean restart;
}