/*******************************************************************************
 * This file is part of the Polyglot extensible compiler framework.
 *
 * Copyright (c) 2000-2008 Polyglot project group, Cornell University
 * Copyright (c) 2006-2008 IBM Corporation
 * All rights reserved.
 *
 * This program and the accompanying materials are made available under
 * the terms of the Eclipse Public License v1.0 which accompanies this
 * distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * This program and the accompanying materials are made available under
 * the terms of the Lesser GNU Public License v2.0 which accompanies this
 * distribution.
 * 
 * The development of the Polyglot project has been supported by a
 * number of funding sources, including DARPA Contract F30602-99-1-0533,
 * monitored by USAF Rome Laboratory, ONR Grant N00014-01-1-0968, NSF
 * Grants CNS-0208642, CNS-0430161, and CCF-0133302, an Alfred P. Sloan
 * Research Fellowship, and an Intel Research Ph.D. Fellowship.
 *
 * See README for contributors.
 ******************************************************************************/

package polyglot.types;

import polyglot.types.*;
import polyglot.util.*;
import java.util.*;

/**
 * A <code>ProcedureInstance_c</code> contains the type information for a Java
 * procedure (either a method or a constructor).
 */
public abstract class ProcedureInstance_c extends TypeObject_c
                                       implements ProcedureInstance
{
    protected ReferenceType container;
    protected Flags flags;
    protected List formalTypes;
    protected List throwTypes;

    /** Used for deserializing types. */
    protected ProcedureInstance_c() { }

    public ProcedureInstance_c(TypeSystem ts, Position pos,
			       ReferenceType container,
			       Flags flags, List formalTypes, List excTypes) {
        super(ts, pos);
	this.container = container;
	this.flags = flags;
	this.formalTypes = TypedList.copyAndCheck(formalTypes, Type.class, true);
	this.throwTypes = TypedList.copyAndCheck(excTypes, Type.class, true);
    }
    
    public ReferenceType container() {
        return container;
    }

    public Flags flags() {
        return flags;
    }

    public List formalTypes() {
        return Collections.unmodifiableList(formalTypes);
    }

    public List throwTypes() {
        return Collections.unmodifiableList(throwTypes);
    }

    /**
     * @param container The container to set.
     */
    public void setContainer(ReferenceType container) {
        this.container = container;
    }
    
    /**
     * @param flags The flags to set.
     */
    public void setFlags(Flags flags) {
        this.flags = flags;
    }
    
    /**
     * @param formalTypes The formalTypes to set.
     */
    public void setFormalTypes(List formalTypes) {
        this.formalTypes = TypedList.copyAndCheck(formalTypes, Type.class, true);
    }
    
    /**
     * @param throwTypes The throwTypes to set.
     */
    public void setThrowTypes(List throwTypes) {
        this.throwTypes = TypedList.copyAndCheck(throwTypes, Type.class, true);
    }
     
    public int hashCode() {
        return container.hashCode() + flags.hashCode();
    }

    public boolean equalsImpl(TypeObject o) {
        if (o instanceof ProcedureInstance) {
	    ProcedureInstance i = (ProcedureInstance) o;
	    // FIXME: Check excTypes too?
	    return flags.equals(i.flags())
	        && ts.hasFormals(this, i.formalTypes());
	}

	return false;
    }

    protected boolean listIsCanonical(List l) {
	for (Iterator i = l.iterator(); i.hasNext(); ) {
	    TypeObject o = (TypeObject) i.next();
	    if (! o.isCanonical()) {
		return false;
	    }
	}

	return true;
    }

    public final boolean moreSpecific(ProcedureInstance p) {
        return ts.moreSpecific(this, p);
    }


    /**
     * Returns whether <code>this</code> is <i>more specific</i> than
     * <code>p</code>, where <i>more specific</i> is defined as JLS
     * 15.12.2.2.
     *<p>
     * <b>Note:</b> There is a fair amount of guesswork since the JLS
     * does not include any info regarding Java 1.2, so all inner class
     * rules are found empirically using jikes and javac.
     */
    public boolean moreSpecificImpl(ProcedureInstance p) {
        ProcedureInstance p1 = this;
        ProcedureInstance p2 = p;

        // rule 1:
        ReferenceType t1 = null;
        ReferenceType t2 = null;
        
        if (p1 instanceof MemberInstance) {
            if (p1 instanceof Declaration) {
                t1 = ((MemberInstance) ((Declaration) p1).declaration()).container();
            }
            else {
                t1 = ((MemberInstance) p1).container();
            }
        }
        if (p2 instanceof MemberInstance) {
            if (p2 instanceof Declaration) {
                t2 = ((MemberInstance) ((Declaration) p2).declaration()).container();
            }
            else {
                t2 = ((MemberInstance) p2).container();
            }
        }
        
        if (t1 != null && t2 != null) {
            if (t1.isClass() && t2.isClass()) {
                if (! t1.isSubtype(t2) &&
                        ! t1.toClass().isEnclosed(t2.toClass())) {
                    return false;
                }
            }
            else {
                if (! t1.isSubtype(t2)) {
                    return false;
                }
            }
        }

        // rule 2:
        return p2.callValid(p1.formalTypes());
    }

    /** Returns true if the procedure has the given formal parameter types. */
    public final boolean hasFormals(List formalTypes) {
        return ts.hasFormals(this, formalTypes);
    }

    /** Returns true if the procedure has the given formal parameter types. */
    public boolean hasFormalsImpl(List formalTypes) {
        List l1 = this.formalTypes();
        List l2 = formalTypes;

        Iterator i1 = l1.iterator();
        Iterator i2 = l2.iterator();

        while (i1.hasNext() && i2.hasNext()) {
            Type t1 = (Type) i1.next();
            Type t2 = (Type) i2.next();

            if (! ts.equals(t1, t2)) {
                return false;
            }
        }

        return ! (i1.hasNext() || i2.hasNext());
    }

    /** Returns true iff <code>this</code> throws fewer exceptions than
     * <code>p</code>. */
    public final boolean throwsSubset(ProcedureInstance p) {
        return ts.throwsSubset(this, p);
    }

    /** Returns true iff <code>this</code> throws fewer exceptions than
     * <code>p</code>. */
    public boolean throwsSubsetImpl(ProcedureInstance p) {
        SubtypeSet s1 = new SubtypeSet(ts.Throwable());
        SubtypeSet s2 = new SubtypeSet(ts.Throwable());

        s1.addAll(this.throwTypes());
        s2.addAll(p.throwTypes());

        for (Iterator i = s1.iterator(); i.hasNext(); ) {
            Type t = (Type) i.next();
            if (! ts.isUncheckedException(t) && ! s2.contains(t)) {
                return false;
            }
        }

        return true;
    }

    /** Returns true if a call can be made with the given argument types. */
    public final boolean callValid(List argTypes) {
        return ts.callValid(this, argTypes);
    }

    /** Returns true if a call can be made with the given argument types. */
    public boolean callValidImpl(List argTypes) {
        List l1 = this.formalTypes();
        List l2 = argTypes;

        Iterator i1 = l1.iterator();
        Iterator i2 = l2.iterator();

        while (i1.hasNext() && i2.hasNext()) {
            Type t1 = (Type) i1.next();
            Type t2 = (Type) i2.next();

            if (! ts.isImplicitCastValid(t2, t1)) {
                return false;
            }
        }

        return ! (i1.hasNext() || i2.hasNext());
    }
}
