//
// This is the grammar specification from the Final Draft of the generic spec.
//
////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2007 IBM Corporation.
// All rights reserved. This program and the accompanying materials
// are made available under the terms of the Eclipse Public License v1.0
// which accompanies this distribution, and is available at
// http://www.eclipse.org/legal/epl-v10.html
//
//Contributors:
//    Philippe Charles (pcharles@us.ibm.com) - initial API and implementation

////////////////////////////////////////////////////////////////////////////////

package org.eclipse.imp.lpg.parser;

import lpg.runtime.*;

import java.lang.*;
public class JavaParser implements RuleAction
{
    private PrsStream prsStream = null;
    
    private boolean unimplementedSymbolsWarning = false;

    private static ParseTable prsTable = new JavaParserprs();
    public ParseTable getParseTable() { return prsTable; }

    private BacktrackingParser btParser = null;
    public BacktrackingParser getParser() { return btParser; }

    private void setResult(Object object) { btParser.setSym1(object); }
    public Object getRhsSym(int i) { return btParser.getSym(i); }

    public int getRhsTokenIndex(int i) { return btParser.getToken(i); }
    public IToken getRhsIToken(int i) { return prsStream.getIToken(getRhsTokenIndex(i)); }
    
    public int getRhsFirstTokenIndex(int i) { return btParser.getFirstToken(i); }
    public IToken getRhsFirstIToken(int i) { return prsStream.getIToken(getRhsFirstTokenIndex(i)); }

    public int getRhsLastTokenIndex(int i) { return btParser.getLastToken(i); }
    public IToken getRhsLastIToken(int i) { return prsStream.getIToken(getRhsLastTokenIndex(i)); }

    public int getLeftSpan() { return btParser.getFirstToken(); }
    public IToken getLeftIToken()  { return prsStream.getIToken(getLeftSpan()); }

    public int getRightSpan() { return btParser.getLastToken(); }
    public IToken getRightIToken() { return prsStream.getIToken(getRightSpan()); }

    public int getRhsErrorTokenIndex(int i)
    {
        int index = btParser.getToken(i);
        IToken err = prsStream.getIToken(index);
        return (err instanceof ErrorToken ? index : 0);
    }
    public ErrorToken getRhsErrorIToken(int i)
    {
        int index = btParser.getToken(i);
        IToken err = prsStream.getIToken(index);
        return (ErrorToken) (err instanceof ErrorToken ? err : null);
    }

    public void reset(ILexStream lexStream)
    {
        prsStream = new PrsStream(lexStream);
        btParser.reset(prsStream);

        try
        {
            prsStream.remapTerminalSymbols(orderedTerminalSymbols(), prsTable.getEoftSymbol());
        }
        catch(NullExportedSymbolsException e) {
        }
        catch(NullTerminalSymbolsException e) {
        }
        catch(UnimplementedTerminalsException e)
        {
            if (unimplementedSymbolsWarning) {
                java.util.ArrayList unimplemented_symbols = e.getSymbols();
                System.out.println("The Lexer will not scan the following token(s):");
                for (int i = 0; i < unimplemented_symbols.size(); i++)
                {
                    Integer id = (Integer) unimplemented_symbols.get(i);
                    System.out.println("    " + JavaParsersym.orderedTerminalSymbols[id.intValue()]);               
                }
                System.out.println();
            }
        }
        catch(UndefinedEofSymbolException e)
        {
            throw new Error(new UndefinedEofSymbolException
                                ("The Lexer does not implement the Eof symbol " +
                                 JavaParsersym.orderedTerminalSymbols[prsTable.getEoftSymbol()]));
        } 
    }
    
    public JavaParser()
    {
        try
        {
            btParser = new BacktrackingParser(prsStream, prsTable, (RuleAction) this);
        }
        catch (NotBacktrackParseTableException e)
        {
            throw new Error(new NotBacktrackParseTableException
                                ("Regenerate JavaParserprs.java with -BACKTRACK option"));
        }
        catch (BadParseSymFileException e)
        {
            throw new Error(new BadParseSymFileException("Bad Parser Symbol File -- JavaParsersym.java"));
        }
    }
    
    public JavaParser(ILexStream lexStream)
    {
        this();
        reset(lexStream);
    }
    
    public int numTokenKinds() { return JavaParsersym.numTokenKinds; }
    public String[] orderedTerminalSymbols() { return JavaParsersym.orderedTerminalSymbols; }
    public String getTokenKindName(int kind) { return JavaParsersym.orderedTerminalSymbols[kind]; }
    public int getEOFTokenKind() { return prsTable.getEoftSymbol(); }
    public IPrsStream getIPrsStream() { return prsStream; }

    /**
     * @deprecated replaced by {@link #getIPrsStream()}
     *
     */
    public PrsStream getPrsStream() { return prsStream; }

    /**
     * @deprecated replaced by {@link #getIPrsStream()}
     *
     */
    public PrsStream getParseStream() { return prsStream; }

    public Ast parser()
    {
        return parser(null, 0);
    }
    
    public Ast parser(Monitor monitor)
    {
        return parser(monitor, 0);
    }
    
    public Ast parser(int error_repair_count)
    {
        return parser(null, error_repair_count);
    }

    public Ast parser(Monitor monitor, int error_repair_count)
    {
        btParser.setMonitor(monitor);
        
        try
        {
            return (Ast) btParser.fuzzyParse(error_repair_count);
        }
        catch (BadParseException e)
        {
            prsStream.reset(e.error_token); // point to error token

            DiagnoseParser diagnoseParser = new DiagnoseParser(prsStream, prsTable);
            diagnoseParser.diagnose(e.error_token);
        }

        return null;
    }

    //
    // Additional entry points, if any
    //
    
    public Ast parseClassBodyDeclarationsopt()
    {
        return parseClassBodyDeclarationsopt(null, 0);
    }
        
    public Ast parseClassBodyDeclarationsopt(Monitor monitor)
    {
        return parseClassBodyDeclarationsopt(monitor, 0);
    }
        
    public Ast parseClassBodyDeclarationsopt(int error_repair_count)
    {
        return parseClassBodyDeclarationsopt(null, error_repair_count);
    }
        
    public Ast parseClassBodyDeclarationsopt(Monitor monitor, int error_repair_count)
    {
        btParser.setMonitor(monitor);
        
        try
        {
            return (Ast) btParser.fuzzyParseEntry(JavaParsersym.TK_ClassBodyDeclarationsoptMarker, error_repair_count);
        }
        catch (BadParseException e)
        {
            prsStream.reset(e.error_token); // point to error token

            DiagnoseParser diagnoseParser = new DiagnoseParser(prsStream, prsTable);
            diagnoseParser.diagnoseEntry(JavaParsersym.TK_ClassBodyDeclarationsoptMarker, e.error_token);
        }

        return null;
    }

    public Ast parseLPGUserAction()
    {
        return parseLPGUserAction(null, 0);
    }
        
    public Ast parseLPGUserAction(Monitor monitor)
    {
        return parseLPGUserAction(monitor, 0);
    }
        
    public Ast parseLPGUserAction(int error_repair_count)
    {
        return parseLPGUserAction(null, error_repair_count);
    }
        
    public Ast parseLPGUserAction(Monitor monitor, int error_repair_count)
    {
        btParser.setMonitor(monitor);
        
        try
        {
            return (Ast) btParser.fuzzyParseEntry(JavaParsersym.TK_LPGUserActionMarker, error_repair_count);
        }
        catch (BadParseException e)
        {
            prsStream.reset(e.error_token); // point to error token

            DiagnoseParser diagnoseParser = new DiagnoseParser(prsStream, prsTable);
            diagnoseParser.diagnoseEntry(JavaParsersym.TK_LPGUserActionMarker, e.error_token);
        }

        return null;
    }

    static public abstract class Ast implements IAst
    {
        public IAst getNextAst() { return null; }
        protected IToken leftIToken,
                         rightIToken;
        protected IAst parent = null;
        protected void setParent(IAst parent) { this.parent = parent; }
        public IAst getParent() { return parent; }

        public IToken getLeftIToken() { return leftIToken; }
        public IToken getRightIToken() { return rightIToken; }
        public IToken[] getPrecedingAdjuncts() { return leftIToken.getPrecedingAdjuncts(); }
        public IToken[] getFollowingAdjuncts() { return rightIToken.getFollowingAdjuncts(); }

        public String toString()
        {
            return leftIToken.getILexStream().toString(leftIToken.getStartOffset(), rightIToken.getEndOffset());
        }

        public Ast(IToken token) { this.leftIToken = this.rightIToken = token; }
        public Ast(IToken leftIToken, IToken rightIToken)
        {
            this.leftIToken = leftIToken;
            this.rightIToken = rightIToken;
        }

        void initialize() {}

        /**
         * A list of all children of this node, excluding the null ones.
         */
        public java.util.ArrayList getChildren()
        {
            java.util.ArrayList list = getAllChildren();
            int k = -1;
            for (int i = 0; i < list.size(); i++)
            {
                Object element = list.get(i);
                if (element != null)
                {
                    if (++k != i)
                        list.set(k, element);
                }
            }
            for (int i = list.size() - 1; i > k; i--) // remove extraneous elements
                list.remove(i);
            return list;
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public abstract java.util.ArrayList getAllChildren();

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Ast)) return false;
            Ast other = (Ast) o;
            return getLeftIToken().getILexStream() == other.getLeftIToken().getILexStream() &&
                   getLeftIToken().getTokenIndex() == other.getLeftIToken().getTokenIndex() &&
                   getRightIToken().getILexStream() == other.getRightIToken().getILexStream() &&
                   getRightIToken().getTokenIndex() == other.getRightIToken().getTokenIndex();
        }

        public int hashCode()
        {
            int hash = 7;
            if (getLeftIToken().getILexStream() != null) hash = hash * 31 + getLeftIToken().getILexStream().hashCode();
            hash = hash * 31 + getLeftIToken().getTokenIndex();
            if (getRightIToken().getILexStream() != null) hash = hash * 31 + getRightIToken().getILexStream().hashCode();
            hash = hash * 31 + getRightIToken().getTokenIndex();
            return hash;
        }
        public abstract void accept(IAstVisitor v);
    }

    static public abstract class AbstractAstList extends Ast
    {
        private boolean leftRecursive;
        private java.util.ArrayList list;
        public int size() { return list.size(); }
        public Ast getElementAt(int i) { return (Ast) list.get(leftRecursive ? i : list.size() - 1 - i); }
        public java.util.ArrayList getArrayList()
        {
            if (! leftRecursive) // reverse the list 
            {
                for (int i = 0, n = list.size() - 1; i < n; i++, n--)
                {
                    Object ith = list.get(i),
                           nth = list.get(n);
                    list.set(i, nth);
                    list.set(n, ith);
                }
                leftRecursive = true;
            }
            return list;
        }
        public void add(Ast element)
        {
            list.add(element);
            if (leftRecursive)
                 rightIToken = element.getRightIToken();
            else leftIToken = element.getLeftIToken();
        }

        public AbstractAstList(IToken leftIToken, IToken rightIToken, boolean leftRecursive)
        {
            super(leftIToken, rightIToken);
            this.leftRecursive = leftRecursive;
            list = new java.util.ArrayList();
        }

        public AbstractAstList(Ast element, boolean leftRecursive)
        {
            this(element.getLeftIToken(), element.getRightIToken(), leftRecursive);
            list.add(element);
        }

        /**
         * Make a copy of the list and return it. Note that we obtain the local list by
         * invoking getArrayList so as to make sure that the list we return is in proper order.
         */
        public java.util.ArrayList getAllChildren()
        {
            return (java.util.ArrayList) getArrayList().clone();
        }

    }

    static public class AstToken extends Ast implements IAstToken
    {
        public AstToken(IToken token) { super(token); }
        public IToken getIToken() { return leftIToken; }
        public String toString() { return leftIToken.toString(); }

        /**
         * A token class has no children. So, we return the empty list.
         */
        public java.util.ArrayList getAllChildren() { return new java.util.ArrayList(); }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AstToken)) return false;
            AstToken other = (AstToken) o;
            return getIToken().getILexStream() == other.getIToken().getILexStream() &&
                   getIToken().getTokenIndex() == other.getIToken().getTokenIndex();
        }

        public int hashCode()
        {
            int hash = 7;
            if (getIToken().getILexStream() != null) hash = hash * 31 + getIToken().getILexStream().hashCode();
            hash = hash * 31 + getIToken().getTokenIndex();
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>PrimitiveType
     *<li>ClassType
     *<li>ArrayType
     *<li>TypeDeclaration
     *<li>NormalClassDeclaration
     *<li>ClassMemberDeclaration
     *<li>FieldDeclaration
     *<li>MethodDeclaration
     *<li>ResultType
     *<li>VariableModifier
     *<li>MethodBody
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>InterfaceMemberDeclaration
     *<li>ConstantDeclaration
     *<li>AbstractMethodDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>Block
     *<li>BlockStatements
     *<li>LocalVariableDeclarationStatement
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>Commaopt
     *<li>Ellipsisopt
     *<li>LPGUserAction0
     *<li>LPGUserAction1
     *<li>LPGUserAction2
     *<li>LPGUserAction3
     *<li>LPGUserAction4
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *<li>ClassModifier0
     *<li>ClassModifier1
     *<li>ClassModifier2
     *<li>ClassModifier3
     *<li>ClassModifier4
     *<li>ClassModifier5
     *<li>ClassModifier6
     *<li>FieldModifier0
     *<li>FieldModifier1
     *<li>FieldModifier2
     *<li>FieldModifier3
     *<li>FieldModifier4
     *<li>FieldModifier5
     *<li>FieldModifier6
     *<li>MethodModifier0
     *<li>MethodModifier1
     *<li>MethodModifier2
     *<li>MethodModifier3
     *<li>MethodModifier4
     *<li>MethodModifier5
     *<li>MethodModifier6
     *<li>MethodModifier7
     *<li>MethodModifier8
     *<li>ConstructorModifier0
     *<li>ConstructorModifier1
     *<li>ConstructorModifier2
     *<li>InterfaceModifier0
     *<li>InterfaceModifier1
     *<li>InterfaceModifier2
     *<li>InterfaceModifier3
     *<li>InterfaceModifier4
     *<li>InterfaceModifier5
     *<li>ConstantModifier0
     *<li>ConstantModifier1
     *<li>ConstantModifier2
     *<li>AbstractMethodModifier0
     *<li>AbstractMethodModifier1
     *<li>AnnotationTypeElementDeclaration0
     *<li>AnnotationTypeElementDeclaration1
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>AssignmentOperator0
     *<li>AssignmentOperator1
     *<li>AssignmentOperator2
     *<li>AssignmentOperator3
     *<li>AssignmentOperator4
     *<li>AssignmentOperator5
     *<li>AssignmentOperator6
     *<li>AssignmentOperator7
     *<li>AssignmentOperator8
     *<li>AssignmentOperator9
     *<li>AssignmentOperator10
     *<li>AssignmentOperator11
     *</ul>
     *</b>
     */
    public interface IAstToken
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>Commaopt</b>
     */
    public interface ICommaopt extends IAstToken {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>Ellipsisopt</b>
     */
    public interface IEllipsisopt extends IAstToken {}

    /**
     * is implemented by <b>CompilationUnit</b>
     */
    public interface ICompilationUnit
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>ClassBodyDeclarations
     *<li>ClassMemberDeclaration
     *<li>FieldDeclaration
     *<li>MethodDeclaration
     *<li>StaticInitializer
     *<li>ConstructorDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>Block
     *</ul>
     *</b>
     */
    public interface IClassBodyDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>Block
     *<li>BlockStatements
     *<li>LocalVariableDeclarationStatement
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>LPGUserAction0
     *<li>LPGUserAction1
     *<li>LPGUserAction2
     *<li>LPGUserAction3
     *<li>LPGUserAction4
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface ILPGUserAction extends IAstToken {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>identifier</b>
     */
    public interface Iidentifier extends IAstToken, ITypeName, ITypeVariable, IPackageName, IExpressionName, IMethodName, IPackageOrTypeName, IAmbiguousName, IVariableDeclaratorId, ISimpleTypeName, ISimpleName, IEnumConstant, Iidentifieropt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>Block
     *<li>BlockStatements
     *<li>LocalVariableDeclarationStatement
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IBlockStatementsopt extends ILPGUserAction {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>PrimitiveType
     *<li>ClassType
     *<li>ArrayType
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *</ul>
     *</b>
     */
    public interface IType extends IResultType {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>PrimitiveType
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *</ul>
     *</b>
     */
    public interface IPrimitiveType extends IType, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ClassType
     *<li>ArrayType
     *</ul>
     *</b>
     */
    public interface IReferenceType extends IType, IActualTypeArgument {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *</ul>
     *</b>
     */
    public interface INumericType extends IPrimitiveType {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *</ul>
     *</b>
     */
    public interface IIntegralType extends INumericType, IAstToken {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *</ul>
     *</b>
     */
    public interface IFloatingPointType extends INumericType, IAstToken {}

    /**
     * is implemented by <b>ClassType</b>
     */
    public interface IClassOrInterfaceType extends IReferenceType {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>identifier</b>
     */
    public interface ITypeVariable extends IReferenceType, IExceptionType {}

    /**
     * is implemented by <b>ArrayType</b>
     */
    public interface IArrayType extends IReferenceType {}

    /**
     * is implemented by <b>ClassType</b>
     */
    public interface IClassType extends IClassOrInterfaceType, IExceptionType {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>TypeName
     *</ul>
     *</b>
     */
    public interface ITypeName extends IClassName {}

    /**
     * is implemented by <b>TypeArguments</b>
     */
    public interface ITypeArgumentsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>InterfaceType</b>
     */
    public interface IInterfaceType extends IInterfaceTypeList {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>TypeName
     *</ul>
     *</b>
     */
    public interface IClassName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>TypeParameter</b>
     */
    public interface ITypeParameter extends ITypeParameterList {}

    /**
     * is implemented by <b>TypeBound</b>
     */
    public interface ITypeBoundopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>TypeBound</b>
     */
    public interface ITypeBound extends ITypeBoundopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AdditionalBoundList
     *<li>AdditionalBound
     *</ul>
     *</b>
     */
    public interface IAdditionalBoundListopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AdditionalBoundList
     *<li>AdditionalBound
     *</ul>
     *</b>
     */
    public interface IAdditionalBoundList extends IAdditionalBoundListopt {}

    /**
     * is implemented by <b>AdditionalBound</b>
     */
    public interface IAdditionalBound extends IAdditionalBoundList {}

    /**
     * is implemented by <b>TypeArguments</b>
     */
    public interface ITypeArguments extends ITypeArgumentsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ClassType
     *<li>ArrayType
     *<li>ActualTypeArgumentList
     *<li>Wildcard
     *</ul>
     *</b>
     */
    public interface IActualTypeArgumentList
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ClassType
     *<li>ArrayType
     *<li>Wildcard
     *</ul>
     *</b>
     */
    public interface IActualTypeArgument extends IActualTypeArgumentList {}

    /**
     * is implemented by <b>Wildcard</b>
     */
    public interface IWildcard extends IActualTypeArgument {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>WildcardBounds0
     *<li>WildcardBounds1
     *</ul>
     *</b>
     */
    public interface IWildcardBoundsOpt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>WildcardBounds0
     *<li>WildcardBounds1
     *</ul>
     *</b>
     */
    public interface IWildcardBounds extends IWildcardBoundsOpt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>PackageName
     *</ul>
     *</b>
     */
    public interface IPackageName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *</ul>
     *</b>
     */
    public interface IExpressionName extends IPostfixExpression, ILeftHandSide {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>AmbiguousName
     *</ul>
     *</b>
     */
    public interface IAmbiguousName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>MethodName
     *</ul>
     *</b>
     */
    public interface IMethodName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>PackageOrTypeName
     *</ul>
     *</b>
     */
    public interface IPackageOrTypeName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>PackageDeclaration</b>
     */
    public interface IPackageDeclarationopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ImportDeclarations
     *<li>SingleTypeImportDeclaration
     *<li>TypeImportOnDemandDeclaration
     *<li>SingleStaticImportDeclaration
     *<li>StaticImportOnDemandDeclaration
     *</ul>
     *</b>
     */
    public interface IImportDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>TypeDeclarations
     *<li>TypeDeclaration
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface ITypeDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ImportDeclarations
     *<li>SingleTypeImportDeclaration
     *<li>TypeImportOnDemandDeclaration
     *<li>SingleStaticImportDeclaration
     *<li>StaticImportOnDemandDeclaration
     *</ul>
     *</b>
     */
    public interface IImportDeclarations extends IImportDeclarationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SingleTypeImportDeclaration
     *<li>TypeImportOnDemandDeclaration
     *<li>SingleStaticImportDeclaration
     *<li>StaticImportOnDemandDeclaration
     *</ul>
     *</b>
     */
    public interface IImportDeclaration extends IImportDeclarations {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>TypeDeclarations
     *<li>TypeDeclaration
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface ITypeDeclarations extends ITypeDeclarationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>TypeDeclaration
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface ITypeDeclaration extends ITypeDeclarations, IAstToken {}

    /**
     * is implemented by <b>PackageDeclaration</b>
     */
    public interface IPackageDeclaration extends IPackageDeclarationopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IAnnotationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>SingleTypeImportDeclaration</b>
     */
    public interface ISingleTypeImportDeclaration extends IImportDeclaration {}

    /**
     * is implemented by <b>TypeImportOnDemandDeclaration</b>
     */
    public interface ITypeImportOnDemandDeclaration extends IImportDeclaration {}

    /**
     * is implemented by <b>SingleStaticImportDeclaration</b>
     */
    public interface ISingleStaticImportDeclaration extends IImportDeclaration {}

    /**
     * is implemented by <b>StaticImportOnDemandDeclaration</b>
     */
    public interface IStaticImportOnDemandDeclaration extends IImportDeclaration {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *</ul>
     *</b>
     */
    public interface IClassDeclaration extends ITypeDeclaration, IClassMemberDeclaration, IInterfaceMemberDeclaration, IAnnotationTypeElementDeclaration, IBlockStatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface IInterfaceDeclaration extends ITypeDeclaration, IClassMemberDeclaration, IInterfaceMemberDeclaration, IAnnotationTypeElementDeclaration {}

    /**
     * is implemented by <b>NormalClassDeclaration</b>
     */
    public interface INormalClassDeclaration extends IClassDeclaration {}

    /**
     * is implemented by <b>EnumDeclaration</b>
     */
    public interface IEnumDeclaration extends IClassDeclaration, IAnnotationTypeElementDeclaration {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ClassModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ClassModifier0
     *<li>ClassModifier1
     *<li>ClassModifier2
     *<li>ClassModifier3
     *<li>ClassModifier4
     *<li>ClassModifier5
     *<li>ClassModifier6
     *</ul>
     *</b>
     */
    public interface IClassModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>TypeParameters</b>
     */
    public interface ITypeParametersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Super</b>
     */
    public interface ISuperopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Interfaces</b>
     */
    public interface IInterfacesopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>ClassBody</b>
     */
    public interface IClassBody extends IClassBodyopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ClassModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ClassModifier0
     *<li>ClassModifier1
     *<li>ClassModifier2
     *<li>ClassModifier3
     *<li>ClassModifier4
     *<li>ClassModifier5
     *<li>ClassModifier6
     *</ul>
     *</b>
     */
    public interface IClassModifiers extends IClassModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ClassModifier0
     *<li>ClassModifier1
     *<li>ClassModifier2
     *<li>ClassModifier3
     *<li>ClassModifier4
     *<li>ClassModifier5
     *<li>ClassModifier6
     *</ul>
     *</b>
     */
    public interface IClassModifier extends IClassModifiers, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IAnnotation extends IClassModifier, IFieldModifier, IInterfaceModifier, IConstantModifier, IAnnotations, IElementValue {}

    /**
     * is implemented by <b>TypeParameters</b>
     */
    public interface ITypeParameters extends ITypeParametersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>TypeParameter
     *<li>TypeParameterList
     *</ul>
     *</b>
     */
    public interface ITypeParameterList
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Super</b>
     */
    public interface ISuper extends ISuperopt {}

    /**
     * is implemented by <b>Interfaces</b>
     */
    public interface IInterfaces extends IInterfacesopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>InterfaceType
     *<li>InterfaceTypeList
     *</ul>
     *</b>
     */
    public interface IInterfaceTypeList
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>ClassBodyDeclarations
     *<li>ClassMemberDeclaration
     *<li>FieldDeclaration
     *<li>MethodDeclaration
     *<li>StaticInitializer
     *<li>ConstructorDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>Block
     *</ul>
     *</b>
     */
    public interface IClassBodyDeclarations extends IClassBodyDeclarationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>ClassMemberDeclaration
     *<li>FieldDeclaration
     *<li>MethodDeclaration
     *<li>StaticInitializer
     *<li>ConstructorDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>Block
     *</ul>
     *</b>
     */
    public interface IClassBodyDeclaration extends IClassBodyDeclarations {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>ClassMemberDeclaration
     *<li>FieldDeclaration
     *<li>MethodDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface IClassMemberDeclaration extends IClassBodyDeclaration, IAstToken {}

    /**
     * is implemented by <b>Block</b>
     */
    public interface IInstanceInitializer extends IClassBodyDeclaration {}

    /**
     * is implemented by <b>StaticInitializer</b>
     */
    public interface IStaticInitializer extends IClassBodyDeclaration {}

    /**
     * is implemented by <b>ConstructorDeclaration</b>
     */
    public interface IConstructorDeclaration extends IClassBodyDeclaration {}

    /**
     * is implemented by <b>FieldDeclaration</b>
     */
    public interface IFieldDeclaration extends IClassMemberDeclaration {}

    /**
     * is implemented by <b>MethodDeclaration</b>
     */
    public interface IMethodDeclaration extends IClassMemberDeclaration {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FieldModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>FieldModifier0
     *<li>FieldModifier1
     *<li>FieldModifier2
     *<li>FieldModifier3
     *<li>FieldModifier4
     *<li>FieldModifier5
     *<li>FieldModifier6
     *</ul>
     *</b>
     */
    public interface IFieldModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>VariableDeclarators
     *<li>VariableDeclarator
     *<li>VariableDeclaratorId
     *</ul>
     *</b>
     */
    public interface IVariableDeclarators
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>VariableDeclarator
     *<li>VariableDeclaratorId
     *</ul>
     *</b>
     */
    public interface IVariableDeclarator extends IVariableDeclarators {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>VariableDeclaratorId
     *</ul>
     *</b>
     */
    public interface IVariableDeclaratorId extends IVariableDeclarator {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>ArrayInitializer
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IVariableInitializer extends IVariableInitializers {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IExpression extends IVariableInitializer, IArgumentList, IConstantExpression, IExpressionopt {}

    /**
     * is implemented by <b>ArrayInitializer</b>
     */
    public interface IArrayInitializer extends IVariableInitializer {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FieldModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>FieldModifier0
     *<li>FieldModifier1
     *<li>FieldModifier2
     *<li>FieldModifier3
     *<li>FieldModifier4
     *<li>FieldModifier5
     *<li>FieldModifier6
     *</ul>
     *</b>
     */
    public interface IFieldModifiers extends IFieldModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>FieldModifier0
     *<li>FieldModifier1
     *<li>FieldModifier2
     *<li>FieldModifier3
     *<li>FieldModifier4
     *<li>FieldModifier5
     *<li>FieldModifier6
     *</ul>
     *</b>
     */
    public interface IFieldModifier extends IFieldModifiers, IAstToken {}

    /**
     * is implemented by <b>MethodHeader</b>
     */
    public interface IMethodHeader
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>MethodBody
     *<li>Block
     *</ul>
     *</b>
     */
    public interface IMethodBody extends IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>MethodModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>MethodModifier0
     *<li>MethodModifier1
     *<li>MethodModifier2
     *<li>MethodModifier3
     *<li>MethodModifier4
     *<li>MethodModifier5
     *<li>MethodModifier6
     *<li>MethodModifier7
     *<li>MethodModifier8
     *</ul>
     *</b>
     */
    public interface IMethodModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>PrimitiveType
     *<li>ClassType
     *<li>ArrayType
     *<li>ResultType
     *<li>IntegralType0
     *<li>IntegralType1
     *<li>IntegralType2
     *<li>IntegralType3
     *<li>IntegralType4
     *<li>FloatingPointType0
     *<li>FloatingPointType1
     *</ul>
     *</b>
     */
    public interface IResultType extends IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>MethodDeclarator0
     *<li>MethodDeclarator1
     *</ul>
     *</b>
     */
    public interface IMethodDeclarator
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Throws</b>
     */
    public interface IThrowsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FormalParameterList
     *<li>LastFormalParameter
     *</ul>
     *</b>
     */
    public interface IFormalParameterListopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FormalParameterList
     *<li>LastFormalParameter
     *</ul>
     *</b>
     */
    public interface IFormalParameterList extends IFormalParameterListopt {}

    /**
     * is implemented by <b>LastFormalParameter</b>
     */
    public interface ILastFormalParameter extends IFormalParameterList {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FormalParameters
     *<li>FormalParameter
     *</ul>
     *</b>
     */
    public interface IFormalParameters
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>FormalParameter</b>
     */
    public interface IFormalParameter extends IFormalParameters {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>VariableModifiers
     *<li>VariableModifier
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IVariableModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>VariableModifiers
     *<li>VariableModifier
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IVariableModifiers extends IVariableModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>VariableModifier
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IVariableModifier extends IVariableModifiers, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *</ul>
     *</b>
     */
    public interface IAnnotations extends IVariableModifier, IMethodModifier, IConstructorModifier, IAbstractMethodModifier, IAnnotationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>MethodModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>MethodModifier0
     *<li>MethodModifier1
     *<li>MethodModifier2
     *<li>MethodModifier3
     *<li>MethodModifier4
     *<li>MethodModifier5
     *<li>MethodModifier6
     *<li>MethodModifier7
     *<li>MethodModifier8
     *</ul>
     *</b>
     */
    public interface IMethodModifiers extends IMethodModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>MethodModifier0
     *<li>MethodModifier1
     *<li>MethodModifier2
     *<li>MethodModifier3
     *<li>MethodModifier4
     *<li>MethodModifier5
     *<li>MethodModifier6
     *<li>MethodModifier7
     *<li>MethodModifier8
     *</ul>
     *</b>
     */
    public interface IMethodModifier extends IMethodModifiers, IAstToken {}

    /**
     * is implemented by <b>Throws</b>
     */
    public interface IThrows extends IThrowsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ClassType
     *<li>ExceptionTypeList
     *</ul>
     *</b>
     */
    public interface IExceptionTypeList
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ClassType
     *</ul>
     *</b>
     */
    public interface IExceptionType extends IExceptionTypeList {}

    /**
     * is implemented by <b>Block</b>
     */
    public interface IBlock extends IMethodBody, IInstanceInitializer, IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ConstructorModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstructorModifier0
     *<li>ConstructorModifier1
     *<li>ConstructorModifier2
     *</ul>
     *</b>
     */
    public interface IConstructorModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>ConstructorDeclarator</b>
     */
    public interface IConstructorDeclarator
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>ConstructorBody</b>
     */
    public interface IConstructorBody
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>identifier</b>
     */
    public interface ISimpleTypeName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ConstructorModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstructorModifier0
     *<li>ConstructorModifier1
     *<li>ConstructorModifier2
     *</ul>
     *</b>
     */
    public interface IConstructorModifiers extends IConstructorModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstructorModifier0
     *<li>ConstructorModifier1
     *<li>ConstructorModifier2
     *</ul>
     *</b>
     */
    public interface IConstructorModifier extends IConstructorModifiers, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ExplicitConstructorInvocation0
     *<li>ExplicitConstructorInvocation1
     *<li>ExplicitConstructorInvocation2
     *</ul>
     *</b>
     */
    public interface IExplicitConstructorInvocationopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ExplicitConstructorInvocation0
     *<li>ExplicitConstructorInvocation1
     *<li>ExplicitConstructorInvocation2
     *</ul>
     *</b>
     */
    public interface IExplicitConstructorInvocation extends IExplicitConstructorInvocationopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>ArgumentList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IArgumentListopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *</ul>
     *</b>
     */
    public interface IPrimary extends IPostfixExpression {}

    /**
     * is implemented by <b>EnumBody</b>
     */
    public interface IEnumBody
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>EnumConstants
     *<li>EnumConstant
     *</ul>
     *</b>
     */
    public interface IEnumConstantsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>EnumBodyDeclarations</b>
     */
    public interface IEnumBodyDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>EnumConstants
     *<li>EnumConstant
     *</ul>
     *</b>
     */
    public interface IEnumConstants extends IEnumConstantsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>EnumConstant
     *</ul>
     *</b>
     */
    public interface IEnumConstant extends IEnumConstants {}

    /**
     * is implemented by <b>Arguments</b>
     */
    public interface IArgumentsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>ClassBody</b>
     */
    public interface IClassBodyopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Arguments</b>
     */
    public interface IArguments extends IArgumentsopt {}

    /**
     * is implemented by <b>EnumBodyDeclarations</b>
     */
    public interface IEnumBodyDeclarations extends IEnumBodyDeclarationsopt {}

    /**
     * is implemented by <b>NormalInterfaceDeclaration</b>
     */
    public interface INormalInterfaceDeclaration extends IInterfaceDeclaration {}

    /**
     * is implemented by <b>AnnotationTypeDeclaration</b>
     */
    public interface IAnnotationTypeDeclaration extends IInterfaceDeclaration, IAnnotationTypeElementDeclaration {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>InterfaceModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>InterfaceModifier0
     *<li>InterfaceModifier1
     *<li>InterfaceModifier2
     *<li>InterfaceModifier3
     *<li>InterfaceModifier4
     *<li>InterfaceModifier5
     *</ul>
     *</b>
     */
    public interface IInterfaceModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ExtendsInterfaces0
     *<li>ExtendsInterfaces1
     *</ul>
     *</b>
     */
    public interface IExtendsInterfacesopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>InterfaceBody</b>
     */
    public interface IInterfaceBody
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>InterfaceModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>InterfaceModifier0
     *<li>InterfaceModifier1
     *<li>InterfaceModifier2
     *<li>InterfaceModifier3
     *<li>InterfaceModifier4
     *<li>InterfaceModifier5
     *</ul>
     *</b>
     */
    public interface IInterfaceModifiers extends IInterfaceModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>InterfaceModifier0
     *<li>InterfaceModifier1
     *<li>InterfaceModifier2
     *<li>InterfaceModifier3
     *<li>InterfaceModifier4
     *<li>InterfaceModifier5
     *</ul>
     *</b>
     */
    public interface IInterfaceModifier extends IInterfaceModifiers, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ExtendsInterfaces0
     *<li>ExtendsInterfaces1
     *</ul>
     *</b>
     */
    public interface IExtendsInterfaces extends IExtendsInterfacesopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>InterfaceMemberDeclarations
     *<li>InterfaceMemberDeclaration
     *<li>ConstantDeclaration
     *<li>AbstractMethodDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface IInterfaceMemberDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>InterfaceMemberDeclarations
     *<li>InterfaceMemberDeclaration
     *<li>ConstantDeclaration
     *<li>AbstractMethodDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface IInterfaceMemberDeclarations extends IInterfaceMemberDeclarationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>InterfaceMemberDeclaration
     *<li>ConstantDeclaration
     *<li>AbstractMethodDeclaration
     *<li>AnnotationTypeDeclaration
     *</ul>
     *</b>
     */
    public interface IInterfaceMemberDeclaration extends IInterfaceMemberDeclarations, IAstToken {}

    /**
     * is implemented by <b>ConstantDeclaration</b>
     */
    public interface IConstantDeclaration extends IInterfaceMemberDeclaration, IAnnotationTypeElementDeclaration {}

    /**
     * is implemented by <b>AbstractMethodDeclaration</b>
     */
    public interface IAbstractMethodDeclaration extends IInterfaceMemberDeclaration {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ConstantModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstantModifier0
     *<li>ConstantModifier1
     *<li>ConstantModifier2
     *</ul>
     *</b>
     */
    public interface IConstantModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ConstantModifiers
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstantModifier0
     *<li>ConstantModifier1
     *<li>ConstantModifier2
     *</ul>
     *</b>
     */
    public interface IConstantModifiers extends IConstantModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>ConstantModifier0
     *<li>ConstantModifier1
     *<li>ConstantModifier2
     *</ul>
     *</b>
     */
    public interface IConstantModifier extends IConstantModifiers, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AbstractMethodModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>AbstractMethodModifier0
     *<li>AbstractMethodModifier1
     *</ul>
     *</b>
     */
    public interface IAbstractMethodModifiersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AbstractMethodModifiers
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>AbstractMethodModifier0
     *<li>AbstractMethodModifier1
     *</ul>
     *</b>
     */
    public interface IAbstractMethodModifiers extends IAbstractMethodModifiersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Annotations
     *<li>NormalAnnotation
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>AbstractMethodModifier0
     *<li>AbstractMethodModifier1
     *</ul>
     *</b>
     */
    public interface IAbstractMethodModifier extends IAbstractMethodModifiers, IAstToken {}

    /**
     * is implemented by <b>AnnotationTypeBody</b>
     */
    public interface IAnnotationTypeBody
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>ConstantDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>AnnotationTypeElementDeclarations
     *<li>AnnotationTypeElementDeclaration0
     *<li>AnnotationTypeElementDeclaration1
     *</ul>
     *</b>
     */
    public interface IAnnotationTypeElementDeclarationsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>ConstantDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>AnnotationTypeElementDeclarations
     *<li>AnnotationTypeElementDeclaration0
     *<li>AnnotationTypeElementDeclaration1
     *</ul>
     *</b>
     */
    public interface IAnnotationTypeElementDeclarations extends IAnnotationTypeElementDeclarationsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>NormalInterfaceDeclaration
     *<li>ConstantDeclaration
     *<li>AnnotationTypeDeclaration
     *<li>AnnotationTypeElementDeclaration0
     *<li>AnnotationTypeElementDeclaration1
     *</ul>
     *</b>
     */
    public interface IAnnotationTypeElementDeclaration extends IAnnotationTypeElementDeclarations, IAstToken {}

    /**
     * is implemented by <b>DefaultValue</b>
     */
    public interface IDefaultValueopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>DefaultValue</b>
     */
    public interface IDefaultValue extends IDefaultValueopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>NormalAnnotation
     *<li>ElementValueArrayInitializer
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IElementValue extends IElementValues {}

    /**
     * is implemented by <b>NormalAnnotation</b>
     */
    public interface INormalAnnotation extends IAnnotation {}

    /**
     * is implemented by <b>MarkerAnnotation</b>
     */
    public interface IMarkerAnnotation extends IAnnotation {}

    /**
     * is implemented by <b>SingleElementAnnotation</b>
     */
    public interface ISingleElementAnnotation extends IAnnotation {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ElementValuePairs
     *<li>ElementValuePair
     *</ul>
     *</b>
     */
    public interface IElementValuePairsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ElementValuePairs
     *<li>ElementValuePair
     *</ul>
     *</b>
     */
    public interface IElementValuePairs extends IElementValuePairsopt {}

    /**
     * is implemented by <b>ElementValuePair</b>
     */
    public interface IElementValuePair extends IElementValuePairs {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>identifier</b>
     */
    public interface ISimpleName
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IConditionalExpression extends IElementValue, IAssignmentExpression {}

    /**
     * is implemented by <b>ElementValueArrayInitializer</b>
     */
    public interface IElementValueArrayInitializer extends IElementValue {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>NormalAnnotation
     *<li>ElementValueArrayInitializer
     *<li>ElementValues
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IElementValuesopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>NormalAnnotation
     *<li>ElementValueArrayInitializer
     *<li>ElementValues
     *<li>MarkerAnnotation
     *<li>SingleElementAnnotation
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IElementValues extends IElementValuesopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>ArrayInitializer
     *<li>VariableInitializers
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IVariableInitializersopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>ArrayInitializer
     *<li>VariableInitializers
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IVariableInitializers extends IVariableInitializersopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>Block
     *<li>BlockStatements
     *<li>LocalVariableDeclarationStatement
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IBlockStatements extends IBlockStatementsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>NormalClassDeclaration
     *<li>EnumDeclaration
     *<li>Block
     *<li>LocalVariableDeclarationStatement
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IBlockStatement extends IBlockStatements {}

    /**
     * is implemented by <b>LocalVariableDeclarationStatement</b>
     */
    public interface ILocalVariableDeclarationStatement extends IBlockStatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Block
     *<li>IfThenStatement
     *<li>IfThenElseStatement
     *<li>EmptyStatement
     *<li>LabeledStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatement
     *<li>DoStatement
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IStatement extends IBlockStatement {}

    /**
     * is implemented by <b>LocalVariableDeclaration</b>
     */
    public interface ILocalVariableDeclaration extends IForInit {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Block
     *<li>EmptyStatement
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>DoStatement
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IStatementWithoutTrailingSubstatement extends IStatement, IStatementNoShortIf {}

    /**
     * is implemented by <b>LabeledStatement</b>
     */
    public interface ILabeledStatement extends IStatement {}

    /**
     * is implemented by <b>IfThenStatement</b>
     */
    public interface IIfThenStatement extends IStatement {}

    /**
     * is implemented by <b>IfThenElseStatement</b>
     */
    public interface IIfThenElseStatement extends IStatement {}

    /**
     * is implemented by <b>WhileStatement</b>
     */
    public interface IWhileStatement extends IStatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>BasicForStatement
     *<li>EnhancedForStatement
     *</ul>
     *</b>
     */
    public interface IForStatement extends IStatement {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>EmptyStatement</b>
     */
    public interface IEmptyStatement extends IStatementWithoutTrailingSubstatement, IAstToken {}

    /**
     * is implemented by <b>ExpressionStatement</b>
     */
    public interface IExpressionStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AssertStatement0
     *<li>AssertStatement1
     *</ul>
     *</b>
     */
    public interface IAssertStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>SwitchStatement</b>
     */
    public interface ISwitchStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>DoStatement</b>
     */
    public interface IDoStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>BreakStatement</b>
     */
    public interface IBreakStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>ContinueStatement</b>
     */
    public interface IContinueStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>ReturnStatement</b>
     */
    public interface IReturnStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>SynchronizedStatement</b>
     */
    public interface ISynchronizedStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by <b>ThrowStatement</b>
     */
    public interface IThrowStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface ITryStatement extends IStatementWithoutTrailingSubstatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Block
     *<li>IfThenElseStatementNoShortIf
     *<li>EmptyStatement
     *<li>LabeledStatementNoShortIf
     *<li>ExpressionStatement
     *<li>SwitchStatement
     *<li>WhileStatementNoShortIf
     *<li>DoStatement
     *<li>ForStatementNoShortIf
     *<li>BreakStatement
     *<li>ContinueStatement
     *<li>ReturnStatement
     *<li>ThrowStatement
     *<li>SynchronizedStatement
     *<li>AssertStatement0
     *<li>AssertStatement1
     *<li>TryStatement0
     *<li>TryStatement1
     *</ul>
     *</b>
     */
    public interface IStatementNoShortIf
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>LabeledStatementNoShortIf</b>
     */
    public interface ILabeledStatementNoShortIf extends IStatementNoShortIf {}

    /**
     * is implemented by <b>IfThenElseStatementNoShortIf</b>
     */
    public interface IIfThenElseStatementNoShortIf extends IStatementNoShortIf {}

    /**
     * is implemented by <b>WhileStatementNoShortIf</b>
     */
    public interface IWhileStatementNoShortIf extends IStatementNoShortIf {}

    /**
     * is implemented by <b>ForStatementNoShortIf</b>
     */
    public interface IForStatementNoShortIf extends IStatementNoShortIf {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IStatementExpression extends IStatementExpressionList {}

    /**
     * is implemented by <b>Assignment</b>
     */
    public interface IAssignment extends IStatementExpression, IAssignmentExpression {}

    /**
     * is implemented by <b>PreIncrementExpression</b>
     */
    public interface IPreIncrementExpression extends IStatementExpression, IUnaryExpression {}

    /**
     * is implemented by <b>PreDecrementExpression</b>
     */
    public interface IPreDecrementExpression extends IStatementExpression, IUnaryExpression {}

    /**
     * is implemented by <b>PostIncrementExpression</b>
     */
    public interface IPostIncrementExpression extends IStatementExpression, IPostfixExpression {}

    /**
     * is implemented by <b>PostDecrementExpression</b>
     */
    public interface IPostDecrementExpression extends IStatementExpression, IPostfixExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IMethodInvocation extends IStatementExpression, IPrimaryNoNewArray {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *</ul>
     *</b>
     */
    public interface IClassInstanceCreationExpression extends IStatementExpression, IPrimaryNoNewArray {}

    /**
     * is implemented by <b>SwitchBlock</b>
     */
    public interface ISwitchBlock
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SwitchBlockStatementGroups
     *<li>SwitchBlockStatementGroup
     *</ul>
     *</b>
     */
    public interface ISwitchBlockStatementGroupsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SwitchLabels
     *<li>SwitchLabel0
     *<li>SwitchLabel1
     *<li>SwitchLabel2
     *</ul>
     *</b>
     */
    public interface ISwitchLabelsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SwitchBlockStatementGroups
     *<li>SwitchBlockStatementGroup
     *</ul>
     *</b>
     */
    public interface ISwitchBlockStatementGroups extends ISwitchBlockStatementGroupsopt {}

    /**
     * is implemented by <b>SwitchBlockStatementGroup</b>
     */
    public interface ISwitchBlockStatementGroup extends ISwitchBlockStatementGroups {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SwitchLabels
     *<li>SwitchLabel0
     *<li>SwitchLabel1
     *<li>SwitchLabel2
     *</ul>
     *</b>
     */
    public interface ISwitchLabels extends ISwitchLabelsopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>SwitchLabel0
     *<li>SwitchLabel1
     *<li>SwitchLabel2
     *</ul>
     *</b>
     */
    public interface ISwitchLabel extends ISwitchLabels {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IConstantExpression
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>BasicForStatement</b>
     */
    public interface IBasicForStatement extends IForStatement {}

    /**
     * is implemented by <b>EnhancedForStatement</b>
     */
    public interface IEnhancedForStatement extends IForStatement {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>LocalVariableDeclaration
     *<li>StatementExpressionList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IForInitopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IExpressionopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>StatementExpressionList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IForUpdateopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>LocalVariableDeclaration
     *<li>StatementExpressionList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IForInit extends IForInitopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>StatementExpressionList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IStatementExpressionList extends IForInit, IForUpdate {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>StatementExpressionList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>Assignment
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *</ul>
     *</b>
     */
    public interface IForUpdate extends IForUpdateopt {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by <b>identifier</b>
     */
    public interface Iidentifieropt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Catches
     *<li>CatchClause
     *</ul>
     *</b>
     */
    public interface ICatches extends ICatchesopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Catches
     *<li>CatchClause
     *</ul>
     *</b>
     */
    public interface ICatchesopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>Finally</b>
     */
    public interface IFinally
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by <b>CatchClause</b>
     */
    public interface ICatchClause extends ICatches {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *</ul>
     *</b>
     */
    public interface IPrimaryNoNewArray extends IPrimary, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *</ul>
     *</b>
     */
    public interface IArrayCreationExpression extends IPrimary {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *</ul>
     *</b>
     */
    public interface ILiteral extends IPrimaryNoNewArray, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *</ul>
     *</b>
     */
    public interface IFieldAccess extends IPrimaryNoNewArray, ILeftHandSide {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *</ul>
     *</b>
     */
    public interface IArrayAccess extends IPrimaryNoNewArray, ILeftHandSide {}

    /**
     * is always implemented by <b>AstToken</b>. It is also implemented by:
     *<b>
     *<ul>
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *</ul>
     *</b>
     */
    public interface IBooleanLiteral extends ILiteral, IAstToken {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>ArgumentList
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IArgumentList extends IArgumentListopt {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>DimExprs
     *<li>DimExpr
     *</ul>
     *</b>
     */
    public interface IDimExprs
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Dims0
     *<li>Dims1
     *</ul>
     *</b>
     */
    public interface IDimsopt
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>Dims0
     *<li>Dims1
     *</ul>
     *</b>
     */
    public interface IDims extends IDimsopt {}

    /**
     * is implemented by <b>DimExpr</b>
     */
    public interface IDimExpr extends IDimExprs {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *</ul>
     *</b>
     */
    public interface IPostfixExpression extends IUnaryExpressionNotPlusMinus {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *</ul>
     *</b>
     */
    public interface IUnaryExpression extends IMultiplicativeExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *</ul>
     *</b>
     */
    public interface IUnaryExpressionNotPlusMinus extends IUnaryExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>CastExpression0
     *<li>CastExpression1
     *</ul>
     *</b>
     */
    public interface ICastExpression extends IUnaryExpressionNotPlusMinus {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *</ul>
     *</b>
     */
    public interface IMultiplicativeExpression extends IAdditiveExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *</ul>
     *</b>
     */
    public interface IAdditiveExpression extends IShiftExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *</ul>
     *</b>
     */
    public interface IShiftExpression extends IRelationalExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *</ul>
     *</b>
     */
    public interface IRelationalExpression extends IEqualityExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IEqualityExpression extends IAndExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IAndExpression extends IExclusiveOrExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IExclusiveOrExpression extends IInclusiveOrExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IInclusiveOrExpression extends IConditionalAndExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IConditionalAndExpression extends IConditionalOrExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IConditionalOrExpression extends IConditionalExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>PostIncrementExpression
     *<li>PostDecrementExpression
     *<li>PreIncrementExpression
     *<li>PreDecrementExpression
     *<li>AndExpression
     *<li>ExclusiveOrExpression
     *<li>InclusiveOrExpression
     *<li>ConditionalAndExpression
     *<li>ConditionalOrExpression
     *<li>ConditionalExpression
     *<li>Assignment
     *<li>PrimaryNoNewArray0
     *<li>PrimaryNoNewArray1
     *<li>PrimaryNoNewArray2
     *<li>PrimaryNoNewArray3
     *<li>PrimaryNoNewArray4
     *<li>Literal0
     *<li>Literal1
     *<li>Literal2
     *<li>Literal3
     *<li>Literal4
     *<li>Literal5
     *<li>Literal6
     *<li>BooleanLiteral0
     *<li>BooleanLiteral1
     *<li>ClassInstanceCreationExpression0
     *<li>ClassInstanceCreationExpression1
     *<li>ArrayCreationExpression0
     *<li>ArrayCreationExpression1
     *<li>ArrayCreationExpression2
     *<li>ArrayCreationExpression3
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>MethodInvocation0
     *<li>MethodInvocation1
     *<li>MethodInvocation2
     *<li>MethodInvocation3
     *<li>MethodInvocation4
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *<li>UnaryExpression0
     *<li>UnaryExpression1
     *<li>UnaryExpressionNotPlusMinus0
     *<li>UnaryExpressionNotPlusMinus1
     *<li>CastExpression0
     *<li>CastExpression1
     *<li>MultiplicativeExpression0
     *<li>MultiplicativeExpression1
     *<li>MultiplicativeExpression2
     *<li>AdditiveExpression0
     *<li>AdditiveExpression1
     *<li>ShiftExpression0
     *<li>ShiftExpression1
     *<li>ShiftExpression2
     *<li>RelationalExpression0
     *<li>RelationalExpression1
     *<li>RelationalExpression2
     *<li>RelationalExpression3
     *<li>RelationalExpression4
     *<li>EqualityExpression0
     *<li>EqualityExpression1
     *</ul>
     *</b>
     */
    public interface IAssignmentExpression extends IExpression {}

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>identifier
     *<li>ExpressionName
     *<li>FieldAccess0
     *<li>FieldAccess1
     *<li>FieldAccess2
     *<li>ArrayAccess0
     *<li>ArrayAccess1
     *</ul>
     *</b>
     */
    public interface ILeftHandSide
    {
        public IToken getLeftIToken();
        public IToken getRightIToken();

        void accept(IAstVisitor v);
    }

    /**
     * is implemented by:
     *<b>
     *<ul>
     *<li>AssignmentOperator0
     *<li>AssignmentOperator1
     *<li>AssignmentOperator2
     *<li>AssignmentOperator3
     *<li>AssignmentOperator4
     *<li>AssignmentOperator5
     *<li>AssignmentOperator6
     *<li>AssignmentOperator7
     *<li>AssignmentOperator8
     *<li>AssignmentOperator9
     *<li>AssignmentOperator10
     *<li>AssignmentOperator11
     *</ul>
     *</b>
     */
    public interface IAssignmentOperator extends IAstToken {}

    /**
     *<b>
     *<li>Rule 3:  identifier ::= IDENTIFIER
     *</b>
     */
    static public class identifier extends AstToken implements Iidentifier
    {
        private JavaParser environment;
        public JavaParser getEnvironment() { return environment; }

        public IToken getIDENTIFIER() { return leftIToken; }

        public identifier(JavaParser environment, IToken token)        {
            super(token);
            this.environment = environment;
            initialize();
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }

                void initialize()
                {
                    if (getIDENTIFIER().getKind() != JavaParserprs.TK_IDENTIFIER)
                        System.out.println("Turning keyword " + getIDENTIFIER().toString() + " into an identifier");
                }
        }

    /**
     *<em>
     *<li>Rule 12:  PrimitiveType ::= NumericType
     *</em>
     *<p>
     *<b>
     *<li>Rule 13:  PrimitiveType ::= boolean
     *</b>
     */
    static public class PrimitiveType extends AstToken implements IPrimitiveType
    {
        public IToken getboolean() { return leftIToken; }

        public PrimitiveType(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 27:  ClassType ::= TypeName TypeArgumentsopt
     *</b>
     */
    static public class ClassType extends Ast implements IClassType
    {
        private ITypeName _TypeName;
        private TypeArguments _TypeArgumentsopt;

        public ITypeName getTypeName() { return _TypeName; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }

        public ClassType(IToken leftIToken, IToken rightIToken,
                         ITypeName _TypeName,
                         TypeArguments _TypeArgumentsopt)
        {
            super(leftIToken, rightIToken);

            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeName);
            list.add(_TypeArgumentsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassType)) return false;
            if (! super.equals(o)) return false;
            ClassType other = (ClassType) o;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeName.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 28:  InterfaceType ::= TypeName TypeArgumentsopt
     *</b>
     */
    static public class InterfaceType extends Ast implements IInterfaceType
    {
        private ITypeName _TypeName;
        private TypeArguments _TypeArgumentsopt;

        public ITypeName getTypeName() { return _TypeName; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }

        public InterfaceType(IToken leftIToken, IToken rightIToken,
                             ITypeName _TypeName,
                             TypeArguments _TypeArgumentsopt)
        {
            super(leftIToken, rightIToken);

            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeName);
            list.add(_TypeArgumentsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InterfaceType)) return false;
            if (! super.equals(o)) return false;
            InterfaceType other = (InterfaceType) o;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeName.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 29:  TypeName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 30:  TypeName ::= TypeName . identifier
     *</b>
     */
    static public class TypeName extends Ast implements ITypeName
    {
        private ITypeName _TypeName;
        private AstToken _DOT;
        private identifier _identifier;

        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public TypeName(IToken leftIToken, IToken rightIToken,
                        ITypeName _TypeName,
                        AstToken _DOT,
                        identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeName)) return false;
            if (! super.equals(o)) return false;
            TypeName other = (TypeName) o;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 33:  ArrayType ::= Type [ ]
     *</b>
     */
    static public class ArrayType extends Ast implements IArrayType
    {
        private IType _Type;
        private AstToken _LBRACKET;
        private AstToken _RBRACKET;

        public IType getType() { return _Type; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public ArrayType(IToken leftIToken, IToken rightIToken,
                         IType _Type,
                         AstToken _LBRACKET,
                         AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Type);
            list.add(_LBRACKET);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayType)) return false;
            if (! super.equals(o)) return false;
            ArrayType other = (ArrayType) o;
            if (! _Type.equals(other._Type)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Type.accept(v);
                _LBRACKET.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 34:  TypeParameter ::= TypeVariable TypeBoundopt
     *</b>
     */
    static public class TypeParameter extends Ast implements ITypeParameter
    {
        private identifier _TypeVariable;
        private TypeBound _TypeBoundopt;

        public identifier getTypeVariable() { return _TypeVariable; }
        /**
         * The value returned by <b>getTypeBoundopt</b> may be <b>null</b>
         */
        public TypeBound getTypeBoundopt() { return _TypeBoundopt; }

        public TypeParameter(IToken leftIToken, IToken rightIToken,
                             identifier _TypeVariable,
                             TypeBound _TypeBoundopt)
        {
            super(leftIToken, rightIToken);

            this._TypeVariable = _TypeVariable;
            ((Ast) _TypeVariable).setParent(this);
            this._TypeBoundopt = _TypeBoundopt;
            if (_TypeBoundopt != null) ((Ast) _TypeBoundopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeVariable);
            list.add(_TypeBoundopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeParameter)) return false;
            if (! super.equals(o)) return false;
            TypeParameter other = (TypeParameter) o;
            if (! _TypeVariable.equals(other._TypeVariable)) return false;
            if (_TypeBoundopt == null)
                if (other._TypeBoundopt != null) return false;
                else; // continue
            else if (! _TypeBoundopt.equals(other._TypeBoundopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeVariable.hashCode());
            hash = hash * 31 + (_TypeBoundopt == null ? 0 : _TypeBoundopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeVariable.accept(v);
                if (_TypeBoundopt != null) _TypeBoundopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 35:  TypeBound ::= extends ClassOrInterfaceType AdditionalBoundListopt
     *</b>
     */
    static public class TypeBound extends Ast implements ITypeBound
    {
        private AstToken _extends;
        private ClassType _ClassOrInterfaceType;
        private IAdditionalBoundListopt _AdditionalBoundListopt;

        public AstToken getextends() { return _extends; }
        public ClassType getClassOrInterfaceType() { return _ClassOrInterfaceType; }
        /**
         * The value returned by <b>getAdditionalBoundListopt</b> may be <b>null</b>
         */
        public IAdditionalBoundListopt getAdditionalBoundListopt() { return _AdditionalBoundListopt; }

        public TypeBound(IToken leftIToken, IToken rightIToken,
                         AstToken _extends,
                         ClassType _ClassOrInterfaceType,
                         IAdditionalBoundListopt _AdditionalBoundListopt)
        {
            super(leftIToken, rightIToken);

            this._extends = _extends;
            ((Ast) _extends).setParent(this);
            this._ClassOrInterfaceType = _ClassOrInterfaceType;
            ((Ast) _ClassOrInterfaceType).setParent(this);
            this._AdditionalBoundListopt = _AdditionalBoundListopt;
            if (_AdditionalBoundListopt != null) ((Ast) _AdditionalBoundListopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_extends);
            list.add(_ClassOrInterfaceType);
            list.add(_AdditionalBoundListopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeBound)) return false;
            if (! super.equals(o)) return false;
            TypeBound other = (TypeBound) o;
            if (! _extends.equals(other._extends)) return false;
            if (! _ClassOrInterfaceType.equals(other._ClassOrInterfaceType)) return false;
            if (_AdditionalBoundListopt == null)
                if (other._AdditionalBoundListopt != null) return false;
                else; // continue
            else if (! _AdditionalBoundListopt.equals(other._AdditionalBoundListopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_extends.hashCode());
            hash = hash * 31 + (_ClassOrInterfaceType.hashCode());
            hash = hash * 31 + (_AdditionalBoundListopt == null ? 0 : _AdditionalBoundListopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _extends.accept(v);
                _ClassOrInterfaceType.accept(v);
                if (_AdditionalBoundListopt != null) _AdditionalBoundListopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 36:  AdditionalBoundList ::= AdditionalBound
     *</em>
     *<p>
     *<b>
     *<li>Rule 37:  AdditionalBoundList ::= AdditionalBoundList AdditionalBound
     *</b>
     */
    static public class AdditionalBoundList extends Ast implements IAdditionalBoundList
    {
        private IAdditionalBoundList _AdditionalBoundList;
        private AdditionalBound _AdditionalBound;

        public IAdditionalBoundList getAdditionalBoundList() { return _AdditionalBoundList; }
        public AdditionalBound getAdditionalBound() { return _AdditionalBound; }

        public AdditionalBoundList(IToken leftIToken, IToken rightIToken,
                                   IAdditionalBoundList _AdditionalBoundList,
                                   AdditionalBound _AdditionalBound)
        {
            super(leftIToken, rightIToken);

            this._AdditionalBoundList = _AdditionalBoundList;
            ((Ast) _AdditionalBoundList).setParent(this);
            this._AdditionalBound = _AdditionalBound;
            ((Ast) _AdditionalBound).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AdditionalBoundList);
            list.add(_AdditionalBound);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AdditionalBoundList)) return false;
            if (! super.equals(o)) return false;
            AdditionalBoundList other = (AdditionalBoundList) o;
            if (! _AdditionalBoundList.equals(other._AdditionalBoundList)) return false;
            if (! _AdditionalBound.equals(other._AdditionalBound)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AdditionalBoundList.hashCode());
            hash = hash * 31 + (_AdditionalBound.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AdditionalBoundList.accept(v);
                _AdditionalBound.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 38:  AdditionalBound ::= & InterfaceType
     *</b>
     */
    static public class AdditionalBound extends Ast implements IAdditionalBound
    {
        private AstToken _AND;
        private InterfaceType _InterfaceType;

        public AstToken getAND() { return _AND; }
        public InterfaceType getInterfaceType() { return _InterfaceType; }

        public AdditionalBound(IToken leftIToken, IToken rightIToken,
                               AstToken _AND,
                               InterfaceType _InterfaceType)
        {
            super(leftIToken, rightIToken);

            this._AND = _AND;
            ((Ast) _AND).setParent(this);
            this._InterfaceType = _InterfaceType;
            ((Ast) _InterfaceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AND);
            list.add(_InterfaceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AdditionalBound)) return false;
            if (! super.equals(o)) return false;
            AdditionalBound other = (AdditionalBound) o;
            if (! _AND.equals(other._AND)) return false;
            if (! _InterfaceType.equals(other._InterfaceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AND.hashCode());
            hash = hash * 31 + (_InterfaceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AND.accept(v);
                _InterfaceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 39:  TypeArguments ::= < ActualTypeArgumentList >
     *</b>
     */
    static public class TypeArguments extends Ast implements ITypeArguments
    {
        private AstToken _LESS;
        private IActualTypeArgumentList _ActualTypeArgumentList;
        private AstToken _GREATER;

        public AstToken getLESS() { return _LESS; }
        public IActualTypeArgumentList getActualTypeArgumentList() { return _ActualTypeArgumentList; }
        public AstToken getGREATER() { return _GREATER; }

        public TypeArguments(IToken leftIToken, IToken rightIToken,
                             AstToken _LESS,
                             IActualTypeArgumentList _ActualTypeArgumentList,
                             AstToken _GREATER)
        {
            super(leftIToken, rightIToken);

            this._LESS = _LESS;
            ((Ast) _LESS).setParent(this);
            this._ActualTypeArgumentList = _ActualTypeArgumentList;
            ((Ast) _ActualTypeArgumentList).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LESS);
            list.add(_ActualTypeArgumentList);
            list.add(_GREATER);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeArguments)) return false;
            if (! super.equals(o)) return false;
            TypeArguments other = (TypeArguments) o;
            if (! _LESS.equals(other._LESS)) return false;
            if (! _ActualTypeArgumentList.equals(other._ActualTypeArgumentList)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LESS.hashCode());
            hash = hash * 31 + (_ActualTypeArgumentList.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LESS.accept(v);
                _ActualTypeArgumentList.accept(v);
                _GREATER.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 40:  ActualTypeArgumentList ::= ActualTypeArgument
     *</em>
     *<p>
     *<b>
     *<li>Rule 41:  ActualTypeArgumentList ::= ActualTypeArgumentList , ActualTypeArgument
     *</b>
     */
    static public class ActualTypeArgumentList extends Ast implements IActualTypeArgumentList
    {
        private IActualTypeArgumentList _ActualTypeArgumentList;
        private AstToken _COMMA;
        private IActualTypeArgument _ActualTypeArgument;

        public IActualTypeArgumentList getActualTypeArgumentList() { return _ActualTypeArgumentList; }
        public AstToken getCOMMA() { return _COMMA; }
        public IActualTypeArgument getActualTypeArgument() { return _ActualTypeArgument; }

        public ActualTypeArgumentList(IToken leftIToken, IToken rightIToken,
                                      IActualTypeArgumentList _ActualTypeArgumentList,
                                      AstToken _COMMA,
                                      IActualTypeArgument _ActualTypeArgument)
        {
            super(leftIToken, rightIToken);

            this._ActualTypeArgumentList = _ActualTypeArgumentList;
            ((Ast) _ActualTypeArgumentList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._ActualTypeArgument = _ActualTypeArgument;
            ((Ast) _ActualTypeArgument).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ActualTypeArgumentList);
            list.add(_COMMA);
            list.add(_ActualTypeArgument);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ActualTypeArgumentList)) return false;
            if (! super.equals(o)) return false;
            ActualTypeArgumentList other = (ActualTypeArgumentList) o;
            if (! _ActualTypeArgumentList.equals(other._ActualTypeArgumentList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _ActualTypeArgument.equals(other._ActualTypeArgument)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ActualTypeArgumentList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_ActualTypeArgument.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ActualTypeArgumentList.accept(v);
                _COMMA.accept(v);
                _ActualTypeArgument.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 44:  Wildcard ::= ? WildcardBoundsOpt
     *</b>
     */
    static public class Wildcard extends Ast implements IWildcard
    {
        private AstToken _QUESTION;
        private IWildcardBoundsOpt _WildcardBoundsOpt;

        public AstToken getQUESTION() { return _QUESTION; }
        /**
         * The value returned by <b>getWildcardBoundsOpt</b> may be <b>null</b>
         */
        public IWildcardBoundsOpt getWildcardBoundsOpt() { return _WildcardBoundsOpt; }

        public Wildcard(IToken leftIToken, IToken rightIToken,
                        AstToken _QUESTION,
                        IWildcardBoundsOpt _WildcardBoundsOpt)
        {
            super(leftIToken, rightIToken);

            this._QUESTION = _QUESTION;
            ((Ast) _QUESTION).setParent(this);
            this._WildcardBoundsOpt = _WildcardBoundsOpt;
            if (_WildcardBoundsOpt != null) ((Ast) _WildcardBoundsOpt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_QUESTION);
            list.add(_WildcardBoundsOpt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Wildcard)) return false;
            if (! super.equals(o)) return false;
            Wildcard other = (Wildcard) o;
            if (! _QUESTION.equals(other._QUESTION)) return false;
            if (_WildcardBoundsOpt == null)
                if (other._WildcardBoundsOpt != null) return false;
                else; // continue
            else if (! _WildcardBoundsOpt.equals(other._WildcardBoundsOpt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_QUESTION.hashCode());
            hash = hash * 31 + (_WildcardBoundsOpt == null ? 0 : _WildcardBoundsOpt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _QUESTION.accept(v);
                if (_WildcardBoundsOpt != null) _WildcardBoundsOpt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 47:  PackageName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 48:  PackageName ::= PackageName . identifier
     *</b>
     */
    static public class PackageName extends Ast implements IPackageName
    {
        private IPackageName _PackageName;
        private AstToken _DOT;
        private identifier _identifier;

        public IPackageName getPackageName() { return _PackageName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public PackageName(IToken leftIToken, IToken rightIToken,
                           IPackageName _PackageName,
                           AstToken _DOT,
                           identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._PackageName = _PackageName;
            ((Ast) _PackageName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PackageName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PackageName)) return false;
            if (! super.equals(o)) return false;
            PackageName other = (PackageName) o;
            if (! _PackageName.equals(other._PackageName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PackageName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PackageName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 49:  ExpressionName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 50:  ExpressionName ::= AmbiguousName . identifier
     *</b>
     */
    static public class ExpressionName extends Ast implements IExpressionName
    {
        private IAmbiguousName _AmbiguousName;
        private AstToken _DOT;
        private identifier _identifier;

        public IAmbiguousName getAmbiguousName() { return _AmbiguousName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public ExpressionName(IToken leftIToken, IToken rightIToken,
                              IAmbiguousName _AmbiguousName,
                              AstToken _DOT,
                              identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._AmbiguousName = _AmbiguousName;
            ((Ast) _AmbiguousName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AmbiguousName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExpressionName)) return false;
            if (! super.equals(o)) return false;
            ExpressionName other = (ExpressionName) o;
            if (! _AmbiguousName.equals(other._AmbiguousName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AmbiguousName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AmbiguousName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 51:  MethodName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 52:  MethodName ::= AmbiguousName . identifier
     *</b>
     */
    static public class MethodName extends Ast implements IMethodName
    {
        private IAmbiguousName _AmbiguousName;
        private AstToken _DOT;
        private identifier _identifier;

        public IAmbiguousName getAmbiguousName() { return _AmbiguousName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public MethodName(IToken leftIToken, IToken rightIToken,
                          IAmbiguousName _AmbiguousName,
                          AstToken _DOT,
                          identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._AmbiguousName = _AmbiguousName;
            ((Ast) _AmbiguousName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AmbiguousName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodName)) return false;
            if (! super.equals(o)) return false;
            MethodName other = (MethodName) o;
            if (! _AmbiguousName.equals(other._AmbiguousName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AmbiguousName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AmbiguousName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 53:  PackageOrTypeName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 54:  PackageOrTypeName ::= PackageOrTypeName . identifier
     *</b>
     */
    static public class PackageOrTypeName extends Ast implements IPackageOrTypeName
    {
        private IPackageOrTypeName _PackageOrTypeName;
        private AstToken _DOT;
        private identifier _identifier;

        public IPackageOrTypeName getPackageOrTypeName() { return _PackageOrTypeName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public PackageOrTypeName(IToken leftIToken, IToken rightIToken,
                                 IPackageOrTypeName _PackageOrTypeName,
                                 AstToken _DOT,
                                 identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._PackageOrTypeName = _PackageOrTypeName;
            ((Ast) _PackageOrTypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PackageOrTypeName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PackageOrTypeName)) return false;
            if (! super.equals(o)) return false;
            PackageOrTypeName other = (PackageOrTypeName) o;
            if (! _PackageOrTypeName.equals(other._PackageOrTypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PackageOrTypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PackageOrTypeName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 55:  AmbiguousName ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 56:  AmbiguousName ::= AmbiguousName . identifier
     *</b>
     */
    static public class AmbiguousName extends Ast implements IAmbiguousName
    {
        private IAmbiguousName _AmbiguousName;
        private AstToken _DOT;
        private identifier _identifier;

        public IAmbiguousName getAmbiguousName() { return _AmbiguousName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public AmbiguousName(IToken leftIToken, IToken rightIToken,
                             IAmbiguousName _AmbiguousName,
                             AstToken _DOT,
                             identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._AmbiguousName = _AmbiguousName;
            ((Ast) _AmbiguousName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AmbiguousName);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AmbiguousName)) return false;
            if (! super.equals(o)) return false;
            AmbiguousName other = (AmbiguousName) o;
            if (! _AmbiguousName.equals(other._AmbiguousName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AmbiguousName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AmbiguousName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 57:  CompilationUnit ::= PackageDeclarationopt ImportDeclarationsopt TypeDeclarationsopt
     *</b>
     */
    static public class CompilationUnit extends Ast implements ICompilationUnit
    {
        private PackageDeclaration _PackageDeclarationopt;
        private IImportDeclarationsopt _ImportDeclarationsopt;
        private ITypeDeclarationsopt _TypeDeclarationsopt;

        /**
         * The value returned by <b>getPackageDeclarationopt</b> may be <b>null</b>
         */
        public PackageDeclaration getPackageDeclarationopt() { return _PackageDeclarationopt; }
        /**
         * The value returned by <b>getImportDeclarationsopt</b> may be <b>null</b>
         */
        public IImportDeclarationsopt getImportDeclarationsopt() { return _ImportDeclarationsopt; }
        /**
         * The value returned by <b>getTypeDeclarationsopt</b> may be <b>null</b>
         */
        public ITypeDeclarationsopt getTypeDeclarationsopt() { return _TypeDeclarationsopt; }

        public CompilationUnit(IToken leftIToken, IToken rightIToken,
                               PackageDeclaration _PackageDeclarationopt,
                               IImportDeclarationsopt _ImportDeclarationsopt,
                               ITypeDeclarationsopt _TypeDeclarationsopt)
        {
            super(leftIToken, rightIToken);

            this._PackageDeclarationopt = _PackageDeclarationopt;
            if (_PackageDeclarationopt != null) ((Ast) _PackageDeclarationopt).setParent(this);
            this._ImportDeclarationsopt = _ImportDeclarationsopt;
            if (_ImportDeclarationsopt != null) ((Ast) _ImportDeclarationsopt).setParent(this);
            this._TypeDeclarationsopt = _TypeDeclarationsopt;
            if (_TypeDeclarationsopt != null) ((Ast) _TypeDeclarationsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PackageDeclarationopt);
            list.add(_ImportDeclarationsopt);
            list.add(_TypeDeclarationsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof CompilationUnit)) return false;
            if (! super.equals(o)) return false;
            CompilationUnit other = (CompilationUnit) o;
            if (_PackageDeclarationopt == null)
                if (other._PackageDeclarationopt != null) return false;
                else; // continue
            else if (! _PackageDeclarationopt.equals(other._PackageDeclarationopt)) return false;
            if (_ImportDeclarationsopt == null)
                if (other._ImportDeclarationsopt != null) return false;
                else; // continue
            else if (! _ImportDeclarationsopt.equals(other._ImportDeclarationsopt)) return false;
            if (_TypeDeclarationsopt == null)
                if (other._TypeDeclarationsopt != null) return false;
                else; // continue
            else if (! _TypeDeclarationsopt.equals(other._TypeDeclarationsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PackageDeclarationopt == null ? 0 : _PackageDeclarationopt.hashCode());
            hash = hash * 31 + (_ImportDeclarationsopt == null ? 0 : _ImportDeclarationsopt.hashCode());
            hash = hash * 31 + (_TypeDeclarationsopt == null ? 0 : _TypeDeclarationsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_PackageDeclarationopt != null) _PackageDeclarationopt.accept(v);
                if (_ImportDeclarationsopt != null) _ImportDeclarationsopt.accept(v);
                if (_TypeDeclarationsopt != null) _TypeDeclarationsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 58:  ImportDeclarations ::= ImportDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 59:  ImportDeclarations ::= ImportDeclarations ImportDeclaration
     *</b>
     */
    static public class ImportDeclarations extends Ast implements IImportDeclarations
    {
        private IImportDeclarations _ImportDeclarations;
        private IImportDeclaration _ImportDeclaration;

        public IImportDeclarations getImportDeclarations() { return _ImportDeclarations; }
        public IImportDeclaration getImportDeclaration() { return _ImportDeclaration; }

        public ImportDeclarations(IToken leftIToken, IToken rightIToken,
                                  IImportDeclarations _ImportDeclarations,
                                  IImportDeclaration _ImportDeclaration)
        {
            super(leftIToken, rightIToken);

            this._ImportDeclarations = _ImportDeclarations;
            ((Ast) _ImportDeclarations).setParent(this);
            this._ImportDeclaration = _ImportDeclaration;
            ((Ast) _ImportDeclaration).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ImportDeclarations);
            list.add(_ImportDeclaration);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ImportDeclarations)) return false;
            if (! super.equals(o)) return false;
            ImportDeclarations other = (ImportDeclarations) o;
            if (! _ImportDeclarations.equals(other._ImportDeclarations)) return false;
            if (! _ImportDeclaration.equals(other._ImportDeclaration)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ImportDeclarations.hashCode());
            hash = hash * 31 + (_ImportDeclaration.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ImportDeclarations.accept(v);
                _ImportDeclaration.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 60:  TypeDeclarations ::= TypeDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 61:  TypeDeclarations ::= TypeDeclarations TypeDeclaration
     *</b>
     */
    static public class TypeDeclarations extends Ast implements ITypeDeclarations
    {
        private ITypeDeclarations _TypeDeclarations;
        private ITypeDeclaration _TypeDeclaration;

        public ITypeDeclarations getTypeDeclarations() { return _TypeDeclarations; }
        public ITypeDeclaration getTypeDeclaration() { return _TypeDeclaration; }

        public TypeDeclarations(IToken leftIToken, IToken rightIToken,
                                ITypeDeclarations _TypeDeclarations,
                                ITypeDeclaration _TypeDeclaration)
        {
            super(leftIToken, rightIToken);

            this._TypeDeclarations = _TypeDeclarations;
            ((Ast) _TypeDeclarations).setParent(this);
            this._TypeDeclaration = _TypeDeclaration;
            ((Ast) _TypeDeclaration).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeDeclarations);
            list.add(_TypeDeclaration);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeDeclarations)) return false;
            if (! super.equals(o)) return false;
            TypeDeclarations other = (TypeDeclarations) o;
            if (! _TypeDeclarations.equals(other._TypeDeclarations)) return false;
            if (! _TypeDeclaration.equals(other._TypeDeclaration)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeDeclarations.hashCode());
            hash = hash * 31 + (_TypeDeclaration.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeDeclarations.accept(v);
                _TypeDeclaration.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 62:  PackageDeclaration ::= Annotationsopt package PackageName ;
     *</b>
     */
    static public class PackageDeclaration extends Ast implements IPackageDeclaration
    {
        private IAnnotationsopt _Annotationsopt;
        private AstToken _package;
        private IPackageName _PackageName;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getAnnotationsopt</b> may be <b>null</b>
         */
        public IAnnotationsopt getAnnotationsopt() { return _Annotationsopt; }
        public AstToken getpackage() { return _package; }
        public IPackageName getPackageName() { return _PackageName; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public PackageDeclaration(IToken leftIToken, IToken rightIToken,
                                  IAnnotationsopt _Annotationsopt,
                                  AstToken _package,
                                  IPackageName _PackageName,
                                  AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._Annotationsopt = _Annotationsopt;
            if (_Annotationsopt != null) ((Ast) _Annotationsopt).setParent(this);
            this._package = _package;
            ((Ast) _package).setParent(this);
            this._PackageName = _PackageName;
            ((Ast) _PackageName).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Annotationsopt);
            list.add(_package);
            list.add(_PackageName);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PackageDeclaration)) return false;
            if (! super.equals(o)) return false;
            PackageDeclaration other = (PackageDeclaration) o;
            if (_Annotationsopt == null)
                if (other._Annotationsopt != null) return false;
                else; // continue
            else if (! _Annotationsopt.equals(other._Annotationsopt)) return false;
            if (! _package.equals(other._package)) return false;
            if (! _PackageName.equals(other._PackageName)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Annotationsopt == null ? 0 : _Annotationsopt.hashCode());
            hash = hash * 31 + (_package.hashCode());
            hash = hash * 31 + (_PackageName.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_Annotationsopt != null) _Annotationsopt.accept(v);
                _package.accept(v);
                _PackageName.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 67:  SingleTypeImportDeclaration ::= import TypeName ;
     *</b>
     */
    static public class SingleTypeImportDeclaration extends Ast implements ISingleTypeImportDeclaration
    {
        private AstToken _import;
        private ITypeName _TypeName;
        private AstToken _SEMICOLON;

        public AstToken getimport() { return _import; }
        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public SingleTypeImportDeclaration(IToken leftIToken, IToken rightIToken,
                                           AstToken _import,
                                           ITypeName _TypeName,
                                           AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._import = _import;
            ((Ast) _import).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_import);
            list.add(_TypeName);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SingleTypeImportDeclaration)) return false;
            if (! super.equals(o)) return false;
            SingleTypeImportDeclaration other = (SingleTypeImportDeclaration) o;
            if (! _import.equals(other._import)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_import.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _import.accept(v);
                _TypeName.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 68:  TypeImportOnDemandDeclaration ::= import PackageOrTypeName . * ;
     *</b>
     */
    static public class TypeImportOnDemandDeclaration extends Ast implements ITypeImportOnDemandDeclaration
    {
        private AstToken _import;
        private IPackageOrTypeName _PackageOrTypeName;
        private AstToken _DOT;
        private AstToken _MULTIPLY;
        private AstToken _SEMICOLON;

        public AstToken getimport() { return _import; }
        public IPackageOrTypeName getPackageOrTypeName() { return _PackageOrTypeName; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getMULTIPLY() { return _MULTIPLY; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public TypeImportOnDemandDeclaration(IToken leftIToken, IToken rightIToken,
                                             AstToken _import,
                                             IPackageOrTypeName _PackageOrTypeName,
                                             AstToken _DOT,
                                             AstToken _MULTIPLY,
                                             AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._import = _import;
            ((Ast) _import).setParent(this);
            this._PackageOrTypeName = _PackageOrTypeName;
            ((Ast) _PackageOrTypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._MULTIPLY = _MULTIPLY;
            ((Ast) _MULTIPLY).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_import);
            list.add(_PackageOrTypeName);
            list.add(_DOT);
            list.add(_MULTIPLY);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeImportOnDemandDeclaration)) return false;
            if (! super.equals(o)) return false;
            TypeImportOnDemandDeclaration other = (TypeImportOnDemandDeclaration) o;
            if (! _import.equals(other._import)) return false;
            if (! _PackageOrTypeName.equals(other._PackageOrTypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _MULTIPLY.equals(other._MULTIPLY)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_import.hashCode());
            hash = hash * 31 + (_PackageOrTypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_MULTIPLY.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _import.accept(v);
                _PackageOrTypeName.accept(v);
                _DOT.accept(v);
                _MULTIPLY.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 69:  SingleStaticImportDeclaration ::= import static TypeName . identifier ;
     *</b>
     */
    static public class SingleStaticImportDeclaration extends Ast implements ISingleStaticImportDeclaration
    {
        private AstToken _import;
        private AstToken _static;
        private ITypeName _TypeName;
        private AstToken _DOT;
        private identifier _identifier;
        private AstToken _SEMICOLON;

        public AstToken getimport() { return _import; }
        public AstToken getstatic() { return _static; }
        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public SingleStaticImportDeclaration(IToken leftIToken, IToken rightIToken,
                                             AstToken _import,
                                             AstToken _static,
                                             ITypeName _TypeName,
                                             AstToken _DOT,
                                             identifier _identifier,
                                             AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._import = _import;
            ((Ast) _import).setParent(this);
            this._static = _static;
            ((Ast) _static).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_import);
            list.add(_static);
            list.add(_TypeName);
            list.add(_DOT);
            list.add(_identifier);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SingleStaticImportDeclaration)) return false;
            if (! super.equals(o)) return false;
            SingleStaticImportDeclaration other = (SingleStaticImportDeclaration) o;
            if (! _import.equals(other._import)) return false;
            if (! _static.equals(other._static)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_import.hashCode());
            hash = hash * 31 + (_static.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _import.accept(v);
                _static.accept(v);
                _TypeName.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 70:  StaticImportOnDemandDeclaration ::= import static TypeName . * ;
     *</b>
     */
    static public class StaticImportOnDemandDeclaration extends Ast implements IStaticImportOnDemandDeclaration
    {
        private AstToken _import;
        private AstToken _static;
        private ITypeName _TypeName;
        private AstToken _DOT;
        private AstToken _MULTIPLY;
        private AstToken _SEMICOLON;

        public AstToken getimport() { return _import; }
        public AstToken getstatic() { return _static; }
        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getMULTIPLY() { return _MULTIPLY; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public StaticImportOnDemandDeclaration(IToken leftIToken, IToken rightIToken,
                                               AstToken _import,
                                               AstToken _static,
                                               ITypeName _TypeName,
                                               AstToken _DOT,
                                               AstToken _MULTIPLY,
                                               AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._import = _import;
            ((Ast) _import).setParent(this);
            this._static = _static;
            ((Ast) _static).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._MULTIPLY = _MULTIPLY;
            ((Ast) _MULTIPLY).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_import);
            list.add(_static);
            list.add(_TypeName);
            list.add(_DOT);
            list.add(_MULTIPLY);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof StaticImportOnDemandDeclaration)) return false;
            if (! super.equals(o)) return false;
            StaticImportOnDemandDeclaration other = (StaticImportOnDemandDeclaration) o;
            if (! _import.equals(other._import)) return false;
            if (! _static.equals(other._static)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _MULTIPLY.equals(other._MULTIPLY)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_import.hashCode());
            hash = hash * 31 + (_static.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_MULTIPLY.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _import.accept(v);
                _static.accept(v);
                _TypeName.accept(v);
                _DOT.accept(v);
                _MULTIPLY.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 71:  TypeDeclaration ::= ClassDeclaration
     *<li>Rule 72:  TypeDeclaration ::= InterfaceDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 73:  TypeDeclaration ::= ;
     *</b>
     */
    static public class TypeDeclaration extends AstToken implements ITypeDeclaration
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public TypeDeclaration(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 76:  NormalClassDeclaration ::= ClassModifiersopt class identifier TypeParametersopt Superopt Interfacesopt ClassBody
     *</b>
     */
    static public class NormalClassDeclaration extends Ast implements INormalClassDeclaration
    {
        private IClassModifiersopt _ClassModifiersopt;
        private AstToken _class;
        private identifier _identifier;
        private TypeParameters _TypeParametersopt;
        private Super _Superopt;
        private Interfaces _Interfacesopt;
        private ClassBody _ClassBody;

        /**
         * The value returned by <b>getClassModifiersopt</b> may be <b>null</b>
         */
        public IClassModifiersopt getClassModifiersopt() { return _ClassModifiersopt; }
        public AstToken getclass() { return _class; }
        public identifier getidentifier() { return _identifier; }
        /**
         * The value returned by <b>getTypeParametersopt</b> may be <b>null</b>
         */
        public TypeParameters getTypeParametersopt() { return _TypeParametersopt; }
        /**
         * The value returned by <b>getSuperopt</b> may be <b>null</b>
         */
        public Super getSuperopt() { return _Superopt; }
        /**
         * The value returned by <b>getInterfacesopt</b> may be <b>null</b>
         */
        public Interfaces getInterfacesopt() { return _Interfacesopt; }
        public ClassBody getClassBody() { return _ClassBody; }

        public NormalClassDeclaration(IToken leftIToken, IToken rightIToken,
                                      IClassModifiersopt _ClassModifiersopt,
                                      AstToken _class,
                                      identifier _identifier,
                                      TypeParameters _TypeParametersopt,
                                      Super _Superopt,
                                      Interfaces _Interfacesopt,
                                      ClassBody _ClassBody)
        {
            super(leftIToken, rightIToken);

            this._ClassModifiersopt = _ClassModifiersopt;
            if (_ClassModifiersopt != null) ((Ast) _ClassModifiersopt).setParent(this);
            this._class = _class;
            ((Ast) _class).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._TypeParametersopt = _TypeParametersopt;
            if (_TypeParametersopt != null) ((Ast) _TypeParametersopt).setParent(this);
            this._Superopt = _Superopt;
            if (_Superopt != null) ((Ast) _Superopt).setParent(this);
            this._Interfacesopt = _Interfacesopt;
            if (_Interfacesopt != null) ((Ast) _Interfacesopt).setParent(this);
            this._ClassBody = _ClassBody;
            ((Ast) _ClassBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassModifiersopt);
            list.add(_class);
            list.add(_identifier);
            list.add(_TypeParametersopt);
            list.add(_Superopt);
            list.add(_Interfacesopt);
            list.add(_ClassBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof NormalClassDeclaration)) return false;
            if (! super.equals(o)) return false;
            NormalClassDeclaration other = (NormalClassDeclaration) o;
            if (_ClassModifiersopt == null)
                if (other._ClassModifiersopt != null) return false;
                else; // continue
            else if (! _ClassModifiersopt.equals(other._ClassModifiersopt)) return false;
            if (! _class.equals(other._class)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (_TypeParametersopt == null)
                if (other._TypeParametersopt != null) return false;
                else; // continue
            else if (! _TypeParametersopt.equals(other._TypeParametersopt)) return false;
            if (_Superopt == null)
                if (other._Superopt != null) return false;
                else; // continue
            else if (! _Superopt.equals(other._Superopt)) return false;
            if (_Interfacesopt == null)
                if (other._Interfacesopt != null) return false;
                else; // continue
            else if (! _Interfacesopt.equals(other._Interfacesopt)) return false;
            if (! _ClassBody.equals(other._ClassBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassModifiersopt == null ? 0 : _ClassModifiersopt.hashCode());
            hash = hash * 31 + (_class.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_TypeParametersopt == null ? 0 : _TypeParametersopt.hashCode());
            hash = hash * 31 + (_Superopt == null ? 0 : _Superopt.hashCode());
            hash = hash * 31 + (_Interfacesopt == null ? 0 : _Interfacesopt.hashCode());
            hash = hash * 31 + (_ClassBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_ClassModifiersopt != null) _ClassModifiersopt.accept(v);
                _class.accept(v);
                _identifier.accept(v);
                if (_TypeParametersopt != null) _TypeParametersopt.accept(v);
                if (_Superopt != null) _Superopt.accept(v);
                if (_Interfacesopt != null) _Interfacesopt.accept(v);
                _ClassBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 77:  ClassModifiers ::= ClassModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 78:  ClassModifiers ::= ClassModifiers ClassModifier
     *</b>
     */
    static public class ClassModifiers extends Ast implements IClassModifiers
    {
        private IClassModifiers _ClassModifiers;
        private IClassModifier _ClassModifier;

        public IClassModifiers getClassModifiers() { return _ClassModifiers; }
        public IClassModifier getClassModifier() { return _ClassModifier; }

        public ClassModifiers(IToken leftIToken, IToken rightIToken,
                              IClassModifiers _ClassModifiers,
                              IClassModifier _ClassModifier)
        {
            super(leftIToken, rightIToken);

            this._ClassModifiers = _ClassModifiers;
            ((Ast) _ClassModifiers).setParent(this);
            this._ClassModifier = _ClassModifier;
            ((Ast) _ClassModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassModifiers);
            list.add(_ClassModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassModifiers)) return false;
            if (! super.equals(o)) return false;
            ClassModifiers other = (ClassModifiers) o;
            if (! _ClassModifiers.equals(other._ClassModifiers)) return false;
            if (! _ClassModifier.equals(other._ClassModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassModifiers.hashCode());
            hash = hash * 31 + (_ClassModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ClassModifiers.accept(v);
                _ClassModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 87:  TypeParameters ::= < TypeParameterList >
     *</b>
     */
    static public class TypeParameters extends Ast implements ITypeParameters
    {
        private AstToken _LESS;
        private ITypeParameterList _TypeParameterList;
        private AstToken _GREATER;

        public AstToken getLESS() { return _LESS; }
        public ITypeParameterList getTypeParameterList() { return _TypeParameterList; }
        public AstToken getGREATER() { return _GREATER; }

        public TypeParameters(IToken leftIToken, IToken rightIToken,
                              AstToken _LESS,
                              ITypeParameterList _TypeParameterList,
                              AstToken _GREATER)
        {
            super(leftIToken, rightIToken);

            this._LESS = _LESS;
            ((Ast) _LESS).setParent(this);
            this._TypeParameterList = _TypeParameterList;
            ((Ast) _TypeParameterList).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LESS);
            list.add(_TypeParameterList);
            list.add(_GREATER);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeParameters)) return false;
            if (! super.equals(o)) return false;
            TypeParameters other = (TypeParameters) o;
            if (! _LESS.equals(other._LESS)) return false;
            if (! _TypeParameterList.equals(other._TypeParameterList)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LESS.hashCode());
            hash = hash * 31 + (_TypeParameterList.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LESS.accept(v);
                _TypeParameterList.accept(v);
                _GREATER.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 88:  TypeParameterList ::= TypeParameter
     *</em>
     *<p>
     *<b>
     *<li>Rule 89:  TypeParameterList ::= TypeParameterList , TypeParameter
     *</b>
     */
    static public class TypeParameterList extends Ast implements ITypeParameterList
    {
        private ITypeParameterList _TypeParameterList;
        private AstToken _COMMA;
        private TypeParameter _TypeParameter;

        public ITypeParameterList getTypeParameterList() { return _TypeParameterList; }
        public AstToken getCOMMA() { return _COMMA; }
        public TypeParameter getTypeParameter() { return _TypeParameter; }

        public TypeParameterList(IToken leftIToken, IToken rightIToken,
                                 ITypeParameterList _TypeParameterList,
                                 AstToken _COMMA,
                                 TypeParameter _TypeParameter)
        {
            super(leftIToken, rightIToken);

            this._TypeParameterList = _TypeParameterList;
            ((Ast) _TypeParameterList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._TypeParameter = _TypeParameter;
            ((Ast) _TypeParameter).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeParameterList);
            list.add(_COMMA);
            list.add(_TypeParameter);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TypeParameterList)) return false;
            if (! super.equals(o)) return false;
            TypeParameterList other = (TypeParameterList) o;
            if (! _TypeParameterList.equals(other._TypeParameterList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _TypeParameter.equals(other._TypeParameter)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeParameterList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_TypeParameter.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeParameterList.accept(v);
                _COMMA.accept(v);
                _TypeParameter.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 90:  Super ::= extends ClassType
     *</b>
     */
    static public class Super extends Ast implements ISuper
    {
        private AstToken _extends;
        private ClassType _ClassType;

        public AstToken getextends() { return _extends; }
        public ClassType getClassType() { return _ClassType; }

        public Super(IToken leftIToken, IToken rightIToken,
                     AstToken _extends,
                     ClassType _ClassType)
        {
            super(leftIToken, rightIToken);

            this._extends = _extends;
            ((Ast) _extends).setParent(this);
            this._ClassType = _ClassType;
            ((Ast) _ClassType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_extends);
            list.add(_ClassType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Super)) return false;
            if (! super.equals(o)) return false;
            Super other = (Super) o;
            if (! _extends.equals(other._extends)) return false;
            if (! _ClassType.equals(other._ClassType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_extends.hashCode());
            hash = hash * 31 + (_ClassType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _extends.accept(v);
                _ClassType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 91:  Interfaces ::= implements InterfaceTypeList
     *</b>
     */
    static public class Interfaces extends Ast implements IInterfaces
    {
        private AstToken _implements;
        private IInterfaceTypeList _InterfaceTypeList;

        public AstToken getimplements() { return _implements; }
        public IInterfaceTypeList getInterfaceTypeList() { return _InterfaceTypeList; }

        public Interfaces(IToken leftIToken, IToken rightIToken,
                          AstToken _implements,
                          IInterfaceTypeList _InterfaceTypeList)
        {
            super(leftIToken, rightIToken);

            this._implements = _implements;
            ((Ast) _implements).setParent(this);
            this._InterfaceTypeList = _InterfaceTypeList;
            ((Ast) _InterfaceTypeList).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_implements);
            list.add(_InterfaceTypeList);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Interfaces)) return false;
            if (! super.equals(o)) return false;
            Interfaces other = (Interfaces) o;
            if (! _implements.equals(other._implements)) return false;
            if (! _InterfaceTypeList.equals(other._InterfaceTypeList)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_implements.hashCode());
            hash = hash * 31 + (_InterfaceTypeList.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _implements.accept(v);
                _InterfaceTypeList.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 92:  InterfaceTypeList ::= InterfaceType
     *</em>
     *<p>
     *<b>
     *<li>Rule 93:  InterfaceTypeList ::= InterfaceTypeList , InterfaceType
     *</b>
     */
    static public class InterfaceTypeList extends Ast implements IInterfaceTypeList
    {
        private IInterfaceTypeList _InterfaceTypeList;
        private AstToken _COMMA;
        private InterfaceType _InterfaceType;

        public IInterfaceTypeList getInterfaceTypeList() { return _InterfaceTypeList; }
        public AstToken getCOMMA() { return _COMMA; }
        public InterfaceType getInterfaceType() { return _InterfaceType; }

        public InterfaceTypeList(IToken leftIToken, IToken rightIToken,
                                 IInterfaceTypeList _InterfaceTypeList,
                                 AstToken _COMMA,
                                 InterfaceType _InterfaceType)
        {
            super(leftIToken, rightIToken);

            this._InterfaceTypeList = _InterfaceTypeList;
            ((Ast) _InterfaceTypeList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._InterfaceType = _InterfaceType;
            ((Ast) _InterfaceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InterfaceTypeList);
            list.add(_COMMA);
            list.add(_InterfaceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InterfaceTypeList)) return false;
            if (! super.equals(o)) return false;
            InterfaceTypeList other = (InterfaceTypeList) o;
            if (! _InterfaceTypeList.equals(other._InterfaceTypeList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _InterfaceType.equals(other._InterfaceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InterfaceTypeList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_InterfaceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _InterfaceTypeList.accept(v);
                _COMMA.accept(v);
                _InterfaceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 94:  ClassBody ::= { ClassBodyDeclarationsopt }
     *</b>
     */
    static public class ClassBody extends Ast implements IClassBody
    {
        private AstToken _LBRACE;
        private IClassBodyDeclarationsopt _ClassBodyDeclarationsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getClassBodyDeclarationsopt</b> may be <b>null</b>
         */
        public IClassBodyDeclarationsopt getClassBodyDeclarationsopt() { return _ClassBodyDeclarationsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public ClassBody(IToken leftIToken, IToken rightIToken,
                         AstToken _LBRACE,
                         IClassBodyDeclarationsopt _ClassBodyDeclarationsopt,
                         AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._ClassBodyDeclarationsopt = _ClassBodyDeclarationsopt;
            if (_ClassBodyDeclarationsopt != null) ((Ast) _ClassBodyDeclarationsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_ClassBodyDeclarationsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassBody)) return false;
            if (! super.equals(o)) return false;
            ClassBody other = (ClassBody) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_ClassBodyDeclarationsopt == null)
                if (other._ClassBodyDeclarationsopt != null) return false;
                else; // continue
            else if (! _ClassBodyDeclarationsopt.equals(other._ClassBodyDeclarationsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_ClassBodyDeclarationsopt == null ? 0 : _ClassBodyDeclarationsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_ClassBodyDeclarationsopt != null) _ClassBodyDeclarationsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 95:  ClassBodyDeclarations ::= ClassBodyDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 96:  ClassBodyDeclarations ::= ClassBodyDeclarations ClassBodyDeclaration
     *</b>
     */
    static public class ClassBodyDeclarations extends Ast implements IClassBodyDeclarations
    {
        private IClassBodyDeclarations _ClassBodyDeclarations;
        private IClassBodyDeclaration _ClassBodyDeclaration;

        public IClassBodyDeclarations getClassBodyDeclarations() { return _ClassBodyDeclarations; }
        public IClassBodyDeclaration getClassBodyDeclaration() { return _ClassBodyDeclaration; }

        public ClassBodyDeclarations(IToken leftIToken, IToken rightIToken,
                                     IClassBodyDeclarations _ClassBodyDeclarations,
                                     IClassBodyDeclaration _ClassBodyDeclaration)
        {
            super(leftIToken, rightIToken);

            this._ClassBodyDeclarations = _ClassBodyDeclarations;
            ((Ast) _ClassBodyDeclarations).setParent(this);
            this._ClassBodyDeclaration = _ClassBodyDeclaration;
            ((Ast) _ClassBodyDeclaration).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassBodyDeclarations);
            list.add(_ClassBodyDeclaration);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassBodyDeclarations)) return false;
            if (! super.equals(o)) return false;
            ClassBodyDeclarations other = (ClassBodyDeclarations) o;
            if (! _ClassBodyDeclarations.equals(other._ClassBodyDeclarations)) return false;
            if (! _ClassBodyDeclaration.equals(other._ClassBodyDeclaration)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassBodyDeclarations.hashCode());
            hash = hash * 31 + (_ClassBodyDeclaration.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ClassBodyDeclarations.accept(v);
                _ClassBodyDeclaration.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 101:  ClassMemberDeclaration ::= FieldDeclaration
     *<li>Rule 102:  ClassMemberDeclaration ::= MethodDeclaration
     *<li>Rule 103:  ClassMemberDeclaration ::= ClassDeclaration
     *<li>Rule 104:  ClassMemberDeclaration ::= InterfaceDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 105:  ClassMemberDeclaration ::= ;
     *</b>
     */
    static public class ClassMemberDeclaration extends AstToken implements IClassMemberDeclaration
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public ClassMemberDeclaration(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 106:  FieldDeclaration ::= FieldModifiersopt Type VariableDeclarators ;
     *</b>
     */
    static public class FieldDeclaration extends Ast implements IFieldDeclaration
    {
        private IFieldModifiersopt _FieldModifiersopt;
        private IType _Type;
        private IVariableDeclarators _VariableDeclarators;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getFieldModifiersopt</b> may be <b>null</b>
         */
        public IFieldModifiersopt getFieldModifiersopt() { return _FieldModifiersopt; }
        public IType getType() { return _Type; }
        public IVariableDeclarators getVariableDeclarators() { return _VariableDeclarators; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public FieldDeclaration(IToken leftIToken, IToken rightIToken,
                                IFieldModifiersopt _FieldModifiersopt,
                                IType _Type,
                                IVariableDeclarators _VariableDeclarators,
                                AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._FieldModifiersopt = _FieldModifiersopt;
            if (_FieldModifiersopt != null) ((Ast) _FieldModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._VariableDeclarators = _VariableDeclarators;
            ((Ast) _VariableDeclarators).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_FieldModifiersopt);
            list.add(_Type);
            list.add(_VariableDeclarators);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FieldDeclaration)) return false;
            if (! super.equals(o)) return false;
            FieldDeclaration other = (FieldDeclaration) o;
            if (_FieldModifiersopt == null)
                if (other._FieldModifiersopt != null) return false;
                else; // continue
            else if (! _FieldModifiersopt.equals(other._FieldModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (! _VariableDeclarators.equals(other._VariableDeclarators)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_FieldModifiersopt == null ? 0 : _FieldModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_VariableDeclarators.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_FieldModifiersopt != null) _FieldModifiersopt.accept(v);
                _Type.accept(v);
                _VariableDeclarators.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 107:  VariableDeclarators ::= VariableDeclarator
     *</em>
     *<p>
     *<b>
     *<li>Rule 108:  VariableDeclarators ::= VariableDeclarators , VariableDeclarator
     *</b>
     */
    static public class VariableDeclarators extends Ast implements IVariableDeclarators
    {
        private IVariableDeclarators _VariableDeclarators;
        private AstToken _COMMA;
        private IVariableDeclarator _VariableDeclarator;

        public IVariableDeclarators getVariableDeclarators() { return _VariableDeclarators; }
        public AstToken getCOMMA() { return _COMMA; }
        public IVariableDeclarator getVariableDeclarator() { return _VariableDeclarator; }

        public VariableDeclarators(IToken leftIToken, IToken rightIToken,
                                   IVariableDeclarators _VariableDeclarators,
                                   AstToken _COMMA,
                                   IVariableDeclarator _VariableDeclarator)
        {
            super(leftIToken, rightIToken);

            this._VariableDeclarators = _VariableDeclarators;
            ((Ast) _VariableDeclarators).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._VariableDeclarator = _VariableDeclarator;
            ((Ast) _VariableDeclarator).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableDeclarators);
            list.add(_COMMA);
            list.add(_VariableDeclarator);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof VariableDeclarators)) return false;
            if (! super.equals(o)) return false;
            VariableDeclarators other = (VariableDeclarators) o;
            if (! _VariableDeclarators.equals(other._VariableDeclarators)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _VariableDeclarator.equals(other._VariableDeclarator)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableDeclarators.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_VariableDeclarator.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _VariableDeclarators.accept(v);
                _COMMA.accept(v);
                _VariableDeclarator.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 109:  VariableDeclarator ::= VariableDeclaratorId
     *</em>
     *<p>
     *<b>
     *<li>Rule 110:  VariableDeclarator ::= VariableDeclaratorId = VariableInitializer
     *</b>
     */
    static public class VariableDeclarator extends Ast implements IVariableDeclarator
    {
        private IVariableDeclaratorId _VariableDeclaratorId;
        private AstToken _EQUAL;
        private IVariableInitializer _VariableInitializer;

        public IVariableDeclaratorId getVariableDeclaratorId() { return _VariableDeclaratorId; }
        public AstToken getEQUAL() { return _EQUAL; }
        public IVariableInitializer getVariableInitializer() { return _VariableInitializer; }

        public VariableDeclarator(IToken leftIToken, IToken rightIToken,
                                  IVariableDeclaratorId _VariableDeclaratorId,
                                  AstToken _EQUAL,
                                  IVariableInitializer _VariableInitializer)
        {
            super(leftIToken, rightIToken);

            this._VariableDeclaratorId = _VariableDeclaratorId;
            ((Ast) _VariableDeclaratorId).setParent(this);
            this._EQUAL = _EQUAL;
            ((Ast) _EQUAL).setParent(this);
            this._VariableInitializer = _VariableInitializer;
            ((Ast) _VariableInitializer).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableDeclaratorId);
            list.add(_EQUAL);
            list.add(_VariableInitializer);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof VariableDeclarator)) return false;
            if (! super.equals(o)) return false;
            VariableDeclarator other = (VariableDeclarator) o;
            if (! _VariableDeclaratorId.equals(other._VariableDeclaratorId)) return false;
            if (! _EQUAL.equals(other._EQUAL)) return false;
            if (! _VariableInitializer.equals(other._VariableInitializer)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableDeclaratorId.hashCode());
            hash = hash * 31 + (_EQUAL.hashCode());
            hash = hash * 31 + (_VariableInitializer.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _VariableDeclaratorId.accept(v);
                _EQUAL.accept(v);
                _VariableInitializer.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 111:  VariableDeclaratorId ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 112:  VariableDeclaratorId ::= VariableDeclaratorId [ ]
     *</b>
     */
    static public class VariableDeclaratorId extends Ast implements IVariableDeclaratorId
    {
        private IVariableDeclaratorId _VariableDeclaratorId;
        private AstToken _LBRACKET;
        private AstToken _RBRACKET;

        public IVariableDeclaratorId getVariableDeclaratorId() { return _VariableDeclaratorId; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public VariableDeclaratorId(IToken leftIToken, IToken rightIToken,
                                    IVariableDeclaratorId _VariableDeclaratorId,
                                    AstToken _LBRACKET,
                                    AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._VariableDeclaratorId = _VariableDeclaratorId;
            ((Ast) _VariableDeclaratorId).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableDeclaratorId);
            list.add(_LBRACKET);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof VariableDeclaratorId)) return false;
            if (! super.equals(o)) return false;
            VariableDeclaratorId other = (VariableDeclaratorId) o;
            if (! _VariableDeclaratorId.equals(other._VariableDeclaratorId)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableDeclaratorId.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _VariableDeclaratorId.accept(v);
                _LBRACKET.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 115:  FieldModifiers ::= FieldModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 116:  FieldModifiers ::= FieldModifiers FieldModifier
     *</b>
     */
    static public class FieldModifiers extends Ast implements IFieldModifiers
    {
        private IFieldModifiers _FieldModifiers;
        private IFieldModifier _FieldModifier;

        public IFieldModifiers getFieldModifiers() { return _FieldModifiers; }
        public IFieldModifier getFieldModifier() { return _FieldModifier; }

        public FieldModifiers(IToken leftIToken, IToken rightIToken,
                              IFieldModifiers _FieldModifiers,
                              IFieldModifier _FieldModifier)
        {
            super(leftIToken, rightIToken);

            this._FieldModifiers = _FieldModifiers;
            ((Ast) _FieldModifiers).setParent(this);
            this._FieldModifier = _FieldModifier;
            ((Ast) _FieldModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_FieldModifiers);
            list.add(_FieldModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FieldModifiers)) return false;
            if (! super.equals(o)) return false;
            FieldModifiers other = (FieldModifiers) o;
            if (! _FieldModifiers.equals(other._FieldModifiers)) return false;
            if (! _FieldModifier.equals(other._FieldModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_FieldModifiers.hashCode());
            hash = hash * 31 + (_FieldModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _FieldModifiers.accept(v);
                _FieldModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 125:  MethodDeclaration ::= MethodHeader MethodBody
     *</b>
     */
    static public class MethodDeclaration extends Ast implements IMethodDeclaration
    {
        private MethodHeader _MethodHeader;
        private IMethodBody _MethodBody;

        public MethodHeader getMethodHeader() { return _MethodHeader; }
        public IMethodBody getMethodBody() { return _MethodBody; }

        public MethodDeclaration(IToken leftIToken, IToken rightIToken,
                                 MethodHeader _MethodHeader,
                                 IMethodBody _MethodBody)
        {
            super(leftIToken, rightIToken);

            this._MethodHeader = _MethodHeader;
            ((Ast) _MethodHeader).setParent(this);
            this._MethodBody = _MethodBody;
            ((Ast) _MethodBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MethodHeader);
            list.add(_MethodBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodDeclaration)) return false;
            if (! super.equals(o)) return false;
            MethodDeclaration other = (MethodDeclaration) o;
            if (! _MethodHeader.equals(other._MethodHeader)) return false;
            if (! _MethodBody.equals(other._MethodBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MethodHeader.hashCode());
            hash = hash * 31 + (_MethodBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MethodHeader.accept(v);
                _MethodBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 126:  MethodHeader ::= MethodModifiersopt TypeParametersopt ResultType MethodDeclarator Throwsopt
     *</b>
     */
    static public class MethodHeader extends Ast implements IMethodHeader
    {
        private IMethodModifiersopt _MethodModifiersopt;
        private TypeParameters _TypeParametersopt;
        private IResultType _ResultType;
        private IMethodDeclarator _MethodDeclarator;
        private Throws _Throwsopt;

        /**
         * The value returned by <b>getMethodModifiersopt</b> may be <b>null</b>
         */
        public IMethodModifiersopt getMethodModifiersopt() { return _MethodModifiersopt; }
        /**
         * The value returned by <b>getTypeParametersopt</b> may be <b>null</b>
         */
        public TypeParameters getTypeParametersopt() { return _TypeParametersopt; }
        public IResultType getResultType() { return _ResultType; }
        public IMethodDeclarator getMethodDeclarator() { return _MethodDeclarator; }
        /**
         * The value returned by <b>getThrowsopt</b> may be <b>null</b>
         */
        public Throws getThrowsopt() { return _Throwsopt; }

        public MethodHeader(IToken leftIToken, IToken rightIToken,
                            IMethodModifiersopt _MethodModifiersopt,
                            TypeParameters _TypeParametersopt,
                            IResultType _ResultType,
                            IMethodDeclarator _MethodDeclarator,
                            Throws _Throwsopt)
        {
            super(leftIToken, rightIToken);

            this._MethodModifiersopt = _MethodModifiersopt;
            if (_MethodModifiersopt != null) ((Ast) _MethodModifiersopt).setParent(this);
            this._TypeParametersopt = _TypeParametersopt;
            if (_TypeParametersopt != null) ((Ast) _TypeParametersopt).setParent(this);
            this._ResultType = _ResultType;
            ((Ast) _ResultType).setParent(this);
            this._MethodDeclarator = _MethodDeclarator;
            ((Ast) _MethodDeclarator).setParent(this);
            this._Throwsopt = _Throwsopt;
            if (_Throwsopt != null) ((Ast) _Throwsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MethodModifiersopt);
            list.add(_TypeParametersopt);
            list.add(_ResultType);
            list.add(_MethodDeclarator);
            list.add(_Throwsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodHeader)) return false;
            if (! super.equals(o)) return false;
            MethodHeader other = (MethodHeader) o;
            if (_MethodModifiersopt == null)
                if (other._MethodModifiersopt != null) return false;
                else; // continue
            else if (! _MethodModifiersopt.equals(other._MethodModifiersopt)) return false;
            if (_TypeParametersopt == null)
                if (other._TypeParametersopt != null) return false;
                else; // continue
            else if (! _TypeParametersopt.equals(other._TypeParametersopt)) return false;
            if (! _ResultType.equals(other._ResultType)) return false;
            if (! _MethodDeclarator.equals(other._MethodDeclarator)) return false;
            if (_Throwsopt == null)
                if (other._Throwsopt != null) return false;
                else; // continue
            else if (! _Throwsopt.equals(other._Throwsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MethodModifiersopt == null ? 0 : _MethodModifiersopt.hashCode());
            hash = hash * 31 + (_TypeParametersopt == null ? 0 : _TypeParametersopt.hashCode());
            hash = hash * 31 + (_ResultType.hashCode());
            hash = hash * 31 + (_MethodDeclarator.hashCode());
            hash = hash * 31 + (_Throwsopt == null ? 0 : _Throwsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_MethodModifiersopt != null) _MethodModifiersopt.accept(v);
                if (_TypeParametersopt != null) _TypeParametersopt.accept(v);
                _ResultType.accept(v);
                _MethodDeclarator.accept(v);
                if (_Throwsopt != null) _Throwsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 127:  ResultType ::= Type
     *</em>
     *<p>
     *<b>
     *<li>Rule 128:  ResultType ::= void
     *</b>
     */
    static public class ResultType extends AstToken implements IResultType
    {
        public IToken getvoid() { return leftIToken; }

        public ResultType(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 131:  FormalParameterList ::= LastFormalParameter
     *</em>
     *<p>
     *<b>
     *<li>Rule 132:  FormalParameterList ::= FormalParameters , LastFormalParameter
     *</b>
     */
    static public class FormalParameterList extends Ast implements IFormalParameterList
    {
        private IFormalParameters _FormalParameters;
        private AstToken _COMMA;
        private LastFormalParameter _LastFormalParameter;

        public IFormalParameters getFormalParameters() { return _FormalParameters; }
        public AstToken getCOMMA() { return _COMMA; }
        public LastFormalParameter getLastFormalParameter() { return _LastFormalParameter; }

        public FormalParameterList(IToken leftIToken, IToken rightIToken,
                                   IFormalParameters _FormalParameters,
                                   AstToken _COMMA,
                                   LastFormalParameter _LastFormalParameter)
        {
            super(leftIToken, rightIToken);

            this._FormalParameters = _FormalParameters;
            ((Ast) _FormalParameters).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._LastFormalParameter = _LastFormalParameter;
            ((Ast) _LastFormalParameter).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_FormalParameters);
            list.add(_COMMA);
            list.add(_LastFormalParameter);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FormalParameterList)) return false;
            if (! super.equals(o)) return false;
            FormalParameterList other = (FormalParameterList) o;
            if (! _FormalParameters.equals(other._FormalParameters)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _LastFormalParameter.equals(other._LastFormalParameter)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_FormalParameters.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_LastFormalParameter.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _FormalParameters.accept(v);
                _COMMA.accept(v);
                _LastFormalParameter.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 133:  FormalParameters ::= FormalParameter
     *</em>
     *<p>
     *<b>
     *<li>Rule 134:  FormalParameters ::= FormalParameters , FormalParameter
     *</b>
     */
    static public class FormalParameters extends Ast implements IFormalParameters
    {
        private IFormalParameters _FormalParameters;
        private AstToken _COMMA;
        private FormalParameter _FormalParameter;

        public IFormalParameters getFormalParameters() { return _FormalParameters; }
        public AstToken getCOMMA() { return _COMMA; }
        public FormalParameter getFormalParameter() { return _FormalParameter; }

        public FormalParameters(IToken leftIToken, IToken rightIToken,
                                IFormalParameters _FormalParameters,
                                AstToken _COMMA,
                                FormalParameter _FormalParameter)
        {
            super(leftIToken, rightIToken);

            this._FormalParameters = _FormalParameters;
            ((Ast) _FormalParameters).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._FormalParameter = _FormalParameter;
            ((Ast) _FormalParameter).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_FormalParameters);
            list.add(_COMMA);
            list.add(_FormalParameter);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FormalParameters)) return false;
            if (! super.equals(o)) return false;
            FormalParameters other = (FormalParameters) o;
            if (! _FormalParameters.equals(other._FormalParameters)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _FormalParameter.equals(other._FormalParameter)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_FormalParameters.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_FormalParameter.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _FormalParameters.accept(v);
                _COMMA.accept(v);
                _FormalParameter.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 135:  FormalParameter ::= VariableModifiersopt Type VariableDeclaratorId
     *</b>
     */
    static public class FormalParameter extends Ast implements IFormalParameter
    {
        private IVariableModifiersopt _VariableModifiersopt;
        private IType _Type;
        private IVariableDeclaratorId _VariableDeclaratorId;

        /**
         * The value returned by <b>getVariableModifiersopt</b> may be <b>null</b>
         */
        public IVariableModifiersopt getVariableModifiersopt() { return _VariableModifiersopt; }
        public IType getType() { return _Type; }
        public IVariableDeclaratorId getVariableDeclaratorId() { return _VariableDeclaratorId; }

        public FormalParameter(IToken leftIToken, IToken rightIToken,
                               IVariableModifiersopt _VariableModifiersopt,
                               IType _Type,
                               IVariableDeclaratorId _VariableDeclaratorId)
        {
            super(leftIToken, rightIToken);

            this._VariableModifiersopt = _VariableModifiersopt;
            if (_VariableModifiersopt != null) ((Ast) _VariableModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._VariableDeclaratorId = _VariableDeclaratorId;
            ((Ast) _VariableDeclaratorId).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableModifiersopt);
            list.add(_Type);
            list.add(_VariableDeclaratorId);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FormalParameter)) return false;
            if (! super.equals(o)) return false;
            FormalParameter other = (FormalParameter) o;
            if (_VariableModifiersopt == null)
                if (other._VariableModifiersopt != null) return false;
                else; // continue
            else if (! _VariableModifiersopt.equals(other._VariableModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (! _VariableDeclaratorId.equals(other._VariableDeclaratorId)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableModifiersopt == null ? 0 : _VariableModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_VariableDeclaratorId.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_VariableModifiersopt != null) _VariableModifiersopt.accept(v);
                _Type.accept(v);
                _VariableDeclaratorId.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 136:  VariableModifiers ::= VariableModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 137:  VariableModifiers ::= VariableModifiers VariableModifier
     *</b>
     */
    static public class VariableModifiers extends Ast implements IVariableModifiers
    {
        private IVariableModifiers _VariableModifiers;
        private IVariableModifier _VariableModifier;

        public IVariableModifiers getVariableModifiers() { return _VariableModifiers; }
        public IVariableModifier getVariableModifier() { return _VariableModifier; }

        public VariableModifiers(IToken leftIToken, IToken rightIToken,
                                 IVariableModifiers _VariableModifiers,
                                 IVariableModifier _VariableModifier)
        {
            super(leftIToken, rightIToken);

            this._VariableModifiers = _VariableModifiers;
            ((Ast) _VariableModifiers).setParent(this);
            this._VariableModifier = _VariableModifier;
            ((Ast) _VariableModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableModifiers);
            list.add(_VariableModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof VariableModifiers)) return false;
            if (! super.equals(o)) return false;
            VariableModifiers other = (VariableModifiers) o;
            if (! _VariableModifiers.equals(other._VariableModifiers)) return false;
            if (! _VariableModifier.equals(other._VariableModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableModifiers.hashCode());
            hash = hash * 31 + (_VariableModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _VariableModifiers.accept(v);
                _VariableModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 139:  VariableModifier ::= Annotations
     *</em>
     *<p>
     *<b>
     *<li>Rule 138:  VariableModifier ::= final
     *</b>
     */
    static public class VariableModifier extends AstToken implements IVariableModifier
    {
        public IToken getfinal() { return leftIToken; }

        public VariableModifier(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 140:  LastFormalParameter ::= VariableModifiersopt Type ...opt VariableDeclaratorId
     *</b>
     */
    static public class LastFormalParameter extends Ast implements ILastFormalParameter
    {
        private IVariableModifiersopt _VariableModifiersopt;
        private IType _Type;
        private Ellipsisopt _Ellipsisopt;
        private IVariableDeclaratorId _VariableDeclaratorId;

        /**
         * The value returned by <b>getVariableModifiersopt</b> may be <b>null</b>
         */
        public IVariableModifiersopt getVariableModifiersopt() { return _VariableModifiersopt; }
        public IType getType() { return _Type; }
        /**
         * The value returned by <b>getEllipsisopt</b> may be <b>null</b>
         */
        public Ellipsisopt getEllipsisopt() { return _Ellipsisopt; }
        public IVariableDeclaratorId getVariableDeclaratorId() { return _VariableDeclaratorId; }

        public LastFormalParameter(IToken leftIToken, IToken rightIToken,
                                   IVariableModifiersopt _VariableModifiersopt,
                                   IType _Type,
                                   Ellipsisopt _Ellipsisopt,
                                   IVariableDeclaratorId _VariableDeclaratorId)
        {
            super(leftIToken, rightIToken);

            this._VariableModifiersopt = _VariableModifiersopt;
            if (_VariableModifiersopt != null) ((Ast) _VariableModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._Ellipsisopt = _Ellipsisopt;
            if (_Ellipsisopt != null) ((Ast) _Ellipsisopt).setParent(this);
            this._VariableDeclaratorId = _VariableDeclaratorId;
            ((Ast) _VariableDeclaratorId).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableModifiersopt);
            list.add(_Type);
            list.add(_Ellipsisopt);
            list.add(_VariableDeclaratorId);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LastFormalParameter)) return false;
            if (! super.equals(o)) return false;
            LastFormalParameter other = (LastFormalParameter) o;
            if (_VariableModifiersopt == null)
                if (other._VariableModifiersopt != null) return false;
                else; // continue
            else if (! _VariableModifiersopt.equals(other._VariableModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (_Ellipsisopt == null)
                if (other._Ellipsisopt != null) return false;
                else; // continue
            else if (! _Ellipsisopt.equals(other._Ellipsisopt)) return false;
            if (! _VariableDeclaratorId.equals(other._VariableDeclaratorId)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableModifiersopt == null ? 0 : _VariableModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_Ellipsisopt == null ? 0 : _Ellipsisopt.hashCode());
            hash = hash * 31 + (_VariableDeclaratorId.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_VariableModifiersopt != null) _VariableModifiersopt.accept(v);
                _Type.accept(v);
                if (_Ellipsisopt != null) _Ellipsisopt.accept(v);
                _VariableDeclaratorId.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 141:  MethodModifiers ::= MethodModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 142:  MethodModifiers ::= MethodModifiers MethodModifier
     *</b>
     */
    static public class MethodModifiers extends Ast implements IMethodModifiers
    {
        private IMethodModifiers _MethodModifiers;
        private IMethodModifier _MethodModifier;

        public IMethodModifiers getMethodModifiers() { return _MethodModifiers; }
        public IMethodModifier getMethodModifier() { return _MethodModifier; }

        public MethodModifiers(IToken leftIToken, IToken rightIToken,
                               IMethodModifiers _MethodModifiers,
                               IMethodModifier _MethodModifier)
        {
            super(leftIToken, rightIToken);

            this._MethodModifiers = _MethodModifiers;
            ((Ast) _MethodModifiers).setParent(this);
            this._MethodModifier = _MethodModifier;
            ((Ast) _MethodModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MethodModifiers);
            list.add(_MethodModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodModifiers)) return false;
            if (! super.equals(o)) return false;
            MethodModifiers other = (MethodModifiers) o;
            if (! _MethodModifiers.equals(other._MethodModifiers)) return false;
            if (! _MethodModifier.equals(other._MethodModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MethodModifiers.hashCode());
            hash = hash * 31 + (_MethodModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MethodModifiers.accept(v);
                _MethodModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 153:  Throws ::= throws ExceptionTypeList
     *</b>
     */
    static public class Throws extends Ast implements IThrows
    {
        private AstToken _throws;
        private IExceptionTypeList _ExceptionTypeList;

        public AstToken getthrows() { return _throws; }
        public IExceptionTypeList getExceptionTypeList() { return _ExceptionTypeList; }

        public Throws(IToken leftIToken, IToken rightIToken,
                      AstToken _throws,
                      IExceptionTypeList _ExceptionTypeList)
        {
            super(leftIToken, rightIToken);

            this._throws = _throws;
            ((Ast) _throws).setParent(this);
            this._ExceptionTypeList = _ExceptionTypeList;
            ((Ast) _ExceptionTypeList).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_throws);
            list.add(_ExceptionTypeList);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Throws)) return false;
            if (! super.equals(o)) return false;
            Throws other = (Throws) o;
            if (! _throws.equals(other._throws)) return false;
            if (! _ExceptionTypeList.equals(other._ExceptionTypeList)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_throws.hashCode());
            hash = hash * 31 + (_ExceptionTypeList.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _throws.accept(v);
                _ExceptionTypeList.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 154:  ExceptionTypeList ::= ExceptionType
     *</em>
     *<p>
     *<b>
     *<li>Rule 155:  ExceptionTypeList ::= ExceptionTypeList , ExceptionType
     *</b>
     */
    static public class ExceptionTypeList extends Ast implements IExceptionTypeList
    {
        private IExceptionTypeList _ExceptionTypeList;
        private AstToken _COMMA;
        private IExceptionType _ExceptionType;

        public IExceptionTypeList getExceptionTypeList() { return _ExceptionTypeList; }
        public AstToken getCOMMA() { return _COMMA; }
        public IExceptionType getExceptionType() { return _ExceptionType; }

        public ExceptionTypeList(IToken leftIToken, IToken rightIToken,
                                 IExceptionTypeList _ExceptionTypeList,
                                 AstToken _COMMA,
                                 IExceptionType _ExceptionType)
        {
            super(leftIToken, rightIToken);

            this._ExceptionTypeList = _ExceptionTypeList;
            ((Ast) _ExceptionTypeList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._ExceptionType = _ExceptionType;
            ((Ast) _ExceptionType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ExceptionTypeList);
            list.add(_COMMA);
            list.add(_ExceptionType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExceptionTypeList)) return false;
            if (! super.equals(o)) return false;
            ExceptionTypeList other = (ExceptionTypeList) o;
            if (! _ExceptionTypeList.equals(other._ExceptionTypeList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _ExceptionType.equals(other._ExceptionType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ExceptionTypeList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_ExceptionType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ExceptionTypeList.accept(v);
                _COMMA.accept(v);
                _ExceptionType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 158:  MethodBody ::= Block
     *</em>
     *<p>
     *<b>
     *<li>Rule 159:  MethodBody ::= ;
     *</b>
     */
    static public class MethodBody extends AstToken implements IMethodBody
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public MethodBody(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 161:  StaticInitializer ::= static Block
     *</b>
     */
    static public class StaticInitializer extends Ast implements IStaticInitializer
    {
        private AstToken _static;
        private Block _Block;

        public AstToken getstatic() { return _static; }
        public Block getBlock() { return _Block; }

        public StaticInitializer(IToken leftIToken, IToken rightIToken,
                                 AstToken _static,
                                 Block _Block)
        {
            super(leftIToken, rightIToken);

            this._static = _static;
            ((Ast) _static).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_static);
            list.add(_Block);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof StaticInitializer)) return false;
            if (! super.equals(o)) return false;
            StaticInitializer other = (StaticInitializer) o;
            if (! _static.equals(other._static)) return false;
            if (! _Block.equals(other._Block)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_static.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _static.accept(v);
                _Block.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 162:  ConstructorDeclaration ::= ConstructorModifiersopt ConstructorDeclarator Throwsopt ConstructorBody
     *</b>
     */
    static public class ConstructorDeclaration extends Ast implements IConstructorDeclaration
    {
        private IConstructorModifiersopt _ConstructorModifiersopt;
        private ConstructorDeclarator _ConstructorDeclarator;
        private Throws _Throwsopt;
        private ConstructorBody _ConstructorBody;

        /**
         * The value returned by <b>getConstructorModifiersopt</b> may be <b>null</b>
         */
        public IConstructorModifiersopt getConstructorModifiersopt() { return _ConstructorModifiersopt; }
        public ConstructorDeclarator getConstructorDeclarator() { return _ConstructorDeclarator; }
        /**
         * The value returned by <b>getThrowsopt</b> may be <b>null</b>
         */
        public Throws getThrowsopt() { return _Throwsopt; }
        public ConstructorBody getConstructorBody() { return _ConstructorBody; }

        public ConstructorDeclaration(IToken leftIToken, IToken rightIToken,
                                      IConstructorModifiersopt _ConstructorModifiersopt,
                                      ConstructorDeclarator _ConstructorDeclarator,
                                      Throws _Throwsopt,
                                      ConstructorBody _ConstructorBody)
        {
            super(leftIToken, rightIToken);

            this._ConstructorModifiersopt = _ConstructorModifiersopt;
            if (_ConstructorModifiersopt != null) ((Ast) _ConstructorModifiersopt).setParent(this);
            this._ConstructorDeclarator = _ConstructorDeclarator;
            ((Ast) _ConstructorDeclarator).setParent(this);
            this._Throwsopt = _Throwsopt;
            if (_Throwsopt != null) ((Ast) _Throwsopt).setParent(this);
            this._ConstructorBody = _ConstructorBody;
            ((Ast) _ConstructorBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConstructorModifiersopt);
            list.add(_ConstructorDeclarator);
            list.add(_Throwsopt);
            list.add(_ConstructorBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstructorDeclaration)) return false;
            if (! super.equals(o)) return false;
            ConstructorDeclaration other = (ConstructorDeclaration) o;
            if (_ConstructorModifiersopt == null)
                if (other._ConstructorModifiersopt != null) return false;
                else; // continue
            else if (! _ConstructorModifiersopt.equals(other._ConstructorModifiersopt)) return false;
            if (! _ConstructorDeclarator.equals(other._ConstructorDeclarator)) return false;
            if (_Throwsopt == null)
                if (other._Throwsopt != null) return false;
                else; // continue
            else if (! _Throwsopt.equals(other._Throwsopt)) return false;
            if (! _ConstructorBody.equals(other._ConstructorBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConstructorModifiersopt == null ? 0 : _ConstructorModifiersopt.hashCode());
            hash = hash * 31 + (_ConstructorDeclarator.hashCode());
            hash = hash * 31 + (_Throwsopt == null ? 0 : _Throwsopt.hashCode());
            hash = hash * 31 + (_ConstructorBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_ConstructorModifiersopt != null) _ConstructorModifiersopt.accept(v);
                _ConstructorDeclarator.accept(v);
                if (_Throwsopt != null) _Throwsopt.accept(v);
                _ConstructorBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 163:  ConstructorDeclarator ::= TypeParametersopt SimpleTypeName ( FormalParameterListopt )
     *</b>
     */
    static public class ConstructorDeclarator extends Ast implements IConstructorDeclarator
    {
        private TypeParameters _TypeParametersopt;
        private identifier _SimpleTypeName;
        private AstToken _LPAREN;
        private IFormalParameterListopt _FormalParameterListopt;
        private AstToken _RPAREN;

        /**
         * The value returned by <b>getTypeParametersopt</b> may be <b>null</b>
         */
        public TypeParameters getTypeParametersopt() { return _TypeParametersopt; }
        public identifier getSimpleTypeName() { return _SimpleTypeName; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getFormalParameterListopt</b> may be <b>null</b>
         */
        public IFormalParameterListopt getFormalParameterListopt() { return _FormalParameterListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public ConstructorDeclarator(IToken leftIToken, IToken rightIToken,
                                     TypeParameters _TypeParametersopt,
                                     identifier _SimpleTypeName,
                                     AstToken _LPAREN,
                                     IFormalParameterListopt _FormalParameterListopt,
                                     AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._TypeParametersopt = _TypeParametersopt;
            if (_TypeParametersopt != null) ((Ast) _TypeParametersopt).setParent(this);
            this._SimpleTypeName = _SimpleTypeName;
            ((Ast) _SimpleTypeName).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._FormalParameterListopt = _FormalParameterListopt;
            if (_FormalParameterListopt != null) ((Ast) _FormalParameterListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeParametersopt);
            list.add(_SimpleTypeName);
            list.add(_LPAREN);
            list.add(_FormalParameterListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstructorDeclarator)) return false;
            if (! super.equals(o)) return false;
            ConstructorDeclarator other = (ConstructorDeclarator) o;
            if (_TypeParametersopt == null)
                if (other._TypeParametersopt != null) return false;
                else; // continue
            else if (! _TypeParametersopt.equals(other._TypeParametersopt)) return false;
            if (! _SimpleTypeName.equals(other._SimpleTypeName)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_FormalParameterListopt == null)
                if (other._FormalParameterListopt != null) return false;
                else; // continue
            else if (! _FormalParameterListopt.equals(other._FormalParameterListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeParametersopt == null ? 0 : _TypeParametersopt.hashCode());
            hash = hash * 31 + (_SimpleTypeName.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_FormalParameterListopt == null ? 0 : _FormalParameterListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_TypeParametersopt != null) _TypeParametersopt.accept(v);
                _SimpleTypeName.accept(v);
                _LPAREN.accept(v);
                if (_FormalParameterListopt != null) _FormalParameterListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 165:  ConstructorModifiers ::= ConstructorModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 166:  ConstructorModifiers ::= ConstructorModifiers ConstructorModifier
     *</b>
     */
    static public class ConstructorModifiers extends Ast implements IConstructorModifiers
    {
        private IConstructorModifiers _ConstructorModifiers;
        private IConstructorModifier _ConstructorModifier;

        public IConstructorModifiers getConstructorModifiers() { return _ConstructorModifiers; }
        public IConstructorModifier getConstructorModifier() { return _ConstructorModifier; }

        public ConstructorModifiers(IToken leftIToken, IToken rightIToken,
                                    IConstructorModifiers _ConstructorModifiers,
                                    IConstructorModifier _ConstructorModifier)
        {
            super(leftIToken, rightIToken);

            this._ConstructorModifiers = _ConstructorModifiers;
            ((Ast) _ConstructorModifiers).setParent(this);
            this._ConstructorModifier = _ConstructorModifier;
            ((Ast) _ConstructorModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConstructorModifiers);
            list.add(_ConstructorModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstructorModifiers)) return false;
            if (! super.equals(o)) return false;
            ConstructorModifiers other = (ConstructorModifiers) o;
            if (! _ConstructorModifiers.equals(other._ConstructorModifiers)) return false;
            if (! _ConstructorModifier.equals(other._ConstructorModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConstructorModifiers.hashCode());
            hash = hash * 31 + (_ConstructorModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ConstructorModifiers.accept(v);
                _ConstructorModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 171:  ConstructorBody ::= { ExplicitConstructorInvocationopt BlockStatementsopt }
     *</b>
     */
    static public class ConstructorBody extends Ast implements IConstructorBody
    {
        private AstToken _LBRACE;
        private IExplicitConstructorInvocationopt _ExplicitConstructorInvocationopt;
        private IBlockStatementsopt _BlockStatementsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getExplicitConstructorInvocationopt</b> may be <b>null</b>
         */
        public IExplicitConstructorInvocationopt getExplicitConstructorInvocationopt() { return _ExplicitConstructorInvocationopt; }
        /**
         * The value returned by <b>getBlockStatementsopt</b> may be <b>null</b>
         */
        public IBlockStatementsopt getBlockStatementsopt() { return _BlockStatementsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public ConstructorBody(IToken leftIToken, IToken rightIToken,
                               AstToken _LBRACE,
                               IExplicitConstructorInvocationopt _ExplicitConstructorInvocationopt,
                               IBlockStatementsopt _BlockStatementsopt,
                               AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._ExplicitConstructorInvocationopt = _ExplicitConstructorInvocationopt;
            if (_ExplicitConstructorInvocationopt != null) ((Ast) _ExplicitConstructorInvocationopt).setParent(this);
            this._BlockStatementsopt = _BlockStatementsopt;
            if (_BlockStatementsopt != null) ((Ast) _BlockStatementsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_ExplicitConstructorInvocationopt);
            list.add(_BlockStatementsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstructorBody)) return false;
            if (! super.equals(o)) return false;
            ConstructorBody other = (ConstructorBody) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_ExplicitConstructorInvocationopt == null)
                if (other._ExplicitConstructorInvocationopt != null) return false;
                else; // continue
            else if (! _ExplicitConstructorInvocationopt.equals(other._ExplicitConstructorInvocationopt)) return false;
            if (_BlockStatementsopt == null)
                if (other._BlockStatementsopt != null) return false;
                else; // continue
            else if (! _BlockStatementsopt.equals(other._BlockStatementsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_ExplicitConstructorInvocationopt == null ? 0 : _ExplicitConstructorInvocationopt.hashCode());
            hash = hash * 31 + (_BlockStatementsopt == null ? 0 : _BlockStatementsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_ExplicitConstructorInvocationopt != null) _ExplicitConstructorInvocationopt.accept(v);
                if (_BlockStatementsopt != null) _BlockStatementsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 175:  EnumDeclaration ::= ClassModifiersopt enum identifier Interfacesopt EnumBody
     *</b>
     */
    static public class EnumDeclaration extends Ast implements IEnumDeclaration
    {
        private IClassModifiersopt _ClassModifiersopt;
        private AstToken _enum;
        private identifier _identifier;
        private Interfaces _Interfacesopt;
        private EnumBody _EnumBody;

        /**
         * The value returned by <b>getClassModifiersopt</b> may be <b>null</b>
         */
        public IClassModifiersopt getClassModifiersopt() { return _ClassModifiersopt; }
        public AstToken getenum() { return _enum; }
        public identifier getidentifier() { return _identifier; }
        /**
         * The value returned by <b>getInterfacesopt</b> may be <b>null</b>
         */
        public Interfaces getInterfacesopt() { return _Interfacesopt; }
        public EnumBody getEnumBody() { return _EnumBody; }

        public EnumDeclaration(IToken leftIToken, IToken rightIToken,
                               IClassModifiersopt _ClassModifiersopt,
                               AstToken _enum,
                               identifier _identifier,
                               Interfaces _Interfacesopt,
                               EnumBody _EnumBody)
        {
            super(leftIToken, rightIToken);

            this._ClassModifiersopt = _ClassModifiersopt;
            if (_ClassModifiersopt != null) ((Ast) _ClassModifiersopt).setParent(this);
            this._enum = _enum;
            ((Ast) _enum).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._Interfacesopt = _Interfacesopt;
            if (_Interfacesopt != null) ((Ast) _Interfacesopt).setParent(this);
            this._EnumBody = _EnumBody;
            ((Ast) _EnumBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassModifiersopt);
            list.add(_enum);
            list.add(_identifier);
            list.add(_Interfacesopt);
            list.add(_EnumBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnumDeclaration)) return false;
            if (! super.equals(o)) return false;
            EnumDeclaration other = (EnumDeclaration) o;
            if (_ClassModifiersopt == null)
                if (other._ClassModifiersopt != null) return false;
                else; // continue
            else if (! _ClassModifiersopt.equals(other._ClassModifiersopt)) return false;
            if (! _enum.equals(other._enum)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (_Interfacesopt == null)
                if (other._Interfacesopt != null) return false;
                else; // continue
            else if (! _Interfacesopt.equals(other._Interfacesopt)) return false;
            if (! _EnumBody.equals(other._EnumBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassModifiersopt == null ? 0 : _ClassModifiersopt.hashCode());
            hash = hash * 31 + (_enum.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_Interfacesopt == null ? 0 : _Interfacesopt.hashCode());
            hash = hash * 31 + (_EnumBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_ClassModifiersopt != null) _ClassModifiersopt.accept(v);
                _enum.accept(v);
                _identifier.accept(v);
                if (_Interfacesopt != null) _Interfacesopt.accept(v);
                _EnumBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 176:  EnumBody ::= { EnumConstantsopt ,opt EnumBodyDeclarationsopt }
     *</b>
     */
    static public class EnumBody extends Ast implements IEnumBody
    {
        private AstToken _LBRACE;
        private IEnumConstantsopt _EnumConstantsopt;
        private Commaopt _Commaopt;
        private EnumBodyDeclarations _EnumBodyDeclarationsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getEnumConstantsopt</b> may be <b>null</b>
         */
        public IEnumConstantsopt getEnumConstantsopt() { return _EnumConstantsopt; }
        /**
         * The value returned by <b>getCommaopt</b> may be <b>null</b>
         */
        public Commaopt getCommaopt() { return _Commaopt; }
        /**
         * The value returned by <b>getEnumBodyDeclarationsopt</b> may be <b>null</b>
         */
        public EnumBodyDeclarations getEnumBodyDeclarationsopt() { return _EnumBodyDeclarationsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public EnumBody(IToken leftIToken, IToken rightIToken,
                        AstToken _LBRACE,
                        IEnumConstantsopt _EnumConstantsopt,
                        Commaopt _Commaopt,
                        EnumBodyDeclarations _EnumBodyDeclarationsopt,
                        AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._EnumConstantsopt = _EnumConstantsopt;
            if (_EnumConstantsopt != null) ((Ast) _EnumConstantsopt).setParent(this);
            this._Commaopt = _Commaopt;
            if (_Commaopt != null) ((Ast) _Commaopt).setParent(this);
            this._EnumBodyDeclarationsopt = _EnumBodyDeclarationsopt;
            if (_EnumBodyDeclarationsopt != null) ((Ast) _EnumBodyDeclarationsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_EnumConstantsopt);
            list.add(_Commaopt);
            list.add(_EnumBodyDeclarationsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnumBody)) return false;
            if (! super.equals(o)) return false;
            EnumBody other = (EnumBody) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_EnumConstantsopt == null)
                if (other._EnumConstantsopt != null) return false;
                else; // continue
            else if (! _EnumConstantsopt.equals(other._EnumConstantsopt)) return false;
            if (_Commaopt == null)
                if (other._Commaopt != null) return false;
                else; // continue
            else if (! _Commaopt.equals(other._Commaopt)) return false;
            if (_EnumBodyDeclarationsopt == null)
                if (other._EnumBodyDeclarationsopt != null) return false;
                else; // continue
            else if (! _EnumBodyDeclarationsopt.equals(other._EnumBodyDeclarationsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_EnumConstantsopt == null ? 0 : _EnumConstantsopt.hashCode());
            hash = hash * 31 + (_Commaopt == null ? 0 : _Commaopt.hashCode());
            hash = hash * 31 + (_EnumBodyDeclarationsopt == null ? 0 : _EnumBodyDeclarationsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_EnumConstantsopt != null) _EnumConstantsopt.accept(v);
                if (_Commaopt != null) _Commaopt.accept(v);
                if (_EnumBodyDeclarationsopt != null) _EnumBodyDeclarationsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 177:  EnumConstants ::= EnumConstant
     *</em>
     *<p>
     *<b>
     *<li>Rule 178:  EnumConstants ::= EnumConstants , EnumConstant
     *</b>
     */
    static public class EnumConstants extends Ast implements IEnumConstants
    {
        private IEnumConstants _EnumConstants;
        private AstToken _COMMA;
        private IEnumConstant _EnumConstant;

        public IEnumConstants getEnumConstants() { return _EnumConstants; }
        public AstToken getCOMMA() { return _COMMA; }
        public IEnumConstant getEnumConstant() { return _EnumConstant; }

        public EnumConstants(IToken leftIToken, IToken rightIToken,
                             IEnumConstants _EnumConstants,
                             AstToken _COMMA,
                             IEnumConstant _EnumConstant)
        {
            super(leftIToken, rightIToken);

            this._EnumConstants = _EnumConstants;
            ((Ast) _EnumConstants).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._EnumConstant = _EnumConstant;
            ((Ast) _EnumConstant).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_EnumConstants);
            list.add(_COMMA);
            list.add(_EnumConstant);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnumConstants)) return false;
            if (! super.equals(o)) return false;
            EnumConstants other = (EnumConstants) o;
            if (! _EnumConstants.equals(other._EnumConstants)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _EnumConstant.equals(other._EnumConstant)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_EnumConstants.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_EnumConstant.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _EnumConstants.accept(v);
                _COMMA.accept(v);
                _EnumConstant.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 307:  EnumConstant ::= identifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 179:  EnumConstant ::= Annotationsopt identifier Argumentsopt ClassBodyopt
     *</b>
     */
    static public class EnumConstant extends Ast implements IEnumConstant
    {
        private IAnnotationsopt _Annotationsopt;
        private identifier _identifier;
        private Arguments _Argumentsopt;
        private ClassBody _ClassBodyopt;

        /**
         * The value returned by <b>getAnnotationsopt</b> may be <b>null</b>
         */
        public IAnnotationsopt getAnnotationsopt() { return _Annotationsopt; }
        public identifier getidentifier() { return _identifier; }
        /**
         * The value returned by <b>getArgumentsopt</b> may be <b>null</b>
         */
        public Arguments getArgumentsopt() { return _Argumentsopt; }
        /**
         * The value returned by <b>getClassBodyopt</b> may be <b>null</b>
         */
        public ClassBody getClassBodyopt() { return _ClassBodyopt; }

        public EnumConstant(IToken leftIToken, IToken rightIToken,
                            IAnnotationsopt _Annotationsopt,
                            identifier _identifier,
                            Arguments _Argumentsopt,
                            ClassBody _ClassBodyopt)
        {
            super(leftIToken, rightIToken);

            this._Annotationsopt = _Annotationsopt;
            if (_Annotationsopt != null) ((Ast) _Annotationsopt).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._Argumentsopt = _Argumentsopt;
            if (_Argumentsopt != null) ((Ast) _Argumentsopt).setParent(this);
            this._ClassBodyopt = _ClassBodyopt;
            if (_ClassBodyopt != null) ((Ast) _ClassBodyopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Annotationsopt);
            list.add(_identifier);
            list.add(_Argumentsopt);
            list.add(_ClassBodyopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnumConstant)) return false;
            if (! super.equals(o)) return false;
            EnumConstant other = (EnumConstant) o;
            if (_Annotationsopt == null)
                if (other._Annotationsopt != null) return false;
                else; // continue
            else if (! _Annotationsopt.equals(other._Annotationsopt)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (_Argumentsopt == null)
                if (other._Argumentsopt != null) return false;
                else; // continue
            else if (! _Argumentsopt.equals(other._Argumentsopt)) return false;
            if (_ClassBodyopt == null)
                if (other._ClassBodyopt != null) return false;
                else; // continue
            else if (! _ClassBodyopt.equals(other._ClassBodyopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Annotationsopt == null ? 0 : _Annotationsopt.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_Argumentsopt == null ? 0 : _Argumentsopt.hashCode());
            hash = hash * 31 + (_ClassBodyopt == null ? 0 : _ClassBodyopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_Annotationsopt != null) _Annotationsopt.accept(v);
                _identifier.accept(v);
                if (_Argumentsopt != null) _Argumentsopt.accept(v);
                if (_ClassBodyopt != null) _ClassBodyopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 180:  Arguments ::= ( ArgumentListopt )
     *</b>
     */
    static public class Arguments extends Ast implements IArguments
    {
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public Arguments(IToken leftIToken, IToken rightIToken,
                         AstToken _LPAREN,
                         IArgumentListopt _ArgumentListopt,
                         AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Arguments)) return false;
            if (! super.equals(o)) return false;
            Arguments other = (Arguments) o;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 181:  EnumBodyDeclarations ::= ; ClassBodyDeclarationsopt
     *</b>
     */
    static public class EnumBodyDeclarations extends Ast implements IEnumBodyDeclarations
    {
        private AstToken _SEMICOLON;
        private IClassBodyDeclarationsopt _ClassBodyDeclarationsopt;

        public AstToken getSEMICOLON() { return _SEMICOLON; }
        /**
         * The value returned by <b>getClassBodyDeclarationsopt</b> may be <b>null</b>
         */
        public IClassBodyDeclarationsopt getClassBodyDeclarationsopt() { return _ClassBodyDeclarationsopt; }

        public EnumBodyDeclarations(IToken leftIToken, IToken rightIToken,
                                    AstToken _SEMICOLON,
                                    IClassBodyDeclarationsopt _ClassBodyDeclarationsopt)
        {
            super(leftIToken, rightIToken);

            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            this._ClassBodyDeclarationsopt = _ClassBodyDeclarationsopt;
            if (_ClassBodyDeclarationsopt != null) ((Ast) _ClassBodyDeclarationsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_SEMICOLON);
            list.add(_ClassBodyDeclarationsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnumBodyDeclarations)) return false;
            if (! super.equals(o)) return false;
            EnumBodyDeclarations other = (EnumBodyDeclarations) o;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            if (_ClassBodyDeclarationsopt == null)
                if (other._ClassBodyDeclarationsopt != null) return false;
                else; // continue
            else if (! _ClassBodyDeclarationsopt.equals(other._ClassBodyDeclarationsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_SEMICOLON.hashCode());
            hash = hash * 31 + (_ClassBodyDeclarationsopt == null ? 0 : _ClassBodyDeclarationsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _SEMICOLON.accept(v);
                if (_ClassBodyDeclarationsopt != null) _ClassBodyDeclarationsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 184:  NormalInterfaceDeclaration ::= InterfaceModifiersopt interface identifier TypeParametersopt ExtendsInterfacesopt InterfaceBody
     *</b>
     */
    static public class NormalInterfaceDeclaration extends Ast implements INormalInterfaceDeclaration
    {
        private IInterfaceModifiersopt _InterfaceModifiersopt;
        private AstToken _interface;
        private identifier _identifier;
        private TypeParameters _TypeParametersopt;
        private IExtendsInterfacesopt _ExtendsInterfacesopt;
        private InterfaceBody _InterfaceBody;

        /**
         * The value returned by <b>getInterfaceModifiersopt</b> may be <b>null</b>
         */
        public IInterfaceModifiersopt getInterfaceModifiersopt() { return _InterfaceModifiersopt; }
        public AstToken getinterface() { return _interface; }
        public identifier getidentifier() { return _identifier; }
        /**
         * The value returned by <b>getTypeParametersopt</b> may be <b>null</b>
         */
        public TypeParameters getTypeParametersopt() { return _TypeParametersopt; }
        /**
         * The value returned by <b>getExtendsInterfacesopt</b> may be <b>null</b>
         */
        public IExtendsInterfacesopt getExtendsInterfacesopt() { return _ExtendsInterfacesopt; }
        public InterfaceBody getInterfaceBody() { return _InterfaceBody; }

        public NormalInterfaceDeclaration(IToken leftIToken, IToken rightIToken,
                                          IInterfaceModifiersopt _InterfaceModifiersopt,
                                          AstToken _interface,
                                          identifier _identifier,
                                          TypeParameters _TypeParametersopt,
                                          IExtendsInterfacesopt _ExtendsInterfacesopt,
                                          InterfaceBody _InterfaceBody)
        {
            super(leftIToken, rightIToken);

            this._InterfaceModifiersopt = _InterfaceModifiersopt;
            if (_InterfaceModifiersopt != null) ((Ast) _InterfaceModifiersopt).setParent(this);
            this._interface = _interface;
            ((Ast) _interface).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._TypeParametersopt = _TypeParametersopt;
            if (_TypeParametersopt != null) ((Ast) _TypeParametersopt).setParent(this);
            this._ExtendsInterfacesopt = _ExtendsInterfacesopt;
            if (_ExtendsInterfacesopt != null) ((Ast) _ExtendsInterfacesopt).setParent(this);
            this._InterfaceBody = _InterfaceBody;
            ((Ast) _InterfaceBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InterfaceModifiersopt);
            list.add(_interface);
            list.add(_identifier);
            list.add(_TypeParametersopt);
            list.add(_ExtendsInterfacesopt);
            list.add(_InterfaceBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof NormalInterfaceDeclaration)) return false;
            if (! super.equals(o)) return false;
            NormalInterfaceDeclaration other = (NormalInterfaceDeclaration) o;
            if (_InterfaceModifiersopt == null)
                if (other._InterfaceModifiersopt != null) return false;
                else; // continue
            else if (! _InterfaceModifiersopt.equals(other._InterfaceModifiersopt)) return false;
            if (! _interface.equals(other._interface)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (_TypeParametersopt == null)
                if (other._TypeParametersopt != null) return false;
                else; // continue
            else if (! _TypeParametersopt.equals(other._TypeParametersopt)) return false;
            if (_ExtendsInterfacesopt == null)
                if (other._ExtendsInterfacesopt != null) return false;
                else; // continue
            else if (! _ExtendsInterfacesopt.equals(other._ExtendsInterfacesopt)) return false;
            if (! _InterfaceBody.equals(other._InterfaceBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InterfaceModifiersopt == null ? 0 : _InterfaceModifiersopt.hashCode());
            hash = hash * 31 + (_interface.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_TypeParametersopt == null ? 0 : _TypeParametersopt.hashCode());
            hash = hash * 31 + (_ExtendsInterfacesopt == null ? 0 : _ExtendsInterfacesopt.hashCode());
            hash = hash * 31 + (_InterfaceBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_InterfaceModifiersopt != null) _InterfaceModifiersopt.accept(v);
                _interface.accept(v);
                _identifier.accept(v);
                if (_TypeParametersopt != null) _TypeParametersopt.accept(v);
                if (_ExtendsInterfacesopt != null) _ExtendsInterfacesopt.accept(v);
                _InterfaceBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 185:  InterfaceModifiers ::= InterfaceModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 186:  InterfaceModifiers ::= InterfaceModifiers InterfaceModifier
     *</b>
     */
    static public class InterfaceModifiers extends Ast implements IInterfaceModifiers
    {
        private IInterfaceModifiers _InterfaceModifiers;
        private IInterfaceModifier _InterfaceModifier;

        public IInterfaceModifiers getInterfaceModifiers() { return _InterfaceModifiers; }
        public IInterfaceModifier getInterfaceModifier() { return _InterfaceModifier; }

        public InterfaceModifiers(IToken leftIToken, IToken rightIToken,
                                  IInterfaceModifiers _InterfaceModifiers,
                                  IInterfaceModifier _InterfaceModifier)
        {
            super(leftIToken, rightIToken);

            this._InterfaceModifiers = _InterfaceModifiers;
            ((Ast) _InterfaceModifiers).setParent(this);
            this._InterfaceModifier = _InterfaceModifier;
            ((Ast) _InterfaceModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InterfaceModifiers);
            list.add(_InterfaceModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InterfaceModifiers)) return false;
            if (! super.equals(o)) return false;
            InterfaceModifiers other = (InterfaceModifiers) o;
            if (! _InterfaceModifiers.equals(other._InterfaceModifiers)) return false;
            if (! _InterfaceModifier.equals(other._InterfaceModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InterfaceModifiers.hashCode());
            hash = hash * 31 + (_InterfaceModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _InterfaceModifiers.accept(v);
                _InterfaceModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 196:  InterfaceBody ::= { InterfaceMemberDeclarationsopt }
     *</b>
     */
    static public class InterfaceBody extends Ast implements IInterfaceBody
    {
        private AstToken _LBRACE;
        private IInterfaceMemberDeclarationsopt _InterfaceMemberDeclarationsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getInterfaceMemberDeclarationsopt</b> may be <b>null</b>
         */
        public IInterfaceMemberDeclarationsopt getInterfaceMemberDeclarationsopt() { return _InterfaceMemberDeclarationsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public InterfaceBody(IToken leftIToken, IToken rightIToken,
                             AstToken _LBRACE,
                             IInterfaceMemberDeclarationsopt _InterfaceMemberDeclarationsopt,
                             AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._InterfaceMemberDeclarationsopt = _InterfaceMemberDeclarationsopt;
            if (_InterfaceMemberDeclarationsopt != null) ((Ast) _InterfaceMemberDeclarationsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_InterfaceMemberDeclarationsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InterfaceBody)) return false;
            if (! super.equals(o)) return false;
            InterfaceBody other = (InterfaceBody) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_InterfaceMemberDeclarationsopt == null)
                if (other._InterfaceMemberDeclarationsopt != null) return false;
                else; // continue
            else if (! _InterfaceMemberDeclarationsopt.equals(other._InterfaceMemberDeclarationsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_InterfaceMemberDeclarationsopt == null ? 0 : _InterfaceMemberDeclarationsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_InterfaceMemberDeclarationsopt != null) _InterfaceMemberDeclarationsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 197:  InterfaceMemberDeclarations ::= InterfaceMemberDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 198:  InterfaceMemberDeclarations ::= InterfaceMemberDeclarations InterfaceMemberDeclaration
     *</b>
     */
    static public class InterfaceMemberDeclarations extends Ast implements IInterfaceMemberDeclarations
    {
        private IInterfaceMemberDeclarations _InterfaceMemberDeclarations;
        private IInterfaceMemberDeclaration _InterfaceMemberDeclaration;

        public IInterfaceMemberDeclarations getInterfaceMemberDeclarations() { return _InterfaceMemberDeclarations; }
        public IInterfaceMemberDeclaration getInterfaceMemberDeclaration() { return _InterfaceMemberDeclaration; }

        public InterfaceMemberDeclarations(IToken leftIToken, IToken rightIToken,
                                           IInterfaceMemberDeclarations _InterfaceMemberDeclarations,
                                           IInterfaceMemberDeclaration _InterfaceMemberDeclaration)
        {
            super(leftIToken, rightIToken);

            this._InterfaceMemberDeclarations = _InterfaceMemberDeclarations;
            ((Ast) _InterfaceMemberDeclarations).setParent(this);
            this._InterfaceMemberDeclaration = _InterfaceMemberDeclaration;
            ((Ast) _InterfaceMemberDeclaration).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InterfaceMemberDeclarations);
            list.add(_InterfaceMemberDeclaration);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InterfaceMemberDeclarations)) return false;
            if (! super.equals(o)) return false;
            InterfaceMemberDeclarations other = (InterfaceMemberDeclarations) o;
            if (! _InterfaceMemberDeclarations.equals(other._InterfaceMemberDeclarations)) return false;
            if (! _InterfaceMemberDeclaration.equals(other._InterfaceMemberDeclaration)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InterfaceMemberDeclarations.hashCode());
            hash = hash * 31 + (_InterfaceMemberDeclaration.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _InterfaceMemberDeclarations.accept(v);
                _InterfaceMemberDeclaration.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 199:  InterfaceMemberDeclaration ::= ConstantDeclaration
     *<li>Rule 200:  InterfaceMemberDeclaration ::= AbstractMethodDeclaration
     *<li>Rule 201:  InterfaceMemberDeclaration ::= ClassDeclaration
     *<li>Rule 202:  InterfaceMemberDeclaration ::= InterfaceDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 203:  InterfaceMemberDeclaration ::= ;
     *</b>
     */
    static public class InterfaceMemberDeclaration extends AstToken implements IInterfaceMemberDeclaration
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public InterfaceMemberDeclaration(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 204:  ConstantDeclaration ::= ConstantModifiersopt Type VariableDeclarators
     *</b>
     */
    static public class ConstantDeclaration extends Ast implements IConstantDeclaration
    {
        private IConstantModifiersopt _ConstantModifiersopt;
        private IType _Type;
        private IVariableDeclarators _VariableDeclarators;

        /**
         * The value returned by <b>getConstantModifiersopt</b> may be <b>null</b>
         */
        public IConstantModifiersopt getConstantModifiersopt() { return _ConstantModifiersopt; }
        public IType getType() { return _Type; }
        public IVariableDeclarators getVariableDeclarators() { return _VariableDeclarators; }

        public ConstantDeclaration(IToken leftIToken, IToken rightIToken,
                                   IConstantModifiersopt _ConstantModifiersopt,
                                   IType _Type,
                                   IVariableDeclarators _VariableDeclarators)
        {
            super(leftIToken, rightIToken);

            this._ConstantModifiersopt = _ConstantModifiersopt;
            if (_ConstantModifiersopt != null) ((Ast) _ConstantModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._VariableDeclarators = _VariableDeclarators;
            ((Ast) _VariableDeclarators).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConstantModifiersopt);
            list.add(_Type);
            list.add(_VariableDeclarators);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstantDeclaration)) return false;
            if (! super.equals(o)) return false;
            ConstantDeclaration other = (ConstantDeclaration) o;
            if (_ConstantModifiersopt == null)
                if (other._ConstantModifiersopt != null) return false;
                else; // continue
            else if (! _ConstantModifiersopt.equals(other._ConstantModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (! _VariableDeclarators.equals(other._VariableDeclarators)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConstantModifiersopt == null ? 0 : _ConstantModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_VariableDeclarators.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_ConstantModifiersopt != null) _ConstantModifiersopt.accept(v);
                _Type.accept(v);
                _VariableDeclarators.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 205:  ConstantModifiers ::= ConstantModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 206:  ConstantModifiers ::= ConstantModifiers ConstantModifier
     *</b>
     */
    static public class ConstantModifiers extends Ast implements IConstantModifiers
    {
        private IConstantModifiers _ConstantModifiers;
        private IConstantModifier _ConstantModifier;

        public IConstantModifiers getConstantModifiers() { return _ConstantModifiers; }
        public IConstantModifier getConstantModifier() { return _ConstantModifier; }

        public ConstantModifiers(IToken leftIToken, IToken rightIToken,
                                 IConstantModifiers _ConstantModifiers,
                                 IConstantModifier _ConstantModifier)
        {
            super(leftIToken, rightIToken);

            this._ConstantModifiers = _ConstantModifiers;
            ((Ast) _ConstantModifiers).setParent(this);
            this._ConstantModifier = _ConstantModifier;
            ((Ast) _ConstantModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConstantModifiers);
            list.add(_ConstantModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConstantModifiers)) return false;
            if (! super.equals(o)) return false;
            ConstantModifiers other = (ConstantModifiers) o;
            if (! _ConstantModifiers.equals(other._ConstantModifiers)) return false;
            if (! _ConstantModifier.equals(other._ConstantModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConstantModifiers.hashCode());
            hash = hash * 31 + (_ConstantModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ConstantModifiers.accept(v);
                _ConstantModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 211:  AbstractMethodDeclaration ::= AbstractMethodModifiersopt TypeParametersopt ResultType MethodDeclarator Throwsopt ;
     *</b>
     */
    static public class AbstractMethodDeclaration extends Ast implements IAbstractMethodDeclaration
    {
        private IAbstractMethodModifiersopt _AbstractMethodModifiersopt;
        private TypeParameters _TypeParametersopt;
        private IResultType _ResultType;
        private IMethodDeclarator _MethodDeclarator;
        private Throws _Throwsopt;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getAbstractMethodModifiersopt</b> may be <b>null</b>
         */
        public IAbstractMethodModifiersopt getAbstractMethodModifiersopt() { return _AbstractMethodModifiersopt; }
        /**
         * The value returned by <b>getTypeParametersopt</b> may be <b>null</b>
         */
        public TypeParameters getTypeParametersopt() { return _TypeParametersopt; }
        public IResultType getResultType() { return _ResultType; }
        public IMethodDeclarator getMethodDeclarator() { return _MethodDeclarator; }
        /**
         * The value returned by <b>getThrowsopt</b> may be <b>null</b>
         */
        public Throws getThrowsopt() { return _Throwsopt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public AbstractMethodDeclaration(IToken leftIToken, IToken rightIToken,
                                         IAbstractMethodModifiersopt _AbstractMethodModifiersopt,
                                         TypeParameters _TypeParametersopt,
                                         IResultType _ResultType,
                                         IMethodDeclarator _MethodDeclarator,
                                         Throws _Throwsopt,
                                         AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._AbstractMethodModifiersopt = _AbstractMethodModifiersopt;
            if (_AbstractMethodModifiersopt != null) ((Ast) _AbstractMethodModifiersopt).setParent(this);
            this._TypeParametersopt = _TypeParametersopt;
            if (_TypeParametersopt != null) ((Ast) _TypeParametersopt).setParent(this);
            this._ResultType = _ResultType;
            ((Ast) _ResultType).setParent(this);
            this._MethodDeclarator = _MethodDeclarator;
            ((Ast) _MethodDeclarator).setParent(this);
            this._Throwsopt = _Throwsopt;
            if (_Throwsopt != null) ((Ast) _Throwsopt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AbstractMethodModifiersopt);
            list.add(_TypeParametersopt);
            list.add(_ResultType);
            list.add(_MethodDeclarator);
            list.add(_Throwsopt);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AbstractMethodDeclaration)) return false;
            if (! super.equals(o)) return false;
            AbstractMethodDeclaration other = (AbstractMethodDeclaration) o;
            if (_AbstractMethodModifiersopt == null)
                if (other._AbstractMethodModifiersopt != null) return false;
                else; // continue
            else if (! _AbstractMethodModifiersopt.equals(other._AbstractMethodModifiersopt)) return false;
            if (_TypeParametersopt == null)
                if (other._TypeParametersopt != null) return false;
                else; // continue
            else if (! _TypeParametersopt.equals(other._TypeParametersopt)) return false;
            if (! _ResultType.equals(other._ResultType)) return false;
            if (! _MethodDeclarator.equals(other._MethodDeclarator)) return false;
            if (_Throwsopt == null)
                if (other._Throwsopt != null) return false;
                else; // continue
            else if (! _Throwsopt.equals(other._Throwsopt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AbstractMethodModifiersopt == null ? 0 : _AbstractMethodModifiersopt.hashCode());
            hash = hash * 31 + (_TypeParametersopt == null ? 0 : _TypeParametersopt.hashCode());
            hash = hash * 31 + (_ResultType.hashCode());
            hash = hash * 31 + (_MethodDeclarator.hashCode());
            hash = hash * 31 + (_Throwsopt == null ? 0 : _Throwsopt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_AbstractMethodModifiersopt != null) _AbstractMethodModifiersopt.accept(v);
                if (_TypeParametersopt != null) _TypeParametersopt.accept(v);
                _ResultType.accept(v);
                _MethodDeclarator.accept(v);
                if (_Throwsopt != null) _Throwsopt.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 212:  AbstractMethodModifiers ::= AbstractMethodModifier
     *</em>
     *<p>
     *<b>
     *<li>Rule 213:  AbstractMethodModifiers ::= AbstractMethodModifiers AbstractMethodModifier
     *</b>
     */
    static public class AbstractMethodModifiers extends Ast implements IAbstractMethodModifiers
    {
        private IAbstractMethodModifiers _AbstractMethodModifiers;
        private IAbstractMethodModifier _AbstractMethodModifier;

        public IAbstractMethodModifiers getAbstractMethodModifiers() { return _AbstractMethodModifiers; }
        public IAbstractMethodModifier getAbstractMethodModifier() { return _AbstractMethodModifier; }

        public AbstractMethodModifiers(IToken leftIToken, IToken rightIToken,
                                       IAbstractMethodModifiers _AbstractMethodModifiers,
                                       IAbstractMethodModifier _AbstractMethodModifier)
        {
            super(leftIToken, rightIToken);

            this._AbstractMethodModifiers = _AbstractMethodModifiers;
            ((Ast) _AbstractMethodModifiers).setParent(this);
            this._AbstractMethodModifier = _AbstractMethodModifier;
            ((Ast) _AbstractMethodModifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AbstractMethodModifiers);
            list.add(_AbstractMethodModifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AbstractMethodModifiers)) return false;
            if (! super.equals(o)) return false;
            AbstractMethodModifiers other = (AbstractMethodModifiers) o;
            if (! _AbstractMethodModifiers.equals(other._AbstractMethodModifiers)) return false;
            if (! _AbstractMethodModifier.equals(other._AbstractMethodModifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AbstractMethodModifiers.hashCode());
            hash = hash * 31 + (_AbstractMethodModifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AbstractMethodModifiers.accept(v);
                _AbstractMethodModifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 217:  AnnotationTypeDeclaration ::= InterfaceModifiersopt @ interface identifier AnnotationTypeBody
     *</b>
     */
    static public class AnnotationTypeDeclaration extends Ast implements IAnnotationTypeDeclaration
    {
        private IInterfaceModifiersopt _InterfaceModifiersopt;
        private AstToken _AT;
        private AstToken _interface;
        private identifier _identifier;
        private AnnotationTypeBody _AnnotationTypeBody;

        /**
         * The value returned by <b>getInterfaceModifiersopt</b> may be <b>null</b>
         */
        public IInterfaceModifiersopt getInterfaceModifiersopt() { return _InterfaceModifiersopt; }
        public AstToken getAT() { return _AT; }
        public AstToken getinterface() { return _interface; }
        public identifier getidentifier() { return _identifier; }
        public AnnotationTypeBody getAnnotationTypeBody() { return _AnnotationTypeBody; }

        public AnnotationTypeDeclaration(IToken leftIToken, IToken rightIToken,
                                         IInterfaceModifiersopt _InterfaceModifiersopt,
                                         AstToken _AT,
                                         AstToken _interface,
                                         identifier _identifier,
                                         AnnotationTypeBody _AnnotationTypeBody)
        {
            super(leftIToken, rightIToken);

            this._InterfaceModifiersopt = _InterfaceModifiersopt;
            if (_InterfaceModifiersopt != null) ((Ast) _InterfaceModifiersopt).setParent(this);
            this._AT = _AT;
            ((Ast) _AT).setParent(this);
            this._interface = _interface;
            ((Ast) _interface).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._AnnotationTypeBody = _AnnotationTypeBody;
            ((Ast) _AnnotationTypeBody).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InterfaceModifiersopt);
            list.add(_AT);
            list.add(_interface);
            list.add(_identifier);
            list.add(_AnnotationTypeBody);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AnnotationTypeDeclaration)) return false;
            if (! super.equals(o)) return false;
            AnnotationTypeDeclaration other = (AnnotationTypeDeclaration) o;
            if (_InterfaceModifiersopt == null)
                if (other._InterfaceModifiersopt != null) return false;
                else; // continue
            else if (! _InterfaceModifiersopt.equals(other._InterfaceModifiersopt)) return false;
            if (! _AT.equals(other._AT)) return false;
            if (! _interface.equals(other._interface)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _AnnotationTypeBody.equals(other._AnnotationTypeBody)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InterfaceModifiersopt == null ? 0 : _InterfaceModifiersopt.hashCode());
            hash = hash * 31 + (_AT.hashCode());
            hash = hash * 31 + (_interface.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_AnnotationTypeBody.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_InterfaceModifiersopt != null) _InterfaceModifiersopt.accept(v);
                _AT.accept(v);
                _interface.accept(v);
                _identifier.accept(v);
                _AnnotationTypeBody.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 218:  AnnotationTypeBody ::= { AnnotationTypeElementDeclarationsopt }
     *</b>
     */
    static public class AnnotationTypeBody extends Ast implements IAnnotationTypeBody
    {
        private AstToken _LBRACE;
        private IAnnotationTypeElementDeclarationsopt _AnnotationTypeElementDeclarationsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getAnnotationTypeElementDeclarationsopt</b> may be <b>null</b>
         */
        public IAnnotationTypeElementDeclarationsopt getAnnotationTypeElementDeclarationsopt() { return _AnnotationTypeElementDeclarationsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public AnnotationTypeBody(IToken leftIToken, IToken rightIToken,
                                  AstToken _LBRACE,
                                  IAnnotationTypeElementDeclarationsopt _AnnotationTypeElementDeclarationsopt,
                                  AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._AnnotationTypeElementDeclarationsopt = _AnnotationTypeElementDeclarationsopt;
            if (_AnnotationTypeElementDeclarationsopt != null) ((Ast) _AnnotationTypeElementDeclarationsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_AnnotationTypeElementDeclarationsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AnnotationTypeBody)) return false;
            if (! super.equals(o)) return false;
            AnnotationTypeBody other = (AnnotationTypeBody) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_AnnotationTypeElementDeclarationsopt == null)
                if (other._AnnotationTypeElementDeclarationsopt != null) return false;
                else; // continue
            else if (! _AnnotationTypeElementDeclarationsopt.equals(other._AnnotationTypeElementDeclarationsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_AnnotationTypeElementDeclarationsopt == null ? 0 : _AnnotationTypeElementDeclarationsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_AnnotationTypeElementDeclarationsopt != null) _AnnotationTypeElementDeclarationsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 219:  AnnotationTypeElementDeclarations ::= AnnotationTypeElementDeclaration
     *</em>
     *<p>
     *<b>
     *<li>Rule 220:  AnnotationTypeElementDeclarations ::= AnnotationTypeElementDeclarations AnnotationTypeElementDeclaration
     *</b>
     */
    static public class AnnotationTypeElementDeclarations extends Ast implements IAnnotationTypeElementDeclarations
    {
        private IAnnotationTypeElementDeclarations _AnnotationTypeElementDeclarations;
        private IAnnotationTypeElementDeclaration _AnnotationTypeElementDeclaration;

        public IAnnotationTypeElementDeclarations getAnnotationTypeElementDeclarations() { return _AnnotationTypeElementDeclarations; }
        public IAnnotationTypeElementDeclaration getAnnotationTypeElementDeclaration() { return _AnnotationTypeElementDeclaration; }

        public AnnotationTypeElementDeclarations(IToken leftIToken, IToken rightIToken,
                                                 IAnnotationTypeElementDeclarations _AnnotationTypeElementDeclarations,
                                                 IAnnotationTypeElementDeclaration _AnnotationTypeElementDeclaration)
        {
            super(leftIToken, rightIToken);

            this._AnnotationTypeElementDeclarations = _AnnotationTypeElementDeclarations;
            ((Ast) _AnnotationTypeElementDeclarations).setParent(this);
            this._AnnotationTypeElementDeclaration = _AnnotationTypeElementDeclaration;
            ((Ast) _AnnotationTypeElementDeclaration).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AnnotationTypeElementDeclarations);
            list.add(_AnnotationTypeElementDeclaration);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AnnotationTypeElementDeclarations)) return false;
            if (! super.equals(o)) return false;
            AnnotationTypeElementDeclarations other = (AnnotationTypeElementDeclarations) o;
            if (! _AnnotationTypeElementDeclarations.equals(other._AnnotationTypeElementDeclarations)) return false;
            if (! _AnnotationTypeElementDeclaration.equals(other._AnnotationTypeElementDeclaration)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AnnotationTypeElementDeclarations.hashCode());
            hash = hash * 31 + (_AnnotationTypeElementDeclaration.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AnnotationTypeElementDeclarations.accept(v);
                _AnnotationTypeElementDeclaration.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 228:  DefaultValue ::= default ElementValue
     *</b>
     */
    static public class DefaultValue extends Ast implements IDefaultValue
    {
        private AstToken _default;
        private IElementValue _ElementValue;

        public AstToken getdefault() { return _default; }
        public IElementValue getElementValue() { return _ElementValue; }

        public DefaultValue(IToken leftIToken, IToken rightIToken,
                            AstToken _default,
                            IElementValue _ElementValue)
        {
            super(leftIToken, rightIToken);

            this._default = _default;
            ((Ast) _default).setParent(this);
            this._ElementValue = _ElementValue;
            ((Ast) _ElementValue).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_default);
            list.add(_ElementValue);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof DefaultValue)) return false;
            if (! super.equals(o)) return false;
            DefaultValue other = (DefaultValue) o;
            if (! _default.equals(other._default)) return false;
            if (! _ElementValue.equals(other._ElementValue)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_default.hashCode());
            hash = hash * 31 + (_ElementValue.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _default.accept(v);
                _ElementValue.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 229:  Annotations ::= Annotation
     *</em>
     *<p>
     *<b>
     *<li>Rule 230:  Annotations ::= Annotations Annotation
     *</b>
     */
    static public class Annotations extends Ast implements IAnnotations
    {
        private IAnnotations _Annotations;
        private IAnnotation _Annotation;

        public IAnnotations getAnnotations() { return _Annotations; }
        public IAnnotation getAnnotation() { return _Annotation; }

        public Annotations(IToken leftIToken, IToken rightIToken,
                           IAnnotations _Annotations,
                           IAnnotation _Annotation)
        {
            super(leftIToken, rightIToken);

            this._Annotations = _Annotations;
            ((Ast) _Annotations).setParent(this);
            this._Annotation = _Annotation;
            ((Ast) _Annotation).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Annotations);
            list.add(_Annotation);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Annotations)) return false;
            if (! super.equals(o)) return false;
            Annotations other = (Annotations) o;
            if (! _Annotations.equals(other._Annotations)) return false;
            if (! _Annotation.equals(other._Annotation)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Annotations.hashCode());
            hash = hash * 31 + (_Annotation.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Annotations.accept(v);
                _Annotation.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 234:  NormalAnnotation ::= @ TypeName ( ElementValuePairsopt )
     *</b>
     */
    static public class NormalAnnotation extends Ast implements INormalAnnotation
    {
        private AstToken _AT;
        private ITypeName _TypeName;
        private AstToken _LPAREN;
        private IElementValuePairsopt _ElementValuePairsopt;
        private AstToken _RPAREN;

        public AstToken getAT() { return _AT; }
        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getElementValuePairsopt</b> may be <b>null</b>
         */
        public IElementValuePairsopt getElementValuePairsopt() { return _ElementValuePairsopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public NormalAnnotation(IToken leftIToken, IToken rightIToken,
                                AstToken _AT,
                                ITypeName _TypeName,
                                AstToken _LPAREN,
                                IElementValuePairsopt _ElementValuePairsopt,
                                AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._AT = _AT;
            ((Ast) _AT).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ElementValuePairsopt = _ElementValuePairsopt;
            if (_ElementValuePairsopt != null) ((Ast) _ElementValuePairsopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AT);
            list.add(_TypeName);
            list.add(_LPAREN);
            list.add(_ElementValuePairsopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof NormalAnnotation)) return false;
            if (! super.equals(o)) return false;
            NormalAnnotation other = (NormalAnnotation) o;
            if (! _AT.equals(other._AT)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ElementValuePairsopt == null)
                if (other._ElementValuePairsopt != null) return false;
                else; // continue
            else if (! _ElementValuePairsopt.equals(other._ElementValuePairsopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AT.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ElementValuePairsopt == null ? 0 : _ElementValuePairsopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AT.accept(v);
                _TypeName.accept(v);
                _LPAREN.accept(v);
                if (_ElementValuePairsopt != null) _ElementValuePairsopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 235:  ElementValuePairs ::= ElementValuePair
     *</em>
     *<p>
     *<b>
     *<li>Rule 236:  ElementValuePairs ::= ElementValuePairs , ElementValuePair
     *</b>
     */
    static public class ElementValuePairs extends Ast implements IElementValuePairs
    {
        private IElementValuePairs _ElementValuePairs;
        private AstToken _COMMA;
        private ElementValuePair _ElementValuePair;

        public IElementValuePairs getElementValuePairs() { return _ElementValuePairs; }
        public AstToken getCOMMA() { return _COMMA; }
        public ElementValuePair getElementValuePair() { return _ElementValuePair; }

        public ElementValuePairs(IToken leftIToken, IToken rightIToken,
                                 IElementValuePairs _ElementValuePairs,
                                 AstToken _COMMA,
                                 ElementValuePair _ElementValuePair)
        {
            super(leftIToken, rightIToken);

            this._ElementValuePairs = _ElementValuePairs;
            ((Ast) _ElementValuePairs).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._ElementValuePair = _ElementValuePair;
            ((Ast) _ElementValuePair).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ElementValuePairs);
            list.add(_COMMA);
            list.add(_ElementValuePair);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ElementValuePairs)) return false;
            if (! super.equals(o)) return false;
            ElementValuePairs other = (ElementValuePairs) o;
            if (! _ElementValuePairs.equals(other._ElementValuePairs)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _ElementValuePair.equals(other._ElementValuePair)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ElementValuePairs.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_ElementValuePair.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ElementValuePairs.accept(v);
                _COMMA.accept(v);
                _ElementValuePair.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 237:  ElementValuePair ::= SimpleName = ElementValue
     *</b>
     */
    static public class ElementValuePair extends Ast implements IElementValuePair
    {
        private identifier _SimpleName;
        private AstToken _EQUAL;
        private IElementValue _ElementValue;

        public identifier getSimpleName() { return _SimpleName; }
        public AstToken getEQUAL() { return _EQUAL; }
        public IElementValue getElementValue() { return _ElementValue; }

        public ElementValuePair(IToken leftIToken, IToken rightIToken,
                                identifier _SimpleName,
                                AstToken _EQUAL,
                                IElementValue _ElementValue)
        {
            super(leftIToken, rightIToken);

            this._SimpleName = _SimpleName;
            ((Ast) _SimpleName).setParent(this);
            this._EQUAL = _EQUAL;
            ((Ast) _EQUAL).setParent(this);
            this._ElementValue = _ElementValue;
            ((Ast) _ElementValue).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_SimpleName);
            list.add(_EQUAL);
            list.add(_ElementValue);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ElementValuePair)) return false;
            if (! super.equals(o)) return false;
            ElementValuePair other = (ElementValuePair) o;
            if (! _SimpleName.equals(other._SimpleName)) return false;
            if (! _EQUAL.equals(other._EQUAL)) return false;
            if (! _ElementValue.equals(other._ElementValue)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_SimpleName.hashCode());
            hash = hash * 31 + (_EQUAL.hashCode());
            hash = hash * 31 + (_ElementValue.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _SimpleName.accept(v);
                _EQUAL.accept(v);
                _ElementValue.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 242:  ElementValueArrayInitializer ::= { ElementValuesopt ,opt }
     *</b>
     */
    static public class ElementValueArrayInitializer extends Ast implements IElementValueArrayInitializer
    {
        private AstToken _LBRACE;
        private IElementValuesopt _ElementValuesopt;
        private Commaopt _Commaopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getElementValuesopt</b> may be <b>null</b>
         */
        public IElementValuesopt getElementValuesopt() { return _ElementValuesopt; }
        /**
         * The value returned by <b>getCommaopt</b> may be <b>null</b>
         */
        public Commaopt getCommaopt() { return _Commaopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public ElementValueArrayInitializer(IToken leftIToken, IToken rightIToken,
                                            AstToken _LBRACE,
                                            IElementValuesopt _ElementValuesopt,
                                            Commaopt _Commaopt,
                                            AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._ElementValuesopt = _ElementValuesopt;
            if (_ElementValuesopt != null) ((Ast) _ElementValuesopt).setParent(this);
            this._Commaopt = _Commaopt;
            if (_Commaopt != null) ((Ast) _Commaopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_ElementValuesopt);
            list.add(_Commaopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ElementValueArrayInitializer)) return false;
            if (! super.equals(o)) return false;
            ElementValueArrayInitializer other = (ElementValueArrayInitializer) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_ElementValuesopt == null)
                if (other._ElementValuesopt != null) return false;
                else; // continue
            else if (! _ElementValuesopt.equals(other._ElementValuesopt)) return false;
            if (_Commaopt == null)
                if (other._Commaopt != null) return false;
                else; // continue
            else if (! _Commaopt.equals(other._Commaopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_ElementValuesopt == null ? 0 : _ElementValuesopt.hashCode());
            hash = hash * 31 + (_Commaopt == null ? 0 : _Commaopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_ElementValuesopt != null) _ElementValuesopt.accept(v);
                if (_Commaopt != null) _Commaopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 243:  ElementValues ::= ElementValue
     *</em>
     *<p>
     *<b>
     *<li>Rule 244:  ElementValues ::= ElementValues , ElementValue
     *</b>
     */
    static public class ElementValues extends Ast implements IElementValues
    {
        private IElementValues _ElementValues;
        private AstToken _COMMA;
        private IElementValue _ElementValue;

        public IElementValues getElementValues() { return _ElementValues; }
        public AstToken getCOMMA() { return _COMMA; }
        public IElementValue getElementValue() { return _ElementValue; }

        public ElementValues(IToken leftIToken, IToken rightIToken,
                             IElementValues _ElementValues,
                             AstToken _COMMA,
                             IElementValue _ElementValue)
        {
            super(leftIToken, rightIToken);

            this._ElementValues = _ElementValues;
            ((Ast) _ElementValues).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._ElementValue = _ElementValue;
            ((Ast) _ElementValue).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ElementValues);
            list.add(_COMMA);
            list.add(_ElementValue);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ElementValues)) return false;
            if (! super.equals(o)) return false;
            ElementValues other = (ElementValues) o;
            if (! _ElementValues.equals(other._ElementValues)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _ElementValue.equals(other._ElementValue)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ElementValues.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_ElementValue.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ElementValues.accept(v);
                _COMMA.accept(v);
                _ElementValue.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 245:  MarkerAnnotation ::= @ TypeName
     *</b>
     */
    static public class MarkerAnnotation extends Ast implements IMarkerAnnotation
    {
        private AstToken _AT;
        private ITypeName _TypeName;

        public AstToken getAT() { return _AT; }
        public ITypeName getTypeName() { return _TypeName; }

        public MarkerAnnotation(IToken leftIToken, IToken rightIToken,
                                AstToken _AT,
                                ITypeName _TypeName)
        {
            super(leftIToken, rightIToken);

            this._AT = _AT;
            ((Ast) _AT).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AT);
            list.add(_TypeName);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MarkerAnnotation)) return false;
            if (! super.equals(o)) return false;
            MarkerAnnotation other = (MarkerAnnotation) o;
            if (! _AT.equals(other._AT)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AT.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AT.accept(v);
                _TypeName.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 246:  SingleElementAnnotation ::= @ TypeName ( ElementValue )
     *</b>
     */
    static public class SingleElementAnnotation extends Ast implements ISingleElementAnnotation
    {
        private AstToken _AT;
        private ITypeName _TypeName;
        private AstToken _LPAREN;
        private IElementValue _ElementValue;
        private AstToken _RPAREN;

        public AstToken getAT() { return _AT; }
        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IElementValue getElementValue() { return _ElementValue; }
        public AstToken getRPAREN() { return _RPAREN; }

        public SingleElementAnnotation(IToken leftIToken, IToken rightIToken,
                                       AstToken _AT,
                                       ITypeName _TypeName,
                                       AstToken _LPAREN,
                                       IElementValue _ElementValue,
                                       AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._AT = _AT;
            ((Ast) _AT).setParent(this);
            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ElementValue = _ElementValue;
            ((Ast) _ElementValue).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AT);
            list.add(_TypeName);
            list.add(_LPAREN);
            list.add(_ElementValue);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SingleElementAnnotation)) return false;
            if (! super.equals(o)) return false;
            SingleElementAnnotation other = (SingleElementAnnotation) o;
            if (! _AT.equals(other._AT)) return false;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _ElementValue.equals(other._ElementValue)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AT.hashCode());
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ElementValue.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AT.accept(v);
                _TypeName.accept(v);
                _LPAREN.accept(v);
                _ElementValue.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 247:  ArrayInitializer ::= { VariableInitializersopt ,opt }
     *</b>
     */
    static public class ArrayInitializer extends Ast implements IArrayInitializer
    {
        private AstToken _LBRACE;
        private IVariableInitializersopt _VariableInitializersopt;
        private Commaopt _Commaopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getVariableInitializersopt</b> may be <b>null</b>
         */
        public IVariableInitializersopt getVariableInitializersopt() { return _VariableInitializersopt; }
        /**
         * The value returned by <b>getCommaopt</b> may be <b>null</b>
         */
        public Commaopt getCommaopt() { return _Commaopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public ArrayInitializer(IToken leftIToken, IToken rightIToken,
                                AstToken _LBRACE,
                                IVariableInitializersopt _VariableInitializersopt,
                                Commaopt _Commaopt,
                                AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._VariableInitializersopt = _VariableInitializersopt;
            if (_VariableInitializersopt != null) ((Ast) _VariableInitializersopt).setParent(this);
            this._Commaopt = _Commaopt;
            if (_Commaopt != null) ((Ast) _Commaopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_VariableInitializersopt);
            list.add(_Commaopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayInitializer)) return false;
            if (! super.equals(o)) return false;
            ArrayInitializer other = (ArrayInitializer) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_VariableInitializersopt == null)
                if (other._VariableInitializersopt != null) return false;
                else; // continue
            else if (! _VariableInitializersopt.equals(other._VariableInitializersopt)) return false;
            if (_Commaopt == null)
                if (other._Commaopt != null) return false;
                else; // continue
            else if (! _Commaopt.equals(other._Commaopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_VariableInitializersopt == null ? 0 : _VariableInitializersopt.hashCode());
            hash = hash * 31 + (_Commaopt == null ? 0 : _Commaopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_VariableInitializersopt != null) _VariableInitializersopt.accept(v);
                if (_Commaopt != null) _Commaopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 248:  VariableInitializers ::= VariableInitializer
     *</em>
     *<p>
     *<b>
     *<li>Rule 249:  VariableInitializers ::= VariableInitializers , VariableInitializer
     *</b>
     */
    static public class VariableInitializers extends Ast implements IVariableInitializers
    {
        private IVariableInitializers _VariableInitializers;
        private AstToken _COMMA;
        private IVariableInitializer _VariableInitializer;

        public IVariableInitializers getVariableInitializers() { return _VariableInitializers; }
        public AstToken getCOMMA() { return _COMMA; }
        public IVariableInitializer getVariableInitializer() { return _VariableInitializer; }

        public VariableInitializers(IToken leftIToken, IToken rightIToken,
                                    IVariableInitializers _VariableInitializers,
                                    AstToken _COMMA,
                                    IVariableInitializer _VariableInitializer)
        {
            super(leftIToken, rightIToken);

            this._VariableInitializers = _VariableInitializers;
            ((Ast) _VariableInitializers).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._VariableInitializer = _VariableInitializer;
            ((Ast) _VariableInitializer).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableInitializers);
            list.add(_COMMA);
            list.add(_VariableInitializer);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof VariableInitializers)) return false;
            if (! super.equals(o)) return false;
            VariableInitializers other = (VariableInitializers) o;
            if (! _VariableInitializers.equals(other._VariableInitializers)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _VariableInitializer.equals(other._VariableInitializer)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableInitializers.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_VariableInitializer.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _VariableInitializers.accept(v);
                _COMMA.accept(v);
                _VariableInitializer.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 250:  Block ::= { BlockStatementsopt }
     *</b>
     */
    static public class Block extends Ast implements IBlock
    {
        private AstToken _LBRACE;
        private IBlockStatementsopt _BlockStatementsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getBlockStatementsopt</b> may be <b>null</b>
         */
        public IBlockStatementsopt getBlockStatementsopt() { return _BlockStatementsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public Block(IToken leftIToken, IToken rightIToken,
                     AstToken _LBRACE,
                     IBlockStatementsopt _BlockStatementsopt,
                     AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._BlockStatementsopt = _BlockStatementsopt;
            if (_BlockStatementsopt != null) ((Ast) _BlockStatementsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_BlockStatementsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Block)) return false;
            if (! super.equals(o)) return false;
            Block other = (Block) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_BlockStatementsopt == null)
                if (other._BlockStatementsopt != null) return false;
                else; // continue
            else if (! _BlockStatementsopt.equals(other._BlockStatementsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_BlockStatementsopt == null ? 0 : _BlockStatementsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_BlockStatementsopt != null) _BlockStatementsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 251:  BlockStatements ::= BlockStatement
     *</em>
     *<p>
     *<b>
     *<li>Rule 252:  BlockStatements ::= BlockStatements BlockStatement
     *</b>
     */
    static public class BlockStatements extends Ast implements IBlockStatements
    {
        private IBlockStatements _BlockStatements;
        private IBlockStatement _BlockStatement;

        public IBlockStatements getBlockStatements() { return _BlockStatements; }
        public IBlockStatement getBlockStatement() { return _BlockStatement; }

        public BlockStatements(IToken leftIToken, IToken rightIToken,
                               IBlockStatements _BlockStatements,
                               IBlockStatement _BlockStatement)
        {
            super(leftIToken, rightIToken);

            this._BlockStatements = _BlockStatements;
            ((Ast) _BlockStatements).setParent(this);
            this._BlockStatement = _BlockStatement;
            ((Ast) _BlockStatement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_BlockStatements);
            list.add(_BlockStatement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof BlockStatements)) return false;
            if (! super.equals(o)) return false;
            BlockStatements other = (BlockStatements) o;
            if (! _BlockStatements.equals(other._BlockStatements)) return false;
            if (! _BlockStatement.equals(other._BlockStatement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_BlockStatements.hashCode());
            hash = hash * 31 + (_BlockStatement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _BlockStatements.accept(v);
                _BlockStatement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 256:  LocalVariableDeclarationStatement ::= LocalVariableDeclaration ;
     *</b>
     */
    static public class LocalVariableDeclarationStatement extends Ast implements ILocalVariableDeclarationStatement
    {
        private LocalVariableDeclaration _LocalVariableDeclaration;
        private AstToken _SEMICOLON;

        public LocalVariableDeclaration getLocalVariableDeclaration() { return _LocalVariableDeclaration; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public LocalVariableDeclarationStatement(IToken leftIToken, IToken rightIToken,
                                                 LocalVariableDeclaration _LocalVariableDeclaration,
                                                 AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._LocalVariableDeclaration = _LocalVariableDeclaration;
            ((Ast) _LocalVariableDeclaration).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LocalVariableDeclaration);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LocalVariableDeclarationStatement)) return false;
            if (! super.equals(o)) return false;
            LocalVariableDeclarationStatement other = (LocalVariableDeclarationStatement) o;
            if (! _LocalVariableDeclaration.equals(other._LocalVariableDeclaration)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LocalVariableDeclaration.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LocalVariableDeclaration.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 257:  LocalVariableDeclaration ::= VariableModifiersopt Type VariableDeclarators
     *</b>
     */
    static public class LocalVariableDeclaration extends Ast implements ILocalVariableDeclaration
    {
        private IVariableModifiersopt _VariableModifiersopt;
        private IType _Type;
        private IVariableDeclarators _VariableDeclarators;

        /**
         * The value returned by <b>getVariableModifiersopt</b> may be <b>null</b>
         */
        public IVariableModifiersopt getVariableModifiersopt() { return _VariableModifiersopt; }
        public IType getType() { return _Type; }
        public IVariableDeclarators getVariableDeclarators() { return _VariableDeclarators; }

        public LocalVariableDeclaration(IToken leftIToken, IToken rightIToken,
                                        IVariableModifiersopt _VariableModifiersopt,
                                        IType _Type,
                                        IVariableDeclarators _VariableDeclarators)
        {
            super(leftIToken, rightIToken);

            this._VariableModifiersopt = _VariableModifiersopt;
            if (_VariableModifiersopt != null) ((Ast) _VariableModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._VariableDeclarators = _VariableDeclarators;
            ((Ast) _VariableDeclarators).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_VariableModifiersopt);
            list.add(_Type);
            list.add(_VariableDeclarators);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LocalVariableDeclaration)) return false;
            if (! super.equals(o)) return false;
            LocalVariableDeclaration other = (LocalVariableDeclaration) o;
            if (_VariableModifiersopt == null)
                if (other._VariableModifiersopt != null) return false;
                else; // continue
            else if (! _VariableModifiersopt.equals(other._VariableModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (! _VariableDeclarators.equals(other._VariableDeclarators)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_VariableModifiersopt == null ? 0 : _VariableModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_VariableDeclarators.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_VariableModifiersopt != null) _VariableModifiersopt.accept(v);
                _Type.accept(v);
                _VariableDeclarators.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 281:  IfThenStatement ::= if ( Expression ) Statement
     *</b>
     */
    static public class IfThenStatement extends Ast implements IIfThenStatement
    {
        private AstToken _if;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatement _Statement;

        public AstToken getif() { return _if; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatement getStatement() { return _Statement; }

        public IfThenStatement(IToken leftIToken, IToken rightIToken,
                               AstToken _if,
                               AstToken _LPAREN,
                               IExpression _Expression,
                               AstToken _RPAREN,
                               IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._if = _if;
            ((Ast) _if).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_if);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof IfThenStatement)) return false;
            if (! super.equals(o)) return false;
            IfThenStatement other = (IfThenStatement) o;
            if (! _if.equals(other._if)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_if.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _if.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 282:  IfThenElseStatement ::= if ( Expression ) StatementNoShortIf else Statement
     *</b>
     */
    static public class IfThenElseStatement extends Ast implements IIfThenElseStatement
    {
        private AstToken _if;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatementNoShortIf _StatementNoShortIf;
        private AstToken _else;
        private IStatement _Statement;

        public AstToken getif() { return _if; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatementNoShortIf getStatementNoShortIf() { return _StatementNoShortIf; }
        public AstToken getelse() { return _else; }
        public IStatement getStatement() { return _Statement; }

        public IfThenElseStatement(IToken leftIToken, IToken rightIToken,
                                   AstToken _if,
                                   AstToken _LPAREN,
                                   IExpression _Expression,
                                   AstToken _RPAREN,
                                   IStatementNoShortIf _StatementNoShortIf,
                                   AstToken _else,
                                   IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._if = _if;
            ((Ast) _if).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._StatementNoShortIf = _StatementNoShortIf;
            ((Ast) _StatementNoShortIf).setParent(this);
            this._else = _else;
            ((Ast) _else).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_if);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_StatementNoShortIf);
            list.add(_else);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof IfThenElseStatement)) return false;
            if (! super.equals(o)) return false;
            IfThenElseStatement other = (IfThenElseStatement) o;
            if (! _if.equals(other._if)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _StatementNoShortIf.equals(other._StatementNoShortIf)) return false;
            if (! _else.equals(other._else)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_if.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_StatementNoShortIf.hashCode());
            hash = hash * 31 + (_else.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _if.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _StatementNoShortIf.accept(v);
                _else.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 283:  IfThenElseStatementNoShortIf ::= if ( Expression ) StatementNoShortIf else StatementNoShortIf
     *</b>
     */
    static public class IfThenElseStatementNoShortIf extends Ast implements IIfThenElseStatementNoShortIf
    {
        private AstToken _if;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatementNoShortIf _StatementNoShortIf;
        private AstToken _else;
        private IStatementNoShortIf _StatementNoShortIf7;

        public AstToken getif() { return _if; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatementNoShortIf getStatementNoShortIf() { return _StatementNoShortIf; }
        public AstToken getelse() { return _else; }
        public IStatementNoShortIf getStatementNoShortIf7() { return _StatementNoShortIf7; }

        public IfThenElseStatementNoShortIf(IToken leftIToken, IToken rightIToken,
                                            AstToken _if,
                                            AstToken _LPAREN,
                                            IExpression _Expression,
                                            AstToken _RPAREN,
                                            IStatementNoShortIf _StatementNoShortIf,
                                            AstToken _else,
                                            IStatementNoShortIf _StatementNoShortIf7)
        {
            super(leftIToken, rightIToken);

            this._if = _if;
            ((Ast) _if).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._StatementNoShortIf = _StatementNoShortIf;
            ((Ast) _StatementNoShortIf).setParent(this);
            this._else = _else;
            ((Ast) _else).setParent(this);
            this._StatementNoShortIf7 = _StatementNoShortIf7;
            ((Ast) _StatementNoShortIf7).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_if);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_StatementNoShortIf);
            list.add(_else);
            list.add(_StatementNoShortIf7);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof IfThenElseStatementNoShortIf)) return false;
            if (! super.equals(o)) return false;
            IfThenElseStatementNoShortIf other = (IfThenElseStatementNoShortIf) o;
            if (! _if.equals(other._if)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _StatementNoShortIf.equals(other._StatementNoShortIf)) return false;
            if (! _else.equals(other._else)) return false;
            if (! _StatementNoShortIf7.equals(other._StatementNoShortIf7)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_if.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_StatementNoShortIf.hashCode());
            hash = hash * 31 + (_else.hashCode());
            hash = hash * 31 + (_StatementNoShortIf7.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _if.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _StatementNoShortIf.accept(v);
                _else.accept(v);
                _StatementNoShortIf7.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 284:  EmptyStatement ::= ;
     *</b>
     */
    static public class EmptyStatement extends AstToken implements IEmptyStatement
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public EmptyStatement(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 285:  LabeledStatement ::= identifier : Statement
     *</b>
     */
    static public class LabeledStatement extends Ast implements ILabeledStatement
    {
        private identifier _identifier;
        private AstToken _COLON;
        private IStatement _Statement;

        public identifier getidentifier() { return _identifier; }
        public AstToken getCOLON() { return _COLON; }
        public IStatement getStatement() { return _Statement; }

        public LabeledStatement(IToken leftIToken, IToken rightIToken,
                                identifier _identifier,
                                AstToken _COLON,
                                IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_identifier);
            list.add(_COLON);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LabeledStatement)) return false;
            if (! super.equals(o)) return false;
            LabeledStatement other = (LabeledStatement) o;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _identifier.accept(v);
                _COLON.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 286:  LabeledStatementNoShortIf ::= identifier : StatementNoShortIf
     *</b>
     */
    static public class LabeledStatementNoShortIf extends Ast implements ILabeledStatementNoShortIf
    {
        private identifier _identifier;
        private AstToken _COLON;
        private IStatementNoShortIf _StatementNoShortIf;

        public identifier getidentifier() { return _identifier; }
        public AstToken getCOLON() { return _COLON; }
        public IStatementNoShortIf getStatementNoShortIf() { return _StatementNoShortIf; }

        public LabeledStatementNoShortIf(IToken leftIToken, IToken rightIToken,
                                         identifier _identifier,
                                         AstToken _COLON,
                                         IStatementNoShortIf _StatementNoShortIf)
        {
            super(leftIToken, rightIToken);

            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            this._StatementNoShortIf = _StatementNoShortIf;
            ((Ast) _StatementNoShortIf).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_identifier);
            list.add(_COLON);
            list.add(_StatementNoShortIf);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LabeledStatementNoShortIf)) return false;
            if (! super.equals(o)) return false;
            LabeledStatementNoShortIf other = (LabeledStatementNoShortIf) o;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            if (! _StatementNoShortIf.equals(other._StatementNoShortIf)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            hash = hash * 31 + (_StatementNoShortIf.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _identifier.accept(v);
                _COLON.accept(v);
                _StatementNoShortIf.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 287:  ExpressionStatement ::= StatementExpression ;
     *</b>
     */
    static public class ExpressionStatement extends Ast implements IExpressionStatement
    {
        private IStatementExpression _StatementExpression;
        private AstToken _SEMICOLON;

        public IStatementExpression getStatementExpression() { return _StatementExpression; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ExpressionStatement(IToken leftIToken, IToken rightIToken,
                                   IStatementExpression _StatementExpression,
                                   AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._StatementExpression = _StatementExpression;
            ((Ast) _StatementExpression).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_StatementExpression);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExpressionStatement)) return false;
            if (! super.equals(o)) return false;
            ExpressionStatement other = (ExpressionStatement) o;
            if (! _StatementExpression.equals(other._StatementExpression)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_StatementExpression.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _StatementExpression.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 297:  SwitchStatement ::= switch ( Expression ) SwitchBlock
     *</b>
     */
    static public class SwitchStatement extends Ast implements ISwitchStatement
    {
        private AstToken _switch;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private SwitchBlock _SwitchBlock;

        public AstToken getswitch() { return _switch; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public SwitchBlock getSwitchBlock() { return _SwitchBlock; }

        public SwitchStatement(IToken leftIToken, IToken rightIToken,
                               AstToken _switch,
                               AstToken _LPAREN,
                               IExpression _Expression,
                               AstToken _RPAREN,
                               SwitchBlock _SwitchBlock)
        {
            super(leftIToken, rightIToken);

            this._switch = _switch;
            ((Ast) _switch).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._SwitchBlock = _SwitchBlock;
            ((Ast) _SwitchBlock).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_switch);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_SwitchBlock);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchStatement)) return false;
            if (! super.equals(o)) return false;
            SwitchStatement other = (SwitchStatement) o;
            if (! _switch.equals(other._switch)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _SwitchBlock.equals(other._SwitchBlock)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_switch.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_SwitchBlock.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _switch.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _SwitchBlock.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 298:  SwitchBlock ::= { SwitchBlockStatementGroupsopt SwitchLabelsopt }
     *</b>
     */
    static public class SwitchBlock extends Ast implements ISwitchBlock
    {
        private AstToken _LBRACE;
        private ISwitchBlockStatementGroupsopt _SwitchBlockStatementGroupsopt;
        private ISwitchLabelsopt _SwitchLabelsopt;
        private AstToken _RBRACE;

        public AstToken getLBRACE() { return _LBRACE; }
        /**
         * The value returned by <b>getSwitchBlockStatementGroupsopt</b> may be <b>null</b>
         */
        public ISwitchBlockStatementGroupsopt getSwitchBlockStatementGroupsopt() { return _SwitchBlockStatementGroupsopt; }
        /**
         * The value returned by <b>getSwitchLabelsopt</b> may be <b>null</b>
         */
        public ISwitchLabelsopt getSwitchLabelsopt() { return _SwitchLabelsopt; }
        public AstToken getRBRACE() { return _RBRACE; }

        public SwitchBlock(IToken leftIToken, IToken rightIToken,
                           AstToken _LBRACE,
                           ISwitchBlockStatementGroupsopt _SwitchBlockStatementGroupsopt,
                           ISwitchLabelsopt _SwitchLabelsopt,
                           AstToken _RBRACE)
        {
            super(leftIToken, rightIToken);

            this._LBRACE = _LBRACE;
            ((Ast) _LBRACE).setParent(this);
            this._SwitchBlockStatementGroupsopt = _SwitchBlockStatementGroupsopt;
            if (_SwitchBlockStatementGroupsopt != null) ((Ast) _SwitchBlockStatementGroupsopt).setParent(this);
            this._SwitchLabelsopt = _SwitchLabelsopt;
            if (_SwitchLabelsopt != null) ((Ast) _SwitchLabelsopt).setParent(this);
            this._RBRACE = _RBRACE;
            ((Ast) _RBRACE).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACE);
            list.add(_SwitchBlockStatementGroupsopt);
            list.add(_SwitchLabelsopt);
            list.add(_RBRACE);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchBlock)) return false;
            if (! super.equals(o)) return false;
            SwitchBlock other = (SwitchBlock) o;
            if (! _LBRACE.equals(other._LBRACE)) return false;
            if (_SwitchBlockStatementGroupsopt == null)
                if (other._SwitchBlockStatementGroupsopt != null) return false;
                else; // continue
            else if (! _SwitchBlockStatementGroupsopt.equals(other._SwitchBlockStatementGroupsopt)) return false;
            if (_SwitchLabelsopt == null)
                if (other._SwitchLabelsopt != null) return false;
                else; // continue
            else if (! _SwitchLabelsopt.equals(other._SwitchLabelsopt)) return false;
            if (! _RBRACE.equals(other._RBRACE)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACE.hashCode());
            hash = hash * 31 + (_SwitchBlockStatementGroupsopt == null ? 0 : _SwitchBlockStatementGroupsopt.hashCode());
            hash = hash * 31 + (_SwitchLabelsopt == null ? 0 : _SwitchLabelsopt.hashCode());
            hash = hash * 31 + (_RBRACE.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACE.accept(v);
                if (_SwitchBlockStatementGroupsopt != null) _SwitchBlockStatementGroupsopt.accept(v);
                if (_SwitchLabelsopt != null) _SwitchLabelsopt.accept(v);
                _RBRACE.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 299:  SwitchBlockStatementGroups ::= SwitchBlockStatementGroup
     *</em>
     *<p>
     *<b>
     *<li>Rule 300:  SwitchBlockStatementGroups ::= SwitchBlockStatementGroups SwitchBlockStatementGroup
     *</b>
     */
    static public class SwitchBlockStatementGroups extends Ast implements ISwitchBlockStatementGroups
    {
        private ISwitchBlockStatementGroups _SwitchBlockStatementGroups;
        private SwitchBlockStatementGroup _SwitchBlockStatementGroup;

        public ISwitchBlockStatementGroups getSwitchBlockStatementGroups() { return _SwitchBlockStatementGroups; }
        public SwitchBlockStatementGroup getSwitchBlockStatementGroup() { return _SwitchBlockStatementGroup; }

        public SwitchBlockStatementGroups(IToken leftIToken, IToken rightIToken,
                                          ISwitchBlockStatementGroups _SwitchBlockStatementGroups,
                                          SwitchBlockStatementGroup _SwitchBlockStatementGroup)
        {
            super(leftIToken, rightIToken);

            this._SwitchBlockStatementGroups = _SwitchBlockStatementGroups;
            ((Ast) _SwitchBlockStatementGroups).setParent(this);
            this._SwitchBlockStatementGroup = _SwitchBlockStatementGroup;
            ((Ast) _SwitchBlockStatementGroup).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_SwitchBlockStatementGroups);
            list.add(_SwitchBlockStatementGroup);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchBlockStatementGroups)) return false;
            if (! super.equals(o)) return false;
            SwitchBlockStatementGroups other = (SwitchBlockStatementGroups) o;
            if (! _SwitchBlockStatementGroups.equals(other._SwitchBlockStatementGroups)) return false;
            if (! _SwitchBlockStatementGroup.equals(other._SwitchBlockStatementGroup)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_SwitchBlockStatementGroups.hashCode());
            hash = hash * 31 + (_SwitchBlockStatementGroup.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _SwitchBlockStatementGroups.accept(v);
                _SwitchBlockStatementGroup.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 301:  SwitchBlockStatementGroup ::= SwitchLabels BlockStatements
     *</b>
     */
    static public class SwitchBlockStatementGroup extends Ast implements ISwitchBlockStatementGroup
    {
        private ISwitchLabels _SwitchLabels;
        private IBlockStatements _BlockStatements;

        public ISwitchLabels getSwitchLabels() { return _SwitchLabels; }
        public IBlockStatements getBlockStatements() { return _BlockStatements; }

        public SwitchBlockStatementGroup(IToken leftIToken, IToken rightIToken,
                                         ISwitchLabels _SwitchLabels,
                                         IBlockStatements _BlockStatements)
        {
            super(leftIToken, rightIToken);

            this._SwitchLabels = _SwitchLabels;
            ((Ast) _SwitchLabels).setParent(this);
            this._BlockStatements = _BlockStatements;
            ((Ast) _BlockStatements).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_SwitchLabels);
            list.add(_BlockStatements);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchBlockStatementGroup)) return false;
            if (! super.equals(o)) return false;
            SwitchBlockStatementGroup other = (SwitchBlockStatementGroup) o;
            if (! _SwitchLabels.equals(other._SwitchLabels)) return false;
            if (! _BlockStatements.equals(other._BlockStatements)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_SwitchLabels.hashCode());
            hash = hash * 31 + (_BlockStatements.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _SwitchLabels.accept(v);
                _BlockStatements.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 302:  SwitchLabels ::= SwitchLabel
     *</em>
     *<p>
     *<b>
     *<li>Rule 303:  SwitchLabels ::= SwitchLabels SwitchLabel
     *</b>
     */
    static public class SwitchLabels extends Ast implements ISwitchLabels
    {
        private ISwitchLabels _SwitchLabels;
        private ISwitchLabel _SwitchLabel;

        public ISwitchLabels getSwitchLabels() { return _SwitchLabels; }
        public ISwitchLabel getSwitchLabel() { return _SwitchLabel; }

        public SwitchLabels(IToken leftIToken, IToken rightIToken,
                            ISwitchLabels _SwitchLabels,
                            ISwitchLabel _SwitchLabel)
        {
            super(leftIToken, rightIToken);

            this._SwitchLabels = _SwitchLabels;
            ((Ast) _SwitchLabels).setParent(this);
            this._SwitchLabel = _SwitchLabel;
            ((Ast) _SwitchLabel).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_SwitchLabels);
            list.add(_SwitchLabel);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchLabels)) return false;
            if (! super.equals(o)) return false;
            SwitchLabels other = (SwitchLabels) o;
            if (! _SwitchLabels.equals(other._SwitchLabels)) return false;
            if (! _SwitchLabel.equals(other._SwitchLabel)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_SwitchLabels.hashCode());
            hash = hash * 31 + (_SwitchLabel.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _SwitchLabels.accept(v);
                _SwitchLabel.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 308:  WhileStatement ::= while ( Expression ) Statement
     *</b>
     */
    static public class WhileStatement extends Ast implements IWhileStatement
    {
        private AstToken _while;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatement _Statement;

        public AstToken getwhile() { return _while; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatement getStatement() { return _Statement; }

        public WhileStatement(IToken leftIToken, IToken rightIToken,
                              AstToken _while,
                              AstToken _LPAREN,
                              IExpression _Expression,
                              AstToken _RPAREN,
                              IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._while = _while;
            ((Ast) _while).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_while);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof WhileStatement)) return false;
            if (! super.equals(o)) return false;
            WhileStatement other = (WhileStatement) o;
            if (! _while.equals(other._while)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_while.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _while.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 309:  WhileStatementNoShortIf ::= while ( Expression ) StatementNoShortIf
     *</b>
     */
    static public class WhileStatementNoShortIf extends Ast implements IWhileStatementNoShortIf
    {
        private AstToken _while;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatementNoShortIf _StatementNoShortIf;

        public AstToken getwhile() { return _while; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatementNoShortIf getStatementNoShortIf() { return _StatementNoShortIf; }

        public WhileStatementNoShortIf(IToken leftIToken, IToken rightIToken,
                                       AstToken _while,
                                       AstToken _LPAREN,
                                       IExpression _Expression,
                                       AstToken _RPAREN,
                                       IStatementNoShortIf _StatementNoShortIf)
        {
            super(leftIToken, rightIToken);

            this._while = _while;
            ((Ast) _while).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._StatementNoShortIf = _StatementNoShortIf;
            ((Ast) _StatementNoShortIf).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_while);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_StatementNoShortIf);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof WhileStatementNoShortIf)) return false;
            if (! super.equals(o)) return false;
            WhileStatementNoShortIf other = (WhileStatementNoShortIf) o;
            if (! _while.equals(other._while)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _StatementNoShortIf.equals(other._StatementNoShortIf)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_while.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_StatementNoShortIf.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _while.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _StatementNoShortIf.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 310:  DoStatement ::= do Statement while ( Expression ) ;
     *</b>
     */
    static public class DoStatement extends Ast implements IDoStatement
    {
        private AstToken _do;
        private IStatement _Statement;
        private AstToken _while;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private AstToken _SEMICOLON;

        public AstToken getdo() { return _do; }
        public IStatement getStatement() { return _Statement; }
        public AstToken getwhile() { return _while; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public DoStatement(IToken leftIToken, IToken rightIToken,
                           AstToken _do,
                           IStatement _Statement,
                           AstToken _while,
                           AstToken _LPAREN,
                           IExpression _Expression,
                           AstToken _RPAREN,
                           AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._do = _do;
            ((Ast) _do).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            this._while = _while;
            ((Ast) _while).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_do);
            list.add(_Statement);
            list.add(_while);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof DoStatement)) return false;
            if (! super.equals(o)) return false;
            DoStatement other = (DoStatement) o;
            if (! _do.equals(other._do)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            if (! _while.equals(other._while)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_do.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            hash = hash * 31 + (_while.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _do.accept(v);
                _Statement.accept(v);
                _while.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 313:  BasicForStatement ::= for ( ForInitopt ; Expressionopt ; ForUpdateopt ) Statement
     *</b>
     */
    static public class BasicForStatement extends Ast implements IBasicForStatement
    {
        private AstToken _for;
        private AstToken _LPAREN;
        private IForInitopt _ForInitopt;
        private AstToken _SEMICOLON;
        private IExpressionopt _Expressionopt;
        private AstToken _SEMICOLON6;
        private IForUpdateopt _ForUpdateopt;
        private AstToken _RPAREN;
        private IStatement _Statement;

        public AstToken getfor() { return _for; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getForInitopt</b> may be <b>null</b>
         */
        public IForInitopt getForInitopt() { return _ForInitopt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }
        /**
         * The value returned by <b>getExpressionopt</b> may be <b>null</b>
         */
        public IExpressionopt getExpressionopt() { return _Expressionopt; }
        public AstToken getSEMICOLON6() { return _SEMICOLON6; }
        /**
         * The value returned by <b>getForUpdateopt</b> may be <b>null</b>
         */
        public IForUpdateopt getForUpdateopt() { return _ForUpdateopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatement getStatement() { return _Statement; }

        public BasicForStatement(IToken leftIToken, IToken rightIToken,
                                 AstToken _for,
                                 AstToken _LPAREN,
                                 IForInitopt _ForInitopt,
                                 AstToken _SEMICOLON,
                                 IExpressionopt _Expressionopt,
                                 AstToken _SEMICOLON6,
                                 IForUpdateopt _ForUpdateopt,
                                 AstToken _RPAREN,
                                 IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._for = _for;
            ((Ast) _for).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ForInitopt = _ForInitopt;
            if (_ForInitopt != null) ((Ast) _ForInitopt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            this._Expressionopt = _Expressionopt;
            if (_Expressionopt != null) ((Ast) _Expressionopt).setParent(this);
            this._SEMICOLON6 = _SEMICOLON6;
            ((Ast) _SEMICOLON6).setParent(this);
            this._ForUpdateopt = _ForUpdateopt;
            if (_ForUpdateopt != null) ((Ast) _ForUpdateopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_for);
            list.add(_LPAREN);
            list.add(_ForInitopt);
            list.add(_SEMICOLON);
            list.add(_Expressionopt);
            list.add(_SEMICOLON6);
            list.add(_ForUpdateopt);
            list.add(_RPAREN);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof BasicForStatement)) return false;
            if (! super.equals(o)) return false;
            BasicForStatement other = (BasicForStatement) o;
            if (! _for.equals(other._for)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ForInitopt == null)
                if (other._ForInitopt != null) return false;
                else; // continue
            else if (! _ForInitopt.equals(other._ForInitopt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            if (_Expressionopt == null)
                if (other._Expressionopt != null) return false;
                else; // continue
            else if (! _Expressionopt.equals(other._Expressionopt)) return false;
            if (! _SEMICOLON6.equals(other._SEMICOLON6)) return false;
            if (_ForUpdateopt == null)
                if (other._ForUpdateopt != null) return false;
                else; // continue
            else if (! _ForUpdateopt.equals(other._ForUpdateopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_for.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ForInitopt == null ? 0 : _ForInitopt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            hash = hash * 31 + (_Expressionopt == null ? 0 : _Expressionopt.hashCode());
            hash = hash * 31 + (_SEMICOLON6.hashCode());
            hash = hash * 31 + (_ForUpdateopt == null ? 0 : _ForUpdateopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _for.accept(v);
                _LPAREN.accept(v);
                if (_ForInitopt != null) _ForInitopt.accept(v);
                _SEMICOLON.accept(v);
                if (_Expressionopt != null) _Expressionopt.accept(v);
                _SEMICOLON6.accept(v);
                if (_ForUpdateopt != null) _ForUpdateopt.accept(v);
                _RPAREN.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 314:  ForStatementNoShortIf ::= for ( ForInitopt ; Expressionopt ; ForUpdateopt ) StatementNoShortIf
     *</b>
     */
    static public class ForStatementNoShortIf extends Ast implements IForStatementNoShortIf
    {
        private AstToken _for;
        private AstToken _LPAREN;
        private IForInitopt _ForInitopt;
        private AstToken _SEMICOLON;
        private IExpressionopt _Expressionopt;
        private AstToken _SEMICOLON6;
        private IForUpdateopt _ForUpdateopt;
        private AstToken _RPAREN;
        private IStatementNoShortIf _StatementNoShortIf;

        public AstToken getfor() { return _for; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getForInitopt</b> may be <b>null</b>
         */
        public IForInitopt getForInitopt() { return _ForInitopt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }
        /**
         * The value returned by <b>getExpressionopt</b> may be <b>null</b>
         */
        public IExpressionopt getExpressionopt() { return _Expressionopt; }
        public AstToken getSEMICOLON6() { return _SEMICOLON6; }
        /**
         * The value returned by <b>getForUpdateopt</b> may be <b>null</b>
         */
        public IForUpdateopt getForUpdateopt() { return _ForUpdateopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatementNoShortIf getStatementNoShortIf() { return _StatementNoShortIf; }

        public ForStatementNoShortIf(IToken leftIToken, IToken rightIToken,
                                     AstToken _for,
                                     AstToken _LPAREN,
                                     IForInitopt _ForInitopt,
                                     AstToken _SEMICOLON,
                                     IExpressionopt _Expressionopt,
                                     AstToken _SEMICOLON6,
                                     IForUpdateopt _ForUpdateopt,
                                     AstToken _RPAREN,
                                     IStatementNoShortIf _StatementNoShortIf)
        {
            super(leftIToken, rightIToken);

            this._for = _for;
            ((Ast) _for).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ForInitopt = _ForInitopt;
            if (_ForInitopt != null) ((Ast) _ForInitopt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            this._Expressionopt = _Expressionopt;
            if (_Expressionopt != null) ((Ast) _Expressionopt).setParent(this);
            this._SEMICOLON6 = _SEMICOLON6;
            ((Ast) _SEMICOLON6).setParent(this);
            this._ForUpdateopt = _ForUpdateopt;
            if (_ForUpdateopt != null) ((Ast) _ForUpdateopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._StatementNoShortIf = _StatementNoShortIf;
            ((Ast) _StatementNoShortIf).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_for);
            list.add(_LPAREN);
            list.add(_ForInitopt);
            list.add(_SEMICOLON);
            list.add(_Expressionopt);
            list.add(_SEMICOLON6);
            list.add(_ForUpdateopt);
            list.add(_RPAREN);
            list.add(_StatementNoShortIf);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ForStatementNoShortIf)) return false;
            if (! super.equals(o)) return false;
            ForStatementNoShortIf other = (ForStatementNoShortIf) o;
            if (! _for.equals(other._for)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ForInitopt == null)
                if (other._ForInitopt != null) return false;
                else; // continue
            else if (! _ForInitopt.equals(other._ForInitopt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            if (_Expressionopt == null)
                if (other._Expressionopt != null) return false;
                else; // continue
            else if (! _Expressionopt.equals(other._Expressionopt)) return false;
            if (! _SEMICOLON6.equals(other._SEMICOLON6)) return false;
            if (_ForUpdateopt == null)
                if (other._ForUpdateopt != null) return false;
                else; // continue
            else if (! _ForUpdateopt.equals(other._ForUpdateopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _StatementNoShortIf.equals(other._StatementNoShortIf)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_for.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ForInitopt == null ? 0 : _ForInitopt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            hash = hash * 31 + (_Expressionopt == null ? 0 : _Expressionopt.hashCode());
            hash = hash * 31 + (_SEMICOLON6.hashCode());
            hash = hash * 31 + (_ForUpdateopt == null ? 0 : _ForUpdateopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_StatementNoShortIf.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _for.accept(v);
                _LPAREN.accept(v);
                if (_ForInitopt != null) _ForInitopt.accept(v);
                _SEMICOLON.accept(v);
                if (_Expressionopt != null) _Expressionopt.accept(v);
                _SEMICOLON6.accept(v);
                if (_ForUpdateopt != null) _ForUpdateopt.accept(v);
                _RPAREN.accept(v);
                _StatementNoShortIf.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 318:  StatementExpressionList ::= StatementExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 319:  StatementExpressionList ::= StatementExpressionList , StatementExpression
     *</b>
     */
    static public class StatementExpressionList extends Ast implements IStatementExpressionList
    {
        private IStatementExpressionList _StatementExpressionList;
        private AstToken _COMMA;
        private IStatementExpression _StatementExpression;

        public IStatementExpressionList getStatementExpressionList() { return _StatementExpressionList; }
        public AstToken getCOMMA() { return _COMMA; }
        public IStatementExpression getStatementExpression() { return _StatementExpression; }

        public StatementExpressionList(IToken leftIToken, IToken rightIToken,
                                       IStatementExpressionList _StatementExpressionList,
                                       AstToken _COMMA,
                                       IStatementExpression _StatementExpression)
        {
            super(leftIToken, rightIToken);

            this._StatementExpressionList = _StatementExpressionList;
            ((Ast) _StatementExpressionList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._StatementExpression = _StatementExpression;
            ((Ast) _StatementExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_StatementExpressionList);
            list.add(_COMMA);
            list.add(_StatementExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof StatementExpressionList)) return false;
            if (! super.equals(o)) return false;
            StatementExpressionList other = (StatementExpressionList) o;
            if (! _StatementExpressionList.equals(other._StatementExpressionList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _StatementExpression.equals(other._StatementExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_StatementExpressionList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_StatementExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _StatementExpressionList.accept(v);
                _COMMA.accept(v);
                _StatementExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 320:  EnhancedForStatement ::= for ( FormalParameter : Expression ) Statement
     *</b>
     */
    static public class EnhancedForStatement extends Ast implements IEnhancedForStatement
    {
        private AstToken _for;
        private AstToken _LPAREN;
        private FormalParameter _FormalParameter;
        private AstToken _COLON;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private IStatement _Statement;

        public AstToken getfor() { return _for; }
        public AstToken getLPAREN() { return _LPAREN; }
        public FormalParameter getFormalParameter() { return _FormalParameter; }
        public AstToken getCOLON() { return _COLON; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IStatement getStatement() { return _Statement; }

        public EnhancedForStatement(IToken leftIToken, IToken rightIToken,
                                    AstToken _for,
                                    AstToken _LPAREN,
                                    FormalParameter _FormalParameter,
                                    AstToken _COLON,
                                    IExpression _Expression,
                                    AstToken _RPAREN,
                                    IStatement _Statement)
        {
            super(leftIToken, rightIToken);

            this._for = _for;
            ((Ast) _for).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._FormalParameter = _FormalParameter;
            ((Ast) _FormalParameter).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Statement = _Statement;
            ((Ast) _Statement).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_for);
            list.add(_LPAREN);
            list.add(_FormalParameter);
            list.add(_COLON);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_Statement);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EnhancedForStatement)) return false;
            if (! super.equals(o)) return false;
            EnhancedForStatement other = (EnhancedForStatement) o;
            if (! _for.equals(other._for)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _FormalParameter.equals(other._FormalParameter)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Statement.equals(other._Statement)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_for.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_FormalParameter.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Statement.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _for.accept(v);
                _LPAREN.accept(v);
                _FormalParameter.accept(v);
                _COLON.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _Statement.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 321:  BreakStatement ::= break identifieropt ;
     *</b>
     */
    static public class BreakStatement extends Ast implements IBreakStatement
    {
        private AstToken _break;
        private identifier _identifieropt;
        private AstToken _SEMICOLON;

        public AstToken getbreak() { return _break; }
        /**
         * The value returned by <b>getidentifieropt</b> may be <b>null</b>
         */
        public identifier getidentifieropt() { return _identifieropt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public BreakStatement(IToken leftIToken, IToken rightIToken,
                              AstToken _break,
                              identifier _identifieropt,
                              AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._break = _break;
            ((Ast) _break).setParent(this);
            this._identifieropt = _identifieropt;
            if (_identifieropt != null) ((Ast) _identifieropt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_break);
            list.add(_identifieropt);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof BreakStatement)) return false;
            if (! super.equals(o)) return false;
            BreakStatement other = (BreakStatement) o;
            if (! _break.equals(other._break)) return false;
            if (_identifieropt == null)
                if (other._identifieropt != null) return false;
                else; // continue
            else if (! _identifieropt.equals(other._identifieropt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_break.hashCode());
            hash = hash * 31 + (_identifieropt == null ? 0 : _identifieropt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _break.accept(v);
                if (_identifieropt != null) _identifieropt.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 322:  ContinueStatement ::= continue identifieropt ;
     *</b>
     */
    static public class ContinueStatement extends Ast implements IContinueStatement
    {
        private AstToken _continue;
        private identifier _identifieropt;
        private AstToken _SEMICOLON;

        public AstToken getcontinue() { return _continue; }
        /**
         * The value returned by <b>getidentifieropt</b> may be <b>null</b>
         */
        public identifier getidentifieropt() { return _identifieropt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ContinueStatement(IToken leftIToken, IToken rightIToken,
                                 AstToken _continue,
                                 identifier _identifieropt,
                                 AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._continue = _continue;
            ((Ast) _continue).setParent(this);
            this._identifieropt = _identifieropt;
            if (_identifieropt != null) ((Ast) _identifieropt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_continue);
            list.add(_identifieropt);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ContinueStatement)) return false;
            if (! super.equals(o)) return false;
            ContinueStatement other = (ContinueStatement) o;
            if (! _continue.equals(other._continue)) return false;
            if (_identifieropt == null)
                if (other._identifieropt != null) return false;
                else; // continue
            else if (! _identifieropt.equals(other._identifieropt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_continue.hashCode());
            hash = hash * 31 + (_identifieropt == null ? 0 : _identifieropt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _continue.accept(v);
                if (_identifieropt != null) _identifieropt.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 323:  ReturnStatement ::= return Expressionopt ;
     *</b>
     */
    static public class ReturnStatement extends Ast implements IReturnStatement
    {
        private AstToken _return;
        private IExpressionopt _Expressionopt;
        private AstToken _SEMICOLON;

        public AstToken getreturn() { return _return; }
        /**
         * The value returned by <b>getExpressionopt</b> may be <b>null</b>
         */
        public IExpressionopt getExpressionopt() { return _Expressionopt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ReturnStatement(IToken leftIToken, IToken rightIToken,
                               AstToken _return,
                               IExpressionopt _Expressionopt,
                               AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._return = _return;
            ((Ast) _return).setParent(this);
            this._Expressionopt = _Expressionopt;
            if (_Expressionopt != null) ((Ast) _Expressionopt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_return);
            list.add(_Expressionopt);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ReturnStatement)) return false;
            if (! super.equals(o)) return false;
            ReturnStatement other = (ReturnStatement) o;
            if (! _return.equals(other._return)) return false;
            if (_Expressionopt == null)
                if (other._Expressionopt != null) return false;
                else; // continue
            else if (! _Expressionopt.equals(other._Expressionopt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_return.hashCode());
            hash = hash * 31 + (_Expressionopt == null ? 0 : _Expressionopt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _return.accept(v);
                if (_Expressionopt != null) _Expressionopt.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 324:  ThrowStatement ::= throw Expression ;
     *</b>
     */
    static public class ThrowStatement extends Ast implements IThrowStatement
    {
        private AstToken _throw;
        private IExpression _Expression;
        private AstToken _SEMICOLON;

        public AstToken getthrow() { return _throw; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ThrowStatement(IToken leftIToken, IToken rightIToken,
                              AstToken _throw,
                              IExpression _Expression,
                              AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._throw = _throw;
            ((Ast) _throw).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_throw);
            list.add(_Expression);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ThrowStatement)) return false;
            if (! super.equals(o)) return false;
            ThrowStatement other = (ThrowStatement) o;
            if (! _throw.equals(other._throw)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_throw.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _throw.accept(v);
                _Expression.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 325:  SynchronizedStatement ::= synchronized ( Expression ) Block
     *</b>
     */
    static public class SynchronizedStatement extends Ast implements ISynchronizedStatement
    {
        private AstToken _synchronized;
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;
        private Block _Block;

        public AstToken getsynchronized() { return _synchronized; }
        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }
        public Block getBlock() { return _Block; }

        public SynchronizedStatement(IToken leftIToken, IToken rightIToken,
                                     AstToken _synchronized,
                                     AstToken _LPAREN,
                                     IExpression _Expression,
                                     AstToken _RPAREN,
                                     Block _Block)
        {
            super(leftIToken, rightIToken);

            this._synchronized = _synchronized;
            ((Ast) _synchronized).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_synchronized);
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            list.add(_Block);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SynchronizedStatement)) return false;
            if (! super.equals(o)) return false;
            SynchronizedStatement other = (SynchronizedStatement) o;
            if (! _synchronized.equals(other._synchronized)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Block.equals(other._Block)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_synchronized.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _synchronized.accept(v);
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
                _Block.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 328:  Catches ::= CatchClause
     *</em>
     *<p>
     *<b>
     *<li>Rule 329:  Catches ::= Catches CatchClause
     *</b>
     */
    static public class Catches extends Ast implements ICatches
    {
        private ICatches _Catches;
        private CatchClause _CatchClause;

        public ICatches getCatches() { return _Catches; }
        public CatchClause getCatchClause() { return _CatchClause; }

        public Catches(IToken leftIToken, IToken rightIToken,
                       ICatches _Catches,
                       CatchClause _CatchClause)
        {
            super(leftIToken, rightIToken);

            this._Catches = _Catches;
            ((Ast) _Catches).setParent(this);
            this._CatchClause = _CatchClause;
            ((Ast) _CatchClause).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Catches);
            list.add(_CatchClause);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Catches)) return false;
            if (! super.equals(o)) return false;
            Catches other = (Catches) o;
            if (! _Catches.equals(other._Catches)) return false;
            if (! _CatchClause.equals(other._CatchClause)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Catches.hashCode());
            hash = hash * 31 + (_CatchClause.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Catches.accept(v);
                _CatchClause.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 330:  CatchClause ::= catch ( FormalParameter ) Block
     *</b>
     */
    static public class CatchClause extends Ast implements ICatchClause
    {
        private AstToken _catch;
        private AstToken _LPAREN;
        private FormalParameter _FormalParameter;
        private AstToken _RPAREN;
        private Block _Block;

        public AstToken getcatch() { return _catch; }
        public AstToken getLPAREN() { return _LPAREN; }
        public FormalParameter getFormalParameter() { return _FormalParameter; }
        public AstToken getRPAREN() { return _RPAREN; }
        public Block getBlock() { return _Block; }

        public CatchClause(IToken leftIToken, IToken rightIToken,
                           AstToken _catch,
                           AstToken _LPAREN,
                           FormalParameter _FormalParameter,
                           AstToken _RPAREN,
                           Block _Block)
        {
            super(leftIToken, rightIToken);

            this._catch = _catch;
            ((Ast) _catch).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._FormalParameter = _FormalParameter;
            ((Ast) _FormalParameter).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_catch);
            list.add(_LPAREN);
            list.add(_FormalParameter);
            list.add(_RPAREN);
            list.add(_Block);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof CatchClause)) return false;
            if (! super.equals(o)) return false;
            CatchClause other = (CatchClause) o;
            if (! _catch.equals(other._catch)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _FormalParameter.equals(other._FormalParameter)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _Block.equals(other._Block)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_catch.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_FormalParameter.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _catch.accept(v);
                _LPAREN.accept(v);
                _FormalParameter.accept(v);
                _RPAREN.accept(v);
                _Block.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 331:  Finally ::= finally Block
     *</b>
     */
    static public class Finally extends Ast implements IFinally
    {
        private AstToken _finally;
        private Block _Block;

        public AstToken getfinally() { return _finally; }
        public Block getBlock() { return _Block; }

        public Finally(IToken leftIToken, IToken rightIToken,
                       AstToken _finally,
                       Block _Block)
        {
            super(leftIToken, rightIToken);

            this._finally = _finally;
            ((Ast) _finally).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_finally);
            list.add(_Block);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Finally)) return false;
            if (! super.equals(o)) return false;
            Finally other = (Finally) o;
            if (! _finally.equals(other._finally)) return false;
            if (! _Block.equals(other._Block)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_finally.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _finally.accept(v);
                _Block.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 356:  ArgumentList ::= Expression
     *</em>
     *<p>
     *<b>
     *<li>Rule 357:  ArgumentList ::= ArgumentList , Expression
     *</b>
     */
    static public class ArgumentList extends Ast implements IArgumentList
    {
        private IArgumentList _ArgumentList;
        private AstToken _COMMA;
        private IExpression _Expression;

        public IArgumentList getArgumentList() { return _ArgumentList; }
        public AstToken getCOMMA() { return _COMMA; }
        public IExpression getExpression() { return _Expression; }

        public ArgumentList(IToken leftIToken, IToken rightIToken,
                            IArgumentList _ArgumentList,
                            AstToken _COMMA,
                            IExpression _Expression)
        {
            super(leftIToken, rightIToken);

            this._ArgumentList = _ArgumentList;
            ((Ast) _ArgumentList).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ArgumentList);
            list.add(_COMMA);
            list.add(_Expression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArgumentList)) return false;
            if (! super.equals(o)) return false;
            ArgumentList other = (ArgumentList) o;
            if (! _ArgumentList.equals(other._ArgumentList)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ArgumentList.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ArgumentList.accept(v);
                _COMMA.accept(v);
                _Expression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 362:  DimExprs ::= DimExpr
     *</em>
     *<p>
     *<b>
     *<li>Rule 363:  DimExprs ::= DimExprs DimExpr
     *</b>
     */
    static public class DimExprs extends Ast implements IDimExprs
    {
        private IDimExprs _DimExprs;
        private DimExpr _DimExpr;

        public IDimExprs getDimExprs() { return _DimExprs; }
        public DimExpr getDimExpr() { return _DimExpr; }

        public DimExprs(IToken leftIToken, IToken rightIToken,
                        IDimExprs _DimExprs,
                        DimExpr _DimExpr)
        {
            super(leftIToken, rightIToken);

            this._DimExprs = _DimExprs;
            ((Ast) _DimExprs).setParent(this);
            this._DimExpr = _DimExpr;
            ((Ast) _DimExpr).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_DimExprs);
            list.add(_DimExpr);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof DimExprs)) return false;
            if (! super.equals(o)) return false;
            DimExprs other = (DimExprs) o;
            if (! _DimExprs.equals(other._DimExprs)) return false;
            if (! _DimExpr.equals(other._DimExpr)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_DimExprs.hashCode());
            hash = hash * 31 + (_DimExpr.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _DimExprs.accept(v);
                _DimExpr.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 364:  DimExpr ::= [ Expression ]
     *</b>
     */
    static public class DimExpr extends Ast implements IDimExpr
    {
        private AstToken _LBRACKET;
        private IExpression _Expression;
        private AstToken _RBRACKET;

        public AstToken getLBRACKET() { return _LBRACKET; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public DimExpr(IToken leftIToken, IToken rightIToken,
                       AstToken _LBRACKET,
                       IExpression _Expression,
                       AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACKET);
            list.add(_Expression);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof DimExpr)) return false;
            if (! super.equals(o)) return false;
            DimExpr other = (DimExpr) o;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACKET.accept(v);
                _Expression.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 381:  PostIncrementExpression ::= PostfixExpression ++
     *</b>
     */
    static public class PostIncrementExpression extends Ast implements IPostIncrementExpression
    {
        private IPostfixExpression _PostfixExpression;
        private AstToken _PLUS_PLUS;

        public IPostfixExpression getPostfixExpression() { return _PostfixExpression; }
        public AstToken getPLUS_PLUS() { return _PLUS_PLUS; }

        public PostIncrementExpression(IToken leftIToken, IToken rightIToken,
                                       IPostfixExpression _PostfixExpression,
                                       AstToken _PLUS_PLUS)
        {
            super(leftIToken, rightIToken);

            this._PostfixExpression = _PostfixExpression;
            ((Ast) _PostfixExpression).setParent(this);
            this._PLUS_PLUS = _PLUS_PLUS;
            ((Ast) _PLUS_PLUS).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PostfixExpression);
            list.add(_PLUS_PLUS);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PostIncrementExpression)) return false;
            if (! super.equals(o)) return false;
            PostIncrementExpression other = (PostIncrementExpression) o;
            if (! _PostfixExpression.equals(other._PostfixExpression)) return false;
            if (! _PLUS_PLUS.equals(other._PLUS_PLUS)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PostfixExpression.hashCode());
            hash = hash * 31 + (_PLUS_PLUS.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PostfixExpression.accept(v);
                _PLUS_PLUS.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 382:  PostDecrementExpression ::= PostfixExpression --
     *</b>
     */
    static public class PostDecrementExpression extends Ast implements IPostDecrementExpression
    {
        private IPostfixExpression _PostfixExpression;
        private AstToken _MINUS_MINUS;

        public IPostfixExpression getPostfixExpression() { return _PostfixExpression; }
        public AstToken getMINUS_MINUS() { return _MINUS_MINUS; }

        public PostDecrementExpression(IToken leftIToken, IToken rightIToken,
                                       IPostfixExpression _PostfixExpression,
                                       AstToken _MINUS_MINUS)
        {
            super(leftIToken, rightIToken);

            this._PostfixExpression = _PostfixExpression;
            ((Ast) _PostfixExpression).setParent(this);
            this._MINUS_MINUS = _MINUS_MINUS;
            ((Ast) _MINUS_MINUS).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PostfixExpression);
            list.add(_MINUS_MINUS);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PostDecrementExpression)) return false;
            if (! super.equals(o)) return false;
            PostDecrementExpression other = (PostDecrementExpression) o;
            if (! _PostfixExpression.equals(other._PostfixExpression)) return false;
            if (! _MINUS_MINUS.equals(other._MINUS_MINUS)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PostfixExpression.hashCode());
            hash = hash * 31 + (_MINUS_MINUS.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PostfixExpression.accept(v);
                _MINUS_MINUS.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 388:  PreIncrementExpression ::= ++ UnaryExpression
     *</b>
     */
    static public class PreIncrementExpression extends Ast implements IPreIncrementExpression
    {
        private AstToken _PLUS_PLUS;
        private IUnaryExpression _UnaryExpression;

        public AstToken getPLUS_PLUS() { return _PLUS_PLUS; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public PreIncrementExpression(IToken leftIToken, IToken rightIToken,
                                      AstToken _PLUS_PLUS,
                                      IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._PLUS_PLUS = _PLUS_PLUS;
            ((Ast) _PLUS_PLUS).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PLUS_PLUS);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PreIncrementExpression)) return false;
            if (! super.equals(o)) return false;
            PreIncrementExpression other = (PreIncrementExpression) o;
            if (! _PLUS_PLUS.equals(other._PLUS_PLUS)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PLUS_PLUS.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PLUS_PLUS.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 389:  PreDecrementExpression ::= -- UnaryExpression
     *</b>
     */
    static public class PreDecrementExpression extends Ast implements IPreDecrementExpression
    {
        private AstToken _MINUS_MINUS;
        private IUnaryExpression _UnaryExpression;

        public AstToken getMINUS_MINUS() { return _MINUS_MINUS; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public PreDecrementExpression(IToken leftIToken, IToken rightIToken,
                                      AstToken _MINUS_MINUS,
                                      IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._MINUS_MINUS = _MINUS_MINUS;
            ((Ast) _MINUS_MINUS).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MINUS_MINUS);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PreDecrementExpression)) return false;
            if (! super.equals(o)) return false;
            PreDecrementExpression other = (PreDecrementExpression) o;
            if (! _MINUS_MINUS.equals(other._MINUS_MINUS)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MINUS_MINUS.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MINUS_MINUS.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 416:  AndExpression ::= EqualityExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 417:  AndExpression ::= AndExpression & EqualityExpression
     *</b>
     */
    static public class AndExpression extends Ast implements IAndExpression
    {
        private IAndExpression _AndExpression;
        private AstToken _AND;
        private IEqualityExpression _EqualityExpression;

        public IAndExpression getAndExpression() { return _AndExpression; }
        public AstToken getAND() { return _AND; }
        public IEqualityExpression getEqualityExpression() { return _EqualityExpression; }

        public AndExpression(IToken leftIToken, IToken rightIToken,
                             IAndExpression _AndExpression,
                             AstToken _AND,
                             IEqualityExpression _EqualityExpression)
        {
            super(leftIToken, rightIToken);

            this._AndExpression = _AndExpression;
            ((Ast) _AndExpression).setParent(this);
            this._AND = _AND;
            ((Ast) _AND).setParent(this);
            this._EqualityExpression = _EqualityExpression;
            ((Ast) _EqualityExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AndExpression);
            list.add(_AND);
            list.add(_EqualityExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AndExpression)) return false;
            if (! super.equals(o)) return false;
            AndExpression other = (AndExpression) o;
            if (! _AndExpression.equals(other._AndExpression)) return false;
            if (! _AND.equals(other._AND)) return false;
            if (! _EqualityExpression.equals(other._EqualityExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AndExpression.hashCode());
            hash = hash * 31 + (_AND.hashCode());
            hash = hash * 31 + (_EqualityExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AndExpression.accept(v);
                _AND.accept(v);
                _EqualityExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 418:  ExclusiveOrExpression ::= AndExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 419:  ExclusiveOrExpression ::= ExclusiveOrExpression ^ AndExpression
     *</b>
     */
    static public class ExclusiveOrExpression extends Ast implements IExclusiveOrExpression
    {
        private IExclusiveOrExpression _ExclusiveOrExpression;
        private AstToken _XOR;
        private IAndExpression _AndExpression;

        public IExclusiveOrExpression getExclusiveOrExpression() { return _ExclusiveOrExpression; }
        public AstToken getXOR() { return _XOR; }
        public IAndExpression getAndExpression() { return _AndExpression; }

        public ExclusiveOrExpression(IToken leftIToken, IToken rightIToken,
                                     IExclusiveOrExpression _ExclusiveOrExpression,
                                     AstToken _XOR,
                                     IAndExpression _AndExpression)
        {
            super(leftIToken, rightIToken);

            this._ExclusiveOrExpression = _ExclusiveOrExpression;
            ((Ast) _ExclusiveOrExpression).setParent(this);
            this._XOR = _XOR;
            ((Ast) _XOR).setParent(this);
            this._AndExpression = _AndExpression;
            ((Ast) _AndExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ExclusiveOrExpression);
            list.add(_XOR);
            list.add(_AndExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExclusiveOrExpression)) return false;
            if (! super.equals(o)) return false;
            ExclusiveOrExpression other = (ExclusiveOrExpression) o;
            if (! _ExclusiveOrExpression.equals(other._ExclusiveOrExpression)) return false;
            if (! _XOR.equals(other._XOR)) return false;
            if (! _AndExpression.equals(other._AndExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ExclusiveOrExpression.hashCode());
            hash = hash * 31 + (_XOR.hashCode());
            hash = hash * 31 + (_AndExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ExclusiveOrExpression.accept(v);
                _XOR.accept(v);
                _AndExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 420:  InclusiveOrExpression ::= ExclusiveOrExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 421:  InclusiveOrExpression ::= InclusiveOrExpression | ExclusiveOrExpression
     *</b>
     */
    static public class InclusiveOrExpression extends Ast implements IInclusiveOrExpression
    {
        private IInclusiveOrExpression _InclusiveOrExpression;
        private AstToken _OR;
        private IExclusiveOrExpression _ExclusiveOrExpression;

        public IInclusiveOrExpression getInclusiveOrExpression() { return _InclusiveOrExpression; }
        public AstToken getOR() { return _OR; }
        public IExclusiveOrExpression getExclusiveOrExpression() { return _ExclusiveOrExpression; }

        public InclusiveOrExpression(IToken leftIToken, IToken rightIToken,
                                     IInclusiveOrExpression _InclusiveOrExpression,
                                     AstToken _OR,
                                     IExclusiveOrExpression _ExclusiveOrExpression)
        {
            super(leftIToken, rightIToken);

            this._InclusiveOrExpression = _InclusiveOrExpression;
            ((Ast) _InclusiveOrExpression).setParent(this);
            this._OR = _OR;
            ((Ast) _OR).setParent(this);
            this._ExclusiveOrExpression = _ExclusiveOrExpression;
            ((Ast) _ExclusiveOrExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_InclusiveOrExpression);
            list.add(_OR);
            list.add(_ExclusiveOrExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof InclusiveOrExpression)) return false;
            if (! super.equals(o)) return false;
            InclusiveOrExpression other = (InclusiveOrExpression) o;
            if (! _InclusiveOrExpression.equals(other._InclusiveOrExpression)) return false;
            if (! _OR.equals(other._OR)) return false;
            if (! _ExclusiveOrExpression.equals(other._ExclusiveOrExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_InclusiveOrExpression.hashCode());
            hash = hash * 31 + (_OR.hashCode());
            hash = hash * 31 + (_ExclusiveOrExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _InclusiveOrExpression.accept(v);
                _OR.accept(v);
                _ExclusiveOrExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 422:  ConditionalAndExpression ::= InclusiveOrExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 423:  ConditionalAndExpression ::= ConditionalAndExpression && InclusiveOrExpression
     *</b>
     */
    static public class ConditionalAndExpression extends Ast implements IConditionalAndExpression
    {
        private IConditionalAndExpression _ConditionalAndExpression;
        private AstToken _AND_AND;
        private IInclusiveOrExpression _InclusiveOrExpression;

        public IConditionalAndExpression getConditionalAndExpression() { return _ConditionalAndExpression; }
        public AstToken getAND_AND() { return _AND_AND; }
        public IInclusiveOrExpression getInclusiveOrExpression() { return _InclusiveOrExpression; }

        public ConditionalAndExpression(IToken leftIToken, IToken rightIToken,
                                        IConditionalAndExpression _ConditionalAndExpression,
                                        AstToken _AND_AND,
                                        IInclusiveOrExpression _InclusiveOrExpression)
        {
            super(leftIToken, rightIToken);

            this._ConditionalAndExpression = _ConditionalAndExpression;
            ((Ast) _ConditionalAndExpression).setParent(this);
            this._AND_AND = _AND_AND;
            ((Ast) _AND_AND).setParent(this);
            this._InclusiveOrExpression = _InclusiveOrExpression;
            ((Ast) _InclusiveOrExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConditionalAndExpression);
            list.add(_AND_AND);
            list.add(_InclusiveOrExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConditionalAndExpression)) return false;
            if (! super.equals(o)) return false;
            ConditionalAndExpression other = (ConditionalAndExpression) o;
            if (! _ConditionalAndExpression.equals(other._ConditionalAndExpression)) return false;
            if (! _AND_AND.equals(other._AND_AND)) return false;
            if (! _InclusiveOrExpression.equals(other._InclusiveOrExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConditionalAndExpression.hashCode());
            hash = hash * 31 + (_AND_AND.hashCode());
            hash = hash * 31 + (_InclusiveOrExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ConditionalAndExpression.accept(v);
                _AND_AND.accept(v);
                _InclusiveOrExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 424:  ConditionalOrExpression ::= ConditionalAndExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 425:  ConditionalOrExpression ::= ConditionalOrExpression || ConditionalAndExpression
     *</b>
     */
    static public class ConditionalOrExpression extends Ast implements IConditionalOrExpression
    {
        private IConditionalOrExpression _ConditionalOrExpression;
        private AstToken _OR_OR;
        private IConditionalAndExpression _ConditionalAndExpression;

        public IConditionalOrExpression getConditionalOrExpression() { return _ConditionalOrExpression; }
        public AstToken getOR_OR() { return _OR_OR; }
        public IConditionalAndExpression getConditionalAndExpression() { return _ConditionalAndExpression; }

        public ConditionalOrExpression(IToken leftIToken, IToken rightIToken,
                                       IConditionalOrExpression _ConditionalOrExpression,
                                       AstToken _OR_OR,
                                       IConditionalAndExpression _ConditionalAndExpression)
        {
            super(leftIToken, rightIToken);

            this._ConditionalOrExpression = _ConditionalOrExpression;
            ((Ast) _ConditionalOrExpression).setParent(this);
            this._OR_OR = _OR_OR;
            ((Ast) _OR_OR).setParent(this);
            this._ConditionalAndExpression = _ConditionalAndExpression;
            ((Ast) _ConditionalAndExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConditionalOrExpression);
            list.add(_OR_OR);
            list.add(_ConditionalAndExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConditionalOrExpression)) return false;
            if (! super.equals(o)) return false;
            ConditionalOrExpression other = (ConditionalOrExpression) o;
            if (! _ConditionalOrExpression.equals(other._ConditionalOrExpression)) return false;
            if (! _OR_OR.equals(other._OR_OR)) return false;
            if (! _ConditionalAndExpression.equals(other._ConditionalAndExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConditionalOrExpression.hashCode());
            hash = hash * 31 + (_OR_OR.hashCode());
            hash = hash * 31 + (_ConditionalAndExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ConditionalOrExpression.accept(v);
                _OR_OR.accept(v);
                _ConditionalAndExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 426:  ConditionalExpression ::= ConditionalOrExpression
     *</em>
     *<p>
     *<b>
     *<li>Rule 427:  ConditionalExpression ::= ConditionalOrExpression ? Expression : ConditionalExpression
     *</b>
     */
    static public class ConditionalExpression extends Ast implements IConditionalExpression
    {
        private IConditionalOrExpression _ConditionalOrExpression;
        private AstToken _QUESTION;
        private IExpression _Expression;
        private AstToken _COLON;
        private IConditionalExpression _ConditionalExpression;

        public IConditionalOrExpression getConditionalOrExpression() { return _ConditionalOrExpression; }
        public AstToken getQUESTION() { return _QUESTION; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getCOLON() { return _COLON; }
        public IConditionalExpression getConditionalExpression() { return _ConditionalExpression; }

        public ConditionalExpression(IToken leftIToken, IToken rightIToken,
                                     IConditionalOrExpression _ConditionalOrExpression,
                                     AstToken _QUESTION,
                                     IExpression _Expression,
                                     AstToken _COLON,
                                     IConditionalExpression _ConditionalExpression)
        {
            super(leftIToken, rightIToken);

            this._ConditionalOrExpression = _ConditionalOrExpression;
            ((Ast) _ConditionalOrExpression).setParent(this);
            this._QUESTION = _QUESTION;
            ((Ast) _QUESTION).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            this._ConditionalExpression = _ConditionalExpression;
            ((Ast) _ConditionalExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ConditionalOrExpression);
            list.add(_QUESTION);
            list.add(_Expression);
            list.add(_COLON);
            list.add(_ConditionalExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ConditionalExpression)) return false;
            if (! super.equals(o)) return false;
            ConditionalExpression other = (ConditionalExpression) o;
            if (! _ConditionalOrExpression.equals(other._ConditionalOrExpression)) return false;
            if (! _QUESTION.equals(other._QUESTION)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            if (! _ConditionalExpression.equals(other._ConditionalExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ConditionalOrExpression.hashCode());
            hash = hash * 31 + (_QUESTION.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            hash = hash * 31 + (_ConditionalExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ConditionalOrExpression.accept(v);
                _QUESTION.accept(v);
                _Expression.accept(v);
                _COLON.accept(v);
                _ConditionalExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 430:  Assignment ::= LeftHandSide AssignmentOperator AssignmentExpression
     *</b>
     */
    static public class Assignment extends Ast implements IAssignment
    {
        private ILeftHandSide _LeftHandSide;
        private IAssignmentOperator _AssignmentOperator;
        private IAssignmentExpression _AssignmentExpression;

        public ILeftHandSide getLeftHandSide() { return _LeftHandSide; }
        public IAssignmentOperator getAssignmentOperator() { return _AssignmentOperator; }
        public IAssignmentExpression getAssignmentExpression() { return _AssignmentExpression; }

        public Assignment(IToken leftIToken, IToken rightIToken,
                          ILeftHandSide _LeftHandSide,
                          IAssignmentOperator _AssignmentOperator,
                          IAssignmentExpression _AssignmentExpression)
        {
            super(leftIToken, rightIToken);

            this._LeftHandSide = _LeftHandSide;
            ((Ast) _LeftHandSide).setParent(this);
            this._AssignmentOperator = _AssignmentOperator;
            ((Ast) _AssignmentOperator).setParent(this);
            this._AssignmentExpression = _AssignmentExpression;
            ((Ast) _AssignmentExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LeftHandSide);
            list.add(_AssignmentOperator);
            list.add(_AssignmentExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Assignment)) return false;
            if (! super.equals(o)) return false;
            Assignment other = (Assignment) o;
            if (! _LeftHandSide.equals(other._LeftHandSide)) return false;
            if (! _AssignmentOperator.equals(other._AssignmentOperator)) return false;
            if (! _AssignmentExpression.equals(other._AssignmentExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LeftHandSide.hashCode());
            hash = hash * 31 + (_AssignmentOperator.hashCode());
            hash = hash * 31 + (_AssignmentExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LeftHandSide.accept(v);
                _AssignmentOperator.accept(v);
                _AssignmentExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 492:  ,opt ::= $Empty
     *</em>
     *<p>
     *<b>
     *<li>Rule 493:  ,opt ::= ,
     *</b>
     */
    static public class Commaopt extends AstToken implements ICommaopt
    {
        public IToken getCOMMA() { return leftIToken; }

        public Commaopt(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<em>
     *<li>Rule 504:  ...opt ::= $Empty
     *</em>
     *<p>
     *<b>
     *<li>Rule 505:  ...opt ::= ...
     *</b>
     */
    static public class Ellipsisopt extends AstToken implements IEllipsisopt
    {
        public IToken getELLIPSIS() { return leftIToken; }

        public Ellipsisopt(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 5:  LPGUserAction ::= $BeginAction BlockStatementsopt $EndAction
     *</b>
     */
    static public class LPGUserAction0 extends Ast implements ILPGUserAction
    {
        private AstToken _BeginAction;
        private IBlockStatementsopt _BlockStatementsopt;
        private AstToken _EndAction;

        public AstToken getBeginAction() { return _BeginAction; }
        /**
         * The value returned by <b>getBlockStatementsopt</b> may be <b>null</b>
         */
        public IBlockStatementsopt getBlockStatementsopt() { return _BlockStatementsopt; }
        public AstToken getEndAction() { return _EndAction; }

        public LPGUserAction0(IToken leftIToken, IToken rightIToken,
                              AstToken _BeginAction,
                              IBlockStatementsopt _BlockStatementsopt,
                              AstToken _EndAction)
        {
            super(leftIToken, rightIToken);

            this._BeginAction = _BeginAction;
            ((Ast) _BeginAction).setParent(this);
            this._BlockStatementsopt = _BlockStatementsopt;
            if (_BlockStatementsopt != null) ((Ast) _BlockStatementsopt).setParent(this);
            this._EndAction = _EndAction;
            ((Ast) _EndAction).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_BeginAction);
            list.add(_BlockStatementsopt);
            list.add(_EndAction);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LPGUserAction0)) return false;
            if (! super.equals(o)) return false;
            LPGUserAction0 other = (LPGUserAction0) o;
            if (! _BeginAction.equals(other._BeginAction)) return false;
            if (_BlockStatementsopt == null)
                if (other._BlockStatementsopt != null) return false;
                else; // continue
            else if (! _BlockStatementsopt.equals(other._BlockStatementsopt)) return false;
            if (! _EndAction.equals(other._EndAction)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_BeginAction.hashCode());
            hash = hash * 31 + (_BlockStatementsopt == null ? 0 : _BlockStatementsopt.hashCode());
            hash = hash * 31 + (_EndAction.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _BeginAction.accept(v);
                if (_BlockStatementsopt != null) _BlockStatementsopt.accept(v);
                _EndAction.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 6:  LPGUserAction ::= $BeginJava BlockStatementsopt $EndJava
     *</b>
     */
    static public class LPGUserAction1 extends Ast implements ILPGUserAction
    {
        private AstToken _BeginJava;
        private IBlockStatementsopt _BlockStatementsopt;
        private AstToken _EndJava;

        public AstToken getBeginJava() { return _BeginJava; }
        /**
         * The value returned by <b>getBlockStatementsopt</b> may be <b>null</b>
         */
        public IBlockStatementsopt getBlockStatementsopt() { return _BlockStatementsopt; }
        public AstToken getEndJava() { return _EndJava; }

        public LPGUserAction1(IToken leftIToken, IToken rightIToken,
                              AstToken _BeginJava,
                              IBlockStatementsopt _BlockStatementsopt,
                              AstToken _EndJava)
        {
            super(leftIToken, rightIToken);

            this._BeginJava = _BeginJava;
            ((Ast) _BeginJava).setParent(this);
            this._BlockStatementsopt = _BlockStatementsopt;
            if (_BlockStatementsopt != null) ((Ast) _BlockStatementsopt).setParent(this);
            this._EndJava = _EndJava;
            ((Ast) _EndJava).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_BeginJava);
            list.add(_BlockStatementsopt);
            list.add(_EndJava);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof LPGUserAction1)) return false;
            if (! super.equals(o)) return false;
            LPGUserAction1 other = (LPGUserAction1) o;
            if (! _BeginJava.equals(other._BeginJava)) return false;
            if (_BlockStatementsopt == null)
                if (other._BlockStatementsopt != null) return false;
                else; // continue
            else if (! _BlockStatementsopt.equals(other._BlockStatementsopt)) return false;
            if (! _EndJava.equals(other._EndJava)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_BeginJava.hashCode());
            hash = hash * 31 + (_BlockStatementsopt == null ? 0 : _BlockStatementsopt.hashCode());
            hash = hash * 31 + (_EndJava.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _BeginJava.accept(v);
                if (_BlockStatementsopt != null) _BlockStatementsopt.accept(v);
                _EndJava.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 7:  LPGUserAction ::= $NoAction
     *</b>
     */
    static public class LPGUserAction2 extends AstToken implements ILPGUserAction
    {
        public IToken getNoAction() { return leftIToken; }

        public LPGUserAction2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 8:  LPGUserAction ::= $NullAction
     *</b>
     */
    static public class LPGUserAction3 extends AstToken implements ILPGUserAction
    {
        public IToken getNullAction() { return leftIToken; }

        public LPGUserAction3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 9:  LPGUserAction ::= $BadAction
     *</b>
     */
    static public class LPGUserAction4 extends AstToken implements ILPGUserAction
    {
        public IToken getBadAction() { return leftIToken; }

        public LPGUserAction4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 16:  IntegralType ::= byte
     *</b>
     */
    static public class IntegralType0 extends AstToken implements IIntegralType
    {
        public IToken getbyte() { return leftIToken; }

        public IntegralType0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 17:  IntegralType ::= short
     *</b>
     */
    static public class IntegralType1 extends AstToken implements IIntegralType
    {
        public IToken getshort() { return leftIToken; }

        public IntegralType1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 18:  IntegralType ::= int
     *</b>
     */
    static public class IntegralType2 extends AstToken implements IIntegralType
    {
        public IToken getint() { return leftIToken; }

        public IntegralType2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 19:  IntegralType ::= long
     *</b>
     */
    static public class IntegralType3 extends AstToken implements IIntegralType
    {
        public IToken getlong() { return leftIToken; }

        public IntegralType3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 20:  IntegralType ::= char
     *</b>
     */
    static public class IntegralType4 extends AstToken implements IIntegralType
    {
        public IToken getchar() { return leftIToken; }

        public IntegralType4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 21:  FloatingPointType ::= float
     *</b>
     */
    static public class FloatingPointType0 extends AstToken implements IFloatingPointType
    {
        public IToken getfloat() { return leftIToken; }

        public FloatingPointType0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 22:  FloatingPointType ::= double
     *</b>
     */
    static public class FloatingPointType1 extends AstToken implements IFloatingPointType
    {
        public IToken getdouble() { return leftIToken; }

        public FloatingPointType1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 45:  WildcardBounds ::= extends ReferenceType
     *</b>
     */
    static public class WildcardBounds0 extends Ast implements IWildcardBounds
    {
        private AstToken _extends;
        private IReferenceType _ReferenceType;

        public AstToken getextends() { return _extends; }
        public IReferenceType getReferenceType() { return _ReferenceType; }

        public WildcardBounds0(IToken leftIToken, IToken rightIToken,
                               AstToken _extends,
                               IReferenceType _ReferenceType)
        {
            super(leftIToken, rightIToken);

            this._extends = _extends;
            ((Ast) _extends).setParent(this);
            this._ReferenceType = _ReferenceType;
            ((Ast) _ReferenceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_extends);
            list.add(_ReferenceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof WildcardBounds0)) return false;
            if (! super.equals(o)) return false;
            WildcardBounds0 other = (WildcardBounds0) o;
            if (! _extends.equals(other._extends)) return false;
            if (! _ReferenceType.equals(other._ReferenceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_extends.hashCode());
            hash = hash * 31 + (_ReferenceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _extends.accept(v);
                _ReferenceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 46:  WildcardBounds ::= super ReferenceType
     *</b>
     */
    static public class WildcardBounds1 extends Ast implements IWildcardBounds
    {
        private AstToken _super;
        private IReferenceType _ReferenceType;

        public AstToken getsuper() { return _super; }
        public IReferenceType getReferenceType() { return _ReferenceType; }

        public WildcardBounds1(IToken leftIToken, IToken rightIToken,
                               AstToken _super,
                               IReferenceType _ReferenceType)
        {
            super(leftIToken, rightIToken);

            this._super = _super;
            ((Ast) _super).setParent(this);
            this._ReferenceType = _ReferenceType;
            ((Ast) _ReferenceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_super);
            list.add(_ReferenceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof WildcardBounds1)) return false;
            if (! super.equals(o)) return false;
            WildcardBounds1 other = (WildcardBounds1) o;
            if (! _super.equals(other._super)) return false;
            if (! _ReferenceType.equals(other._ReferenceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_ReferenceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _super.accept(v);
                _ReferenceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 80:  ClassModifier ::= public
     *</b>
     */
    static public class ClassModifier0 extends AstToken implements IClassModifier
    {
        public IToken getpublic() { return leftIToken; }

        public ClassModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 81:  ClassModifier ::= protected
     *</b>
     */
    static public class ClassModifier1 extends AstToken implements IClassModifier
    {
        public IToken getprotected() { return leftIToken; }

        public ClassModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 82:  ClassModifier ::= private
     *</b>
     */
    static public class ClassModifier2 extends AstToken implements IClassModifier
    {
        public IToken getprivate() { return leftIToken; }

        public ClassModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 83:  ClassModifier ::= abstract
     *</b>
     */
    static public class ClassModifier3 extends AstToken implements IClassModifier
    {
        public IToken getabstract() { return leftIToken; }

        public ClassModifier3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 84:  ClassModifier ::= static
     *</b>
     */
    static public class ClassModifier4 extends AstToken implements IClassModifier
    {
        public IToken getstatic() { return leftIToken; }

        public ClassModifier4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 85:  ClassModifier ::= final
     *</b>
     */
    static public class ClassModifier5 extends AstToken implements IClassModifier
    {
        public IToken getfinal() { return leftIToken; }

        public ClassModifier5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 86:  ClassModifier ::= strictfp
     *</b>
     */
    static public class ClassModifier6 extends AstToken implements IClassModifier
    {
        public IToken getstrictfp() { return leftIToken; }

        public ClassModifier6(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 118:  FieldModifier ::= public
     *</b>
     */
    static public class FieldModifier0 extends AstToken implements IFieldModifier
    {
        public IToken getpublic() { return leftIToken; }

        public FieldModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 119:  FieldModifier ::= protected
     *</b>
     */
    static public class FieldModifier1 extends AstToken implements IFieldModifier
    {
        public IToken getprotected() { return leftIToken; }

        public FieldModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 120:  FieldModifier ::= private
     *</b>
     */
    static public class FieldModifier2 extends AstToken implements IFieldModifier
    {
        public IToken getprivate() { return leftIToken; }

        public FieldModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 121:  FieldModifier ::= static
     *</b>
     */
    static public class FieldModifier3 extends AstToken implements IFieldModifier
    {
        public IToken getstatic() { return leftIToken; }

        public FieldModifier3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 122:  FieldModifier ::= final
     *</b>
     */
    static public class FieldModifier4 extends AstToken implements IFieldModifier
    {
        public IToken getfinal() { return leftIToken; }

        public FieldModifier4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 123:  FieldModifier ::= transient
     *</b>
     */
    static public class FieldModifier5 extends AstToken implements IFieldModifier
    {
        public IToken gettransient() { return leftIToken; }

        public FieldModifier5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 124:  FieldModifier ::= volatile
     *</b>
     */
    static public class FieldModifier6 extends AstToken implements IFieldModifier
    {
        public IToken getvolatile() { return leftIToken; }

        public FieldModifier6(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 129:  MethodDeclarator ::= identifier ( FormalParameterListopt )
     *</b>
     */
    static public class MethodDeclarator0 extends Ast implements IMethodDeclarator
    {
        private identifier _identifier;
        private AstToken _LPAREN;
        private IFormalParameterListopt _FormalParameterListopt;
        private AstToken _RPAREN;

        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getFormalParameterListopt</b> may be <b>null</b>
         */
        public IFormalParameterListopt getFormalParameterListopt() { return _FormalParameterListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodDeclarator0(IToken leftIToken, IToken rightIToken,
                                 identifier _identifier,
                                 AstToken _LPAREN,
                                 IFormalParameterListopt _FormalParameterListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._FormalParameterListopt = _FormalParameterListopt;
            if (_FormalParameterListopt != null) ((Ast) _FormalParameterListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_FormalParameterListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodDeclarator0)) return false;
            if (! super.equals(o)) return false;
            MethodDeclarator0 other = (MethodDeclarator0) o;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_FormalParameterListopt == null)
                if (other._FormalParameterListopt != null) return false;
                else; // continue
            else if (! _FormalParameterListopt.equals(other._FormalParameterListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_FormalParameterListopt == null ? 0 : _FormalParameterListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _identifier.accept(v);
                _LPAREN.accept(v);
                if (_FormalParameterListopt != null) _FormalParameterListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 130:  MethodDeclarator ::= MethodDeclarator [ ]
     *</b>
     */
    static public class MethodDeclarator1 extends Ast implements IMethodDeclarator
    {
        private IMethodDeclarator _MethodDeclarator;
        private AstToken _LBRACKET;
        private AstToken _RBRACKET;

        public IMethodDeclarator getMethodDeclarator() { return _MethodDeclarator; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public MethodDeclarator1(IToken leftIToken, IToken rightIToken,
                                 IMethodDeclarator _MethodDeclarator,
                                 AstToken _LBRACKET,
                                 AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._MethodDeclarator = _MethodDeclarator;
            ((Ast) _MethodDeclarator).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MethodDeclarator);
            list.add(_LBRACKET);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodDeclarator1)) return false;
            if (! super.equals(o)) return false;
            MethodDeclarator1 other = (MethodDeclarator1) o;
            if (! _MethodDeclarator.equals(other._MethodDeclarator)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MethodDeclarator.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MethodDeclarator.accept(v);
                _LBRACKET.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 144:  MethodModifier ::= public
     *</b>
     */
    static public class MethodModifier0 extends AstToken implements IMethodModifier
    {
        public IToken getpublic() { return leftIToken; }

        public MethodModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 145:  MethodModifier ::= protected
     *</b>
     */
    static public class MethodModifier1 extends AstToken implements IMethodModifier
    {
        public IToken getprotected() { return leftIToken; }

        public MethodModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 146:  MethodModifier ::= private
     *</b>
     */
    static public class MethodModifier2 extends AstToken implements IMethodModifier
    {
        public IToken getprivate() { return leftIToken; }

        public MethodModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 147:  MethodModifier ::= abstract
     *</b>
     */
    static public class MethodModifier3 extends AstToken implements IMethodModifier
    {
        public IToken getabstract() { return leftIToken; }

        public MethodModifier3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 148:  MethodModifier ::= static
     *</b>
     */
    static public class MethodModifier4 extends AstToken implements IMethodModifier
    {
        public IToken getstatic() { return leftIToken; }

        public MethodModifier4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 149:  MethodModifier ::= final
     *</b>
     */
    static public class MethodModifier5 extends AstToken implements IMethodModifier
    {
        public IToken getfinal() { return leftIToken; }

        public MethodModifier5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 150:  MethodModifier ::= synchronized
     *</b>
     */
    static public class MethodModifier6 extends AstToken implements IMethodModifier
    {
        public IToken getsynchronized() { return leftIToken; }

        public MethodModifier6(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 151:  MethodModifier ::= native
     *</b>
     */
    static public class MethodModifier7 extends AstToken implements IMethodModifier
    {
        public IToken getnative() { return leftIToken; }

        public MethodModifier7(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 152:  MethodModifier ::= strictfp
     *</b>
     */
    static public class MethodModifier8 extends AstToken implements IMethodModifier
    {
        public IToken getstrictfp() { return leftIToken; }

        public MethodModifier8(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 168:  ConstructorModifier ::= public
     *</b>
     */
    static public class ConstructorModifier0 extends AstToken implements IConstructorModifier
    {
        public IToken getpublic() { return leftIToken; }

        public ConstructorModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 169:  ConstructorModifier ::= protected
     *</b>
     */
    static public class ConstructorModifier1 extends AstToken implements IConstructorModifier
    {
        public IToken getprotected() { return leftIToken; }

        public ConstructorModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 170:  ConstructorModifier ::= private
     *</b>
     */
    static public class ConstructorModifier2 extends AstToken implements IConstructorModifier
    {
        public IToken getprivate() { return leftIToken; }

        public ConstructorModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 172:  ExplicitConstructorInvocation ::= TypeArgumentsopt this ( ArgumentListopt ) ;
     *</b>
     */
    static public class ExplicitConstructorInvocation0 extends Ast implements IExplicitConstructorInvocation
    {
        private TypeArguments _TypeArgumentsopt;
        private AstToken _this;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public AstToken getthis() { return _this; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ExplicitConstructorInvocation0(IToken leftIToken, IToken rightIToken,
                                              TypeArguments _TypeArgumentsopt,
                                              AstToken _this,
                                              AstToken _LPAREN,
                                              IArgumentListopt _ArgumentListopt,
                                              AstToken _RPAREN,
                                              AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._this = _this;
            ((Ast) _this).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeArgumentsopt);
            list.add(_this);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExplicitConstructorInvocation0)) return false;
            if (! super.equals(o)) return false;
            ExplicitConstructorInvocation0 other = (ExplicitConstructorInvocation0) o;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _this.equals(other._this)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_this.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _this.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 173:  ExplicitConstructorInvocation ::= TypeArgumentsopt super ( ArgumentListopt ) ;
     *</b>
     */
    static public class ExplicitConstructorInvocation1 extends Ast implements IExplicitConstructorInvocation
    {
        private TypeArguments _TypeArgumentsopt;
        private AstToken _super;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public AstToken getsuper() { return _super; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ExplicitConstructorInvocation1(IToken leftIToken, IToken rightIToken,
                                              TypeArguments _TypeArgumentsopt,
                                              AstToken _super,
                                              AstToken _LPAREN,
                                              IArgumentListopt _ArgumentListopt,
                                              AstToken _RPAREN,
                                              AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._super = _super;
            ((Ast) _super).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeArgumentsopt);
            list.add(_super);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExplicitConstructorInvocation1)) return false;
            if (! super.equals(o)) return false;
            ExplicitConstructorInvocation1 other = (ExplicitConstructorInvocation1) o;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _super.equals(other._super)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _super.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 174:  ExplicitConstructorInvocation ::= Primary . TypeArgumentsopt super ( ArgumentListopt ) ;
     *</b>
     */
    static public class ExplicitConstructorInvocation2 extends Ast implements IExplicitConstructorInvocation
    {
        private IPrimary _Primary;
        private AstToken _DOT;
        private TypeArguments _TypeArgumentsopt;
        private AstToken _super;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;
        private AstToken _SEMICOLON;

        public IPrimary getPrimary() { return _Primary; }
        public AstToken getDOT() { return _DOT; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public AstToken getsuper() { return _super; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public ExplicitConstructorInvocation2(IToken leftIToken, IToken rightIToken,
                                              IPrimary _Primary,
                                              AstToken _DOT,
                                              TypeArguments _TypeArgumentsopt,
                                              AstToken _super,
                                              AstToken _LPAREN,
                                              IArgumentListopt _ArgumentListopt,
                                              AstToken _RPAREN,
                                              AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._Primary = _Primary;
            ((Ast) _Primary).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._super = _super;
            ((Ast) _super).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Primary);
            list.add(_DOT);
            list.add(_TypeArgumentsopt);
            list.add(_super);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExplicitConstructorInvocation2)) return false;
            if (! super.equals(o)) return false;
            ExplicitConstructorInvocation2 other = (ExplicitConstructorInvocation2) o;
            if (! _Primary.equals(other._Primary)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _super.equals(other._super)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Primary.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Primary.accept(v);
                _DOT.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _super.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 188:  InterfaceModifier ::= public
     *</b>
     */
    static public class InterfaceModifier0 extends AstToken implements IInterfaceModifier
    {
        public IToken getpublic() { return leftIToken; }

        public InterfaceModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 189:  InterfaceModifier ::= protected
     *</b>
     */
    static public class InterfaceModifier1 extends AstToken implements IInterfaceModifier
    {
        public IToken getprotected() { return leftIToken; }

        public InterfaceModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 190:  InterfaceModifier ::= private
     *</b>
     */
    static public class InterfaceModifier2 extends AstToken implements IInterfaceModifier
    {
        public IToken getprivate() { return leftIToken; }

        public InterfaceModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 191:  InterfaceModifier ::= abstract
     *</b>
     */
    static public class InterfaceModifier3 extends AstToken implements IInterfaceModifier
    {
        public IToken getabstract() { return leftIToken; }

        public InterfaceModifier3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 192:  InterfaceModifier ::= static
     *</b>
     */
    static public class InterfaceModifier4 extends AstToken implements IInterfaceModifier
    {
        public IToken getstatic() { return leftIToken; }

        public InterfaceModifier4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 193:  InterfaceModifier ::= strictfp
     *</b>
     */
    static public class InterfaceModifier5 extends AstToken implements IInterfaceModifier
    {
        public IToken getstrictfp() { return leftIToken; }

        public InterfaceModifier5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 194:  ExtendsInterfaces ::= extends InterfaceType
     *</b>
     */
    static public class ExtendsInterfaces0 extends Ast implements IExtendsInterfaces
    {
        private AstToken _extends;
        private InterfaceType _InterfaceType;

        public AstToken getextends() { return _extends; }
        public InterfaceType getInterfaceType() { return _InterfaceType; }

        public ExtendsInterfaces0(IToken leftIToken, IToken rightIToken,
                                  AstToken _extends,
                                  InterfaceType _InterfaceType)
        {
            super(leftIToken, rightIToken);

            this._extends = _extends;
            ((Ast) _extends).setParent(this);
            this._InterfaceType = _InterfaceType;
            ((Ast) _InterfaceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_extends);
            list.add(_InterfaceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExtendsInterfaces0)) return false;
            if (! super.equals(o)) return false;
            ExtendsInterfaces0 other = (ExtendsInterfaces0) o;
            if (! _extends.equals(other._extends)) return false;
            if (! _InterfaceType.equals(other._InterfaceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_extends.hashCode());
            hash = hash * 31 + (_InterfaceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _extends.accept(v);
                _InterfaceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 195:  ExtendsInterfaces ::= ExtendsInterfaces , InterfaceType
     *</b>
     */
    static public class ExtendsInterfaces1 extends Ast implements IExtendsInterfaces
    {
        private IExtendsInterfaces _ExtendsInterfaces;
        private AstToken _COMMA;
        private InterfaceType _InterfaceType;

        public IExtendsInterfaces getExtendsInterfaces() { return _ExtendsInterfaces; }
        public AstToken getCOMMA() { return _COMMA; }
        public InterfaceType getInterfaceType() { return _InterfaceType; }

        public ExtendsInterfaces1(IToken leftIToken, IToken rightIToken,
                                  IExtendsInterfaces _ExtendsInterfaces,
                                  AstToken _COMMA,
                                  InterfaceType _InterfaceType)
        {
            super(leftIToken, rightIToken);

            this._ExtendsInterfaces = _ExtendsInterfaces;
            ((Ast) _ExtendsInterfaces).setParent(this);
            this._COMMA = _COMMA;
            ((Ast) _COMMA).setParent(this);
            this._InterfaceType = _InterfaceType;
            ((Ast) _InterfaceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ExtendsInterfaces);
            list.add(_COMMA);
            list.add(_InterfaceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ExtendsInterfaces1)) return false;
            if (! super.equals(o)) return false;
            ExtendsInterfaces1 other = (ExtendsInterfaces1) o;
            if (! _ExtendsInterfaces.equals(other._ExtendsInterfaces)) return false;
            if (! _COMMA.equals(other._COMMA)) return false;
            if (! _InterfaceType.equals(other._InterfaceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ExtendsInterfaces.hashCode());
            hash = hash * 31 + (_COMMA.hashCode());
            hash = hash * 31 + (_InterfaceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ExtendsInterfaces.accept(v);
                _COMMA.accept(v);
                _InterfaceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 208:  ConstantModifier ::= public
     *</b>
     */
    static public class ConstantModifier0 extends AstToken implements IConstantModifier
    {
        public IToken getpublic() { return leftIToken; }

        public ConstantModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 209:  ConstantModifier ::= static
     *</b>
     */
    static public class ConstantModifier1 extends AstToken implements IConstantModifier
    {
        public IToken getstatic() { return leftIToken; }

        public ConstantModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 210:  ConstantModifier ::= final
     *</b>
     */
    static public class ConstantModifier2 extends AstToken implements IConstantModifier
    {
        public IToken getfinal() { return leftIToken; }

        public ConstantModifier2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 215:  AbstractMethodModifier ::= public
     *</b>
     */
    static public class AbstractMethodModifier0 extends AstToken implements IAbstractMethodModifier
    {
        public IToken getpublic() { return leftIToken; }

        public AbstractMethodModifier0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 216:  AbstractMethodModifier ::= abstract
     *</b>
     */
    static public class AbstractMethodModifier1 extends AstToken implements IAbstractMethodModifier
    {
        public IToken getabstract() { return leftIToken; }

        public AbstractMethodModifier1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 221:  AnnotationTypeElementDeclaration ::= AbstractMethodModifiersopt Type identifier ( ) DefaultValueopt ;
     *</b>
     */
    static public class AnnotationTypeElementDeclaration0 extends Ast implements IAnnotationTypeElementDeclaration
    {
        private IAbstractMethodModifiersopt _AbstractMethodModifiersopt;
        private IType _Type;
        private identifier _identifier;
        private AstToken _LPAREN;
        private AstToken _RPAREN;
        private DefaultValue _DefaultValueopt;
        private AstToken _SEMICOLON;

        /**
         * The value returned by <b>getAbstractMethodModifiersopt</b> may be <b>null</b>
         */
        public IAbstractMethodModifiersopt getAbstractMethodModifiersopt() { return _AbstractMethodModifiersopt; }
        public IType getType() { return _Type; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        public AstToken getRPAREN() { return _RPAREN; }
        /**
         * The value returned by <b>getDefaultValueopt</b> may be <b>null</b>
         */
        public DefaultValue getDefaultValueopt() { return _DefaultValueopt; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public AnnotationTypeElementDeclaration0(IToken leftIToken, IToken rightIToken,
                                                 IAbstractMethodModifiersopt _AbstractMethodModifiersopt,
                                                 IType _Type,
                                                 identifier _identifier,
                                                 AstToken _LPAREN,
                                                 AstToken _RPAREN,
                                                 DefaultValue _DefaultValueopt,
                                                 AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._AbstractMethodModifiersopt = _AbstractMethodModifiersopt;
            if (_AbstractMethodModifiersopt != null) ((Ast) _AbstractMethodModifiersopt).setParent(this);
            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._DefaultValueopt = _DefaultValueopt;
            if (_DefaultValueopt != null) ((Ast) _DefaultValueopt).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AbstractMethodModifiersopt);
            list.add(_Type);
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_RPAREN);
            list.add(_DefaultValueopt);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AnnotationTypeElementDeclaration0)) return false;
            if (! super.equals(o)) return false;
            AnnotationTypeElementDeclaration0 other = (AnnotationTypeElementDeclaration0) o;
            if (_AbstractMethodModifiersopt == null)
                if (other._AbstractMethodModifiersopt != null) return false;
                else; // continue
            else if (! _AbstractMethodModifiersopt.equals(other._AbstractMethodModifiersopt)) return false;
            if (! _Type.equals(other._Type)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (_DefaultValueopt == null)
                if (other._DefaultValueopt != null) return false;
                else; // continue
            else if (! _DefaultValueopt.equals(other._DefaultValueopt)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AbstractMethodModifiersopt == null ? 0 : _AbstractMethodModifiersopt.hashCode());
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_DefaultValueopt == null ? 0 : _DefaultValueopt.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                if (_AbstractMethodModifiersopt != null) _AbstractMethodModifiersopt.accept(v);
                _Type.accept(v);
                _identifier.accept(v);
                _LPAREN.accept(v);
                _RPAREN.accept(v);
                if (_DefaultValueopt != null) _DefaultValueopt.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 227:  AnnotationTypeElementDeclaration ::= ;
     *</b>
     */
    static public class AnnotationTypeElementDeclaration1 extends AstToken implements IAnnotationTypeElementDeclaration
    {
        public IToken getSEMICOLON() { return leftIToken; }

        public AnnotationTypeElementDeclaration1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 295:  AssertStatement ::= assert Expression ;
     *</b>
     */
    static public class AssertStatement0 extends Ast implements IAssertStatement
    {
        private AstToken _assert;
        private IExpression _Expression;
        private AstToken _SEMICOLON;

        public AstToken getassert() { return _assert; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public AssertStatement0(IToken leftIToken, IToken rightIToken,
                                AstToken _assert,
                                IExpression _Expression,
                                AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._assert = _assert;
            ((Ast) _assert).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_assert);
            list.add(_Expression);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AssertStatement0)) return false;
            if (! super.equals(o)) return false;
            AssertStatement0 other = (AssertStatement0) o;
            if (! _assert.equals(other._assert)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_assert.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _assert.accept(v);
                _Expression.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 296:  AssertStatement ::= assert Expression : Expression ;
     *</b>
     */
    static public class AssertStatement1 extends Ast implements IAssertStatement
    {
        private AstToken _assert;
        private IExpression _Expression;
        private AstToken _COLON;
        private IExpression _Expression4;
        private AstToken _SEMICOLON;

        public AstToken getassert() { return _assert; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getCOLON() { return _COLON; }
        public IExpression getExpression4() { return _Expression4; }
        public AstToken getSEMICOLON() { return _SEMICOLON; }

        public AssertStatement1(IToken leftIToken, IToken rightIToken,
                                AstToken _assert,
                                IExpression _Expression,
                                AstToken _COLON,
                                IExpression _Expression4,
                                AstToken _SEMICOLON)
        {
            super(leftIToken, rightIToken);

            this._assert = _assert;
            ((Ast) _assert).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            this._Expression4 = _Expression4;
            ((Ast) _Expression4).setParent(this);
            this._SEMICOLON = _SEMICOLON;
            ((Ast) _SEMICOLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_assert);
            list.add(_Expression);
            list.add(_COLON);
            list.add(_Expression4);
            list.add(_SEMICOLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AssertStatement1)) return false;
            if (! super.equals(o)) return false;
            AssertStatement1 other = (AssertStatement1) o;
            if (! _assert.equals(other._assert)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            if (! _Expression4.equals(other._Expression4)) return false;
            if (! _SEMICOLON.equals(other._SEMICOLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_assert.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            hash = hash * 31 + (_Expression4.hashCode());
            hash = hash * 31 + (_SEMICOLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _assert.accept(v);
                _Expression.accept(v);
                _COLON.accept(v);
                _Expression4.accept(v);
                _SEMICOLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 304:  SwitchLabel ::= case ConstantExpression :
     *</b>
     */
    static public class SwitchLabel0 extends Ast implements ISwitchLabel
    {
        private AstToken _case;
        private IConstantExpression _ConstantExpression;
        private AstToken _COLON;

        public AstToken getcase() { return _case; }
        public IConstantExpression getConstantExpression() { return _ConstantExpression; }
        public AstToken getCOLON() { return _COLON; }

        public SwitchLabel0(IToken leftIToken, IToken rightIToken,
                            AstToken _case,
                            IConstantExpression _ConstantExpression,
                            AstToken _COLON)
        {
            super(leftIToken, rightIToken);

            this._case = _case;
            ((Ast) _case).setParent(this);
            this._ConstantExpression = _ConstantExpression;
            ((Ast) _ConstantExpression).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_case);
            list.add(_ConstantExpression);
            list.add(_COLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchLabel0)) return false;
            if (! super.equals(o)) return false;
            SwitchLabel0 other = (SwitchLabel0) o;
            if (! _case.equals(other._case)) return false;
            if (! _ConstantExpression.equals(other._ConstantExpression)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_case.hashCode());
            hash = hash * 31 + (_ConstantExpression.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _case.accept(v);
                _ConstantExpression.accept(v);
                _COLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 305:  SwitchLabel ::= case EnumConstant :
     *</b>
     */
    static public class SwitchLabel1 extends Ast implements ISwitchLabel
    {
        private AstToken _case;
        private IEnumConstant _EnumConstant;
        private AstToken _COLON;

        public AstToken getcase() { return _case; }
        public IEnumConstant getEnumConstant() { return _EnumConstant; }
        public AstToken getCOLON() { return _COLON; }

        public SwitchLabel1(IToken leftIToken, IToken rightIToken,
                            AstToken _case,
                            IEnumConstant _EnumConstant,
                            AstToken _COLON)
        {
            super(leftIToken, rightIToken);

            this._case = _case;
            ((Ast) _case).setParent(this);
            this._EnumConstant = _EnumConstant;
            ((Ast) _EnumConstant).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_case);
            list.add(_EnumConstant);
            list.add(_COLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchLabel1)) return false;
            if (! super.equals(o)) return false;
            SwitchLabel1 other = (SwitchLabel1) o;
            if (! _case.equals(other._case)) return false;
            if (! _EnumConstant.equals(other._EnumConstant)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_case.hashCode());
            hash = hash * 31 + (_EnumConstant.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _case.accept(v);
                _EnumConstant.accept(v);
                _COLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 306:  SwitchLabel ::= default :
     *</b>
     */
    static public class SwitchLabel2 extends Ast implements ISwitchLabel
    {
        private AstToken _default;
        private AstToken _COLON;

        public AstToken getdefault() { return _default; }
        public AstToken getCOLON() { return _COLON; }

        public SwitchLabel2(IToken leftIToken, IToken rightIToken,
                            AstToken _default,
                            AstToken _COLON)
        {
            super(leftIToken, rightIToken);

            this._default = _default;
            ((Ast) _default).setParent(this);
            this._COLON = _COLON;
            ((Ast) _COLON).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_default);
            list.add(_COLON);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof SwitchLabel2)) return false;
            if (! super.equals(o)) return false;
            SwitchLabel2 other = (SwitchLabel2) o;
            if (! _default.equals(other._default)) return false;
            if (! _COLON.equals(other._COLON)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_default.hashCode());
            hash = hash * 31 + (_COLON.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _default.accept(v);
                _COLON.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 326:  TryStatement ::= try Block Catches
     *</b>
     */
    static public class TryStatement0 extends Ast implements ITryStatement
    {
        private AstToken _try;
        private Block _Block;
        private ICatches _Catches;

        public AstToken gettry() { return _try; }
        public Block getBlock() { return _Block; }
        public ICatches getCatches() { return _Catches; }

        public TryStatement0(IToken leftIToken, IToken rightIToken,
                             AstToken _try,
                             Block _Block,
                             ICatches _Catches)
        {
            super(leftIToken, rightIToken);

            this._try = _try;
            ((Ast) _try).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            this._Catches = _Catches;
            ((Ast) _Catches).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_try);
            list.add(_Block);
            list.add(_Catches);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TryStatement0)) return false;
            if (! super.equals(o)) return false;
            TryStatement0 other = (TryStatement0) o;
            if (! _try.equals(other._try)) return false;
            if (! _Block.equals(other._Block)) return false;
            if (! _Catches.equals(other._Catches)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_try.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            hash = hash * 31 + (_Catches.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _try.accept(v);
                _Block.accept(v);
                _Catches.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 327:  TryStatement ::= try Block Catchesopt Finally
     *</b>
     */
    static public class TryStatement1 extends Ast implements ITryStatement
    {
        private AstToken _try;
        private Block _Block;
        private ICatchesopt _Catchesopt;
        private Finally _Finally;

        public AstToken gettry() { return _try; }
        public Block getBlock() { return _Block; }
        /**
         * The value returned by <b>getCatchesopt</b> may be <b>null</b>
         */
        public ICatchesopt getCatchesopt() { return _Catchesopt; }
        public Finally getFinally() { return _Finally; }

        public TryStatement1(IToken leftIToken, IToken rightIToken,
                             AstToken _try,
                             Block _Block,
                             ICatchesopt _Catchesopt,
                             Finally _Finally)
        {
            super(leftIToken, rightIToken);

            this._try = _try;
            ((Ast) _try).setParent(this);
            this._Block = _Block;
            ((Ast) _Block).setParent(this);
            this._Catchesopt = _Catchesopt;
            if (_Catchesopt != null) ((Ast) _Catchesopt).setParent(this);
            this._Finally = _Finally;
            ((Ast) _Finally).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_try);
            list.add(_Block);
            list.add(_Catchesopt);
            list.add(_Finally);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof TryStatement1)) return false;
            if (! super.equals(o)) return false;
            TryStatement1 other = (TryStatement1) o;
            if (! _try.equals(other._try)) return false;
            if (! _Block.equals(other._Block)) return false;
            if (_Catchesopt == null)
                if (other._Catchesopt != null) return false;
                else; // continue
            else if (! _Catchesopt.equals(other._Catchesopt)) return false;
            if (! _Finally.equals(other._Finally)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_try.hashCode());
            hash = hash * 31 + (_Block.hashCode());
            hash = hash * 31 + (_Catchesopt == null ? 0 : _Catchesopt.hashCode());
            hash = hash * 31 + (_Finally.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _try.accept(v);
                _Block.accept(v);
                if (_Catchesopt != null) _Catchesopt.accept(v);
                _Finally.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 335:  PrimaryNoNewArray ::= Type . class
     *</b>
     */
    static public class PrimaryNoNewArray0 extends Ast implements IPrimaryNoNewArray
    {
        private IType _Type;
        private AstToken _DOT;
        private AstToken _class;

        public IType getType() { return _Type; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getclass() { return _class; }

        public PrimaryNoNewArray0(IToken leftIToken, IToken rightIToken,
                                  IType _Type,
                                  AstToken _DOT,
                                  AstToken _class)
        {
            super(leftIToken, rightIToken);

            this._Type = _Type;
            ((Ast) _Type).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._class = _class;
            ((Ast) _class).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Type);
            list.add(_DOT);
            list.add(_class);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PrimaryNoNewArray0)) return false;
            if (! super.equals(o)) return false;
            PrimaryNoNewArray0 other = (PrimaryNoNewArray0) o;
            if (! _Type.equals(other._Type)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _class.equals(other._class)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Type.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_class.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Type.accept(v);
                _DOT.accept(v);
                _class.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 336:  PrimaryNoNewArray ::= void . class
     *</b>
     */
    static public class PrimaryNoNewArray1 extends Ast implements IPrimaryNoNewArray
    {
        private AstToken _void;
        private AstToken _DOT;
        private AstToken _class;

        public AstToken getvoid() { return _void; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getclass() { return _class; }

        public PrimaryNoNewArray1(IToken leftIToken, IToken rightIToken,
                                  AstToken _void,
                                  AstToken _DOT,
                                  AstToken _class)
        {
            super(leftIToken, rightIToken);

            this._void = _void;
            ((Ast) _void).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._class = _class;
            ((Ast) _class).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_void);
            list.add(_DOT);
            list.add(_class);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PrimaryNoNewArray1)) return false;
            if (! super.equals(o)) return false;
            PrimaryNoNewArray1 other = (PrimaryNoNewArray1) o;
            if (! _void.equals(other._void)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _class.equals(other._class)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_void.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_class.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _void.accept(v);
                _DOT.accept(v);
                _class.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 337:  PrimaryNoNewArray ::= this
     *</b>
     */
    static public class PrimaryNoNewArray2 extends AstToken implements IPrimaryNoNewArray
    {
        public IToken getthis() { return leftIToken; }

        public PrimaryNoNewArray2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 338:  PrimaryNoNewArray ::= ClassName . this
     *</b>
     */
    static public class PrimaryNoNewArray3 extends Ast implements IPrimaryNoNewArray
    {
        private IClassName _ClassName;
        private AstToken _DOT;
        private AstToken _this;

        public IClassName getClassName() { return _ClassName; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getthis() { return _this; }

        public PrimaryNoNewArray3(IToken leftIToken, IToken rightIToken,
                                  IClassName _ClassName,
                                  AstToken _DOT,
                                  AstToken _this)
        {
            super(leftIToken, rightIToken);

            this._ClassName = _ClassName;
            ((Ast) _ClassName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._this = _this;
            ((Ast) _this).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassName);
            list.add(_DOT);
            list.add(_this);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PrimaryNoNewArray3)) return false;
            if (! super.equals(o)) return false;
            PrimaryNoNewArray3 other = (PrimaryNoNewArray3) o;
            if (! _ClassName.equals(other._ClassName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _this.equals(other._this)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_this.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ClassName.accept(v);
                _DOT.accept(v);
                _this.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 339:  PrimaryNoNewArray ::= ( Expression )
     *</b>
     */
    static public class PrimaryNoNewArray4 extends Ast implements IPrimaryNoNewArray
    {
        private AstToken _LPAREN;
        private IExpression _Expression;
        private AstToken _RPAREN;

        public AstToken getLPAREN() { return _LPAREN; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRPAREN() { return _RPAREN; }

        public PrimaryNoNewArray4(IToken leftIToken, IToken rightIToken,
                                  AstToken _LPAREN,
                                  IExpression _Expression,
                                  AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LPAREN);
            list.add(_Expression);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof PrimaryNoNewArray4)) return false;
            if (! super.equals(o)) return false;
            PrimaryNoNewArray4 other = (PrimaryNoNewArray4) o;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LPAREN.accept(v);
                _Expression.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 344:  Literal ::= IntegerLiteral
     *</b>
     */
    static public class Literal0 extends AstToken implements ILiteral
    {
        public IToken getIntegerLiteral() { return leftIToken; }

        public Literal0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 345:  Literal ::= LongLiteral
     *</b>
     */
    static public class Literal1 extends AstToken implements ILiteral
    {
        public IToken getLongLiteral() { return leftIToken; }

        public Literal1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 346:  Literal ::= FloatingPointLiteral
     *</b>
     */
    static public class Literal2 extends AstToken implements ILiteral
    {
        public IToken getFloatingPointLiteral() { return leftIToken; }

        public Literal2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 347:  Literal ::= DoubleLiteral
     *</b>
     */
    static public class Literal3 extends AstToken implements ILiteral
    {
        public IToken getDoubleLiteral() { return leftIToken; }

        public Literal3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 349:  Literal ::= CharacterLiteral
     *</b>
     */
    static public class Literal4 extends AstToken implements ILiteral
    {
        public IToken getCharacterLiteral() { return leftIToken; }

        public Literal4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 350:  Literal ::= StringLiteral
     *</b>
     */
    static public class Literal5 extends AstToken implements ILiteral
    {
        public IToken getStringLiteral() { return leftIToken; }

        public Literal5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 351:  Literal ::= null
     *</b>
     */
    static public class Literal6 extends AstToken implements ILiteral
    {
        public IToken getnull() { return leftIToken; }

        public Literal6(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 352:  BooleanLiteral ::= true
     *</b>
     */
    static public class BooleanLiteral0 extends AstToken implements IBooleanLiteral
    {
        public IToken gettrue() { return leftIToken; }

        public BooleanLiteral0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 353:  BooleanLiteral ::= false
     *</b>
     */
    static public class BooleanLiteral1 extends AstToken implements IBooleanLiteral
    {
        public IToken getfalse() { return leftIToken; }

        public BooleanLiteral1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 354:  ClassInstanceCreationExpression ::= new TypeArgumentsopt ClassOrInterfaceType TypeArgumentsopt ( ArgumentListopt ) ClassBodyopt
     *</b>
     */
    static public class ClassInstanceCreationExpression0 extends Ast implements IClassInstanceCreationExpression
    {
        private AstToken _new;
        private TypeArguments _TypeArgumentsopt;
        private ClassType _ClassOrInterfaceType;
        private TypeArguments _TypeArgumentsopt4;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;
        private ClassBody _ClassBodyopt;

        public AstToken getnew() { return _new; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public ClassType getClassOrInterfaceType() { return _ClassOrInterfaceType; }
        /**
         * The value returned by <b>getTypeArgumentsopt4</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt4() { return _TypeArgumentsopt4; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        /**
         * The value returned by <b>getClassBodyopt</b> may be <b>null</b>
         */
        public ClassBody getClassBodyopt() { return _ClassBodyopt; }

        public ClassInstanceCreationExpression0(IToken leftIToken, IToken rightIToken,
                                                AstToken _new,
                                                TypeArguments _TypeArgumentsopt,
                                                ClassType _ClassOrInterfaceType,
                                                TypeArguments _TypeArgumentsopt4,
                                                AstToken _LPAREN,
                                                IArgumentListopt _ArgumentListopt,
                                                AstToken _RPAREN,
                                                ClassBody _ClassBodyopt)
        {
            super(leftIToken, rightIToken);

            this._new = _new;
            ((Ast) _new).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._ClassOrInterfaceType = _ClassOrInterfaceType;
            ((Ast) _ClassOrInterfaceType).setParent(this);
            this._TypeArgumentsopt4 = _TypeArgumentsopt4;
            if (_TypeArgumentsopt4 != null) ((Ast) _TypeArgumentsopt4).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._ClassBodyopt = _ClassBodyopt;
            if (_ClassBodyopt != null) ((Ast) _ClassBodyopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_new);
            list.add(_TypeArgumentsopt);
            list.add(_ClassOrInterfaceType);
            list.add(_TypeArgumentsopt4);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            list.add(_ClassBodyopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassInstanceCreationExpression0)) return false;
            if (! super.equals(o)) return false;
            ClassInstanceCreationExpression0 other = (ClassInstanceCreationExpression0) o;
            if (! _new.equals(other._new)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _ClassOrInterfaceType.equals(other._ClassOrInterfaceType)) return false;
            if (_TypeArgumentsopt4 == null)
                if (other._TypeArgumentsopt4 != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt4.equals(other._TypeArgumentsopt4)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (_ClassBodyopt == null)
                if (other._ClassBodyopt != null) return false;
                else; // continue
            else if (! _ClassBodyopt.equals(other._ClassBodyopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_ClassOrInterfaceType.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt4 == null ? 0 : _TypeArgumentsopt4.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_ClassBodyopt == null ? 0 : _ClassBodyopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _new.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _ClassOrInterfaceType.accept(v);
                if (_TypeArgumentsopt4 != null) _TypeArgumentsopt4.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
                if (_ClassBodyopt != null) _ClassBodyopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 355:  ClassInstanceCreationExpression ::= Primary . new TypeArgumentsopt identifier TypeArgumentsopt ( ArgumentListopt ) ClassBodyopt
     *</b>
     */
    static public class ClassInstanceCreationExpression1 extends Ast implements IClassInstanceCreationExpression
    {
        private IPrimary _Primary;
        private AstToken _DOT;
        private AstToken _new;
        private TypeArguments _TypeArgumentsopt;
        private identifier _identifier;
        private TypeArguments _TypeArgumentsopt6;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;
        private ClassBody _ClassBodyopt;

        public IPrimary getPrimary() { return _Primary; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getnew() { return _new; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public identifier getidentifier() { return _identifier; }
        /**
         * The value returned by <b>getTypeArgumentsopt6</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt6() { return _TypeArgumentsopt6; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        /**
         * The value returned by <b>getClassBodyopt</b> may be <b>null</b>
         */
        public ClassBody getClassBodyopt() { return _ClassBodyopt; }

        public ClassInstanceCreationExpression1(IToken leftIToken, IToken rightIToken,
                                                IPrimary _Primary,
                                                AstToken _DOT,
                                                AstToken _new,
                                                TypeArguments _TypeArgumentsopt,
                                                identifier _identifier,
                                                TypeArguments _TypeArgumentsopt6,
                                                AstToken _LPAREN,
                                                IArgumentListopt _ArgumentListopt,
                                                AstToken _RPAREN,
                                                ClassBody _ClassBodyopt)
        {
            super(leftIToken, rightIToken);

            this._Primary = _Primary;
            ((Ast) _Primary).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._new = _new;
            ((Ast) _new).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._TypeArgumentsopt6 = _TypeArgumentsopt6;
            if (_TypeArgumentsopt6 != null) ((Ast) _TypeArgumentsopt6).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._ClassBodyopt = _ClassBodyopt;
            if (_ClassBodyopt != null) ((Ast) _ClassBodyopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Primary);
            list.add(_DOT);
            list.add(_new);
            list.add(_TypeArgumentsopt);
            list.add(_identifier);
            list.add(_TypeArgumentsopt6);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            list.add(_ClassBodyopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ClassInstanceCreationExpression1)) return false;
            if (! super.equals(o)) return false;
            ClassInstanceCreationExpression1 other = (ClassInstanceCreationExpression1) o;
            if (! _Primary.equals(other._Primary)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _new.equals(other._new)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (_TypeArgumentsopt6 == null)
                if (other._TypeArgumentsopt6 != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt6.equals(other._TypeArgumentsopt6)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (_ClassBodyopt == null)
                if (other._ClassBodyopt != null) return false;
                else; // continue
            else if (! _ClassBodyopt.equals(other._ClassBodyopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Primary.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt6 == null ? 0 : _TypeArgumentsopt6.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_ClassBodyopt == null ? 0 : _ClassBodyopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Primary.accept(v);
                _DOT.accept(v);
                _new.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _identifier.accept(v);
                if (_TypeArgumentsopt6 != null) _TypeArgumentsopt6.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
                if (_ClassBodyopt != null) _ClassBodyopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 358:  ArrayCreationExpression ::= new PrimitiveType DimExprs Dimsopt
     *</b>
     */
    static public class ArrayCreationExpression0 extends Ast implements IArrayCreationExpression
    {
        private AstToken _new;
        private IPrimitiveType _PrimitiveType;
        private IDimExprs _DimExprs;
        private IDimsopt _Dimsopt;

        public AstToken getnew() { return _new; }
        public IPrimitiveType getPrimitiveType() { return _PrimitiveType; }
        public IDimExprs getDimExprs() { return _DimExprs; }
        /**
         * The value returned by <b>getDimsopt</b> may be <b>null</b>
         */
        public IDimsopt getDimsopt() { return _Dimsopt; }

        public ArrayCreationExpression0(IToken leftIToken, IToken rightIToken,
                                        AstToken _new,
                                        IPrimitiveType _PrimitiveType,
                                        IDimExprs _DimExprs,
                                        IDimsopt _Dimsopt)
        {
            super(leftIToken, rightIToken);

            this._new = _new;
            ((Ast) _new).setParent(this);
            this._PrimitiveType = _PrimitiveType;
            ((Ast) _PrimitiveType).setParent(this);
            this._DimExprs = _DimExprs;
            ((Ast) _DimExprs).setParent(this);
            this._Dimsopt = _Dimsopt;
            if (_Dimsopt != null) ((Ast) _Dimsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_new);
            list.add(_PrimitiveType);
            list.add(_DimExprs);
            list.add(_Dimsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayCreationExpression0)) return false;
            if (! super.equals(o)) return false;
            ArrayCreationExpression0 other = (ArrayCreationExpression0) o;
            if (! _new.equals(other._new)) return false;
            if (! _PrimitiveType.equals(other._PrimitiveType)) return false;
            if (! _DimExprs.equals(other._DimExprs)) return false;
            if (_Dimsopt == null)
                if (other._Dimsopt != null) return false;
                else; // continue
            else if (! _Dimsopt.equals(other._Dimsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_PrimitiveType.hashCode());
            hash = hash * 31 + (_DimExprs.hashCode());
            hash = hash * 31 + (_Dimsopt == null ? 0 : _Dimsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _new.accept(v);
                _PrimitiveType.accept(v);
                _DimExprs.accept(v);
                if (_Dimsopt != null) _Dimsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 359:  ArrayCreationExpression ::= new ClassOrInterfaceType DimExprs Dimsopt
     *</b>
     */
    static public class ArrayCreationExpression1 extends Ast implements IArrayCreationExpression
    {
        private AstToken _new;
        private ClassType _ClassOrInterfaceType;
        private IDimExprs _DimExprs;
        private IDimsopt _Dimsopt;

        public AstToken getnew() { return _new; }
        public ClassType getClassOrInterfaceType() { return _ClassOrInterfaceType; }
        public IDimExprs getDimExprs() { return _DimExprs; }
        /**
         * The value returned by <b>getDimsopt</b> may be <b>null</b>
         */
        public IDimsopt getDimsopt() { return _Dimsopt; }

        public ArrayCreationExpression1(IToken leftIToken, IToken rightIToken,
                                        AstToken _new,
                                        ClassType _ClassOrInterfaceType,
                                        IDimExprs _DimExprs,
                                        IDimsopt _Dimsopt)
        {
            super(leftIToken, rightIToken);

            this._new = _new;
            ((Ast) _new).setParent(this);
            this._ClassOrInterfaceType = _ClassOrInterfaceType;
            ((Ast) _ClassOrInterfaceType).setParent(this);
            this._DimExprs = _DimExprs;
            ((Ast) _DimExprs).setParent(this);
            this._Dimsopt = _Dimsopt;
            if (_Dimsopt != null) ((Ast) _Dimsopt).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_new);
            list.add(_ClassOrInterfaceType);
            list.add(_DimExprs);
            list.add(_Dimsopt);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayCreationExpression1)) return false;
            if (! super.equals(o)) return false;
            ArrayCreationExpression1 other = (ArrayCreationExpression1) o;
            if (! _new.equals(other._new)) return false;
            if (! _ClassOrInterfaceType.equals(other._ClassOrInterfaceType)) return false;
            if (! _DimExprs.equals(other._DimExprs)) return false;
            if (_Dimsopt == null)
                if (other._Dimsopt != null) return false;
                else; // continue
            else if (! _Dimsopt.equals(other._Dimsopt)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_ClassOrInterfaceType.hashCode());
            hash = hash * 31 + (_DimExprs.hashCode());
            hash = hash * 31 + (_Dimsopt == null ? 0 : _Dimsopt.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _new.accept(v);
                _ClassOrInterfaceType.accept(v);
                _DimExprs.accept(v);
                if (_Dimsopt != null) _Dimsopt.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 360:  ArrayCreationExpression ::= new PrimitiveType Dims ArrayInitializer
     *</b>
     */
    static public class ArrayCreationExpression2 extends Ast implements IArrayCreationExpression
    {
        private AstToken _new;
        private IPrimitiveType _PrimitiveType;
        private IDims _Dims;
        private ArrayInitializer _ArrayInitializer;

        public AstToken getnew() { return _new; }
        public IPrimitiveType getPrimitiveType() { return _PrimitiveType; }
        public IDims getDims() { return _Dims; }
        public ArrayInitializer getArrayInitializer() { return _ArrayInitializer; }

        public ArrayCreationExpression2(IToken leftIToken, IToken rightIToken,
                                        AstToken _new,
                                        IPrimitiveType _PrimitiveType,
                                        IDims _Dims,
                                        ArrayInitializer _ArrayInitializer)
        {
            super(leftIToken, rightIToken);

            this._new = _new;
            ((Ast) _new).setParent(this);
            this._PrimitiveType = _PrimitiveType;
            ((Ast) _PrimitiveType).setParent(this);
            this._Dims = _Dims;
            ((Ast) _Dims).setParent(this);
            this._ArrayInitializer = _ArrayInitializer;
            ((Ast) _ArrayInitializer).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_new);
            list.add(_PrimitiveType);
            list.add(_Dims);
            list.add(_ArrayInitializer);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayCreationExpression2)) return false;
            if (! super.equals(o)) return false;
            ArrayCreationExpression2 other = (ArrayCreationExpression2) o;
            if (! _new.equals(other._new)) return false;
            if (! _PrimitiveType.equals(other._PrimitiveType)) return false;
            if (! _Dims.equals(other._Dims)) return false;
            if (! _ArrayInitializer.equals(other._ArrayInitializer)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_PrimitiveType.hashCode());
            hash = hash * 31 + (_Dims.hashCode());
            hash = hash * 31 + (_ArrayInitializer.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _new.accept(v);
                _PrimitiveType.accept(v);
                _Dims.accept(v);
                _ArrayInitializer.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 361:  ArrayCreationExpression ::= new ClassOrInterfaceType Dims ArrayInitializer
     *</b>
     */
    static public class ArrayCreationExpression3 extends Ast implements IArrayCreationExpression
    {
        private AstToken _new;
        private ClassType _ClassOrInterfaceType;
        private IDims _Dims;
        private ArrayInitializer _ArrayInitializer;

        public AstToken getnew() { return _new; }
        public ClassType getClassOrInterfaceType() { return _ClassOrInterfaceType; }
        public IDims getDims() { return _Dims; }
        public ArrayInitializer getArrayInitializer() { return _ArrayInitializer; }

        public ArrayCreationExpression3(IToken leftIToken, IToken rightIToken,
                                        AstToken _new,
                                        ClassType _ClassOrInterfaceType,
                                        IDims _Dims,
                                        ArrayInitializer _ArrayInitializer)
        {
            super(leftIToken, rightIToken);

            this._new = _new;
            ((Ast) _new).setParent(this);
            this._ClassOrInterfaceType = _ClassOrInterfaceType;
            ((Ast) _ClassOrInterfaceType).setParent(this);
            this._Dims = _Dims;
            ((Ast) _Dims).setParent(this);
            this._ArrayInitializer = _ArrayInitializer;
            ((Ast) _ArrayInitializer).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_new);
            list.add(_ClassOrInterfaceType);
            list.add(_Dims);
            list.add(_ArrayInitializer);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayCreationExpression3)) return false;
            if (! super.equals(o)) return false;
            ArrayCreationExpression3 other = (ArrayCreationExpression3) o;
            if (! _new.equals(other._new)) return false;
            if (! _ClassOrInterfaceType.equals(other._ClassOrInterfaceType)) return false;
            if (! _Dims.equals(other._Dims)) return false;
            if (! _ArrayInitializer.equals(other._ArrayInitializer)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_new.hashCode());
            hash = hash * 31 + (_ClassOrInterfaceType.hashCode());
            hash = hash * 31 + (_Dims.hashCode());
            hash = hash * 31 + (_ArrayInitializer.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _new.accept(v);
                _ClassOrInterfaceType.accept(v);
                _Dims.accept(v);
                _ArrayInitializer.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 365:  Dims ::= [ ]
     *</b>
     */
    static public class Dims0 extends Ast implements IDims
    {
        private AstToken _LBRACKET;
        private AstToken _RBRACKET;

        public AstToken getLBRACKET() { return _LBRACKET; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public Dims0(IToken leftIToken, IToken rightIToken,
                     AstToken _LBRACKET,
                     AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LBRACKET);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Dims0)) return false;
            if (! super.equals(o)) return false;
            Dims0 other = (Dims0) o;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LBRACKET.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 366:  Dims ::= Dims [ ]
     *</b>
     */
    static public class Dims1 extends Ast implements IDims
    {
        private IDims _Dims;
        private AstToken _LBRACKET;
        private AstToken _RBRACKET;

        public IDims getDims() { return _Dims; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public Dims1(IToken leftIToken, IToken rightIToken,
                     IDims _Dims,
                     AstToken _LBRACKET,
                     AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._Dims = _Dims;
            ((Ast) _Dims).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Dims);
            list.add(_LBRACKET);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof Dims1)) return false;
            if (! super.equals(o)) return false;
            Dims1 other = (Dims1) o;
            if (! _Dims.equals(other._Dims)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Dims.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Dims.accept(v);
                _LBRACKET.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 367:  FieldAccess ::= Primary . identifier
     *</b>
     */
    static public class FieldAccess0 extends Ast implements IFieldAccess
    {
        private IPrimary _Primary;
        private AstToken _DOT;
        private identifier _identifier;

        public IPrimary getPrimary() { return _Primary; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public FieldAccess0(IToken leftIToken, IToken rightIToken,
                            IPrimary _Primary,
                            AstToken _DOT,
                            identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._Primary = _Primary;
            ((Ast) _Primary).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Primary);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FieldAccess0)) return false;
            if (! super.equals(o)) return false;
            FieldAccess0 other = (FieldAccess0) o;
            if (! _Primary.equals(other._Primary)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Primary.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Primary.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 368:  FieldAccess ::= super . identifier
     *</b>
     */
    static public class FieldAccess1 extends Ast implements IFieldAccess
    {
        private AstToken _super;
        private AstToken _DOT;
        private identifier _identifier;

        public AstToken getsuper() { return _super; }
        public AstToken getDOT() { return _DOT; }
        public identifier getidentifier() { return _identifier; }

        public FieldAccess1(IToken leftIToken, IToken rightIToken,
                            AstToken _super,
                            AstToken _DOT,
                            identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._super = _super;
            ((Ast) _super).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_super);
            list.add(_DOT);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FieldAccess1)) return false;
            if (! super.equals(o)) return false;
            FieldAccess1 other = (FieldAccess1) o;
            if (! _super.equals(other._super)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _super.accept(v);
                _DOT.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 369:  FieldAccess ::= ClassName . super . identifier
     *</b>
     */
    static public class FieldAccess2 extends Ast implements IFieldAccess
    {
        private IClassName _ClassName;
        private AstToken _DOT;
        private AstToken _super;
        private AstToken _DOT4;
        private identifier _identifier;

        public IClassName getClassName() { return _ClassName; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getsuper() { return _super; }
        public AstToken getDOT4() { return _DOT4; }
        public identifier getidentifier() { return _identifier; }

        public FieldAccess2(IToken leftIToken, IToken rightIToken,
                            IClassName _ClassName,
                            AstToken _DOT,
                            AstToken _super,
                            AstToken _DOT4,
                            identifier _identifier)
        {
            super(leftIToken, rightIToken);

            this._ClassName = _ClassName;
            ((Ast) _ClassName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._super = _super;
            ((Ast) _super).setParent(this);
            this._DOT4 = _DOT4;
            ((Ast) _DOT4).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassName);
            list.add(_DOT);
            list.add(_super);
            list.add(_DOT4);
            list.add(_identifier);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof FieldAccess2)) return false;
            if (! super.equals(o)) return false;
            FieldAccess2 other = (FieldAccess2) o;
            if (! _ClassName.equals(other._ClassName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _super.equals(other._super)) return false;
            if (! _DOT4.equals(other._DOT4)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_DOT4.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ClassName.accept(v);
                _DOT.accept(v);
                _super.accept(v);
                _DOT4.accept(v);
                _identifier.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 370:  MethodInvocation ::= MethodName ( ArgumentListopt )
     *</b>
     */
    static public class MethodInvocation0 extends Ast implements IMethodInvocation
    {
        private IMethodName _MethodName;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public IMethodName getMethodName() { return _MethodName; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodInvocation0(IToken leftIToken, IToken rightIToken,
                                 IMethodName _MethodName,
                                 AstToken _LPAREN,
                                 IArgumentListopt _ArgumentListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._MethodName = _MethodName;
            ((Ast) _MethodName).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MethodName);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodInvocation0)) return false;
            if (! super.equals(o)) return false;
            MethodInvocation0 other = (MethodInvocation0) o;
            if (! _MethodName.equals(other._MethodName)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MethodName.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MethodName.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 371:  MethodInvocation ::= Primary . TypeArgumentsopt identifier ( ArgumentListopt )
     *</b>
     */
    static public class MethodInvocation1 extends Ast implements IMethodInvocation
    {
        private IPrimary _Primary;
        private AstToken _DOT;
        private TypeArguments _TypeArgumentsopt;
        private identifier _identifier;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public IPrimary getPrimary() { return _Primary; }
        public AstToken getDOT() { return _DOT; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodInvocation1(IToken leftIToken, IToken rightIToken,
                                 IPrimary _Primary,
                                 AstToken _DOT,
                                 TypeArguments _TypeArgumentsopt,
                                 identifier _identifier,
                                 AstToken _LPAREN,
                                 IArgumentListopt _ArgumentListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._Primary = _Primary;
            ((Ast) _Primary).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_Primary);
            list.add(_DOT);
            list.add(_TypeArgumentsopt);
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodInvocation1)) return false;
            if (! super.equals(o)) return false;
            MethodInvocation1 other = (MethodInvocation1) o;
            if (! _Primary.equals(other._Primary)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_Primary.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _Primary.accept(v);
                _DOT.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _identifier.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 372:  MethodInvocation ::= super . TypeArgumentsopt identifier ( ArgumentListopt )
     *</b>
     */
    static public class MethodInvocation2 extends Ast implements IMethodInvocation
    {
        private AstToken _super;
        private AstToken _DOT;
        private TypeArguments _TypeArgumentsopt;
        private identifier _identifier;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public AstToken getsuper() { return _super; }
        public AstToken getDOT() { return _DOT; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodInvocation2(IToken leftIToken, IToken rightIToken,
                                 AstToken _super,
                                 AstToken _DOT,
                                 TypeArguments _TypeArgumentsopt,
                                 identifier _identifier,
                                 AstToken _LPAREN,
                                 IArgumentListopt _ArgumentListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._super = _super;
            ((Ast) _super).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_super);
            list.add(_DOT);
            list.add(_TypeArgumentsopt);
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodInvocation2)) return false;
            if (! super.equals(o)) return false;
            MethodInvocation2 other = (MethodInvocation2) o;
            if (! _super.equals(other._super)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _super.accept(v);
                _DOT.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _identifier.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 373:  MethodInvocation ::= ClassName . super . TypeArgumentsopt identifier ( ArgumentListopt )
     *</b>
     */
    static public class MethodInvocation3 extends Ast implements IMethodInvocation
    {
        private IClassName _ClassName;
        private AstToken _DOT;
        private AstToken _super;
        private AstToken _DOT4;
        private TypeArguments _TypeArgumentsopt;
        private identifier _identifier;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public IClassName getClassName() { return _ClassName; }
        public AstToken getDOT() { return _DOT; }
        public AstToken getsuper() { return _super; }
        public AstToken getDOT4() { return _DOT4; }
        /**
         * The value returned by <b>getTypeArgumentsopt</b> may be <b>null</b>
         */
        public TypeArguments getTypeArgumentsopt() { return _TypeArgumentsopt; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodInvocation3(IToken leftIToken, IToken rightIToken,
                                 IClassName _ClassName,
                                 AstToken _DOT,
                                 AstToken _super,
                                 AstToken _DOT4,
                                 TypeArguments _TypeArgumentsopt,
                                 identifier _identifier,
                                 AstToken _LPAREN,
                                 IArgumentListopt _ArgumentListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._ClassName = _ClassName;
            ((Ast) _ClassName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._super = _super;
            ((Ast) _super).setParent(this);
            this._DOT4 = _DOT4;
            ((Ast) _DOT4).setParent(this);
            this._TypeArgumentsopt = _TypeArgumentsopt;
            if (_TypeArgumentsopt != null) ((Ast) _TypeArgumentsopt).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ClassName);
            list.add(_DOT);
            list.add(_super);
            list.add(_DOT4);
            list.add(_TypeArgumentsopt);
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodInvocation3)) return false;
            if (! super.equals(o)) return false;
            MethodInvocation3 other = (MethodInvocation3) o;
            if (! _ClassName.equals(other._ClassName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _super.equals(other._super)) return false;
            if (! _DOT4.equals(other._DOT4)) return false;
            if (_TypeArgumentsopt == null)
                if (other._TypeArgumentsopt != null) return false;
                else; // continue
            else if (! _TypeArgumentsopt.equals(other._TypeArgumentsopt)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ClassName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_super.hashCode());
            hash = hash * 31 + (_DOT4.hashCode());
            hash = hash * 31 + (_TypeArgumentsopt == null ? 0 : _TypeArgumentsopt.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ClassName.accept(v);
                _DOT.accept(v);
                _super.accept(v);
                _DOT4.accept(v);
                if (_TypeArgumentsopt != null) _TypeArgumentsopt.accept(v);
                _identifier.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 374:  MethodInvocation ::= TypeName . TypeArguments identifier ( ArgumentListopt )
     *</b>
     */
    static public class MethodInvocation4 extends Ast implements IMethodInvocation
    {
        private ITypeName _TypeName;
        private AstToken _DOT;
        private TypeArguments _TypeArguments;
        private identifier _identifier;
        private AstToken _LPAREN;
        private IArgumentListopt _ArgumentListopt;
        private AstToken _RPAREN;

        public ITypeName getTypeName() { return _TypeName; }
        public AstToken getDOT() { return _DOT; }
        public TypeArguments getTypeArguments() { return _TypeArguments; }
        public identifier getidentifier() { return _identifier; }
        public AstToken getLPAREN() { return _LPAREN; }
        /**
         * The value returned by <b>getArgumentListopt</b> may be <b>null</b>
         */
        public IArgumentListopt getArgumentListopt() { return _ArgumentListopt; }
        public AstToken getRPAREN() { return _RPAREN; }

        public MethodInvocation4(IToken leftIToken, IToken rightIToken,
                                 ITypeName _TypeName,
                                 AstToken _DOT,
                                 TypeArguments _TypeArguments,
                                 identifier _identifier,
                                 AstToken _LPAREN,
                                 IArgumentListopt _ArgumentListopt,
                                 AstToken _RPAREN)
        {
            super(leftIToken, rightIToken);

            this._TypeName = _TypeName;
            ((Ast) _TypeName).setParent(this);
            this._DOT = _DOT;
            ((Ast) _DOT).setParent(this);
            this._TypeArguments = _TypeArguments;
            ((Ast) _TypeArguments).setParent(this);
            this._identifier = _identifier;
            ((Ast) _identifier).setParent(this);
            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ArgumentListopt = _ArgumentListopt;
            if (_ArgumentListopt != null) ((Ast) _ArgumentListopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TypeName);
            list.add(_DOT);
            list.add(_TypeArguments);
            list.add(_identifier);
            list.add(_LPAREN);
            list.add(_ArgumentListopt);
            list.add(_RPAREN);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MethodInvocation4)) return false;
            if (! super.equals(o)) return false;
            MethodInvocation4 other = (MethodInvocation4) o;
            if (! _TypeName.equals(other._TypeName)) return false;
            if (! _DOT.equals(other._DOT)) return false;
            if (! _TypeArguments.equals(other._TypeArguments)) return false;
            if (! _identifier.equals(other._identifier)) return false;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (_ArgumentListopt == null)
                if (other._ArgumentListopt != null) return false;
                else; // continue
            else if (! _ArgumentListopt.equals(other._ArgumentListopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TypeName.hashCode());
            hash = hash * 31 + (_DOT.hashCode());
            hash = hash * 31 + (_TypeArguments.hashCode());
            hash = hash * 31 + (_identifier.hashCode());
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ArgumentListopt == null ? 0 : _ArgumentListopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TypeName.accept(v);
                _DOT.accept(v);
                _TypeArguments.accept(v);
                _identifier.accept(v);
                _LPAREN.accept(v);
                if (_ArgumentListopt != null) _ArgumentListopt.accept(v);
                _RPAREN.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 375:  ArrayAccess ::= ExpressionName [ Expression ]
     *</b>
     */
    static public class ArrayAccess0 extends Ast implements IArrayAccess
    {
        private IExpressionName _ExpressionName;
        private AstToken _LBRACKET;
        private IExpression _Expression;
        private AstToken _RBRACKET;

        public IExpressionName getExpressionName() { return _ExpressionName; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public ArrayAccess0(IToken leftIToken, IToken rightIToken,
                            IExpressionName _ExpressionName,
                            AstToken _LBRACKET,
                            IExpression _Expression,
                            AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._ExpressionName = _ExpressionName;
            ((Ast) _ExpressionName).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ExpressionName);
            list.add(_LBRACKET);
            list.add(_Expression);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayAccess0)) return false;
            if (! super.equals(o)) return false;
            ArrayAccess0 other = (ArrayAccess0) o;
            if (! _ExpressionName.equals(other._ExpressionName)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ExpressionName.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ExpressionName.accept(v);
                _LBRACKET.accept(v);
                _Expression.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 376:  ArrayAccess ::= PrimaryNoNewArray [ Expression ]
     *</b>
     */
    static public class ArrayAccess1 extends Ast implements IArrayAccess
    {
        private IPrimaryNoNewArray _PrimaryNoNewArray;
        private AstToken _LBRACKET;
        private IExpression _Expression;
        private AstToken _RBRACKET;

        public IPrimaryNoNewArray getPrimaryNoNewArray() { return _PrimaryNoNewArray; }
        public AstToken getLBRACKET() { return _LBRACKET; }
        public IExpression getExpression() { return _Expression; }
        public AstToken getRBRACKET() { return _RBRACKET; }

        public ArrayAccess1(IToken leftIToken, IToken rightIToken,
                            IPrimaryNoNewArray _PrimaryNoNewArray,
                            AstToken _LBRACKET,
                            IExpression _Expression,
                            AstToken _RBRACKET)
        {
            super(leftIToken, rightIToken);

            this._PrimaryNoNewArray = _PrimaryNoNewArray;
            ((Ast) _PrimaryNoNewArray).setParent(this);
            this._LBRACKET = _LBRACKET;
            ((Ast) _LBRACKET).setParent(this);
            this._Expression = _Expression;
            ((Ast) _Expression).setParent(this);
            this._RBRACKET = _RBRACKET;
            ((Ast) _RBRACKET).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PrimaryNoNewArray);
            list.add(_LBRACKET);
            list.add(_Expression);
            list.add(_RBRACKET);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ArrayAccess1)) return false;
            if (! super.equals(o)) return false;
            ArrayAccess1 other = (ArrayAccess1) o;
            if (! _PrimaryNoNewArray.equals(other._PrimaryNoNewArray)) return false;
            if (! _LBRACKET.equals(other._LBRACKET)) return false;
            if (! _Expression.equals(other._Expression)) return false;
            if (! _RBRACKET.equals(other._RBRACKET)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PrimaryNoNewArray.hashCode());
            hash = hash * 31 + (_LBRACKET.hashCode());
            hash = hash * 31 + (_Expression.hashCode());
            hash = hash * 31 + (_RBRACKET.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PrimaryNoNewArray.accept(v);
                _LBRACKET.accept(v);
                _Expression.accept(v);
                _RBRACKET.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 385:  UnaryExpression ::= + UnaryExpression
     *</b>
     */
    static public class UnaryExpression0 extends Ast implements IUnaryExpression
    {
        private AstToken _PLUS;
        private IUnaryExpression _UnaryExpression;

        public AstToken getPLUS() { return _PLUS; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public UnaryExpression0(IToken leftIToken, IToken rightIToken,
                                AstToken _PLUS,
                                IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._PLUS = _PLUS;
            ((Ast) _PLUS).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_PLUS);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof UnaryExpression0)) return false;
            if (! super.equals(o)) return false;
            UnaryExpression0 other = (UnaryExpression0) o;
            if (! _PLUS.equals(other._PLUS)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_PLUS.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _PLUS.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 386:  UnaryExpression ::= - UnaryExpression
     *</b>
     */
    static public class UnaryExpression1 extends Ast implements IUnaryExpression
    {
        private AstToken _MINUS;
        private IUnaryExpression _UnaryExpression;

        public AstToken getMINUS() { return _MINUS; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public UnaryExpression1(IToken leftIToken, IToken rightIToken,
                                AstToken _MINUS,
                                IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._MINUS = _MINUS;
            ((Ast) _MINUS).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MINUS);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof UnaryExpression1)) return false;
            if (! super.equals(o)) return false;
            UnaryExpression1 other = (UnaryExpression1) o;
            if (! _MINUS.equals(other._MINUS)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MINUS.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MINUS.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 391:  UnaryExpressionNotPlusMinus ::= ~ UnaryExpression
     *</b>
     */
    static public class UnaryExpressionNotPlusMinus0 extends Ast implements IUnaryExpressionNotPlusMinus
    {
        private AstToken _TWIDDLE;
        private IUnaryExpression _UnaryExpression;

        public AstToken getTWIDDLE() { return _TWIDDLE; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public UnaryExpressionNotPlusMinus0(IToken leftIToken, IToken rightIToken,
                                            AstToken _TWIDDLE,
                                            IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._TWIDDLE = _TWIDDLE;
            ((Ast) _TWIDDLE).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_TWIDDLE);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof UnaryExpressionNotPlusMinus0)) return false;
            if (! super.equals(o)) return false;
            UnaryExpressionNotPlusMinus0 other = (UnaryExpressionNotPlusMinus0) o;
            if (! _TWIDDLE.equals(other._TWIDDLE)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_TWIDDLE.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _TWIDDLE.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 392:  UnaryExpressionNotPlusMinus ::= ! UnaryExpression
     *</b>
     */
    static public class UnaryExpressionNotPlusMinus1 extends Ast implements IUnaryExpressionNotPlusMinus
    {
        private AstToken _NOT;
        private IUnaryExpression _UnaryExpression;

        public AstToken getNOT() { return _NOT; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public UnaryExpressionNotPlusMinus1(IToken leftIToken, IToken rightIToken,
                                            AstToken _NOT,
                                            IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._NOT = _NOT;
            ((Ast) _NOT).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_NOT);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof UnaryExpressionNotPlusMinus1)) return false;
            if (! super.equals(o)) return false;
            UnaryExpressionNotPlusMinus1 other = (UnaryExpressionNotPlusMinus1) o;
            if (! _NOT.equals(other._NOT)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_NOT.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _NOT.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 394:  CastExpression ::= ( PrimitiveType Dimsopt ) UnaryExpression
     *</b>
     */
    static public class CastExpression0 extends Ast implements ICastExpression
    {
        private AstToken _LPAREN;
        private IPrimitiveType _PrimitiveType;
        private IDimsopt _Dimsopt;
        private AstToken _RPAREN;
        private IUnaryExpression _UnaryExpression;

        public AstToken getLPAREN() { return _LPAREN; }
        public IPrimitiveType getPrimitiveType() { return _PrimitiveType; }
        /**
         * The value returned by <b>getDimsopt</b> may be <b>null</b>
         */
        public IDimsopt getDimsopt() { return _Dimsopt; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public CastExpression0(IToken leftIToken, IToken rightIToken,
                               AstToken _LPAREN,
                               IPrimitiveType _PrimitiveType,
                               IDimsopt _Dimsopt,
                               AstToken _RPAREN,
                               IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._PrimitiveType = _PrimitiveType;
            ((Ast) _PrimitiveType).setParent(this);
            this._Dimsopt = _Dimsopt;
            if (_Dimsopt != null) ((Ast) _Dimsopt).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LPAREN);
            list.add(_PrimitiveType);
            list.add(_Dimsopt);
            list.add(_RPAREN);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof CastExpression0)) return false;
            if (! super.equals(o)) return false;
            CastExpression0 other = (CastExpression0) o;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _PrimitiveType.equals(other._PrimitiveType)) return false;
            if (_Dimsopt == null)
                if (other._Dimsopt != null) return false;
                else; // continue
            else if (! _Dimsopt.equals(other._Dimsopt)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_PrimitiveType.hashCode());
            hash = hash * 31 + (_Dimsopt == null ? 0 : _Dimsopt.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LPAREN.accept(v);
                _PrimitiveType.accept(v);
                if (_Dimsopt != null) _Dimsopt.accept(v);
                _RPAREN.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 395:  CastExpression ::= ( ReferenceType ) UnaryExpressionNotPlusMinus
     *</b>
     */
    static public class CastExpression1 extends Ast implements ICastExpression
    {
        private AstToken _LPAREN;
        private IReferenceType _ReferenceType;
        private AstToken _RPAREN;
        private IUnaryExpressionNotPlusMinus _UnaryExpressionNotPlusMinus;

        public AstToken getLPAREN() { return _LPAREN; }
        public IReferenceType getReferenceType() { return _ReferenceType; }
        public AstToken getRPAREN() { return _RPAREN; }
        public IUnaryExpressionNotPlusMinus getUnaryExpressionNotPlusMinus() { return _UnaryExpressionNotPlusMinus; }

        public CastExpression1(IToken leftIToken, IToken rightIToken,
                               AstToken _LPAREN,
                               IReferenceType _ReferenceType,
                               AstToken _RPAREN,
                               IUnaryExpressionNotPlusMinus _UnaryExpressionNotPlusMinus)
        {
            super(leftIToken, rightIToken);

            this._LPAREN = _LPAREN;
            ((Ast) _LPAREN).setParent(this);
            this._ReferenceType = _ReferenceType;
            ((Ast) _ReferenceType).setParent(this);
            this._RPAREN = _RPAREN;
            ((Ast) _RPAREN).setParent(this);
            this._UnaryExpressionNotPlusMinus = _UnaryExpressionNotPlusMinus;
            ((Ast) _UnaryExpressionNotPlusMinus).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_LPAREN);
            list.add(_ReferenceType);
            list.add(_RPAREN);
            list.add(_UnaryExpressionNotPlusMinus);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof CastExpression1)) return false;
            if (! super.equals(o)) return false;
            CastExpression1 other = (CastExpression1) o;
            if (! _LPAREN.equals(other._LPAREN)) return false;
            if (! _ReferenceType.equals(other._ReferenceType)) return false;
            if (! _RPAREN.equals(other._RPAREN)) return false;
            if (! _UnaryExpressionNotPlusMinus.equals(other._UnaryExpressionNotPlusMinus)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_LPAREN.hashCode());
            hash = hash * 31 + (_ReferenceType.hashCode());
            hash = hash * 31 + (_RPAREN.hashCode());
            hash = hash * 31 + (_UnaryExpressionNotPlusMinus.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _LPAREN.accept(v);
                _ReferenceType.accept(v);
                _RPAREN.accept(v);
                _UnaryExpressionNotPlusMinus.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 397:  MultiplicativeExpression ::= MultiplicativeExpression * UnaryExpression
     *</b>
     */
    static public class MultiplicativeExpression0 extends Ast implements IMultiplicativeExpression
    {
        private IMultiplicativeExpression _MultiplicativeExpression;
        private AstToken _MULTIPLY;
        private IUnaryExpression _UnaryExpression;

        public IMultiplicativeExpression getMultiplicativeExpression() { return _MultiplicativeExpression; }
        public AstToken getMULTIPLY() { return _MULTIPLY; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public MultiplicativeExpression0(IToken leftIToken, IToken rightIToken,
                                         IMultiplicativeExpression _MultiplicativeExpression,
                                         AstToken _MULTIPLY,
                                         IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._MultiplicativeExpression = _MultiplicativeExpression;
            ((Ast) _MultiplicativeExpression).setParent(this);
            this._MULTIPLY = _MULTIPLY;
            ((Ast) _MULTIPLY).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MultiplicativeExpression);
            list.add(_MULTIPLY);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MultiplicativeExpression0)) return false;
            if (! super.equals(o)) return false;
            MultiplicativeExpression0 other = (MultiplicativeExpression0) o;
            if (! _MultiplicativeExpression.equals(other._MultiplicativeExpression)) return false;
            if (! _MULTIPLY.equals(other._MULTIPLY)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MultiplicativeExpression.hashCode());
            hash = hash * 31 + (_MULTIPLY.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MultiplicativeExpression.accept(v);
                _MULTIPLY.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 398:  MultiplicativeExpression ::= MultiplicativeExpression / UnaryExpression
     *</b>
     */
    static public class MultiplicativeExpression1 extends Ast implements IMultiplicativeExpression
    {
        private IMultiplicativeExpression _MultiplicativeExpression;
        private AstToken _DIVIDE;
        private IUnaryExpression _UnaryExpression;

        public IMultiplicativeExpression getMultiplicativeExpression() { return _MultiplicativeExpression; }
        public AstToken getDIVIDE() { return _DIVIDE; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public MultiplicativeExpression1(IToken leftIToken, IToken rightIToken,
                                         IMultiplicativeExpression _MultiplicativeExpression,
                                         AstToken _DIVIDE,
                                         IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._MultiplicativeExpression = _MultiplicativeExpression;
            ((Ast) _MultiplicativeExpression).setParent(this);
            this._DIVIDE = _DIVIDE;
            ((Ast) _DIVIDE).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MultiplicativeExpression);
            list.add(_DIVIDE);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MultiplicativeExpression1)) return false;
            if (! super.equals(o)) return false;
            MultiplicativeExpression1 other = (MultiplicativeExpression1) o;
            if (! _MultiplicativeExpression.equals(other._MultiplicativeExpression)) return false;
            if (! _DIVIDE.equals(other._DIVIDE)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MultiplicativeExpression.hashCode());
            hash = hash * 31 + (_DIVIDE.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MultiplicativeExpression.accept(v);
                _DIVIDE.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 399:  MultiplicativeExpression ::= MultiplicativeExpression % UnaryExpression
     *</b>
     */
    static public class MultiplicativeExpression2 extends Ast implements IMultiplicativeExpression
    {
        private IMultiplicativeExpression _MultiplicativeExpression;
        private AstToken _REMAINDER;
        private IUnaryExpression _UnaryExpression;

        public IMultiplicativeExpression getMultiplicativeExpression() { return _MultiplicativeExpression; }
        public AstToken getREMAINDER() { return _REMAINDER; }
        public IUnaryExpression getUnaryExpression() { return _UnaryExpression; }

        public MultiplicativeExpression2(IToken leftIToken, IToken rightIToken,
                                         IMultiplicativeExpression _MultiplicativeExpression,
                                         AstToken _REMAINDER,
                                         IUnaryExpression _UnaryExpression)
        {
            super(leftIToken, rightIToken);

            this._MultiplicativeExpression = _MultiplicativeExpression;
            ((Ast) _MultiplicativeExpression).setParent(this);
            this._REMAINDER = _REMAINDER;
            ((Ast) _REMAINDER).setParent(this);
            this._UnaryExpression = _UnaryExpression;
            ((Ast) _UnaryExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_MultiplicativeExpression);
            list.add(_REMAINDER);
            list.add(_UnaryExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof MultiplicativeExpression2)) return false;
            if (! super.equals(o)) return false;
            MultiplicativeExpression2 other = (MultiplicativeExpression2) o;
            if (! _MultiplicativeExpression.equals(other._MultiplicativeExpression)) return false;
            if (! _REMAINDER.equals(other._REMAINDER)) return false;
            if (! _UnaryExpression.equals(other._UnaryExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_MultiplicativeExpression.hashCode());
            hash = hash * 31 + (_REMAINDER.hashCode());
            hash = hash * 31 + (_UnaryExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _MultiplicativeExpression.accept(v);
                _REMAINDER.accept(v);
                _UnaryExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 401:  AdditiveExpression ::= AdditiveExpression + MultiplicativeExpression
     *</b>
     */
    static public class AdditiveExpression0 extends Ast implements IAdditiveExpression
    {
        private IAdditiveExpression _AdditiveExpression;
        private AstToken _PLUS;
        private IMultiplicativeExpression _MultiplicativeExpression;

        public IAdditiveExpression getAdditiveExpression() { return _AdditiveExpression; }
        public AstToken getPLUS() { return _PLUS; }
        public IMultiplicativeExpression getMultiplicativeExpression() { return _MultiplicativeExpression; }

        public AdditiveExpression0(IToken leftIToken, IToken rightIToken,
                                   IAdditiveExpression _AdditiveExpression,
                                   AstToken _PLUS,
                                   IMultiplicativeExpression _MultiplicativeExpression)
        {
            super(leftIToken, rightIToken);

            this._AdditiveExpression = _AdditiveExpression;
            ((Ast) _AdditiveExpression).setParent(this);
            this._PLUS = _PLUS;
            ((Ast) _PLUS).setParent(this);
            this._MultiplicativeExpression = _MultiplicativeExpression;
            ((Ast) _MultiplicativeExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AdditiveExpression);
            list.add(_PLUS);
            list.add(_MultiplicativeExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AdditiveExpression0)) return false;
            if (! super.equals(o)) return false;
            AdditiveExpression0 other = (AdditiveExpression0) o;
            if (! _AdditiveExpression.equals(other._AdditiveExpression)) return false;
            if (! _PLUS.equals(other._PLUS)) return false;
            if (! _MultiplicativeExpression.equals(other._MultiplicativeExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AdditiveExpression.hashCode());
            hash = hash * 31 + (_PLUS.hashCode());
            hash = hash * 31 + (_MultiplicativeExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AdditiveExpression.accept(v);
                _PLUS.accept(v);
                _MultiplicativeExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 402:  AdditiveExpression ::= AdditiveExpression - MultiplicativeExpression
     *</b>
     */
    static public class AdditiveExpression1 extends Ast implements IAdditiveExpression
    {
        private IAdditiveExpression _AdditiveExpression;
        private AstToken _MINUS;
        private IMultiplicativeExpression _MultiplicativeExpression;

        public IAdditiveExpression getAdditiveExpression() { return _AdditiveExpression; }
        public AstToken getMINUS() { return _MINUS; }
        public IMultiplicativeExpression getMultiplicativeExpression() { return _MultiplicativeExpression; }

        public AdditiveExpression1(IToken leftIToken, IToken rightIToken,
                                   IAdditiveExpression _AdditiveExpression,
                                   AstToken _MINUS,
                                   IMultiplicativeExpression _MultiplicativeExpression)
        {
            super(leftIToken, rightIToken);

            this._AdditiveExpression = _AdditiveExpression;
            ((Ast) _AdditiveExpression).setParent(this);
            this._MINUS = _MINUS;
            ((Ast) _MINUS).setParent(this);
            this._MultiplicativeExpression = _MultiplicativeExpression;
            ((Ast) _MultiplicativeExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_AdditiveExpression);
            list.add(_MINUS);
            list.add(_MultiplicativeExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AdditiveExpression1)) return false;
            if (! super.equals(o)) return false;
            AdditiveExpression1 other = (AdditiveExpression1) o;
            if (! _AdditiveExpression.equals(other._AdditiveExpression)) return false;
            if (! _MINUS.equals(other._MINUS)) return false;
            if (! _MultiplicativeExpression.equals(other._MultiplicativeExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_AdditiveExpression.hashCode());
            hash = hash * 31 + (_MINUS.hashCode());
            hash = hash * 31 + (_MultiplicativeExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _AdditiveExpression.accept(v);
                _MINUS.accept(v);
                _MultiplicativeExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 404:  ShiftExpression ::= ShiftExpression << AdditiveExpression
     *</b>
     */
    static public class ShiftExpression0 extends Ast implements IShiftExpression
    {
        private IShiftExpression _ShiftExpression;
        private AstToken _LEFT_SHIFT;
        private IAdditiveExpression _AdditiveExpression;

        public IShiftExpression getShiftExpression() { return _ShiftExpression; }
        public AstToken getLEFT_SHIFT() { return _LEFT_SHIFT; }
        public IAdditiveExpression getAdditiveExpression() { return _AdditiveExpression; }

        public ShiftExpression0(IToken leftIToken, IToken rightIToken,
                                IShiftExpression _ShiftExpression,
                                AstToken _LEFT_SHIFT,
                                IAdditiveExpression _AdditiveExpression)
        {
            super(leftIToken, rightIToken);

            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            this._LEFT_SHIFT = _LEFT_SHIFT;
            ((Ast) _LEFT_SHIFT).setParent(this);
            this._AdditiveExpression = _AdditiveExpression;
            ((Ast) _AdditiveExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ShiftExpression);
            list.add(_LEFT_SHIFT);
            list.add(_AdditiveExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ShiftExpression0)) return false;
            if (! super.equals(o)) return false;
            ShiftExpression0 other = (ShiftExpression0) o;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            if (! _LEFT_SHIFT.equals(other._LEFT_SHIFT)) return false;
            if (! _AdditiveExpression.equals(other._AdditiveExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ShiftExpression.hashCode());
            hash = hash * 31 + (_LEFT_SHIFT.hashCode());
            hash = hash * 31 + (_AdditiveExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ShiftExpression.accept(v);
                _LEFT_SHIFT.accept(v);
                _AdditiveExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 405:  ShiftExpression ::= ShiftExpression > > AdditiveExpression
     *</b>
     */
    static public class ShiftExpression1 extends Ast implements IShiftExpression
    {
        private IShiftExpression _ShiftExpression;
        private AstToken _GREATER;
        private AstToken _GREATER3;
        private IAdditiveExpression _AdditiveExpression;

        public IShiftExpression getShiftExpression() { return _ShiftExpression; }
        public AstToken getGREATER() { return _GREATER; }
        public AstToken getGREATER3() { return _GREATER3; }
        public IAdditiveExpression getAdditiveExpression() { return _AdditiveExpression; }

        public ShiftExpression1(IToken leftIToken, IToken rightIToken,
                                IShiftExpression _ShiftExpression,
                                AstToken _GREATER,
                                AstToken _GREATER3,
                                IAdditiveExpression _AdditiveExpression)
        {
            super(leftIToken, rightIToken);

            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._GREATER3 = _GREATER3;
            ((Ast) _GREATER3).setParent(this);
            this._AdditiveExpression = _AdditiveExpression;
            ((Ast) _AdditiveExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ShiftExpression);
            list.add(_GREATER);
            list.add(_GREATER3);
            list.add(_AdditiveExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ShiftExpression1)) return false;
            if (! super.equals(o)) return false;
            ShiftExpression1 other = (ShiftExpression1) o;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _GREATER3.equals(other._GREATER3)) return false;
            if (! _AdditiveExpression.equals(other._AdditiveExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ShiftExpression.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_GREATER3.hashCode());
            hash = hash * 31 + (_AdditiveExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ShiftExpression.accept(v);
                _GREATER.accept(v);
                _GREATER3.accept(v);
                _AdditiveExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 406:  ShiftExpression ::= ShiftExpression > > > AdditiveExpression
     *</b>
     */
    static public class ShiftExpression2 extends Ast implements IShiftExpression
    {
        private IShiftExpression _ShiftExpression;
        private AstToken _GREATER;
        private AstToken _GREATER3;
        private AstToken _GREATER4;
        private IAdditiveExpression _AdditiveExpression;

        public IShiftExpression getShiftExpression() { return _ShiftExpression; }
        public AstToken getGREATER() { return _GREATER; }
        public AstToken getGREATER3() { return _GREATER3; }
        public AstToken getGREATER4() { return _GREATER4; }
        public IAdditiveExpression getAdditiveExpression() { return _AdditiveExpression; }

        public ShiftExpression2(IToken leftIToken, IToken rightIToken,
                                IShiftExpression _ShiftExpression,
                                AstToken _GREATER,
                                AstToken _GREATER3,
                                AstToken _GREATER4,
                                IAdditiveExpression _AdditiveExpression)
        {
            super(leftIToken, rightIToken);

            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._GREATER3 = _GREATER3;
            ((Ast) _GREATER3).setParent(this);
            this._GREATER4 = _GREATER4;
            ((Ast) _GREATER4).setParent(this);
            this._AdditiveExpression = _AdditiveExpression;
            ((Ast) _AdditiveExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_ShiftExpression);
            list.add(_GREATER);
            list.add(_GREATER3);
            list.add(_GREATER4);
            list.add(_AdditiveExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof ShiftExpression2)) return false;
            if (! super.equals(o)) return false;
            ShiftExpression2 other = (ShiftExpression2) o;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _GREATER3.equals(other._GREATER3)) return false;
            if (! _GREATER4.equals(other._GREATER4)) return false;
            if (! _AdditiveExpression.equals(other._AdditiveExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_ShiftExpression.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_GREATER3.hashCode());
            hash = hash * 31 + (_GREATER4.hashCode());
            hash = hash * 31 + (_AdditiveExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _ShiftExpression.accept(v);
                _GREATER.accept(v);
                _GREATER3.accept(v);
                _GREATER4.accept(v);
                _AdditiveExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 408:  RelationalExpression ::= RelationalExpression < ShiftExpression
     *</b>
     */
    static public class RelationalExpression0 extends Ast implements IRelationalExpression
    {
        private IRelationalExpression _RelationalExpression;
        private AstToken _LESS;
        private IShiftExpression _ShiftExpression;

        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }
        public AstToken getLESS() { return _LESS; }
        public IShiftExpression getShiftExpression() { return _ShiftExpression; }

        public RelationalExpression0(IToken leftIToken, IToken rightIToken,
                                     IRelationalExpression _RelationalExpression,
                                     AstToken _LESS,
                                     IShiftExpression _ShiftExpression)
        {
            super(leftIToken, rightIToken);

            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            this._LESS = _LESS;
            ((Ast) _LESS).setParent(this);
            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_RelationalExpression);
            list.add(_LESS);
            list.add(_ShiftExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof RelationalExpression0)) return false;
            if (! super.equals(o)) return false;
            RelationalExpression0 other = (RelationalExpression0) o;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            if (! _LESS.equals(other._LESS)) return false;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_RelationalExpression.hashCode());
            hash = hash * 31 + (_LESS.hashCode());
            hash = hash * 31 + (_ShiftExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _RelationalExpression.accept(v);
                _LESS.accept(v);
                _ShiftExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 409:  RelationalExpression ::= RelationalExpression > ShiftExpression
     *</b>
     */
    static public class RelationalExpression1 extends Ast implements IRelationalExpression
    {
        private IRelationalExpression _RelationalExpression;
        private AstToken _GREATER;
        private IShiftExpression _ShiftExpression;

        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }
        public AstToken getGREATER() { return _GREATER; }
        public IShiftExpression getShiftExpression() { return _ShiftExpression; }

        public RelationalExpression1(IToken leftIToken, IToken rightIToken,
                                     IRelationalExpression _RelationalExpression,
                                     AstToken _GREATER,
                                     IShiftExpression _ShiftExpression)
        {
            super(leftIToken, rightIToken);

            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_RelationalExpression);
            list.add(_GREATER);
            list.add(_ShiftExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof RelationalExpression1)) return false;
            if (! super.equals(o)) return false;
            RelationalExpression1 other = (RelationalExpression1) o;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_RelationalExpression.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_ShiftExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _RelationalExpression.accept(v);
                _GREATER.accept(v);
                _ShiftExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 410:  RelationalExpression ::= RelationalExpression <= ShiftExpression
     *</b>
     */
    static public class RelationalExpression2 extends Ast implements IRelationalExpression
    {
        private IRelationalExpression _RelationalExpression;
        private AstToken _LESS_EQUAL;
        private IShiftExpression _ShiftExpression;

        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }
        public AstToken getLESS_EQUAL() { return _LESS_EQUAL; }
        public IShiftExpression getShiftExpression() { return _ShiftExpression; }

        public RelationalExpression2(IToken leftIToken, IToken rightIToken,
                                     IRelationalExpression _RelationalExpression,
                                     AstToken _LESS_EQUAL,
                                     IShiftExpression _ShiftExpression)
        {
            super(leftIToken, rightIToken);

            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            this._LESS_EQUAL = _LESS_EQUAL;
            ((Ast) _LESS_EQUAL).setParent(this);
            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_RelationalExpression);
            list.add(_LESS_EQUAL);
            list.add(_ShiftExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof RelationalExpression2)) return false;
            if (! super.equals(o)) return false;
            RelationalExpression2 other = (RelationalExpression2) o;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            if (! _LESS_EQUAL.equals(other._LESS_EQUAL)) return false;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_RelationalExpression.hashCode());
            hash = hash * 31 + (_LESS_EQUAL.hashCode());
            hash = hash * 31 + (_ShiftExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _RelationalExpression.accept(v);
                _LESS_EQUAL.accept(v);
                _ShiftExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 411:  RelationalExpression ::= RelationalExpression > = ShiftExpression
     *</b>
     */
    static public class RelationalExpression3 extends Ast implements IRelationalExpression
    {
        private IRelationalExpression _RelationalExpression;
        private AstToken _GREATER;
        private AstToken _EQUAL;
        private IShiftExpression _ShiftExpression;

        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }
        public AstToken getGREATER() { return _GREATER; }
        public AstToken getEQUAL() { return _EQUAL; }
        public IShiftExpression getShiftExpression() { return _ShiftExpression; }

        public RelationalExpression3(IToken leftIToken, IToken rightIToken,
                                     IRelationalExpression _RelationalExpression,
                                     AstToken _GREATER,
                                     AstToken _EQUAL,
                                     IShiftExpression _ShiftExpression)
        {
            super(leftIToken, rightIToken);

            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._EQUAL = _EQUAL;
            ((Ast) _EQUAL).setParent(this);
            this._ShiftExpression = _ShiftExpression;
            ((Ast) _ShiftExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_RelationalExpression);
            list.add(_GREATER);
            list.add(_EQUAL);
            list.add(_ShiftExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof RelationalExpression3)) return false;
            if (! super.equals(o)) return false;
            RelationalExpression3 other = (RelationalExpression3) o;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _EQUAL.equals(other._EQUAL)) return false;
            if (! _ShiftExpression.equals(other._ShiftExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_RelationalExpression.hashCode());
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_EQUAL.hashCode());
            hash = hash * 31 + (_ShiftExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _RelationalExpression.accept(v);
                _GREATER.accept(v);
                _EQUAL.accept(v);
                _ShiftExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 412:  RelationalExpression ::= RelationalExpression instanceof ReferenceType
     *</b>
     */
    static public class RelationalExpression4 extends Ast implements IRelationalExpression
    {
        private IRelationalExpression _RelationalExpression;
        private AstToken _instanceof;
        private IReferenceType _ReferenceType;

        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }
        public AstToken getinstanceof() { return _instanceof; }
        public IReferenceType getReferenceType() { return _ReferenceType; }

        public RelationalExpression4(IToken leftIToken, IToken rightIToken,
                                     IRelationalExpression _RelationalExpression,
                                     AstToken _instanceof,
                                     IReferenceType _ReferenceType)
        {
            super(leftIToken, rightIToken);

            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            this._instanceof = _instanceof;
            ((Ast) _instanceof).setParent(this);
            this._ReferenceType = _ReferenceType;
            ((Ast) _ReferenceType).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_RelationalExpression);
            list.add(_instanceof);
            list.add(_ReferenceType);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof RelationalExpression4)) return false;
            if (! super.equals(o)) return false;
            RelationalExpression4 other = (RelationalExpression4) o;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            if (! _instanceof.equals(other._instanceof)) return false;
            if (! _ReferenceType.equals(other._ReferenceType)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_RelationalExpression.hashCode());
            hash = hash * 31 + (_instanceof.hashCode());
            hash = hash * 31 + (_ReferenceType.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _RelationalExpression.accept(v);
                _instanceof.accept(v);
                _ReferenceType.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 414:  EqualityExpression ::= EqualityExpression == RelationalExpression
     *</b>
     */
    static public class EqualityExpression0 extends Ast implements IEqualityExpression
    {
        private IEqualityExpression _EqualityExpression;
        private AstToken _EQUAL_EQUAL;
        private IRelationalExpression _RelationalExpression;

        public IEqualityExpression getEqualityExpression() { return _EqualityExpression; }
        public AstToken getEQUAL_EQUAL() { return _EQUAL_EQUAL; }
        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }

        public EqualityExpression0(IToken leftIToken, IToken rightIToken,
                                   IEqualityExpression _EqualityExpression,
                                   AstToken _EQUAL_EQUAL,
                                   IRelationalExpression _RelationalExpression)
        {
            super(leftIToken, rightIToken);

            this._EqualityExpression = _EqualityExpression;
            ((Ast) _EqualityExpression).setParent(this);
            this._EQUAL_EQUAL = _EQUAL_EQUAL;
            ((Ast) _EQUAL_EQUAL).setParent(this);
            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_EqualityExpression);
            list.add(_EQUAL_EQUAL);
            list.add(_RelationalExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EqualityExpression0)) return false;
            if (! super.equals(o)) return false;
            EqualityExpression0 other = (EqualityExpression0) o;
            if (! _EqualityExpression.equals(other._EqualityExpression)) return false;
            if (! _EQUAL_EQUAL.equals(other._EQUAL_EQUAL)) return false;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_EqualityExpression.hashCode());
            hash = hash * 31 + (_EQUAL_EQUAL.hashCode());
            hash = hash * 31 + (_RelationalExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _EqualityExpression.accept(v);
                _EQUAL_EQUAL.accept(v);
                _RelationalExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 415:  EqualityExpression ::= EqualityExpression != RelationalExpression
     *</b>
     */
    static public class EqualityExpression1 extends Ast implements IEqualityExpression
    {
        private IEqualityExpression _EqualityExpression;
        private AstToken _NOT_EQUAL;
        private IRelationalExpression _RelationalExpression;

        public IEqualityExpression getEqualityExpression() { return _EqualityExpression; }
        public AstToken getNOT_EQUAL() { return _NOT_EQUAL; }
        public IRelationalExpression getRelationalExpression() { return _RelationalExpression; }

        public EqualityExpression1(IToken leftIToken, IToken rightIToken,
                                   IEqualityExpression _EqualityExpression,
                                   AstToken _NOT_EQUAL,
                                   IRelationalExpression _RelationalExpression)
        {
            super(leftIToken, rightIToken);

            this._EqualityExpression = _EqualityExpression;
            ((Ast) _EqualityExpression).setParent(this);
            this._NOT_EQUAL = _NOT_EQUAL;
            ((Ast) _NOT_EQUAL).setParent(this);
            this._RelationalExpression = _RelationalExpression;
            ((Ast) _RelationalExpression).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_EqualityExpression);
            list.add(_NOT_EQUAL);
            list.add(_RelationalExpression);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof EqualityExpression1)) return false;
            if (! super.equals(o)) return false;
            EqualityExpression1 other = (EqualityExpression1) o;
            if (! _EqualityExpression.equals(other._EqualityExpression)) return false;
            if (! _NOT_EQUAL.equals(other._NOT_EQUAL)) return false;
            if (! _RelationalExpression.equals(other._RelationalExpression)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_EqualityExpression.hashCode());
            hash = hash * 31 + (_NOT_EQUAL.hashCode());
            hash = hash * 31 + (_RelationalExpression.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _EqualityExpression.accept(v);
                _NOT_EQUAL.accept(v);
                _RelationalExpression.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 434:  AssignmentOperator ::= =
     *</b>
     */
    static public class AssignmentOperator0 extends AstToken implements IAssignmentOperator
    {
        public IToken getEQUAL() { return leftIToken; }

        public AssignmentOperator0(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 435:  AssignmentOperator ::= *=
     *</b>
     */
    static public class AssignmentOperator1 extends AstToken implements IAssignmentOperator
    {
        public IToken getMULTIPLY_EQUAL() { return leftIToken; }

        public AssignmentOperator1(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 436:  AssignmentOperator ::= /=
     *</b>
     */
    static public class AssignmentOperator2 extends AstToken implements IAssignmentOperator
    {
        public IToken getDIVIDE_EQUAL() { return leftIToken; }

        public AssignmentOperator2(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 437:  AssignmentOperator ::= %=
     *</b>
     */
    static public class AssignmentOperator3 extends AstToken implements IAssignmentOperator
    {
        public IToken getREMAINDER_EQUAL() { return leftIToken; }

        public AssignmentOperator3(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 438:  AssignmentOperator ::= +=
     *</b>
     */
    static public class AssignmentOperator4 extends AstToken implements IAssignmentOperator
    {
        public IToken getPLUS_EQUAL() { return leftIToken; }

        public AssignmentOperator4(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 439:  AssignmentOperator ::= -=
     *</b>
     */
    static public class AssignmentOperator5 extends AstToken implements IAssignmentOperator
    {
        public IToken getMINUS_EQUAL() { return leftIToken; }

        public AssignmentOperator5(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 440:  AssignmentOperator ::= <<=
     *</b>
     */
    static public class AssignmentOperator6 extends AstToken implements IAssignmentOperator
    {
        public IToken getLEFT_SHIFT_EQUAL() { return leftIToken; }

        public AssignmentOperator6(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 441:  AssignmentOperator ::= > > =
     *</b>
     */
    static public class AssignmentOperator7 extends Ast implements IAssignmentOperator
    {
        private AstToken _GREATER;
        private AstToken _GREATER2;
        private AstToken _EQUAL;

        public AstToken getGREATER() { return _GREATER; }
        public AstToken getGREATER2() { return _GREATER2; }
        public AstToken getEQUAL() { return _EQUAL; }

        public AssignmentOperator7(IToken leftIToken, IToken rightIToken,
                                   AstToken _GREATER,
                                   AstToken _GREATER2,
                                   AstToken _EQUAL)
        {
            super(leftIToken, rightIToken);

            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._GREATER2 = _GREATER2;
            ((Ast) _GREATER2).setParent(this);
            this._EQUAL = _EQUAL;
            ((Ast) _EQUAL).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_GREATER);
            list.add(_GREATER2);
            list.add(_EQUAL);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AssignmentOperator7)) return false;
            if (! super.equals(o)) return false;
            AssignmentOperator7 other = (AssignmentOperator7) o;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _GREATER2.equals(other._GREATER2)) return false;
            if (! _EQUAL.equals(other._EQUAL)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_GREATER2.hashCode());
            hash = hash * 31 + (_EQUAL.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _GREATER.accept(v);
                _GREATER2.accept(v);
                _EQUAL.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 442:  AssignmentOperator ::= > > > =
     *</b>
     */
    static public class AssignmentOperator8 extends Ast implements IAssignmentOperator
    {
        private AstToken _GREATER;
        private AstToken _GREATER2;
        private AstToken _GREATER3;
        private AstToken _EQUAL;

        public AstToken getGREATER() { return _GREATER; }
        public AstToken getGREATER2() { return _GREATER2; }
        public AstToken getGREATER3() { return _GREATER3; }
        public AstToken getEQUAL() { return _EQUAL; }

        public AssignmentOperator8(IToken leftIToken, IToken rightIToken,
                                   AstToken _GREATER,
                                   AstToken _GREATER2,
                                   AstToken _GREATER3,
                                   AstToken _EQUAL)
        {
            super(leftIToken, rightIToken);

            this._GREATER = _GREATER;
            ((Ast) _GREATER).setParent(this);
            this._GREATER2 = _GREATER2;
            ((Ast) _GREATER2).setParent(this);
            this._GREATER3 = _GREATER3;
            ((Ast) _GREATER3).setParent(this);
            this._EQUAL = _EQUAL;
            ((Ast) _EQUAL).setParent(this);
            initialize();
        }

        /**
         * A list of all children of this node, including the null ones.
         */
        public java.util.ArrayList getAllChildren()
        {
            java.util.ArrayList list = new java.util.ArrayList();
            list.add(_GREATER);
            list.add(_GREATER2);
            list.add(_GREATER3);
            list.add(_EQUAL);
            return list;
        }

        public boolean equals(Object o)
        {
            if (o == this) return true;
            if (! (o instanceof AssignmentOperator8)) return false;
            if (! super.equals(o)) return false;
            AssignmentOperator8 other = (AssignmentOperator8) o;
            if (! _GREATER.equals(other._GREATER)) return false;
            if (! _GREATER2.equals(other._GREATER2)) return false;
            if (! _GREATER3.equals(other._GREATER3)) return false;
            if (! _EQUAL.equals(other._EQUAL)) return false;
            return true;
        }

        public int hashCode()
        {
            int hash = super.hashCode();
            hash = hash * 31 + (_GREATER.hashCode());
            hash = hash * 31 + (_GREATER2.hashCode());
            hash = hash * 31 + (_GREATER3.hashCode());
            hash = hash * 31 + (_EQUAL.hashCode());
            return hash;
        }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            boolean checkChildren = v.visit(this);
            if (checkChildren)
            {
                _GREATER.accept(v);
                _GREATER2.accept(v);
                _GREATER3.accept(v);
                _EQUAL.accept(v);
            }
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 443:  AssignmentOperator ::= &=
     *</b>
     */
    static public class AssignmentOperator9 extends AstToken implements IAssignmentOperator
    {
        public IToken getAND_EQUAL() { return leftIToken; }

        public AssignmentOperator9(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 444:  AssignmentOperator ::= ^=
     *</b>
     */
    static public class AssignmentOperator10 extends AstToken implements IAssignmentOperator
    {
        public IToken getXOR_EQUAL() { return leftIToken; }

        public AssignmentOperator10(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    /**
     *<b>
     *<li>Rule 445:  AssignmentOperator ::= |=
     *</b>
     */
    static public class AssignmentOperator11 extends AstToken implements IAssignmentOperator
    {
        public IToken getOR_EQUAL() { return leftIToken; }

        public AssignmentOperator11(IToken token) { super(token); initialize(); }

        public void accept(IAstVisitor v)
        {
            if (! v.preVisit(this)) return;
            enter((Visitor) v);
            v.postVisit(this);
        }

        public void enter(Visitor v)
        {
            v.visit(this);
            v.endVisit(this);
        }
    }

    public interface Visitor extends IAstVisitor
    {
        boolean visit(Ast n);
        void endVisit(Ast n);

        boolean visit(AstToken n);
        void endVisit(AstToken n);

        boolean visit(identifier n);
        void endVisit(identifier n);

        boolean visit(PrimitiveType n);
        void endVisit(PrimitiveType n);

        boolean visit(ClassType n);
        void endVisit(ClassType n);

        boolean visit(InterfaceType n);
        void endVisit(InterfaceType n);

        boolean visit(TypeName n);
        void endVisit(TypeName n);

        boolean visit(ArrayType n);
        void endVisit(ArrayType n);

        boolean visit(TypeParameter n);
        void endVisit(TypeParameter n);

        boolean visit(TypeBound n);
        void endVisit(TypeBound n);

        boolean visit(AdditionalBoundList n);
        void endVisit(AdditionalBoundList n);

        boolean visit(AdditionalBound n);
        void endVisit(AdditionalBound n);

        boolean visit(TypeArguments n);
        void endVisit(TypeArguments n);

        boolean visit(ActualTypeArgumentList n);
        void endVisit(ActualTypeArgumentList n);

        boolean visit(Wildcard n);
        void endVisit(Wildcard n);

        boolean visit(PackageName n);
        void endVisit(PackageName n);

        boolean visit(ExpressionName n);
        void endVisit(ExpressionName n);

        boolean visit(MethodName n);
        void endVisit(MethodName n);

        boolean visit(PackageOrTypeName n);
        void endVisit(PackageOrTypeName n);

        boolean visit(AmbiguousName n);
        void endVisit(AmbiguousName n);

        boolean visit(CompilationUnit n);
        void endVisit(CompilationUnit n);

        boolean visit(ImportDeclarations n);
        void endVisit(ImportDeclarations n);

        boolean visit(TypeDeclarations n);
        void endVisit(TypeDeclarations n);

        boolean visit(PackageDeclaration n);
        void endVisit(PackageDeclaration n);

        boolean visit(SingleTypeImportDeclaration n);
        void endVisit(SingleTypeImportDeclaration n);

        boolean visit(TypeImportOnDemandDeclaration n);
        void endVisit(TypeImportOnDemandDeclaration n);

        boolean visit(SingleStaticImportDeclaration n);
        void endVisit(SingleStaticImportDeclaration n);

        boolean visit(StaticImportOnDemandDeclaration n);
        void endVisit(StaticImportOnDemandDeclaration n);

        boolean visit(TypeDeclaration n);
        void endVisit(TypeDeclaration n);

        boolean visit(NormalClassDeclaration n);
        void endVisit(NormalClassDeclaration n);

        boolean visit(ClassModifiers n);
        void endVisit(ClassModifiers n);

        boolean visit(TypeParameters n);
        void endVisit(TypeParameters n);

        boolean visit(TypeParameterList n);
        void endVisit(TypeParameterList n);

        boolean visit(Super n);
        void endVisit(Super n);

        boolean visit(Interfaces n);
        void endVisit(Interfaces n);

        boolean visit(InterfaceTypeList n);
        void endVisit(InterfaceTypeList n);

        boolean visit(ClassBody n);
        void endVisit(ClassBody n);

        boolean visit(ClassBodyDeclarations n);
        void endVisit(ClassBodyDeclarations n);

        boolean visit(ClassMemberDeclaration n);
        void endVisit(ClassMemberDeclaration n);

        boolean visit(FieldDeclaration n);
        void endVisit(FieldDeclaration n);

        boolean visit(VariableDeclarators n);
        void endVisit(VariableDeclarators n);

        boolean visit(VariableDeclarator n);
        void endVisit(VariableDeclarator n);

        boolean visit(VariableDeclaratorId n);
        void endVisit(VariableDeclaratorId n);

        boolean visit(FieldModifiers n);
        void endVisit(FieldModifiers n);

        boolean visit(MethodDeclaration n);
        void endVisit(MethodDeclaration n);

        boolean visit(MethodHeader n);
        void endVisit(MethodHeader n);

        boolean visit(ResultType n);
        void endVisit(ResultType n);

        boolean visit(FormalParameterList n);
        void endVisit(FormalParameterList n);

        boolean visit(FormalParameters n);
        void endVisit(FormalParameters n);

        boolean visit(FormalParameter n);
        void endVisit(FormalParameter n);

        boolean visit(VariableModifiers n);
        void endVisit(VariableModifiers n);

        boolean visit(VariableModifier n);
        void endVisit(VariableModifier n);

        boolean visit(LastFormalParameter n);
        void endVisit(LastFormalParameter n);

        boolean visit(MethodModifiers n);
        void endVisit(MethodModifiers n);

        boolean visit(Throws n);
        void endVisit(Throws n);

        boolean visit(ExceptionTypeList n);
        void endVisit(ExceptionTypeList n);

        boolean visit(MethodBody n);
        void endVisit(MethodBody n);

        boolean visit(StaticInitializer n);
        void endVisit(StaticInitializer n);

        boolean visit(ConstructorDeclaration n);
        void endVisit(ConstructorDeclaration n);

        boolean visit(ConstructorDeclarator n);
        void endVisit(ConstructorDeclarator n);

        boolean visit(ConstructorModifiers n);
        void endVisit(ConstructorModifiers n);

        boolean visit(ConstructorBody n);
        void endVisit(ConstructorBody n);

        boolean visit(EnumDeclaration n);
        void endVisit(EnumDeclaration n);

        boolean visit(EnumBody n);
        void endVisit(EnumBody n);

        boolean visit(EnumConstants n);
        void endVisit(EnumConstants n);

        boolean visit(EnumConstant n);
        void endVisit(EnumConstant n);

        boolean visit(Arguments n);
        void endVisit(Arguments n);

        boolean visit(EnumBodyDeclarations n);
        void endVisit(EnumBodyDeclarations n);

        boolean visit(NormalInterfaceDeclaration n);
        void endVisit(NormalInterfaceDeclaration n);

        boolean visit(InterfaceModifiers n);
        void endVisit(InterfaceModifiers n);

        boolean visit(InterfaceBody n);
        void endVisit(InterfaceBody n);

        boolean visit(InterfaceMemberDeclarations n);
        void endVisit(InterfaceMemberDeclarations n);

        boolean visit(InterfaceMemberDeclaration n);
        void endVisit(InterfaceMemberDeclaration n);

        boolean visit(ConstantDeclaration n);
        void endVisit(ConstantDeclaration n);

        boolean visit(ConstantModifiers n);
        void endVisit(ConstantModifiers n);

        boolean visit(AbstractMethodDeclaration n);
        void endVisit(AbstractMethodDeclaration n);

        boolean visit(AbstractMethodModifiers n);
        void endVisit(AbstractMethodModifiers n);

        boolean visit(AnnotationTypeDeclaration n);
        void endVisit(AnnotationTypeDeclaration n);

        boolean visit(AnnotationTypeBody n);
        void endVisit(AnnotationTypeBody n);

        boolean visit(AnnotationTypeElementDeclarations n);
        void endVisit(AnnotationTypeElementDeclarations n);

        boolean visit(DefaultValue n);
        void endVisit(DefaultValue n);

        boolean visit(Annotations n);
        void endVisit(Annotations n);

        boolean visit(NormalAnnotation n);
        void endVisit(NormalAnnotation n);

        boolean visit(ElementValuePairs n);
        void endVisit(ElementValuePairs n);

        boolean visit(ElementValuePair n);
        void endVisit(ElementValuePair n);

        boolean visit(ElementValueArrayInitializer n);
        void endVisit(ElementValueArrayInitializer n);

        boolean visit(ElementValues n);
        void endVisit(ElementValues n);

        boolean visit(MarkerAnnotation n);
        void endVisit(MarkerAnnotation n);

        boolean visit(SingleElementAnnotation n);
        void endVisit(SingleElementAnnotation n);

        boolean visit(ArrayInitializer n);
        void endVisit(ArrayInitializer n);

        boolean visit(VariableInitializers n);
        void endVisit(VariableInitializers n);

        boolean visit(Block n);
        void endVisit(Block n);

        boolean visit(BlockStatements n);
        void endVisit(BlockStatements n);

        boolean visit(LocalVariableDeclarationStatement n);
        void endVisit(LocalVariableDeclarationStatement n);

        boolean visit(LocalVariableDeclaration n);
        void endVisit(LocalVariableDeclaration n);

        boolean visit(IfThenStatement n);
        void endVisit(IfThenStatement n);

        boolean visit(IfThenElseStatement n);
        void endVisit(IfThenElseStatement n);

        boolean visit(IfThenElseStatementNoShortIf n);
        void endVisit(IfThenElseStatementNoShortIf n);

        boolean visit(EmptyStatement n);
        void endVisit(EmptyStatement n);

        boolean visit(LabeledStatement n);
        void endVisit(LabeledStatement n);

        boolean visit(LabeledStatementNoShortIf n);
        void endVisit(LabeledStatementNoShortIf n);

        boolean visit(ExpressionStatement n);
        void endVisit(ExpressionStatement n);

        boolean visit(SwitchStatement n);
        void endVisit(SwitchStatement n);

        boolean visit(SwitchBlock n);
        void endVisit(SwitchBlock n);

        boolean visit(SwitchBlockStatementGroups n);
        void endVisit(SwitchBlockStatementGroups n);

        boolean visit(SwitchBlockStatementGroup n);
        void endVisit(SwitchBlockStatementGroup n);

        boolean visit(SwitchLabels n);
        void endVisit(SwitchLabels n);

        boolean visit(WhileStatement n);
        void endVisit(WhileStatement n);

        boolean visit(WhileStatementNoShortIf n);
        void endVisit(WhileStatementNoShortIf n);

        boolean visit(DoStatement n);
        void endVisit(DoStatement n);

        boolean visit(BasicForStatement n);
        void endVisit(BasicForStatement n);

        boolean visit(ForStatementNoShortIf n);
        void endVisit(ForStatementNoShortIf n);

        boolean visit(StatementExpressionList n);
        void endVisit(StatementExpressionList n);

        boolean visit(EnhancedForStatement n);
        void endVisit(EnhancedForStatement n);

        boolean visit(BreakStatement n);
        void endVisit(BreakStatement n);

        boolean visit(ContinueStatement n);
        void endVisit(ContinueStatement n);

        boolean visit(ReturnStatement n);
        void endVisit(ReturnStatement n);

        boolean visit(ThrowStatement n);
        void endVisit(ThrowStatement n);

        boolean visit(SynchronizedStatement n);
        void endVisit(SynchronizedStatement n);

        boolean visit(Catches n);
        void endVisit(Catches n);

        boolean visit(CatchClause n);
        void endVisit(CatchClause n);

        boolean visit(Finally n);
        void endVisit(Finally n);

        boolean visit(ArgumentList n);
        void endVisit(ArgumentList n);

        boolean visit(DimExprs n);
        void endVisit(DimExprs n);

        boolean visit(DimExpr n);
        void endVisit(DimExpr n);

        boolean visit(PostIncrementExpression n);
        void endVisit(PostIncrementExpression n);

        boolean visit(PostDecrementExpression n);
        void endVisit(PostDecrementExpression n);

        boolean visit(PreIncrementExpression n);
        void endVisit(PreIncrementExpression n);

        boolean visit(PreDecrementExpression n);
        void endVisit(PreDecrementExpression n);

        boolean visit(AndExpression n);
        void endVisit(AndExpression n);

        boolean visit(ExclusiveOrExpression n);
        void endVisit(ExclusiveOrExpression n);

        boolean visit(InclusiveOrExpression n);
        void endVisit(InclusiveOrExpression n);

        boolean visit(ConditionalAndExpression n);
        void endVisit(ConditionalAndExpression n);

        boolean visit(ConditionalOrExpression n);
        void endVisit(ConditionalOrExpression n);

        boolean visit(ConditionalExpression n);
        void endVisit(ConditionalExpression n);

        boolean visit(Assignment n);
        void endVisit(Assignment n);

        boolean visit(Commaopt n);
        void endVisit(Commaopt n);

        boolean visit(Ellipsisopt n);
        void endVisit(Ellipsisopt n);

        boolean visit(LPGUserAction0 n);
        void endVisit(LPGUserAction0 n);

        boolean visit(LPGUserAction1 n);
        void endVisit(LPGUserAction1 n);

        boolean visit(LPGUserAction2 n);
        void endVisit(LPGUserAction2 n);

        boolean visit(LPGUserAction3 n);
        void endVisit(LPGUserAction3 n);

        boolean visit(LPGUserAction4 n);
        void endVisit(LPGUserAction4 n);

        boolean visit(IntegralType0 n);
        void endVisit(IntegralType0 n);

        boolean visit(IntegralType1 n);
        void endVisit(IntegralType1 n);

        boolean visit(IntegralType2 n);
        void endVisit(IntegralType2 n);

        boolean visit(IntegralType3 n);
        void endVisit(IntegralType3 n);

        boolean visit(IntegralType4 n);
        void endVisit(IntegralType4 n);

        boolean visit(FloatingPointType0 n);
        void endVisit(FloatingPointType0 n);

        boolean visit(FloatingPointType1 n);
        void endVisit(FloatingPointType1 n);

        boolean visit(WildcardBounds0 n);
        void endVisit(WildcardBounds0 n);

        boolean visit(WildcardBounds1 n);
        void endVisit(WildcardBounds1 n);

        boolean visit(ClassModifier0 n);
        void endVisit(ClassModifier0 n);

        boolean visit(ClassModifier1 n);
        void endVisit(ClassModifier1 n);

        boolean visit(ClassModifier2 n);
        void endVisit(ClassModifier2 n);

        boolean visit(ClassModifier3 n);
        void endVisit(ClassModifier3 n);

        boolean visit(ClassModifier4 n);
        void endVisit(ClassModifier4 n);

        boolean visit(ClassModifier5 n);
        void endVisit(ClassModifier5 n);

        boolean visit(ClassModifier6 n);
        void endVisit(ClassModifier6 n);

        boolean visit(FieldModifier0 n);
        void endVisit(FieldModifier0 n);

        boolean visit(FieldModifier1 n);
        void endVisit(FieldModifier1 n);

        boolean visit(FieldModifier2 n);
        void endVisit(FieldModifier2 n);

        boolean visit(FieldModifier3 n);
        void endVisit(FieldModifier3 n);

        boolean visit(FieldModifier4 n);
        void endVisit(FieldModifier4 n);

        boolean visit(FieldModifier5 n);
        void endVisit(FieldModifier5 n);

        boolean visit(FieldModifier6 n);
        void endVisit(FieldModifier6 n);

        boolean visit(MethodDeclarator0 n);
        void endVisit(MethodDeclarator0 n);

        boolean visit(MethodDeclarator1 n);
        void endVisit(MethodDeclarator1 n);

        boolean visit(MethodModifier0 n);
        void endVisit(MethodModifier0 n);

        boolean visit(MethodModifier1 n);
        void endVisit(MethodModifier1 n);

        boolean visit(MethodModifier2 n);
        void endVisit(MethodModifier2 n);

        boolean visit(MethodModifier3 n);
        void endVisit(MethodModifier3 n);

        boolean visit(MethodModifier4 n);
        void endVisit(MethodModifier4 n);

        boolean visit(MethodModifier5 n);
        void endVisit(MethodModifier5 n);

        boolean visit(MethodModifier6 n);
        void endVisit(MethodModifier6 n);

        boolean visit(MethodModifier7 n);
        void endVisit(MethodModifier7 n);

        boolean visit(MethodModifier8 n);
        void endVisit(MethodModifier8 n);

        boolean visit(ConstructorModifier0 n);
        void endVisit(ConstructorModifier0 n);

        boolean visit(ConstructorModifier1 n);
        void endVisit(ConstructorModifier1 n);

        boolean visit(ConstructorModifier2 n);
        void endVisit(ConstructorModifier2 n);

        boolean visit(ExplicitConstructorInvocation0 n);
        void endVisit(ExplicitConstructorInvocation0 n);

        boolean visit(ExplicitConstructorInvocation1 n);
        void endVisit(ExplicitConstructorInvocation1 n);

        boolean visit(ExplicitConstructorInvocation2 n);
        void endVisit(ExplicitConstructorInvocation2 n);

        boolean visit(InterfaceModifier0 n);
        void endVisit(InterfaceModifier0 n);

        boolean visit(InterfaceModifier1 n);
        void endVisit(InterfaceModifier1 n);

        boolean visit(InterfaceModifier2 n);
        void endVisit(InterfaceModifier2 n);

        boolean visit(InterfaceModifier3 n);
        void endVisit(InterfaceModifier3 n);

        boolean visit(InterfaceModifier4 n);
        void endVisit(InterfaceModifier4 n);

        boolean visit(InterfaceModifier5 n);
        void endVisit(InterfaceModifier5 n);

        boolean visit(ExtendsInterfaces0 n);
        void endVisit(ExtendsInterfaces0 n);

        boolean visit(ExtendsInterfaces1 n);
        void endVisit(ExtendsInterfaces1 n);

        boolean visit(ConstantModifier0 n);
        void endVisit(ConstantModifier0 n);

        boolean visit(ConstantModifier1 n);
        void endVisit(ConstantModifier1 n);

        boolean visit(ConstantModifier2 n);
        void endVisit(ConstantModifier2 n);

        boolean visit(AbstractMethodModifier0 n);
        void endVisit(AbstractMethodModifier0 n);

        boolean visit(AbstractMethodModifier1 n);
        void endVisit(AbstractMethodModifier1 n);

        boolean visit(AnnotationTypeElementDeclaration0 n);
        void endVisit(AnnotationTypeElementDeclaration0 n);

        boolean visit(AnnotationTypeElementDeclaration1 n);
        void endVisit(AnnotationTypeElementDeclaration1 n);

        boolean visit(AssertStatement0 n);
        void endVisit(AssertStatement0 n);

        boolean visit(AssertStatement1 n);
        void endVisit(AssertStatement1 n);

        boolean visit(SwitchLabel0 n);
        void endVisit(SwitchLabel0 n);

        boolean visit(SwitchLabel1 n);
        void endVisit(SwitchLabel1 n);

        boolean visit(SwitchLabel2 n);
        void endVisit(SwitchLabel2 n);

        boolean visit(TryStatement0 n);
        void endVisit(TryStatement0 n);

        boolean visit(TryStatement1 n);
        void endVisit(TryStatement1 n);

        boolean visit(PrimaryNoNewArray0 n);
        void endVisit(PrimaryNoNewArray0 n);

        boolean visit(PrimaryNoNewArray1 n);
        void endVisit(PrimaryNoNewArray1 n);

        boolean visit(PrimaryNoNewArray2 n);
        void endVisit(PrimaryNoNewArray2 n);

        boolean visit(PrimaryNoNewArray3 n);
        void endVisit(PrimaryNoNewArray3 n);

        boolean visit(PrimaryNoNewArray4 n);
        void endVisit(PrimaryNoNewArray4 n);

        boolean visit(Literal0 n);
        void endVisit(Literal0 n);

        boolean visit(Literal1 n);
        void endVisit(Literal1 n);

        boolean visit(Literal2 n);
        void endVisit(Literal2 n);

        boolean visit(Literal3 n);
        void endVisit(Literal3 n);

        boolean visit(Literal4 n);
        void endVisit(Literal4 n);

        boolean visit(Literal5 n);
        void endVisit(Literal5 n);

        boolean visit(Literal6 n);
        void endVisit(Literal6 n);

        boolean visit(BooleanLiteral0 n);
        void endVisit(BooleanLiteral0 n);

        boolean visit(BooleanLiteral1 n);
        void endVisit(BooleanLiteral1 n);

        boolean visit(ClassInstanceCreationExpression0 n);
        void endVisit(ClassInstanceCreationExpression0 n);

        boolean visit(ClassInstanceCreationExpression1 n);
        void endVisit(ClassInstanceCreationExpression1 n);

        boolean visit(ArrayCreationExpression0 n);
        void endVisit(ArrayCreationExpression0 n);

        boolean visit(ArrayCreationExpression1 n);
        void endVisit(ArrayCreationExpression1 n);

        boolean visit(ArrayCreationExpression2 n);
        void endVisit(ArrayCreationExpression2 n);

        boolean visit(ArrayCreationExpression3 n);
        void endVisit(ArrayCreationExpression3 n);

        boolean visit(Dims0 n);
        void endVisit(Dims0 n);

        boolean visit(Dims1 n);
        void endVisit(Dims1 n);

        boolean visit(FieldAccess0 n);
        void endVisit(FieldAccess0 n);

        boolean visit(FieldAccess1 n);
        void endVisit(FieldAccess1 n);

        boolean visit(FieldAccess2 n);
        void endVisit(FieldAccess2 n);

        boolean visit(MethodInvocation0 n);
        void endVisit(MethodInvocation0 n);

        boolean visit(MethodInvocation1 n);
        void endVisit(MethodInvocation1 n);

        boolean visit(MethodInvocation2 n);
        void endVisit(MethodInvocation2 n);

        boolean visit(MethodInvocation3 n);
        void endVisit(MethodInvocation3 n);

        boolean visit(MethodInvocation4 n);
        void endVisit(MethodInvocation4 n);

        boolean visit(ArrayAccess0 n);
        void endVisit(ArrayAccess0 n);

        boolean visit(ArrayAccess1 n);
        void endVisit(ArrayAccess1 n);

        boolean visit(UnaryExpression0 n);
        void endVisit(UnaryExpression0 n);

        boolean visit(UnaryExpression1 n);
        void endVisit(UnaryExpression1 n);

        boolean visit(UnaryExpressionNotPlusMinus0 n);
        void endVisit(UnaryExpressionNotPlusMinus0 n);

        boolean visit(UnaryExpressionNotPlusMinus1 n);
        void endVisit(UnaryExpressionNotPlusMinus1 n);

        boolean visit(CastExpression0 n);
        void endVisit(CastExpression0 n);

        boolean visit(CastExpression1 n);
        void endVisit(CastExpression1 n);

        boolean visit(MultiplicativeExpression0 n);
        void endVisit(MultiplicativeExpression0 n);

        boolean visit(MultiplicativeExpression1 n);
        void endVisit(MultiplicativeExpression1 n);

        boolean visit(MultiplicativeExpression2 n);
        void endVisit(MultiplicativeExpression2 n);

        boolean visit(AdditiveExpression0 n);
        void endVisit(AdditiveExpression0 n);

        boolean visit(AdditiveExpression1 n);
        void endVisit(AdditiveExpression1 n);

        boolean visit(ShiftExpression0 n);
        void endVisit(ShiftExpression0 n);

        boolean visit(ShiftExpression1 n);
        void endVisit(ShiftExpression1 n);

        boolean visit(ShiftExpression2 n);
        void endVisit(ShiftExpression2 n);

        boolean visit(RelationalExpression0 n);
        void endVisit(RelationalExpression0 n);

        boolean visit(RelationalExpression1 n);
        void endVisit(RelationalExpression1 n);

        boolean visit(RelationalExpression2 n);
        void endVisit(RelationalExpression2 n);

        boolean visit(RelationalExpression3 n);
        void endVisit(RelationalExpression3 n);

        boolean visit(RelationalExpression4 n);
        void endVisit(RelationalExpression4 n);

        boolean visit(EqualityExpression0 n);
        void endVisit(EqualityExpression0 n);

        boolean visit(EqualityExpression1 n);
        void endVisit(EqualityExpression1 n);

        boolean visit(AssignmentOperator0 n);
        void endVisit(AssignmentOperator0 n);

        boolean visit(AssignmentOperator1 n);
        void endVisit(AssignmentOperator1 n);

        boolean visit(AssignmentOperator2 n);
        void endVisit(AssignmentOperator2 n);

        boolean visit(AssignmentOperator3 n);
        void endVisit(AssignmentOperator3 n);

        boolean visit(AssignmentOperator4 n);
        void endVisit(AssignmentOperator4 n);

        boolean visit(AssignmentOperator5 n);
        void endVisit(AssignmentOperator5 n);

        boolean visit(AssignmentOperator6 n);
        void endVisit(AssignmentOperator6 n);

        boolean visit(AssignmentOperator7 n);
        void endVisit(AssignmentOperator7 n);

        boolean visit(AssignmentOperator8 n);
        void endVisit(AssignmentOperator8 n);

        boolean visit(AssignmentOperator9 n);
        void endVisit(AssignmentOperator9 n);

        boolean visit(AssignmentOperator10 n);
        void endVisit(AssignmentOperator10 n);

        boolean visit(AssignmentOperator11 n);
        void endVisit(AssignmentOperator11 n);

    }

    static public abstract class AbstractVisitor implements Visitor
    {
        public abstract void unimplementedVisitor(String s);

        public boolean preVisit(IAst element) { return true; }

        public void postVisit(IAst element) {}

        public boolean visit(AstToken n) { unimplementedVisitor("visit(AstToken)"); return true; }
        public void endVisit(AstToken n) { unimplementedVisitor("endVisit(AstToken)"); }

        public boolean visit(identifier n) { unimplementedVisitor("visit(identifier)"); return true; }
        public void endVisit(identifier n) { unimplementedVisitor("endVisit(identifier)"); }

        public boolean visit(PrimitiveType n) { unimplementedVisitor("visit(PrimitiveType)"); return true; }
        public void endVisit(PrimitiveType n) { unimplementedVisitor("endVisit(PrimitiveType)"); }

        public boolean visit(ClassType n) { unimplementedVisitor("visit(ClassType)"); return true; }
        public void endVisit(ClassType n) { unimplementedVisitor("endVisit(ClassType)"); }

        public boolean visit(InterfaceType n) { unimplementedVisitor("visit(InterfaceType)"); return true; }
        public void endVisit(InterfaceType n) { unimplementedVisitor("endVisit(InterfaceType)"); }

        public boolean visit(TypeName n) { unimplementedVisitor("visit(TypeName)"); return true; }
        public void endVisit(TypeName n) { unimplementedVisitor("endVisit(TypeName)"); }

        public boolean visit(ArrayType n) { unimplementedVisitor("visit(ArrayType)"); return true; }
        public void endVisit(ArrayType n) { unimplementedVisitor("endVisit(ArrayType)"); }

        public boolean visit(TypeParameter n) { unimplementedVisitor("visit(TypeParameter)"); return true; }
        public void endVisit(TypeParameter n) { unimplementedVisitor("endVisit(TypeParameter)"); }

        public boolean visit(TypeBound n) { unimplementedVisitor("visit(TypeBound)"); return true; }
        public void endVisit(TypeBound n) { unimplementedVisitor("endVisit(TypeBound)"); }

        public boolean visit(AdditionalBoundList n) { unimplementedVisitor("visit(AdditionalBoundList)"); return true; }
        public void endVisit(AdditionalBoundList n) { unimplementedVisitor("endVisit(AdditionalBoundList)"); }

        public boolean visit(AdditionalBound n) { unimplementedVisitor("visit(AdditionalBound)"); return true; }
        public void endVisit(AdditionalBound n) { unimplementedVisitor("endVisit(AdditionalBound)"); }

        public boolean visit(TypeArguments n) { unimplementedVisitor("visit(TypeArguments)"); return true; }
        public void endVisit(TypeArguments n) { unimplementedVisitor("endVisit(TypeArguments)"); }

        public boolean visit(ActualTypeArgumentList n) { unimplementedVisitor("visit(ActualTypeArgumentList)"); return true; }
        public void endVisit(ActualTypeArgumentList n) { unimplementedVisitor("endVisit(ActualTypeArgumentList)"); }

        public boolean visit(Wildcard n) { unimplementedVisitor("visit(Wildcard)"); return true; }
        public void endVisit(Wildcard n) { unimplementedVisitor("endVisit(Wildcard)"); }

        public boolean visit(PackageName n) { unimplementedVisitor("visit(PackageName)"); return true; }
        public void endVisit(PackageName n) { unimplementedVisitor("endVisit(PackageName)"); }

        public boolean visit(ExpressionName n) { unimplementedVisitor("visit(ExpressionName)"); return true; }
        public void endVisit(ExpressionName n) { unimplementedVisitor("endVisit(ExpressionName)"); }

        public boolean visit(MethodName n) { unimplementedVisitor("visit(MethodName)"); return true; }
        public void endVisit(MethodName n) { unimplementedVisitor("endVisit(MethodName)"); }

        public boolean visit(PackageOrTypeName n) { unimplementedVisitor("visit(PackageOrTypeName)"); return true; }
        public void endVisit(PackageOrTypeName n) { unimplementedVisitor("endVisit(PackageOrTypeName)"); }

        public boolean visit(AmbiguousName n) { unimplementedVisitor("visit(AmbiguousName)"); return true; }
        public void endVisit(AmbiguousName n) { unimplementedVisitor("endVisit(AmbiguousName)"); }

        public boolean visit(CompilationUnit n) { unimplementedVisitor("visit(CompilationUnit)"); return true; }
        public void endVisit(CompilationUnit n) { unimplementedVisitor("endVisit(CompilationUnit)"); }

        public boolean visit(ImportDeclarations n) { unimplementedVisitor("visit(ImportDeclarations)"); return true; }
        public void endVisit(ImportDeclarations n) { unimplementedVisitor("endVisit(ImportDeclarations)"); }

        public boolean visit(TypeDeclarations n) { unimplementedVisitor("visit(TypeDeclarations)"); return true; }
        public void endVisit(TypeDeclarations n) { unimplementedVisitor("endVisit(TypeDeclarations)"); }

        public boolean visit(PackageDeclaration n) { unimplementedVisitor("visit(PackageDeclaration)"); return true; }
        public void endVisit(PackageDeclaration n) { unimplementedVisitor("endVisit(PackageDeclaration)"); }

        public boolean visit(SingleTypeImportDeclaration n) { unimplementedVisitor("visit(SingleTypeImportDeclaration)"); return true; }
        public void endVisit(SingleTypeImportDeclaration n) { unimplementedVisitor("endVisit(SingleTypeImportDeclaration)"); }

        public boolean visit(TypeImportOnDemandDeclaration n) { unimplementedVisitor("visit(TypeImportOnDemandDeclaration)"); return true; }
        public void endVisit(TypeImportOnDemandDeclaration n) { unimplementedVisitor("endVisit(TypeImportOnDemandDeclaration)"); }

        public boolean visit(SingleStaticImportDeclaration n) { unimplementedVisitor("visit(SingleStaticImportDeclaration)"); return true; }
        public void endVisit(SingleStaticImportDeclaration n) { unimplementedVisitor("endVisit(SingleStaticImportDeclaration)"); }

        public boolean visit(StaticImportOnDemandDeclaration n) { unimplementedVisitor("visit(StaticImportOnDemandDeclaration)"); return true; }
        public void endVisit(StaticImportOnDemandDeclaration n) { unimplementedVisitor("endVisit(StaticImportOnDemandDeclaration)"); }

        public boolean visit(TypeDeclaration n) { unimplementedVisitor("visit(TypeDeclaration)"); return true; }
        public void endVisit(TypeDeclaration n) { unimplementedVisitor("endVisit(TypeDeclaration)"); }

        public boolean visit(NormalClassDeclaration n) { unimplementedVisitor("visit(NormalClassDeclaration)"); return true; }
        public void endVisit(NormalClassDeclaration n) { unimplementedVisitor("endVisit(NormalClassDeclaration)"); }

        public boolean visit(ClassModifiers n) { unimplementedVisitor("visit(ClassModifiers)"); return true; }
        public void endVisit(ClassModifiers n) { unimplementedVisitor("endVisit(ClassModifiers)"); }

        public boolean visit(TypeParameters n) { unimplementedVisitor("visit(TypeParameters)"); return true; }
        public void endVisit(TypeParameters n) { unimplementedVisitor("endVisit(TypeParameters)"); }

        public boolean visit(TypeParameterList n) { unimplementedVisitor("visit(TypeParameterList)"); return true; }
        public void endVisit(TypeParameterList n) { unimplementedVisitor("endVisit(TypeParameterList)"); }

        public boolean visit(Super n) { unimplementedVisitor("visit(Super)"); return true; }
        public void endVisit(Super n) { unimplementedVisitor("endVisit(Super)"); }

        public boolean visit(Interfaces n) { unimplementedVisitor("visit(Interfaces)"); return true; }
        public void endVisit(Interfaces n) { unimplementedVisitor("endVisit(Interfaces)"); }

        public boolean visit(InterfaceTypeList n) { unimplementedVisitor("visit(InterfaceTypeList)"); return true; }
        public void endVisit(InterfaceTypeList n) { unimplementedVisitor("endVisit(InterfaceTypeList)"); }

        public boolean visit(ClassBody n) { unimplementedVisitor("visit(ClassBody)"); return true; }
        public void endVisit(ClassBody n) { unimplementedVisitor("endVisit(ClassBody)"); }

        public boolean visit(ClassBodyDeclarations n) { unimplementedVisitor("visit(ClassBodyDeclarations)"); return true; }
        public void endVisit(ClassBodyDeclarations n) { unimplementedVisitor("endVisit(ClassBodyDeclarations)"); }

        public boolean visit(ClassMemberDeclaration n) { unimplementedVisitor("visit(ClassMemberDeclaration)"); return true; }
        public void endVisit(ClassMemberDeclaration n) { unimplementedVisitor("endVisit(ClassMemberDeclaration)"); }

        public boolean visit(FieldDeclaration n) { unimplementedVisitor("visit(FieldDeclaration)"); return true; }
        public void endVisit(FieldDeclaration n) { unimplementedVisitor("endVisit(FieldDeclaration)"); }

        public boolean visit(VariableDeclarators n) { unimplementedVisitor("visit(VariableDeclarators)"); return true; }
        public void endVisit(VariableDeclarators n) { unimplementedVisitor("endVisit(VariableDeclarators)"); }

        public boolean visit(VariableDeclarator n) { unimplementedVisitor("visit(VariableDeclarator)"); return true; }
        public void endVisit(VariableDeclarator n) { unimplementedVisitor("endVisit(VariableDeclarator)"); }

        public boolean visit(VariableDeclaratorId n) { unimplementedVisitor("visit(VariableDeclaratorId)"); return true; }
        public void endVisit(VariableDeclaratorId n) { unimplementedVisitor("endVisit(VariableDeclaratorId)"); }

        public boolean visit(FieldModifiers n) { unimplementedVisitor("visit(FieldModifiers)"); return true; }
        public void endVisit(FieldModifiers n) { unimplementedVisitor("endVisit(FieldModifiers)"); }

        public boolean visit(MethodDeclaration n) { unimplementedVisitor("visit(MethodDeclaration)"); return true; }
        public void endVisit(MethodDeclaration n) { unimplementedVisitor("endVisit(MethodDeclaration)"); }

        public boolean visit(MethodHeader n) { unimplementedVisitor("visit(MethodHeader)"); return true; }
        public void endVisit(MethodHeader n) { unimplementedVisitor("endVisit(MethodHeader)"); }

        public boolean visit(ResultType n) { unimplementedVisitor("visit(ResultType)"); return true; }
        public void endVisit(ResultType n) { unimplementedVisitor("endVisit(ResultType)"); }

        public boolean visit(FormalParameterList n) { unimplementedVisitor("visit(FormalParameterList)"); return true; }
        public void endVisit(FormalParameterList n) { unimplementedVisitor("endVisit(FormalParameterList)"); }

        public boolean visit(FormalParameters n) { unimplementedVisitor("visit(FormalParameters)"); return true; }
        public void endVisit(FormalParameters n) { unimplementedVisitor("endVisit(FormalParameters)"); }

        public boolean visit(FormalParameter n) { unimplementedVisitor("visit(FormalParameter)"); return true; }
        public void endVisit(FormalParameter n) { unimplementedVisitor("endVisit(FormalParameter)"); }

        public boolean visit(VariableModifiers n) { unimplementedVisitor("visit(VariableModifiers)"); return true; }
        public void endVisit(VariableModifiers n) { unimplementedVisitor("endVisit(VariableModifiers)"); }

        public boolean visit(VariableModifier n) { unimplementedVisitor("visit(VariableModifier)"); return true; }
        public void endVisit(VariableModifier n) { unimplementedVisitor("endVisit(VariableModifier)"); }

        public boolean visit(LastFormalParameter n) { unimplementedVisitor("visit(LastFormalParameter)"); return true; }
        public void endVisit(LastFormalParameter n) { unimplementedVisitor("endVisit(LastFormalParameter)"); }

        public boolean visit(MethodModifiers n) { unimplementedVisitor("visit(MethodModifiers)"); return true; }
        public void endVisit(MethodModifiers n) { unimplementedVisitor("endVisit(MethodModifiers)"); }

        public boolean visit(Throws n) { unimplementedVisitor("visit(Throws)"); return true; }
        public void endVisit(Throws n) { unimplementedVisitor("endVisit(Throws)"); }

        public boolean visit(ExceptionTypeList n) { unimplementedVisitor("visit(ExceptionTypeList)"); return true; }
        public void endVisit(ExceptionTypeList n) { unimplementedVisitor("endVisit(ExceptionTypeList)"); }

        public boolean visit(MethodBody n) { unimplementedVisitor("visit(MethodBody)"); return true; }
        public void endVisit(MethodBody n) { unimplementedVisitor("endVisit(MethodBody)"); }

        public boolean visit(StaticInitializer n) { unimplementedVisitor("visit(StaticInitializer)"); return true; }
        public void endVisit(StaticInitializer n) { unimplementedVisitor("endVisit(StaticInitializer)"); }

        public boolean visit(ConstructorDeclaration n) { unimplementedVisitor("visit(ConstructorDeclaration)"); return true; }
        public void endVisit(ConstructorDeclaration n) { unimplementedVisitor("endVisit(ConstructorDeclaration)"); }

        public boolean visit(ConstructorDeclarator n) { unimplementedVisitor("visit(ConstructorDeclarator)"); return true; }
        public void endVisit(ConstructorDeclarator n) { unimplementedVisitor("endVisit(ConstructorDeclarator)"); }

        public boolean visit(ConstructorModifiers n) { unimplementedVisitor("visit(ConstructorModifiers)"); return true; }
        public void endVisit(ConstructorModifiers n) { unimplementedVisitor("endVisit(ConstructorModifiers)"); }

        public boolean visit(ConstructorBody n) { unimplementedVisitor("visit(ConstructorBody)"); return true; }
        public void endVisit(ConstructorBody n) { unimplementedVisitor("endVisit(ConstructorBody)"); }

        public boolean visit(EnumDeclaration n) { unimplementedVisitor("visit(EnumDeclaration)"); return true; }
        public void endVisit(EnumDeclaration n) { unimplementedVisitor("endVisit(EnumDeclaration)"); }

        public boolean visit(EnumBody n) { unimplementedVisitor("visit(EnumBody)"); return true; }
        public void endVisit(EnumBody n) { unimplementedVisitor("endVisit(EnumBody)"); }

        public boolean visit(EnumConstants n) { unimplementedVisitor("visit(EnumConstants)"); return true; }
        public void endVisit(EnumConstants n) { unimplementedVisitor("endVisit(EnumConstants)"); }

        public boolean visit(EnumConstant n) { unimplementedVisitor("visit(EnumConstant)"); return true; }
        public void endVisit(EnumConstant n) { unimplementedVisitor("endVisit(EnumConstant)"); }

        public boolean visit(Arguments n) { unimplementedVisitor("visit(Arguments)"); return true; }
        public void endVisit(Arguments n) { unimplementedVisitor("endVisit(Arguments)"); }

        public boolean visit(EnumBodyDeclarations n) { unimplementedVisitor("visit(EnumBodyDeclarations)"); return true; }
        public void endVisit(EnumBodyDeclarations n) { unimplementedVisitor("endVisit(EnumBodyDeclarations)"); }

        public boolean visit(NormalInterfaceDeclaration n) { unimplementedVisitor("visit(NormalInterfaceDeclaration)"); return true; }
        public void endVisit(NormalInterfaceDeclaration n) { unimplementedVisitor("endVisit(NormalInterfaceDeclaration)"); }

        public boolean visit(InterfaceModifiers n) { unimplementedVisitor("visit(InterfaceModifiers)"); return true; }
        public void endVisit(InterfaceModifiers n) { unimplementedVisitor("endVisit(InterfaceModifiers)"); }

        public boolean visit(InterfaceBody n) { unimplementedVisitor("visit(InterfaceBody)"); return true; }
        public void endVisit(InterfaceBody n) { unimplementedVisitor("endVisit(InterfaceBody)"); }

        public boolean visit(InterfaceMemberDeclarations n) { unimplementedVisitor("visit(InterfaceMemberDeclarations)"); return true; }
        public void endVisit(InterfaceMemberDeclarations n) { unimplementedVisitor("endVisit(InterfaceMemberDeclarations)"); }

        public boolean visit(InterfaceMemberDeclaration n) { unimplementedVisitor("visit(InterfaceMemberDeclaration)"); return true; }
        public void endVisit(InterfaceMemberDeclaration n) { unimplementedVisitor("endVisit(InterfaceMemberDeclaration)"); }

        public boolean visit(ConstantDeclaration n) { unimplementedVisitor("visit(ConstantDeclaration)"); return true; }
        public void endVisit(ConstantDeclaration n) { unimplementedVisitor("endVisit(ConstantDeclaration)"); }

        public boolean visit(ConstantModifiers n) { unimplementedVisitor("visit(ConstantModifiers)"); return true; }
        public void endVisit(ConstantModifiers n) { unimplementedVisitor("endVisit(ConstantModifiers)"); }

        public boolean visit(AbstractMethodDeclaration n) { unimplementedVisitor("visit(AbstractMethodDeclaration)"); return true; }
        public void endVisit(AbstractMethodDeclaration n) { unimplementedVisitor("endVisit(AbstractMethodDeclaration)"); }

        public boolean visit(AbstractMethodModifiers n) { unimplementedVisitor("visit(AbstractMethodModifiers)"); return true; }
        public void endVisit(AbstractMethodModifiers n) { unimplementedVisitor("endVisit(AbstractMethodModifiers)"); }

        public boolean visit(AnnotationTypeDeclaration n) { unimplementedVisitor("visit(AnnotationTypeDeclaration)"); return true; }
        public void endVisit(AnnotationTypeDeclaration n) { unimplementedVisitor("endVisit(AnnotationTypeDeclaration)"); }

        public boolean visit(AnnotationTypeBody n) { unimplementedVisitor("visit(AnnotationTypeBody)"); return true; }
        public void endVisit(AnnotationTypeBody n) { unimplementedVisitor("endVisit(AnnotationTypeBody)"); }

        public boolean visit(AnnotationTypeElementDeclarations n) { unimplementedVisitor("visit(AnnotationTypeElementDeclarations)"); return true; }
        public void endVisit(AnnotationTypeElementDeclarations n) { unimplementedVisitor("endVisit(AnnotationTypeElementDeclarations)"); }

        public boolean visit(DefaultValue n) { unimplementedVisitor("visit(DefaultValue)"); return true; }
        public void endVisit(DefaultValue n) { unimplementedVisitor("endVisit(DefaultValue)"); }

        public boolean visit(Annotations n) { unimplementedVisitor("visit(Annotations)"); return true; }
        public void endVisit(Annotations n) { unimplementedVisitor("endVisit(Annotations)"); }

        public boolean visit(NormalAnnotation n) { unimplementedVisitor("visit(NormalAnnotation)"); return true; }
        public void endVisit(NormalAnnotation n) { unimplementedVisitor("endVisit(NormalAnnotation)"); }

        public boolean visit(ElementValuePairs n) { unimplementedVisitor("visit(ElementValuePairs)"); return true; }
        public void endVisit(ElementValuePairs n) { unimplementedVisitor("endVisit(ElementValuePairs)"); }

        public boolean visit(ElementValuePair n) { unimplementedVisitor("visit(ElementValuePair)"); return true; }
        public void endVisit(ElementValuePair n) { unimplementedVisitor("endVisit(ElementValuePair)"); }

        public boolean visit(ElementValueArrayInitializer n) { unimplementedVisitor("visit(ElementValueArrayInitializer)"); return true; }
        public void endVisit(ElementValueArrayInitializer n) { unimplementedVisitor("endVisit(ElementValueArrayInitializer)"); }

        public boolean visit(ElementValues n) { unimplementedVisitor("visit(ElementValues)"); return true; }
        public void endVisit(ElementValues n) { unimplementedVisitor("endVisit(ElementValues)"); }

        public boolean visit(MarkerAnnotation n) { unimplementedVisitor("visit(MarkerAnnotation)"); return true; }
        public void endVisit(MarkerAnnotation n) { unimplementedVisitor("endVisit(MarkerAnnotation)"); }

        public boolean visit(SingleElementAnnotation n) { unimplementedVisitor("visit(SingleElementAnnotation)"); return true; }
        public void endVisit(SingleElementAnnotation n) { unimplementedVisitor("endVisit(SingleElementAnnotation)"); }

        public boolean visit(ArrayInitializer n) { unimplementedVisitor("visit(ArrayInitializer)"); return true; }
        public void endVisit(ArrayInitializer n) { unimplementedVisitor("endVisit(ArrayInitializer)"); }

        public boolean visit(VariableInitializers n) { unimplementedVisitor("visit(VariableInitializers)"); return true; }
        public void endVisit(VariableInitializers n) { unimplementedVisitor("endVisit(VariableInitializers)"); }

        public boolean visit(Block n) { unimplementedVisitor("visit(Block)"); return true; }
        public void endVisit(Block n) { unimplementedVisitor("endVisit(Block)"); }

        public boolean visit(BlockStatements n) { unimplementedVisitor("visit(BlockStatements)"); return true; }
        public void endVisit(BlockStatements n) { unimplementedVisitor("endVisit(BlockStatements)"); }

        public boolean visit(LocalVariableDeclarationStatement n) { unimplementedVisitor("visit(LocalVariableDeclarationStatement)"); return true; }
        public void endVisit(LocalVariableDeclarationStatement n) { unimplementedVisitor("endVisit(LocalVariableDeclarationStatement)"); }

        public boolean visit(LocalVariableDeclaration n) { unimplementedVisitor("visit(LocalVariableDeclaration)"); return true; }
        public void endVisit(LocalVariableDeclaration n) { unimplementedVisitor("endVisit(LocalVariableDeclaration)"); }

        public boolean visit(IfThenStatement n) { unimplementedVisitor("visit(IfThenStatement)"); return true; }
        public void endVisit(IfThenStatement n) { unimplementedVisitor("endVisit(IfThenStatement)"); }

        public boolean visit(IfThenElseStatement n) { unimplementedVisitor("visit(IfThenElseStatement)"); return true; }
        public void endVisit(IfThenElseStatement n) { unimplementedVisitor("endVisit(IfThenElseStatement)"); }

        public boolean visit(IfThenElseStatementNoShortIf n) { unimplementedVisitor("visit(IfThenElseStatementNoShortIf)"); return true; }
        public void endVisit(IfThenElseStatementNoShortIf n) { unimplementedVisitor("endVisit(IfThenElseStatementNoShortIf)"); }

        public boolean visit(EmptyStatement n) { unimplementedVisitor("visit(EmptyStatement)"); return true; }
        public void endVisit(EmptyStatement n) { unimplementedVisitor("endVisit(EmptyStatement)"); }

        public boolean visit(LabeledStatement n) { unimplementedVisitor("visit(LabeledStatement)"); return true; }
        public void endVisit(LabeledStatement n) { unimplementedVisitor("endVisit(LabeledStatement)"); }

        public boolean visit(LabeledStatementNoShortIf n) { unimplementedVisitor("visit(LabeledStatementNoShortIf)"); return true; }
        public void endVisit(LabeledStatementNoShortIf n) { unimplementedVisitor("endVisit(LabeledStatementNoShortIf)"); }

        public boolean visit(ExpressionStatement n) { unimplementedVisitor("visit(ExpressionStatement)"); return true; }
        public void endVisit(ExpressionStatement n) { unimplementedVisitor("endVisit(ExpressionStatement)"); }

        public boolean visit(SwitchStatement n) { unimplementedVisitor("visit(SwitchStatement)"); return true; }
        public void endVisit(SwitchStatement n) { unimplementedVisitor("endVisit(SwitchStatement)"); }

        public boolean visit(SwitchBlock n) { unimplementedVisitor("visit(SwitchBlock)"); return true; }
        public void endVisit(SwitchBlock n) { unimplementedVisitor("endVisit(SwitchBlock)"); }

        public boolean visit(SwitchBlockStatementGroups n) { unimplementedVisitor("visit(SwitchBlockStatementGroups)"); return true; }
        public void endVisit(SwitchBlockStatementGroups n) { unimplementedVisitor("endVisit(SwitchBlockStatementGroups)"); }

        public boolean visit(SwitchBlockStatementGroup n) { unimplementedVisitor("visit(SwitchBlockStatementGroup)"); return true; }
        public void endVisit(SwitchBlockStatementGroup n) { unimplementedVisitor("endVisit(SwitchBlockStatementGroup)"); }

        public boolean visit(SwitchLabels n) { unimplementedVisitor("visit(SwitchLabels)"); return true; }
        public void endVisit(SwitchLabels n) { unimplementedVisitor("endVisit(SwitchLabels)"); }

        public boolean visit(WhileStatement n) { unimplementedVisitor("visit(WhileStatement)"); return true; }
        public void endVisit(WhileStatement n) { unimplementedVisitor("endVisit(WhileStatement)"); }

        public boolean visit(WhileStatementNoShortIf n) { unimplementedVisitor("visit(WhileStatementNoShortIf)"); return true; }
        public void endVisit(WhileStatementNoShortIf n) { unimplementedVisitor("endVisit(WhileStatementNoShortIf)"); }

        public boolean visit(DoStatement n) { unimplementedVisitor("visit(DoStatement)"); return true; }
        public void endVisit(DoStatement n) { unimplementedVisitor("endVisit(DoStatement)"); }

        public boolean visit(BasicForStatement n) { unimplementedVisitor("visit(BasicForStatement)"); return true; }
        public void endVisit(BasicForStatement n) { unimplementedVisitor("endVisit(BasicForStatement)"); }

        public boolean visit(ForStatementNoShortIf n) { unimplementedVisitor("visit(ForStatementNoShortIf)"); return true; }
        public void endVisit(ForStatementNoShortIf n) { unimplementedVisitor("endVisit(ForStatementNoShortIf)"); }

        public boolean visit(StatementExpressionList n) { unimplementedVisitor("visit(StatementExpressionList)"); return true; }
        public void endVisit(StatementExpressionList n) { unimplementedVisitor("endVisit(StatementExpressionList)"); }

        public boolean visit(EnhancedForStatement n) { unimplementedVisitor("visit(EnhancedForStatement)"); return true; }
        public void endVisit(EnhancedForStatement n) { unimplementedVisitor("endVisit(EnhancedForStatement)"); }

        public boolean visit(BreakStatement n) { unimplementedVisitor("visit(BreakStatement)"); return true; }
        public void endVisit(BreakStatement n) { unimplementedVisitor("endVisit(BreakStatement)"); }

        public boolean visit(ContinueStatement n) { unimplementedVisitor("visit(ContinueStatement)"); return true; }
        public void endVisit(ContinueStatement n) { unimplementedVisitor("endVisit(ContinueStatement)"); }

        public boolean visit(ReturnStatement n) { unimplementedVisitor("visit(ReturnStatement)"); return true; }
        public void endVisit(ReturnStatement n) { unimplementedVisitor("endVisit(ReturnStatement)"); }

        public boolean visit(ThrowStatement n) { unimplementedVisitor("visit(ThrowStatement)"); return true; }
        public void endVisit(ThrowStatement n) { unimplementedVisitor("endVisit(ThrowStatement)"); }

        public boolean visit(SynchronizedStatement n) { unimplementedVisitor("visit(SynchronizedStatement)"); return true; }
        public void endVisit(SynchronizedStatement n) { unimplementedVisitor("endVisit(SynchronizedStatement)"); }

        public boolean visit(Catches n) { unimplementedVisitor("visit(Catches)"); return true; }
        public void endVisit(Catches n) { unimplementedVisitor("endVisit(Catches)"); }

        public boolean visit(CatchClause n) { unimplementedVisitor("visit(CatchClause)"); return true; }
        public void endVisit(CatchClause n) { unimplementedVisitor("endVisit(CatchClause)"); }

        public boolean visit(Finally n) { unimplementedVisitor("visit(Finally)"); return true; }
        public void endVisit(Finally n) { unimplementedVisitor("endVisit(Finally)"); }

        public boolean visit(ArgumentList n) { unimplementedVisitor("visit(ArgumentList)"); return true; }
        public void endVisit(ArgumentList n) { unimplementedVisitor("endVisit(ArgumentList)"); }

        public boolean visit(DimExprs n) { unimplementedVisitor("visit(DimExprs)"); return true; }
        public void endVisit(DimExprs n) { unimplementedVisitor("endVisit(DimExprs)"); }

        public boolean visit(DimExpr n) { unimplementedVisitor("visit(DimExpr)"); return true; }
        public void endVisit(DimExpr n) { unimplementedVisitor("endVisit(DimExpr)"); }

        public boolean visit(PostIncrementExpression n) { unimplementedVisitor("visit(PostIncrementExpression)"); return true; }
        public void endVisit(PostIncrementExpression n) { unimplementedVisitor("endVisit(PostIncrementExpression)"); }

        public boolean visit(PostDecrementExpression n) { unimplementedVisitor("visit(PostDecrementExpression)"); return true; }
        public void endVisit(PostDecrementExpression n) { unimplementedVisitor("endVisit(PostDecrementExpression)"); }

        public boolean visit(PreIncrementExpression n) { unimplementedVisitor("visit(PreIncrementExpression)"); return true; }
        public void endVisit(PreIncrementExpression n) { unimplementedVisitor("endVisit(PreIncrementExpression)"); }

        public boolean visit(PreDecrementExpression n) { unimplementedVisitor("visit(PreDecrementExpression)"); return true; }
        public void endVisit(PreDecrementExpression n) { unimplementedVisitor("endVisit(PreDecrementExpression)"); }

        public boolean visit(AndExpression n) { unimplementedVisitor("visit(AndExpression)"); return true; }
        public void endVisit(AndExpression n) { unimplementedVisitor("endVisit(AndExpression)"); }

        public boolean visit(ExclusiveOrExpression n) { unimplementedVisitor("visit(ExclusiveOrExpression)"); return true; }
        public void endVisit(ExclusiveOrExpression n) { unimplementedVisitor("endVisit(ExclusiveOrExpression)"); }

        public boolean visit(InclusiveOrExpression n) { unimplementedVisitor("visit(InclusiveOrExpression)"); return true; }
        public void endVisit(InclusiveOrExpression n) { unimplementedVisitor("endVisit(InclusiveOrExpression)"); }

        public boolean visit(ConditionalAndExpression n) { unimplementedVisitor("visit(ConditionalAndExpression)"); return true; }
        public void endVisit(ConditionalAndExpression n) { unimplementedVisitor("endVisit(ConditionalAndExpression)"); }

        public boolean visit(ConditionalOrExpression n) { unimplementedVisitor("visit(ConditionalOrExpression)"); return true; }
        public void endVisit(ConditionalOrExpression n) { unimplementedVisitor("endVisit(ConditionalOrExpression)"); }

        public boolean visit(ConditionalExpression n) { unimplementedVisitor("visit(ConditionalExpression)"); return true; }
        public void endVisit(ConditionalExpression n) { unimplementedVisitor("endVisit(ConditionalExpression)"); }

        public boolean visit(Assignment n) { unimplementedVisitor("visit(Assignment)"); return true; }
        public void endVisit(Assignment n) { unimplementedVisitor("endVisit(Assignment)"); }

        public boolean visit(Commaopt n) { unimplementedVisitor("visit(Commaopt)"); return true; }
        public void endVisit(Commaopt n) { unimplementedVisitor("endVisit(Commaopt)"); }

        public boolean visit(Ellipsisopt n) { unimplementedVisitor("visit(Ellipsisopt)"); return true; }
        public void endVisit(Ellipsisopt n) { unimplementedVisitor("endVisit(Ellipsisopt)"); }

        public boolean visit(LPGUserAction0 n) { unimplementedVisitor("visit(LPGUserAction0)"); return true; }
        public void endVisit(LPGUserAction0 n) { unimplementedVisitor("endVisit(LPGUserAction0)"); }

        public boolean visit(LPGUserAction1 n) { unimplementedVisitor("visit(LPGUserAction1)"); return true; }
        public void endVisit(LPGUserAction1 n) { unimplementedVisitor("endVisit(LPGUserAction1)"); }

        public boolean visit(LPGUserAction2 n) { unimplementedVisitor("visit(LPGUserAction2)"); return true; }
        public void endVisit(LPGUserAction2 n) { unimplementedVisitor("endVisit(LPGUserAction2)"); }

        public boolean visit(LPGUserAction3 n) { unimplementedVisitor("visit(LPGUserAction3)"); return true; }
        public void endVisit(LPGUserAction3 n) { unimplementedVisitor("endVisit(LPGUserAction3)"); }

        public boolean visit(LPGUserAction4 n) { unimplementedVisitor("visit(LPGUserAction4)"); return true; }
        public void endVisit(LPGUserAction4 n) { unimplementedVisitor("endVisit(LPGUserAction4)"); }

        public boolean visit(IntegralType0 n) { unimplementedVisitor("visit(IntegralType0)"); return true; }
        public void endVisit(IntegralType0 n) { unimplementedVisitor("endVisit(IntegralType0)"); }

        public boolean visit(IntegralType1 n) { unimplementedVisitor("visit(IntegralType1)"); return true; }
        public void endVisit(IntegralType1 n) { unimplementedVisitor("endVisit(IntegralType1)"); }

        public boolean visit(IntegralType2 n) { unimplementedVisitor("visit(IntegralType2)"); return true; }
        public void endVisit(IntegralType2 n) { unimplementedVisitor("endVisit(IntegralType2)"); }

        public boolean visit(IntegralType3 n) { unimplementedVisitor("visit(IntegralType3)"); return true; }
        public void endVisit(IntegralType3 n) { unimplementedVisitor("endVisit(IntegralType3)"); }

        public boolean visit(IntegralType4 n) { unimplementedVisitor("visit(IntegralType4)"); return true; }
        public void endVisit(IntegralType4 n) { unimplementedVisitor("endVisit(IntegralType4)"); }

        public boolean visit(FloatingPointType0 n) { unimplementedVisitor("visit(FloatingPointType0)"); return true; }
        public void endVisit(FloatingPointType0 n) { unimplementedVisitor("endVisit(FloatingPointType0)"); }

        public boolean visit(FloatingPointType1 n) { unimplementedVisitor("visit(FloatingPointType1)"); return true; }
        public void endVisit(FloatingPointType1 n) { unimplementedVisitor("endVisit(FloatingPointType1)"); }

        public boolean visit(WildcardBounds0 n) { unimplementedVisitor("visit(WildcardBounds0)"); return true; }
        public void endVisit(WildcardBounds0 n) { unimplementedVisitor("endVisit(WildcardBounds0)"); }

        public boolean visit(WildcardBounds1 n) { unimplementedVisitor("visit(WildcardBounds1)"); return true; }
        public void endVisit(WildcardBounds1 n) { unimplementedVisitor("endVisit(WildcardBounds1)"); }

        public boolean visit(ClassModifier0 n) { unimplementedVisitor("visit(ClassModifier0)"); return true; }
        public void endVisit(ClassModifier0 n) { unimplementedVisitor("endVisit(ClassModifier0)"); }

        public boolean visit(ClassModifier1 n) { unimplementedVisitor("visit(ClassModifier1)"); return true; }
        public void endVisit(ClassModifier1 n) { unimplementedVisitor("endVisit(ClassModifier1)"); }

        public boolean visit(ClassModifier2 n) { unimplementedVisitor("visit(ClassModifier2)"); return true; }
        public void endVisit(ClassModifier2 n) { unimplementedVisitor("endVisit(ClassModifier2)"); }

        public boolean visit(ClassModifier3 n) { unimplementedVisitor("visit(ClassModifier3)"); return true; }
        public void endVisit(ClassModifier3 n) { unimplementedVisitor("endVisit(ClassModifier3)"); }

        public boolean visit(ClassModifier4 n) { unimplementedVisitor("visit(ClassModifier4)"); return true; }
        public void endVisit(ClassModifier4 n) { unimplementedVisitor("endVisit(ClassModifier4)"); }

        public boolean visit(ClassModifier5 n) { unimplementedVisitor("visit(ClassModifier5)"); return true; }
        public void endVisit(ClassModifier5 n) { unimplementedVisitor("endVisit(ClassModifier5)"); }

        public boolean visit(ClassModifier6 n) { unimplementedVisitor("visit(ClassModifier6)"); return true; }
        public void endVisit(ClassModifier6 n) { unimplementedVisitor("endVisit(ClassModifier6)"); }

        public boolean visit(FieldModifier0 n) { unimplementedVisitor("visit(FieldModifier0)"); return true; }
        public void endVisit(FieldModifier0 n) { unimplementedVisitor("endVisit(FieldModifier0)"); }

        public boolean visit(FieldModifier1 n) { unimplementedVisitor("visit(FieldModifier1)"); return true; }
        public void endVisit(FieldModifier1 n) { unimplementedVisitor("endVisit(FieldModifier1)"); }

        public boolean visit(FieldModifier2 n) { unimplementedVisitor("visit(FieldModifier2)"); return true; }
        public void endVisit(FieldModifier2 n) { unimplementedVisitor("endVisit(FieldModifier2)"); }

        public boolean visit(FieldModifier3 n) { unimplementedVisitor("visit(FieldModifier3)"); return true; }
        public void endVisit(FieldModifier3 n) { unimplementedVisitor("endVisit(FieldModifier3)"); }

        public boolean visit(FieldModifier4 n) { unimplementedVisitor("visit(FieldModifier4)"); return true; }
        public void endVisit(FieldModifier4 n) { unimplementedVisitor("endVisit(FieldModifier4)"); }

        public boolean visit(FieldModifier5 n) { unimplementedVisitor("visit(FieldModifier5)"); return true; }
        public void endVisit(FieldModifier5 n) { unimplementedVisitor("endVisit(FieldModifier5)"); }

        public boolean visit(FieldModifier6 n) { unimplementedVisitor("visit(FieldModifier6)"); return true; }
        public void endVisit(FieldModifier6 n) { unimplementedVisitor("endVisit(FieldModifier6)"); }

        public boolean visit(MethodDeclarator0 n) { unimplementedVisitor("visit(MethodDeclarator0)"); return true; }
        public void endVisit(MethodDeclarator0 n) { unimplementedVisitor("endVisit(MethodDeclarator0)"); }

        public boolean visit(MethodDeclarator1 n) { unimplementedVisitor("visit(MethodDeclarator1)"); return true; }
        public void endVisit(MethodDeclarator1 n) { unimplementedVisitor("endVisit(MethodDeclarator1)"); }

        public boolean visit(MethodModifier0 n) { unimplementedVisitor("visit(MethodModifier0)"); return true; }
        public void endVisit(MethodModifier0 n) { unimplementedVisitor("endVisit(MethodModifier0)"); }

        public boolean visit(MethodModifier1 n) { unimplementedVisitor("visit(MethodModifier1)"); return true; }
        public void endVisit(MethodModifier1 n) { unimplementedVisitor("endVisit(MethodModifier1)"); }

        public boolean visit(MethodModifier2 n) { unimplementedVisitor("visit(MethodModifier2)"); return true; }
        public void endVisit(MethodModifier2 n) { unimplementedVisitor("endVisit(MethodModifier2)"); }

        public boolean visit(MethodModifier3 n) { unimplementedVisitor("visit(MethodModifier3)"); return true; }
        public void endVisit(MethodModifier3 n) { unimplementedVisitor("endVisit(MethodModifier3)"); }

        public boolean visit(MethodModifier4 n) { unimplementedVisitor("visit(MethodModifier4)"); return true; }
        public void endVisit(MethodModifier4 n) { unimplementedVisitor("endVisit(MethodModifier4)"); }

        public boolean visit(MethodModifier5 n) { unimplementedVisitor("visit(MethodModifier5)"); return true; }
        public void endVisit(MethodModifier5 n) { unimplementedVisitor("endVisit(MethodModifier5)"); }

        public boolean visit(MethodModifier6 n) { unimplementedVisitor("visit(MethodModifier6)"); return true; }
        public void endVisit(MethodModifier6 n) { unimplementedVisitor("endVisit(MethodModifier6)"); }

        public boolean visit(MethodModifier7 n) { unimplementedVisitor("visit(MethodModifier7)"); return true; }
        public void endVisit(MethodModifier7 n) { unimplementedVisitor("endVisit(MethodModifier7)"); }

        public boolean visit(MethodModifier8 n) { unimplementedVisitor("visit(MethodModifier8)"); return true; }
        public void endVisit(MethodModifier8 n) { unimplementedVisitor("endVisit(MethodModifier8)"); }

        public boolean visit(ConstructorModifier0 n) { unimplementedVisitor("visit(ConstructorModifier0)"); return true; }
        public void endVisit(ConstructorModifier0 n) { unimplementedVisitor("endVisit(ConstructorModifier0)"); }

        public boolean visit(ConstructorModifier1 n) { unimplementedVisitor("visit(ConstructorModifier1)"); return true; }
        public void endVisit(ConstructorModifier1 n) { unimplementedVisitor("endVisit(ConstructorModifier1)"); }

        public boolean visit(ConstructorModifier2 n) { unimplementedVisitor("visit(ConstructorModifier2)"); return true; }
        public void endVisit(ConstructorModifier2 n) { unimplementedVisitor("endVisit(ConstructorModifier2)"); }

        public boolean visit(ExplicitConstructorInvocation0 n) { unimplementedVisitor("visit(ExplicitConstructorInvocation0)"); return true; }
        public void endVisit(ExplicitConstructorInvocation0 n) { unimplementedVisitor("endVisit(ExplicitConstructorInvocation0)"); }

        public boolean visit(ExplicitConstructorInvocation1 n) { unimplementedVisitor("visit(ExplicitConstructorInvocation1)"); return true; }
        public void endVisit(ExplicitConstructorInvocation1 n) { unimplementedVisitor("endVisit(ExplicitConstructorInvocation1)"); }

        public boolean visit(ExplicitConstructorInvocation2 n) { unimplementedVisitor("visit(ExplicitConstructorInvocation2)"); return true; }
        public void endVisit(ExplicitConstructorInvocation2 n) { unimplementedVisitor("endVisit(ExplicitConstructorInvocation2)"); }

        public boolean visit(InterfaceModifier0 n) { unimplementedVisitor("visit(InterfaceModifier0)"); return true; }
        public void endVisit(InterfaceModifier0 n) { unimplementedVisitor("endVisit(InterfaceModifier0)"); }

        public boolean visit(InterfaceModifier1 n) { unimplementedVisitor("visit(InterfaceModifier1)"); return true; }
        public void endVisit(InterfaceModifier1 n) { unimplementedVisitor("endVisit(InterfaceModifier1)"); }

        public boolean visit(InterfaceModifier2 n) { unimplementedVisitor("visit(InterfaceModifier2)"); return true; }
        public void endVisit(InterfaceModifier2 n) { unimplementedVisitor("endVisit(InterfaceModifier2)"); }

        public boolean visit(InterfaceModifier3 n) { unimplementedVisitor("visit(InterfaceModifier3)"); return true; }
        public void endVisit(InterfaceModifier3 n) { unimplementedVisitor("endVisit(InterfaceModifier3)"); }

        public boolean visit(InterfaceModifier4 n) { unimplementedVisitor("visit(InterfaceModifier4)"); return true; }
        public void endVisit(InterfaceModifier4 n) { unimplementedVisitor("endVisit(InterfaceModifier4)"); }

        public boolean visit(InterfaceModifier5 n) { unimplementedVisitor("visit(InterfaceModifier5)"); return true; }
        public void endVisit(InterfaceModifier5 n) { unimplementedVisitor("endVisit(InterfaceModifier5)"); }

        public boolean visit(ExtendsInterfaces0 n) { unimplementedVisitor("visit(ExtendsInterfaces0)"); return true; }
        public void endVisit(ExtendsInterfaces0 n) { unimplementedVisitor("endVisit(ExtendsInterfaces0)"); }

        public boolean visit(ExtendsInterfaces1 n) { unimplementedVisitor("visit(ExtendsInterfaces1)"); return true; }
        public void endVisit(ExtendsInterfaces1 n) { unimplementedVisitor("endVisit(ExtendsInterfaces1)"); }

        public boolean visit(ConstantModifier0 n) { unimplementedVisitor("visit(ConstantModifier0)"); return true; }
        public void endVisit(ConstantModifier0 n) { unimplementedVisitor("endVisit(ConstantModifier0)"); }

        public boolean visit(ConstantModifier1 n) { unimplementedVisitor("visit(ConstantModifier1)"); return true; }
        public void endVisit(ConstantModifier1 n) { unimplementedVisitor("endVisit(ConstantModifier1)"); }

        public boolean visit(ConstantModifier2 n) { unimplementedVisitor("visit(ConstantModifier2)"); return true; }
        public void endVisit(ConstantModifier2 n) { unimplementedVisitor("endVisit(ConstantModifier2)"); }

        public boolean visit(AbstractMethodModifier0 n) { unimplementedVisitor("visit(AbstractMethodModifier0)"); return true; }
        public void endVisit(AbstractMethodModifier0 n) { unimplementedVisitor("endVisit(AbstractMethodModifier0)"); }

        public boolean visit(AbstractMethodModifier1 n) { unimplementedVisitor("visit(AbstractMethodModifier1)"); return true; }
        public void endVisit(AbstractMethodModifier1 n) { unimplementedVisitor("endVisit(AbstractMethodModifier1)"); }

        public boolean visit(AnnotationTypeElementDeclaration0 n) { unimplementedVisitor("visit(AnnotationTypeElementDeclaration0)"); return true; }
        public void endVisit(AnnotationTypeElementDeclaration0 n) { unimplementedVisitor("endVisit(AnnotationTypeElementDeclaration0)"); }

        public boolean visit(AnnotationTypeElementDeclaration1 n) { unimplementedVisitor("visit(AnnotationTypeElementDeclaration1)"); return true; }
        public void endVisit(AnnotationTypeElementDeclaration1 n) { unimplementedVisitor("endVisit(AnnotationTypeElementDeclaration1)"); }

        public boolean visit(AssertStatement0 n) { unimplementedVisitor("visit(AssertStatement0)"); return true; }
        public void endVisit(AssertStatement0 n) { unimplementedVisitor("endVisit(AssertStatement0)"); }

        public boolean visit(AssertStatement1 n) { unimplementedVisitor("visit(AssertStatement1)"); return true; }
        public void endVisit(AssertStatement1 n) { unimplementedVisitor("endVisit(AssertStatement1)"); }

        public boolean visit(SwitchLabel0 n) { unimplementedVisitor("visit(SwitchLabel0)"); return true; }
        public void endVisit(SwitchLabel0 n) { unimplementedVisitor("endVisit(SwitchLabel0)"); }

        public boolean visit(SwitchLabel1 n) { unimplementedVisitor("visit(SwitchLabel1)"); return true; }
        public void endVisit(SwitchLabel1 n) { unimplementedVisitor("endVisit(SwitchLabel1)"); }

        public boolean visit(SwitchLabel2 n) { unimplementedVisitor("visit(SwitchLabel2)"); return true; }
        public void endVisit(SwitchLabel2 n) { unimplementedVisitor("endVisit(SwitchLabel2)"); }

        public boolean visit(TryStatement0 n) { unimplementedVisitor("visit(TryStatement0)"); return true; }
        public void endVisit(TryStatement0 n) { unimplementedVisitor("endVisit(TryStatement0)"); }

        public boolean visit(TryStatement1 n) { unimplementedVisitor("visit(TryStatement1)"); return true; }
        public void endVisit(TryStatement1 n) { unimplementedVisitor("endVisit(TryStatement1)"); }

        public boolean visit(PrimaryNoNewArray0 n) { unimplementedVisitor("visit(PrimaryNoNewArray0)"); return true; }
        public void endVisit(PrimaryNoNewArray0 n) { unimplementedVisitor("endVisit(PrimaryNoNewArray0)"); }

        public boolean visit(PrimaryNoNewArray1 n) { unimplementedVisitor("visit(PrimaryNoNewArray1)"); return true; }
        public void endVisit(PrimaryNoNewArray1 n) { unimplementedVisitor("endVisit(PrimaryNoNewArray1)"); }

        public boolean visit(PrimaryNoNewArray2 n) { unimplementedVisitor("visit(PrimaryNoNewArray2)"); return true; }
        public void endVisit(PrimaryNoNewArray2 n) { unimplementedVisitor("endVisit(PrimaryNoNewArray2)"); }

        public boolean visit(PrimaryNoNewArray3 n) { unimplementedVisitor("visit(PrimaryNoNewArray3)"); return true; }
        public void endVisit(PrimaryNoNewArray3 n) { unimplementedVisitor("endVisit(PrimaryNoNewArray3)"); }

        public boolean visit(PrimaryNoNewArray4 n) { unimplementedVisitor("visit(PrimaryNoNewArray4)"); return true; }
        public void endVisit(PrimaryNoNewArray4 n) { unimplementedVisitor("endVisit(PrimaryNoNewArray4)"); }

        public boolean visit(Literal0 n) { unimplementedVisitor("visit(Literal0)"); return true; }
        public void endVisit(Literal0 n) { unimplementedVisitor("endVisit(Literal0)"); }

        public boolean visit(Literal1 n) { unimplementedVisitor("visit(Literal1)"); return true; }
        public void endVisit(Literal1 n) { unimplementedVisitor("endVisit(Literal1)"); }

        public boolean visit(Literal2 n) { unimplementedVisitor("visit(Literal2)"); return true; }
        public void endVisit(Literal2 n) { unimplementedVisitor("endVisit(Literal2)"); }

        public boolean visit(Literal3 n) { unimplementedVisitor("visit(Literal3)"); return true; }
        public void endVisit(Literal3 n) { unimplementedVisitor("endVisit(Literal3)"); }

        public boolean visit(Literal4 n) { unimplementedVisitor("visit(Literal4)"); return true; }
        public void endVisit(Literal4 n) { unimplementedVisitor("endVisit(Literal4)"); }

        public boolean visit(Literal5 n) { unimplementedVisitor("visit(Literal5)"); return true; }
        public void endVisit(Literal5 n) { unimplementedVisitor("endVisit(Literal5)"); }

        public boolean visit(Literal6 n) { unimplementedVisitor("visit(Literal6)"); return true; }
        public void endVisit(Literal6 n) { unimplementedVisitor("endVisit(Literal6)"); }

        public boolean visit(BooleanLiteral0 n) { unimplementedVisitor("visit(BooleanLiteral0)"); return true; }
        public void endVisit(BooleanLiteral0 n) { unimplementedVisitor("endVisit(BooleanLiteral0)"); }

        public boolean visit(BooleanLiteral1 n) { unimplementedVisitor("visit(BooleanLiteral1)"); return true; }
        public void endVisit(BooleanLiteral1 n) { unimplementedVisitor("endVisit(BooleanLiteral1)"); }

        public boolean visit(ClassInstanceCreationExpression0 n) { unimplementedVisitor("visit(ClassInstanceCreationExpression0)"); return true; }
        public void endVisit(ClassInstanceCreationExpression0 n) { unimplementedVisitor("endVisit(ClassInstanceCreationExpression0)"); }

        public boolean visit(ClassInstanceCreationExpression1 n) { unimplementedVisitor("visit(ClassInstanceCreationExpression1)"); return true; }
        public void endVisit(ClassInstanceCreationExpression1 n) { unimplementedVisitor("endVisit(ClassInstanceCreationExpression1)"); }

        public boolean visit(ArrayCreationExpression0 n) { unimplementedVisitor("visit(ArrayCreationExpression0)"); return true; }
        public void endVisit(ArrayCreationExpression0 n) { unimplementedVisitor("endVisit(ArrayCreationExpression0)"); }

        public boolean visit(ArrayCreationExpression1 n) { unimplementedVisitor("visit(ArrayCreationExpression1)"); return true; }
        public void endVisit(ArrayCreationExpression1 n) { unimplementedVisitor("endVisit(ArrayCreationExpression1)"); }

        public boolean visit(ArrayCreationExpression2 n) { unimplementedVisitor("visit(ArrayCreationExpression2)"); return true; }
        public void endVisit(ArrayCreationExpression2 n) { unimplementedVisitor("endVisit(ArrayCreationExpression2)"); }

        public boolean visit(ArrayCreationExpression3 n) { unimplementedVisitor("visit(ArrayCreationExpression3)"); return true; }
        public void endVisit(ArrayCreationExpression3 n) { unimplementedVisitor("endVisit(ArrayCreationExpression3)"); }

        public boolean visit(Dims0 n) { unimplementedVisitor("visit(Dims0)"); return true; }
        public void endVisit(Dims0 n) { unimplementedVisitor("endVisit(Dims0)"); }

        public boolean visit(Dims1 n) { unimplementedVisitor("visit(Dims1)"); return true; }
        public void endVisit(Dims1 n) { unimplementedVisitor("endVisit(Dims1)"); }

        public boolean visit(FieldAccess0 n) { unimplementedVisitor("visit(FieldAccess0)"); return true; }
        public void endVisit(FieldAccess0 n) { unimplementedVisitor("endVisit(FieldAccess0)"); }

        public boolean visit(FieldAccess1 n) { unimplementedVisitor("visit(FieldAccess1)"); return true; }
        public void endVisit(FieldAccess1 n) { unimplementedVisitor("endVisit(FieldAccess1)"); }

        public boolean visit(FieldAccess2 n) { unimplementedVisitor("visit(FieldAccess2)"); return true; }
        public void endVisit(FieldAccess2 n) { unimplementedVisitor("endVisit(FieldAccess2)"); }

        public boolean visit(MethodInvocation0 n) { unimplementedVisitor("visit(MethodInvocation0)"); return true; }
        public void endVisit(MethodInvocation0 n) { unimplementedVisitor("endVisit(MethodInvocation0)"); }

        public boolean visit(MethodInvocation1 n) { unimplementedVisitor("visit(MethodInvocation1)"); return true; }
        public void endVisit(MethodInvocation1 n) { unimplementedVisitor("endVisit(MethodInvocation1)"); }

        public boolean visit(MethodInvocation2 n) { unimplementedVisitor("visit(MethodInvocation2)"); return true; }
        public void endVisit(MethodInvocation2 n) { unimplementedVisitor("endVisit(MethodInvocation2)"); }

        public boolean visit(MethodInvocation3 n) { unimplementedVisitor("visit(MethodInvocation3)"); return true; }
        public void endVisit(MethodInvocation3 n) { unimplementedVisitor("endVisit(MethodInvocation3)"); }

        public boolean visit(MethodInvocation4 n) { unimplementedVisitor("visit(MethodInvocation4)"); return true; }
        public void endVisit(MethodInvocation4 n) { unimplementedVisitor("endVisit(MethodInvocation4)"); }

        public boolean visit(ArrayAccess0 n) { unimplementedVisitor("visit(ArrayAccess0)"); return true; }
        public void endVisit(ArrayAccess0 n) { unimplementedVisitor("endVisit(ArrayAccess0)"); }

        public boolean visit(ArrayAccess1 n) { unimplementedVisitor("visit(ArrayAccess1)"); return true; }
        public void endVisit(ArrayAccess1 n) { unimplementedVisitor("endVisit(ArrayAccess1)"); }

        public boolean visit(UnaryExpression0 n) { unimplementedVisitor("visit(UnaryExpression0)"); return true; }
        public void endVisit(UnaryExpression0 n) { unimplementedVisitor("endVisit(UnaryExpression0)"); }

        public boolean visit(UnaryExpression1 n) { unimplementedVisitor("visit(UnaryExpression1)"); return true; }
        public void endVisit(UnaryExpression1 n) { unimplementedVisitor("endVisit(UnaryExpression1)"); }

        public boolean visit(UnaryExpressionNotPlusMinus0 n) { unimplementedVisitor("visit(UnaryExpressionNotPlusMinus0)"); return true; }
        public void endVisit(UnaryExpressionNotPlusMinus0 n) { unimplementedVisitor("endVisit(UnaryExpressionNotPlusMinus0)"); }

        public boolean visit(UnaryExpressionNotPlusMinus1 n) { unimplementedVisitor("visit(UnaryExpressionNotPlusMinus1)"); return true; }
        public void endVisit(UnaryExpressionNotPlusMinus1 n) { unimplementedVisitor("endVisit(UnaryExpressionNotPlusMinus1)"); }

        public boolean visit(CastExpression0 n) { unimplementedVisitor("visit(CastExpression0)"); return true; }
        public void endVisit(CastExpression0 n) { unimplementedVisitor("endVisit(CastExpression0)"); }

        public boolean visit(CastExpression1 n) { unimplementedVisitor("visit(CastExpression1)"); return true; }
        public void endVisit(CastExpression1 n) { unimplementedVisitor("endVisit(CastExpression1)"); }

        public boolean visit(MultiplicativeExpression0 n) { unimplementedVisitor("visit(MultiplicativeExpression0)"); return true; }
        public void endVisit(MultiplicativeExpression0 n) { unimplementedVisitor("endVisit(MultiplicativeExpression0)"); }

        public boolean visit(MultiplicativeExpression1 n) { unimplementedVisitor("visit(MultiplicativeExpression1)"); return true; }
        public void endVisit(MultiplicativeExpression1 n) { unimplementedVisitor("endVisit(MultiplicativeExpression1)"); }

        public boolean visit(MultiplicativeExpression2 n) { unimplementedVisitor("visit(MultiplicativeExpression2)"); return true; }
        public void endVisit(MultiplicativeExpression2 n) { unimplementedVisitor("endVisit(MultiplicativeExpression2)"); }

        public boolean visit(AdditiveExpression0 n) { unimplementedVisitor("visit(AdditiveExpression0)"); return true; }
        public void endVisit(AdditiveExpression0 n) { unimplementedVisitor("endVisit(AdditiveExpression0)"); }

        public boolean visit(AdditiveExpression1 n) { unimplementedVisitor("visit(AdditiveExpression1)"); return true; }
        public void endVisit(AdditiveExpression1 n) { unimplementedVisitor("endVisit(AdditiveExpression1)"); }

        public boolean visit(ShiftExpression0 n) { unimplementedVisitor("visit(ShiftExpression0)"); return true; }
        public void endVisit(ShiftExpression0 n) { unimplementedVisitor("endVisit(ShiftExpression0)"); }

        public boolean visit(ShiftExpression1 n) { unimplementedVisitor("visit(ShiftExpression1)"); return true; }
        public void endVisit(ShiftExpression1 n) { unimplementedVisitor("endVisit(ShiftExpression1)"); }

        public boolean visit(ShiftExpression2 n) { unimplementedVisitor("visit(ShiftExpression2)"); return true; }
        public void endVisit(ShiftExpression2 n) { unimplementedVisitor("endVisit(ShiftExpression2)"); }

        public boolean visit(RelationalExpression0 n) { unimplementedVisitor("visit(RelationalExpression0)"); return true; }
        public void endVisit(RelationalExpression0 n) { unimplementedVisitor("endVisit(RelationalExpression0)"); }

        public boolean visit(RelationalExpression1 n) { unimplementedVisitor("visit(RelationalExpression1)"); return true; }
        public void endVisit(RelationalExpression1 n) { unimplementedVisitor("endVisit(RelationalExpression1)"); }

        public boolean visit(RelationalExpression2 n) { unimplementedVisitor("visit(RelationalExpression2)"); return true; }
        public void endVisit(RelationalExpression2 n) { unimplementedVisitor("endVisit(RelationalExpression2)"); }

        public boolean visit(RelationalExpression3 n) { unimplementedVisitor("visit(RelationalExpression3)"); return true; }
        public void endVisit(RelationalExpression3 n) { unimplementedVisitor("endVisit(RelationalExpression3)"); }

        public boolean visit(RelationalExpression4 n) { unimplementedVisitor("visit(RelationalExpression4)"); return true; }
        public void endVisit(RelationalExpression4 n) { unimplementedVisitor("endVisit(RelationalExpression4)"); }

        public boolean visit(EqualityExpression0 n) { unimplementedVisitor("visit(EqualityExpression0)"); return true; }
        public void endVisit(EqualityExpression0 n) { unimplementedVisitor("endVisit(EqualityExpression0)"); }

        public boolean visit(EqualityExpression1 n) { unimplementedVisitor("visit(EqualityExpression1)"); return true; }
        public void endVisit(EqualityExpression1 n) { unimplementedVisitor("endVisit(EqualityExpression1)"); }

        public boolean visit(AssignmentOperator0 n) { unimplementedVisitor("visit(AssignmentOperator0)"); return true; }
        public void endVisit(AssignmentOperator0 n) { unimplementedVisitor("endVisit(AssignmentOperator0)"); }

        public boolean visit(AssignmentOperator1 n) { unimplementedVisitor("visit(AssignmentOperator1)"); return true; }
        public void endVisit(AssignmentOperator1 n) { unimplementedVisitor("endVisit(AssignmentOperator1)"); }

        public boolean visit(AssignmentOperator2 n) { unimplementedVisitor("visit(AssignmentOperator2)"); return true; }
        public void endVisit(AssignmentOperator2 n) { unimplementedVisitor("endVisit(AssignmentOperator2)"); }

        public boolean visit(AssignmentOperator3 n) { unimplementedVisitor("visit(AssignmentOperator3)"); return true; }
        public void endVisit(AssignmentOperator3 n) { unimplementedVisitor("endVisit(AssignmentOperator3)"); }

        public boolean visit(AssignmentOperator4 n) { unimplementedVisitor("visit(AssignmentOperator4)"); return true; }
        public void endVisit(AssignmentOperator4 n) { unimplementedVisitor("endVisit(AssignmentOperator4)"); }

        public boolean visit(AssignmentOperator5 n) { unimplementedVisitor("visit(AssignmentOperator5)"); return true; }
        public void endVisit(AssignmentOperator5 n) { unimplementedVisitor("endVisit(AssignmentOperator5)"); }

        public boolean visit(AssignmentOperator6 n) { unimplementedVisitor("visit(AssignmentOperator6)"); return true; }
        public void endVisit(AssignmentOperator6 n) { unimplementedVisitor("endVisit(AssignmentOperator6)"); }

        public boolean visit(AssignmentOperator7 n) { unimplementedVisitor("visit(AssignmentOperator7)"); return true; }
        public void endVisit(AssignmentOperator7 n) { unimplementedVisitor("endVisit(AssignmentOperator7)"); }

        public boolean visit(AssignmentOperator8 n) { unimplementedVisitor("visit(AssignmentOperator8)"); return true; }
        public void endVisit(AssignmentOperator8 n) { unimplementedVisitor("endVisit(AssignmentOperator8)"); }

        public boolean visit(AssignmentOperator9 n) { unimplementedVisitor("visit(AssignmentOperator9)"); return true; }
        public void endVisit(AssignmentOperator9 n) { unimplementedVisitor("endVisit(AssignmentOperator9)"); }

        public boolean visit(AssignmentOperator10 n) { unimplementedVisitor("visit(AssignmentOperator10)"); return true; }
        public void endVisit(AssignmentOperator10 n) { unimplementedVisitor("endVisit(AssignmentOperator10)"); }

        public boolean visit(AssignmentOperator11 n) { unimplementedVisitor("visit(AssignmentOperator11)"); return true; }
        public void endVisit(AssignmentOperator11 n) { unimplementedVisitor("endVisit(AssignmentOperator11)"); }


        public boolean visit(Ast n)
        {
            if (n instanceof AstToken) return visit((AstToken) n);
            else if (n instanceof identifier) return visit((identifier) n);
            else if (n instanceof PrimitiveType) return visit((PrimitiveType) n);
            else if (n instanceof ClassType) return visit((ClassType) n);
            else if (n instanceof InterfaceType) return visit((InterfaceType) n);
            else if (n instanceof TypeName) return visit((TypeName) n);
            else if (n instanceof ArrayType) return visit((ArrayType) n);
            else if (n instanceof TypeParameter) return visit((TypeParameter) n);
            else if (n instanceof TypeBound) return visit((TypeBound) n);
            else if (n instanceof AdditionalBoundList) return visit((AdditionalBoundList) n);
            else if (n instanceof AdditionalBound) return visit((AdditionalBound) n);
            else if (n instanceof TypeArguments) return visit((TypeArguments) n);
            else if (n instanceof ActualTypeArgumentList) return visit((ActualTypeArgumentList) n);
            else if (n instanceof Wildcard) return visit((Wildcard) n);
            else if (n instanceof PackageName) return visit((PackageName) n);
            else if (n instanceof ExpressionName) return visit((ExpressionName) n);
            else if (n instanceof MethodName) return visit((MethodName) n);
            else if (n instanceof PackageOrTypeName) return visit((PackageOrTypeName) n);
            else if (n instanceof AmbiguousName) return visit((AmbiguousName) n);
            else if (n instanceof CompilationUnit) return visit((CompilationUnit) n);
            else if (n instanceof ImportDeclarations) return visit((ImportDeclarations) n);
            else if (n instanceof TypeDeclarations) return visit((TypeDeclarations) n);
            else if (n instanceof PackageDeclaration) return visit((PackageDeclaration) n);
            else if (n instanceof SingleTypeImportDeclaration) return visit((SingleTypeImportDeclaration) n);
            else if (n instanceof TypeImportOnDemandDeclaration) return visit((TypeImportOnDemandDeclaration) n);
            else if (n instanceof SingleStaticImportDeclaration) return visit((SingleStaticImportDeclaration) n);
            else if (n instanceof StaticImportOnDemandDeclaration) return visit((StaticImportOnDemandDeclaration) n);
            else if (n instanceof TypeDeclaration) return visit((TypeDeclaration) n);
            else if (n instanceof NormalClassDeclaration) return visit((NormalClassDeclaration) n);
            else if (n instanceof ClassModifiers) return visit((ClassModifiers) n);
            else if (n instanceof TypeParameters) return visit((TypeParameters) n);
            else if (n instanceof TypeParameterList) return visit((TypeParameterList) n);
            else if (n instanceof Super) return visit((Super) n);
            else if (n instanceof Interfaces) return visit((Interfaces) n);
            else if (n instanceof InterfaceTypeList) return visit((InterfaceTypeList) n);
            else if (n instanceof ClassBody) return visit((ClassBody) n);
            else if (n instanceof ClassBodyDeclarations) return visit((ClassBodyDeclarations) n);
            else if (n instanceof ClassMemberDeclaration) return visit((ClassMemberDeclaration) n);
            else if (n instanceof FieldDeclaration) return visit((FieldDeclaration) n);
            else if (n instanceof VariableDeclarators) return visit((VariableDeclarators) n);
            else if (n instanceof VariableDeclarator) return visit((VariableDeclarator) n);
            else if (n instanceof VariableDeclaratorId) return visit((VariableDeclaratorId) n);
            else if (n instanceof FieldModifiers) return visit((FieldModifiers) n);
            else if (n instanceof MethodDeclaration) return visit((MethodDeclaration) n);
            else if (n instanceof MethodHeader) return visit((MethodHeader) n);
            else if (n instanceof ResultType) return visit((ResultType) n);
            else if (n instanceof FormalParameterList) return visit((FormalParameterList) n);
            else if (n instanceof FormalParameters) return visit((FormalParameters) n);
            else if (n instanceof FormalParameter) return visit((FormalParameter) n);
            else if (n instanceof VariableModifiers) return visit((VariableModifiers) n);
            else if (n instanceof VariableModifier) return visit((VariableModifier) n);
            else if (n instanceof LastFormalParameter) return visit((LastFormalParameter) n);
            else if (n instanceof MethodModifiers) return visit((MethodModifiers) n);
            else if (n instanceof Throws) return visit((Throws) n);
            else if (n instanceof ExceptionTypeList) return visit((ExceptionTypeList) n);
            else if (n instanceof MethodBody) return visit((MethodBody) n);
            else if (n instanceof StaticInitializer) return visit((StaticInitializer) n);
            else if (n instanceof ConstructorDeclaration) return visit((ConstructorDeclaration) n);
            else if (n instanceof ConstructorDeclarator) return visit((ConstructorDeclarator) n);
            else if (n instanceof ConstructorModifiers) return visit((ConstructorModifiers) n);
            else if (n instanceof ConstructorBody) return visit((ConstructorBody) n);
            else if (n instanceof EnumDeclaration) return visit((EnumDeclaration) n);
            else if (n instanceof EnumBody) return visit((EnumBody) n);
            else if (n instanceof EnumConstants) return visit((EnumConstants) n);
            else if (n instanceof EnumConstant) return visit((EnumConstant) n);
            else if (n instanceof Arguments) return visit((Arguments) n);
            else if (n instanceof EnumBodyDeclarations) return visit((EnumBodyDeclarations) n);
            else if (n instanceof NormalInterfaceDeclaration) return visit((NormalInterfaceDeclaration) n);
            else if (n instanceof InterfaceModifiers) return visit((InterfaceModifiers) n);
            else if (n instanceof InterfaceBody) return visit((InterfaceBody) n);
            else if (n instanceof InterfaceMemberDeclarations) return visit((InterfaceMemberDeclarations) n);
            else if (n instanceof InterfaceMemberDeclaration) return visit((InterfaceMemberDeclaration) n);
            else if (n instanceof ConstantDeclaration) return visit((ConstantDeclaration) n);
            else if (n instanceof ConstantModifiers) return visit((ConstantModifiers) n);
            else if (n instanceof AbstractMethodDeclaration) return visit((AbstractMethodDeclaration) n);
            else if (n instanceof AbstractMethodModifiers) return visit((AbstractMethodModifiers) n);
            else if (n instanceof AnnotationTypeDeclaration) return visit((AnnotationTypeDeclaration) n);
            else if (n instanceof AnnotationTypeBody) return visit((AnnotationTypeBody) n);
            else if (n instanceof AnnotationTypeElementDeclarations) return visit((AnnotationTypeElementDeclarations) n);
            else if (n instanceof DefaultValue) return visit((DefaultValue) n);
            else if (n instanceof Annotations) return visit((Annotations) n);
            else if (n instanceof NormalAnnotation) return visit((NormalAnnotation) n);
            else if (n instanceof ElementValuePairs) return visit((ElementValuePairs) n);
            else if (n instanceof ElementValuePair) return visit((ElementValuePair) n);
            else if (n instanceof ElementValueArrayInitializer) return visit((ElementValueArrayInitializer) n);
            else if (n instanceof ElementValues) return visit((ElementValues) n);
            else if (n instanceof MarkerAnnotation) return visit((MarkerAnnotation) n);
            else if (n instanceof SingleElementAnnotation) return visit((SingleElementAnnotation) n);
            else if (n instanceof ArrayInitializer) return visit((ArrayInitializer) n);
            else if (n instanceof VariableInitializers) return visit((VariableInitializers) n);
            else if (n instanceof Block) return visit((Block) n);
            else if (n instanceof BlockStatements) return visit((BlockStatements) n);
            else if (n instanceof LocalVariableDeclarationStatement) return visit((LocalVariableDeclarationStatement) n);
            else if (n instanceof LocalVariableDeclaration) return visit((LocalVariableDeclaration) n);
            else if (n instanceof IfThenStatement) return visit((IfThenStatement) n);
            else if (n instanceof IfThenElseStatement) return visit((IfThenElseStatement) n);
            else if (n instanceof IfThenElseStatementNoShortIf) return visit((IfThenElseStatementNoShortIf) n);
            else if (n instanceof EmptyStatement) return visit((EmptyStatement) n);
            else if (n instanceof LabeledStatement) return visit((LabeledStatement) n);
            else if (n instanceof LabeledStatementNoShortIf) return visit((LabeledStatementNoShortIf) n);
            else if (n instanceof ExpressionStatement) return visit((ExpressionStatement) n);
            else if (n instanceof SwitchStatement) return visit((SwitchStatement) n);
            else if (n instanceof SwitchBlock) return visit((SwitchBlock) n);
            else if (n instanceof SwitchBlockStatementGroups) return visit((SwitchBlockStatementGroups) n);
            else if (n instanceof SwitchBlockStatementGroup) return visit((SwitchBlockStatementGroup) n);
            else if (n instanceof SwitchLabels) return visit((SwitchLabels) n);
            else if (n instanceof WhileStatement) return visit((WhileStatement) n);
            else if (n instanceof WhileStatementNoShortIf) return visit((WhileStatementNoShortIf) n);
            else if (n instanceof DoStatement) return visit((DoStatement) n);
            else if (n instanceof BasicForStatement) return visit((BasicForStatement) n);
            else if (n instanceof ForStatementNoShortIf) return visit((ForStatementNoShortIf) n);
            else if (n instanceof StatementExpressionList) return visit((StatementExpressionList) n);
            else if (n instanceof EnhancedForStatement) return visit((EnhancedForStatement) n);
            else if (n instanceof BreakStatement) return visit((BreakStatement) n);
            else if (n instanceof ContinueStatement) return visit((ContinueStatement) n);
            else if (n instanceof ReturnStatement) return visit((ReturnStatement) n);
            else if (n instanceof ThrowStatement) return visit((ThrowStatement) n);
            else if (n instanceof SynchronizedStatement) return visit((SynchronizedStatement) n);
            else if (n instanceof Catches) return visit((Catches) n);
            else if (n instanceof CatchClause) return visit((CatchClause) n);
            else if (n instanceof Finally) return visit((Finally) n);
            else if (n instanceof ArgumentList) return visit((ArgumentList) n);
            else if (n instanceof DimExprs) return visit((DimExprs) n);
            else if (n instanceof DimExpr) return visit((DimExpr) n);
            else if (n instanceof PostIncrementExpression) return visit((PostIncrementExpression) n);
            else if (n instanceof PostDecrementExpression) return visit((PostDecrementExpression) n);
            else if (n instanceof PreIncrementExpression) return visit((PreIncrementExpression) n);
            else if (n instanceof PreDecrementExpression) return visit((PreDecrementExpression) n);
            else if (n instanceof AndExpression) return visit((AndExpression) n);
            else if (n instanceof ExclusiveOrExpression) return visit((ExclusiveOrExpression) n);
            else if (n instanceof InclusiveOrExpression) return visit((InclusiveOrExpression) n);
            else if (n instanceof ConditionalAndExpression) return visit((ConditionalAndExpression) n);
            else if (n instanceof ConditionalOrExpression) return visit((ConditionalOrExpression) n);
            else if (n instanceof ConditionalExpression) return visit((ConditionalExpression) n);
            else if (n instanceof Assignment) return visit((Assignment) n);
            else if (n instanceof Commaopt) return visit((Commaopt) n);
            else if (n instanceof Ellipsisopt) return visit((Ellipsisopt) n);
            else if (n instanceof LPGUserAction0) return visit((LPGUserAction0) n);
            else if (n instanceof LPGUserAction1) return visit((LPGUserAction1) n);
            else if (n instanceof LPGUserAction2) return visit((LPGUserAction2) n);
            else if (n instanceof LPGUserAction3) return visit((LPGUserAction3) n);
            else if (n instanceof LPGUserAction4) return visit((LPGUserAction4) n);
            else if (n instanceof IntegralType0) return visit((IntegralType0) n);
            else if (n instanceof IntegralType1) return visit((IntegralType1) n);
            else if (n instanceof IntegralType2) return visit((IntegralType2) n);
            else if (n instanceof IntegralType3) return visit((IntegralType3) n);
            else if (n instanceof IntegralType4) return visit((IntegralType4) n);
            else if (n instanceof FloatingPointType0) return visit((FloatingPointType0) n);
            else if (n instanceof FloatingPointType1) return visit((FloatingPointType1) n);
            else if (n instanceof WildcardBounds0) return visit((WildcardBounds0) n);
            else if (n instanceof WildcardBounds1) return visit((WildcardBounds1) n);
            else if (n instanceof ClassModifier0) return visit((ClassModifier0) n);
            else if (n instanceof ClassModifier1) return visit((ClassModifier1) n);
            else if (n instanceof ClassModifier2) return visit((ClassModifier2) n);
            else if (n instanceof ClassModifier3) return visit((ClassModifier3) n);
            else if (n instanceof ClassModifier4) return visit((ClassModifier4) n);
            else if (n instanceof ClassModifier5) return visit((ClassModifier5) n);
            else if (n instanceof ClassModifier6) return visit((ClassModifier6) n);
            else if (n instanceof FieldModifier0) return visit((FieldModifier0) n);
            else if (n instanceof FieldModifier1) return visit((FieldModifier1) n);
            else if (n instanceof FieldModifier2) return visit((FieldModifier2) n);
            else if (n instanceof FieldModifier3) return visit((FieldModifier3) n);
            else if (n instanceof FieldModifier4) return visit((FieldModifier4) n);
            else if (n instanceof FieldModifier5) return visit((FieldModifier5) n);
            else if (n instanceof FieldModifier6) return visit((FieldModifier6) n);
            else if (n instanceof MethodDeclarator0) return visit((MethodDeclarator0) n);
            else if (n instanceof MethodDeclarator1) return visit((MethodDeclarator1) n);
            else if (n instanceof MethodModifier0) return visit((MethodModifier0) n);
            else if (n instanceof MethodModifier1) return visit((MethodModifier1) n);
            else if (n instanceof MethodModifier2) return visit((MethodModifier2) n);
            else if (n instanceof MethodModifier3) return visit((MethodModifier3) n);
            else if (n instanceof MethodModifier4) return visit((MethodModifier4) n);
            else if (n instanceof MethodModifier5) return visit((MethodModifier5) n);
            else if (n instanceof MethodModifier6) return visit((MethodModifier6) n);
            else if (n instanceof MethodModifier7) return visit((MethodModifier7) n);
            else if (n instanceof MethodModifier8) return visit((MethodModifier8) n);
            else if (n instanceof ConstructorModifier0) return visit((ConstructorModifier0) n);
            else if (n instanceof ConstructorModifier1) return visit((ConstructorModifier1) n);
            else if (n instanceof ConstructorModifier2) return visit((ConstructorModifier2) n);
            else if (n instanceof ExplicitConstructorInvocation0) return visit((ExplicitConstructorInvocation0) n);
            else if (n instanceof ExplicitConstructorInvocation1) return visit((ExplicitConstructorInvocation1) n);
            else if (n instanceof ExplicitConstructorInvocation2) return visit((ExplicitConstructorInvocation2) n);
            else if (n instanceof InterfaceModifier0) return visit((InterfaceModifier0) n);
            else if (n instanceof InterfaceModifier1) return visit((InterfaceModifier1) n);
            else if (n instanceof InterfaceModifier2) return visit((InterfaceModifier2) n);
            else if (n instanceof InterfaceModifier3) return visit((InterfaceModifier3) n);
            else if (n instanceof InterfaceModifier4) return visit((InterfaceModifier4) n);
            else if (n instanceof InterfaceModifier5) return visit((InterfaceModifier5) n);
            else if (n instanceof ExtendsInterfaces0) return visit((ExtendsInterfaces0) n);
            else if (n instanceof ExtendsInterfaces1) return visit((ExtendsInterfaces1) n);
            else if (n instanceof ConstantModifier0) return visit((ConstantModifier0) n);
            else if (n instanceof ConstantModifier1) return visit((ConstantModifier1) n);
            else if (n instanceof ConstantModifier2) return visit((ConstantModifier2) n);
            else if (n instanceof AbstractMethodModifier0) return visit((AbstractMethodModifier0) n);
            else if (n instanceof AbstractMethodModifier1) return visit((AbstractMethodModifier1) n);
            else if (n instanceof AnnotationTypeElementDeclaration0) return visit((AnnotationTypeElementDeclaration0) n);
            else if (n instanceof AnnotationTypeElementDeclaration1) return visit((AnnotationTypeElementDeclaration1) n);
            else if (n instanceof AssertStatement0) return visit((AssertStatement0) n);
            else if (n instanceof AssertStatement1) return visit((AssertStatement1) n);
            else if (n instanceof SwitchLabel0) return visit((SwitchLabel0) n);
            else if (n instanceof SwitchLabel1) return visit((SwitchLabel1) n);
            else if (n instanceof SwitchLabel2) return visit((SwitchLabel2) n);
            else if (n instanceof TryStatement0) return visit((TryStatement0) n);
            else if (n instanceof TryStatement1) return visit((TryStatement1) n);
            else if (n instanceof PrimaryNoNewArray0) return visit((PrimaryNoNewArray0) n);
            else if (n instanceof PrimaryNoNewArray1) return visit((PrimaryNoNewArray1) n);
            else if (n instanceof PrimaryNoNewArray2) return visit((PrimaryNoNewArray2) n);
            else if (n instanceof PrimaryNoNewArray3) return visit((PrimaryNoNewArray3) n);
            else if (n instanceof PrimaryNoNewArray4) return visit((PrimaryNoNewArray4) n);
            else if (n instanceof Literal0) return visit((Literal0) n);
            else if (n instanceof Literal1) return visit((Literal1) n);
            else if (n instanceof Literal2) return visit((Literal2) n);
            else if (n instanceof Literal3) return visit((Literal3) n);
            else if (n instanceof Literal4) return visit((Literal4) n);
            else if (n instanceof Literal5) return visit((Literal5) n);
            else if (n instanceof Literal6) return visit((Literal6) n);
            else if (n instanceof BooleanLiteral0) return visit((BooleanLiteral0) n);
            else if (n instanceof BooleanLiteral1) return visit((BooleanLiteral1) n);
            else if (n instanceof ClassInstanceCreationExpression0) return visit((ClassInstanceCreationExpression0) n);
            else if (n instanceof ClassInstanceCreationExpression1) return visit((ClassInstanceCreationExpression1) n);
            else if (n instanceof ArrayCreationExpression0) return visit((ArrayCreationExpression0) n);
            else if (n instanceof ArrayCreationExpression1) return visit((ArrayCreationExpression1) n);
            else if (n instanceof ArrayCreationExpression2) return visit((ArrayCreationExpression2) n);
            else if (n instanceof ArrayCreationExpression3) return visit((ArrayCreationExpression3) n);
            else if (n instanceof Dims0) return visit((Dims0) n);
            else if (n instanceof Dims1) return visit((Dims1) n);
            else if (n instanceof FieldAccess0) return visit((FieldAccess0) n);
            else if (n instanceof FieldAccess1) return visit((FieldAccess1) n);
            else if (n instanceof FieldAccess2) return visit((FieldAccess2) n);
            else if (n instanceof MethodInvocation0) return visit((MethodInvocation0) n);
            else if (n instanceof MethodInvocation1) return visit((MethodInvocation1) n);
            else if (n instanceof MethodInvocation2) return visit((MethodInvocation2) n);
            else if (n instanceof MethodInvocation3) return visit((MethodInvocation3) n);
            else if (n instanceof MethodInvocation4) return visit((MethodInvocation4) n);
            else if (n instanceof ArrayAccess0) return visit((ArrayAccess0) n);
            else if (n instanceof ArrayAccess1) return visit((ArrayAccess1) n);
            else if (n instanceof UnaryExpression0) return visit((UnaryExpression0) n);
            else if (n instanceof UnaryExpression1) return visit((UnaryExpression1) n);
            else if (n instanceof UnaryExpressionNotPlusMinus0) return visit((UnaryExpressionNotPlusMinus0) n);
            else if (n instanceof UnaryExpressionNotPlusMinus1) return visit((UnaryExpressionNotPlusMinus1) n);
            else if (n instanceof CastExpression0) return visit((CastExpression0) n);
            else if (n instanceof CastExpression1) return visit((CastExpression1) n);
            else if (n instanceof MultiplicativeExpression0) return visit((MultiplicativeExpression0) n);
            else if (n instanceof MultiplicativeExpression1) return visit((MultiplicativeExpression1) n);
            else if (n instanceof MultiplicativeExpression2) return visit((MultiplicativeExpression2) n);
            else if (n instanceof AdditiveExpression0) return visit((AdditiveExpression0) n);
            else if (n instanceof AdditiveExpression1) return visit((AdditiveExpression1) n);
            else if (n instanceof ShiftExpression0) return visit((ShiftExpression0) n);
            else if (n instanceof ShiftExpression1) return visit((ShiftExpression1) n);
            else if (n instanceof ShiftExpression2) return visit((ShiftExpression2) n);
            else if (n instanceof RelationalExpression0) return visit((RelationalExpression0) n);
            else if (n instanceof RelationalExpression1) return visit((RelationalExpression1) n);
            else if (n instanceof RelationalExpression2) return visit((RelationalExpression2) n);
            else if (n instanceof RelationalExpression3) return visit((RelationalExpression3) n);
            else if (n instanceof RelationalExpression4) return visit((RelationalExpression4) n);
            else if (n instanceof EqualityExpression0) return visit((EqualityExpression0) n);
            else if (n instanceof EqualityExpression1) return visit((EqualityExpression1) n);
            else if (n instanceof AssignmentOperator0) return visit((AssignmentOperator0) n);
            else if (n instanceof AssignmentOperator1) return visit((AssignmentOperator1) n);
            else if (n instanceof AssignmentOperator2) return visit((AssignmentOperator2) n);
            else if (n instanceof AssignmentOperator3) return visit((AssignmentOperator3) n);
            else if (n instanceof AssignmentOperator4) return visit((AssignmentOperator4) n);
            else if (n instanceof AssignmentOperator5) return visit((AssignmentOperator5) n);
            else if (n instanceof AssignmentOperator6) return visit((AssignmentOperator6) n);
            else if (n instanceof AssignmentOperator7) return visit((AssignmentOperator7) n);
            else if (n instanceof AssignmentOperator8) return visit((AssignmentOperator8) n);
            else if (n instanceof AssignmentOperator9) return visit((AssignmentOperator9) n);
            else if (n instanceof AssignmentOperator10) return visit((AssignmentOperator10) n);
            else if (n instanceof AssignmentOperator11) return visit((AssignmentOperator11) n);
            throw new UnsupportedOperationException("visit(" + n.getClass().toString() + ")");
        }
        public void endVisit(Ast n)
        {
            if (n instanceof AstToken) endVisit((AstToken) n);
            else if (n instanceof identifier) endVisit((identifier) n);
            else if (n instanceof PrimitiveType) endVisit((PrimitiveType) n);
            else if (n instanceof ClassType) endVisit((ClassType) n);
            else if (n instanceof InterfaceType) endVisit((InterfaceType) n);
            else if (n instanceof TypeName) endVisit((TypeName) n);
            else if (n instanceof ArrayType) endVisit((ArrayType) n);
            else if (n instanceof TypeParameter) endVisit((TypeParameter) n);
            else if (n instanceof TypeBound) endVisit((TypeBound) n);
            else if (n instanceof AdditionalBoundList) endVisit((AdditionalBoundList) n);
            else if (n instanceof AdditionalBound) endVisit((AdditionalBound) n);
            else if (n instanceof TypeArguments) endVisit((TypeArguments) n);
            else if (n instanceof ActualTypeArgumentList) endVisit((ActualTypeArgumentList) n);
            else if (n instanceof Wildcard) endVisit((Wildcard) n);
            else if (n instanceof PackageName) endVisit((PackageName) n);
            else if (n instanceof ExpressionName) endVisit((ExpressionName) n);
            else if (n instanceof MethodName) endVisit((MethodName) n);
            else if (n instanceof PackageOrTypeName) endVisit((PackageOrTypeName) n);
            else if (n instanceof AmbiguousName) endVisit((AmbiguousName) n);
            else if (n instanceof CompilationUnit) endVisit((CompilationUnit) n);
            else if (n instanceof ImportDeclarations) endVisit((ImportDeclarations) n);
            else if (n instanceof TypeDeclarations) endVisit((TypeDeclarations) n);
            else if (n instanceof PackageDeclaration) endVisit((PackageDeclaration) n);
            else if (n instanceof SingleTypeImportDeclaration) endVisit((SingleTypeImportDeclaration) n);
            else if (n instanceof TypeImportOnDemandDeclaration) endVisit((TypeImportOnDemandDeclaration) n);
            else if (n instanceof SingleStaticImportDeclaration) endVisit((SingleStaticImportDeclaration) n);
            else if (n instanceof StaticImportOnDemandDeclaration) endVisit((StaticImportOnDemandDeclaration) n);
            else if (n instanceof TypeDeclaration) endVisit((TypeDeclaration) n);
            else if (n instanceof NormalClassDeclaration) endVisit((NormalClassDeclaration) n);
            else if (n instanceof ClassModifiers) endVisit((ClassModifiers) n);
            else if (n instanceof TypeParameters) endVisit((TypeParameters) n);
            else if (n instanceof TypeParameterList) endVisit((TypeParameterList) n);
            else if (n instanceof Super) endVisit((Super) n);
            else if (n instanceof Interfaces) endVisit((Interfaces) n);
            else if (n instanceof InterfaceTypeList) endVisit((InterfaceTypeList) n);
            else if (n instanceof ClassBody) endVisit((ClassBody) n);
            else if (n instanceof ClassBodyDeclarations) endVisit((ClassBodyDeclarations) n);
            else if (n instanceof ClassMemberDeclaration) endVisit((ClassMemberDeclaration) n);
            else if (n instanceof FieldDeclaration) endVisit((FieldDeclaration) n);
            else if (n instanceof VariableDeclarators) endVisit((VariableDeclarators) n);
            else if (n instanceof VariableDeclarator) endVisit((VariableDeclarator) n);
            else if (n instanceof VariableDeclaratorId) endVisit((VariableDeclaratorId) n);
            else if (n instanceof FieldModifiers) endVisit((FieldModifiers) n);
            else if (n instanceof MethodDeclaration) endVisit((MethodDeclaration) n);
            else if (n instanceof MethodHeader) endVisit((MethodHeader) n);
            else if (n instanceof ResultType) endVisit((ResultType) n);
            else if (n instanceof FormalParameterList) endVisit((FormalParameterList) n);
            else if (n instanceof FormalParameters) endVisit((FormalParameters) n);
            else if (n instanceof FormalParameter) endVisit((FormalParameter) n);
            else if (n instanceof VariableModifiers) endVisit((VariableModifiers) n);
            else if (n instanceof VariableModifier) endVisit((VariableModifier) n);
            else if (n instanceof LastFormalParameter) endVisit((LastFormalParameter) n);
            else if (n instanceof MethodModifiers) endVisit((MethodModifiers) n);
            else if (n instanceof Throws) endVisit((Throws) n);
            else if (n instanceof ExceptionTypeList) endVisit((ExceptionTypeList) n);
            else if (n instanceof MethodBody) endVisit((MethodBody) n);
            else if (n instanceof StaticInitializer) endVisit((StaticInitializer) n);
            else if (n instanceof ConstructorDeclaration) endVisit((ConstructorDeclaration) n);
            else if (n instanceof ConstructorDeclarator) endVisit((ConstructorDeclarator) n);
            else if (n instanceof ConstructorModifiers) endVisit((ConstructorModifiers) n);
            else if (n instanceof ConstructorBody) endVisit((ConstructorBody) n);
            else if (n instanceof EnumDeclaration) endVisit((EnumDeclaration) n);
            else if (n instanceof EnumBody) endVisit((EnumBody) n);
            else if (n instanceof EnumConstants) endVisit((EnumConstants) n);
            else if (n instanceof EnumConstant) endVisit((EnumConstant) n);
            else if (n instanceof Arguments) endVisit((Arguments) n);
            else if (n instanceof EnumBodyDeclarations) endVisit((EnumBodyDeclarations) n);
            else if (n instanceof NormalInterfaceDeclaration) endVisit((NormalInterfaceDeclaration) n);
            else if (n instanceof InterfaceModifiers) endVisit((InterfaceModifiers) n);
            else if (n instanceof InterfaceBody) endVisit((InterfaceBody) n);
            else if (n instanceof InterfaceMemberDeclarations) endVisit((InterfaceMemberDeclarations) n);
            else if (n instanceof InterfaceMemberDeclaration) endVisit((InterfaceMemberDeclaration) n);
            else if (n instanceof ConstantDeclaration) endVisit((ConstantDeclaration) n);
            else if (n instanceof ConstantModifiers) endVisit((ConstantModifiers) n);
            else if (n instanceof AbstractMethodDeclaration) endVisit((AbstractMethodDeclaration) n);
            else if (n instanceof AbstractMethodModifiers) endVisit((AbstractMethodModifiers) n);
            else if (n instanceof AnnotationTypeDeclaration) endVisit((AnnotationTypeDeclaration) n);
            else if (n instanceof AnnotationTypeBody) endVisit((AnnotationTypeBody) n);
            else if (n instanceof AnnotationTypeElementDeclarations) endVisit((AnnotationTypeElementDeclarations) n);
            else if (n instanceof DefaultValue) endVisit((DefaultValue) n);
            else if (n instanceof Annotations) endVisit((Annotations) n);
            else if (n instanceof NormalAnnotation) endVisit((NormalAnnotation) n);
            else if (n instanceof ElementValuePairs) endVisit((ElementValuePairs) n);
            else if (n instanceof ElementValuePair) endVisit((ElementValuePair) n);
            else if (n instanceof ElementValueArrayInitializer) endVisit((ElementValueArrayInitializer) n);
            else if (n instanceof ElementValues) endVisit((ElementValues) n);
            else if (n instanceof MarkerAnnotation) endVisit((MarkerAnnotation) n);
            else if (n instanceof SingleElementAnnotation) endVisit((SingleElementAnnotation) n);
            else if (n instanceof ArrayInitializer) endVisit((ArrayInitializer) n);
            else if (n instanceof VariableInitializers) endVisit((VariableInitializers) n);
            else if (n instanceof Block) endVisit((Block) n);
            else if (n instanceof BlockStatements) endVisit((BlockStatements) n);
            else if (n instanceof LocalVariableDeclarationStatement) endVisit((LocalVariableDeclarationStatement) n);
            else if (n instanceof LocalVariableDeclaration) endVisit((LocalVariableDeclaration) n);
            else if (n instanceof IfThenStatement) endVisit((IfThenStatement) n);
            else if (n instanceof IfThenElseStatement) endVisit((IfThenElseStatement) n);
            else if (n instanceof IfThenElseStatementNoShortIf) endVisit((IfThenElseStatementNoShortIf) n);
            else if (n instanceof EmptyStatement) endVisit((EmptyStatement) n);
            else if (n instanceof LabeledStatement) endVisit((LabeledStatement) n);
            else if (n instanceof LabeledStatementNoShortIf) endVisit((LabeledStatementNoShortIf) n);
            else if (n instanceof ExpressionStatement) endVisit((ExpressionStatement) n);
            else if (n instanceof SwitchStatement) endVisit((SwitchStatement) n);
            else if (n instanceof SwitchBlock) endVisit((SwitchBlock) n);
            else if (n instanceof SwitchBlockStatementGroups) endVisit((SwitchBlockStatementGroups) n);
            else if (n instanceof SwitchBlockStatementGroup) endVisit((SwitchBlockStatementGroup) n);
            else if (n instanceof SwitchLabels) endVisit((SwitchLabels) n);
            else if (n instanceof WhileStatement) endVisit((WhileStatement) n);
            else if (n instanceof WhileStatementNoShortIf) endVisit((WhileStatementNoShortIf) n);
            else if (n instanceof DoStatement) endVisit((DoStatement) n);
            else if (n instanceof BasicForStatement) endVisit((BasicForStatement) n);
            else if (n instanceof ForStatementNoShortIf) endVisit((ForStatementNoShortIf) n);
            else if (n instanceof StatementExpressionList) endVisit((StatementExpressionList) n);
            else if (n instanceof EnhancedForStatement) endVisit((EnhancedForStatement) n);
            else if (n instanceof BreakStatement) endVisit((BreakStatement) n);
            else if (n instanceof ContinueStatement) endVisit((ContinueStatement) n);
            else if (n instanceof ReturnStatement) endVisit((ReturnStatement) n);
            else if (n instanceof ThrowStatement) endVisit((ThrowStatement) n);
            else if (n instanceof SynchronizedStatement) endVisit((SynchronizedStatement) n);
            else if (n instanceof Catches) endVisit((Catches) n);
            else if (n instanceof CatchClause) endVisit((CatchClause) n);
            else if (n instanceof Finally) endVisit((Finally) n);
            else if (n instanceof ArgumentList) endVisit((ArgumentList) n);
            else if (n instanceof DimExprs) endVisit((DimExprs) n);
            else if (n instanceof DimExpr) endVisit((DimExpr) n);
            else if (n instanceof PostIncrementExpression) endVisit((PostIncrementExpression) n);
            else if (n instanceof PostDecrementExpression) endVisit((PostDecrementExpression) n);
            else if (n instanceof PreIncrementExpression) endVisit((PreIncrementExpression) n);
            else if (n instanceof PreDecrementExpression) endVisit((PreDecrementExpression) n);
            else if (n instanceof AndExpression) endVisit((AndExpression) n);
            else if (n instanceof ExclusiveOrExpression) endVisit((ExclusiveOrExpression) n);
            else if (n instanceof InclusiveOrExpression) endVisit((InclusiveOrExpression) n);
            else if (n instanceof ConditionalAndExpression) endVisit((ConditionalAndExpression) n);
            else if (n instanceof ConditionalOrExpression) endVisit((ConditionalOrExpression) n);
            else if (n instanceof ConditionalExpression) endVisit((ConditionalExpression) n);
            else if (n instanceof Assignment) endVisit((Assignment) n);
            else if (n instanceof Commaopt) endVisit((Commaopt) n);
            else if (n instanceof Ellipsisopt) endVisit((Ellipsisopt) n);
            else if (n instanceof LPGUserAction0) endVisit((LPGUserAction0) n);
            else if (n instanceof LPGUserAction1) endVisit((LPGUserAction1) n);
            else if (n instanceof LPGUserAction2) endVisit((LPGUserAction2) n);
            else if (n instanceof LPGUserAction3) endVisit((LPGUserAction3) n);
            else if (n instanceof LPGUserAction4) endVisit((LPGUserAction4) n);
            else if (n instanceof IntegralType0) endVisit((IntegralType0) n);
            else if (n instanceof IntegralType1) endVisit((IntegralType1) n);
            else if (n instanceof IntegralType2) endVisit((IntegralType2) n);
            else if (n instanceof IntegralType3) endVisit((IntegralType3) n);
            else if (n instanceof IntegralType4) endVisit((IntegralType4) n);
            else if (n instanceof FloatingPointType0) endVisit((FloatingPointType0) n);
            else if (n instanceof FloatingPointType1) endVisit((FloatingPointType1) n);
            else if (n instanceof WildcardBounds0) endVisit((WildcardBounds0) n);
            else if (n instanceof WildcardBounds1) endVisit((WildcardBounds1) n);
            else if (n instanceof ClassModifier0) endVisit((ClassModifier0) n);
            else if (n instanceof ClassModifier1) endVisit((ClassModifier1) n);
            else if (n instanceof ClassModifier2) endVisit((ClassModifier2) n);
            else if (n instanceof ClassModifier3) endVisit((ClassModifier3) n);
            else if (n instanceof ClassModifier4) endVisit((ClassModifier4) n);
            else if (n instanceof ClassModifier5) endVisit((ClassModifier5) n);
            else if (n instanceof ClassModifier6) endVisit((ClassModifier6) n);
            else if (n instanceof FieldModifier0) endVisit((FieldModifier0) n);
            else if (n instanceof FieldModifier1) endVisit((FieldModifier1) n);
            else if (n instanceof FieldModifier2) endVisit((FieldModifier2) n);
            else if (n instanceof FieldModifier3) endVisit((FieldModifier3) n);
            else if (n instanceof FieldModifier4) endVisit((FieldModifier4) n);
            else if (n instanceof FieldModifier5) endVisit((FieldModifier5) n);
            else if (n instanceof FieldModifier6) endVisit((FieldModifier6) n);
            else if (n instanceof MethodDeclarator0) endVisit((MethodDeclarator0) n);
            else if (n instanceof MethodDeclarator1) endVisit((MethodDeclarator1) n);
            else if (n instanceof MethodModifier0) endVisit((MethodModifier0) n);
            else if (n instanceof MethodModifier1) endVisit((MethodModifier1) n);
            else if (n instanceof MethodModifier2) endVisit((MethodModifier2) n);
            else if (n instanceof MethodModifier3) endVisit((MethodModifier3) n);
            else if (n instanceof MethodModifier4) endVisit((MethodModifier4) n);
            else if (n instanceof MethodModifier5) endVisit((MethodModifier5) n);
            else if (n instanceof MethodModifier6) endVisit((MethodModifier6) n);
            else if (n instanceof MethodModifier7) endVisit((MethodModifier7) n);
            else if (n instanceof MethodModifier8) endVisit((MethodModifier8) n);
            else if (n instanceof ConstructorModifier0) endVisit((ConstructorModifier0) n);
            else if (n instanceof ConstructorModifier1) endVisit((ConstructorModifier1) n);
            else if (n instanceof ConstructorModifier2) endVisit((ConstructorModifier2) n);
            else if (n instanceof ExplicitConstructorInvocation0) endVisit((ExplicitConstructorInvocation0) n);
            else if (n instanceof ExplicitConstructorInvocation1) endVisit((ExplicitConstructorInvocation1) n);
            else if (n instanceof ExplicitConstructorInvocation2) endVisit((ExplicitConstructorInvocation2) n);
            else if (n instanceof InterfaceModifier0) endVisit((InterfaceModifier0) n);
            else if (n instanceof InterfaceModifier1) endVisit((InterfaceModifier1) n);
            else if (n instanceof InterfaceModifier2) endVisit((InterfaceModifier2) n);
            else if (n instanceof InterfaceModifier3) endVisit((InterfaceModifier3) n);
            else if (n instanceof InterfaceModifier4) endVisit((InterfaceModifier4) n);
            else if (n instanceof InterfaceModifier5) endVisit((InterfaceModifier5) n);
            else if (n instanceof ExtendsInterfaces0) endVisit((ExtendsInterfaces0) n);
            else if (n instanceof ExtendsInterfaces1) endVisit((ExtendsInterfaces1) n);
            else if (n instanceof ConstantModifier0) endVisit((ConstantModifier0) n);
            else if (n instanceof ConstantModifier1) endVisit((ConstantModifier1) n);
            else if (n instanceof ConstantModifier2) endVisit((ConstantModifier2) n);
            else if (n instanceof AbstractMethodModifier0) endVisit((AbstractMethodModifier0) n);
            else if (n instanceof AbstractMethodModifier1) endVisit((AbstractMethodModifier1) n);
            else if (n instanceof AnnotationTypeElementDeclaration0) endVisit((AnnotationTypeElementDeclaration0) n);
            else if (n instanceof AnnotationTypeElementDeclaration1) endVisit((AnnotationTypeElementDeclaration1) n);
            else if (n instanceof AssertStatement0) endVisit((AssertStatement0) n);
            else if (n instanceof AssertStatement1) endVisit((AssertStatement1) n);
            else if (n instanceof SwitchLabel0) endVisit((SwitchLabel0) n);
            else if (n instanceof SwitchLabel1) endVisit((SwitchLabel1) n);
            else if (n instanceof SwitchLabel2) endVisit((SwitchLabel2) n);
            else if (n instanceof TryStatement0) endVisit((TryStatement0) n);
            else if (n instanceof TryStatement1) endVisit((TryStatement1) n);
            else if (n instanceof PrimaryNoNewArray0) endVisit((PrimaryNoNewArray0) n);
            else if (n instanceof PrimaryNoNewArray1) endVisit((PrimaryNoNewArray1) n);
            else if (n instanceof PrimaryNoNewArray2) endVisit((PrimaryNoNewArray2) n);
            else if (n instanceof PrimaryNoNewArray3) endVisit((PrimaryNoNewArray3) n);
            else if (n instanceof PrimaryNoNewArray4) endVisit((PrimaryNoNewArray4) n);
            else if (n instanceof Literal0) endVisit((Literal0) n);
            else if (n instanceof Literal1) endVisit((Literal1) n);
            else if (n instanceof Literal2) endVisit((Literal2) n);
            else if (n instanceof Literal3) endVisit((Literal3) n);
            else if (n instanceof Literal4) endVisit((Literal4) n);
            else if (n instanceof Literal5) endVisit((Literal5) n);
            else if (n instanceof Literal6) endVisit((Literal6) n);
            else if (n instanceof BooleanLiteral0) endVisit((BooleanLiteral0) n);
            else if (n instanceof BooleanLiteral1) endVisit((BooleanLiteral1) n);
            else if (n instanceof ClassInstanceCreationExpression0) endVisit((ClassInstanceCreationExpression0) n);
            else if (n instanceof ClassInstanceCreationExpression1) endVisit((ClassInstanceCreationExpression1) n);
            else if (n instanceof ArrayCreationExpression0) endVisit((ArrayCreationExpression0) n);
            else if (n instanceof ArrayCreationExpression1) endVisit((ArrayCreationExpression1) n);
            else if (n instanceof ArrayCreationExpression2) endVisit((ArrayCreationExpression2) n);
            else if (n instanceof ArrayCreationExpression3) endVisit((ArrayCreationExpression3) n);
            else if (n instanceof Dims0) endVisit((Dims0) n);
            else if (n instanceof Dims1) endVisit((Dims1) n);
            else if (n instanceof FieldAccess0) endVisit((FieldAccess0) n);
            else if (n instanceof FieldAccess1) endVisit((FieldAccess1) n);
            else if (n instanceof FieldAccess2) endVisit((FieldAccess2) n);
            else if (n instanceof MethodInvocation0) endVisit((MethodInvocation0) n);
            else if (n instanceof MethodInvocation1) endVisit((MethodInvocation1) n);
            else if (n instanceof MethodInvocation2) endVisit((MethodInvocation2) n);
            else if (n instanceof MethodInvocation3) endVisit((MethodInvocation3) n);
            else if (n instanceof MethodInvocation4) endVisit((MethodInvocation4) n);
            else if (n instanceof ArrayAccess0) endVisit((ArrayAccess0) n);
            else if (n instanceof ArrayAccess1) endVisit((ArrayAccess1) n);
            else if (n instanceof UnaryExpression0) endVisit((UnaryExpression0) n);
            else if (n instanceof UnaryExpression1) endVisit((UnaryExpression1) n);
            else if (n instanceof UnaryExpressionNotPlusMinus0) endVisit((UnaryExpressionNotPlusMinus0) n);
            else if (n instanceof UnaryExpressionNotPlusMinus1) endVisit((UnaryExpressionNotPlusMinus1) n);
            else if (n instanceof CastExpression0) endVisit((CastExpression0) n);
            else if (n instanceof CastExpression1) endVisit((CastExpression1) n);
            else if (n instanceof MultiplicativeExpression0) endVisit((MultiplicativeExpression0) n);
            else if (n instanceof MultiplicativeExpression1) endVisit((MultiplicativeExpression1) n);
            else if (n instanceof MultiplicativeExpression2) endVisit((MultiplicativeExpression2) n);
            else if (n instanceof AdditiveExpression0) endVisit((AdditiveExpression0) n);
            else if (n instanceof AdditiveExpression1) endVisit((AdditiveExpression1) n);
            else if (n instanceof ShiftExpression0) endVisit((ShiftExpression0) n);
            else if (n instanceof ShiftExpression1) endVisit((ShiftExpression1) n);
            else if (n instanceof ShiftExpression2) endVisit((ShiftExpression2) n);
            else if (n instanceof RelationalExpression0) endVisit((RelationalExpression0) n);
            else if (n instanceof RelationalExpression1) endVisit((RelationalExpression1) n);
            else if (n instanceof RelationalExpression2) endVisit((RelationalExpression2) n);
            else if (n instanceof RelationalExpression3) endVisit((RelationalExpression3) n);
            else if (n instanceof RelationalExpression4) endVisit((RelationalExpression4) n);
            else if (n instanceof EqualityExpression0) endVisit((EqualityExpression0) n);
            else if (n instanceof EqualityExpression1) endVisit((EqualityExpression1) n);
            else if (n instanceof AssignmentOperator0) endVisit((AssignmentOperator0) n);
            else if (n instanceof AssignmentOperator1) endVisit((AssignmentOperator1) n);
            else if (n instanceof AssignmentOperator2) endVisit((AssignmentOperator2) n);
            else if (n instanceof AssignmentOperator3) endVisit((AssignmentOperator3) n);
            else if (n instanceof AssignmentOperator4) endVisit((AssignmentOperator4) n);
            else if (n instanceof AssignmentOperator5) endVisit((AssignmentOperator5) n);
            else if (n instanceof AssignmentOperator6) endVisit((AssignmentOperator6) n);
            else if (n instanceof AssignmentOperator7) endVisit((AssignmentOperator7) n);
            else if (n instanceof AssignmentOperator8) endVisit((AssignmentOperator8) n);
            else if (n instanceof AssignmentOperator9) endVisit((AssignmentOperator9) n);
            else if (n instanceof AssignmentOperator10) endVisit((AssignmentOperator10) n);
            else if (n instanceof AssignmentOperator11) endVisit((AssignmentOperator11) n);
            throw new UnsupportedOperationException("visit(" + n.getClass().toString() + ")");
        }
    }

    public void ruleAction(int ruleNumber)
    {
        switch (ruleNumber)
        {

            //
            // Rule 3:  identifier ::= IDENTIFIER
            //
            case 3: {
               //#line 184 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new identifier(JavaParser.this, getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 4:  LPGUserAction ::= BlockStatementsopt
            //
            case 4:
                break;
            //
            // Rule 5:  LPGUserAction ::= $BeginAction BlockStatementsopt $EndAction
            //
            case 5: {
               //#line 194 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LPGUserAction0(getLeftIToken(), getRightIToken(),
                                       new AstToken(getRhsIToken(1)),
                                       (IBlockStatementsopt)getRhsSym(2),
                                       new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 6:  LPGUserAction ::= $BeginJava BlockStatementsopt $EndJava
            //
            case 6: {
               //#line 195 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LPGUserAction1(getLeftIToken(), getRightIToken(),
                                       new AstToken(getRhsIToken(1)),
                                       (IBlockStatementsopt)getRhsSym(2),
                                       new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 7:  LPGUserAction ::= $NoAction
            //
            case 7: {
               //#line 196 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LPGUserAction2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 8:  LPGUserAction ::= $NullAction
            //
            case 8: {
               //#line 197 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LPGUserAction3(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 9:  LPGUserAction ::= $BadAction
            //
            case 9: {
               //#line 198 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LPGUserAction4(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 10:  Type ::= PrimitiveType
            //
            case 10:
                break;
            //
            // Rule 11:  Type ::= ReferenceType
            //
            case 11:
                break;
            //
            // Rule 12:  PrimitiveType ::= NumericType
            //
            case 12:
                break;
            //
            // Rule 13:  PrimitiveType ::= boolean
            //
            case 13: {
               //#line 207 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new PrimitiveType(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 14:  NumericType ::= IntegralType
            //
            case 14:
                break;
            //
            // Rule 15:  NumericType ::= FloatingPointType
            //
            case 15:
                break;
            //
            // Rule 16:  IntegralType ::= byte
            //
            case 16: {
               //#line 212 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new IntegralType0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 17:  IntegralType ::= short
            //
            case 17: {
               //#line 213 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new IntegralType1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 18:  IntegralType ::= int
            //
            case 18: {
               //#line 214 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new IntegralType2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 19:  IntegralType ::= long
            //
            case 19: {
               //#line 215 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new IntegralType3(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 20:  IntegralType ::= char
            //
            case 20: {
               //#line 216 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new IntegralType4(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 21:  FloatingPointType ::= float
            //
            case 21: {
               //#line 218 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FloatingPointType0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 22:  FloatingPointType ::= double
            //
            case 22: {
               //#line 219 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FloatingPointType1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 23:  ReferenceType ::= ClassOrInterfaceType
            //
            case 23:
                break;
            //
            // Rule 24:  ReferenceType ::= TypeVariable
            //
            case 24:
                break;
            //
            // Rule 25:  ReferenceType ::= ArrayType
            //
            case 25:
                break;
            //
            // Rule 26:  ClassOrInterfaceType ::= ClassType
            //
            case 26:
                break;
            //
            // Rule 27:  ClassType ::= TypeName TypeArgumentsopt
            //
            case 27: {
               //#line 231 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassType(getLeftIToken(), getRightIToken(),
                                  (ITypeName)getRhsSym(1),
                                  (TypeArguments)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 28:  InterfaceType ::= TypeName TypeArgumentsopt
            //
            case 28: {
               //#line 233 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new InterfaceType(getLeftIToken(), getRightIToken(),
                                      (ITypeName)getRhsSym(1),
                                      (TypeArguments)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 29:  TypeName ::= identifier
            //
            case 29:
                break;
            //
            // Rule 30:  TypeName ::= TypeName . identifier
            //
            case 30: {
               //#line 236 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeName(getLeftIToken(), getRightIToken(),
                                 (ITypeName)getRhsSym(1),
                                 new AstToken(getRhsIToken(2)),
                                 (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 31:  ClassName ::= TypeName
            //
            case 31:
                break;
            //
            // Rule 32:  TypeVariable ::= identifier
            //
            case 32:
                break;
            //
            // Rule 33:  ArrayType ::= Type [ ]
            //
            case 33: {
               //#line 242 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ArrayType(getLeftIToken(), getRightIToken(),
                                  (IType)getRhsSym(1),
                                  new AstToken(getRhsIToken(2)),
                                  new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 34:  TypeParameter ::= TypeVariable TypeBoundopt
            //
            case 34: {
               //#line 244 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeParameter(getLeftIToken(), getRightIToken(),
                                      (identifier)getRhsSym(1),
                                      (TypeBound)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 35:  TypeBound ::= extends ClassOrInterfaceType AdditionalBoundListopt
            //
            case 35: {
               //#line 246 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeBound(getLeftIToken(), getRightIToken(),
                                  new AstToken(getRhsIToken(1)),
                                  (ClassType)getRhsSym(2),
                                  (IAdditionalBoundListopt)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 36:  AdditionalBoundList ::= AdditionalBound
            //
            case 36:
                break;
            //
            // Rule 37:  AdditionalBoundList ::= AdditionalBoundList AdditionalBound
            //
            case 37: {
               //#line 249 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new AdditionalBoundList(getLeftIToken(), getRightIToken(),
                                            (IAdditionalBoundList)getRhsSym(1),
                                            (AdditionalBound)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 38:  AdditionalBound ::= & InterfaceType
            //
            case 38: {
               //#line 251 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new AdditionalBound(getLeftIToken(), getRightIToken(),
                                        new AstToken(getRhsIToken(1)),
                                        (InterfaceType)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 39:  TypeArguments ::= < ActualTypeArgumentList >
            //
            case 39: {
               //#line 253 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeArguments(getLeftIToken(), getRightIToken(),
                                      new AstToken(getRhsIToken(1)),
                                      (IActualTypeArgumentList)getRhsSym(2),
                                      new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 40:  ActualTypeArgumentList ::= ActualTypeArgument
            //
            case 40:
                break;
            //
            // Rule 41:  ActualTypeArgumentList ::= ActualTypeArgumentList , ActualTypeArgument
            //
            case 41: {
               //#line 256 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ActualTypeArgumentList(getLeftIToken(), getRightIToken(),
                                               (IActualTypeArgumentList)getRhsSym(1),
                                               new AstToken(getRhsIToken(2)),
                                               (IActualTypeArgument)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 42:  ActualTypeArgument ::= ReferenceType
            //
            case 42:
                break;
            //
            // Rule 43:  ActualTypeArgument ::= Wildcard
            //
            case 43:
                break;
            //
            // Rule 44:  Wildcard ::= ? WildcardBoundsOpt
            //
            case 44: {
               //#line 261 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new Wildcard(getLeftIToken(), getRightIToken(),
                                 new AstToken(getRhsIToken(1)),
                                 (IWildcardBoundsOpt)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 45:  WildcardBounds ::= extends ReferenceType
            //
            case 45: {
               //#line 263 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new WildcardBounds0(getLeftIToken(), getRightIToken(),
                                        new AstToken(getRhsIToken(1)),
                                        (IReferenceType)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 46:  WildcardBounds ::= super ReferenceType
            //
            case 46: {
               //#line 264 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new WildcardBounds1(getLeftIToken(), getRightIToken(),
                                        new AstToken(getRhsIToken(1)),
                                        (IReferenceType)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 47:  PackageName ::= identifier
            //
            case 47:
                break;
            //
            // Rule 48:  PackageName ::= PackageName . identifier
            //
            case 48: {
               //#line 271 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new PackageName(getLeftIToken(), getRightIToken(),
                                    (IPackageName)getRhsSym(1),
                                    new AstToken(getRhsIToken(2)),
                                    (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 49:  ExpressionName ::= identifier
            //
            case 49:
                break;
            //
            // Rule 50:  ExpressionName ::= AmbiguousName . identifier
            //
            case 50: {
               //#line 280 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ExpressionName(getLeftIToken(), getRightIToken(),
                                       (IAmbiguousName)getRhsSym(1),
                                       new AstToken(getRhsIToken(2)),
                                       (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 51:  MethodName ::= identifier
            //
            case 51:
                break;
            //
            // Rule 52:  MethodName ::= AmbiguousName . identifier
            //
            case 52: {
               //#line 283 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodName(getLeftIToken(), getRightIToken(),
                                   (IAmbiguousName)getRhsSym(1),
                                   new AstToken(getRhsIToken(2)),
                                   (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 53:  PackageOrTypeName ::= identifier
            //
            case 53:
                break;
            //
            // Rule 54:  PackageOrTypeName ::= PackageOrTypeName . identifier
            //
            case 54: {
               //#line 286 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new PackageOrTypeName(getLeftIToken(), getRightIToken(),
                                          (IPackageOrTypeName)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 55:  AmbiguousName ::= identifier
            //
            case 55:
                break;
            //
            // Rule 56:  AmbiguousName ::= AmbiguousName . identifier
            //
            case 56: {
               //#line 289 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new AmbiguousName(getLeftIToken(), getRightIToken(),
                                      (IAmbiguousName)getRhsSym(1),
                                      new AstToken(getRhsIToken(2)),
                                      (identifier)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 57:  CompilationUnit ::= PackageDeclarationopt ImportDeclarationsopt TypeDeclarationsopt
            //
            case 57: {
               //#line 293 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new CompilationUnit(getLeftIToken(), getRightIToken(),
                                        (PackageDeclaration)getRhsSym(1),
                                        (IImportDeclarationsopt)getRhsSym(2),
                                        (ITypeDeclarationsopt)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 58:  ImportDeclarations ::= ImportDeclaration
            //
            case 58:
                break;
            //
            // Rule 59:  ImportDeclarations ::= ImportDeclarations ImportDeclaration
            //
            case 59: {
               //#line 296 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ImportDeclarations(getLeftIToken(), getRightIToken(),
                                           (IImportDeclarations)getRhsSym(1),
                                           (IImportDeclaration)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 60:  TypeDeclarations ::= TypeDeclaration
            //
            case 60:
                break;
            //
            // Rule 61:  TypeDeclarations ::= TypeDeclarations TypeDeclaration
            //
            case 61: {
               //#line 299 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeDeclarations(getLeftIToken(), getRightIToken(),
                                         (ITypeDeclarations)getRhsSym(1),
                                         (ITypeDeclaration)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 62:  PackageDeclaration ::= Annotationsopt package PackageName ;
            //
            case 62: {
               //#line 301 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new PackageDeclaration(getLeftIToken(), getRightIToken(),
                                           (IAnnotationsopt)getRhsSym(1),
                                           new AstToken(getRhsIToken(2)),
                                           (IPackageName)getRhsSym(3),
                                           new AstToken(getRhsIToken(4)))
                );
                break;
            }
            //
            // Rule 63:  ImportDeclaration ::= SingleTypeImportDeclaration
            //
            case 63:
                break;
            //
            // Rule 64:  ImportDeclaration ::= TypeImportOnDemandDeclaration
            //
            case 64:
                break;
            //
            // Rule 65:  ImportDeclaration ::= SingleStaticImportDeclaration
            //
            case 65:
                break;
            //
            // Rule 66:  ImportDeclaration ::= StaticImportOnDemandDeclaration
            //
            case 66:
                break;
            //
            // Rule 67:  SingleTypeImportDeclaration ::= import TypeName ;
            //
            case 67: {
               //#line 308 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new SingleTypeImportDeclaration(getLeftIToken(), getRightIToken(),
                                                    new AstToken(getRhsIToken(1)),
                                                    (ITypeName)getRhsSym(2),
                                                    new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 68:  TypeImportOnDemandDeclaration ::= import PackageOrTypeName . * ;
            //
            case 68: {
               //#line 310 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeImportOnDemandDeclaration(getLeftIToken(), getRightIToken(),
                                                      new AstToken(getRhsIToken(1)),
                                                      (IPackageOrTypeName)getRhsSym(2),
                                                      new AstToken(getRhsIToken(3)),
                                                      new AstToken(getRhsIToken(4)),
                                                      new AstToken(getRhsIToken(5)))
                );
                break;
            }
            //
            // Rule 69:  SingleStaticImportDeclaration ::= import static TypeName . identifier ;
            //
            case 69: {
               //#line 312 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new SingleStaticImportDeclaration(getLeftIToken(), getRightIToken(),
                                                      new AstToken(getRhsIToken(1)),
                                                      new AstToken(getRhsIToken(2)),
                                                      (ITypeName)getRhsSym(3),
                                                      new AstToken(getRhsIToken(4)),
                                                      (identifier)getRhsSym(5),
                                                      new AstToken(getRhsIToken(6)))
                );
                break;
            }
            //
            // Rule 70:  StaticImportOnDemandDeclaration ::= import static TypeName . * ;
            //
            case 70: {
               //#line 314 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new StaticImportOnDemandDeclaration(getLeftIToken(), getRightIToken(),
                                                        new AstToken(getRhsIToken(1)),
                                                        new AstToken(getRhsIToken(2)),
                                                        (ITypeName)getRhsSym(3),
                                                        new AstToken(getRhsIToken(4)),
                                                        new AstToken(getRhsIToken(5)),
                                                        new AstToken(getRhsIToken(6)))
                );
                break;
            }
            //
            // Rule 71:  TypeDeclaration ::= ClassDeclaration
            //
            case 71:
                break;
            //
            // Rule 72:  TypeDeclaration ::= InterfaceDeclaration
            //
            case 72:
                break;
            //
            // Rule 73:  TypeDeclaration ::= ;
            //
            case 73: {
               //#line 318 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeDeclaration(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 74:  ClassDeclaration ::= NormalClassDeclaration
            //
            case 74:
                break;
            //
            // Rule 75:  ClassDeclaration ::= EnumDeclaration
            //
            case 75:
                break;
            //
            // Rule 76:  NormalClassDeclaration ::= ClassModifiersopt class identifier TypeParametersopt Superopt Interfacesopt ClassBody
            //
            case 76: {
               //#line 325 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new NormalClassDeclaration(getLeftIToken(), getRightIToken(),
                                               (IClassModifiersopt)getRhsSym(1),
                                               new AstToken(getRhsIToken(2)),
                                               (identifier)getRhsSym(3),
                                               (TypeParameters)getRhsSym(4),
                                               (Super)getRhsSym(5),
                                               (Interfaces)getRhsSym(6),
                                               (ClassBody)getRhsSym(7))
                );
                break;
            }
            //
            // Rule 77:  ClassModifiers ::= ClassModifier
            //
            case 77:
                break;
            //
            // Rule 78:  ClassModifiers ::= ClassModifiers ClassModifier
            //
            case 78: {
               //#line 328 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifiers(getLeftIToken(), getRightIToken(),
                                       (IClassModifiers)getRhsSym(1),
                                       (IClassModifier)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 79:  ClassModifier ::= Annotation
            //
            case 79:
                break;
            //
            // Rule 80:  ClassModifier ::= public
            //
            case 80: {
               //#line 331 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 81:  ClassModifier ::= protected
            //
            case 81: {
               //#line 332 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 82:  ClassModifier ::= private
            //
            case 82: {
               //#line 333 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 83:  ClassModifier ::= abstract
            //
            case 83: {
               //#line 334 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier3(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 84:  ClassModifier ::= static
            //
            case 84: {
               //#line 335 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier4(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 85:  ClassModifier ::= final
            //
            case 85: {
               //#line 336 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier5(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 86:  ClassModifier ::= strictfp
            //
            case 86: {
               //#line 337 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassModifier6(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 87:  TypeParameters ::= < TypeParameterList >
            //
            case 87: {
               //#line 339 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeParameters(getLeftIToken(), getRightIToken(),
                                       new AstToken(getRhsIToken(1)),
                                       (ITypeParameterList)getRhsSym(2),
                                       new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 88:  TypeParameterList ::= TypeParameter
            //
            case 88:
                break;
            //
            // Rule 89:  TypeParameterList ::= TypeParameterList , TypeParameter
            //
            case 89: {
               //#line 342 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new TypeParameterList(getLeftIToken(), getRightIToken(),
                                          (ITypeParameterList)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          (TypeParameter)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 90:  Super ::= extends ClassType
            //
            case 90: {
               //#line 344 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new Super(getLeftIToken(), getRightIToken(),
                              new AstToken(getRhsIToken(1)),
                              (ClassType)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 91:  Interfaces ::= implements InterfaceTypeList
            //
            case 91: {
               //#line 351 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new Interfaces(getLeftIToken(), getRightIToken(),
                                   new AstToken(getRhsIToken(1)),
                                   (IInterfaceTypeList)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 92:  InterfaceTypeList ::= InterfaceType
            //
            case 92:
                break;
            //
            // Rule 93:  InterfaceTypeList ::= InterfaceTypeList , InterfaceType
            //
            case 93: {
               //#line 354 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new InterfaceTypeList(getLeftIToken(), getRightIToken(),
                                          (IInterfaceTypeList)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          (InterfaceType)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 94:  ClassBody ::= { ClassBodyDeclarationsopt }
            //
            case 94: {
               //#line 361 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassBody(getLeftIToken(), getRightIToken(),
                                  new AstToken(getRhsIToken(1)),
                                  (IClassBodyDeclarationsopt)getRhsSym(2),
                                  new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 95:  ClassBodyDeclarations ::= ClassBodyDeclaration
            //
            case 95:
                break;
            //
            // Rule 96:  ClassBodyDeclarations ::= ClassBodyDeclarations ClassBodyDeclaration
            //
            case 96: {
               //#line 364 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassBodyDeclarations(getLeftIToken(), getRightIToken(),
                                              (IClassBodyDeclarations)getRhsSym(1),
                                              (IClassBodyDeclaration)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 97:  ClassBodyDeclaration ::= ClassMemberDeclaration
            //
            case 97:
                break;
            //
            // Rule 98:  ClassBodyDeclaration ::= InstanceInitializer
            //
            case 98:
                break;
            //
            // Rule 99:  ClassBodyDeclaration ::= StaticInitializer
            //
            case 99:
                break;
            //
            // Rule 100:  ClassBodyDeclaration ::= ConstructorDeclaration
            //
            case 100:
                break;
            //
            // Rule 101:  ClassMemberDeclaration ::= FieldDeclaration
            //
            case 101:
                break;
            //
            // Rule 102:  ClassMemberDeclaration ::= MethodDeclaration
            //
            case 102:
                break;
            //
            // Rule 103:  ClassMemberDeclaration ::= ClassDeclaration
            //
            case 103:
                break;
            //
            // Rule 104:  ClassMemberDeclaration ::= InterfaceDeclaration
            //
            case 104:
                break;
            //
            // Rule 105:  ClassMemberDeclaration ::= ;
            //
            case 105: {
               //#line 375 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ClassMemberDeclaration(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 106:  FieldDeclaration ::= FieldModifiersopt Type VariableDeclarators ;
            //
            case 106: {
               //#line 377 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldDeclaration(getLeftIToken(), getRightIToken(),
                                         (IFieldModifiersopt)getRhsSym(1),
                                         (IType)getRhsSym(2),
                                         (IVariableDeclarators)getRhsSym(3),
                                         new AstToken(getRhsIToken(4)))
                );
                break;
            }
            //
            // Rule 107:  VariableDeclarators ::= VariableDeclarator
            //
            case 107:
                break;
            //
            // Rule 108:  VariableDeclarators ::= VariableDeclarators , VariableDeclarator
            //
            case 108: {
               //#line 380 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new VariableDeclarators(getLeftIToken(), getRightIToken(),
                                            (IVariableDeclarators)getRhsSym(1),
                                            new AstToken(getRhsIToken(2)),
                                            (IVariableDeclarator)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 109:  VariableDeclarator ::= VariableDeclaratorId
            //
            case 109:
                break;
            //
            // Rule 110:  VariableDeclarator ::= VariableDeclaratorId = VariableInitializer
            //
            case 110: {
               //#line 383 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new VariableDeclarator(getLeftIToken(), getRightIToken(),
                                           (IVariableDeclaratorId)getRhsSym(1),
                                           new AstToken(getRhsIToken(2)),
                                           (IVariableInitializer)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 111:  VariableDeclaratorId ::= identifier
            //
            case 111:
                break;
            //
            // Rule 112:  VariableDeclaratorId ::= VariableDeclaratorId [ ]
            //
            case 112: {
               //#line 386 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new VariableDeclaratorId(getLeftIToken(), getRightIToken(),
                                             (IVariableDeclaratorId)getRhsSym(1),
                                             new AstToken(getRhsIToken(2)),
                                             new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 113:  VariableInitializer ::= Expression
            //
            case 113:
                break;
            //
            // Rule 114:  VariableInitializer ::= ArrayInitializer
            //
            case 114:
                break;
            //
            // Rule 115:  FieldModifiers ::= FieldModifier
            //
            case 115:
                break;
            //
            // Rule 116:  FieldModifiers ::= FieldModifiers FieldModifier
            //
            case 116: {
               //#line 392 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifiers(getLeftIToken(), getRightIToken(),
                                       (IFieldModifiers)getRhsSym(1),
                                       (IFieldModifier)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 117:  FieldModifier ::= Annotation
            //
            case 117:
                break;
            //
            // Rule 118:  FieldModifier ::= public
            //
            case 118: {
               //#line 395 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 119:  FieldModifier ::= protected
            //
            case 119: {
               //#line 396 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 120:  FieldModifier ::= private
            //
            case 120: {
               //#line 397 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 121:  FieldModifier ::= static
            //
            case 121: {
               //#line 398 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier3(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 122:  FieldModifier ::= final
            //
            case 122: {
               //#line 399 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier4(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 123:  FieldModifier ::= transient
            //
            case 123: {
               //#line 400 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier5(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 124:  FieldModifier ::= volatile
            //
            case 124: {
               //#line 401 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FieldModifier6(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 125:  MethodDeclaration ::= MethodHeader MethodBody
            //
            case 125: {
               //#line 403 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodDeclaration(getLeftIToken(), getRightIToken(),
                                          (MethodHeader)getRhsSym(1),
                                          (IMethodBody)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 126:  MethodHeader ::= MethodModifiersopt TypeParametersopt ResultType MethodDeclarator Throwsopt
            //
            case 126: {
               //#line 405 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodHeader(getLeftIToken(), getRightIToken(),
                                     (IMethodModifiersopt)getRhsSym(1),
                                     (TypeParameters)getRhsSym(2),
                                     (IResultType)getRhsSym(3),
                                     (IMethodDeclarator)getRhsSym(4),
                                     (Throws)getRhsSym(5))
                );
                break;
            }
            //
            // Rule 127:  ResultType ::= Type
            //
            case 127:
                break;
            //
            // Rule 128:  ResultType ::= void
            //
            case 128: {
               //#line 408 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ResultType(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 129:  MethodDeclarator ::= identifier ( FormalParameterListopt )
            //
            case 129: {
               //#line 410 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodDeclarator0(getLeftIToken(), getRightIToken(),
                                          (identifier)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          (IFormalParameterListopt)getRhsSym(3),
                                          new AstToken(getRhsIToken(4)))
                );
                break;
            }
            //
            // Rule 130:  MethodDeclarator ::= MethodDeclarator [ ]
            //
            case 130: {
               //#line 412 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodDeclarator1(getLeftIToken(), getRightIToken(),
                                          (IMethodDeclarator)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          new AstToken(getRhsIToken(3)))
                );
                break;
            }
            //
            // Rule 131:  FormalParameterList ::= LastFormalParameter
            //
            case 131:
                break;
            //
            // Rule 132:  FormalParameterList ::= FormalParameters , LastFormalParameter
            //
            case 132: {
               //#line 415 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FormalParameterList(getLeftIToken(), getRightIToken(),
                                            (IFormalParameters)getRhsSym(1),
                                            new AstToken(getRhsIToken(2)),
                                            (LastFormalParameter)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 133:  FormalParameters ::= FormalParameter
            //
            case 133:
                break;
            //
            // Rule 134:  FormalParameters ::= FormalParameters , FormalParameter
            //
            case 134: {
               //#line 418 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FormalParameters(getLeftIToken(), getRightIToken(),
                                         (IFormalParameters)getRhsSym(1),
                                         new AstToken(getRhsIToken(2)),
                                         (FormalParameter)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 135:  FormalParameter ::= VariableModifiersopt Type VariableDeclaratorId
            //
            case 135: {
               //#line 420 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new FormalParameter(getLeftIToken(), getRightIToken(),
                                        (IVariableModifiersopt)getRhsSym(1),
                                        (IType)getRhsSym(2),
                                        (IVariableDeclaratorId)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 136:  VariableModifiers ::= VariableModifier
            //
            case 136:
                break;
            //
            // Rule 137:  VariableModifiers ::= VariableModifiers VariableModifier
            //
            case 137: {
               //#line 423 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new VariableModifiers(getLeftIToken(), getRightIToken(),
                                          (IVariableModifiers)getRhsSym(1),
                                          (IVariableModifier)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 138:  VariableModifier ::= final
            //
            case 138: {
               //#line 425 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new VariableModifier(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 139:  VariableModifier ::= Annotations
            //
            case 139:
                break;
            //
            // Rule 140:  LastFormalParameter ::= VariableModifiersopt Type ...opt VariableDeclaratorId
            //
            case 140: {
               //#line 428 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new LastFormalParameter(getLeftIToken(), getRightIToken(),
                                            (IVariableModifiersopt)getRhsSym(1),
                                            (IType)getRhsSym(2),
                                            (Ellipsisopt)getRhsSym(3),
                                            (IVariableDeclaratorId)getRhsSym(4))
                );
                break;
            }
            //
            // Rule 141:  MethodModifiers ::= MethodModifier
            //
            case 141:
                break;
            //
            // Rule 142:  MethodModifiers ::= MethodModifiers MethodModifier
            //
            case 142: {
               //#line 437 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifiers(getLeftIToken(), getRightIToken(),
                                        (IMethodModifiers)getRhsSym(1),
                                        (IMethodModifier)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 143:  MethodModifier ::= Annotations
            //
            case 143:
                break;
            //
            // Rule 144:  MethodModifier ::= public
            //
            case 144: {
               //#line 440 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 145:  MethodModifier ::= protected
            //
            case 145: {
               //#line 441 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 146:  MethodModifier ::= private
            //
            case 146: {
               //#line 442 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 147:  MethodModifier ::= abstract
            //
            case 147: {
               //#line 443 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier3(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 148:  MethodModifier ::= static
            //
            case 148: {
               //#line 444 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier4(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 149:  MethodModifier ::= final
            //
            case 149: {
               //#line 445 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier5(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 150:  MethodModifier ::= synchronized
            //
            case 150: {
               //#line 446 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier6(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 151:  MethodModifier ::= native
            //
            case 151: {
               //#line 447 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier7(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 152:  MethodModifier ::= strictfp
            //
            case 152: {
               //#line 448 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodModifier8(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 153:  Throws ::= throws ExceptionTypeList
            //
            case 153: {
               //#line 450 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new Throws(getLeftIToken(), getRightIToken(),
                               new AstToken(getRhsIToken(1)),
                               (IExceptionTypeList)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 154:  ExceptionTypeList ::= ExceptionType
            //
            case 154:
                break;
            //
            // Rule 155:  ExceptionTypeList ::= ExceptionTypeList , ExceptionType
            //
            case 155: {
               //#line 453 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ExceptionTypeList(getLeftIToken(), getRightIToken(),
                                          (IExceptionTypeList)getRhsSym(1),
                                          new AstToken(getRhsIToken(2)),
                                          (IExceptionType)getRhsSym(3))
                );
                break;
            }
            //
            // Rule 156:  ExceptionType ::= ClassType
            //
            case 156:
                break;
            //
            // Rule 157:  ExceptionType ::= TypeVariable
            //
            case 157:
                break;
            //
            // Rule 158:  MethodBody ::= Block
            //
            case 158:
                break;
            //
            // Rule 159:  MethodBody ::= ;
            //
            case 159: {
               //#line 459 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new MethodBody(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 160:  InstanceInitializer ::= Block
            //
            case 160:
                break;
            //
            // Rule 161:  StaticInitializer ::= static Block
            //
            case 161: {
               //#line 463 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new StaticInitializer(getLeftIToken(), getRightIToken(),
                                          new AstToken(getRhsIToken(1)),
                                          (Block)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 162:  ConstructorDeclaration ::= ConstructorModifiersopt ConstructorDeclarator Throwsopt ConstructorBody
            //
            case 162: {
               //#line 465 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorDeclaration(getLeftIToken(), getRightIToken(),
                                               (IConstructorModifiersopt)getRhsSym(1),
                                               (ConstructorDeclarator)getRhsSym(2),
                                               (Throws)getRhsSym(3),
                                               (ConstructorBody)getRhsSym(4))
                );
                break;
            }
            //
            // Rule 163:  ConstructorDeclarator ::= TypeParametersopt SimpleTypeName ( FormalParameterListopt )
            //
            case 163: {
               //#line 467 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorDeclarator(getLeftIToken(), getRightIToken(),
                                              (TypeParameters)getRhsSym(1),
                                              (identifier)getRhsSym(2),
                                              new AstToken(getRhsIToken(3)),
                                              (IFormalParameterListopt)getRhsSym(4),
                                              new AstToken(getRhsIToken(5)))
                );
                break;
            }
            //
            // Rule 164:  SimpleTypeName ::= identifier
            //
            case 164:
                break;
            //
            // Rule 165:  ConstructorModifiers ::= ConstructorModifier
            //
            case 165:
                break;
            //
            // Rule 166:  ConstructorModifiers ::= ConstructorModifiers ConstructorModifier
            //
            case 166: {
               //#line 472 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorModifiers(getLeftIToken(), getRightIToken(),
                                             (IConstructorModifiers)getRhsSym(1),
                                             (IConstructorModifier)getRhsSym(2))
                );
                break;
            }
            //
            // Rule 167:  ConstructorModifier ::= Annotations
            //
            case 167:
                break;
            //
            // Rule 168:  ConstructorModifier ::= public
            //
            case 168: {
               //#line 475 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorModifier0(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 169:  ConstructorModifier ::= protected
            //
            case 169: {
               //#line 476 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorModifier1(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 170:  ConstructorModifier ::= private
            //
            case 170: {
               //#line 477 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new ConstructorModifier2(getRhsIToken(1))
                );
                break;
            }
            //
            // Rule 171:  ConstructorBody ::= { ExplicitConstructorInvocationopt BlockStatementsopt }
            //
            case 171: {
               //#line 479 "/Developer/eclipse-3.3.2/plugins/lpg.generator_2.0.16/templates/java/btParserTemplateF.gi"
                setResult(
                    new 