/*******************************************************************************
 * Copyright (c) 2012 GK Software AG and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Stephan Herrmann - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.internal.ui.text.correction.proposals;

import java.util.Collection;

import org.eclipse.core.runtime.CoreException;

import org.eclipse.text.edits.TextEditGroup;

import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.Block;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.Expression;
import org.eclipse.jdt.core.dom.FieldAccess;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.IfStatement;
import org.eclipse.jdt.core.dom.InfixExpression;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.ParameterizedType;
import org.eclipse.jdt.core.dom.PrimitiveType;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.ReturnStatement;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.Statement;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.VariableDeclarationFragment;
import org.eclipse.jdt.core.dom.VariableDeclarationStatement;
import org.eclipse.jdt.core.dom.rewrite.ASTRewrite;
import org.eclipse.jdt.core.dom.rewrite.ImportRewrite;
import org.eclipse.jdt.core.dom.rewrite.ListRewrite;

import org.eclipse.jdt.internal.corext.codemanipulation.StubUtility;
import org.eclipse.jdt.internal.corext.dom.ASTNodeFactory;
import org.eclipse.jdt.internal.corext.dom.ASTNodes;
import org.eclipse.jdt.internal.corext.dom.ScopeAnalyzer;
import org.eclipse.jdt.internal.corext.fix.FixMessages;
import org.eclipse.jdt.internal.corext.fix.LinkedProposalPositionGroup;

import org.eclipse.jdt.internal.ui.JavaPluginImages;

/**
 * Fix for {@link IProblem#NullableFieldReference}:
 * Extract the field reference to a fresh local variable.
 * Add a null check for that local variable and move
 * the dereference into the then-block of this null-check:
 * <pre>
 * {@code @Nullable Exception e;}
 * void test() {
 *     e.printStackTrace();
 * }</pre>
 * will be converted to:
 * <pre>
 * {@code @Nullable Exception e;}
 * void test() {
 *     Exception e2 = e;
 *     if (e2 != null) {
 *         e2.printStackTrace();
 *     } else {
 *         // TODO handle null value
 *     }
 * }</pre>
 * 
 * @since 3.9
 */
public class ExtractToNullCheckedLocalProposal extends LinkedCorrectionProposal {
	
	private static final String LOCAL_NAME_POSITION_GROUP = "localName"; //$NON-NLS-1$
	
	private SimpleName fieldReference;
	private CompilationUnit compilationUnit;
	private ASTNode enclosingMethod; // MethodDeclaration or Initializer 

	public ExtractToNullCheckedLocalProposal(ICompilationUnit cu, CompilationUnit compilationUnit, SimpleName fieldReference, ASTNode enclosingMethod) {
		super(FixMessages.ExtractToNullCheckedLocalProposal_extractToCheckedLocal_proposalName, cu, null, 100, JavaPluginImages.get(JavaPluginImages.IMG_CORRECTION_CHANGE));
		this.compilationUnit= compilationUnit;
		this.fieldReference= fieldReference;
		this.enclosingMethod= enclosingMethod;
	}
	
	@Override
	protected ASTRewrite getRewrite() throws CoreException {
		
		// infrastructure:
		AST ast= this.compilationUnit.getAST();
		ASTRewrite rewrite= ASTRewrite.create(ast);
		ImportRewrite imports= ImportRewrite.create(this.compilationUnit, true);
		TextEditGroup group= new TextEditGroup(FixMessages.ExtractToNullCheckedLocalProposal_extractCheckedLocal_editName);
		LinkedProposalPositionGroup localNameGroup= new LinkedProposalPositionGroup(LOCAL_NAME_POSITION_GROUP);
		getLinkedProposalModel().addPositionGroup(localNameGroup);

		// AST context:
		Statement stmt= (Statement) ASTNodes.getParent(this.fieldReference, Statement.class);
		ASTNode parent= stmt.getParent();
		ListRewrite blockRewrite= null;
		Block block;
		if (parent instanceof Block) {
			// modifying statement list of the parent block
			block= (Block) parent;
			blockRewrite= rewrite.getListRewrite(block, Block.STATEMENTS_PROPERTY);
		} else {
			// replacing statement with a new block
			block= ast.newBlock();			
		}

		Expression toReplace;
		ASTNode directParent= this.fieldReference.getParent();
		if (directParent instanceof FieldAccess) {
			toReplace= (Expression) directParent;
		} else if (directParent instanceof QualifiedName && this.fieldReference.getLocationInParent() == QualifiedName.NAME_PROPERTY) {
			toReplace= (Expression) directParent;
		} else {
			toReplace= this.fieldReference;
		}

		// new local declaration initialized from the field reference
		VariableDeclarationFragment local = ast.newVariableDeclarationFragment();
		VariableDeclarationStatement localDecl= ast.newVariableDeclarationStatement(local);
		// ... type
		localDecl.setType(newType(toReplace.resolveTypeBinding(), ast, imports));
		localDecl.modifiers().add(ast.newModifier(Modifier.ModifierKeyword.FINAL_KEYWORD));
		// ... name
		String localName= proposeLocalName(this.fieldReference, this.compilationUnit, getCompilationUnit().getJavaProject());
		local.setName(ast.newSimpleName(localName));
		// ... initialization
		local.setInitializer((Expression) ASTNode.copySubtree(ast, toReplace));
		
		if (blockRewrite != null)
			blockRewrite.insertBefore(localDecl, stmt, group);
		else
			block.statements().add(localDecl);
		
		// prepare replacing stmt with a wrapper
		Statement statementToMove = (Statement) 
				(blockRewrite != null ? blockRewrite.createMoveTarget(stmt, stmt) : rewrite.createMoveTarget(stmt));
		
		// if statement:
		IfStatement ifStmt= ast.newIfStatement();
		
		// condition:
		InfixExpression nullCheck= ast.newInfixExpression();
		nullCheck.setLeftOperand(ast.newSimpleName(localName));
		nullCheck.setRightOperand(ast.newNullLiteral());
		nullCheck.setOperator(InfixExpression.Operator.NOT_EQUALS);
		ifStmt.setExpression(nullCheck);
		
		// then block: the original statement
		Block thenBlock = ast.newBlock();
		thenBlock.statements().add(statementToMove);
		ifStmt.setThenStatement(thenBlock);
		// ... but with the field reference replaced by the new local:
		SimpleName dereferencedName= ast.newSimpleName(localName);
		rewrite.replace(toReplace, dereferencedName, group);

		
		// else block: a TODO comment
		Block elseBlock = ast.newBlock();
		String elseStatement= "// TODO "+FixMessages.ExtractToNullCheckedLocalProposal_todoHandleNullDescription; //$NON-NLS-1$
		if (stmt instanceof ReturnStatement) {
			Type returnType= newType(((ReturnStatement)stmt).getExpression().resolveTypeBinding(), ast, imports);
			ReturnStatement returnStatement= ast.newReturnStatement();
			returnStatement.setExpression(ASTNodeFactory.newDefaultExpression(ast, returnType, 0));
			elseStatement+= '\n'+ASTNodes.asFormattedString(returnStatement, 0, String.valueOf('\n'), getCompilationUnit().getJavaProject().getOptions(true));
		}

		ReturnStatement todoNode= (ReturnStatement) rewrite.createStringPlaceholder(elseStatement, ASTNode.RETURN_STATEMENT);
		elseBlock.statements().add(todoNode);
		ifStmt.setElseStatement(elseBlock);
		
		// link all three occurrences of the new local variable:
		addLinkedPosition(rewrite.track(local.getName()), true/*first*/, LOCAL_NAME_POSITION_GROUP);
		addLinkedPosition(rewrite.track(nullCheck.getLeftOperand()), false, LOCAL_NAME_POSITION_GROUP);
		addLinkedPosition(rewrite.track(dereferencedName), false, LOCAL_NAME_POSITION_GROUP);

		if (blockRewrite != null) {
			// inside a block replace old statement with wrapping if-statement
			blockRewrite.replace(stmt, ifStmt, group);
		} else {
			// did not have a block: add if-statement to new block
			block.statements().add(ifStmt);
			// and replace the single statement with this block
			rewrite.replace(stmt, block, group);
		}
		
		return rewrite;
	}

	String proposeLocalName(SimpleName fieldName, CompilationUnit root, IJavaProject javaProject) {
		// don't propose names that are already in use:
		Collection<String> variableNames= new ScopeAnalyzer(root).getUsedVariableNames(this.enclosingMethod.getStartPosition(), this.enclosingMethod.getLength());
		String[] names = new String[variableNames.size()+1];
		variableNames.toArray(names);
		// don't propose the field name itself, either:
		String identifier= fieldName.getIdentifier();
		names[names.length-1] = identifier;
		return StubUtility.getLocalNameSuggestions(javaProject, identifier, 0, names)[0];
	}

	/** 
	 * Create a fresh type reference
	 * @param typeBinding the type we want to refer to
	 * @param ast AST for creating new nodes
	 * @param imports use this for optimal type names
	 * @return a fully features non-null type reference (can be parameterized and/or array).
	 */
	public static Type newType(ITypeBinding typeBinding, AST ast, ImportRewrite imports) {
		// unwrap array type:
		int dimensions= typeBinding.getDimensions();
		if (dimensions > 0)
			typeBinding= typeBinding.getElementType();
		
		// unwrap parameterized type:
		ITypeBinding[] typeArguments= typeBinding.getTypeArguments();
		typeBinding= typeBinding.getErasure();	

		// create leaf type:
		Type elementType = (typeBinding.isPrimitive())
					? ast.newPrimitiveType(PrimitiveType.toCode(typeBinding.getName()))
					: ast.newSimpleType(ast.newName(imports.addImport(typeBinding)));

		// re-wrap as parameterized type:
		if (typeArguments.length > 0) {
			ParameterizedType parameterizedType= ast.newParameterizedType(elementType);
			for (ITypeBinding typeArgument : typeArguments)
				parameterizedType.typeArguments().add(newType(typeArgument, ast, imports));
			elementType = parameterizedType;
		}
		
		// re-wrap as array type:
		if (dimensions > 0)
			return ast.newArrayType(elementType, dimensions);
		else
			return elementType;
	}
}