/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id$
 * /
 *******************************************************************************/

package org.eclipse.jet;


import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.Platform;
import org.eclipse.jet.internal.InternalJET2Platform;
import org.eclipse.jet.internal.l10n.JET2Messages;
import org.eclipse.jet.internal.xpath.AnnotationManagerImpl;
import org.eclipse.jet.internal.xpath.functions.StringFunction;
import org.eclipse.jet.taglib.JET2TagException;
import org.eclipse.jet.xpath.DefaultXPathFunctionResolver;
import org.eclipse.jet.xpath.IAnnotationManager;
import org.eclipse.jet.xpath.NodeSet;
import org.eclipse.jet.xpath.XPath;
import org.eclipse.jet.xpath.XPathException;
import org.eclipse.jet.xpath.XPathExpression;
import org.eclipse.jet.xpath.XPathFactory;
import org.eclipse.jet.xpath.XPathFunctionMetaData;
import org.eclipse.jet.xpath.XPathRuntimeException;
import org.eclipse.jet.xpath.XPathUtil;
import org.eclipse.jet.xpath.XPathVariableResolver;
import org.eclipse.jet.xpath.inspector.AddElementException;
import org.eclipse.jet.xpath.inspector.CopyElementException;
import org.eclipse.jet.xpath.inspector.ExpandedName;
import org.eclipse.jet.xpath.inspector.IElementInspector;
import org.eclipse.jet.xpath.inspector.INodeInspector;
import org.eclipse.jet.xpath.inspector.InspectorManager;
import org.eclipse.jet.xpath.inspector.InvalidChildException;
import org.eclipse.jet.xpath.inspector.SimpleElementRequiresValueException;
import org.eclipse.jet.xpath.inspector.INodeInspector.NodeKind;


/**
 * Context Extender that understands XPath processing.
 *
 */
public final class XPathContextExtender extends AbstractContextExtender implements XPathVariableResolver
{

  private static boolean DEBUG = InternalJET2Platform.getDefault().isDebugging()
  && Boolean.valueOf(Platform.getDebugOption("org.eclipse.jet/debug/xpath/compilations")).booleanValue(); //$NON-NLS-1$

  private static final class ContextData
  {
    private final IAnnotationManager annotationManager = new AnnotationManagerImpl();
    
    private DefaultXPathFunctionResolver customFunctionResolver = null;
    
    private final Map knownXPathExpressions = new HashMap();
  }

  /**
   * @param context
   * @deprecated Use {@link #getInstance(JET2Context)}. This method will be made private in the near future.
   */
  public XPathContextExtender(JET2Context context)
  {
    super(context);
  }

  /**
   * Factory method for XPathContextExtenders
   * @param context the JET2Context that is extended
   * @return an XPathContextExtender
   */
  public static XPathContextExtender getInstance(JET2Context context)
  {
    return new XPathContextExtender(context);
  }
  /* (non-Javadoc)
   * @see org.eclipse.jet.AbstractContextExtender#createExtendedData(org.eclipse.jet.JET2Context)
   */
  protected Object createExtendedData(JET2Context context)
  {
    final ContextData contextData = new ContextData();
    
    XPath xp = XPathFactory.newInstance().newXPath(contextData.annotationManager);
    // Add one a custom function resolver which delegates to the resolver installed by default.
    contextData.customFunctionResolver = new DefaultXPathFunctionResolver(xp.getXPathFunctionResolver());
    
    return contextData;
  }

  private ContextData getData()
  {
    return (ContextData)getExtendedData();
  }

  public Object resolveVariable(String name)
  {
    //		return getData().xpathVariableMap.get(name);
    if (getContext().hasVariable(name))
    {
      try
      {
        return getContext().getVariable(name);
      }
      catch (JET2TagException e)
      {
        // wont' happen, we checked to make sure its ok.;
        return null;
      }
    }
    else
    {
      return null;
    }
  }

  /**
   * Resolve the given XPath expression as a string value. Note that if the XPath expression
   * returns an empty Node set, this method returns <code>null</code>
   * @param xpathContextObject the xpath context
   * @param selectXPath the XPath expression
   * @return the string value of the XPath expression, or <code>null</code> if the expression resulted in an empty node set.
   * @throws JET2TagException if an error occurs during expression evaluation
   */
  public String resolveAsString(Object xpathContextObject, String selectXPath) throws JET2TagException
  {
    Object resultObject = resolveAsObject(xpathContextObject, selectXPath);

    String result = null;
    if(resultObject instanceof NodeSet)
    {
      if(((NodeSet)resultObject).size() > 0)
      {
        result = XPathUtil.xpathString(resultObject);
      }
    } 
    else if(resultObject != null)
    {
      result = XPathUtil.xpathString(resultObject);
    }
    return result;
  }

  public Object resolveSingle(Object xpathContextObject, String selectXPath) throws JET2TagException
  {
    try
    {
      XPathExpression expr = compileXPath(selectXPath);
      return expr.evaluateAsSingleNode(xpathContextObject);
    }
    catch (XPathException e)
    {
      throw new JET2TagException(e);
    }

  }

  /**
   * @param selectXPath
   * @return
   * @throws XPathException
   */
  private XPathExpression compileXPath(String selectXPath) throws XPathException
  {
    Object result = getData().knownXPathExpressions.get(selectXPath);
    if (result == null)
    {
      if(DEBUG) System.out.println("XPath compile of " + selectXPath); //$NON-NLS-1$
      XPath xp = XPathFactory.newInstance().newXPath(getData().annotationManager);
      // install the custom resolver.
      xp.setXPathFunctionResolver(getData().customFunctionResolver);
      xp.setXPathVariableResolver(this);
      try
      {
        result = xp.compile(selectXPath);
        if(DEBUG) System.out.println("  compiled to " + result); //$NON-NLS-1$
      }
      catch (XPathException e)
      {
        result = e;
        if(DEBUG) System.out.println("  exception " + result); //$NON-NLS-1$
      }
      getData().knownXPathExpressions.put(selectXPath, result);
    } else {
      if(DEBUG) System.out.println("XPath cache hit on " + selectXPath); //$NON-NLS-1$
    }
    
    if(result instanceof XPathExpression) {
      return (XPathExpression) result;
    }
    else
    {
      throw (XPathException) result;
    }
  }

  public Object currentXPathContextObject()
  {
    return getContext().getSource();
  }

  public Object[] resolve(Object xpathContextObject, String selectXPath) throws JET2TagException
  {
    try
    {
      XPathExpression expr = compileXPath(selectXPath);
      return expr.evaluateAsNodeSet(xpathContextObject).toArray();
    }
    catch (XPathException e)
    {
      throw new JET2TagException(e);
    }
  }

  /**
   * Resolve an xpath expression as a boolean result according to the
   * XPath rules.
   * <p>
   * TODO Add link to XPath 1.0 spec for clarification
   * </p>
   * @param xpathContext the XPath context object
   * @param testXPath the XPath expression
   * @return <code>true</code> or <code>false</code>
   * @throws JET2TagException if an error occurs in evaluating the expression.
   */
  public boolean resolveTest(Object xpathContext, String testXPath) throws JET2TagException
  {
    try
    {
      XPathExpression expr = compileXPath(testXPath);
      return expr.evaluateAsBoolean(xpathContext);
    }
    catch (XPathException e)
    {
      throw new JET2TagException(e);
    }
  }

  public boolean setAttribute(Object element, String name, String bodyContent) throws JET2TagException
  {
    IElementInspector elementInspector = getElementInspector(element);

    boolean isSet = false;
    try
    {
      isSet = elementInspector.createAttribute(element, name, bodyContent);
    }
    catch (UnsupportedOperationException e)
    {
      // continue, ;
    }
    
    if(!isSet && getData().annotationManager != null) {
      Object annotation = getData().annotationManager.getAnnotationObject(element);
      elementInspector = getElementInspector(annotation);
      isSet = elementInspector.createAttribute(annotation, name, bodyContent);
    }
    return isSet;
  }

  /**
   * Resolve dynamic XPath expressions {...} within the pass value
   * @param value a string containing zero or more dynamic xpath expressions
   * @return the string with all dynamic xpath expressions resolved
   * @throws JET2TagException if an Xpath evaluation error occurs
   */
  public String resolveDynamic(String value) throws JET2TagException
  {
    Object context = currentXPathContextObject();
    Pattern dynXpathPattern = Pattern.compile("\\{(.*?)}"); // look for a sequence of characters between { and } //$NON-NLS-1$
    StringBuffer buffer = new StringBuffer(value);
    Matcher matcher = dynXpathPattern.matcher(buffer);
    int i = 0;
    while (i < buffer.length() && matcher.find(i))
    {
      String xpath = matcher.group(1);
      String resolved = resolveAsString(context, xpath);
      if (resolved == null)
      {
        String msg = JET2Messages.XPath_DynamicExpressionIsNull;
        throw new JET2TagException(MessageFormat.format(msg, new Object []{ xpath }));
      }
      buffer.replace(matcher.start(), matcher.end(), resolved);
      // next scan starts at end of this match, adjusted for the size difference between
      // the replacement text (resolved.length()) and the replaced text xpath.length() + 2 (for { & }).
      i = matcher.end() + resolved.length() - (xpath.length() + 2);
    }
    return buffer.toString();
  }

  private IElementInspector getElementInspector(Object element) throws JET2TagException
  {
    final INodeInspector inspector = InspectorManager.getInstance().getInspector(element);
    if (inspector == null || inspector.getNodeKind(element) != NodeKind.ELEMENT || !(inspector instanceof IElementInspector))
    {
      throw new JET2TagException(JET2Messages.XPath_NotAnElement);
    }
    return (IElementInspector)inspector;
  }

  public Object addElement(Object parent, String name) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(parent);

    try
    {
      return inspector.addElement(parent, new ExpandedName(name), null);
    }
    catch (SimpleElementRequiresValueException e)
    {
      throw new JET2TagException(JET2Messages.XPath_UseAddTextElement);
    }
    catch (InvalidChildException e)
    {
      // cannot happend, as we're passing null as the third argument;
      throw new JET2TagException(e);
    }
    catch(UnsupportedOperationException e)
    {
      throw convertToTagException(e);
    }
  }

  public void removeElement(Object element) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(element);

    try
    {
      inspector.removeElement(element);
    }
    catch (UnsupportedOperationException e)
    {
      throw convertToTagException(e);
    }

  }

  /**
   * @param e
   * @return
   */
  private JET2TagException convertToTagException(UnsupportedOperationException e)
  {
    // FIXME: add a message for this
    return new JET2TagException(e.toString());
  }

  /**
   * Copy <code>srcElement</code> as a new element with the specified name under <code>tgtParent</code>.
   * If <code>recursive</code> is <code>true</code>, then all the contained children of <code>srcElement</code>
   * are copied, otherwise, only the element and its attributes are copied.
   * @param srcElement the element to copy
   * @param tgtParent the parent element that will contain the copy
   * @param name the name of the copied element
   * @param recursive <code>true</code> if contained chidren are to be copied, too.
   * @return the newly copied element
   * @throws JET2TagException an error occurs
   */
  public Object copyElement(Object srcElement, Object tgtParent, String name, boolean recursive) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(tgtParent);

    try
    {
      return inspector.copyElement(tgtParent, srcElement, name, recursive);
    }
    catch (CopyElementException e)
    {
      throw new JET2TagException(e.getLocalizedMessage(), e);
    }
    catch (UnsupportedOperationException e)
    {
      throw convertToTagException(e);
    }
  }

  /**
   * Create a new text (simple) element whose content is set to <code>bodyContent</code>.
   * @param parentElement the parent of the new element.
   * @param name the name of the new element.
   * @param bodyContent the content.
   * @return the new Element.
   * @throws JET2TagException if the element cannot be added.
   */
  public Object addTextElement(Object parentElement, String name, String bodyContent) throws JET2TagException
  {
    return addTextElement(parentElement, name, bodyContent, false);
  }

  /**
   * Create a new text (simple) element whose content is set to <code>bodyContent</code>.
   * @param parentElement the parent of the new element.
   * @param name the name of the new element.
   * @param bodyContent the content.
   * @param asCData if <code>true</code>, create the element as a CDATA section, of possible
   * @return the new element.
   * @throws JET2TagException if the element cannot be added.
   */
  public Object addTextElement(Object parentElement, String name, String bodyContent, boolean asCData) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(parentElement);

    try
    {
      return inspector.addTextElement(parentElement, name, bodyContent, asCData);
    }
    catch (AddElementException e)
    {
      throw new JET2TagException(e.getLocalizedMessage(), e);
    }
    catch (UnsupportedOperationException e) {
      throw convertToTagException(e);
    }
    
  }
  /**
   * Resolve the XPath expression, returning an object. Unlike the other resolve methods, this
   * method performs no type conversions on the XPath results.
   * @param contextObject the context object to which the XPath expression is relative.
   * @param selectXPath the XPath expression
   * @return the result of the expression evaluation.
   * @throws JET2TagException if an error occurs.
   */
  public Object resolveAsObject(Object contextObject, String selectXPath) throws JET2TagException
  {
    try
    {
      XPathExpression expr = compileXPath(selectXPath);
      return expr.evaluate(contextObject);
    }
    catch (XPathException e)
    {
      throw new JET2TagException(e.getMessage(), e);
    }
    catch(XPathRuntimeException e) {
      throw new JET2TagException(e.getMessage(), e);
    }
  }
  
  /**
   * Return the value of the named attribute on the passed element.
   * @param element the element containing the attribute
   * @param attributeName the attribute name
   * @return the attribute value
   * @throws JET2TagException
   */
  public Object getAttributeValue(Object element, String attributeName) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(element);

    final Object namedAttribute = inspector.getNamedAttribute(element, new ExpandedName(attributeName));
    INodeInspector attrInspector = InspectorManager.getInstance().getInspector(namedAttribute);
    if(attrInspector != null) {
      return attrInspector.stringValueOf(namedAttribute);
    } else {
      throw new JET2TagException(MessageFormat.format(JET2Messages.XPath_NoValueForAttribute, new Object[] {attributeName}));
    }
  }
  
  /**
   * Remove the named attribute from the specified element.
   * @param element the element containing the attribute
   * @param attributeName the attribute to remove
   * @throws JET2TagException if the attribute cannot be removed (because it is required), 
   * or if <code>element</code> is not a recognized element.
   */
  public void removeAttribute(Object element, String attributeName) throws JET2TagException
  {
    IElementInspector inspector = getElementInspector(element);
    try
    {
      inspector.removeAttribute(element, attributeName);
    }
    catch (UnsupportedOperationException e)
    {
      throw convertToTagException(e);
    }
    
  }
  
  /**
   * Return the string value of the passed object. This equivalent to calling the XPath string() function
   * on the passed object.
   * @param object the object to examine.
   * @return the string value. 
   * @throws JET2TagException if an error occurs.
   */
  public String getContent(Object object) throws JET2TagException {
    return StringFunction.evaluate(object);
  }
  
  /**
   * Add the passed list of XPath function definitions to the XPath processor.
   * @param functionData possible empty array of {@link XPathFunctionMetaData} instances.
   */
  public void addCustomFunctions(XPathFunctionMetaData functionData[]) 
  {
    final DefaultXPathFunctionResolver resolver = getData().customFunctionResolver;
    for (int i = 0; i < functionData.length; i++)
    {
      resolver.addFunction(functionData[i]);
    }
  }
}
