/**
 * <copyright>
 *
 * Copyright (c) 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: V2CodeGenVisitor.java,v 1.2 2007/05/02 00:11:29 pelder Exp $
 */
package org.eclipse.jet.internal.compiler;

import org.eclipse.jet.JET2Context;
import org.eclipse.jet.JET2Writer;
import org.eclipse.jet.core.parser.ast.JETASTElement;
import org.eclipse.jet.core.parser.ast.TextElement;
import org.eclipse.jet.core.parser.ast.XMLElement;
import org.eclipse.jet.internal.JavaUtil;

/**
 * AST Visitor used by code generation
 */
public class V2CodeGenVisitor extends SafeJETASTVisitor
{
  private static final String UNDERSCORE = "_"; //$NON-NLS-1$

  private final String textPrefix = "TEXT_"; //$NON-NLS-1$
  private final String tagVarPrefix = "_jettag_"; //$NON-NLS-1$

  /**
   * @param context
   * @param out
   * @param initialIndent
   */
  public V2CodeGenVisitor(JET2Context context, JET2Writer out, String stdIndent)
  {
    super(context, out, stdIndent);
  }
  
  public V2CodeGenVisitor(JET2Context context, JET2Writer out)
  {
    this(context, out, "    "); //$NON-NLS-1$
  }

  public String tagVariableName(XMLElement element)
  {
    return tagVar(element);
  }

  public String tagVar(XMLElement element)
  {
    return tagVarPrefix + tagVariableSuffix(element);
  }

  /**
   * Return the name of a unique variable to refer to the static information about the passed element.
   * @param element
   * @return the variable name
   */
  public String tagInfoVariableName(XMLElement element)
  {
    return tagInfoVar(element);
  }

  /**
   * Return the name of a unique variable to refer to the static information about the passed element.
   * @param element
   * @return the variable name
   */
  public String tagInfoVar(XMLElement element)
  {
    return tagVariableSuffix(element);
  }

  /**
   * Create a unique suffix of all variables that refer to the passed element
   * @param element an XML Element
   * @return the tag variable name
   */
  private String tagVariableSuffix(XMLElement element)
  {
    return element.getNSPrefix() + UNDERSCORE + element.getTagNCName() + UNDERSCORE + element.getLine() + UNDERSCORE
      + element.getColumn();
  }

  public String textVar(int i) {
    return textPrefix + i;
  }
  
  public boolean requiresOwnWriter(XMLElement element) {
    return element.getTagDefinition().requiresNewWriter();
  }
  
  public String parentTagVar(XMLElement element) {
    XMLElement parentXMLElement = null;
    for(JETASTElement astParent = element.getParent(); astParent != null; astParent = astParent.getParent()) 
    {
      if(astParent instanceof XMLElement) {
        parentXMLElement = (XMLElement)astParent;
        break;
      }
    }
    return parentXMLElement == null ? "null" : tagVar(parentXMLElement); //$NON-NLS-1$
  }
  
  public String textExpr(TextElement text) {
    return JavaUtil.asJavaQuoteStringWithNLRemoved(text.getText());
  }
  
  /**
   * Return the text element as an array of Java text constants.
   * @param text the text element
   * @param nlConstantName the name of the constant to use for NL characters
   * @return an array properly formatted Java quoted strings (including quotes) and NL constants
   */
  public String[] textConstants(TextElement text, String nlConstantName) {
    return JavaUtil.asJavaQuotedStrings(text.getText(), nlConstantName);
  }
  
  public String nlsComment(TextElement text) {
    return JavaUtil.nlsCommentsForJavaQuoteStringWithNLRemoved(text.getText());
  }
  
}
