/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id$
 * /
 *******************************************************************************/
package org.eclipse.jet.internal.runtime;


import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jet.AbstractContextExtender;
import org.eclipse.jet.ContextLogEntry;
import org.eclipse.jet.JET2Context;
import org.eclipse.jet.taglib.TagInfo;
import org.eclipse.jet.transform.TransformContextExtender;


/**
 * A JET2Context extender that may be installed for logging runtime tag activity.
 */
public class RuntimeLoggerContextExtender extends AbstractContextExtender
{

  public static final int ERROR_LEVEL = 5;
  public static final int WARNING_LEVEL = 4;
  public static final int INFO_LEVEL = 3;
  public static final int TRACE_LEVEL = 2;
  public static final int DEBUG_LEVEL = 1;
  

  private static final class RuntimeLogger implements JET2Context.LogListener
  {
    private final List listeners = new ArrayList();

    public void log(String message, TagInfo td, String templatePath, int level)
    {
      for (Iterator i = listeners.iterator(); i.hasNext();)
      {
        RuntimeTagLogger logger = (RuntimeTagLogger)i.next();
        logger.log(message, td, templatePath, level);
      }
    }
    public void log(ContextLogEntry entry)
    {
      log(entry.getMessage(), entry.getTagInfo(), entry.getTemplatePath(), getLevel(entry.getSeverity()));
    }
    private int getLevel(int severity)
    {
      switch(severity)
      {
        case ContextLogEntry.ERROR:
        case ContextLogEntry.CANCEL:
          return RuntimeLoggerContextExtender.ERROR_LEVEL;
        case ContextLogEntry.WARNING:
          return RuntimeLoggerContextExtender.WARNING_LEVEL;
        case ContextLogEntry.INFO:
        case ContextLogEntry.OK:
          return RuntimeLoggerContextExtender.INFO_LEVEL;
        default:
          return RuntimeLoggerContextExtender.ERROR_LEVEL;
            
      }
    }
  }

  /**
   * Get the runtime logger instance. This method
   * will not install this context extender. Use {@link #install(JET2Context)}
   * to install the context extender
   * @param context the context
   * @return the installed context extender or <code>null</code>
   */
  public static RuntimeLoggerContextExtender getInstance(JET2Context context) {
    if(context.hasContextExtender(RuntimeLoggerContextExtender.class)) {
      return new RuntimeLoggerContextExtender(context);
    } else {
      return null;
    }
  }
  
  /**
   * 
   * @param context
   */
  public static void install(JET2Context context) {
    new RuntimeLoggerContextExtender(context);
  }
  
  private RuntimeLoggerContextExtender(JET2Context context)
  {
    super(context);
  }

  /* (non-Javadoc)
   * @see org.eclipse.jet.AbstractContextExtender#createExtendedData(org.eclipse.jet.JET2Context)
   */
  protected Object createExtendedData(JET2Context context)
  {
    final RuntimeLogger runtimeLogger = new RuntimeLogger();
    context.addLogListener(runtimeLogger);
    return runtimeLogger;
  }

  private RuntimeLogger getRuntimeLogger()
  {
    return (RuntimeLogger)getExtendedData();
  }

  public void addListener(RuntimeTagLogger logger)
  {
    getRuntimeLogger().listeners.add(logger);
  }

  public void removeListener(RuntimeTagLogger logger)
  {
    getRuntimeLogger().listeners.remove(logger);
  }

  public static void log(JET2Context context, String message, TagInfo td, int level)
  {
    if (context.hasContextExtender(RuntimeLoggerContextExtender.class))
    {
      RuntimeLoggerContextExtender rl = getInstance(context);
      String templatePath = null;
      if(context.hasContextExtender(TransformContextExtender.class)) {
        TransformContextExtender tce = new TransformContextExtender(context);
        templatePath = tce.getTemplatePath();
      }
      rl.log(message, td, templatePath, level);
    }
  }

  public void log(String message, TagInfo td, String templatePath, int level)
  {
    for (Iterator i = getRuntimeLogger().listeners.iterator(); i.hasNext();)
    {
      RuntimeTagLogger logger = (RuntimeTagLogger)i.next();
      logger.log(message, td, templatePath, level);
    }
  }

  public static void log(JET2Context context, String message, TagInfo tagInfo, String templatePath, int level)
  {
    if (context.hasContextExtender(RuntimeLoggerContextExtender.class))
    {
      getInstance(context).getRuntimeLogger().log(message, tagInfo, templatePath, level);
    }
  }
}
