/**
 * <copyright>
 *
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id$
 */
package org.eclipse.jet.internal.xpath.ast;


import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.jet.internal.xpath.NodeSetImpl;
import org.eclipse.jet.xpath.Context;
import org.eclipse.jet.xpath.IAnnotationManager;
import org.eclipse.jet.xpath.NodeSet;
import org.eclipse.jet.xpath.inspector.IElementInspector;
import org.eclipse.jet.xpath.inspector.INodeInspector;
import org.eclipse.jet.xpath.inspector.INodeInspectorExtension1;
import org.eclipse.jet.xpath.inspector.InspectorManager;
import org.eclipse.jet.xpath.inspector.INodeInspector.NodeKind;


/**
 * Class implementing XPath Axes.
 *
 */
public abstract class Axis
{

  /**
   * Interface defining a function that is invoked if a particular object
   * has an annotation.
   * @see Axis#hasAnnotations(Context, Object, AnnotationAction)
   */
  private interface AnnotationAction
  {

    public Object doAction(Object annotation, IElementInspector inspector);
  }

  /**
   * Test whether the given context object has annotations, and if so
   * invoke {@link AnnotationAction#doAction(Object, IElementInspector)}.
   * @param context the XPath Context
   * @param contextNode the context node
   * @param action the action to perform
   * @return the return value of {@link AnnotationAction#doAction(Object, IElementInspector) doAction}
   * or <code>null</code> if the action is not invoked.
   */
  protected final Object hasAnnotations(Context context, Object contextNode, AnnotationAction action)
  {
    if (context.hasAnnotationManager())
    {
      IAnnotationManager annotationManager = context.getAnnotationManager();
      if (annotationManager.hasAnnotations(contextNode))
      {
        final Object annotation = annotationManager.getAnnotationObject(contextNode);
        IElementInspector inspector = (IElementInspector)InspectorManager.getInstance().getInspector(annotation);
        return action.doAction(annotation, inspector);
      }
    }
    return null;
  }

  private static final class ParentAxis extends Axis
  {

    public NodeSet evaluate(NodeTest nodeTest, Context context)
    {
      Object obj = context.getContextNode();

      INodeInspector inspector = context.getContextNodeInspector();
      Object parent = inspector.getParent(obj);

      NodeSet result = NodeSetImpl.EMPTY_SET;
      if (parent != null)
      {
        result = new NodeSetImpl(1);
        result.add(parent);
      }

      nodeTest.filter(result);

      return result;
    }

    public NodeKind principalNodeKind()
    {
      return NodeKind.ELEMENT;
    }

    public String getAxisName()
    {
      return "parent"; //$NON-NLS-1$
    }
  }

  private static final class AttributeAxis extends Axis
  {

    public NodeSet evaluate(final NodeTest nodeTest, Context context)
    {
      NodeSet result = NodeSetImpl.EMPTY_SET;
      final Object contextNode = context.getContextNode();
      INodeInspector inspector = context.getContextNodeInspector();
      if (inspector instanceof IElementInspector && inspector.getNodeKind(contextNode) == INodeInspector.NodeKind.ELEMENT)
      {
        // getAttributes only applies to ELEMENTs
        IElementInspector elementInspector = (IElementInspector)inspector;
        if (nodeTest.isSimpleNameTest())
        {
          Object namedAttribute = elementInspector.getNamedAttribute(contextNode, nodeTest.getNameTestExpandedName());
          if (namedAttribute == null)
          {
            // check for annotations...
            namedAttribute = hasAnnotations(context, contextNode, new AnnotationAction()
              {

                public Object doAction(Object annotation, IElementInspector inspector)
                {
                  return inspector.getNamedAttribute(annotation, nodeTest.getNameTestExpandedName());
                }

              });
          }
          if (namedAttribute != null)
          {
            result = new NodeSetImpl();
            result.add(namedAttribute);
          }
        }
        else
        {
          Object[] attributes = elementInspector.getAttributes(contextNode);
          result = arrayToNodeSet(attributes);
          // add in annotations...
          Object[] annAttributes = (Object[])hasAnnotations(context, contextNode, new AnnotationAction()
            {

              public Object doAction(Object annotation, IElementInspector inspector)
              {

                return inspector.getAttributes(annotation);
              }
            });
          if (annAttributes != null)
          {
            result.addAll(Arrays.asList(annAttributes));
          }
          nodeTest.filter(result);
        }
      }
      return result;
    }

    public NodeKind principalNodeKind()
    {
      return NodeKind.ATTRIBUTE;
    }

    public String getAxisName()
    {
      return "attribute"; //$NON-NLS-1$
    }
  }

  private static final class ChildAxis extends Axis
  {

    public NodeSet evaluate(NodeTest nodeTest, Context context)
    {
      NodeSet result = NodeSetImpl.EMPTY_SET;
      Object contextNode = context.getContextNode();
      INodeInspector inspector = context.getContextNodeInspector();
      NodeKind nodeKind = inspector.getNodeKind(contextNode);
      if (nodeKind == INodeInspector.NodeKind.ELEMENT || nodeKind == INodeInspector.NodeKind.ROOT)
      {
        if (nodeTest.isSimpleNameTest() && inspector instanceof INodeInspectorExtension1)
        {
          Object[] children = ((INodeInspectorExtension1)inspector).getNamedChildren(contextNode, nodeTest.getNameTestExpandedName());
          if (children != null)
          {
            result = arrayToNodeSet(children);
          }
        }
        else
        {
          Object[] children = inspector.getChildren(contextNode);
          result = arrayToNodeSet(children);
          nodeTest.filter(result);

        }
      }
      return result;
    }

    public NodeKind principalNodeKind()
    {
      return NodeKind.ELEMENT;
    }

    public String getAxisName()
    {
      return "child"; //$NON-NLS-1$
    }

  }

  private static final class SelfAxis extends Axis
  {

    public NodeSet evaluate(NodeTest nodeTest, Context context)
    {
      NodeSet nodeSet = new NodeSetImpl(1);
      nodeSet.add(context.getContextNode());
      nodeTest.filter(nodeSet);
      return nodeSet;
    }

    public NodeKind principalNodeKind()
    {
      return NodeKind.ELEMENT;
    }

    public String getAxisName()
    {
      return "self"; //$NON-NLS-1$
    }

  }

  private static final class DescendantOrSelfAxis extends Axis
  {

    public NodeSet evaluate(NodeTest nodeTest, Context context)
    {
      NodeSet result = new NodeSetImpl();
      result.add(context.getContextNode());
      nodeTest.filter(result);
      
      
      NodeSet children = childAxis().evaluate(NodeTest.allNodes(), context);

      int contextPos = 1;
      for (Iterator i = children.iterator(); i.hasNext(); contextPos += 1)
      {
        Object object = (Object)i.next();
        Context subcontext = context.newSubContext(object, contextPos, children.size());
        
        result.addAll(evaluate(nodeTest, subcontext));
      }
      
      return result;
    }

    public NodeKind principalNodeKind()
    {
      return NodeKind.ELEMENT;
    }

    public String getAxisName()
    {
      return "descendant-or-self"; //$NON-NLS-1$
    }

  }

  private static final Axis childAxis = new ChildAxis();

  private static final Axis attributeAxis = new AttributeAxis();

  private static final Axis parentAxis = new ParentAxis();

  private static final Axis selfAxis = new SelfAxis();

  private static final Axis descendantOrSelfAxis = new DescendantOrSelfAxis();

  private static final Map axisByNameMap = new HashMap(15);
  static
  {
    axisByNameMap.put(childAxis.getAxisName(), childAxis);
    axisByNameMap.put(attributeAxis.getAxisName(), attributeAxis);
    axisByNameMap.put(parentAxis.getAxisName(), parentAxis);
    axisByNameMap.put(selfAxis.getAxisName(), selfAxis);
    axisByNameMap.put(descendantOrSelfAxis.getAxisName(), descendantOrSelfAxis);
  }

  /**
   * 
   */
  public Axis()
  {
    super();
  }

  public abstract NodeSet evaluate(NodeTest nodeTest, Context context);

  public abstract NodeKind principalNodeKind();

  public abstract String getAxisName();

  public static Axis childAxis()
  {
    return childAxis;
  }

  public static Axis attributeAxis()
  {
    return attributeAxis;
  }

  public static Axis parentAxis()
  {
    return parentAxis;
  }

  public static Axis descendantOrSelf()
  {
    return descendantOrSelfAxis;
  }

  public static Axis axisByName(String axisName)
  {
    return (Axis)axisByNameMap.get(axisName);
  }

  public static Axis selfAxis()
  {
    return selfAxis;
  }

  public String toString()
  {
    return getAxisName() + "::"; //$NON-NLS-1$
  }

  /**
   * @param attributes
   * @return
   */
  protected NodeSet arrayToNodeSet(Object[] attributes)
  {
    NodeSet result;
    result = new NodeSetImpl(attributes.length);
    for (int i = 0; i < attributes.length; i++)
    {
      result.add(attributes[i]);
    }
    return result;
  }

}
