/**
 * <copyright>
 *
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id$
 */
package org.eclipse.jet.internal;


import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.EMFPlugin;
import org.eclipse.jet.JET2Platform;
import org.eclipse.jet.internal.core.ContextLogEntryFactoryManager;
import org.eclipse.jet.internal.core.equinox.EquinoxContextLogEntryFactory;
import org.eclipse.jet.internal.extensionpoints.ModelInspectorsManager;
import org.eclipse.jet.internal.extensionpoints.ModelLoaderExtManager;
import org.eclipse.jet.internal.extensionpoints.PluginProjectMonitor;
import org.eclipse.jet.internal.extensionpoints.WorkspaceTagLibraryManager;
import org.eclipse.jet.internal.extensionpoints.XPathFunctionsManager;
import org.eclipse.jet.internal.runtime.JETBundleInstaller;
import org.eclipse.jet.internal.runtime.JETBundleManager;
import org.eclipse.jet.internal.taglib.InternalTagLibManager;
import org.eclipse.jet.taglib.TagLibrary;
import org.eclipse.jet.transform.IJETBundleManager;
import org.osgi.framework.BundleContext;


/**
 * The main plugin class to be used in the desktop.
 */
public class InternalJET2Platform extends EMFPlugin.EclipsePlugin
{

  //The shared instance.
  private static InternalJET2Platform plugin;

  private InternalTagLibManager tagLibManager = null;

  private JETBundleInstaller jetBundleInstaller = null;

  private XPathFunctionsManager xpathFunctionsManager = null;
  
  private ModelLoaderExtManager modelLoaderExtManager = null;
  
  private ModelInspectorsManager modelInspectorExtManager = null;
  
  private JETBundleManager bundleManager = null;
 
  private PluginProjectMonitor pluginProjectMonitor = null;
  
  private WorkspaceTagLibraryManager workspaceTagLibraryManager = null;
  
  private SavedStateManager savedStateManager = null;
  
  /**
   * Returns the shared instance.
   * @return the plugin instance
   */
  public static InternalJET2Platform getDefault()
  {
    return plugin;
  }

  /**
   * Write a message of the supplied severity to the Eclipse log
   * @param severity
   * @param msg
   * @param e
   */
  private static void log(int severity, String msg, Exception e)
  {
    getDefault().getLog().log(newStatus(severity, msg, e));
  }

  /**
   * Write an status object to the Eclipse error log.
   * @param result
   */
  public static void log(IStatus result)
  {
    getDefault().getLog().log(result);

  }

  /**
   * Write an error message to the Eclipse log
   * @param msg
   */
  public static void logError(String msg)
  {
    logError(msg, null);
  }

  /**
   * Write an error message to the Eclipse log
   * @param msg
   * @param e
   */
  public static void logError(String msg, Exception e)
  {
    log(IStatus.ERROR, msg, e);
  }

  /**
   * Write an info message to the Eclipse log.
   * @param msg
   */
  public static void logInfo(String msg)
  {
    logInfo(msg, null);
  }

  /**
   * Write an info message to the Eclipse log
   * @param msg
   * @param e
   */
  public static void logInfo(String msg, Exception e)
  {
    log(IStatus.INFO, msg, e);
  }

  /**
   * Write an error message to the Eclipse log
   * @param msg
   */
  public static void logWarning(String msg)
  {
    logWarning(msg, null);
  }

  /**
   * Log a warning message to the Eclipse log
   * @param msg a message, may be <code>null</code>.
   * @param e an exeception, may be <code>null</code>.
   */
  public static void logWarning(String msg, Exception e)
  {
    log(IStatus.WARNING, msg, e);
  }

  /**
   * Create a new IStatus wrapping the passed exception & message. The status refers to the JET2 core
   * plugin.
   * @param severity one of the {@link IStatus} severity codes.
   * @param msg the error message.
   * @param e the exception, may be <code>null</code>.
   * @return the new Status
   */
  public static IStatus newStatus(int severity, String msg, Exception e)
  {
    return new Status(severity, JET2Platform.PLUGIN_ID, IStatus.OK, msg == null ? "" : msg, e); //$NON-NLS-1$
  }

  /**
   * The constructor.
   */
  public InternalJET2Platform()
  {
    plugin = this;
    
    JETActivatorWrapper.INSTANCE.setPlugin(this);
  }

  /**
   * Return the location where the compiled bundles corresponding to JET2 workspace projects are stored.
   * @return the location (absolute path) of the compiled bundles.
   */
  public IPath getCompiledBundleLocation()
  {
    IPath stateLocation = getStateLocation();
    IPath compiledBundleLocation = stateLocation.append("compiledBundles"); //$NON-NLS-1$
    compiledBundleLocation.toFile().mkdirs();
    return compiledBundleLocation;
  }

  /**
   * Return tag library meta information for the specified tag library id.
   * @param id a tag library id (plugin id + "." + tag library id)
   * @return the tab library data or <code>null</code> if the id was not found.
   */
  public TagLibrary getTagLibrary(String id)
  {
    return tagLibManager.getTagLibrary(id);
  }

  /**
   * Return tag library meta information for the specified tag library id. Only tag libraries
   * declared in open plug-in projects in the workspace are considered.
   * @param id a tag library id (plugin id + "." + tag library id)
   * @return the tab library data or <code>null</code> if the id was not found.
   */
  public TagLibrary getWorkspaceTagLibrary(String id)
  {
    return workspaceTagLibraryManager.getTagLibrary(id);
  }
  
  /**
   * Return the project defining the tag library.
   * @param id the tag library id
   * @return the project or <code>null</code> if the tag library is not defined by a workspace project.
   */
  public IProject getProjectDefiningTagLibrary(String id)
  {
    return workspaceTagLibraryManager.getProjectDefiningTagLibrary(id);
  }
  /**
   * Return the manager of transform bundles (compiled JET2 tranformations).
   * @return the transform bundle manager
   */
  public JETBundleInstaller getJETBundleInstaller()
  {
    return jetBundleInstaller;
  }

  /**
   * This method is called upon plug-in activation
   * @param context the OSGi bundle context
   * @throws Exception if the plugin cannot start up
   */
  public void start(BundleContext context) throws Exception
  {
    super.start(context);

    // start the model inspectors early, they are used by other managers...
    modelInspectorExtManager = new ModelInspectorsManager();
    modelInspectorExtManager.startup();
    
    // start the model loader early, it is used by other managers...
    modelLoaderExtManager = new ModelLoaderExtManager();
    modelLoaderExtManager.startup();

    // start the XPath function extensions early, they are used by other managers...
    xpathFunctionsManager = new XPathFunctionsManager();
    xpathFunctionsManager.startup();

    pluginProjectMonitor = new PluginProjectMonitor();
    workspaceTagLibraryManager = new WorkspaceTagLibraryManager();
    workspaceTagLibraryManager.startup(pluginProjectMonitor);
    pluginProjectMonitor.startup();

    tagLibManager = new InternalTagLibManager();
    tagLibManager.startup();

    jetBundleInstaller = new JETBundleInstaller();
    jetBundleInstaller.startup(context);

    bundleManager = new JETBundleManager();
    bundleManager.startup();
    
    savedStateManager = new SavedStateManager(this);
    savedStateManager.startup();
    
    ContextLogEntryFactoryManager.setFactory(new EquinoxContextLogEntryFactory(Platform.getAdapterManager()));
  }

  /**
   * This method is called when the plug-in is stopped
   * @param context the OSGi bundle context
   * @throws Exception if the plugin cannot shutdown
   */
  public void stop(BundleContext context) throws Exception
  {
    tagLibManager.shutdown();
    tagLibManager = null;

    jetBundleInstaller.shutdown();
    jetBundleInstaller = null;
    
    xpathFunctionsManager.shutdown();
    xpathFunctionsManager = null;
    
    modelLoaderExtManager.shutdown();
    modelLoaderExtManager = null;
    
    modelInspectorExtManager.shutdown();
    modelInspectorExtManager = null;
    
    bundleManager.shutdown();
    bundleManager = null;
    
    workspaceTagLibraryManager.shutdown();
    pluginProjectMonitor.shutdown();
    
    savedStateManager.shutdown();
    savedStateManager = null;

    // shut down the plug-in itself last
    super.stop(context);
    plugin = null;
  }

  /**
   * Utility function for writing Debug output
   * @param issuer the issuer of the message, usually <code>this</code>, but may
   * also be a Class object.
   * @param message the debug message.
   */
  public static void debugMessage(Object issuer, String message)
  {
    Class msgClass = null;
    if (issuer instanceof Class)
    {
      msgClass = (Class)issuer;
    }
    else
    {
      msgClass = issuer.getClass();
    }

    System.err.println(msgClass.getName() + ": " + message); //$NON-NLS-1$
  }

  public String[] getKnownTagLibraryIds()
  {
    return tagLibManager.getKnownTagLibraryIds();
  }

  /**
   * @return Returns the xpathFunctionsManager.
   */
  public final XPathFunctionsManager getXPathFunctionsManager()
  {
    return xpathFunctionsManager;
  }

  /**
   * @return Returns the tagLibManager.
   */
  public final InternalTagLibManager getTagLibManager()
  {
    return tagLibManager;
  }
  
  /**
   * @return Returns the bundleManager.
   */
  public final IJETBundleManager getBundleManager()
  {
    return bundleManager;
  }

  public final SavedStateManager getSavedStateManager()
  {
    return savedStateManager;
  }
}
