/**
 * <copyright>
 *
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: IndentTag.java,v 1.2 2008/05/22 15:07:11 pelder Exp $
 */
package org.eclipse.jet.internal.taglib.format;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jet.JET2Context;
import org.eclipse.jet.internal.l10n.JET2Messages;
import org.eclipse.jet.taglib.AbstractFunctionTag;
import org.eclipse.jet.taglib.JET2TagException;
import org.eclipse.jet.taglib.TagInfo;
import org.eclipse.osgi.util.NLS;

/**
 * Implement &lt;f:indent text="..." [depth="n"]&gt;content&lt;/f:indent&gt;
 */
public class IndentTag extends AbstractFunctionTag
{

  private static final String TEXT_ATTR = "text"; //$NON-NLS-1$
  private static final String DEPTH_ATTR = "depth"; //$NON-NLS-1$
  private static final String NL_REGEX = "([\\n][\\r]?|[\\r][\\n]?)"; //$NON-NLS-1$
  private static final Pattern NL_PATTERN = Pattern.compile(NL_REGEX, Pattern.MULTILINE);
  private static final String NL = System.getProperty("line.separator"); //$NON-NLS-1$

  /* (non-Javadoc)
   * @see org.eclipse.jet.taglib.FunctionTag#doFunction(org.eclipse.jet.taglib.TagInfo, org.eclipse.jet.JET2Context, java.lang.String)
   */
  public String doFunction(TagInfo td, JET2Context context, String bodyContent) throws JET2TagException
  {
    final String indentText = getAttribute(TEXT_ATTR);
    final String depthValue = getAttribute(DEPTH_ATTR);
    try
    {
      final int depth = depthValue == null ? 1 : Integer.decode(depthValue).intValue();
      return indent(bodyContent, indentText, depth);
    }
    catch (NumberFormatException e)
    {
      throw new JET2TagException(NLS.bind(JET2Messages.IndentTag_AttributeValueMustBeInteger, DEPTH_ATTR, depthValue));
    }
  }

  public static String indent(String content, String indent, int depth) {
    final String totalIndent = computeTotalIndent(indent, depth);
    if(totalIndent.length() == 0) {
      return content;
    }
    boolean indentIsWhitespace = totalIndent.matches("\\s*"); //$NON-NLS-1$
    
    final Matcher matcher = NL_PATTERN.matcher(content);
    final StringBuffer result = new StringBuffer(content.length() * 2); // a guess
    int lineStart;
    for(lineStart = 0; matcher.find(lineStart); lineStart = matcher.end()) {
      String line = content.substring(lineStart, matcher.start());
      if(line.length() > 0 || !indentIsWhitespace) {
        result.append(totalIndent);
      }
      result.append(line);
      result.append(NL);
      lineStart = matcher.end();
    }
    
    // look for final line without a NL character...
    if(lineStart < content.length()) {
      String line = content.substring(lineStart);
      result.append(totalIndent).append(line).append(NL);
    }
    
    return result.toString();
  }

  /**
   * @param indent
   * @param depth
   * @return
   */
  private static String computeTotalIndent(String indent, int depth)
  {
    final StringBuffer buffer = new StringBuffer(
                                        depth * indent.length() 
                                        );
    if(indent != null) {
      for(int i = 0; i < depth; i++) {
        buffer.append(indent);
      }
    }
    return buffer.toString();
  }
}
