/*******************************************************************************
 * Copyright (c) 2012 BREDEX GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BREDEX GmbH - initial API and implementation 
 *******************************************************************************/
package org.eclipse.jubula.rc.swing.swing.caps;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Container;
import java.awt.event.ContainerEvent;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.StringTokenizer;

import javax.swing.AbstractButton;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.JToggleButton;
import javax.swing.text.JTextComponent;

import org.eclipse.jubula.rc.common.AUTServer;
import org.eclipse.jubula.rc.common.adaptable.AdapterFactoryRegistry;
import org.eclipse.jubula.rc.common.adaptable.ITextRendererAdapter;
import org.eclipse.jubula.rc.common.driver.IEventThreadQueuer;
import org.eclipse.jubula.rc.common.driver.IRobot;
import org.eclipse.jubula.rc.common.driver.IRobotFactory;
import org.eclipse.jubula.rc.common.driver.IRunnable;
import org.eclipse.jubula.rc.common.exception.StepExecutionException;
import org.eclipse.jubula.rc.common.util.KeyStrokeUtil;
import org.eclipse.jubula.rc.swing.driver.RobotFactoryConfig;
import org.eclipse.jubula.rc.swing.swing.driver.KeyCodeConverter;
import org.eclipse.jubula.rc.swing.swing.implclasses.EventListener;
import org.eclipse.jubula.tools.objects.event.EventFactory;
import org.eclipse.jubula.tools.objects.event.TestErrorEvent;

/**
 * Util class for Swing specific aspects.
 * 
 * @author BREDEX GmbH
 */
public class CapUtil {

    /**
     * <code>RENDERER_FALLBACK_TEXT_GETTER_METHOD_1</code>
     */
    public static final String RENDERER_FALLBACK_TEXT_GETTER_METHOD_1 = "getTestableText"; //$NON-NLS-1$

    /**
     * <code>RENDERER_FALLBACK_TEXT_GETTER_METHOD_2</code>
     */
    public static final String RENDERER_FALLBACK_TEXT_GETTER_METHOD_2 = "getText"; //$NON-NLS-1$
      
   
    /**
     * Is true, if a popup menu is shown
     */
    public static class PopupShownCondition implements
        EventListener.Condition {

    /**
     * the popup menu
     */
        private JPopupMenu m_popup = null;
        
    /**
     *
     * @return the popup menu
     */
        public JPopupMenu getPopup() {
            return m_popup;
        }

    /**
     * {@inheritDoc}
     * @param event event
     * @return result of the condition
     */
        public boolean isTrue(AWTEvent event) {
            if (event.getID() != ContainerEvent.COMPONENT_ADDED) {
                return false;
            }
            ContainerEvent ce = (ContainerEvent)event;
            if (ce.getChild() instanceof JPopupMenu) {
                m_popup = (JPopupMenu)ce.getChild();
                return true;
            } else if (ce.getChild() instanceof Container) {
                Container popupContainer = (Container)ce.getChild();
                final int length = popupContainer.getComponents().length;
                for (int i = 0; i < length; i++) {
                    if (popupContainer.getComponents()[i]
                                                       instanceof JPopupMenu) {
        
                        m_popup = (JPopupMenu)popupContainer.getComponents()[i];
                        return true;
                    }
                }
            }
            return false;
        }
    }
    
    /**
     * The robot factory.
     */
    private static IRobotFactory robotFactory;

    /**
     * 
     */
    private CapUtil() { }
    
    /**
     * 
     * @return the Robot
     */
    public static IRobot getRobot() {
        return AUTServer.getInstance().getRobot();
    }

    /**
     * Gets the Robot factory. The factory is created once per instance.
     *
     * @return The Robot factory.
     */
    protected static IRobotFactory getRobotFactory() {
        if (robotFactory == null) {
            robotFactory = new RobotFactoryConfig().getRobotFactory();
        }
        return robotFactory;
    }
    /**
     * @return The event thread queuer.
     */
    protected static IEventThreadQueuer getEventThreadQueuer() {
        return getRobotFactory().getEventThreadQueuer();
    }
    
    /**
     * Presses or releases the given modifier.
     * @param modifier the modifier.
     * @param press if true, the modifier will be pressed.
     * if false, the modifier will be released.
     */
    public static void pressOrReleaseModifiers(String modifier, boolean press) {
        final IRobot robot = getRobot();
        final StringTokenizer modTok = new StringTokenizer(
                KeyStrokeUtil.getModifierString(modifier), " "); //$NON-NLS-1$
        while (modTok.hasMoreTokens()) {
            final String mod = modTok.nextToken();
            final int keyCode = KeyCodeConverter.getKeyCode(mod);
            if (press) {
                robot.keyPress(null, keyCode);
            } else {
                robot.keyRelease(null, keyCode);
            }
        }
    }
     
    /**
     * Casts the passed renderer component to a known type and extracts the
     * rendered text.
     *
     * @param renderer
     *            The renderer.
     * @param queueInEventThread
     *            If <code>true</code>, the text extraction is executed in
     *            the event queue thread.
     * @return The rendered text.
     * @throws StepExecutionException
     *             If the passed renderer is not supported. Supported types are
     *             <code>JLabel</code>, <code>JToggleButton</code>,
     *             <code>AbstractButton</code> and <code>JTextComponent</code>
     *
     */
    public static String getRenderedText(final Component renderer,
        boolean queueInEventThread) throws StepExecutionException {

        if (queueInEventThread) {
            return (String)getEventThreadQueuer().invokeAndWait(
                "getRenderedText", new IRunnable() { //$NON-NLS-1$
                    public Object run() {
                        return getRenderedText(renderer);
                    }
                });
        }

        return getRenderedText(renderer);
    }
    
    /**
     * @param renderer
     *            The component which is used as the renderer
     * @return The string that the renderer displays.
     * @throws StepExecutionException
     *             If the renderer component is not of type <code>JLabel</code>,
     *             <code>JToggleButton</code>, <code>AbstractButton</code>,
     *             <code>JTextComponent</code> or supports one of the fallback 
     *             methods
     */
    public static String getRenderedText(Component renderer)
        throws StepExecutionException {
        String renderedText = resolveRenderedText(renderer);
        if (renderedText != null) {
            return renderedText;
        }
        throw new StepExecutionException(
            "Renderer not supported: " + renderer.getClass(), //$NON-NLS-1$
            EventFactory.createActionError(
                    TestErrorEvent.RENDERER_NOT_SUPPORTED));
    }
    
    /**
     * @param renderer
     *            The component which is used as the renderer
     * @return The string that the renderer displays or <code>null</code> if it
     *         could not be resolved.
     */
    private static String resolveRenderedText(Component renderer) {
        if (renderer instanceof JLabel) {
            return ((JLabel)renderer).getText();
        } else if (renderer instanceof JToggleButton) {
            return ((JToggleButton)renderer).isSelected() ? Boolean.TRUE
                .toString() : Boolean.FALSE.toString();
        } else if (renderer instanceof AbstractButton) {
            return ((AbstractButton)renderer).getText();
        } else if (renderer instanceof JTextComponent) {
            return ((JTextComponent)renderer).getText();
        } 
        // Check if an adapter exists
        ITextRendererAdapter textRendererAdapter = 
            ((ITextRendererAdapter) AdapterFactoryRegistry
                .getInstance().getAdapter(
                        ITextRendererAdapter.class, renderer));
        if (textRendererAdapter != null) {
            return textRendererAdapter.getText();
        } else if (renderer != null) {
            String[] methodNames = new String[] {
                RENDERER_FALLBACK_TEXT_GETTER_METHOD_1,
                RENDERER_FALLBACK_TEXT_GETTER_METHOD_2 };
            for (int i = 0; i < methodNames.length; i++) {
                String text = getTextFromComponent(renderer, methodNames[i]);
                if (text != null) {
                    return text;
                }
            }
        }
        return null;
    }
    
    /**
     * @param obj
     *            the object to invoke the method for
     * @param getterName
     *            the name of the getter Method for string retrival
     * @return the return value of the given method name or <code>null</code> if
     *         something went wrong during method invocation
     */
    private static String getTextFromComponent(Object obj, String getterName) {
        String text = null;
        try {
            Method getter = null;
            Class objClass = obj.getClass();
            try {
                getter = objClass.getDeclaredMethod(getterName, null);
            } catch (NoSuchMethodException e) {
                // ignore
            } catch (SecurityException e) {
                // ignore
            }
            if (getter == null) {
                try {
                    getter = objClass.getMethod(getterName, null);
                } catch (NoSuchMethodException e) {
                    return text;
                } catch (SecurityException e) {
                    return text;
                }
            }
            getter.setAccessible(true);
            Object returnValue = getter.invoke(obj, null);
            if (returnValue instanceof String) {
                text = (String) returnValue;
            }
            return text;
        } catch (SecurityException e) {
            return text;
        } catch (IllegalArgumentException e) {
            return text;
        } catch (IllegalAccessException e) {
            return text;
        } catch (InvocationTargetException e) {
            return text;
        }
    }
}
