/*******************************************************************************
 * Copyright (c) 2004, 2010 BREDEX GmbH.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BREDEX GmbH - initial API and implementation and/or initial documentation
 *******************************************************************************/
package org.eclipse.jubula.client.ui.handlers.open;

import java.util.Iterator;
import java.util.concurrent.atomic.AtomicReference;

import javax.persistence.EntityManager;

import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jubula.client.core.model.ITestResultSummaryPO;
import org.eclipse.jubula.client.core.model.TestResultNode;
import org.eclipse.jubula.client.core.persistence.GeneralStorage;
import org.eclipse.jubula.client.core.persistence.TestResultPM;
import org.eclipse.jubula.client.core.utils.ITreeNodeOperation;
import org.eclipse.jubula.client.core.utils.ITreeTraverserContext;
import org.eclipse.jubula.client.core.utils.TestResultNodeTraverser;
import org.eclipse.jubula.client.ui.constants.CommandIDs;
import org.eclipse.jubula.client.ui.editors.TestResultEditorInput;
import org.eclipse.jubula.client.ui.editors.TestResultViewer;
import org.eclipse.jubula.client.ui.handlers.AbstractSelectionBasedHandler;
import org.eclipse.jubula.client.ui.i18n.Messages;
import org.eclipse.jubula.client.ui.utils.ErrorHandlingUtil;
import org.eclipse.jubula.tools.constants.StringConstants;
import org.eclipse.jubula.tools.messagehandling.MessageIDs;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.handlers.HandlerUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Handler to open the Test Result Viewer.
 *
 * @author BREDEX GmbH
 * @created Jun 14, 2010
 */
public class OpenTestResultViewerHandler 
    extends AbstractSelectionBasedHandler {
    /** the logger */
    private static final Logger LOG = 
        LoggerFactory.getLogger(OpenTestResultViewerHandler.class);
    
    /**
     * {@inheritDoc}
     */
    public Object executeImpl(ExecutionEvent event) throws ExecutionException {
        String summaryIdString = event.getParameter(CommandIDs.
            OPEN_TEST_RESULT_DETAIL_COMMAND_PARAMETER_SUMMARY_ID);
        IEditorInput editorInput = null;
        EntityManager mSession = GeneralStorage.getInstance()
            .getMasterSession();
        if (summaryIdString != null) {
            final Long summaryId = Long.valueOf(summaryIdString);
            if (TestResultPM.hasTestResultDetails(mSession, summaryId)) {
                ITestResultSummaryPO testResultSummary = TestResultPM
                    .getTestResultSummary(mSession, summaryId);
                editorInput = new TestResultEditorInput(testResultSummary);
            } else {
                Display.getDefault().asyncExec(new Runnable() {
                    public void run() {
                        MessageBox mb = new MessageBox(getActiveShell());
                        mb.setMessage(NLS.bind(
                            Messages.TestresultDetailsForSummaryNotFound,
                            summaryId));
                        mb.open();
                    }
                });
            }
        } else {
            editorInput = getEditorInputFromTestResultSummaryView(event,
                mSession);
        }
        openTestResultViewerEditor(event, editorInput);
        return null;
    }

    /**
     * @param event
     *            the execution event
     * @param session
     *            the session to use
     * @throws ExecutionException
     * @return the input if found - otherwise null
     */
    private IEditorInput getEditorInputFromTestResultSummaryView(
        ExecutionEvent event, EntityManager session) throws ExecutionException {
        IStructuredSelection structuredSel = getSelection();
        Iterator selectedTestResultSummaries = structuredSel.iterator();
        IEditorInput editorInput = null;
        while (selectedTestResultSummaries.hasNext()) {
            Object firstElement = selectedTestResultSummaries.next();
            if (firstElement instanceof ITestResultSummaryPO) {
                ITestResultSummaryPO result = 
                    (ITestResultSummaryPO) firstElement;
                if (TestResultPM.hasTestResultDetails(
                    session, result.getId())) {
                    editorInput = new TestResultEditorInput(result);
                }
            } else {
                LOG.info(Messages.SelectedElementIsNotTestResultSummary
                    + StringConstants.DOT);
            }
        }
        return editorInput;
    }

    /**
     * @author BREDEX GmbH
     */
    private static class SearchNodeByCountOperation implements
            ITreeNodeOperation<TestResultNode> {
        /** the node to select*/
        private Long m_initialNodeToSelect;
        
        /** the counter used to count nodes*/
        private Long m_currentNodeCount = new Long(0);

        /** the node */
        private Object m_node = null;
        
        /** flag indicating whether result has been found */ 
        private boolean m_found = false;
        
        /**
         * Constructor
         * 
         * @param nodeToSearchByCount
         *            the node to search by index
         */
        public SearchNodeByCountOperation(Long nodeToSearchByCount) {
            m_initialNodeToSelect = nodeToSearchByCount;
        }
        
        /** {@inheritDoc} */
        public boolean operate(ITreeTraverserContext<TestResultNode> ctx,
            TestResultNode parent, TestResultNode node, 
            boolean alreadyVisited) {
            
            m_currentNodeCount++;
            
            if (m_currentNodeCount.equals(m_initialNodeToSelect)) {
                setNode(node);
                m_found = true;
            }
            
            return !m_found;
        }
        
        /** {@inheritDoc} */
        public void postOperate(ITreeTraverserContext<TestResultNode> ctx,
            TestResultNode parent, TestResultNode node, 
            boolean alreadyVisited) {
            // not necessary
        }

        /**
         * @return the node
         */
        public Object getNode() {
            return m_node;
        }

        /**
         * @param node the node to set
         */
        private void setNode(Object node) {
            m_node = node;
        }
    }
    
    /**
     * @param event
     *            event
     * @param editorInput
     *            editorInput
     * @throws ExecutionException
     */
    private void openTestResultViewerEditor(ExecutionEvent event,
        final IEditorInput editorInput) throws ExecutionException {
        if (editorInput != null) {
            final IWorkbenchPage currentPage = HandlerUtil
                .getActiveSiteChecked(event).getPage();
            final String nodeId = event.getParameter(CommandIDs.
                OPEN_TEST_RESULT_DETAIL_COMMAND_PARAMETER_NODE_ID);
            if (currentPage != null) {
                openEditor(editorInput, currentPage, nodeId);
            }
        } else {
            LOG.info(Messages.NoEditorInputCouldBeCreated);
        }
    }

    /**
     * @param editorInput
     *            editorInput
     * @param currentPage
     *            currentPage
     * @param nodeId
     *            nodeId
     */
    private void openEditor(final IEditorInput editorInput,
        final IWorkbenchPage currentPage, final String nodeId) {
        // necessary when being executed in RAP context !!
        Display currentDisplay = Display.getDefault();
        final AtomicReference<Object> editorRef = new AtomicReference<Object>();
        
        currentDisplay.syncExec(new Runnable() {
            public void run() {
                try {
                    editorRef.set(currentPage.openEditor(editorInput,
                        TestResultViewer.EDITOR_ID));
                } catch (PartInitException e) {
                    ErrorHandlingUtil
                        .createMessageDialog(MessageIDs.E_CANNOT_OPEN_EDITOR);
                }
            }
        });
        
        if (nodeId != null) {
            currentDisplay.syncExec(new Runnable() {
                public void run() {
                    TestResultViewer editor = (TestResultViewer) 
                        editorRef.get();
                    if (editor != null) {
                        SearchNodeByCountOperation operation = 
                            new SearchNodeByCountOperation(
                                Long.valueOf(nodeId));
                        TestResultNodeTraverser traverser = 
                            new TestResultNodeTraverser(
                                editor.getTestResultRootNode(), operation);
                        traverser.traverse();

                        Object nodeToSelect = operation.getNode();

                        if (nodeToSelect != null) {
                            StructuredSelection selection = 
                                new StructuredSelection(nodeToSelect);
                            editor.getTreeViewer().reveal(nodeToSelect);
                            editor.setSelection(selection);
                        }
                    }
                }
            });
        }
    }
}
