/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import com.google.common.base.MoreObjects;
import org.eclipse.lsp4j.debug.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * A GotoTarget describes a code location that can be used as a target in the 'goto' request.
 * <p>
 * The possible goto targets can be determined via the 'gotoTargets' request.
 */
@SuppressWarnings("all")
public class GotoTarget {
  /**
   * Unique identifier for a goto target. This is used in the goto request.
   */
  private int id;

  /**
   * The name of the goto target (shown in the UI).
   */
  @NonNull
  private String label;

  /**
   * The line of the goto target.
   */
  private int line;

  /**
   * The column of the goto target.
   * <p>
   * This is an optional property.
   */
  private Integer column;

  /**
   * The end line of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  private Integer endLine;

  /**
   * The end column of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  private Integer endColumn;

  /**
   * A memory reference for the instruction pointer value represented by this target.
   * <p>
   * This is an optional property.
   */
  private String instructionPointerReference;

  /**
   * Unique identifier for a goto target. This is used in the goto request.
   */
  public int getId() {
    return this.id;
  }

  /**
   * Unique identifier for a goto target. This is used in the goto request.
   */
  public void setId(final int id) {
    this.id = id;
  }

  /**
   * The name of the goto target (shown in the UI).
   */
  @NonNull
  public String getLabel() {
    return this.label;
  }

  /**
   * The name of the goto target (shown in the UI).
   */
  public void setLabel(@NonNull final String label) {
    this.label = Preconditions.checkNotNull(label, "label");
  }

  /**
   * The line of the goto target.
   */
  public int getLine() {
    return this.line;
  }

  /**
   * The line of the goto target.
   */
  public void setLine(final int line) {
    this.line = line;
  }

  /**
   * The column of the goto target.
   * <p>
   * This is an optional property.
   */
  public Integer getColumn() {
    return this.column;
  }

  /**
   * The column of the goto target.
   * <p>
   * This is an optional property.
   */
  public void setColumn(final Integer column) {
    this.column = column;
  }

  /**
   * The end line of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  public Integer getEndLine() {
    return this.endLine;
  }

  /**
   * The end line of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  public void setEndLine(final Integer endLine) {
    this.endLine = endLine;
  }

  /**
   * The end column of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  public Integer getEndColumn() {
    return this.endColumn;
  }

  /**
   * The end column of the range covered by the goto target.
   * <p>
   * This is an optional property.
   */
  public void setEndColumn(final Integer endColumn) {
    this.endColumn = endColumn;
  }

  /**
   * A memory reference for the instruction pointer value represented by this target.
   * <p>
   * This is an optional property.
   */
  public String getInstructionPointerReference() {
    return this.instructionPointerReference;
  }

  /**
   * A memory reference for the instruction pointer value represented by this target.
   * <p>
   * This is an optional property.
   */
  public void setInstructionPointerReference(final String instructionPointerReference) {
    this.instructionPointerReference = instructionPointerReference;
  }

  @Override
  public String toString() {
    MoreObjects.ToStringHelper b = MoreObjects.toStringHelper(this);
    b.add("id", this.id);
    b.add("label", this.label);
    b.add("line", this.line);
    b.add("column", this.column);
    b.add("endLine", this.endLine);
    b.add("endColumn", this.endColumn);
    b.add("instructionPointerReference", this.instructionPointerReference);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    GotoTarget other = (GotoTarget) obj;
    if (other.id != this.id)
      return false;
    if (this.label == null) {
      if (other.label != null)
        return false;
    } else if (!this.label.equals(other.label))
      return false;
    if (other.line != this.line)
      return false;
    if (this.column == null) {
      if (other.column != null)
        return false;
    } else if (!this.column.equals(other.column))
      return false;
    if (this.endLine == null) {
      if (other.endLine != null)
        return false;
    } else if (!this.endLine.equals(other.endLine))
      return false;
    if (this.endColumn == null) {
      if (other.endColumn != null)
        return false;
    } else if (!this.endColumn.equals(other.endColumn))
      return false;
    if (this.instructionPointerReference == null) {
      if (other.instructionPointerReference != null)
        return false;
    } else if (!this.instructionPointerReference.equals(other.instructionPointerReference))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + this.id;
    result = prime * result + ((this.label== null) ? 0 : this.label.hashCode());
    result = prime * result + this.line;
    result = prime * result + ((this.column== null) ? 0 : this.column.hashCode());
    result = prime * result + ((this.endLine== null) ? 0 : this.endLine.hashCode());
    result = prime * result + ((this.endColumn== null) ? 0 : this.endColumn.hashCode());
    return prime * result + ((this.instructionPointerReference== null) ? 0 : this.instructionPointerReference.hashCode());
  }
}
