/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import com.google.common.base.MoreObjects;
import org.eclipse.lsp4j.debug.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Body for 'memory' event.
 * <p>
 * Represents the {@code body} of {@code MemoryEvent} defined in spec.
 * <p>
 * Since 1.49
 */
@SuppressWarnings("all")
public class MemoryEventArguments {
  /**
   * Memory reference of a memory range that has been updated.
   */
  @NonNull
  private String memoryReference;

  /**
   * Starting offset in bytes where memory has been updated. Can be negative.
   */
  private int offset;

  /**
   * Number of bytes updated.
   */
  private int count;

  /**
   * Memory reference of a memory range that has been updated.
   */
  @NonNull
  public String getMemoryReference() {
    return this.memoryReference;
  }

  /**
   * Memory reference of a memory range that has been updated.
   */
  public void setMemoryReference(@NonNull final String memoryReference) {
    this.memoryReference = Preconditions.checkNotNull(memoryReference, "memoryReference");
  }

  /**
   * Starting offset in bytes where memory has been updated. Can be negative.
   */
  public int getOffset() {
    return this.offset;
  }

  /**
   * Starting offset in bytes where memory has been updated. Can be negative.
   */
  public void setOffset(final int offset) {
    this.offset = offset;
  }

  /**
   * Number of bytes updated.
   */
  public int getCount() {
    return this.count;
  }

  /**
   * Number of bytes updated.
   */
  public void setCount(final int count) {
    this.count = count;
  }

  @Override
  public String toString() {
    MoreObjects.ToStringHelper b = MoreObjects.toStringHelper(this);
    b.add("memoryReference", this.memoryReference);
    b.add("offset", this.offset);
    b.add("count", this.count);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    MemoryEventArguments other = (MemoryEventArguments) obj;
    if (this.memoryReference == null) {
      if (other.memoryReference != null)
        return false;
    } else if (!this.memoryReference.equals(other.memoryReference))
      return false;
    if (other.offset != this.offset)
      return false;
    if (other.count != this.count)
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.memoryReference== null) ? 0 : this.memoryReference.hashCode());
    result = prime * result + this.offset;
    return prime * result + this.count;
  }
}
