/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import com.google.common.base.MoreObjects;
import java.util.List;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;
import org.eclipse.lsp4j.util.Preconditions;

/**
 * Represents an outgoing call, e.g. calling a getter from a method or a method from a constructor etc.
 * <p>
 * Since 3.16.0
 */
@SuppressWarnings("all")
public class CallHierarchyOutgoingCall {
  /**
   * The item that is called.
   */
  @NonNull
  private CallHierarchyItem to;

  /**
   * The range at which this item is called. This is the range relative to the caller, i.e. the {@link CallHierarchyOutgoingCallsParams#item}.
   */
  @NonNull
  private List<Range> fromRanges;

  public CallHierarchyOutgoingCall() {
  }

  public CallHierarchyOutgoingCall(@NonNull final CallHierarchyItem to, @NonNull final List<Range> fromRanges) {
    this.to = Preconditions.<CallHierarchyItem>checkNotNull(to, "to");
    this.fromRanges = Preconditions.<List<Range>>checkNotNull(fromRanges, "fromRanges");
  }

  /**
   * The item that is called.
   */
  @NonNull
  public CallHierarchyItem getTo() {
    return this.to;
  }

  /**
   * The item that is called.
   */
  public void setTo(@NonNull final CallHierarchyItem to) {
    this.to = Preconditions.checkNotNull(to, "to");
  }

  /**
   * The range at which this item is called. This is the range relative to the caller, i.e. the {@link CallHierarchyOutgoingCallsParams#item}.
   */
  @NonNull
  public List<Range> getFromRanges() {
    return this.fromRanges;
  }

  /**
   * The range at which this item is called. This is the range relative to the caller, i.e. the {@link CallHierarchyOutgoingCallsParams#item}.
   */
  public void setFromRanges(@NonNull final List<Range> fromRanges) {
    this.fromRanges = Preconditions.checkNotNull(fromRanges, "fromRanges");
  }

  @Override
  public String toString() {
    MoreObjects.ToStringHelper b = MoreObjects.toStringHelper(this);
    b.add("to", this.to);
    b.add("fromRanges", this.fromRanges);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CallHierarchyOutgoingCall other = (CallHierarchyOutgoingCall) obj;
    if (this.to == null) {
      if (other.to != null)
        return false;
    } else if (!this.to.equals(other.to))
      return false;
    if (this.fromRanges == null) {
      if (other.fromRanges != null)
        return false;
    } else if (!this.fromRanges.equals(other.fromRanges))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.to== null) ? 0 : this.to.hashCode());
    return prime * result + ((this.fromRanges== null) ? 0 : this.fromRanges.hashCode());
  }
}
