/**
 * Copyright (c) 2017, 2020 Kichwa Coders Ltd. and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j.debug;

import org.eclipse.lsp4j.debug.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Arguments for 'writeMemory' request.
 * <p>
 * Since 1.48
 */
@SuppressWarnings("all")
public class WriteMemoryArguments {
  /**
   * Memory reference to the base location to which data should be written.
   */
  @NonNull
  private String memoryReference;

  /**
   * Offset (in bytes) to be applied to the reference location before
   * writing data. Can be negative.
   * <p>
   * This is an optional property.
   */
  private Integer offset;

  /**
   * Property to control partial writes. If true, the debug adapter
   * should attempt to write memory even if the entire memory region is not
   * writable. In such a case the debug adapter should stop after hitting the
   * first byte of memory that cannot be written and return the number of bytes
   * written in the response via the 'offset' and 'bytesWritten' properties.
   * If false or missing, a debug adapter should attempt to verify the region is
   * writable before writing, and fail the response if it is not.
   * <p>
   * This is an optional property.
   */
  private Boolean allowPartial;

  /**
   * Bytes to write, encoded using base64.
   */
  @NonNull
  private String data;

  /**
   * Memory reference to the base location to which data should be written.
   */
  @NonNull
  public String getMemoryReference() {
    return this.memoryReference;
  }

  /**
   * Memory reference to the base location to which data should be written.
   */
  public void setMemoryReference(@NonNull final String memoryReference) {
    this.memoryReference = Preconditions.checkNotNull(memoryReference, "memoryReference");
  }

  /**
   * Offset (in bytes) to be applied to the reference location before
   * writing data. Can be negative.
   * <p>
   * This is an optional property.
   */
  public Integer getOffset() {
    return this.offset;
  }

  /**
   * Offset (in bytes) to be applied to the reference location before
   * writing data. Can be negative.
   * <p>
   * This is an optional property.
   */
  public void setOffset(final Integer offset) {
    this.offset = offset;
  }

  /**
   * Property to control partial writes. If true, the debug adapter
   * should attempt to write memory even if the entire memory region is not
   * writable. In such a case the debug adapter should stop after hitting the
   * first byte of memory that cannot be written and return the number of bytes
   * written in the response via the 'offset' and 'bytesWritten' properties.
   * If false or missing, a debug adapter should attempt to verify the region is
   * writable before writing, and fail the response if it is not.
   * <p>
   * This is an optional property.
   */
  public Boolean getAllowPartial() {
    return this.allowPartial;
  }

  /**
   * Property to control partial writes. If true, the debug adapter
   * should attempt to write memory even if the entire memory region is not
   * writable. In such a case the debug adapter should stop after hitting the
   * first byte of memory that cannot be written and return the number of bytes
   * written in the response via the 'offset' and 'bytesWritten' properties.
   * If false or missing, a debug adapter should attempt to verify the region is
   * writable before writing, and fail the response if it is not.
   * <p>
   * This is an optional property.
   */
  public void setAllowPartial(final Boolean allowPartial) {
    this.allowPartial = allowPartial;
  }

  /**
   * Bytes to write, encoded using base64.
   */
  @NonNull
  public String getData() {
    return this.data;
  }

  /**
   * Bytes to write, encoded using base64.
   */
  public void setData(@NonNull final String data) {
    this.data = Preconditions.checkNotNull(data, "data");
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("memoryReference", this.memoryReference);
    b.add("offset", this.offset);
    b.add("allowPartial", this.allowPartial);
    b.add("data", this.data);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    WriteMemoryArguments other = (WriteMemoryArguments) obj;
    if (this.memoryReference == null) {
      if (other.memoryReference != null)
        return false;
    } else if (!this.memoryReference.equals(other.memoryReference))
      return false;
    if (this.offset == null) {
      if (other.offset != null)
        return false;
    } else if (!this.offset.equals(other.offset))
      return false;
    if (this.allowPartial == null) {
      if (other.allowPartial != null)
        return false;
    } else if (!this.allowPartial.equals(other.allowPartial))
      return false;
    if (this.data == null) {
      if (other.data != null)
        return false;
    } else if (!this.data.equals(other.data))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.memoryReference== null) ? 0 : this.memoryReference.hashCode());
    result = prime * result + ((this.offset== null) ? 0 : this.offset.hashCode());
    result = prime * result + ((this.allowPartial== null) ? 0 : this.allowPartial.hashCode());
    return prime * result + ((this.data== null) ? 0 : this.data.hashCode());
  }
}
