
package org.eclipse.mddi.modelbus.adapter.infrastructure.axisasync;

import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

import org.apache.axis.soap.SOAPConnectionImpl;


/**
 * SOAP Connection implementation
*
 * @author Nicolas Garandeau (lip6)
 *
 */
public class SOAPAsyncConnectionImpl extends SOAPAsyncConnection {
 
	/**
	 * The poll object which corresponds to this asynchronous connection
	 */
	private PollObject pollObject;
	
	/**
	 * The synchronous connection used to execute the task
	 */
	private SOAPConnection syncConnection;
	
	
	public SOAPAsyncConnectionImpl(){
		super();
		this.pollObject = new PollObject();
		this.syncConnection = new SOAPConnectionImpl();
	}
    
	
    /**
     * Sends the given message to the specified endpoint in a non-blocking way.
     * The result will be registered in an object that we can poll with 
     * isResultReady() and getResult() methods.
     * @param request the <CODE>SOAPMessage</CODE>
     *     object to be sent
     * @param endpoint an <code>Object</code> that identifies
     *            where the message should be sent. It is required to
     *            support Objects of type
     *            <code>java.lang.String</code>,
     *            <code>java.net.URL</code>, and when JAXM is present
     *            <code>javax.xml.messaging.URLEndpoint</code>
     * @param pool the <CODE>ThreadPoolQueue</CODE> object in which 
     * 		the call task will be executed
     * @return the <CODE>SOAPMessage</CODE> object that is the
     *     response to the message that was sent
     * @throws  SOAPException if there is a SOAP error
     */
	public String call(SOAPMessage request, Object endpoint,ThreadPoolQueue pool) throws SOAPException{
		String connectionId = this.hashCode()+"";
		launchThreadCall(request,endpoint,pool);
	
		return connectionId;
	}
	
	
	 /**
     * To get the information if the result is ready or not
     * @return the boolean that indicates if the result is ready
     */
	public boolean isResultReady(){
		return pollObject.isResultReady();
	
	}
	
	 /**
     * To get the result if the result is ready
     * @return the result object if the result is ready
     * @throws  ResultNotReadyException if the result is not yet ready
     */
	public Object getResult() throws ResultNotReadyException{
		return pollObject.getResult();
	}
	

    /**
     * Closes this <CODE>SOAPAsyncConnection</CODE> object.
     * @throws  SOAPException if there is a SOAP error
     */
	public void close() throws SOAPException {
		syncConnection.close();
	}
	
	/**
	 * The non-blocking method which invokes the request message a the endpointUrl with a given ThreadPoolQueue
	 * @param request the message to invoke
	 * @param endpoint the endpoint where the message should be sent
	 * @param pool the object in which the call task will be executed
	 * @throws SOAPException
	 */
	private void launchThreadCall(SOAPMessage request,Object endpoint,ThreadPoolQueue pool) throws SOAPException{
	
		Task task = new Task(syncConnection,request,endpoint,pollObject);
		pool.execute(task);
	}

 
    /**
     * Class responsible for invoking asynchrnously a request message on an endpoint, with a given pollObject
     */
    private static class Task implements Runnable{
		
		private SOAPConnection connection;
		private SOAPMessage  request;
		private Object endpoint;
		private PollObject po;
		
		public Task(SOAPConnection connection, SOAPMessage request,Object endpoint, PollObject po) {
			super();
			this.connection = connection;
			this.request = request;
			this.endpoint = endpoint;
			this.po = po;
		}
		
		public void run() {
			try {
				SOAPMessage resp = connection.call(request,endpoint);
				po.registerResult(resp);
			} catch (SOAPException e) {
				e.printStackTrace();
			}
		}
	}


}
