/*
 * $RCSfile: RegistryClient.java,v $
 * $Date: 2006/07/14 08:14:55 $
 * $Revision: 1.10 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * RegistryClient.java 
 * 
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 * @version $Revision: 1.10 $ $Date: 2006/07/14 08:14:55 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.registry;

import java.io.IOException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import org.apache.axis.AxisFault;
import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.mddi.modelbus.adapter.infrastructure.LoggerConfigurator;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.ModelUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.AlreadyRegistered;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.NotFound;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.WebRegistry;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.WebRegistryServiceLocator;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.emf_xmi.Emf2XmiConversion;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.emf_xmi.Xmi2EmfConversion;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.QualifiedServiceName;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;
import org.eclipse.mddi.modelbus.description.abstract_.Error;
import org.eclipse.mddi.modelbus.description.abstract_.MetaclassSpecification;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingServiceInterface;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.abstract_.Type;
import org.eclipse.mddi.modelbus.description.concrete.Tool;
import org.eclipse.uml2.impl.UML2PackageImpl;

/**
 * 
 * RegistryClient provides the communication between the Adapter and Registry *
 * 
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 *  
 */
public class RegistryClient {

    String registry_location;

    WebRegistry registry;

    private static Logger logger = Logger.getLogger(RegistryClient.class);

    public RegistryClient(String _registry_location)
            throws RegistryNotAvailableException {
        LoggerConfigurator.configure();
        try {
            registry_location = _registry_location;
            WebRegistryServiceLocator loc = new WebRegistryServiceLocator();
            registry = loc.getWebRegistry(new URL(registry_location));
        } catch (Exception e) {
            throw new RegistryNotAvailableException(
                    "URL= [" + registry_location+"]", e);
        }
    }

    /**
     * set of Tool objects. Local cache of data from Registry
     */
    Collection tools = new HashSet();

    public Tool lookupToolByModelingService(String serviceName,
            ToolSelectionStrategy selector)
            throws RegistryNotAvailableException {

        // optimize performance by calling loadDescriptionsFromRegistry() less
        // often.
        // first we try to look in cache
        // if not found, we refresh the cache by loading from Registry
        Tool t = null;
        //try to lookup in cache
        if (tools != null)
            t = selector.selectTool(serviceName, tools);
        if (t != null)
            return t;
        //otherwise lookup again the registry
        loadDescriptionsFromRegistry(serviceName);
        t = selector.selectTool(serviceName, tools);
        return t;
    }

    /**
     * 
     * getCachedToolDescription
     * 
     * @return a collection of "...modelbus.description.concrete.Tool" objects
     * @throws RegistryNotAvailableException
     * 
     *  
     */
    public Collection getCachedToolDescription()
            throws RegistryNotAvailableException {
        if (tools == null) {
            loadDescriptionsFromRegistry();
        }
        return tools;

    }

    public void loadDescriptionsFromRegistry()
            throws RegistryNotAvailableException {

        //        try {
        String[] results;
        Collection col;
        try {
            results = registry.lookupAllModelingServiceInterfaces();
            col = toEmf(results);
            col = toEmf(results);
            Collection modelingServiceInterfaces = ModelUtil.findElementByType(
                    col, "ModelingServiceInterface");
            Iterator it = modelingServiceInterfaces.iterator();
            while (it.hasNext()) {
                ModelingServiceInterface interf = (ModelingServiceInterface) it
                        .next();
                results = registry.lookupToolsByModelingServiceInterface(interf
                        .getName());
                col = toEmf(results);
                Collection _tools = ModelUtil.findElementByType(col, "Tool");
                tools.addAll(_tools);

                //register all imported meta-models
                registerResources(_tools);
            }
        } catch (RemoteException e) {
            if (e instanceof AxisFault) {
                logger.error("Looking up: " + ((AxisFault) e).dumpToString());

                throw new RegistryNotAvailableException("URL= "
                        + registry_location, e);
            }
            logger.error("RemoteException:", e);
            e.printStackTrace();
        } catch (IOException e) {
            logger.error("IOException:", e);
            e.printStackTrace();
        }
    }

    public void loadDescriptionsFromRegistry(String serviceName)
            throws RegistryNotAvailableException {
        QualifiedServiceName qname = QualifiedServiceName
                .getQualifiedServiceName(serviceName);

        if (qname.interfaceName == null) {
            loadDescriptionsFromRegistry();
            return;
        }
        String[] results = null;
        try {
            results = registry
                    .lookupToolsByModelingServiceInterface(qname.interfaceName);
        } catch (NotFound e) {
            logger.error("NotFound:", e);
            e.printStackTrace();
        } catch (RemoteException e) {
            if (e instanceof AxisFault) {
                logger.error("Looking up: " + ((AxisFault) e).dumpToString());

                throw new RegistryNotAvailableException("URL= "
                        + registry_location, e);
            }
            logger.error("RemoteException:", e);
            e.printStackTrace();
        }
        Collection col = null;
        try {
            col = toEmf(results);
        } catch (IOException e1) {
            logger.error("IOException:", e1);
            e1.printStackTrace();
        }
        Collection _tools = ModelUtil.findElementByType(col, "Tool");
        tools.addAll(_tools);

        //register all imported meta-models
        registerResources(_tools);

    }

    private Collection toEmf(String[] results) throws IOException {
        //DEBUG
        logger.debug("Registry lookup result: ");
        for (int i = 0; i < results.length; i++) {
            logger.debug(results[i]);
        }
        //
        Collection strings = new Vector();
        for (int i = 0; i < results.length; i++) {
            strings.add(results[i]);
        }
        Collection emfobjs = new Xmi2EmfConversion().convertFromStrings(
                strings, false);
        return emfobjs;
    }

    public String registerTool(Tool desc) throws RegistryNotAvailableException {
        String token = new String();
        try {
            Collection emfobjs = new Vector();
            emfobjs.add(desc);
            String s = Emf2XmiConversion.convertToString(emfobjs);
            token = registry.registerTool(s);
            return token;
        } catch (Exception e) {
            if (e instanceof AlreadyRegistered) {
                logger
                        .warn("Registering: Modeling Service is already registred");
                return token;
            } else if (e instanceof AxisFault) {
                logger.error("Registry not available: Fault at the remote site "
                        + ((AxisFault) e).dumpToString());
                throw new RegistryNotAvailableException("URL= ["
                        + registry_location+"]", e);
            } else {
                logger.error("Registry not available");
                throw new RegistryNotAvailableException("URL= ["
                        + registry_location+"]", e);
            }
        }
    }

    public void deregisterTool(String token)
            throws RegistryNotAvailableException {
        try {
            registry.deregisterTool(token);
        } catch (Exception e) {
            throw new RegistryNotAvailableException(
                    "URL= " + registry_location, e);
        }
    }

    /**
     * Tests the availability
     * 
     * @return TODO implement
     */
    public boolean isAvailable() {
        return true;
    }

    /**
     * @return Returns the tools.
     */
    public Collection getTools() {
        return tools;
    }

    /**
     * 
     * registerResources Registers all resources associated with tool
     * description. For example, meta-models.
     * 
     * @param tool
     *  
     */
    public static void registerResources(Tool tool) {

        ModelingServiceInterface msi = tool.getInterface();
        Iterator it = msi.getService().iterator();
        while (it.hasNext()) {
            ModelingService ms = (ModelingService) it.next();
            //check all parameters
            Iterator it2 = ms.getParameter().iterator();
            while (it2.hasNext()) {
                Parameter param = (Parameter) it2.next();
                Type type = param.getType();
                if (type instanceof ModelType) {
                    EPackage metamodel = getMetaModel((ModelType) type);
                    if (metamodel != null)
                        registerMetaModel(metamodel);
                }
            }
            //check all errors
            Iterator it3 = ms.getServiceError().iterator();
            while (it3.hasNext()) {
                Error error = (Error) it3.next();
                Type type = error.getType();
                if (type instanceof ModelType) {
                    EPackage metamodel = getMetaModel((ModelType) type);
                    if (metamodel != null)
                        registerMetaModel(metamodel);
                }

            }
        }
    }

    /**
     * 
     * registerResources regesters resources for all tools
     * 
     * @param tools -
     *            Collection of Tools to register
     *  
     */

    public static void registerResources(Collection tools) {
        Iterator it = tools.iterator();
        while (it.hasNext()) {
            Tool t = (Tool) it.next();
            registerResources(t);
        }
    }

    private static EPackage getMetaModel(ModelType type) {
        ModelType mtype = (ModelType) type;
        Iterator meta_it = mtype.getContent().iterator();
        while (meta_it.hasNext()) {
            MetaclassSpecification metaspec = (MetaclassSpecification) meta_it
                    .next();
            EClass metaclass = metaspec.getMetaClass();

            EPackage metamodel = getMetaModel(metaclass.getEPackage());

            return metamodel;
        }
        return null;
    }

    private static EPackage getMetaModel(EPackage pack) {

        EPackage next_pack = pack.getESuperPackage();
        UML2PackageImpl.init();

        while (next_pack != null) {
            pack = next_pack;
            next_pack = pack.getESuperPackage();
        }
        return pack; //the uppest EPackage = metamodel
    }

    private static void registerMetaModel(EPackage metamodel) {
        String uri = metamodel.getNsURI();
        if (uri == null || "".equals(uri)) {
            EPackage superp = metamodel.getESuperPackage();
            if (superp == null) {
                uri = "http://model/" + metamodel.getName();
            } else {
                uri = superp.getNsURI() + "/" + metamodel.getName();
            }
            metamodel.setNsURI(uri);
        }
        EPackage registered = EPackage.Registry.INSTANCE.getEPackage(metamodel
                .getNsURI());
        if (registered != null && registered != metamodel) {
            logger
                    .warn("The EPackage with NS URI "
                            + metamodel.getNsURI()
                            + " is already loaded! \n"
                            + "It will be kept unchanged, though ModelBus Registry can contain an updated version.");
            return;
        }
        //if new, load it.
        logger.debug("Registring NS URI = " + metamodel.getNsURI()
                + "\for EPackage" + metamodel);
        EPackage.Registry.INSTANCE.put(metamodel.getNsURI(), metamodel);
        //run procedure for all subpackages
        for (Iterator it = metamodel.getESubpackages().iterator(); it.hasNext();) {
            EPackage subp = (EPackage) it.next();
            registerMetaModel(subp);
        }

    }
}
