/*
 * $RCSfile: AbstractBasicModelSerializer.java,v $
 * $Date: 2006/06/02 15:01:00 $
 * $Revision: 1.3 $
 * $Author: xblanc $
 */
 
/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */
 
/*
 * AbstractModelSerializer.java
 * 
 * @author Prawee Sriplakich
 * @version $Revision: 1.3 $ $Date: 2006/06/02 15:01:00 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.serialize;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.Map;
import java.util.Vector;
import java.util.WeakHashMap;

import org.apache.log4j.Logger;

import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;


/**
 * This class provides a part of implementation of ModelSerializer.
 * It hasd the following restrictions
 * <br>
 * - A service parameter is serialized to an XML document 
 * (1-to-1 mapping: 1 parameter = 1 XMI document)
 * <br>
 * - Several parameters can not refer to the same document.
 * <br>
 * - A parameter value can not contain several XMI documents
 * <br>
 * - Models corresponding to different parameter values can not have links to each other
 * (those links will not be serialized).
 * <br>
 * 
 * @author P. Sriplakich
 *
 *
 */
public abstract class AbstractBasicModelSerializer implements ModelSerializer {

    private static Logger logger = Logger.getLogger(AbstractBasicModelSerializer.class);
    
    /**
     * remember the URI of the serialized XMI documents.
     * 
     */
    protected Map model_uri  = new WeakHashMap(); 
    // WeakHashMap does not prevent the objects to be garbage-collected.
    
    
    public SerializedXmiDocument[] serialize(Parameter[] params, Object[] values) {
        Vector docVector = new Vector();        
        for(int i=0; i<params.length; i++) {
            if(params[i].getType() instanceof ModelType) {
              if(params[i].getUpper()!=1) {
                  Object[] array = (Object[]) values[i];
                  for(int j=0; j<array.length; j++) {
                      SerializedXmiDocument doc = serializeDocument(
                              params[i].getName() +"." +j , array[j] );
                      docVector.add(doc);
                  }
              } else {
                  SerializedXmiDocument doc = serializeDocument(
                          params[i].getName(), values[i] );
                  docVector.add(doc);
              }
            }
        }
        return (SerializedXmiDocument[]) docVector
            .toArray(new SerializedXmiDocument[docVector.size()]);        
    }
    
    
    SerializedXmiDocument serializeDocument(String uri, Object value) {
        SerializedXmiDocument doc = new SerializedXmiDocument();
        doc.setUri(uri);
        model_uri.put(value, uri);
        try {            
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            serialize(value, out);
            doc.setXmi(out.toString());
          } catch (Exception e) {
            logger.error(e);
            e.printStackTrace();
          }     
        return doc;
    }
    
    
    
    /**
     * to be override.
     * Serialize an object representing a model to an XMI stream
     * 
     * @param o
     * @return
     * 
     *
     */
    abstract protected void serialize(Object o, OutputStream xmiStream);
    

    public DeserializedModel[] deserialize(SerializedXmiDocument[] documents) {
        DeserializedModel[] result = new DeserializedModel[documents.length];
        for(int i=0; i<documents.length; i++) {
            result[i] = new DeserializedModel();
            result[i].setUri( documents[i].getUri() );
            try {
               ByteArrayInputStream bin = new ByteArrayInputStream(
                       documents[i].getXmi().getBytes());
               Object o = deserialize(bin);
               result[i].setValue(o);
            } catch(Exception e) {
                logger.error(e);
                e.printStackTrace();
            }
        }
        return result;
    }
    
    /**
     * To be override
     * deserialize a stream of an XMI document to an object representing the model.
     * 
     * @param xmiDocument
     * @return
     * 
     *
     */
    abstract protected Object deserialize(InputStream xmiStream);

    public ModelElementReference[] getReferences(
            SerializedXmiDocument[] documents, Parameter p, Object o) {
        ModelElementReference[] result = new ModelElementReference[1];
        result[0] = new ModelElementReference();
        result[0].setUri( (String)model_uri.get(o) );
        result[0].setRef("/");
        return result;
    }
    
    

    public Object dereference(
            DeserializedModel[] data, Parameter p, ModelElementReference[] refs) {
        Vector v = new Vector();
        String uri = refs[0].getUri();
        for(int j=0; j<data.length; j++) {
          if(data[j].getUri().equals(uri)) {
              return data[j].getValue();
          }   
        }
        logger.error("Resource Not Found " + uri);
        return null;
    }
    
    

}
