/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * Created on 5 juil. 2005
 *
 */
package org.eclipse.mddi.modelbus.adapter.test.unit;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Collection;

import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

import org.apache.log4j.Logger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.LoggerConfigurator;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.XmiUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.DefaultMarshaler;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.SoapUtil;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;

import org.eclipse.mddi.modelbus.description.abstract_.AbstractFactory;
import org.eclipse.mddi.modelbus.description.abstract_.EnumerationType;
import org.eclipse.mddi.modelbus.description.abstract_.Error;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.abstract_.PrimitiveType;
import org.eclipse.mddi.modelbus.description.abstract_.impl.AbstractPackageImpl;

import junit.framework.TestCase;

/**
 * Tests marshalling and unmarshalling mechanism. In each test methods, a
 * different type of parameter is marshalled. Then it is unmarshalled. The
 * results of marshalling and unmarshalling are compared. If they are correct,
 * then they must be equal.
 * 
 * @author P. Sriplakich, Andrey Sadovykh (LIP6)
 *  
 */
public class MarshallTest extends TestCase {

    private static Logger logger = Logger.getLogger(MarshallTest.class);

    protected DefaultMarshaler marshaler;

    public static void main(String[] args) {
        LoggerConfigurator.configure();
        junit.textui.TestRunner.run(MarshallTest.class);
    }

    SOAPMessage msg;

    SOAPElement topElem;

    // a factory for creating service description
    AbstractFactory factory;

    /*
     * @see TestCase#setUp()
     */
    protected void setUp() throws Exception {
        try {
            marshaler = new DefaultMarshaler(null);
            msg = SoapUtil.messageFactory.createMessage();
            SOAPBody body = msg.getSOAPBody();
            DefaultMarshaler.setNamespaces(body);
            topElem = body.addChildElement("service1", "modelbus");
            factory = AbstractPackageImpl.init().getAbstractFactory();
        } catch (Exception e) {
            throw new Exception("TestCase can not be initialized", e);
        }
    }

    /*
     * @see TestCase#tearDown()
     */
    protected void tearDown() throws Exception {

    }

    public void testMarshall_string() throws Exception {

        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("string");
        p.setType(t);

        Object parameterValue = "test string";

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_string");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        
        Object unmarshalledParam;
        unmarshalledParam = marshaler.unmarshal(p, topElem);
        assertEquals(parameterValue, unmarshalledParam);

    }

    public void testMarshall_string_array() throws Exception {
        // create parameter description
     
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("string");
        p.setType(t);

        String[] parameterValue = new String[] { "string1", "string2",
                "string3" };

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_string_array");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        String[] unmarshalledParam = (String[]) marshaler.unmarshal(p, topElem);

        for (int i = 0; i < parameterValue.length; i++) {
            assertEquals(parameterValue[i], unmarshalledParam[i]);
        }

    }

    public void testMarshall_boulean_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("boolean");
        p.setType(t);

        Boolean[] parameterValue = new Boolean[] { new Boolean(true),
                new Boolean(false), new Boolean(false) };

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_boolean_array");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Boolean[] unmarshalledParam = (Boolean[]) marshaler.unmarshal(p,
                topElem);

        for (int i = 0; i < parameterValue.length; i++) {
            assertEquals(parameterValue[i], unmarshalledParam[i]);
        }

    }

    public void testMarshall_int_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("integer");
        p.setType(t);

        Integer[] parameterValue = new Integer[] { new Integer(456),
                new Integer(789), new Integer(687) };

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_int_array");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Integer[] unmarshalledParam = (Integer[]) marshaler.unmarshal(p,
                topElem);

        for (int i = 0; i < parameterValue.length; i++) {
            assertEquals(parameterValue[i], unmarshalledParam[i]);
        }

    }

    public void testMarshall_double_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("double");
        p.setType(t);

        Double[] parameterValue = new Double[] { new Double(456.757),
                new Double(789.757), new Double(687.757) };

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_double_array");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Double[] unmarshalledParam = (Double[]) marshaler.unmarshal(p, topElem);

        for (int i = 0; i < parameterValue.length; i++) {
            assertEquals(parameterValue[i], unmarshalledParam[i]);
        }

    }

    public void testMarshall_model_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        ModelType t = factory.createModelType();
        p.setType(t);

        Collection[] parameterValue = new Collection[] {
                Util.createUml2Model(), Util.createUml2Model(),
                Util.createUml2Model() };

        topElem.removeContents();
        //marshal
        logger.info("testMarshall_model_array");

        Parameter[] params = new Parameter[1];
        params[0] = p;
        Object[] values = new Object[1];
        values[0] = parameterValue;
        marshaler.marshal(params, values, topElem);


        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");


        //unmarshal
        Object[] umarshaledValues = marshaler.unmarshal(params, topElem);

        Collection[] unmarshalledParam = (Collection[]) umarshaledValues[0];

//        try {
//            logger.info("unmarshalled model:"
//                    + Emf2XmiConversion.convertToString(unmarshalledParam[1]));
//        } catch (IOException e) {
//            e.printStackTrace();
//        }

        for (int i = 0; i < parameterValue.length; i++) {
            assertTrue(Util.compareModels(parameterValue[i],
                    unmarshalledParam[i]));
        }

    }

    public void testMarshall_null() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("boolean");
        p.setType(t);

        Object parameterValue = null;

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_null");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Object unmarshalledParam = marshaler.unmarshal(p, topElem);

        assertEquals(parameterValue, unmarshalledParam);
    }

    public void testMarshall_boolean() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("boolean");
        p.setType(t);

        Object parameterValue = new Boolean(true);

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_boolean");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Object unmarshalledParam = marshaler.unmarshal(p, topElem);

        assertEquals(parameterValue, unmarshalledParam);
    }

    public void testMarshall_int() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("integer");
        p.setType(t);

        Object parameterValue = new Integer(456573);

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_boolean");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Object unmarshalledParam = marshaler.unmarshal(p, topElem);

        assertEquals(parameterValue, unmarshalledParam);
    }

    public void testMarshall_enum() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        EnumerationType t = factory.createEnumerationType();
        t.getLabels().add("red");
        t.getLabels().add("black");
        t.getLabels().add("white");
        t.getLabels().add("blue");

        p.setType(t);

        Object parameterValue = "red";

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_enum");
        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        Object unmarshalledParam = marshaler.unmarshal(p, topElem);

        logger.info("unmarshall result = \n" + unmarshalledParam + "\n");
        assertEquals(parameterValue, unmarshalledParam);
    }

    public void testMarshall_enum_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        EnumerationType t = factory.createEnumerationType();
        t.getLabels().add("red");
        t.getLabels().add("black");
        t.getLabels().add("white");
        t.getLabels().add("blue");

        p.setType(t);

        String[] parameterValue = new String[] { "red", "blue", "white" };

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_enum_array");

        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        String[] unmarshalledParam = (String[]) marshaler.unmarshal(p,
                topElem);


        for (int i = 0; i < parameterValue.length; i++) {
            assertEquals(parameterValue[i], unmarshalledParam[i]);
        }

    }

    public void testMarshall_double() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("double");
        p.setType(t);

        Object parameterValue = new Double(456573.646546);

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_boolean");
        logger.info("marshall result = \n" + topElem + "\n");
        Object unmarshalledParam = marshaler.unmarshal(p, topElem);

        assertEquals(parameterValue, unmarshalledParam);
    }

    public void testMarshall_model() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        ModelType t = factory.createModelType();
        p.setType(t);

        Collection parameterValue = Util.createUml2Model();

        // logger.info("Size of model1:"+parameterValue.size());

        topElem.removeContents();
        logger.info("testMarshall_model");
        Parameter[] params = new Parameter[1];
        params[0] = p;
        Object[] values = new Object[1];
        values[0] = parameterValue;
        marshaler.marshal(params, values, topElem);

        logger.info("marshall result = \n" + XmiUtil.xmlElemToString(topElem) + "\n");
        

        //unmarshal

        
        Object[] umarshaledValues = marshaler.unmarshal(params,
                topElem);
        
        Collection unmarshalledParam = (Collection) umarshaledValues [0];
        
//        try {
//            logger.info("unmarshalled model:"
//                    + Emf2XmiConversion.convertToString(unmarshalledParam));
//        } catch (IOException e) {
//            e.printStackTrace();
//        }



        // logger.info("Size of model2:"+unmarshalledParam.size());

        boolean test = Util.compareModels(parameterValue, unmarshalledParam);

        logger.info("Compare result:" + test);
        assertTrue(test);
    }

    public void testMarshall_NullError() throws Exception {
        ModelingServiceError e1 = new ModelingServiceError("someName", null);

        SOAPMessage msg;
        //marshal
        msg = SoapUtil.messageFactory.createMessage();
        SOAPFault f = msg.getSOAPBody().addFault();
        marshaler.marshalModelingServiceError(e1, f);

        System.out.println(msg.getSOAPBody());

        //unmarshal
        ModelingServiceError e2 = marshaler.unmarshalModelingServiceError(null,f);

        System.out.println(e2 + "  :  " + e2.getInfo());
        
        boolean test =  (e1.getName().equals(e2.getName()))&&(e1.getInfo()==e2.getInfo());
        
        assertTrue(test);
        

    }

    public void testMarshall_StringError() throws Exception {
        ModelingServiceError e1 = new ModelingServiceError("someName", "someDescription");

        SOAPMessage msg;
        //marshal
        msg = SoapUtil.messageFactory.createMessage();
        SOAPFault f = msg.getSOAPBody().addFault();
        marshaler.marshalModelingServiceError(e1, f);

        System.out.println(msg.getSOAPBody());

        //unmarshal
        ModelingServiceError e2 = marshaler.unmarshalModelingServiceError(null,f);

        System.out.println(e2 + "  :  " + e2.getInfo());
        
        boolean test =  (e1.getName().equals(e2.getName()))&&(((String)e1.getInfo()).equals((String)e2.getInfo()));
        
        assertTrue(test);
        

    }

    public void testMarshall_ModelError() throws Exception {
        // create parameter description
        Error errorType = factory.createError();
        errorType.setName("someName");
        ModelType t = factory.createModelType();
        errorType.setType(t);

        Collection model = Util.createUml2Model();
        ModelingServiceError e1 = new ModelingServiceError(errorType, model);

        SOAPMessage msg;
        //marshal
        msg = SoapUtil.messageFactory.createMessage();
        SOAPFault f = msg.getSOAPBody().addFault();
        marshaler.marshalModelingServiceError(e1, f);

        System.out.println(msg.getSOAPBody());

        //unmarshal
        ModelingServiceError e2 = marshaler.unmarshalModelingServiceError(errorType,f);

        System.out.println(e2 + "  :  " + e2.getInfo());
        
        boolean testModels = Util.compareModels((Collection)e1.getInfo(), (Collection)e2.getInfo());
        
        boolean test =  (e1.getName().equals(e2.getName()))&&testModels;
        
        assertTrue(test);
        
    }    
    
    public void testMarshall_binary() throws Exception {

        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("binary");
        p.setType(t);
        
        String file = "some image or file";

        Object parameterValue = new ByteArrayInputStream(file.getBytes());

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_binary");
        logger.info("marshall result = \n" + topElem + "\n");
        Object unmarshalledParam;
        unmarshalledParam = marshaler.unmarshal(p, topElem);
        
        InputStream is = (InputStream)unmarshalledParam;
        byte[] buffer = new byte [is.available()];
        is.read(buffer);
        
        String testFile = new String (buffer);
        
        logger.info ("Unmarshaled file:"+testFile);
        
        assertEquals(file, testFile);

    }

    public void testMarshall_binary_array() throws Exception {
        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        p.setUpper(-1);
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("binary");
        p.setType(t);

        String[] files = new String[] { "aaaaaaa", "bbbbbbb",
                "cccccccccc" };
        
        Object [] parameterValue = new Object [files.length];
        for (int i =0; i<files.length; i++){
            parameterValue [i] = new ByteArrayInputStream((files[i]).getBytes());
        }
        

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_string_array");
        logger.info("marshall result = \n" + topElem + "\n");
        InputStream[] unmarshalledParam = (InputStream[]) marshaler.unmarshal(p, topElem);

        
        for (int i = 0; i < parameterValue.length; i++) {
            byte[] buffer = new byte [unmarshalledParam[i].available()];
            unmarshalledParam[i].read(buffer);
            
            String testFile = new String (buffer);
            
            logger.info ("Unmarshaled file:"+testFile);
            
            assertEquals(files[i], testFile);

        }

    }

    public void testMarshall_file() throws Exception {

        // create parameter description
        Parameter p = factory.createParameter();
        p.setName("p1");
        PrimitiveType t = factory.createPrimitiveType();
        t.setName("binary");
        p.setType(t);
        
        File ft = new File ("textfile.txt");
        FileInputStream ist = new FileInputStream(ft);
        
        byte[] data = new byte [ist.available()];
        ist.read(data);
        
        String textData = new String(data);
        
        File f = new File ("textfile.txt");
        FileInputStream fis = new FileInputStream(f);

        Object parameterValue = fis;

        topElem.removeContents();
        marshaler.marshal(p, parameterValue, topElem);
        logger.info("testMarshall_binary");
        logger.info("marshall result = \n" + topElem + "\n");
        Object unmarshalledParam;
        unmarshalledParam = marshaler.unmarshal(p, topElem);
        
        InputStream is = (InputStream)unmarshalledParam;
        byte[] buffer = new byte [is.available()];
        is.read(buffer);
        
        String testFile = new String (buffer);
        
        logger.info ("Unmarshaled file:"+testFile);
        
        assertEquals(textData, testFile);

    }
    

}
