

package org.eclipse.mddi.modelbus.adapter.infrastructure.axisasync;

import java.util.LinkedList;

/**
 * a ThreadPoolQueue object is reponsible for executing tasks 
 * given a maximum working thread number.
 * @author Nicolas Garandeau (lip6)
 *
 */
public class ThreadPoolQueue{
	/**
	 * the default number of working threads in the pool
	 */
	private static final int DEFAULT_THREAD_NUMBER = 100;
	
	/**
	 * the number of working threads in the pool
	 */
	private final int nThreads;
	
	/**
	 * the working threads of the pool
	 */
	private final PoolThread[] threads;
	
	/**
	 * the queue that conains the tasks that are waiting for their execution
	 */
	private final LinkedList queue;
	
	/**
	 *  the boolean that indicates if the queue is active or not
	 */
	private boolean activeQueue;
	
	
	public ThreadPoolQueue()
	{
		this(DEFAULT_THREAD_NUMBER);	
	}
	
	
	public ThreadPoolQueue(int nThreads)
	{
		this.nThreads = nThreads;
		queue = new LinkedList();
		threads = new PoolThread[nThreads];
		activeQueue =true;
		
		for (int i=0; i<nThreads; i++) {
			threads[i] = new PoolThread(i+1);
			threads[i].start();
		}
	}
	
	public boolean isEmpty(){
		return queue.isEmpty();
	}
	
	/**
	 * Execute the task r with a free working thread.
	 * If there is no free working thread to execute this task, 
	 * the task is added in the queue, waiting a free working thread.
	 * @param r the task to execute
	 * @throws QueueClosedException if the queue is closed
	 */
	public void execute(Runnable r) throws QueueClosedException{
		
		if(!activeQueue)
			throw new QueueClosedException();
		
		synchronized(queue) {
			queue.addLast(r);
			queue.notify();
		}
	}
	
	/**
	 * Interrupt the working threads after having finished their current task,
	 * wether the queue is empty or not.
	 */
	public void shutdown(){
		activeQueue = false;
		for (int i=0; i<nThreads; i++) {
			threads[i].shutdown();
		}
	}
	
	
	
	/**
	 * Interrupt the working threads after having finished all the tasks,
	 * wether they are current tasks or waiting tasks in the queue.
	 */
	public void close(){
		
		activeQueue = false;
		
		// ensure that queue is empty
		synchronized(queue) {
			while (!queue.isEmpty()) {
				try
				{
					queue.wait();
				}
				catch (InterruptedException e)
				{
					e.printStackTrace();
				}
			}
		}
		
		shutdown();
	}
	
	
	public class QueueClosedException extends RuntimeException{
	}
	
	/**
	 * This class represents a worker thread in the pool
	 * @author Nicolas
	 *
	 */
	private class PoolThread extends Thread {
		private boolean active =true;
		private int id;
		
		public PoolThread(int id){
			this.id=id;
		}
		/**
		 * to run the thread
		 */
		public void run() {
			
			Runnable r;
			while (active) {
				synchronized(queue) {
					while (queue.isEmpty() && active) {		
						try
						{
							queue.wait();
						}
						catch (InterruptedException e)
						{
							e.printStackTrace();
						}
					}
					
					if(!active){
						return; 
					}
					r = (Runnable) queue.removeFirst();
					queue.notify(); 
				}
				
				try {
					r.run();	
				}
				catch (RuntimeException e) {
					e.printStackTrace();
				}
			}
			
		}
		
		/**
		 * to end the thread after having executed its current task
		 */
		public void shutdown(){
			active = false;
			synchronized(queue) {
				queue.notifyAll(); 
			}
		}
	}
}
