/*
 * $RCSfile: ConflictDetectionHelper.java,v $
 * $Date: 2006/05/17 12:12:12 $
 * $Revision: 1.2 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * ConflictDetectionHelper.java Add description 
 * 
 * @author Prawee Sriplakich
 * @version $Revision: 1.2 $ $Date: 2006/05/17 12:12:12 $
 * @see Add references
 *
 * TODO Add description
 * TODO Add references
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.merge;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import org.apache.log4j.Logger;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.Conflict;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.Delete;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.InsertLink;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.MergeFactory;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.ModifyLink;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.ModifyLinkElement;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.ModifyPrimitive;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.NodeLevelSubDelta;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.RemoveLink;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.SubDelta;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.util.SimpleModelLabelProvider;
import org.eclipse.mddi.modelbus.adapter.infrastructure.merge.model.impl.MergePackageImpl;;

public class ConflictDetectionHelper {
    
    static Logger logger = Logger.getLogger("ConflictDetection");
    
    static MergeFactory fac;
    static {
        fac = MergePackageImpl.init().getMergeFactory();
    }
    
    /**
     * Returns the conflict or null if not detected.
     * 
     * The conflict contains the specified ModifityPrimitive instance in one side
     * and one detected ModifyPrimitive instance in the other side
     * 
     *
     */
    public static Conflict detectConcurrentUpdatePrimitive(DeltaQuery q,
            ModifyPrimitive mp) {
        ModifyPrimitive mp2 = getConcurrentModifyPrimitive(q, mp);
        if (mp2 == null)
            return null;
        Conflict conf = fac.createConflict();
        conf.setType(ConflictDetection.CONCURRENT_UPDATE_PRIMITIVE);
        if (DeltaQuery.isLocalSubDelta(mp)) {
            conf.getLocal().add(mp);
            conf.getRemote().add(mp2);
        } else {
            conf.getLocal().add(mp2);
            conf.getRemote().add(mp);
        }
        return conf;
    }

    /**
     * return a conflicting ModifyPrimitive sub-delta or null if no conflict
     */
    public static ModifyPrimitive getConcurrentModifyPrimitive(DeltaQuery q,
            ModifyPrimitive mp) {
        ModifyPrimitive mp2 = (ModifyPrimitive) q.getModifyPrimitive((
                (NodeLevelSubDelta) mp.eContainer())
                .getId(), (EAttribute) mp.getProperty());
        if (mp2 == null)
            return null;
        if (MergeHelper.areDiffrent(mp2.getValue(), mp.getValue())) {
            return mp2;
        }
        return null;
    }

    /**
     * Returns the conflict or null if not detected.
     * 
     * The conflict contains the specified Delete instance in one side
     * and detected ModifyPrimitive instances in the other side
     * 
     *
     */
    public static Conflict detectDeleteUpdate(DeltaQuery q, Delete delete) {
        Collection col = ConflictDetectionHelper.getDeleteModifyPrimitive(q,
                delete);
        if (col.isEmpty())
            return null;
        Conflict conf = fac.createConflict();
        conf.setType(ConflictDetection.DELETE_UPDATE);
        if (DeltaQuery.isLocalSubDelta(delete)) {
            conf.getLocal().add(delete);
            conf.getRemote().addAll(col);
        } else {
            conf.getLocal().addAll(col);
            conf.getRemote().add(delete);
        }
        return conf;
    }

    /**
     * 
     * return a collection containing ModifyPrimitive instances that are
     * conflicting or empty collection if no conflict
     */
    public static Collection getDeleteModifyPrimitive(DeltaQuery q,
            Delete delete) {
        Collection result = new Vector();
        NodeLevelSubDelta sub = q.getNodeLevelSubDelta(delete.getId());
        if (sub instanceof Delete || sub ==null) {
            return result;
        }
        for (Iterator it = sub.getContent().iterator(); it.hasNext();) {
            Object o = it.next();
            if (o instanceof ModifyPrimitive) {
                result.add(o);
            }
        }
        return result;
    }

    /**
     * Returns the conflict or null if not detected.
     * 
     * The conflict contains the spicified InsertLink instance in one side
     * and detected InsertLink/RemoveLink instances in the other side
     * 
     * @param q
     * @param il
     * @return
     * 
     *
     */
    public static Conflict detectOrderConflict(DeltaQuery q, ModifyLinkElement mle) {
        Collection c = null;
        if(mle instanceof InsertLink) {
           c = ConflictDetectionHelper.getOrderConflictWithInsertLink(q,
                   (InsertLink)mle);
        } else if(mle instanceof RemoveLink) {
           c = ConflictDetectionHelper.getOrderConflictWithRemoveLink(q,
                   (RemoveLink)mle);
        }
        if (c.isEmpty())
            return null;
        Conflict conf = fac.createConflict();
        conf.setType(ConflictDetection.ORDER_CONFLICT);
        if (DeltaQuery.isLocalSubDelta(mle)) {
            conf.getLocal().add(mle);
            conf.getRemote().addAll(c);
        } else {
            conf.getLocal().addAll(c);
            conf.getRemote().add(mle);
        }
        return conf;
    }

    /**
     * return a collection containing InsertLink or RemoveLink instances
     * conflicting to the specified InsertLink instance 
     * or empty collection if no conflict
     */
    public static Collection getOrderConflictWithInsertLink(DeltaQuery q,
            InsertLink il) {
        Collection result = new Vector();
        ModifyLink ml1 = (ModifyLink) il.eContainer();
        if (!ml1.getProperty().isOrdered()) {
            return result;
        }
        NodeLevelSubDelta mod1 = (NodeLevelSubDelta) ml1.eContainer();
        ModifyLink ml2 = (ModifyLink) q.getModifyLink(mod1.getId(),
                (EReference) ml1.getProperty());
        if (ml2 == null)
            return result;
        if (ml2.getRemove() != null) {
            if (areConflicting(il, ml2.getRemove())) {
                result.add(ml2.getRemove());
            }
        }
        for (Iterator it = ml2.getInsert().iterator(); it.hasNext();) {
            InsertLink il2 = (InsertLink) it.next();
            if (areConflicting(il, il2) && !areEquivalent(il, il2)) {
                result.add(il2);
            }
        }
        return result;
    }
    
    /**
     * return a collection containing InsertLink or RemoveLink instances
     * conflicting to the specified RemoveLink instance  
     * or empty collection if no conflict
     */    
    public static Collection getOrderConflictWithRemoveLink(DeltaQuery q, RemoveLink rl) {
        Collection result = new Vector();
        ModifyLink ml1 = (ModifyLink) rl.eContainer();
        if (!ml1.getProperty().isOrdered()) {
            return result;
        }
        NodeLevelSubDelta mod1 = (NodeLevelSubDelta) ml1.eContainer();
        ModifyLink ml2 = (ModifyLink) q.getModifyLink(mod1.getId(),
                (EReference) ml1.getProperty());
        if (ml2 == null)
            return result;
        for (Iterator it = ml2.getInsert().iterator(); it.hasNext();) {
            InsertLink il = (InsertLink) it.next();
            if (areConflicting(il, rl)) {
                result.add(il);
            }
        }
        return result;        
    }
    
    

    static boolean areConflicting(ModifyLinkElement il1, ModifyLinkElement il2) {
        Collection c = new Vector(il1.getRef());
        c.retainAll(il2.getRef());
        return !c.isEmpty();
    }


    static boolean areEquivalent(InsertLink e1, InsertLink e2) {
        if (MergeHelper.areDiffrent(e1.getPosAfter(), e2.getPosAfter())) {
            return false;
        }
        if (!e1.getRef().equals(e2.getRef())) {
            return false;
        }
        return true;
    }

    /**
     * Returns the conflict or null if not detected.
     * 
     * The conflict contains Delete instance in one side 
     * and detected InsertLink instances in the other side.
     */    
    public static Conflict detectDanglingLink(DeltaQuery q, Delete delete) {
        Collection col = ConflictDetectionHelper.getDanglingInsertLink(q,
                delete);
        if(col.isEmpty()) {
            return null;
        }
        Conflict conf = fac.createConflict();
        conf.setType(ConflictDetection.DANGLING_LINK);
        if (DeltaQuery.isLocalSubDelta(delete)) {
            conf.getLocal().add(delete);
            conf.getRemote().addAll(col);
        } else {
            conf.getLocal().addAll(col);
            conf.getRemote().add(delete);
        }
        return conf;
    }

    /**
     * return a collection containing InsertLink instances that are conflicting
     * or empty collection if no conflict
     */
    public static Collection getDanglingInsertLink(DeltaQuery q, Delete delete) {
        Collection result = new Vector();
        result.addAll(q.getInsertLinks(delete.getId()));
        result.addAll(q.getOppositeInsertLinks(delete.getId()));
        return result;
    }

    
    
    
    /**
     * 
     * detect multiplicity violation of the value of the Reference r.
     * This is done by testing the application of InsertLink/RemoveLink/Delete instances.
     * 
     * @param o the model element that owns the reference value, or null if this is a new object
     * @param r EReference defining the reference value
     * 
     * @return
     * 
     *
     */    
    public static Conflict checkMultiplicity(String id, EObject o, EReference r, 
            DeltaQuery localQuery, DeltaQuery remoteQuery) { 
        if (MergeHelper.isMainReference(r)) {
            return checkMultiplicityAtMainEnd(id, o, r, localQuery, remoteQuery);
        } else {
            return checkMultiplicityAtOppositeEnd(id, o, r, localQuery, remoteQuery);
        }
   }

    /**
     * 
     * detect multiplicity violation of the value of the Reference r.
     * This is done by testing the application of InsertLink/RemoveLink/Delete instances.
     * The InsertLink/RemoveLink instances must directly apply to o.eGet(r)
     * and not the opposite reference.
     * 
     * @param o the model element that owns the reference value or null if this is a new object
     * @param r EReference defining the reference value
     * 
     *
     */
    private static Conflict checkMultiplicityAtMainEnd(String id, EObject o, EReference r, 
            DeltaQuery localQuery, DeltaQuery remoteQuery) {   
        Collection inserts = new Vector();
        inserts.addAll(localQuery
                .getInsertLinks(id, r));
        inserts.addAll(remoteQuery
                .getInsertLinks(id, r));
        Collection removes = new Vector();
        removes.addAll(localQuery
                .getRemoveLinks(id, r));
        removes.addAll(remoteQuery
                .getRemoveLinks(id, r));
        Collection deletes = new Vector();
        Set ids = getReferenceValue(o, r);  
        deletes.addAll(localQuery.getDeletes(ids));
        deletes.addAll(remoteQuery.getDeletes(ids));

        if( inserts.isEmpty() && removes.isEmpty() && deletes.isEmpty())  {
            return null;
        }
        
        for(Iterator it = inserts.iterator(); it.hasNext(); ) {
            InsertLink il = (InsertLink)it.next();
            ids.addAll(il.getRef());
        }       
        for(Iterator it = removes.iterator(); it.hasNext(); ) {
            RemoveLink rl = (RemoveLink)it.next();
            ids.removeAll(rl.getRef());
        }
        for(Iterator it = deletes.iterator(); it.hasNext(); ) {
            Delete d = (Delete)it.next();
            ids.remove(d.getId());
        }        
        return createMultiplicityConflict(id, o, r, ids, inserts, removes, deletes);
    }

    /**
     * 
     * Test the application of InsertLink and RemoveLink instances.
     * This is done by testing the application of InsertLink and RemoveLink instances.
     * The InsertLink/RemoveLink instances must apply to the opposite end of the Reference r.
     * I.e. the model element o must appear in the InsertLink/RemoveLink.getRef().
     * 
     * @param o the model element that owns the reference value, or null if this is a new object
     * @param r
     * 
     *
     */
    private static Conflict checkMultiplicityAtOppositeEnd(String id, EObject o, EReference r, 
            DeltaQuery localQuery, DeltaQuery remoteQuery) {     
        EReference op = r.getEOpposite();
        Collection inserts = new Vector(localQuery
                .getOppositeInsertLinks(id, op));
        inserts.addAll(remoteQuery
                .getOppositeInsertLinks(id, op));
        Collection removes = new Vector(localQuery
                .getOppositeRemoveLinks(id, op));
        removes.addAll(remoteQuery
                .getOppositeRemoveLinks(id, op)); 
        Collection deletes = new Vector();        
        Set ids = getReferenceValue(o, r);  
        deletes.addAll(localQuery.getDeletes(ids));
        deletes.addAll(remoteQuery.getDeletes(ids));
        
        
        if( inserts.isEmpty() && removes.isEmpty() && deletes.isEmpty())  {
            return null;
        }
        
        for(Iterator it = inserts.iterator(); it.hasNext(); ) {
            InsertLink il = (InsertLink)it.next();
            NodeLevelSubDelta modify = (NodeLevelSubDelta) il.eContainer().eContainer();
            ids.add(modify.getId());
        }       
        for(Iterator it = removes.iterator(); it.hasNext(); ) {
            RemoveLink rl = (RemoveLink)it.next();
            NodeLevelSubDelta modify = (NodeLevelSubDelta) rl.eContainer().eContainer();            
            ids.remove(modify.getId());
        }
        for(Iterator it = deletes.iterator(); it.hasNext(); ) {
            Delete d = (Delete)it.next();
            ids.remove(d.getId());
        }                
        return createMultiplicityConflict(id, o, r, ids, inserts, removes, deletes);
    }
    
    
    private static Conflict createMultiplicityConflict(String id, EObject o, EReference r, Set resultReferenceValue, 
            Collection inserts, Collection removes, Collection deletes) {
        if(!isMultiplicityViolated(r, resultReferenceValue)) {
            return null;
        }
        Conflict conf = fac.createConflict();
        conf.setType(ConflictDetection.MUL_CONFLICT);
        conf.setDetail( SimpleModelLabelProvider.getObjectTypeAndNameOrId(o, id) +" " 
                + r.getName()+" value=" +resultReferenceValue);
        List subdeltas = new Vector();
        subdeltas.addAll(deletes);
        subdeltas.addAll(removes);
        subdeltas.addAll(inserts);
        for(Iterator it= subdeltas.iterator(); it.hasNext();) {
            SubDelta elem = (SubDelta) it.next();
            if(DeltaQuery.isLocalSubDelta(elem)) {
                conf.getLocal().add(elem);
            } else {
                conf.getRemote().add(elem);
            }
        }      
        return conf;
    }
    
    
    public static Set getReferenceValue(EObject o, EReference r) {
        if(o==null) return new HashSet();
        Object v = o.eGet(r);
        Set ids = null;
        if (v instanceof Collection) {
            return new HashSet(MergeHelper.getIDList((Collection) v));
        } else {
            ids = new HashSet();
            if(v!=null) {
                ids.add(MergeHelper.getURIFragment((EObject) v));
            }
            return ids;
        }     
    }
    
    
    public static boolean isMultiplicityViolated(EReference r, Set ids) {
        int s = ids.size();
        if (s < r.getLowerBound())
            return true;
        if (r.getUpperBound() > 0 && s > r.getUpperBound())
            return true;
        return false;          
    }

    public static boolean needMultiplicityCheck(EReference r) {
        if (r.getLowerBound() != 0)
            return true;
        if (r.getUpperBound() != -1)
            return true;
        return false;
    }

    /**
     * 
     * Solve the problem when two elements are concurrently generated with 
     * same IDs.
     * One of the redundant ID must be changed.
     */ 
    public static void changeConflictingIDs(MergeData mergeData, DeltaQuery localQuery, DeltaQuery remoteQuery) {
        Collection c1 = localQuery.getCreatedNodeIDs();
        Collection c2 = remoteQuery.getCreatedNodeIDs();
        Collection redundant = new Vector(c1);
        redundant.retainAll(c2);   
        Collection idSet = new HashSet(mergeData.getRemoteVariant().getIdTable().getIdSet());
        idSet.addAll(mergeData.getLocalVariant().getIdTable().getIdSet());
        for(Iterator it = redundant.iterator(); it.hasNext(); ) {
            String oldID = (String) it.next();
            String newID = mergeData.getLocalVariant().getIdTable().changeId(oldID, idSet);
            idSet.add(newID);
            logger.debug("change ID " + oldID + "->" +newID);
            localQuery.changeID(oldID, newID);                
        }
    }

}
