/*
 * $RCSfile: DefaultMarshaler.java,v $
 * $Date: 2006/06/29 16:46:42 $
 * $Revision: 1.16 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * ModelMarshaller.java
 * 
 * @author Prawee Sriplakich
 * @version $Revision: 1.16 $ $Date: 2006/06/29 16:46:42 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import javax.xml.soap.Detail;
import javax.xml.soap.DetailEntry;
import javax.xml.soap.Name;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPFault;

import org.apache.axis.message.PrefixedQName;
import org.apache.log4j.Logger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.DescriptionUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.*;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.emf_xmi.DefaultModelSerializer;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelTypeMismatchException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.TypeMismatchException;
import org.eclipse.mddi.modelbus.description.abstract_.AbstractFactory;
import org.eclipse.mddi.modelbus.description.abstract_.EnumerationType;
import org.eclipse.mddi.modelbus.description.abstract_.Error;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.abstract_.PrimitiveType;
import org.eclipse.mddi.modelbus.description.abstract_.Type;
import org.eclipse.mddi.modelbus.description.abstract_.impl.AbstractFactoryImpl;
import org.w3c.dom.Element;

import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

/**
 * @author P. Sriplakich, Andrey Sadovykh (LIP6)
 * 
 *  
 */
public class DefaultMarshaler implements Marshaler {

    public static final String ERROR_INFO_TAG = "ErrorInfo";

    public static final String ERROR_FULL_TAG = "modelbus:ModelingServiceError";

    public static final String ERROR_TAG = "ModelingServiceError";

    public static final String SCOPE_TAG = "modelbus:Scope";
    
    
    /*
     * ModelBus Primitive Types
     */
    
    public static final String MB_STRING = "string";
    public static final String MB_INTEGER = "integer";
    public static final String MB_BOOLEAN = "boolean";
    public static final String MB_DOUBLE = "double";
    public static final String MB_BINARY = "binary";
    
   

    private static Logger logger = Logger.getLogger(DefaultMarshaler.class);

    SerializedXmiDocument[] docs;

    DeserializedModel[] models;

    ModelSerializer serializer = new DefaultModelSerializer();

    public DefaultMarshaler(Properties p) {
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.ParameterMarshaler#marshall(org.eclipse.mddi.modelbus.description.abstract_.Parameter,
     *      java.lang.Object, javax.xml.soap.SOAPElement)
     */
    public void marshal(Parameter p, Object parameterValue, SOAPElement topElem)
            throws Exception {
        //if parameter value is null, do nothing
        if (parameterValue == null) {
            return;
        }
        SOAPElement paramSoapElem = topElem.addChildElement(p.getName());
        if (p.getUpper() != 1) {
            if (!(parameterValue instanceof Object[])) {
                throw new SOAPException(
                        "Marshall error, invalid parameter multiplicity - "
                                + p.getName());
            }
            Object[] values = (Object[]) parameterValue;
            for (int i = 0; i < values.length; i++) {
                Object singleValue = values[i];
                SOAPElement itemElem = paramSoapElem
                        .addChildElement("modelbus:item");
                marshallSingleValue(p, singleValue, itemElem);
            }
        } else {
            marshallSingleValue(p, parameterValue, paramSoapElem);
        }
    }

    void marshallSingleValue(Parameter p, Object value, SOAPElement elem)
            throws Exception {
        Type t = p.getType();
        if (t instanceof ModelType) {
            ModelElementReference[] refs = serializer.getReferences(docs, p,
                    value);
            for (int i = 0; i < refs.length; i++) {
                Element elem1 = elem.addChildElement("modelbus:Ref");
                elem1.setAttribute("uri", refs[i].getUri());
                elem1.setAttribute("fragment", refs[i].getRef());
            }
            setEncodingStyle(p, elem);

            // logger.debug("PRAM: " +param.getName() );
            // logger.debug(s);

        }

        else if (t instanceof EnumerationType) {
            String s = value.toString();
            setEncodingStyle(p, elem);
            elem.addTextNode(s);
        } else if (t.getName().equalsIgnoreCase("binary")) {
            //check type of value
            if (!(value instanceof InputStream))
                throw new TypeMismatchException(
                        "Binary parameters should be of java.io.InputStream type");
            //marshal binary data from an Output stream
            BASE64Encoder encoder = new BASE64Encoder();
            InputStream is = (InputStream) value;
            byte[] buffer = new byte[is.available()];
            is.read(buffer);
            String s = encoder.encode(buffer);
            setEncodingStyle(p, elem);
            elem.addTextNode(s);

        } else {
            String s = value.toString();
            setEncodingStyle(p, elem);
            elem.addTextNode(s);
        }
    }

    public static void setEncodingStyle(Parameter p, SOAPElement elem)
            throws SOAPException {
        Type t = p.getType();
        String typeName = t.getName();
        if (t instanceof PrimitiveType) {
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                elem.setAttribute("encoding", "xsd:string");
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                elem.setAttribute("encoding", "xsd:integer");
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                elem.setAttribute("encoding", "xsd:float");
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                elem.setAttribute("encoding", "xsd:boolean");
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                elem.setAttribute("encoding", "xsd:base64Binary");
            } else {
                // this is a user-defined primitive type
                elem.setEncodingStyle(t.getName());
            }
        } else if (t instanceof ModelType) {
            elem.setAttribute("encoding", "modelbus:ModelEncoding");
        } else if (t instanceof EnumerationType) {
            elem.setAttribute("encoding", "modelbus:Enumeration");
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.ParameterMarshaler#unmarshall(org.eclipse.mddi.modelbus.description.abstract_.Parameter,
     *      javax.xml.soap.SOAPElement)
     */
    public Object unmarshal(Parameter p, SOAPElement topElem) throws Exception {
        SOAPElement elem = SoapUtil.getChildByLocalName(topElem, p.getName());
        // if the element is not found in SOAP message, then the parameter value
        // is null
        if (elem == null) {
            return null;
        }
        if (p.getUpper() != 1) {
            List values = new Vector();
            Iterator it = elem.getChildElements();
            while (it.hasNext()) {
                Object o = it.next();
                if (o instanceof SOAPElement) {
                    SOAPElement subElem = (SOAPElement) o;
                    Object v = unmarshallSingleValue(p, subElem);
                    values.add(v);
                }
            }
            Object[] array = getArrayForMultiValueParameter(p, values.size());
            return values.toArray(array);
        } else {
            return unmarshallSingleValue(p, elem);
        }
    }

    /**
     * Creates array for storing parameter values
     * 
     * @param p
     * @param size
     * @return
     */
    static Object[] getArrayForMultiValueParameter(Parameter p, int size) {
        Type t = p.getType();
        if (t instanceof PrimitiveType) {
            String typeName = t.getName();
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                return new String[size];
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                return new Integer[size];
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                return new Double[size];
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                return new Boolean[size];
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                return new ByteArrayInputStream[size];
            } else {
                return null;
            }
        } else if (t instanceof EnumerationType) {
            return new String[size];
        } else if (t instanceof ModelType) {
            return new Collection[size];
        }
        return null;
    }

    Object unmarshallSingleValue(Parameter p, SOAPElement elem)
            throws Exception {
        Type t = p.getType();
        if (t instanceof ModelType) {
            Vector v = new Vector();
            for (Iterator it = elem.getChildElements(); it.hasNext();) {
                SOAPElement elem1 = (SOAPElement) it.next();
                ModelElementReference r = new ModelElementReference();
                v.add(r);
                r.setUri(elem1.getAttribute("uri"));
                r.setRef(elem1.getAttribute("fragment"));
            }
            ModelElementReference[] refs = (ModelElementReference[]) v
                    .toArray(new ModelElementReference[v.size()]);
            Object o = serializer.dereference(models, p, refs);
            return o;
        } else if (t instanceof EnumerationType) {
            String value = elem.getValue();
            return value;
        } else {
            String typeName = t.getName();
            String value = elem.getValue();
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                return value;
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                return new Integer(Integer.parseInt(value));
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                return new Double(Double.parseDouble(value));
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                return Boolean.valueOf(value);
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                BASE64Decoder decoder = new BASE64Decoder();
                byte[] buffer = decoder.decodeBuffer(value);
                return new ByteArrayInputStream(buffer);
            }
        }
        throw new Exception("Deserialize error, unknown Type " + t);
    }

    public void marshalModelingServiceError(ModelingServiceError err,
            SOAPFault f) throws SOAPException {
        Name n1 = new PrefixedQName("http://www.eclipse.org/mddi/modelbus",
                ERROR_INFO_TAG, "modelbus");

        f.setFaultCode(ERROR_TAG);
        f.setFaultString(err.getName());
        if (err.getInfo() != null) {
            Detail detail = f.addDetail();
            DetailEntry entry = detail.addDetailEntry(n1);
            if (err.getErrorType() == null) {

                entry.addTextNode(err.getInfo().toString());
            } else {
                try {
                    marshalModelingServiceError(err.getErrorType(), err
                            .getInfo(), entry);
                } catch (Exception e) {
                    throw new SOAPException(e);
                }
            }

        }
    }

    public void marshalModelingServiceError(Error errorType, Object info,
            SOAPElement element) throws SOAPException, Exception {

        AbstractFactory factory = new AbstractFactoryImpl();
        Parameter parameter = factory.createParameter();
        parameter.setUpper(errorType.getUpper());
        parameter.setName(errorType.getName());
        parameter.setType(errorType.getType());

        Parameter[] params = new Parameter[1];
        params[0] = parameter;

        Object[] values = new Object[1];
        values[0] = info;

        marshal(params, values, element);
    }

    public Object unmarshalModelingServiceError(Error errorType,
            SOAPElement element) throws SOAPException, Exception {
        AbstractFactory factory = new AbstractFactoryImpl();
        Parameter parameter = factory.createParameter();
        parameter.setUpper(errorType.getUpper());
        parameter.setName(errorType.getName());
        parameter.setType(errorType.getType());

        Parameter[] params = new Parameter[1];
        params[0] = parameter;

        Object[] values = unmarshal(params, element);
        return values[0];
    }

    /**
     * Identifies whether this SOAP fault contains ModelingServiceError or other
     * kinds of errors.
     * 
     * @param f
     * @return
     *  
     */
    public static boolean isModelingServiceError(SOAPFault f) {
        String fcode = f.getFaultCode();
        return (fcode.equals(ERROR_FULL_TAG) || fcode.equals(ERROR_TAG));
    }

    public static String getModelingServiceErrorName(SOAPFault f) {

        return f.getFaultString();
    }

    public ModelingServiceError unmarshalModelingServiceError(Error errorType,
            SOAPFault f) throws SOAPException {
        String errName = f.getFaultString();

        Name n1 = new PrefixedQName("http://www.eclipse.org/mddi/modelbus",
                ERROR_INFO_TAG, "modelbus");
        Iterator it = f.getChildElements();
        while (it.hasNext()) {
            SOAPElement el = (SOAPElement) it.next();
            Iterator it2 = el.getChildElements(n1);
            while (it2.hasNext()) {
                SOAPElement infoEl = (SOAPElement) it2.next();
                if (errorType == null) {
                    String info = infoEl.getValue();
                    return new ModelingServiceError(errName, info);
                } else {
                    Object info;
                    try {
                        info = unmarshalModelingServiceError(errorType, infoEl);
                    } catch (Exception e) {
                        throw new SOAPException(e);
                    }
                    return new ModelingServiceError(errorType, info);

                }
            }
        }
        return new ModelingServiceError(errName, null);
    }

    /**
     * Set namespace declarations used for defining encodings
     * 
     * @param body
     * @throws SOAPException
     */
    public static void setNamespaces(SOAPBody body) throws SOAPException {
        body.addNamespaceDeclaration("xsd", "http://www.w3.org/1999/XMLSchema");
        body.addNamespaceDeclaration("xsi",
                "http://www.w3.org/1999/XMLSchema-instance");
        body.addNamespaceDeclaration("modelbus",
                "http://www.eclipse.org/mddi/modelbus");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.Marshaler#marshal(org.eclipse.mddi.modelbus.description.abstract_.Parameter[],
     *      java.lang.Object[], javax.xml.soap.SOAPElement)
     */
    public void marshal(Parameter[] params, Object[] parameterValues,
            SOAPElement topElem) throws Exception {
        if (params == null)
            return;
        if (parameterValues.length != params.length) {
            throw new ModelTypeMismatchException(
                    "Bad number of parameters, got " + params.length
                            + ", expected parameters: "
                            + DescriptionUtil.getParameterNameList(params));
        }
        SOAPElement scopeElement = topElem.addChildElement(SCOPE_TAG);
        docs = serializer.serialize(params, parameterValues);
        for (int i = 0; i < docs.length; i++) {
            SOAPElement elem1 = scopeElement
                    .addChildElement("modelbus:Resource");
            elem1.setAttribute("uri", docs[i].getUri());
            elem1.addTextNode(docs[i].getXmi());
        }
        for (int i = 0; i < params.length; i++) {
            marshal(params[i], parameterValues[i], topElem);
        }
        docs = null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.Marshaler#unmarshal(org.eclipse.mddi.modelbus.description.abstract_.Parameter[],
     *      javax.xml.soap.SOAPElement)
     */
    public Object[] unmarshal(Parameter[] params, SOAPElement topElem)
            throws Exception {
        if (params == null)
            return null;

        SOAPElement scopeElement = SoapUtil.getChildByFullName(topElem,
                SCOPE_TAG);
        Vector v = new Vector();
        for (Iterator it = scopeElement.getChildElements(); it.hasNext();) {
            SOAPElement resElem = (SOAPElement) it.next();
            SerializedXmiDocument doc = new SerializedXmiDocument();
            doc.setUri(resElem.getAttribute("uri"));
            doc.setXmi(resElem.getValue());
            v.add(doc);
        }
        docs = (SerializedXmiDocument[]) v.toArray(new SerializedXmiDocument[v
                .size()]);
        models = serializer.deserialize(docs);
        Object[] unmarshaledParams = new Object[params.length];
        for (int i = 0; i < params.length; i++) {
            unmarshaledParams[i] = unmarshal(params[i], topElem);
        }
        docs = null;
        models = null;
        return unmarshaledParams;
    }

    /**
     * @return Returns the serializer.
     */
    public ModelSerializer getSerializer() {
        return serializer;
    }

    /**
     * @param serializer
     *            The serializer to set.
     */
    public void setSerializer(ModelSerializer serializer) {
        this.serializer = serializer;
    }

}
