/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

package org.eclipse.mddi.modelbus.adapter.user.consumer;

import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;

/**
 * 
 * This is an interface offerred by Adapter component. It allows the consumer
 * tool to consume modeling services provided by other Tools. It is used by
 * consumer tools that want to invoke a modeling service.
 * 
 * <p>
 * Parameter values of the modeling services are represented as Java objects. We
 * use the following mappings to map ModelBys types to Java types
 * 
 * <p>
 * <b>ModelBus types:</b> PrimitiveType (String, Boolean, Integer, Double)
 * <p>
 * <b>Java types:<b> java.lang.String, java.lang.Boolean, java.lang.Integer,
 * java.lang.Double <br>
 * <p>
 * <b>ModelBus types:</b> EnumerationType
 * <p>
 * <b>Java types:<b> java.lang.String. The string represents the literal value
 * of the enumeration <br>
 * <p>
 * <b>ModelBus types:</b> ModelType
 * <p>
 * <b>Java types:<b> java.lang.Collection The collection contains instances of
 * org.eclipse.emf.ecore.EObject (see EMF [REF 6]).
 * <p>
 * The collection contains all model elements (EObject) in the model, including
 * the sub model elements. For example, a UML model contains packages and
 * classes inside the packages. The collection representing this model must
 * contain all packages and all classes in this model. In the case of inout
 * parameters, the members of the collections can be modified when passed to the
 * Modeling Service. <br>
 * <p>
 * <b>ModelBus types:</b> Multi-valued Parameter. I.e., A parameter having
 * upper property not equal to one (-1 or more than one).
 * <p>
 * <b>Java types:<b> Array of the Java types representing elements of the
 * multi-valued parameters.
 * <p>
 * Multi-valued primitive types: String[], Boolean[], Integer[], Double[]
 * <p>
 * Multi-valued enumeration types: String[]
 * <p>
 * Multi-valued model types: Collection[], where each element in the Collection
 * is instance of EObject. <br>
 * <p>
 * <b>Null value representation</b>
 * <p>
 * Single-valued parameters ([upper..lower] = [0..1] or [1..1]). In the case of
 * [0..1], null parameter value is represented by null Java value. In the case
 * of [1..1], the parameter value can not be null.
 * <p>
 * Multi-valued parameters. The null parameter value is represented by an array
 * of size zero.
 * 
 * @author Prawee Sriplakich (LIP6)
 * 
 */
public interface GenericConsumer {

    /**
     * This operation is used for invoking a modeling service in the synchronous
     * style. The adapter has the charge of to communicate with the Registry for
     * finding one provider tool that does provide access to the modeling
     * service.
     * 
     * <p>
     * If several provider tools are accessible, then the adapter chooses one of
     * them (the first in the list returned by the Registry).
     * 
     * @param serviceName
     *            it can be either qualified name or not. qualified name =
     *            [tool] . [Interface] . [service] or [Interface] . [service]
     * 
     * Example: Objecteering.ModelEdition.loadModel or ModelEdition.loadModel
     * 
     * @param parameterValues
     * @return
     * @throws ModelingServiceError
     * @throws ServiceUnknownException
     * @throws NoToolAvailableException
     * @throws ModelTypeMismatchException
     * @throws ModelBusCommunicationException
     * @throws Exception 
     */
    Object[] consume(String serviceName, Object[] parameterValues)
            throws ModelingServiceError, ServiceUnknownException,
            NoToolAvailableException, ModelTypeMismatchException,
            ModelBusCommunicationException;

    /**
     * 
     * This operation is used for invoking a modeling service in the synchronous
     * style using session mechanisme. When using this method, the adapter
     * communicates with the Registry to find a provider tool that does provide
     * access to the modeling service.
     * 
     * <p>
     * If several provider tools are accessible, then the adapter chooses one of
     * them (the first in the list returned by the Registry).
     * 
     * @param serviceName
     *            it can be either qualified name or not. qualified name =
     *            [tool] . [Interface] . [service] or [Interface] . [service]
     * 
     * Example: Objecteering.ModelEdition.loadModel or ModelEdition.loadModel
     * 
     * @param parameterValues
     * @param sessionId
     *            is specified to identify the session in the frame of which,
     *            the invocation is performed
     * @return
     * @throws ModelingServiceError
     * @throws ServiceUnknownException
     * @throws NoToolAvailableException
     * @throws ModelTypeMismatchException
     * @throws ModelBusCommunicationException
     * 
     */
    Object[] consume(String serviceName, Object[] parameterValues,
            String sessionId) throws ModelingServiceError,
            ServiceUnknownException, NoToolAvailableException,
            ModelTypeMismatchException, ModelBusCommunicationException;

    /**
     * 
     * This operation is used when full control of the sessions is required.
     * This operations permits to create a new session
     * 
     * @param serviceName
     * @return
     * @throws ModelBusCommunicationException
     * @throws ServiceUnknownException
     * 
     */
    public String newSession(String serviceName)
            throws ModelBusCommunicationException, ServiceUnknownException;

    /**
     * 
     * This operation is used to terminate a given session
     * 
     * @param sessionId
     * @throws ModelBusCommunicationException
     * 
     */
    public void endSession(String sessionId)
            throws ModelBusCommunicationException;

    /**
     * 
     * getOpenSessions - returns all open sessions
     * 
     * @param serviceName
     * @return
     * 
     */

    public String[] getAllOpenSessions();

    /**
     * 
     * getOpenSessions - returns all open sessions for the given serviceName
     * 
     * @param serviceName
     *            it can be either qualified name or not. qualified name =
     *            [tool] . [Interface] . [service] or [Interface] . [service]
     *            or even [service]
     * 
     * Example: Objecteering.ModelEdition.loadModel or ModelEdition.loadModel
     * 
     * @return session ids
     * 
     */
    public String[] getOpenSessions(String serviceName);

    /**
     * This operation is used for invoking a modeling service in the
     * asynchronous style.When using this method, the adapter
     * communicates with the Registry to find a provider tool that does provide
     * access to the modeling service.
     * 
     * <p>
     * If several provider tools are accessible, then the adapter chooses one of
     * them (the first in the list returned by the Registry).
     * 
     * @param serviceName
     *            it can be either qualified name or not. qualified name =
     *            [tool] . [Interface] . [service] or [Interface] . [service]
     * 
     * Example: Objecteering.ModelEdition.loadModel or ModelEdition.loadModel
     * @param parameterValues
     * @throws ServiceUnknownException
     * @throws NoToolAvailableException
     * @throws ModelTypeMismatchException
     * @throws ModelBusCommunicationException
     */
    String consumeAsync(String serviceName, Object[] parameterValues)
            throws ServiceUnknownException, NoToolAvailableException,
            ModelTypeMismatchException, ModelBusCommunicationException;

    /**
     * This operation is used for invoking a modeling service in the
     * asynchronous style using session mechanisme. When using this method, the adapter
     * communicates with the Registry to find a provider tool that does provide
     * access to the modeling service.
     * 
     * <p>
     * If several provider tools are accessible, then the adapter chooses one of
     * them (the first in the list returned by the Registry).
     * 
     * @param serviceName
     *            it can be either qualified name or not. qualified name =
     *            [tool] . [Interface] . [service] or [Interface] . [service]
     * 
     * Example: Objecteering.ModelEdition.loadModel or ModelEdition.loadModel
     * 
     * @param parameterValues
     * @param sessionId
     *            is specified to identify the session in the frame of which,
     *            the invocation is performed
     * @return
     * @throws ModelingServiceError
     * @throws ServiceUnknownException
     * @throws NoToolAvailableException
     * @throws ModelTypeMismatchException
     * @throws ModelBusCommunicationException
     * 
     */
    String consumeAsync(String serviceName,Object[] parameterValues, String
     sessionId) throws ServiceUnknownException, NoToolAvailableException,
     ModelTypeMismatchException, ModelBusCommunicationException;
    /**
     * This operation is used for asking whether the result of the invocation is
     * ready. This operation can be used in both sessionless and sessionful
     * styles.
     * 
     * <p>
     * If two asynchronous calls have been transmitted for a same modeling
     * service then this operation returns true if one of them has returned.
     * 
     * @param serviceName
     * @return
     * @throws ModelBusCommunicationException
     */
    boolean isResultReady(String serviceName)
            throws ModelBusCommunicationException;

    /**
     * This operation is used for getting the result of an asynchronous
     * invocation.
     * 
     * @param serviceName
     * @return the result of modeling service invocation
     * @throws ModelingServiceError
     *             errors produced by the provider tool
     */
    Object[] getResult(String serviceName) throws ModelingServiceError;

}
