/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * Created on 16 juin 2005
 *
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation;

import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Level;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EContentsEList;

/**
 * @author Prawee Sriplakich (LIP6)
 * 
 * 
 */
public class ModelUtil {



    /**
     * Returns a set of elements that are direct instances of the specified
     * metaclass. The sybclass relation is not considerred.
     * 
     * @param emfobjs
     *            a collection of EObjects
     * @param typeName
     *            the name of the metaclass
     * @return
     * 
     * @see Add references
     * 
     * TODO Add description
     */
    public static Collection findElementByType(Collection emfobjs,
            String typeName) {
        Collection set = new HashSet();
        Iterator it = emfobjs.iterator();
        while (it.hasNext()) {
            EObject o = (EObject) it.next();
            if (o.eClass().getName().equals(typeName)) {
                set.add(o);
            }
        }
        return set;
    }

    public static Collection findElementByType_considerSubtypes(
            Collection emfobjs, EClass type) {
        Collection set = new HashSet();
        Iterator it = emfobjs.iterator();
        while (it.hasNext()) {
            EObject o = (EObject) it.next();
            if (isInstanceOf(o, type)) {
                set.add(o);
            }
        }
        return set;
    }

    public static boolean isInstanceOf(EObject o, EClass c) {
        EClass subc = o.eClass();
        if (subc.equals(c))
            return true;
        if (subc.getEAllSuperTypes().contains(c))
            return true;
        return false;
    }

    public static EObject findElementByName(Collection model, String name) {
        for (Iterator it = model.iterator(); it.hasNext();) {
            EObject o = (EObject) it.next();
            EStructuralFeature f = o.eClass().getEStructuralFeature("name");
            // findStructuralFeature(o.eClass(), "name");
            if (f != null) {
                Object _name = o.eGet(f, false);
                if (_name != null && _name.equals(name)) {
                    return o;
                }
            }
        }
        return null;
    }

    /**
     * 
     * findStructuralFeature
     * 
     * @param c
     * @param featureName
     * @return
     * @deprecated use EClass.getEStructuralFeature(String name) instead
     * 
     */
    public static EStructuralFeature findStructuralFeature(EClass c,
            String featureName) {
        EStructuralFeature f = null;
        for (Iterator it = c.getEAllStructuralFeatures().iterator(); it
                .hasNext();) {
            f = (EStructuralFeature) it.next();
            if (f.getName().equals(featureName)) {
                return f;
            }
        }
        return null;
    }

    /**
     * Returns a set containing elements in <code>eObjects</code> 
     * and their
     * subelements.
     * 
     * 
     * @param eObjects
     * @return
     */
    public static List flattenHirachicalCollection(Collection eObjects) {
        List target = new Vector();
        for(Iterator it = eObjects.iterator(); it.hasNext(); ) {
            EObject o = (EObject) it.next();
            target.add(o);
            Iterator it2 = o.eAllContents();
            while (it2.hasNext()) {
                EObject o2 = (EObject) it2.next();
                target.add(o2);
            }
        }
        return target;
    }

    public static Collection getTopElementsFromResources(Resource[] resources) {
        Collection result = new Vector();
        for (int i = 0; i < resources.length; i++) {
            Resource r = resources[i];
            result.addAll(r.getContents());
        }
        return result;
    }

    /**
     * Find top-level elements in the <code>source</code> and put them in the
     * return collection.
     * 
     * @param model
     * @return
     */
    public static Collection getTopElements(Collection model) {
        Collection target = new HashSet();
        Iterator it = model.iterator();
        while (it.hasNext()) {
            EObject o = (EObject) it.next();
            EObject topElem = getTopElement(o);
            if (!target.contains(topElem)) {
                target.add(topElem);
            }
        }
        return target;
    }

    public static EObject getTopElement(EObject o) {
        EObject topElem = o;
        while (topElem.eContainer() != null) {
            topElem = topElem.eContainer();
        }
        return topElem;
    }

    /**
     * Check whether this collection contains only EObjects
     * 
     * @param c
     * @return false if there is one or more instances that are not EObjects
     *         true if all instances are EObjects or the collection is empty
     */
    public static boolean isEObjectCollection(Collection c) {
        Iterator it = c.iterator();
        while (it.hasNext()) {
            Object o = it.next();
            if (!(o instanceof EObject)) {
                return false;
            }
        }
        return true;
    }

    public static Object getFeatureValue(EObject o, String featureName) {
        EStructuralFeature f = o.eClass().getEStructuralFeature(featureName);
        // findStructuralFeature(o.eClass(), featureName);
        if (f == null) {
            return null;
        }
        return o.eGet(f, false);
    }

    public static Object createFromString(String primitiveType, String value) {
        if (primitiveType.equalsIgnoreCase("string")) {
            return value;
        } else if (primitiveType.equalsIgnoreCase("boolean")) {
            return Boolean.valueOf(value);
        } else if (primitiveType.equalsIgnoreCase("integer")) {
            return new Integer(Integer.parseInt(value));
        } else if (primitiveType.equalsIgnoreCase("double")) {
            return new Double(Double.parseDouble(value));
        }
        System.err.println("unknown primitive " + primitiveType);
        return null;
    }

    public static Collection getTopElemsInFragment(Collection fragment) {
        Collection result = new HashSet();
        for (Iterator it = fragment.iterator(); it.hasNext();) {
            EObject e = ModelUtil.getTopElemInFragment((EObject) it.next(),
                    fragment);
            result.add(e);
        }
        return result;
    }

    /**
     * Returns the top-most element that is included in this model.
     */
    public static EObject getTopElemInFragment(EObject currentElem,
            Collection fragment) {
        EObject topElem = currentElem;
        while (topElem.eContainer() != null
                && fragment.contains(topElem.eContainer())) {
            topElem = topElem.eContainer();
        }
        return topElem;
    }

    /**
     * 
     * find an Adapter corresponding to a specified type
     * 
     * @param target
     * @param type
     * @return
     */
    public static Adapter findAdapter(Notifier target, Object type) {
        for (Iterator adapters = target.eAdapters().iterator(); adapters
                .hasNext();) {
            Adapter adapter = (Adapter) adapters.next();
            if (adapter.isAdapterForType(type)) {
                return adapter;
            }
        }
        return null;
    }

    /**
     * Add resources that are crossed-referenced by the resources (specified as
     * a parameter) to the specified set (specified as the other parameter).
     * <br>
     * 
     * option "ignoredUriPrefixSet": is a set of URI prefixes to be ignored. The
     * cross-referenced resources that have those prefixes will NOT be included
     * in the result. <br>
     * 
     * @param resources
     * @param ignoredUriPrefixSet
     *            is a set of string
     * 
     * 
     */
    public static void addCrossReferencedResources(Set resources,
            Collection ignoredUriPrefixSet) {
        // the queue contains model elements to be examined.
        // add all model elements to the queue.
        List queue = new Vector();
        for (Iterator it = resources.iterator(); it.hasNext();) {
            Resource r = (Resource) it.next();
            for (Iterator it2 = r.getAllContents(); it2.hasNext();) {
                EObject containedObject = (EObject) it2.next();
                queue.add(containedObject);
            }
        }
        // examine elements in the queue
        while (!queue.isEmpty()) {
            EObject o = (EObject) queue.remove(0);
            for (Iterator it = o.eCrossReferences().iterator(); it.hasNext();) {
                EObject referencedObject = (EObject) it.next();

                if (referencedObject instanceof EFactory) {
                    // the EFactory need not be serialized.
                    continue;
                }
                if (referencedObject.eIsProxy()) {
                    // the Proxy are the object that can not be resolved.
                    // therefore, it does not need to be serialized.
                    continue;
                }

                Resource cr = referencedObject.eResource();
                if (cr != null && !resources.contains(cr)) {

                    if (!ModelUtil.uriBeginsWith(cr.getURI(),
                            ignoredUriPrefixSet)) {
                        resources.add(cr);
                        // we need to examine the cross-ref resource from this
                        // resouce too.
                        // Therefore, we add all elements in this resource in
                        // the queue.
                        for (Iterator it2 = cr.getAllContents(); it2.hasNext();) {
                            queue.add(it2.next());
                        }
                    }
                } else if (cr == null) {
                    // find whether the default resource already exists.
                    cr = findOrCreateDefaultResource(resources);
                   
                    EObject top = getTopElement(referencedObject);
                    cr.getContents().add(top);
                    resources.add(cr);
                    // we need to examine the cross-ref resource from this
                    // resouce too.
                    // Therefore, we add all elements in this resource in
                    // the queue.
                    for (Iterator it2 = top.eAllContents(); it2.hasNext();) {
                        queue.add(it2.next());
                    }
                }
            } // end for
        }
    }

    /**
     * 
     * Obtain a set of resources that contained the specified objects. For
     * objects has no resource, a default resource is created for containing
     * them. <br>
     * <br>
     * option "skipCrossReferences": It true, the resources that are referenced
     * by the specified elements will NOT be included in the result. <br>
     * <br>
     * option "ignoredUriPrefixSet": is a set of URI prefixes to be ignored. The
     * cross-referenced resources that have those prefixes will NOT be included
     * in the result. <br>
     * 
     * @param emfobjs
     * @param ignoredUriPrefixSet
     *            (set of String)
     * @return a set of container resources
     * 
     * 
     */
    public static List<Resource> getResourcesFromObjects(Collection emfobjs,
            boolean skipCrossReferences, Collection ignoredUriPrefixSet) {

        Set<Resource>  resources = new HashSet();

        Collection orphanTopElements = new HashSet();
        for (Iterator it = emfobjs.iterator(); it.hasNext();) {
            EObject o = (EObject) it.next();
            Resource r = o.eResource();
            if (r == null) {
                // add this object to resource;
                EObject top = getTopElement(o);
                orphanTopElements.add(top);
            } else {
                resources.add(r);
            }
        }
        // If there are orphan elements, a new resource will be created to
        // contain those elements.
        if (!orphanTopElements.isEmpty()) {
            Resource defaultResource = findOrCreateDefaultResource(resources);
            defaultResource.getContents().addAll(orphanTopElements);
            resources.add(defaultResource);
        }
        // if option skipCrossReferences is false, the cross-referenced
        // resources are searched
        if (!skipCrossReferences) {
            addCrossReferencedResources(resources, ignoredUriPrefixSet);
        }
        return new Vector<Resource>(resources);
    }

    private static Resource findOrCreateDefaultResource(Collection resources) {
        // find whether the default resource already exists.
        for (Iterator it = resources.iterator(); it.hasNext();) {
            Resource r = (Resource) it.next();
            if (ModelBusResourceSet.MB_DEFAULT_RES_URI.equals(r.getURI())) {
                return r;
            }
        }
        // create default resource

        ResourceSet rSet = new ModelBusResourceSet();
        Resource defaultResource = rSet
                .createResource(ModelBusResourceSet.MB_DEFAULT_RES_URI);
        return defaultResource;
    }

    /**
     * Ensure that all objects reachable from this resource set is contained by
     * any resources.
     * Otherwise, add the 'orphan' objects 
     * to one of the resources 
     * in the ResourceSet
     * 
     * @param rs
     */
    public static void ensureContainment(ResourceSet rs) {
        for (int i = 0; i < rs.getResources().size(); i++) {
            Resource r = (Resource) rs.getResources().get(i);
            ensureContainment(r);
        }
    }

    /**
     * Ensure that all objects in this resource is contained by any resources.
     * Otherwise, add the 'orphan' objects 
     * to this resource.
     * 
     * @param r
     */
    public static void ensureContainment(Resource r) {
        // do not use iterator to avoid ConcurrentModificationException
        for (int j = 0; j < r.getContents().size(); j++) {
            EObject o = (EObject) r.getContents().get(j);
            ensureContainmentForAll(r, o);
        }
    }

    /**
     * Ensure that all objects reachable from
     * the specified object is contained by any resources.
     * Otherwise, add the 'orphan' objects 
     * to this resource.
     * 
     * @param r
     * @param object
     */
    private static void ensureContainmentForAll(Resource r, EObject o) {
        ensureContainmentInternal(r, o);
        for (Iterator it = o.eAllContents(); it.hasNext();) {
            EObject child = (EObject) it.next();
            ensureContainmentInternal(r, child);
        }
    }
    
    private static void ensureContainmentInternal(Resource r, EObject o) {
        for (EContentsEList.FeatureIterator featureIterator = (EContentsEList.FeatureIterator) o
                .eCrossReferences().iterator(); featureIterator.hasNext();) {
            InternalEObject target = (InternalEObject) featureIterator
                    .next();
            EReference eReference = (EReference) featureIterator.feature();
            if (!target.eIsProxy() && !eReference.isDerived()
                    && !eReference.isTransient()) {
                if (target.eResource() == null) {
                    EObject top = getTopElement(target);
                    r.getContents().add(top);
                     org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.INFO,"ensureContainment: add " +  getTypeAndNameLabel(top) +" to " + r.getURI());
                }
            }
        }
    }

    /**
     * 
     * test that the URI begins with a specified prefix.
     * 
     * @param uri
     * @param prefixSet
     *            a set of possible prefix. for example { "http://", "pathmap:" }
     * @return
     * 
     * 
     */
    public static boolean uriBeginsWith(URI uri, Collection prefixSet) {
        if (prefixSet == null)
            return false;

        String uriString = uri.toString();
        for (Iterator it = prefixSet.iterator(); it.hasNext();) {
            String prefix = (String) it.next();
            if (uriString.startsWith(prefix)) {
                return true;
            }
        }
        return false;
    }

    public static boolean containsURI(ResourceSet rs, URI uri) {
        for (int i = 0; i < rs.getResources().size(); i++) {
            Resource r = (Resource) rs.getResources().get(i);
            if (r.getURI().equals(uri))
                return true;
        }
        return false;
    }

    public static String getTypeAndNameLabel(EObject o) {
        if (o == null) {
            return "null";
        }
        EStructuralFeature f = o.eClass().getEStructuralFeature("name");
        if (f instanceof EAttribute) {
            return o.eClass().getName() + "_" + o.eGet(f, false);
        }
        return o.eClass().getName();
    }

    /**
     * 
     * check whether the specified EPackage is registered.
     * 
     * @param p
     * @return
     * 
     * 
     */
    public static boolean isRegistered(EPackage p) {
        return EPackage.Registry.INSTANCE.containsValue(p);
    }

    /**
     * 
     * register an EPackage so that it can be used for creating instance models
     * 
     * @param p
     * 
     * 
     */
    public static void registerEPackage(EPackage p) {
        String uri = p.getNsURI();
        if (uri == null || "".equals(uri)) {
            EPackage superp = p.getESuperPackage();
            if (superp == null) {
                uri = "http://model/" + p.getName();
            } else {
                uri = superp.getNsURI() + "/" + p.getName();
            }
            p.setNsURI(uri);
        }
        EPackage registered = EPackage.Registry.INSTANCE.getEPackage(p
                .getNsURI());
        if (registered != null && registered != p) {
             org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.WARNING, "The EPackage for NS URI " + p.getNsURI()
                    + " changed: " + registered + " -> " + p);
        }
        EPackage.Registry.INSTANCE.put(p.getNsURI(), p);
        for (Iterator it = p.getESubpackages().iterator(); it.hasNext();) {
            EPackage subp = (EPackage) it.next();
            registerEPackage(subp);
        }
    }

    /**
     * 
     * Register all Ecore packages in the specified resource
     * 
     * @param r
     * 
     * 
     */
    public static void registerEPackages(Resource r) {
        for (Iterator it = r.getContents().iterator(); it.hasNext();) {
            Object o = it.next();
            if (o instanceof EPackage) {
                registerEPackage((EPackage) o);
            }
        }
    }

    public static Resource findResourceWithURI(String uri, ResourceSet rs) {
        return findResourceWithURI(uri, rs.getResources());
    }

    public static Resource findResourceWithURI(String uri, List resources) {
        for (int i = 0; i < resources.size(); i++) {
            Resource r = (Resource) resources.get(i);
            if (r.getURI().toString().equals(uri)) {
                return r;
            }
        }
        return null;
    }

    public static EObject findObjectForID(String id, ResourceSet rs) {
        if (rs == null) {
            throw new IllegalArgumentException("ResourceSet must be specified");
        }
        List l = rs.getResources();
        for (int i = 0; i < l.size(); i++) {
            Resource r = (Resource) l.get(i);
            EObject o = r.getEObject(id);
            if (o != null) {
                return o;
            }
        }
        return null;
    }

    public static Set getAllProxies(Resource r) {
        Set result = new HashSet();
        for (Iterator it = r.getAllContents(); it.hasNext();) {
            EObject o = (EObject) it.next();
            for (Iterator it2 = o.eCrossReferences().iterator(); it2.hasNext();) {
                EObject o2 = (EObject) it2.next();
                if (o2.eIsProxy()) {
                    result.add(o2);
                }
            }
        }
        return result;
    }

}
