package org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.mb_xmi;


import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.ModelUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.XmiUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;


/**
 * The model serializer that writes persistent object IDs to the XMI file.
 * It is used together with MBXmiResource, which is a resource 
 * that can associate a model witn a perststent ID table. 
 * <p><p>
 * This model serializer can also serialize the models contained in other kind of resources
 * In this case, the ID table must specified explicitely in the contructor.
 * 
 * 
 * @author P. Sriplakich
 *
 *
 */
public class MBModelSerializer {
    
    public boolean option_omit_default_value = true;
    
    
//    /**
//     * By default,  the references to elements outside the collection will not be written.
//     * Setting this option TRUE is used for serializing with XMI diff. 
//     * 
//     */
//    public boolean option_includesReferenceToOutside = false;
    

    
    //IdTable idTable = null;
    
    /**
     * The fragment of model to be serialized, 
     * 'null' value means serializing the whole resource.
     */
    Set fragment = null;
        
    public MBModelSerializer () {    
    }
    


	
    
//	public String serializeAllInIdTable() {
//	    return serializeFragment(idTable.getModel());
//	}

    public String serializeFragment(Collection fragment) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try {
            serializeFragment(fragment, out);
        } catch (IOException e) {
            e.printStackTrace();
            return "";
        }
        return out.toString();
    }
    
    public void serializeFragment(Collection fragment, OutputStream out) throws IOException {    
        Document doc = XmiUtil.createDocument();
        Element docElem = doc.createElementNS(XmiUtil.XMI_NS,"xmi:XMI");
        docElem.setAttribute("xmlns:xmi", XmiUtil.XMI_NS);
        doc.appendChild(docElem);

//           long time;        
//           time = System.currentTimeMillis();
      
        serializeFragment(fragment, docElem);
           
//        time = System.currentTimeMillis() - time;
//        System.out.println(".... [SER]->DOM = " +time);
//        time = System.currentTimeMillis();
           
         XmiUtil.write(docElem,out);
        
//        time = System.currentTimeMillis() - time;
//        System.out.println(".... [SER]->txt = " +time +" (txt size=" +result.length() +")");
           
    }
    
    /**
     * 
     * serialize the fragment in the root XML element.
     * This operation can be used for serializing a fragment to an existing XML document
     * 
     * @param fragment 
     * @param rootElem
     * 
     */
    public void serializeFragment(Collection fragment, Element rootElem) {
        if(fragment.isEmpty()) {
             org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.WARNING,"fragment is empty");
            return;            
        }
        if(fragment instanceof Set) {
            this.fragment = (Set)fragment;
        } else {
            this.fragment = new HashSet(fragment);
        }

//        initIdTable();        
        Document doc = rootElem.getOwnerDocument();
        Collection writtenElems = new HashSet();
        for(Iterator it = this.fragment.iterator(); it.hasNext();) {
           EObject o = (EObject) it.next();
           EObject top = ModelUtil.getTopElemInFragment(o, this.fragment);
           if(!writtenElems.contains(top)) {
             EPackage p = top.eClass().getEPackage();
             defineNameSpace(p, rootElem);
             Element xmlElem = doc.createElement( p.getNsPrefix() +":" +top.eClass().getName());  
             rootElem.appendChild(xmlElem);
             serializeThisElementAndChildren(top, xmlElem);
             writtenElems.add(top);
           }
        }        
    }    
    
    
    public void serializeResource(Resource r, OutputStream out) throws IOException {
        Document doc = XmiUtil.createDocument();
        Element docElem = doc.createElementNS(XmiUtil.XMI_NS,"xmi:XMI");
        docElem.setAttribute("xmlns:xmi", XmiUtil.XMI_NS);
        doc.appendChild(docElem);
        serializeResource(r, docElem);                      
        XmiUtil.write(docElem, out);     
    }

    public void serializeResource(Resource r, Element rootElem) {    
//        initIdTable();
        for(Iterator it = r.getContents().iterator(); it.hasNext();) {
            Document doc = rootElem.getOwnerDocument();
            EObject top = (EObject) it.next();
            EPackage p = top.eClass().getEPackage();
            defineNameSpace(p, rootElem);
            Element xmlElem = doc.createElement( p.getNsPrefix() +":" +top.eClass().getName());  
            rootElem.appendChild(xmlElem);
            serializeThisElementAndChildren(top, xmlElem);
         }         
    }
    
    

    

   
    
    
    /**
     * 
     * Serializes 'o' and its subelements.
     * Only subelements that are in 'fragment' will be written.
     * 
     * 
     * @param o
     * @param fragment
     * @return
     * 
     */
    void serializeThisElementAndChildren(EObject o, Element xmlElem) {
        // assign ID
        IdTable idTable = IdTableRegistry.getIdTable(o);
        String id = idTable.getOrCreateId(o);
        
        
//      Testing that the EGenericType object is volatile!! It can be modified by the EMF internal system
//        if(o.eClass().getName().equals("EGenericType")) {
//            RootLogger.getLogger().log(Level.FINE, "EGenericType " +o.hashCode() +" -> " +id);
//        }
        
        xmlElem.setAttributeNS(XmiUtil.XMI_NS, "xmi:id", id);
        
        
        // --------- for all attributes ----------------
        for(Iterator it = o.eClass().getEAllAttributes().iterator(); it.hasNext(); ) {
          EAttribute a = (EAttribute) it.next();
          if(a.isDerived() || a.isTransient() || !a.isChangeable() || a.isVolatile() ) {
            // do nothing
          } else {              
            serializeAttribute(o, a, xmlElem);
          }
        }
        // --------- for all references --------------------
        for(Iterator it = o.eClass().getEAllReferences().iterator(); it.hasNext(); ) {
          EReference r = (EReference) it.next();  
          if( !omitSerialize(r) ) {
            serializeReference(o, r, xmlElem);
          }
        } //--------- end for --------------
    }
    
    void serializeAttribute(EObject o, EAttribute a, Element xmlElem) {
        if(!a.isMany()) {              
            //serialize as an XML Attribute
            Object value = o.eGet(a, false);
            if(value!=null) {
                if(option_omit_default_value && 
                        a.getDefaultValue()!=null && 
                        value.equals(a.getDefaultValue())) {
                    // do nothing
                } else {
                  xmlElem.setAttribute(a.getName(), value.toString());
                }
            }
        } else {
            //serialize as a set of XML Elements
            List values = (List) o.eGet(a, false);
            Document doc = xmlElem.getOwnerDocument();
            for(int i=0; i<values.size(); i++) {
                Element subXmlElem = doc.createElement(a.getName());
                Object value = values.get(i);
                Text text = doc.createTextNode(value.toString());
                subXmlElem.appendChild(text);
                xmlElem.appendChild(subXmlElem);
            }
        }        
    }
    
    
    void serializeReference(EObject o, EReference r, Element xmlElem) {
          if(r.isContainment()) {
            // reference to the contents of this object
            if(!r.isMany()) {
                //serialize as an XML Element                         
                EObject value = (EObject) o.eGet(r, false);                  
                if(value!=null) {
                  if(fragment==null || fragment.contains(value)) {  
                    serializeReferenceValueAsContent(r, value, xmlElem);              
                  } 
                }
            } else {
                //serialize as a set of XML Elements          
                List values = (List) o.eGet(r, false);
                for(int i=0; i<values.size(); i++) {
                    EObject value = (EObject) values.get(i);  
                    if(fragment==null || fragment.contains(value)) {  
                        serializeReferenceValueAsContent(r, value, xmlElem);              
                    }             
                }
            }             
        
        } else {  // this is non-aggregation reference  
            if(!r.isMany()) {
                //serialize as an XML Element          
                EObject value = (EObject) o.eGet(r, false);                      
                if(value!=null) {
                    if(fragment==null || fragment.contains(value)) {
                      serializeReferenceValueAsLink(o.eResource(), r, value, xmlElem); 
                    }
                }
            } else {
                //serialize as a set of XML Elements          
                List values = (List) o.eGet(r, false);              
                for(int i=0; i<values.size(); i++) {
                    EObject value = (EObject) values.get(i);  
                    if(fragment==null || fragment.contains(value)) {
                      serializeReferenceValueAsLink(o.eResource(), r, value, xmlElem);
                    }
                }
            }                        
        }              
    }
    
    
    /**
     * 
     * 
     * If the reference is in the same resource as the container,
     * then the ID of this element is written.
     * If it is in a different resource, 
     * then resource#ID is written
     * 
     * 
     *
     */
    void serializeReferenceValueAsLink(Resource ownerResource, EReference r, EObject value, Element containerXmlElem) {                   
         String href = getURI(ownerResource, value);
         if(href!=null) {
            Element subXmlElem = containerXmlElem.getOwnerDocument().createElement(r.getName());
            setXmiType(value.eClass(), subXmlElem);   
            
            subXmlElem.setAttribute("href", href);
            setXmiType(value.eClass(), subXmlElem);
            //System.out.println("href=" +idref);
            containerXmlElem.appendChild(subXmlElem);
         }
    }
    
    
    void serializeReferenceValueAsContent(EReference r, EObject value, Element containerXmlElem) {
            Element subXmlElem = containerXmlElem.getOwnerDocument().createElement(r.getName());
            containerXmlElem.appendChild(subXmlElem);
            setXmiType(value.eClass(), subXmlElem);
            serializeThisElementAndChildren(value, subXmlElem);   

    }
    
    /**
     * 
     * create an attribute 'xmi:type=ns1:Class1' in the specified element, 
     * and define the ns1 namespace at the root element.
     * 
     * @param type
     * @param xmlElem element owner the 'xmi:type' attribute
     * 
     *
     */
    static void setXmiType(EClass type, Element xmlElem) {
        EPackage p = type.getEPackage();
        String prefix = p.getNsPrefix();
        xmlElem.setAttribute("xmi:type", prefix +":" +type.getName());    
        defineNameSpace(p, xmlElem);
    }
    
    /**
     * 
     * define the NameSpace of the EPackage of the type in the root XMI element
     * 
     * @param type
     * @param xmlElem
     * 
     *
     */
    static void defineNameSpace(EPackage p, Element xmlElem) {
        String uri = p.getNsURI();
        String prefix = p.getNsPrefix(); 
        Document doc = xmlElem.getOwnerDocument();
        Element root = doc.getDocumentElement();
        root.setAttribute("xmlns:" +prefix, uri);        
    }
    
    
    
    
    static boolean omitSerialize(EReference r) {
        if(r.isDerived() || r.isTransient() || !r.isChangeable() 
        || r.isContainer()  || r.isVolatile()  ) {
            return true; // omit
        }
        if(r.isContainment())  return false; // must serialize
        EReference op = r.getEOpposite();
        if(op==null) return false; // must serialize
        // choose to serialize the end that is "orderred".
        if(r.isOrdered()) return false; // must serialize
        if(op.isOrdered()) return true; // omit
        // choose to serialize the end that is higher alphabetically
        return r.getName().compareTo(op.getName())<0;        
    }
    

    /**
     * 
     * get the URI of the specified object.
     * If the object belongs the mainResource, 
     * the return the short form URI (just only ID).
     * Otherwise return the full URI (resourceURI # ID).
     * 
     * @param o
     * @return
     */
    String getURI(Resource ownerResource, EObject o) { 
        if(o.eIsProxy()) {
            URI uri = ((InternalEObject) o).eProxyURI();
            URI nUri = uri;
            ResourceSet rs = ownerResource.getResourceSet();      
            if(rs!= null) { 
               nUri = rs.getURIConverter().normalize(uri);
            } 
            // RootLogger.getLogger().log(Level.FINE, "href= " +uri +" -> " +nUri);
            if( ownerResource.getURI().toString().equals( 
                    nUri.trimFragment().toString()) ) {    
              // make sure that the object exist
              EObject v = ownerResource.getEObject(nUri.fragment());
              if(v==null || v.eResource()!=ownerResource ) {
                  return null;
              } else {
                  return nUri.fragment();
              }
            } else {
              return nUri.toString();
            }
        }
        Resource targetRes = o.eResource();
        if( (fragment!=null && fragment.contains(o)) // this is the element in the same fragment
                || targetRes==null                   // this is the element with no container resource
                || targetRes==ownerResource          // this is the element in the same resource
                ) {
            IdTable idTable = IdTableRegistry.getIdTable(o);
            return idTable.getOrCreateId(o);
        }
        return targetRes.getURI().toString() +"#" + targetRes.getURIFragment(o);        
    }
    
//    /**
//     * 
//     * get the Resource containing o.
//     * If no resource contains o, then put o in the main resource
//     */
//    Resource getResource(EObject o) {
//        Resource r = o.eResource();
//        if(r!=null) return r;
//        EObject top = ModelUtil.getTopElement(o);
//        mainResource.getContents().add(top);      
//        return mainResource; 
//    }
    
    
    
//    /**
//     * 
//     * ensure that idTable!=null.
//     * If idTable==null, then try to obtain the IdTable from mainResource.
//     * If mainResource is not MBXmiResource, then create a new IdTable
//     */
//    void initIdTable() {
//        if(idTable == null) {
//            if(mainResource instanceof MBXmiResource) {
//              idTable = ((MBXmiResource) mainResource).getIdTable();
//            } else {
//              idTable = new IdTable();
//            }
//        } else {
//            if((mainResource instanceof MBXmiResource) &&
//                    (((MBXmiResource) mainResource).getIdTable()!=idTable) ) {
//                 org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.WARNING,"the IdTable of "+ mainResource.getURI() + " is not used");
//            }
//        }
//    }
        

    


//    
//    /**
//     * @return Returns the idTable.
//     */
//    public IdTable getIdTable() {
//        return idTable;
//    }
//    /**
//     * @param idTable The idTable to set.
//     */
//    public void setIdTable(IdTable idTable) {
//        this.idTable = idTable;
//    }
    

    
}
