/*
 * $RCSfile: RegistryClient.java,v $
 * $Date: 2007/06/09 12:03:58 $
 * $Revision: 1.23 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * RegistryClient.java 
 * 
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 * @version $Revision: 1.23 $ $Date: 2007/06/09 12:03:58 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.registry;

import java.io.IOException;
import java.net.URL;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.logging.Level;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.DescriptionUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.Emf2XmiConversion;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.ModelUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.Xmi2EmfConversion;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.AlreadyRegistered;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.WebRegistry;
import org.eclipse.mddi.modelbus.adapter.infrastructure.registry.ws.WebRegistryServiceLocator;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.QualifiedServiceName;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ServiceUnknownException;
import org.eclipse.mddi.modelbus.description.abstract_.MetaclassSpecification;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingServiceInterface;
import org.eclipse.mddi.modelbus.description.concrete.Tool;

/**
 * 
 * RegistryClient provides the communication between the Adapter and Registry *
 * 
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 * 
 */
public class RegistryClient {

	String registry_location;

	WebRegistry registry;

	public RegistryClient(String _registry_location)
			throws RegistryNotAvailableException {

		try {
			registry_location = _registry_location;
			WebRegistryServiceLocator loc = new WebRegistryServiceLocator();
			registry = loc.getWebRegistry(new URL(registry_location));
		} catch (Exception e) {
			throw new RegistryNotAvailableException("URL= ["
					+ registry_location + "]", e);
		}
	}

	ToolSelectionStrategy defaultSelector = new DefaultToolSelector();

	/**
	 * Cache for the tool descriptions. The Map key is the tool's URL, the map
	 * value is the tool description
	 */
	Map<String,Tool> url2tool = new HashMap<String, Tool>();

	/**
	 * Cache for the modeling service interface descriptions
	 */
	List<ModelingServiceInterface> modelingServiceInterfaces = new Vector();

	public String getRegistryLocation() {
		return registry_location;
	}

	/**
	 * 
	 * find a Service Description
	 * 
	 * @param serviceName
	 *            The qualified or unqualified service name
	 * @return
	 * @throws RegistryNotAvailableException
	 * @throws ServiceUnknownException
	 * 
	 * 
	 */
	public ModelingService findServiceDescription(String serviceName)
			throws RegistryNotAvailableException, ServiceUnknownException {
		ModelingService s = _findServiceDescription(serviceName);
		if (s != null) {
			return s;
		}
		loadDescriptionsFromRegistryForAllInterfaces();
		s = _findServiceDescription(serviceName);
		if(s==null) {
		    throw new ServiceUnknownException(serviceName);
		}
		return s;
	}

	private ModelingService _findServiceDescription(String serviceName)
			throws RegistryNotAvailableException, ServiceUnknownException {
		QualifiedServiceName qname = QualifiedServiceName
				.getQualifiedServiceName(serviceName);
		if (qname.interfaceName != null) {
			for (ModelingServiceInterface inf : modelingServiceInterfaces) {
				if (inf.getName().equals(qname.interfaceName)) {
					ModelingService s = findModelingService(inf,
							qname.serviceName);
					if (s != null) {
						return s;
					}
				}
			}
		} else {
			for (ModelingServiceInterface inf : modelingServiceInterfaces) {
				ModelingService s = findModelingService(inf, qname.serviceName);
				if (s != null) {
					return s;
				}
			}
		}
		return null;
	}

	private ModelingService findModelingService(ModelingServiceInterface inf,
			String name) {
		for (ModelingService s : (List<ModelingService>) inf.getService()) {
			if (s.getName().equals(name)) {
				return s;
			}
		}
		return null;
	}

	/**
	 * 
	 * Find a tool providing the specified modeling service. The service name
	 * can be qualitied or not (qualitied service name = [Interface].[service])
	 * 
	 * @param serviceName
	 * @param selector
	 *            tool selector (if null, the default one will be used)
	 * @return
	 * @throws RegistryNotAvailableException
	 * @throws ServiceUnknownException
	 * @deprecated not user anymore
	 * 
	 */
	public Tool lookupToolByModelingService(String serviceName,
			ToolSelectionStrategy selector)
			throws RegistryNotAvailableException, ServiceUnknownException {

		if (selector == null) {
			selector = defaultSelector;
		}

		// optimize performance by calling loadDescriptionsFromRegistry() less
		// often.
		// first we try to look in cache
		// if not found, we refresh the cache by loading from Registry
		Tool t = null;
		// try to lookup in cache
		t = selector.selectTool(serviceName, url2tool.values());
		if (t != null)
			return t;
		// otherwise lookup again the registry
		loadDescriptionsFromRegistryForAllInterfaces();

		t = selector.selectTool(serviceName, url2tool.values());
		if (t == null) {
			throw new ServiceUnknownException(serviceName);
		}
		return t;
	}

	public List<Tool> lookupToolsByModelingService(String serviceName)
			throws RegistryNotAvailableException, ServiceUnknownException {
	    List<Tool> result = _lookupToolsByModelingService(serviceName);
	    if(!result.isEmpty()) {
	        return result;
	    }
	    loadDescriptionsFromRegistryForAllInterfaces();
	    return _lookupToolsByModelingService(serviceName);
	}
	
	private List<Tool> _lookupToolsByModelingService(String serviceName)
       throws RegistryNotAvailableException, ServiceUnknownException {

        // if the tool is specified in the targetService 
        QualifiedServiceName qname = QualifiedServiceName.getQualifiedServiceName(serviceName);

        List<Tool> result = new Vector<Tool>();
        for(Tool t :  url2tool.values())  {
            if(t.getInterface() != null) {
              for ( ModelingService s : (List<ModelingService>) t.getInterface().getService()) {                
                if( qname.interfaceName!=null && qname.interfaceName.equals(t.getInterface().getName())
                        && s.getName().equals(qname.serviceName) ) {
                    // both interface and service match
                    result.add(t);
                } else if( qname.interfaceName==null && s.getName().equals(qname.serviceName)) {
                    // the interface is unspecified, and the service matches
                    result.add(t);
                } 
              }
            }
        }
        
        return result;
	}

	public void loadDescriptionsFromRegistryForAllInterfaces()
			throws RegistryNotAvailableException {
		try {

			String[] results = registry.lookupAllModelingServiceInterfaces();
			Collection col = toEmf(results);

			modelingServiceInterfaces.clear();
			modelingServiceInterfaces.addAll(ModelUtil.findElementByType(col,
					"ModelingServiceInterface"));
			url2tool.clear();

			Iterator it = modelingServiceInterfaces.iterator();
			while (it.hasNext()) {
				ModelingServiceInterface interf = (ModelingServiceInterface) it
						.next();
				loadDescriptionsFromRegistryByInterface(interf.getName());
			}
		} catch (IOException e) {
			throw new RegistryNotAvailableException(e);
		}
	}

	public void loadDescriptionsFromRegistryByInterface(
			String modelingServiceInterface) throws IOException {

		String[] results = registry
				.lookupToolsByModelingServiceInterface(modelingServiceInterface);
		Collection col = toEmf(results);
		Collection _tools = ModelUtil.findElementByType(col, "Tool");
		for (Iterator it = _tools.iterator(); it.hasNext();) {
			Tool t = (Tool) it.next();
			String url = DescriptionUtil.getProperty(t, "URL");
			if (url != null) {
				url2tool.put(url, t);
			}
		}
		// register all imported meta-models
		Collection modelTypes = ModelUtil.findElementByType(col, "ModelType");
		for (Iterator it2 = modelTypes.iterator(); it2.hasNext();) {
			registerMetaModels((ModelType) it2.next());
		}

	}

	private Collection toEmf(String[] results) throws IOException {
		// DEBUG
		org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger()
				.log(Level.INFO, "Registry lookup result: ");
		for (int i = 0; i < results.length; i++) {
			org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
					.getLogger().log(Level.INFO, results[i]);
		}
		//
		Collection strings = new Vector();
		for (int i = 0; i < results.length; i++) {
			strings.add(results[i]);
		}
		Collection emfobjs = new Xmi2EmfConversion().convertFromStrings(
				strings, false);
		return emfobjs;
	}

	public String registerTool(Tool desc) throws RegistryNotAvailableException {
		String token = new String();
		try {
			Collection emfobjs = new Vector();
			emfobjs.add(desc);
			String s = Emf2XmiConversion.convertToString(emfobjs);
			token = registry.registerTool(s);
			return token;
		} catch (Exception e) {
			if (e instanceof AlreadyRegistered) {
				org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
						.getLogger()
						.log(Level.WARNING,
								"Registering: Modeling Service is already registred");
				return token;
			} else {
				org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
						.getLogger()
						.log(Level.SEVERE, "Registry not available");
				throw new RegistryNotAvailableException("URL= ["
						+ registry_location + "] " + e);
			}
		}
	}

	public void deregisterTool(String token)
			throws RegistryNotAvailableException {
		try {
			registry.deregisterTool(token);
		} catch (Exception e) {
			throw new RegistryNotAvailableException(
					"URL= " + registry_location, e);
		}
	}

	/**
	 * Tests the availability
	 * 
	 * @return TODO implement
	 */
	public boolean isAvailable() {
		return true;
	}

	/**
	 * @return Returns the tools.
	 */
	public Collection getTools() {
		return url2tool.values();
	}

	/**
	 * 
	 * Register the metamodels required by the model type to the EMF metamodel
	 * registry.
	 * 
	 * @param type
	 * @return
	 * 
	 * 
	 */
	private static void registerMetaModels(ModelType type) {
		ModelType mtype = (ModelType) type;
		Iterator meta_it = mtype.getContent().iterator();
		while (meta_it.hasNext()) {
			MetaclassSpecification metaspec = (MetaclassSpecification) meta_it
					.next();
			EClass metaclass = metaspec.getMetaClass();
			if (metaclass == null) {
				org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
						.getLogger().log(
								Level.WARNING,
								"Cannot find the metaclass for ModelType "
										+ type.getName()
										+ " MetaclassSpecification "
										+ metaspec.getName());
			} else {
				if (metaclass.getEPackage() == null) {
					org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
							.getLogger()
							.log(
									Level.WARNING,
									"The metaclass "
											+ metaclass.getName()
											+ " has no container package, at ModelType "
											+ type.getName());
				} else {
					EPackage metamodel = getOuterMostPackage(metaclass
							.getEPackage());
					registerMetaModel(metamodel);
				}
			}
		}

	}

	private static EPackage getOuterMostPackage(EPackage pack) {

		EPackage next_pack = pack.getESuperPackage();

		while (next_pack != null) {
			pack = next_pack;
			next_pack = pack.getESuperPackage();
		}
		return pack; // the uppest EPackage = metamodel
	}

	private static void registerMetaModel(EPackage metamodel) {
		String uri = metamodel.getNsURI();
		if (uri == null || "".equals(uri)) {
			EPackage superp = metamodel.getESuperPackage();
			if (superp == null) {
				uri = "http://model/" + metamodel.getName();
			} else {
				uri = superp.getNsURI() + "/" + metamodel.getName();
			}
			metamodel.setNsURI(uri);
		}
		EPackage registered = EPackage.Registry.INSTANCE.getEPackage(metamodel
				.getNsURI());
		if (registered != null && registered != metamodel) {
			org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
					.getLogger()
					.log(
							Level.WARNING,
							"The EPackage with NS URI "
									+ metamodel.getNsURI()
									+ " is already loaded! \n"
									+ "It will be kept unchanged, though ModelBus Registry can contain an updated version.");
			return;
		}
		// if new, load it.
		org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger()
				.log(
						Level.INFO,
						"Registring NS URI = " + metamodel.getNsURI()
								+ "\for EPackage" + metamodel);
		EPackage.Registry.INSTANCE.put(metamodel.getNsURI(), metamodel);
		// run procedure for all subpackages
		for (Iterator it = metamodel.getESubpackages().iterator(); it.hasNext();) {
			EPackage subp = (EPackage) it.next();
			registerMetaModel(subp);
		}

	}

}
