/*
 * $RCSfile: DefaultModelSerializer.java,v $
 * $Date: 2007/06/09 02:33:11 $
 * $Revision: 1.7 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * DefaultSerializer.java implements for prameter serialisation methods
 * 
 * @author Andrey Sadovykh (LIP6)
 * @version $Revision: 1.7 $ $Date: 2007/06/09 02:33:11 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.serialize;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Level;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.GlobalResourceRegistry;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.ModelUtil;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;

/*
 * DefaultSerializer
 * 
 */

public class DefaultModelSerializer implements ModelSerializer {

    
    /**
     * Map entry specifying the read-only input elements
     */
    public static final String IN_CONTENT = "IN_CONTENT";
    /**
     * Map entry specifying the mutable input elements
     */
    public static final String INOUT_CONTENT = "INOUT_CONTENT";
    /**
     * Map entry specifying the output elements
     */
    public static final String OUT_CONTENT = "OUT_CONTENT";
    
    
    protected boolean skipCrossReferences = false;

    /**
     * the prefixes of the resource URIs that be NOT be serialized, as
     * cross-referenced resources. The default value is {"pathmap://ModelBus" ,
     * "http://" }. <br>
     */
    protected Collection ignoredUriPrefixSet = new Vector(
            GlobalResourceRegistry.GLOBAL_URI_PREFIX_SET);

    /**
     * defines wheter ModelBus can change resource URI if it needs to serialize
     * resources having the same URI.
     */
    public boolean canChangeURI = true;

    protected List<Resource> sentResources;

    protected List<Resource> receivedResources;

    /**
     * The ResourceSet of holding result resources having the same URI as the
     * input resources
     */
    protected ResourceSet inoutResourceSet;

    protected ResourceSet defaultReceivedResourceSet;

    public DefaultModelSerializer() {
        super();
    }

    /**
     * @return Returns the skipCrossReferences.
     */
    public boolean isSkipCrossReferences() {
        return skipCrossReferences;
    }

    /**
     * @param skipCrossReferences
     *            The skipCrossReferences to set.
     */
    public void setSkipCrossReferences(boolean skipCrossReferences) {
        this.skipCrossReferences = skipCrossReferences;
    }

    /**
     * @return Returns the ignoredUriPrefixSet.
     */
    public Collection getIgnoredUriPrefixSet() {
        return ignoredUriPrefixSet;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializer#serialize(org.eclipse.mddi.modelbus.description.abstract_.Parameter[],
     *      java.lang.Object[])
     */
    public SerializedXmiDocument[] serialize(List<Parameter> params,
            Map<String, Object> values) {
        Collection<EObject> allModelElements = getScope(params, values);

        sentResources = ModelUtil.getResourcesFromObjects(
                allModelElements, skipCrossReferences, ignoredUriPrefixSet);

        return serializeResources();
    }

    protected SerializedXmiDocument[] serializeResources() {
        Set<String> uris = new HashSet(); // used for checking redundant URIs
        Vector<SerializedXmiDocument> docVector = new Vector();
        for (Resource r : sentResources) {
            try {
                SerializedXmiDocument doc = new SerializedXmiDocument();
                ByteArrayOutputStream bout = new ByteArrayOutputStream();
                saveResource(r, bout);
                doc.setXmi(bout.toString());
                String uri = r.getURI().toString();
                int i = 0;
                while (canChangeURI && uris.contains(uri)) {
                    uri = r.getURI().trimFileExtension().toString() + i + "."
                            + r.getURI().fileExtension();
                    r.setURI(URI.createURI(uri));
                    RootLogger.getLogger().log(Level.WARNING,
                            "Redundant URI. Changed to " + uri);
                    i++;
                }
                doc.setUri(uri);
                uris.add(uri);
                docVector.add(doc);
            } catch (Exception e) {
                org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                        .getLogger().log(Level.SEVERE, "", e);
            }
        }
        return (SerializedXmiDocument[]) docVector
                .toArray(new SerializedXmiDocument[docVector.size()]);
    }

    /**
     * Return a collection of model elements to be transmitted
     * 
     * @param params
     * @param values
     * @return
     */
    protected Collection<EObject> getScope(List<Parameter> params,
            Map<String, Object> values) {

        Collection<EObject> result = new HashSet();

        List<EObject> content = (List<EObject>) values.get(OUT_CONTENT);
        if (content != null) {
            result.addAll(content);
        }
        List<EObject> inContent = (List<EObject>) values.get(IN_CONTENT);
        if (inContent != null) {
            result.addAll(inContent);
        }
        List<EObject> inoutContent = (List<EObject>) values.get(INOUT_CONTENT);
        if (inoutContent != null) {
            result.addAll(inoutContent);

        }

        for (Parameter p : params) {
            if (p.getType() instanceof ModelType) {

                // the patameter values can be URI
                Object value = values.get(p.getName());
                if (value instanceof List) {
                    result.addAll((Collection<EObject>) value);
                } else if (value instanceof Collection[]) {
                    Collection<EObject>[] modelArray = (Collection<EObject>[]) value;
                    for (int j = 0; j < modelArray.length; j++) {
                        result.addAll(modelArray[j]);
                    }
                }
            }
        }
        return result;
    }

    /**
     * Overriding this method enable us to choose the ResourceSet for creating a
     * result resource
     * 
     * @return
     */
    protected Resource getResultResource(URI uri) {
        ResourceSet rs = getResourceSetFor(uri);
        return rs.createResource(uri);
    }

    /**
     * Get the ResourceSet for creating result resources
     * 
     * @return
     */
    protected ResourceSet getResourceSetFor(URI uri) {
        if (defaultReceivedResourceSet == null) {
            if (sentResources != null && sentResources.size() > 0) {
                defaultReceivedResourceSet = sentResources.get(0)
                        .getResourceSet();
            } else {
                defaultReceivedResourceSet = new ResourceSetImpl();
            }
        }
        if (inoutResourceSet == null) {
            inoutResourceSet = new ResourceSetImpl();
        }
        if (sentResources != null
                && ModelUtil.findResourceWithURI(uri.toString(), sentResources) != null) {
            return inoutResourceSet;
        }
        if (ModelUtil.containsURI(defaultReceivedResourceSet, uri)) {
            return inoutResourceSet;
        }
        return defaultReceivedResourceSet;
    }

    /**
     * Overriding this method enables us to use a new serialization mechanism
     * other than the one provided by Resource
     * 
     * @throws IOException
     */
    protected void saveResource(Resource r, OutputStream out)
            throws IOException {
        r.save(out, null);
    }

    /**
     * Overriding this method enables us to use a new serialization mechanism
     * other than the one provided by Resource
     * 
     * @throws IOException
     */
    protected void loadResource(Resource r, InputStream in) throws IOException {
        r.load(in, null);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializer#deserialize(org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.SerializedXmiDocument[])
     */
    public DeserializedModel[] deserialize(SerializedXmiDocument[] documents) {
        DeserializedModel[] result = new DeserializedModel[documents.length];
        receivedResources = new Vector();
        for (int i = 0; i < documents.length; i++) {
            result[i] = new DeserializedModel();
            result[i].setUri(documents[i].getUri());

            ByteArrayInputStream bin = new ByteArrayInputStream(documents[i]
                    .getXmi().getBytes());
            Resource r = getResultResource(URI.createURI(documents[i].getUri()));
            receivedResources.add(r);
            if (r != null) {
                try {
                    loadResource(r, bin);
                } catch (IOException e) {
                    org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                            .getLogger().log(
                                    Level.SEVERE,
                                    "Error in deserializing "
                                            + documents[i].getUri() + " " + e);
                }
                result[i].setValue(r);
            } else {
                org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                        .getLogger().log(
                                Level.SEVERE,
                                "No resource factory for "
                                        + documents[i].getUri());
            }

        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializer#getReferences(org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.SerializedXmiDocument[],
     *      org.eclipse.mddi.modelbus.description.abstract_.Parameter,
     *      java.lang.Object)
     */
    public ModelElementReference[] getReferences(Parameter p, Object o) {
        Vector<ModelElementReference> v = new Vector();
        for (EObject eo : (Collection<EObject>) o) {
            Resource r = eo.eResource();
            ModelElementReference ref = new ModelElementReference();
            ref.setUri(r.getURI().toString());
            ref.setRef(getReference(eo));
            v.add(ref);
        }
        return (ModelElementReference[]) v.toArray(new ModelElementReference[v
                .size()]);
    }

    /**
     * Overriding this method enables us to use an alternative adressing scheme
     * for locating a model element
     * 
     * @param o
     * @return
     */
    protected String getReference(EObject o) {
        Resource r = o.eResource();
        return r.getURIFragment(o);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializer#dereference(org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.DeserializedModel[],
     *      org.eclipse.mddi.modelbus.description.abstract_.Parameter,
     *      org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelElementReference[])
     */
    public Object dereference(Parameter p, ModelElementReference[] refs) {
        Vector<EObject> v = new Vector();
        for (int i = 0; i < refs.length; i++) {
            String uri = refs[i].getUri();
            String ref = refs[i].getRef();

            Resource r = (Resource) ModelUtil.findResourceWithURI(uri,
                    receivedResources);
            if (r != null) {
                EObject eo = null;
                eo = dereference(r, ref);
                if (eo != null) {
                    v.add(eo);
                } else {
                    RootLogger.getLogger().log(Level.SEVERE,
                            "Fragment Not Found " + uri + ", " + ref);
                    for (EObject o : (List<EObject>) r.getContents()) {
                        if (!v.contains(o)) {
                            RootLogger.getLogger().log(
                                    Level.INFO,
                                    "Parameter p " + p.getName()
                                            + ": Adding top element " + o);
                            v.add(o);
                        }
                    }
                }
            } else {
                RootLogger.getLogger().log(Level.SEVERE,
                        "Resource Not Found " + uri);
            }
        }
        return v;
    }

    /**
     * Overriding this method enables us to use an alternative adressing scheme
     * for locating a model element
     * 
     * @param o
     * @return
     */
    protected EObject dereference(Resource r, String fragment) {
        return r.getEObject(fragment);
    }

    /**
     * @return Returns the canChangeURI.
     */
    public boolean canChangeURI() {
        return canChangeURI;
    }

    /**
     * @param canChangeURI
     *            The canChangeURI to set.
     */
    public void setCanChangeURI(boolean canChangeURI) {
        this.canChangeURI = canChangeURI;
    }

    public List<Resource> getSentResources() {
        return sentResources;
    }

    public List<Resource> getReceivedResources() {
        return receivedResources;
    }

}
