/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * Created on 15 juin 2005
 *
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws;

import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPConnection;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPFault;
import javax.xml.soap.SOAPMessage;

import org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.DescriptionUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.AbstractInvocationSender;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.async.AsyncConnectionManager;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.async.NoAsyncConnectionFoundException;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.async.SOAPAsyncConnection;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.async.SOAPAsyncConnectionFactory;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.async.ThreadPoolQueue;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelBusCommunicationException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelTypeMismatchException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.NoToolAvailableException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ServiceUnknownException;
import org.eclipse.mddi.modelbus.description.abstract_.Error;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;

/**
 * 
 * 
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 * 
 * 
 */
public class DefaultWsInvocationSender extends AbstractInvocationSender {

    /**
     * Async
     */
    protected static final int DEFAULT_WORKING_THREADS_NUMBER = 20;

    /**
     * Async
     */
    protected static Map connexionId2modelingService = new Hashtable();

    /**
     * Async
     */
    protected static ThreadPoolQueue poolQueue = null;

    public SOAPMessage requestMsg;

    public SOAPMessage responseMsg;

    /**
     * in/inout parameter values
     */
    public Map inputs;

    /**
     * out/inout parameter values
     */
    public Map outputs;

    public ModelingService ms;

    /**
     * used in asynchronous call (added by Nicolas G)
     */
    public String connectionId;

    public DefaultWsInvocationSender() {
        if (getMarshaler() == null) {
            setMarshaler(new DefaultMarshaler());
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.user.provider.WsInvocationSender#invoke(org.eclipse.mddi.modelbus.description.concrete.Tool,
     *      org.eclipse.mddi.modelbus.description.abstract_.ModelingService,
     *      java.lang.Object[])
     */
    public Map<String, Object> invoke(ModelingService ms,
            Map<String, Object> inputs) throws ModelBusCommunicationException {

        this.ms = ms;
        this.inputs = inputs;
        try {
          createRequestMessage();
          makeConnection();
          readResponseMessage();
          return outputs;
        } catch(SOAPException e) {
            throw new ModelBusCommunicationException(e);
        }
    }

    public void createRequestMessage() throws SOAPException,
            ModelBusCommunicationException {

        requestMsg = SoapUtil.messageFactory.createMessage();
        
        SOAPBody requestbody = requestMsg.getSOAPBody();
        DefaultMarshaler.setNamespaces(requestbody);
        SOAPElement reqTopElem = requestbody.addChildElement(ms.getName(),
                "modelbus");
        List params = DescriptionUtil.get_in_inout_Parameters(ms);

        // RootLogger.getLogger().log(Level.INFO,"Serializer:"+getMarshaler().getSerializer());

        getMarshaler().marshal(params, inputs, reqTopElem);

    }

    public void makeConnection() throws SOAPException {

        RootLogger.getLogger().log(Level.INFO, "Call URL: " + getToolUrl());

        SOAPConnection con = SoapUtil.connectionFactory.createConnection();

        RootLogger.getLogger().log(
                Level.INFO,
                "Request Message: " + "\n"
                        + formatMessage(requestMsg.getSOAPBody().toString()));

        responseMsg = con.call(requestMsg, getToolUrl().toString());

        con.close();

        RootLogger.getLogger().log(
                Level.INFO,
                "Response Message: " + "\n"
                        + formatMessage(responseMsg.getSOAPBody().toString()));

    }

    /**
     * Show < > " instead of &lt; &quot; &gt;
     * 
     * @param msg
     * @return
     */
    public static String formatMessage(String msg) {

        msg = msg.replace((CharSequence) "&lt;", (CharSequence) "<");
        msg = msg.replace((CharSequence) "&quot;", (CharSequence) "\"");
        msg = msg.replace((CharSequence) "&gt;", (CharSequence) ">");
        return msg;
    }

    // unmarshall response message
    public void readResponseMessage() throws ModelBusCommunicationException,
            ModelingServiceError {

        SOAPBody responsebody = null;
        try {

            responsebody = responseMsg.getSOAPBody();
        } catch (SOAPException e1) {
            throw new ModelBusCommunicationException(
                    "Can't parse response message", e1);
        }
        if (responsebody.hasFault()) {

            SOAPFault f = responsebody.getFault();
            if (DefaultMarshaler.isModelingServiceError(f)) {

                String errorName = DefaultMarshaler
                        .getModelingServiceErrorName(f);
                Error errorType = null;
                if (errorName != null)
                    errorType = DescriptionUtil.getError(ms, errorName);

                ModelingServiceError err = null;

                try {
                    err = getMarshaler().unmarshalModelingServiceError(
                            errorType, f);
                } catch (SOAPException e) {
                    RootLogger.getLogger().log(Level.SEVERE,
                            "Can't unmarshal ModelingServiceError", e);
                }
                throw err;

            } else {
                throw new ModelBusCommunicationException(f.toString()
                        + "[ uri = " + getToolUrl() + "]");
            }

        } else {
            SOAPElement resTopElem = SoapUtil.getFirstChild(responsebody);
            if (resTopElem == null) {
                org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                        .getLogger().log(Level.SEVERE, "Empty SOAP response");
                org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                        .getLogger().log(Level.INFO, responsebody.toString());
                throw new ModelBusCommunicationException("Empty SOAP response");
            }

            // unmarchal params
            List<Parameter> params = getResponseParameters();

            try {
                outputs = getMarshaler().unmarshal(params, resTopElem);
            } catch (Exception e) {
                RootLogger.getLogger().log(Level.WARNING,
                        "Cannot unmarshal response message", e);
                throw new ModelBusCommunicationException(
                        "Cannot unmarshal response message", e);
            }
        }
    }

    /**
     * Redefine this method, if the INOUT access points do not need to be in
     * response
     */
    protected List<Parameter> getResponseParameters() {
        return DescriptionUtil.get_out_inout_Parameters(ms);
    }

    /**
     * Async
     */
    public String invokeAsync(ModelingService ms, Map inputs)
            throws ModelBusCommunicationException {

        this.ms = ms;
        this.inputs = inputs;

        if (poolQueue == null)
            createThreadPoolQueue();

        try {
          responseMsg = SoapUtil.messageFactory.createMessage();

          createRequestMessage();
          makeAsyncConnection();
          return connectionId;
        } catch(SOAPException e) {
                throw new ModelBusCommunicationException(e);
        }
    }

    /**
     * Async
     */
    public void makeAsyncConnection() throws SOAPException {

        SOAPAsyncConnection con = SOAPAsyncConnectionFactory.instance
                .createAsyncConnection();

        connectionId = con.call(requestMsg, getToolUrl().toString(), poolQueue);

        AsyncConnectionManager.getInstance().putAsyncConnection(connectionId,
                con);

        connexionId2modelingService.put(connectionId, ms);
        // to make the ms avaible from the getResult() method, mandatory to
        // unmarshall the message

    }

    public void closeAsyncConnection(String connexionId) throws SOAPException,
            NoAsyncConnectionFoundException {
        SOAPAsyncConnection con = AsyncConnectionManager.getInstance()
                .getAsyncConnection(connexionId);
        // closing the connection
        con.close();
        AsyncConnectionManager.getInstance().removeAsyncConnection(
                con.hashCode() + "");

        // removing the binding connexionId-Ms from the map
        connexionId2modelingService.remove(connexionId);

    }

    public boolean isResultReady(String connexionId)
            throws ServiceUnknownException, NoToolAvailableException,
            ModelTypeMismatchException, ModelBusCommunicationException,
            NoAsyncConnectionFoundException, ModelingServiceError {

        boolean resultReady = false;

        try {
            SOAPAsyncConnection con = AsyncConnectionManager.getInstance()
                    .getAsyncConnection(connexionId);
            resultReady = con.isResultReady();
        } catch (NoAsyncConnectionFoundException e) {
            e.printStackTrace();
        }
        return resultReady;
    }

    public Object[] getResult(String connexionId)
            throws SOAPException, ModelBusCommunicationException {

        SOAPAsyncConnection con = AsyncConnectionManager.getInstance()
                .getAsyncConnection(connexionId);
        responseMsg = (SOAPMessage) con.getResult();

        // unmarshalling
        ModelingService ms = (ModelingService) connexionId2modelingService
                .get(connexionId);
        if (ms == null) {
            throw new ServiceUnknownException();
        }
        readResponseMessage();

        // the getResultMethod can be called only once
        try {
            closeAsyncConnection(connexionId);
        } catch (SOAPException e) {
            throw new ModelBusCommunicationException(e);
        } catch (NoAsyncConnectionFoundException e) {
            throw new ModelBusCommunicationException(e);
        }

        // TODO
        // if the queue is empty, we close the threadpool
        if (poolQueue.isEmpty())
            closeThreadPoolQueue();

        Parameter[] outputParams = (Parameter[]) DescriptionUtil
                .get_out_inout_Parameters(ms).toArray(new Parameter[0]);
        Object[] result = DescriptionUtil.map2Array(outputParams, outputs);
        return result;
    }

    // TODO
    public static void closeThreadPoolQueue() {
        poolQueue.close();
        poolQueue = null;
    }

    // TODO
    public static void createThreadPoolQueue() {
        poolQueue = new ThreadPoolQueue(DEFAULT_WORKING_THREADS_NUMBER);
    }

}