/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

package org.eclipse.mddi.modelbus.adapter.user.consumer.impl;

import java.net.URL;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.DescriptionUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ProviderProxy;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.QualifiedServiceName;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.RepositoryClient;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.SessionManager;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.SessionProviderProxy;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws.TransportManager;
import org.eclipse.mddi.modelbus.adapter.user.AdapterStub;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;
import org.eclipse.mddi.modelbus.adapter.user.consumer.GenericConsumer;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelBusCommunicationException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelTypeMismatchException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.NoToolAvailableException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ServiceUnknownException;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.concrete.Tool;

/**
 * @author Prawee Sriplakich, Andrey Sadovykh (LIP6)
 * 
 */
public class GenericConsumerImpl implements GenericConsumer {

    AdapterStub adapter;
    
    
    public GenericConsumerImpl(AdapterStub adapter) {
        this.adapter = adapter;
    }
    

    public Map consume(URL toolUrl, String targetService, Map<String, Object> inputMap)
            throws ModelBusCommunicationException {
        if (inputMap == null) {
            org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger
                    .getLogger().log(Level.SEVERE, "null inputs");
            throw new ModelTypeMismatchException("null inputs");
        }
        ProviderProxy sender = adapter.getTransportManager()
                .createProviderProxy(toolUrl);

        RepositoryClient client = new RepositoryClient(this);
        sender.getMarshaler().setRepositoryClient(client);

        ModelingService ms = adapter.getRegistryClient()
                .findServiceDescription(targetService);

        return sender.invoke(ms, inputMap);

    }

    public Map consume(String targetService, Map<String, Object> inputMap)
            throws ModelBusCommunicationException {

        // try to find if session is already created
        String[] sessions = getOpenSessions(targetService);
        if (sessions.length > 0) {
            consume(targetService, inputMap, sessions[0]);
        }
                
        List<Tool> toolDescriptions = adapter.findToolDescriptions(targetService);

        // if the tool is specified in the targetService 
        QualifiedServiceName qname = QualifiedServiceName.getQualifiedServiceName(targetService);
        if(qname.toolName!=null) {
            for (Tool t : toolDescriptions) {
                if(t.getName().equals(qname.toolName)) {
                    return consume(targetService, inputMap, t);
                }
            }
            throw new NoToolAvailableException("tool " + qname.toolName);
        }
        
        for (Tool t : toolDescriptions) {
            try {

                return consume(targetService, inputMap, t);

            } catch (ModelingServiceError e) {
                throw e;
            } catch (ModelBusCommunicationException e) {
                // log exception, then retry the next tool.
                RootLogger.getLogger().log(
                        Level.INFO,
                        "tool not available " + t.getName() + " url= "
                                + DescriptionUtil.getUrl(t));
                RootLogger.getLogger().log(Level.INFO, e.toString());
            }
        }
        throw new NoToolAvailableException("already tried " + toolDescriptions);
    }
    
    public Map consume(String targetService, Map<String, Object> inputMap, Tool t)
    throws ModelBusCommunicationException {
        // Check if Provider Tool is SessionEnabled
        if (DescriptionUtil.getIsSessionEnabled(t)) {
            String sessionId = newSession(t);
            return consume(targetService, inputMap, sessionId);
        }
        
        ProviderProxy sender = adapter.getTransportManager().createProviderProxy(t);

        RepositoryClient client = new RepositoryClient(this);
        sender.getMarshaler().setRepositoryClient(client);

        ModelingService ms = DescriptionUtil.getServiceDescription(
                targetService, t);

        return sender.invoke(ms, inputMap);
    }

    /**
     * @deprecated
     */
    public Object[] consume(String targetService, Object[] parameterValues)
            throws ModelBusCommunicationException {

        ModelingService ms = adapter.getRegistryClient()
                .findServiceDescription(targetService);

        Parameter[] inputParams = (Parameter[]) DescriptionUtil
                .get_in_inout_Parameters(ms).toArray(new Parameter[0]);
        Parameter[] outputParams = (Parameter[]) DescriptionUtil
                .get_out_inout_Parameters(ms).toArray(new Parameter[0]);
        Map inputMap, outputMap;
        inputMap = DescriptionUtil.array2Map(inputParams, parameterValues);
        outputMap = consume(targetService, inputMap);
        Object[] outputArray = DescriptionUtil.map2Array(outputParams,
                outputMap);

        return outputArray;

    }

    public Map consume(String targetService, Map<String, Object> inputMap, String sessionId)
            throws ModelBusCommunicationException {

        TransportManager tm = adapter.getTransportManager();
        SessionManager sm = tm.getSessionManager();
        SessionProviderProxy sender = sm.getProviderProxy(sessionId);

        RepositoryClient client = new RepositoryClient(this);
        sender.getMarshaler().setRepositoryClient(client);
        ModelingService ms = adapter.getRegistryClient()
                .findServiceDescription(targetService);
        return sender.invoke(ms, inputMap);
    }

    /**
     * @deprecated
     */
    public Object[] consume(String targetService, Object[] parameterValues,
            String sessionId) throws ModelBusCommunicationException {

        ModelingService ms = adapter.getRegistryClient()
                .findServiceDescription(targetService);

        Parameter[] inputParams = (Parameter[]) DescriptionUtil
                .get_in_inout_Parameters(ms).toArray(new Parameter[0]);
        Parameter[] outputParams = (Parameter[]) DescriptionUtil
                .get_out_inout_Parameters(ms).toArray(new Parameter[0]);
        Map inputMap, outputMap;
        inputMap = DescriptionUtil.array2Map(inputParams, parameterValues);
        outputMap = consume(targetService, inputMap, sessionId);
        Object[] outputArray = DescriptionUtil.map2Array(outputParams,
                outputMap);

        return outputArray;

    }

    public String newSession(String targetService)
            throws ModelBusCommunicationException, ServiceUnknownException {
        List<Tool> toolDescriptions = adapter.findToolDescriptions(targetService);

        for (Tool t : toolDescriptions) {
            try {
                return newSession(t);
            } catch (Exception e) {
                // log exception, then retry the next tool.
                RootLogger.getLogger().log(
                        Level.INFO,
                        "tool not available " + t.getName() + " url= "
                                + DescriptionUtil.getUrl(t));
                RootLogger.getLogger().log(Level.INFO, e.toString());
            }
        }
        throw new NoToolAvailableException("already tried " + toolDescriptions);
    }

    public String newSession(Tool t)
            throws ModelBusCommunicationException, ServiceUnknownException {
            SessionProviderProxy proxy = adapter.getTransportManager().createSessionProviderProxy(t);
            String sessionId = proxy.newSession();
            return sessionId;
    }

    public void endSession(String sessionId)
            throws ModelBusCommunicationException {
        TransportManager tm = adapter.getTransportManager();
        SessionManager sm = tm.getSessionManager();
        SessionProviderProxy proxy = sm.getProviderProxy(sessionId);
        org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger()
                .log(Level.INFO, "Proxy:" + proxy);
        proxy.endSession(sessionId);
    }

    public String[] getAllOpenSessions() {
        String[] sessions = adapter.getTransportManager().getSessionManager()
                .getAllOpenSessions();
        return sessions;
    }

    public String[] getOpenSessions(String targetService) {
        String[] sessions = adapter.getTransportManager().getSessionManager()
                .getOpenSessions(targetService);
        return sessions;

    }

    /**
     * Asynchronous call section
     * 
     */
    private Map connectionId2provider = new Hashtable();

    /**
     * @deprecated
     */
    public String consumeAsync(String targetService, Object[] parameterValues)
    throws ServiceUnknownException, NoToolAvailableException,
    ModelTypeMismatchException, ModelBusCommunicationException {
        ModelingService ms = adapter.getRegistryClient()
        .findServiceDescription(targetService);

        Parameter[] inputParams = (Parameter[]) DescriptionUtil
        .get_in_inout_Parameters(ms).toArray(new Parameter[0]);

        Map inputMap = DescriptionUtil.array2Map(inputParams, parameterValues);
        return consumeAsync(targetService, inputMap);
    }
    
    public String consumeAsync(String targetService, Map<String, Object> inputMap)
            throws ServiceUnknownException, NoToolAvailableException,
            ModelTypeMismatchException, ModelBusCommunicationException {


        // try to find if session is already created
        String[] sessions = getOpenSessions(targetService);
        if (sessions.length > 0) {
            consumeAsync(targetService, inputMap, sessions[0]);
        }

        List<Tool> toolDescriptions = adapter.findToolDescriptions(targetService);

        for (Tool t : toolDescriptions) {
            try {

                // Check if Provider Tool is SessionEnabled
                if (DescriptionUtil.getIsSessionEnabled(t)) {
                    String sessionId = newSession(t);
                    return consumeAsync(targetService, inputMap, sessionId);
                }

                ProviderProxy sender;

                sender = adapter.getTransportManager().createProviderProxy(t);
                RepositoryClient client = new RepositoryClient(this);
                sender.getMarshaler().setRepositoryClient(client);

                ModelingService ms = DescriptionUtil.getServiceDescription(
                        targetService, t);

                String connectionId = sender.invokeAsync(ms, inputMap);

                connectionId2provider.put(connectionId, sender);

                return connectionId;
            } catch (ModelingServiceError e) {
                throw e;
            } catch (Exception e) {
                // log exception, then retry the next tool.
                RootLogger.getLogger().log(
                        Level.INFO,
                        "tool not available " + t.getName() + " url= "
                                + DescriptionUtil.getUrl(t));
                RootLogger.getLogger().log(Level.INFO, e.toString());
            }
        }
        throw new NoToolAvailableException("already tried " + toolDescriptions);
    }
    
    

    public boolean isResultReady(String resultHolderID) {

        ProviderProxy sender = (ProviderProxy) connectionId2provider
                .get(resultHolderID);

        if (sender == null)
            return false;
        else
            try {
                return sender.isResultReady(resultHolderID);
            } catch (Throwable e) {
                e.printStackTrace();
                return false;
            }
    }

    public Object[] getResult(String connectionID) {

        ProviderProxy sender = (ProviderProxy) connectionId2provider
                .get(connectionID);

        try {
            Object[] result = sender.getResult(connectionID);
            connectionId2provider.remove(connectionID);
            return result;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }
    
    /**
     * @deprecated
     */
    public String consumeAsync(String targetService,  Object[] parameterValues,
            String sessionId) throws ServiceUnknownException,
            NoToolAvailableException, ModelTypeMismatchException,
            ModelBusCommunicationException {
        ModelingService ms = adapter.getRegistryClient()
        .findServiceDescription(targetService);

        Parameter[] inputParams = (Parameter[]) DescriptionUtil
        .get_in_inout_Parameters(ms).toArray(new Parameter[0]);

        Map inputMap = DescriptionUtil.array2Map(inputParams, parameterValues);
        return consumeAsync(targetService, inputMap, sessionId);

    }

    public String consumeAsync(String targetService, Map<String, Object> inputMap,
            String sessionId) throws ServiceUnknownException,
            NoToolAvailableException, ModelTypeMismatchException,
            ModelBusCommunicationException {


        TransportManager tm = adapter.getTransportManager();
        SessionManager sm = tm.getSessionManager();
        SessionProviderProxy sender = sm.getProviderProxy(sessionId);

        RepositoryClient client = new RepositoryClient(this);
        sender.getMarshaler().setRepositoryClient(client);

        ModelingService ms = DescriptionUtil.getServiceDescription(targetService,
                sender.getToolDescription());

        String connectionId = sender.invokeAsync(ms, inputMap, sessionId);
        connectionId2provider.put(connectionId, sender);

        return connectionId;

    }
}
