/*
 * $RCSfile: ModelBusResourceSet.java,v $
 * $Date: 2007/06/09 12:03:57 $
 * $Revision: 1.15 $
 * $Author: xblanc $
 */
 
/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */
 
/*
 * ModelBusResourceFactory.java
 * 
 * @author Prawee Sriplakich
 * @version $Revision: 1.15 $ $Date: 2007/06/09 12:03:57 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation;

import java.util.logging.Level;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.GlobalResourceRegistry.GlobalResourceException;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.mb_xmi.MBXmiResourceFactory;
import org.eclipse.mddi.modelbus.description.impl.DescriptionPackageImpl;

/**
 * @author P. Sriplakich
 * 
 * ModelBusResourceSet enables the link resolutions between multiple-resources.
 * It uses GlobalResourceRegistry for resolving shared resources, such as profiles or metamodels.
 * 
 * ModelBusResourceSet resolve a URI for a resource with the following mechanism.
 * 1. It looks in the resources contained in this set. If found, return this resource.
 * 2. It looks in the Global Resource Set. If found, return this resource.
 * 3. return null; (Resource not found) 
 * 
 * 
 * When the resource can not be resolved, 
 * EMF will create proxy objects for the unresolved objects contained in this resource.
 * A proxy object contains the unresolved URI but has empty content. 
 * It only enables the user to navigate, but he can not read its properties.
 *  
 */
public class ModelBusResourceSet extends ResourceSetImpl  {
    

    public static final Resource.Factory xmiResourceFactory = new XMIResourceFactoryImpl();
    
    static GlobalResourceRegistry globalResourceRegistry = new GlobalResourceRegistry();
    
    /**
     * This URI is used for creating a resource for "orphan" EObjects 
     * (objects that are not contained in any resource)
     */
    public static final URI MB_DEFAULT_RES_URI = URI.createURI("default_resource.modelbus");

    

    /** 
     *  Force the initialization of this class.
     */
    public static void init() {
    }
    
    static {
        MBXmiResourceFactory.init();
        UUIDXMIResourceFactory.init();
        DescriptionPackageImpl.init();

        try {
            Resource r = EcorePackage.eINSTANCE.eResource();
            // add Ecore model to global resource            
            getGlobalResourceRegistry().registerResource(r.getURI(), r);
                        
        } catch(Exception e) {
             org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.SEVERE, "cannot bind Ecore model to GlobalRegistry", e);
        }        
    }
    
  
    
    /**
     * @return Returns the globalResourceSet.
     */
    public static GlobalResourceRegistry getGlobalResourceRegistry() {
        return globalResourceRegistry;
    }
    
    public ModelBusResourceSet() {
        // org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.INFO,"created");
    } 
    
    
    
    /**
     * Override the method of ResourceImpl
     * 
     * Returns a resolved resource available outside of the resource set.
     * It is called by {@link #getResource(URI, boolean) getResource(URI, boolean)} 
     * after it has determined that the URI cannot be resolved 
     * based on the existing contents of the resource set.
     * This implementation looks up the URI in the {#getPackageRegistry() local} package registry.
     * Clients may extend this as appropriate.
     * @param uri the URI
     * @param loadOnDemand whether demand loading is required.
     */
    protected Resource delegatedGetResource(URI uri, boolean loadOnDemand) {
      EPackage ePackage = getPackageRegistry().getEPackage(uri.toString());
      if(ePackage != null) return ePackage.eResource();      
      Resource r = globalResourceRegistry.getResource(uri);      
      return r;
    }
    
    
    /**
     * This is a modification of the default implementation of ResourceImpl,
     * in order to take into account the cached resources in the global registry.
     * <br>
     * First, we try to find the resource in the global registry.
     * If found, return it (avoid creating a duplicated resource)
     * Otherwise, create a new resource.
     * If this resource has the global scheme URI, 
     * then register it with the gobal registry.
     */
    public Resource createResource(URI uri)
    {
      Resource result = globalResourceRegistry.getResource(uri);
      if(result!=null) {
          return result;
      }
      result = super.createResource(uri);
      if(result==null) {
          // never return null resource
          result = xmiResourceFactory.createResource(uri);
      }
      if(GlobalResourceRegistry.hasGlobalURIScheme(uri)) {
        try {
            globalResourceRegistry.registerResource(uri, result);
        } catch (GlobalResourceException e) {
             org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.SEVERE, "createResource", e);
        } 
      } 
      return result;
    }  
    
    
           
}
