package org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.mb_xmi;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;
import java.util.logging.Level;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.ClassNotFoundException;
import org.eclipse.emf.ecore.xmi.PackageNotFoundException;
import org.eclipse.emf.ecore.xmi.XMIException;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.ModelUtil;
import org.eclipse.mddi.modelbus.adapter.infrastructure.model_manipulation.XmiUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * @author P. Sriplakich
 * 
 * 
 */
public class MBModelDeserializer {

    public static final String XMI_NS = "http://www.omg.org/XMI";

    public static DocumentBuilder docBuilder = null;
    static {
        try {
            docBuilder = DocumentBuilderFactory.newInstance()
                    .newDocumentBuilder();
        } catch (Exception e) {
            throw new RuntimeException("XML Parser cannot be initialized", e);
        }

    }

    /**
     * get the EClass from the XML element tag name, or from the 'xmi:type'
     * attribute of the XML element.
     * 
     * @throws PackageNotFoundException
     * @throws ClassNotFoundException
     */
    EClass getType(Element xmlElem) throws IOException {
        String typeString = null;
        if (xmlElem.hasAttribute("xmi:type")) {
            typeString = xmlElem.getAttribute("xmi:type");
        } else {
            typeString = xmlElem.getTagName();
        }
        String prefix = getNsPrefix(typeString);
        String className = getLocalName(typeString);
        String uri = lookupNamespace(xmlElem, prefix);
        EPackage p = getPackageFromURI(uri);
        if (p == null) {
            throw new IOException("PackageNotFound uri=" + uri + " element="
                    + xmlElem);
        }
        EClass c = (EClass) (EClass) p.getEClassifier(className);
        if (c == null) {
            throw new IOException("ClassNotFound " + className + "element="
                    + xmlElem);
        }
        return c;
    }


    /**
     * The format used for deserializing
     */
    String format = "EMF";

    Resource resource;

    IdTable idTable = null;

    Collection createdModelElements = new Vector();

    // Information about links to be resolved after all elements have been
    // created.
    List links = new Vector();

    public MBModelDeserializer() {
        this(new MBXmiResource(URI.createURI("uri1")));
    }

    public MBModelDeserializer(Resource r) {
        this.resource = r;
        idTable = IdTableRegistry.getIdTable(r.getResourceSet());
    }

    public void deserialize(InputStream in) throws SAXException, IOException {
        // long time;
        // time = System.currentTimeMillis();

        Document doc = XmiUtil.parseXml(in);
        
        // time = System.currentTimeMillis() - time;
        // System.out.println(".... [DESER]->DOM t=" +time + " (txt size="
        // +xmi.length() +")");
        // time = System.currentTimeMillis();

        createModelElementsFromRootXml(doc.getDocumentElement());
        resolveLinks(links, resource);

        // time = System.currentTimeMillis() - time;
        // System.out.println(".... [DESER]->Model t=" +time +" (elems="
        // +createdModelElements.size() +")");

    }

    public void deserialize(String xmi) throws SAXException, IOException {
        ByteArrayInputStream in = new ByteArrayInputStream(xmi.getBytes());
        deserialize(in);
    }

    /**
     * This method enables us to extract the model that is encoded in a XML
     * subtree
     * 
     * @param rootElem
     * @throws XMIException
     * 
     * 
     */
    public void createModelElementsFromRootXml(Element rootElem)
            throws IOException {
        String tag = getLocalName(rootElem.getTagName());
        if (!tag.equals("XMI")) {
            throw new IOException("Expected xmi:XMI element at root. Please check the XMI format "
                    + resource.getURI());
        }
        NodeList list = rootElem.getChildNodes();
        for (int i = 0; i < list.getLength(); i++) {
            if (list.item(i) instanceof Element) {
                Element xmlElem = (Element) list.item(i);
                EObject o = createModelElementFrom(xmlElem);
                resource.getContents().add(o);
            }
        }
    }

    /**
     * 
     * Create a new model element that is encoded in the specified XML element
     * 
     * @param xmlElem
     * @return the created model element
     * 
     * @throws XMIException
     * 
     */
    EObject createModelElementFrom(Element xmlElem) throws IOException {
        EClass c = getType(xmlElem);
        EObject o = EcoreUtil.create(c);
        // System.out.println("Created " +c.getName());
        createdModelElements.add(o);

        String xmiid = xmlElem.getAttribute("xmi:id");
        if (xmiid.equals("")) {
            printErrMessage("No id for " + xmlElem);
        } else {
            idTable.assignId(xmiid, o);
        }
        
//        Testing that the EGenericType object is volatile!! It can be modified by the EMF internal system
//        if(o.eClass().getName().equals("EGenericType")) {
//            RootLogger.getLogger().log(Level.FINE, "EGenericType " +o.hashCode() +" -> " +xmiid);
//        }
        
        setFeatures(o, xmlElem);
        return o;
    }

    EPackage getPackageFromURI(String uri) {
        EPackage p = (EPackage) EPackage.Registry.INSTANCE.getEPackage(uri);
        if (p == null) {
            try {
                Resource r = resource.getResourceSet().getResource(
                        URI.createURI(uri), true);
                ModelUtil.registerEPackages(r);
                p = (EPackage) EPackage.Registry.INSTANCE.getEPackage(uri);
            } catch (Exception e) {
                 org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.SEVERE, "Package not found " + uri + " " + e);
                return null;
            }
        }
        return p;
    }

    /**
     * Set the features of the specified model element. The feature values are
     * extracted from the specified XML element.
     * 
     * @param o
     *            The model element to be updated
     * @param xmlElem
     * @throws XMIException
     * 
     */
    void setFeatures(EObject o, Element xmlElem) throws IOException {
        EClass c = o.eClass();

        // for all XML attributes
        NamedNodeMap map = xmlElem.getAttributes();
        for (int i = 0; i < map.getLength(); i++) {
            Node n = map.item(i);
            String attName = n.getNodeName();
            String attNsPrefix = getNsPrefix(attName);
            String attLocalName = getLocalName(attName);
            String attValue = n.getNodeValue();

            if (attNsPrefix.length() > 0) {
                if (attName.equals("xmi:id")) {
                    // do nothing
                } else if (attName.equals("xmi:type")) {
                    // do nothing
                } else {
                    printErrMessage("Unknown attribute " + attName);
                }
            } else { // values of EAttribute
                EAttribute watt = (EAttribute) c.getEStructuralFeature(attName);
                // ModelUtil.findStructuralFeature(c, attName);
                if (watt == null) {
                    printErrMessage("Unknown attribute " + attName);
                } else {
                    Object value = EcoreUtil.createFromString(watt
                            .getEAttributeType(), attValue);
                    o.eSet(watt, value);
                }
            }
        }

        // for all XML subelements
        NodeList list = xmlElem.getChildNodes();
        for (int i = 0; i < list.getLength(); i++) {
            if (!(list.item(i) instanceof Element))
                continue;
            Element subXmlElem = (Element) list.item(i);
            String elemName = subXmlElem.getTagName();
            EStructuralFeature f = c.getEStructuralFeature(elemName);
            // ModelUtil.findStructuralFeature(c, elemName);
            if (f == null) {
                printErrMessage("Unknown feature " + elemName);
            }
            if (f instanceof EAttribute) {
                EAttribute watt = (EAttribute) f;
                String text = subXmlElem.getFirstChild().getNodeValue();
                Object value = EcoreUtil.createFromString(watt
                        .getEAttributeType(), text);
                if (watt.isMany()) {
                    List l = (List) o.eGet(watt, false);
                    l.add(value);
                } else {
                    o.eSet(watt, value);
                }

            } else if (f instanceof EReference) {
                EReference r = (EReference) f;
                if (subXmlElem.hasAttribute("xmi:action")
                        && subXmlElem.hasAttribute("href")) {
                    updateReference(o, r, subXmlElem);

                } else if (subXmlElem.hasAttribute("href")) {
                    String idref = subXmlElem.getAttribute("href");
                    EClass type = getType(subXmlElem);
                    links.add(new Link(r, o, idref, type));
                } else {
                    EObject value = createModelElementFrom(subXmlElem);
                    if (r.isMany()) {
                        List l = (List) o.eGet(r);
                        l.add(value);
                    } else {
                        o.eSet(r, value);
                    }
                }
            }
        }
    }

    /**
     * 
     * This method manipulate the XMI difference
     * 
     * @param o
     * @param r
     * @param refXmlElem
     * 
     * 
     */
    void updateReference(EObject o, EReference r, Element refXmlElem) {
        String action = refXmlElem.getAttribute("xmi:action");
        String id = refXmlElem.getAttribute("href");
        EObject referencedElem = resource.getEObject(id);
        List l = (List) o.eGet(r, false);
        if (action.equals("add")) {
            l.add(referencedElem);
        } else if (action.equals("remove")) {
            l.remove(referencedElem);
        }
    }

    static String getNsPrefix(String colontag) {
        return (colontag.indexOf(':') < 0) ? "" : colontag.substring(0,
                colontag.indexOf(':'));
    }

    static String getLocalName(String colontag) {
        return colontag.substring(colontag.indexOf(':') + 1);
    }

    /**
     * 
     * Returns the namespace URI
     * 
     * @param xmlElem
     * @param prefix
     * 
     */
    static String lookupNamespace(Element xmlElem, String prefix) {
        String attName = (prefix.length() == 0) ? "xmlns" : "xmlns:" + prefix;
        Element topElem = xmlElem;
        String uri = topElem.getAttribute(attName);
        while (uri.equals("") && topElem.getParentNode() instanceof Element) {
            topElem = (Element) topElem.getParentNode();
            uri = topElem.getAttribute("xmlns:" + prefix);
        }
        return uri;
    }

    static void printErrMessage(String s) {
         org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.SEVERE, s);
    }

    /**
     * Returns the collection containing all created model elements (flatten)
     */
    public Collection getCreatedModelElements() {
        return createdModelElements;
    }

    static class Link {
        EReference eReference;

        EObject owner;

        String href;

        EClass type;

        Link(EReference r, EObject owner, String href, EClass type) {
            this.eReference = r;
            this.owner = owner;
            this.href = href;
            this.type = type;
        }

    }

    /**
     * @return Returns the format.
     */
    public String getFormat() {
        return format;
    }

    /**
     * @param format
     *            The format to set.
     */
    public void setFormat(String format) {
        this.format = format;
    }

    /**
     * Resolves the links once all model elements have been created
     * 
     * @throws IOException
     * 
     * 
     */
    void resolveLinks(Collection links, Resource res) throws IOException {
        for (Iterator it = links.iterator(); it.hasNext();) {
            Link link = (Link) it.next();
            String id = MBModelDeserializer.getIdPart(link.href);
            String uri = MBModelDeserializer.getUriPart(link.href);
            EObject value = null;
            if (uri == null || resource.getURI().toString().equals(uri)) {
                // This is the element in the same resource
                value = resource.getEObject(id);
                // if the reference can not be resolved by the resource,
                // then try to use IdTable
                if(value ==null) {
                    value = IdTableRegistry.getIdTable(resource).getModelElement(id);
                }

            } else {
                // This is the element in another resource
                value = EcoreUtil.create(link.type);
                ((InternalEObject) value)
                        .eSetProxyURI(URI.createURI(link.href));
            }

            if (value == null) {
                 org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.SEVERE, "Invalid href " + link.href);
                // throw new IOException("Invalid href " +link.href);
            } else {
                if (link.eReference.isMany()) {
                    List values = (List) link.owner
                            .eGet(link.eReference, false);
                    values.add(value);
                } else {
                    link.owner.eSet(link.eReference, value);
                }
            }
        }
        links.clear();
    }

    public static String getUriPart(String href) {
        int i = href.indexOf((int) '#');
        if (i == -1) {
            return null;
        }
        return href.substring(0, i);
    }

    public static String getIdPart(String href) {
        int i = href.indexOf((int) '#');
        if (i == -1) {
            return href;
        }
        return href.substring(i + 1, href.length());
    }

    /**
     * @return Returns the resource.
     */
    public Resource getResource() {
        return resource;
    }

    /**
     * @param resource
     *            The resource to set.
     */
    public void setResource(MBXmiResource resource) {
        this.resource = resource;
    }
}
