/*
 * $RCSfile: DefaultMarshaler.java,v $
 * $Date: 2007/06/09 02:33:10 $
 * $Revision: 1.8 $
 * $Author: xblanc $
 */

/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

/*
 * ModelMarshaller.java
 * 
 * @author Prawee Sriplakich
 * @version $Revision: 1.8 $ $Date: 2007/06/09 02:33:10 $
 */
package org.eclipse.mddi.modelbus.adapter.infrastructure.transport.ws;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.logging.Level;

import javax.xml.soap.Detail;
import javax.xml.soap.DetailEntry;
import javax.xml.soap.Name;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPFault;

import org.apache.axis.message.PrefixedQName;
import org.eclipse.emf.common.util.URI;
import org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.DefaultModelSerializerFactory;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelElementReference;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializer;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.ModelSerializerFactory;
import org.eclipse.mddi.modelbus.adapter.infrastructure.serialize.SerializedXmiDocument;
import org.eclipse.mddi.modelbus.adapter.infrastructure.transport.Repository;
import org.eclipse.mddi.modelbus.adapter.user.ModelingServiceError;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelBusCommunicationException;
import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelTypeMismatchException;
import org.eclipse.mddi.modelbus.description.abstract_.AbstractFactory;
import org.eclipse.mddi.modelbus.description.abstract_.EnumerationType;
import org.eclipse.mddi.modelbus.description.abstract_.Error;
import org.eclipse.mddi.modelbus.description.abstract_.ModelType;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.abstract_.PrimitiveType;
import org.eclipse.mddi.modelbus.description.abstract_.Type;
import org.eclipse.mddi.modelbus.description.abstract_.impl.AbstractFactoryImpl;
import org.w3c.dom.Element;

import sun.misc.BASE64Decoder;
import sun.misc.BASE64Encoder;

/**
 * 
 * This class manage the marshaling of service parameters in SOAP messages.
 * 
 * This is an example of a SOAP message
 * 
 * 
 * <modelbus:SERVICE-NAME xmlns:modelbus="http://www.eclipse.org/mddi/modelbus"
 * modelbus:modelSerializer="DefaultModelSerializer" >
 * 
 * <modelbus:Scope> <modelbus:Resource uri="RESOURCE1.XMI"> ...... XMI data
 * encoded as a String ........... </modelbus:Resource> ..... There can be more
 * model resources </modelbus:Scope>
 * 
 * <modelbus:PARAM1 encoding="modelbus:ModelEncoding"> <modelbus:Ref
 * uri="RESOURCE1.XMI" fragment="_Ri03NgRkEdqunoyive1chA"/> </modelbus:PARAM1>
 * 
 * <modelbus:PARAM2 encoding="modelbus:ModelEncoding"> <modelbus:Ref
 * uri="RESOURCE1.XMI" fragment="_Ri03NgRkEdqunoyive1chA"/> </modelbus:PARAM2>
 * 
 * ...... more parameters
 * 
 * </modelbus:SERVICE-NAME> </soapenv:Body>
 * 
 * 
 * 
 * @author P. Sriplakich, Andrey Sadovykh (LIP6)
 * 
 */
public class DefaultMarshaler implements Marshaler {

    
    public static final String ERROR_INFO_TAG = "ErrorInfo";

    public static final String ERROR_FULL_TAG = "modelbus:ModelingServiceError";

    public static final String ERROR_TAG = "ModelingServiceError";

    public static final String SCOPE_TAG = "modelbus:Scope";
    
    
    public static final String ITEM_TAG = "modelbus:item";
    
    public static final String REF_TAG = "modelbus:Ref";
    
    public static final String RESOURCE_TAG = "modelbus:resource";

    /**
     * The property for specifying the model serializer used for sending the
     * models with the request message. This property enables the received to
     * choose the compatible model serializer for deserializeing the received
     * models.
     */
    public static final String MODEL_SERIALIZER_PROP = "modelbus:modelSerializer";

    
    
    public static final String ENCODING = "encoding";
    
    public static final String ENCODING_MULTIVALUES = "modelbus:multi-values";
    
    /**
     * The encoding property for specifying that the model is sent by reference,
     * i.e., the model's URI is sent.
     */
    public static final String ENCODING_MODEL_URI = "modelbus:Model-URI";
    
    public static final String ENCODING_MODEL = "modelbus:Model"; 

    /*
     * ModelBus Primitive Types
     */

    public static final String MB_STRING = "string";

    public static final String MB_INTEGER = "integer";

    public static final String MB_BOOLEAN = "boolean";

    public static final String MB_DOUBLE = "double";

    public static final String MB_BINARY = "binary";




    ModelSerializerFactory modelSerializerFactory;

    ModelSerializer modelSerializer;
    
    
    /**
     * The Repository client manages the pass by reference
     */
    Repository repositoryClient;
    

    public DefaultMarshaler() {
    }

    /**
     * Marshalls a parameter value 
     * and placed the result in the SOAP Element <code>topElem</code>.
     * 
     * <p>The value of each parameter is placed in a seperate XML tag.
     * For example, let <code>topElem</code> 
     * be the SOAP element &lt;my_service&gt;...&lt;/my_service&gt;.
     * Let the parameter name be "param1".
     * The parameter value will be placed in the SOAP element &lt;param1&gt;...&lt;/param1&gt; inside the element my_service.
     * 
     * <br><br> For model type, it must delegate to marshalModel(...)
     * 
     * @param p the description of the parameter, as defined in ModelBus metamodel.
     * @param parameterValue the value of the parameter to be marshalled
     * @param topElem the SOAPElement where the value of the parameter will be placed. 
     * Its tag name is the service name
     * @throws SOAPException
     * 
     */
    public void marshal(Parameter p, Object parameterValue, SOAPElement topElem)
            throws SOAPException {
        // if parameter value is null, do nothing
        if (parameterValue == null) {
            return;
        }
        SOAPElement paramSoapElem = topElem.addChildElement(p.getName());
        if (parameterValue instanceof Object[]) {
            paramSoapElem.setAttribute(ENCODING, ENCODING_MULTIVALUES);
            Object[] values = (Object[]) parameterValue;
            for (int i = 0; i < values.length; i++) {
                Object singleValue = values[i];
                SOAPElement itemElem = paramSoapElem
                        .addChildElement(ITEM_TAG);
                marshallSingleValue(p, singleValue, itemElem);
            }
        } else {
            marshallSingleValue(p, parameterValue, paramSoapElem);
        }
    }

    void marshallSingleValue(Parameter p, Object value, SOAPElement elem)
            throws SOAPException {
        Type t = p.getType();
        if (t instanceof ModelType) {
            if (value instanceof URI) {
                String s = value.toString();
                setEncodingStyle(p, elem);
                elem.addTextNode(s);
                elem.setAttribute(ENCODING, ENCODING_MODEL_URI);

            } else {
                ModelElementReference[] refs = getModelSerializer().getReferences(p, value);
                for (int i = 0; i < refs.length; i++) {
                    Element elem1 = elem.addChildElement(REF_TAG);
                    elem1.setAttribute("uri", refs[i].getUri());
                    elem1.setAttribute("fragment", refs[i].getRef());
                }
                elem.setAttribute(ENCODING, ENCODING_MODEL);
            }
        }

        else if (t instanceof EnumerationType) {
            String s = value.toString();
            setEncodingStyle(p, elem);
            elem.addTextNode(s);
        } else if (t.getName().equalsIgnoreCase("binary")) {
            // check type of value
            if (!(value instanceof InputStream))
                throw new SOAPException(
                        "Binary parameters should be of java.io.InputStream type");
            // marshal binary data from an Output stream
            BASE64Encoder encoder = new BASE64Encoder();
            InputStream is = (InputStream) value;
            try {
              byte[] buffer = new byte[is.available()];            
              is.read(buffer);
              String s = encoder.encode(buffer);
              setEncodingStyle(p, elem);
              elem.addTextNode(s);
            } catch(IOException e) {
                throw new SOAPException(e);
            }
            


        } else {
            // for other primitive types
            String s = value.toString();
            setEncodingStyle(p, elem);
            elem.addTextNode(s);
        }
    }

    public static void setEncodingStyle(Parameter p, SOAPElement elem)
            throws SOAPException {
        Type t = p.getType();
        String typeName = t.getName();
        if (t instanceof PrimitiveType) {
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                elem.setAttribute(ENCODING, "xsd:string");
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                elem.setAttribute(ENCODING, "xsd:integer");
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                elem.setAttribute(ENCODING, "xsd:float");
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                elem.setAttribute(ENCODING, "xsd:boolean");
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                elem.setAttribute(ENCODING, "xsd:base64Binary");
            } else {
                // this is a user-defined primitive type
                elem.setEncodingStyle(t.getName());
            }
        } else if (t instanceof EnumerationType) {
            elem.setAttribute(ENCODING, "modelbus:Enumeration");
        }
    }

    /**
     * Extracts the parameter value from the SOAP element <code>topElem</code>. 
     * 
     * <br><br> For model type, it must delegate to unmarshalModel(...)
     * 
     * @param p the description of the parameter, as defined in ModelBus metamodel.
     * @param topElem the SOAPElement from which the value of the parameter will be extracted.
     * Its tag name is the service name
     * @return the extracted value
     * @throws SOAPException
     */
    public Object unmarshal(Parameter p, SOAPElement topElem) throws SOAPException, ModelBusCommunicationException {
        SOAPElement elem = SoapUtil.getChildByLocalName(topElem, p.getName());
        // if the element is not found in SOAP message, then the parameter value
        // is null
        if (elem == null) {
            return null;
        }
        if (ENCODING_MULTIVALUES.equals(elem.getAttribute(ENCODING))) {
            List<Object> values = new Vector();
            for(Iterator<Object> it = elem.getChildElements(); it.hasNext(); ) {
                Object o = it.next();
                if (o instanceof SOAPElement) {
                    SOAPElement subElem = (SOAPElement) o;
                    Object v = unmarshallSingleValue(p, subElem);
                    values.add(v);
                }
            }
            Object[] array = getArrayForMultiValueParameter(p, values.size());
            return values.toArray(array);
        } else {
            return unmarshallSingleValue(p, elem);
        }
    }
    

    /**
     * Creates array for storing parameter values
     * 
     * @param p
     * @param size
     * @return
     */
    static Object[] getArrayForMultiValueParameter(Parameter p, int size) {
        Type t = p.getType();
        if (t instanceof PrimitiveType) {
            String typeName = t.getName();
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                return new String[size];
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                return new Integer[size];
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                return new Double[size];
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                return new Boolean[size];
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                return new ByteArrayInputStream[size];
            } else {
                return null;
            }
        } else if (t instanceof EnumerationType) {
            return new String[size];
        } else if (t instanceof ModelType) {
            return new Collection[size];
        }
        return null;
    }

    Object unmarshallSingleValue(Parameter p, SOAPElement elem) throws ModelBusCommunicationException {
        Type t = p.getType();
        if (t instanceof ModelType) {
            String encoding = elem.getAttribute(ENCODING);
            if (ENCODING_MODEL_URI.equals(encoding)) {
                String uri = elem.getValue();
                 org.eclipse.mddi.modelbus.adapter.infrastructure.RootLogger.getLogger().log(Level.INFO,"Unmarshaling Model URI " +uri);
                if(repositoryClient==null) {
                    throw new ModelBusCommunicationException("No repository client");
                }
                List model;
                model = repositoryClient.getModel(uri);
                return model;

            } else {

                Vector v = new Vector();
                for (Iterator it = elem.getChildElements(); it.hasNext();) {
                    SOAPElement elem1 = (SOAPElement) it.next();
                    ModelElementReference r = new ModelElementReference();
                    v.add(r);
                    r.setUri(elem1.getAttribute("uri"));
                    r.setRef(elem1.getAttribute("fragment"));
                }
                ModelElementReference[] refs = (ModelElementReference[]) v
                        .toArray(new ModelElementReference[v.size()]);
                Object o = getModelSerializer().dereference(p, refs);
                return o;
            }

        } else if (t instanceof EnumerationType) {
            String value = elem.getValue();
            return value;
        } else {
            String typeName = t.getName();
            String value = elem.getValue();
            if (typeName.equalsIgnoreCase(MB_STRING)) {
                return value;
            } else if (typeName.equalsIgnoreCase(MB_INTEGER)) {
                return new Integer(Integer.parseInt(value));
            } else if (typeName.equalsIgnoreCase(MB_DOUBLE)) {
                return new Double(Double.parseDouble(value));
            } else if (typeName.equalsIgnoreCase(MB_BOOLEAN)) {
                return Boolean.valueOf(value);
            } else if (typeName.equalsIgnoreCase(MB_BINARY)) {
                BASE64Decoder decoder = new BASE64Decoder();
                byte[] buffer;
                try {
                    buffer = decoder.decodeBuffer(value);
                } catch (IOException e) {
                    throw new ModelTypeMismatchException("decoding error", e);
                }
                return new ByteArrayInputStream(buffer);
            }
        }
        throw new ModelTypeMismatchException("Deserialize error, unknown Type " + t);
    }

    public void marshalModelingServiceError(ModelingServiceError err,
            SOAPFault f) throws SOAPException {
        Name n1 = new PrefixedQName("http://www.eclipse.org/mddi/modelbus",
                ERROR_INFO_TAG, "modelbus");

        f.setFaultCode(ERROR_TAG);
        f.setFaultString(err.getName());
        if (err.getInfo() != null) {
            Detail detail = f.addDetail();
            DetailEntry entry = detail.addDetailEntry(n1);
            if (err.getErrorType() == null) {

                entry.addTextNode(err.getInfo().toString());
            } else {

                    marshalModelingServiceError(err.getErrorType(), err
                            .getInfo(), entry);
   
            }

        }
    }

    public void marshalModelingServiceError(Error errorType, Object info,
            SOAPElement element) throws SOAPException {

        AbstractFactory factory = new AbstractFactoryImpl();
        Parameter parameter = factory.createParameter();
        parameter.setUpper(errorType.getUpper());
        parameter.setName(errorType.getName());
        parameter.setType(errorType.getType());

        List<Parameter> params = new Vector();
        params.add(parameter);

        Map values = new Hashtable();
        values.put(errorType.getName(), info);
        marshal(params, values, element);

    }

    public Object unmarshalModelingServiceError(Error errorType,
            SOAPElement element) throws SOAPException, ModelBusCommunicationException {
        AbstractFactory factory = new AbstractFactoryImpl();
        Parameter parameter = factory.createParameter();
        parameter.setUpper(errorType.getUpper());
        parameter.setName(errorType.getName());
        parameter.setType(errorType.getType());

        List<Parameter> params = new Vector();
        params.add(parameter);

        Map valueMap = unmarshal(params, element);
        return valueMap.get(errorType.getName());
    }

    /**
     * Identifies whether this SOAP fault contains ModelingServiceError or other
     * kinds of errors.
     * 
     * @param f
     * @return
     * 
     */
    public static boolean isModelingServiceError(SOAPFault f) {
        String fcode = f.getFaultCode();
        return (fcode.equals(ERROR_FULL_TAG) || fcode.equals(ERROR_TAG));
    }

    public static String getModelingServiceErrorName(SOAPFault f) {

        return f.getFaultString();
    }

    public ModelingServiceError unmarshalModelingServiceError(Error errorType,
            SOAPFault f) throws SOAPException {
        String errName = f.getFaultString();

        Name n1 = new PrefixedQName("http://www.eclipse.org/mddi/modelbus",
                ERROR_INFO_TAG, "modelbus");
        Iterator it = f.getChildElements();
        while (it.hasNext()) {
            SOAPElement el = (SOAPElement) it.next();
            Iterator it2 = el.getChildElements(n1);
            while (it2.hasNext()) {
                SOAPElement infoEl = (SOAPElement) it2.next();
                if (errorType == null) {
                    String info = infoEl.getValue();
                    return new ModelingServiceError(errName, info);
                } else {
                    Object info;
                    try {
                        info = unmarshalModelingServiceError(errorType, infoEl);
                    } catch (Exception e) {
                        throw new SOAPException(e);
                    }
                    return new ModelingServiceError(errorType, info);

                }
            }
        }
        return new ModelingServiceError(errName, null);
    }

    /**
     * Set namespace declarations used for defining encodings
     * 
     * @param body
     * @throws SOAPException
     */
    public static void setNamespaces(SOAPBody body) throws SOAPException {
        body.addNamespaceDeclaration("xsd", "http://www.w3.org/1999/XMLSchema");
        body.addNamespaceDeclaration("xsi",
                "http://www.w3.org/1999/XMLSchema-instance");
        body.addNamespaceDeclaration("modelbus",
                "http://www.eclipse.org/mddi/modelbus");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.Marshaler#marshal(org.eclipse.mddi.modelbus.description.abstract_.Parameter[],
     *      java.lang.Object[], javax.xml.soap.SOAPElement)
     */
    public void marshal(List<Parameter> params, Map<String, Object> parameterValues,
            SOAPElement topElem) throws SOAPException {

        topElem.setAttribute(MODEL_SERIALIZER_PROP, getModelSerializerFactory()
                .getName());

        // if (parameterValues.length != params.length) {
        // throw new ModelTypeMismatchException(
        // "Bad number of parameters, got " + params.length
        // + ", expected parameters: "
        // + DescriptionUtil.getParameterNameList(params));
        // }
        SOAPElement scopeElement = topElem.addChildElement(SCOPE_TAG);

        SerializedXmiDocument[] docs = getModelSerializer().serialize(params, parameterValues);
        for (int i = 0; i < docs.length; i++) {
            SOAPElement elem1 = scopeElement
                    .addChildElement( RESOURCE_TAG);
            elem1.setAttribute("uri", docs[i].getUri());
            elem1.addTextNode(docs[i].getXmi());
        }
        for (Parameter p : params) {
            Object value = parameterValues.get(p.getName());
            if (value == null) {
                RootLogger.getLogger().log( Level.WARNING 
                        ,"Missing parameter "
                        + p.getName());
            }
            marshal(p, value, topElem);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.mddi.modelbus.adapter.infrastructure.transport.marshal.ws.Marshaler#unmarshal(org.eclipse.mddi.modelbus.description.abstract_.Parameter[],
     *      javax.xml.soap.SOAPElement)
     */
    public Map<String, Object> unmarshal(List<Parameter> params, SOAPElement topElem)
            throws SOAPException, ModelBusCommunicationException {
        if (params == null)
            return null;

        SOAPElement scopeElement = SoapUtil.getChildByFullName(topElem,
                SCOPE_TAG);
        Vector<SerializedXmiDocument> v = new Vector();
        for (Iterator it = scopeElement.getChildElements(); it.hasNext();) {
            SOAPElement resElem = (SOAPElement) it.next();
            SerializedXmiDocument doc = new SerializedXmiDocument();
            doc.setUri(resElem.getAttribute("uri"));
            doc.setXmi(resElem.getValue());
            v.add(doc);
        }
        SerializedXmiDocument[] docs = (SerializedXmiDocument[]) v.toArray(new SerializedXmiDocument[v
                .size()]);
        getModelSerializer().deserialize(docs);
        Map<String, Object> valueMap = new Hashtable();
        for (Parameter p : params) {
            Object value = unmarshal(p, topElem);
            if(value!=null) {
              valueMap.put(p.getName(), value);
            } else {
               RootLogger.getLogger().log (Level.WARNING, 
                       "Null parameter " + p.getName());
            }
        }
        return valueMap;
    }

    /**
     * @return Returns the serializer.
     */
    public ModelSerializer getModelSerializer() {
        if (modelSerializer == null) {

            modelSerializer = getModelSerializerFactory()
                    .createModelSerializer();
        }
        return modelSerializer;
    }

    /**
     * @param serializer
     *            The serializer to set.
     */
    public void setModelSerializerFactory(ModelSerializerFactory f) {
        modelSerializerFactory = f;
    }

    public ModelSerializerFactory getModelSerializerFactory() {
        if (modelSerializerFactory == null) {
            modelSerializerFactory = new DefaultModelSerializerFactory();
        }
        return modelSerializerFactory;
    }

    /**
     * @return Returns the repositoryClient.
     */
    public Repository getRepositoryClient() {
        return repositoryClient;
    }

    /**
     * @param repositoryClient The repositoryClient to set.
     */
    public void setRepositoryClient(Repository repositoryClient) {
        this.repositoryClient = repositoryClient;
    }

}
