/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

package org.eclipse.mddi.modelbus.toolkit.adapter.generator;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;
import java.util.Map.Entry;

import org.eclipse.emf.common.util.EList;
import org.eclipse.mddi.modelbus.description.abstract_.DirectionKind;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingServiceInterface;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.toolkit.DescriptionModelLoader;
import org.eclipse.mddi.modelbus.toolkit.adapter.generator.exception.BadModeException;


/**
 * 
 * Generator of the interfaces and stubs
 * 
 * @author Fateh Bekhouche, Fatima Fadil, Nicolas Garandeau, Nils Henner, Saoussen Kraiem, Rmy-Christophe Schermesser, Marc Schwitzgubel
 *
 */
public class Generator {	

	/**
	 * Constants for the mode of the generator
	 */
	protected static final int INTERFACE_MODE = 0;
	protected static final int STUB_MODE = 1;
	
	/**
	 * Constants for the mode for the method generator
	 */
	protected static final int CONSUME_METHOD_TYPE = 0;
	protected static final int CONSUME_ASYNC_METHOD_TYPE = 1;
	protected static final int IS_RESULT_READY_METHOD_TYPE = 2;
	protected static final int GET_RESULT_METHOD_TYPE = 3;
	
	/**
	 * Constants for the name of the generated class, interface, method
	 */
	protected static final String STUB_POSTFIX_NAME = "Stub";
	protected static final String OUTPUT_HOLDER_NAME = "OutputHolder_";
	protected static final String CONSUME_METHOD_NAME = "consume_";
	protected static final String CONSUME_ASYNC_METHOD_NAME = "consumeAsync_";
	protected static final String IS_RESULT_READY_METHOD_NAME = "isResultReady_";
	protected static final String GET_RESULT_METHOD_NAME = "getResult_";
	
	/**
	 * Constant for the EPL licence
	 */
	protected static final String LICENCE = 
		"/*\n" +
		"* Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.\n" +
		"* All rights reserved.\n" +
		"*\n" +
		"* This software is published under the terms of the ModelBus Software License\n" +
		"* in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even\n" +
		"* the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n" +
		"* A copy of ModelBus Software License is provided with this distribution in\n" +
		"* doc/LICENSE.txt file.\n" +
		"*/\n";
	
	/**
	 * Constant for the CVS header
	 */
	protected static final String CVS = 
		"/*\n" +
		"* $RCSfile: Generator.java,v $\n" +
		"* $Date: 2006/01/31 16:55:33 $\n" +
		"* $Revision: 1.1 $\n" +
		"* $Author: andreys $\n" +
		"*/\n";

	/**
	 * The model loader
	 */
	protected DescriptionModelLoader modelLoader;

	/**
	 * The output holder classes
	 */
	protected Hashtable outputHolderClasses;
	
	/**
	 * The adapter name
	 */
	protected String adapterName;

	/**
	 * The generated mode
	 */
	protected int mode;
	
	/**
	 * 
	 * The default constructor
	 * 
	 * @param modelLoader 
	 */
	public Generator(DescriptionModelLoader modelLoader) {
		super();
		this.modelLoader = modelLoader; 
		outputHolderClasses = new Hashtable();
	}
	
	/**
	 * 
	 * Set the mode
	 * 
	 * @param mode The mode (see INTERFACE_MODE and STUB_MODE)
	 * @throws BadModeException
	 */
	protected void setMode(int mode) throws BadModeException {
		if((mode != INTERFACE_MODE) && (mode != STUB_MODE)) {
			throw new BadModeException();
		}
		this.mode = mode;
	}
	
	/**
	 * 
	 * Generate the adapter code
	 * 
	 * @param msi The ModelingServiceInterface to generate
	 * @return String The code of this ModelingServiceInterface
	 */
	protected String generateAdapterCode(ModelingServiceInterface msi) {
		StringBuffer sb = new StringBuffer();
		
		switch(mode) {
			case INTERFACE_MODE :
				adapterName = msi.getName();
				break;
			case STUB_MODE :
				adapterName = msi.getName() + STUB_POSTFIX_NAME;
				break;
			default :
				//CAN NOT HAPPEND
		}
		
		
		sb.append("import java.util.Collection;\n");
		sb.append("import java.util.Properties;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelBusCommunicationException;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.consumer.ModelTypeMismatchException;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.consumer.NoToolAvailableException;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.consumer.ServiceUnknownException;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.infrastructure.DeploymentException;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.impl.AdapterStubImpl;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.AdapterStub;\n");
		sb.append("import org.eclipse.mddi.modelbus.adapter.user.consumer.GenericConsumer;\n\n");
		
		sb.append("/**\n");
		sb.append(" * " + adapterName + ".java\n");
		sb.append(" *\n");
		sb.append(" * @author auto-generated\n");
		sb.append(" *\n");
		sb.append(" * @version $Revision: 1.1 $ $Date: 2006/01/31 16:55:33 $\n");
		sb.append(" **/\n");
		sb.append("public ");
		switch(mode) {
			case INTERFACE_MODE :
				sb.append("interface");
				break;
			case STUB_MODE :
				sb.append("class");
				break;
			default :
				//CAN NOT HAPPEND
		}
		sb.append(" " + adapterName);
		
		switch(mode) {
			case STUB_MODE :
				sb.append(" implements " + msi.getName());
				break;
			default :
				//Nothind to do
		}
				
		sb.append(" {\n");

		switch(mode) {
			case STUB_MODE :
				sb.append("\tpublic AdapterStub adapter;\n" +
						  "\tpublic GenericConsumer consumer;\n\n" +
						  "\t/**\n" +
						  "\t *\n" +
						  "\t * @param pro Adapter properties\n" +
						  "\t * @throws DeploymentException\n" +
						  "\t **/\n" +
						  "\tpublic " +
						  adapterName +
						  "(Properties pro) throws DeploymentException {\n" +
						  "\t\tadapter = new AdapterStubImpl(pro);\n" +
						  "\t\tconsumer = adapter.getGenericConsumer();\n" +
						  "\t}\n\n");
				break;
			default : 
				//Nothing to do
		}
		
		EList services = msi.eContents();
		
		for(int i=0 ; i<services.size() ; i++){
			ModelingService ms = (ModelingService) services.get(i);
			sb.append(generateService(ms, modelLoader.getTool().getName(), msi.getName()));
			sb.append("\n");
		}
		
		// }
		sb.append("}\n");
		
		return sb.toString();
	}
	
	/**
	 * int toto
	 */
	public int toto;
	
	/**
	 * 
	 * Generate the output holders
	 * 
	 * @param outputDirectory The output directory
	 * @param outputPackageName The output package name
	 * @throws IOException If an I/O error occurs
	 */
	protected void generateOutputHolders(String outputDirectory, String outputPackageName) throws IOException {	
		Set outputHolderClassesSet = outputHolderClasses.entrySet(); 
		for (Iterator iter = outputHolderClassesSet.iterator() ; iter.hasNext() ;) {
			StringBuffer sb = new StringBuffer();	
			OutputHolderClass classe = (OutputHolderClass)((Entry)iter.next()).getValue();
			
			sb.append(LICENCE);
			if(outputPackageName.compareTo("") != 0) {
				sb.append("\npackage " + outputPackageName + ";\n\n");
			}
			
			sb.append("/**\n");
			sb.append(" * " + classe.getName() + ".java\n");
			sb.append(" *\n");
			sb.append(" * @author auto-generated\n");
			sb.append(" *\n");
			sb.append(" * @version $Revision: 1.1 $ $Date: 2006/01/31 16:55:33 $\n");
			sb.append(" **/\n");
			// public class NAME {
			sb.append("public class ");
			sb.append(classe.getName());
			sb.append(" {\n\n");
			
			Vector params = classe.getParametres();
			
			for(int j=0 ; j<params.size() ; j++){
				Parameter parameter =(Parameter) params.elementAt(j);
				String pName = parameter.getName();
				String pType = TypeConversion.generateType(parameter);
				
				sb.append("\t/**\n");
				sb.append("\t * " + pType + " " + pName + "\n");
				sb.append("\t **/\n");
				// public PTYPE PNAME;
				sb.append("\tpublic ");
				sb.append(pType);
				sb.append(" ");
				sb.append(pName);
				sb.append(";\n\n");
			}
			
			// }
			sb.append("}\n");
			
			File file = new File(outputDirectory +
							     File.separator +
							     classe.getName() +
							     ".java");
			
			try {
				FileOutputStream fos = new FileOutputStream(file);
				fos.write(sb.toString().getBytes());
				fos.close();
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			}
		}
	}
 
	/**
	 * 
	 * Generate the methods
	 * 
	 * @param ms The modeling service 
	 * @param toolName The name of the tool
	 * @param modelingServiceInterfaceName The interface name
	 * @param methodType The method type (See CONSUME_METHOD_TYPE, CONSUME_ASYNC_METHOD_TYPE, IS_RESULT_READY_METHOD_TYPE, GET_RESULT_METHOD_TYPE)
	 * @return String The code of the method
	 */
	protected String generateMethod(ModelingService ms, String toolName, String modelingServiceInterfaceName, int methodType) {
		StringBuffer sb = new StringBuffer();
		
		InputHolder[] inputs = createInputHolders(ms);
		OutputHolder[] outputs = createOutputHolders(ms);
		
		String generatedInput = generateInput(inputs);
		String generatedOutput = generateOutput(outputs);
		
		String nomService = ms.getName();
		
		sb.append("\n\t/**\n");
		sb.append("\t *\n");
		sb.append("\t * ");
		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				sb.append(IS_RESULT_READY_METHOD_NAME);
				break;
			case CONSUME_ASYNC_METHOD_TYPE :
				sb.append(CONSUME_ASYNC_METHOD_NAME);
				break;
			case CONSUME_METHOD_TYPE :
				sb.append(CONSUME_METHOD_NAME);
				break;
			case GET_RESULT_METHOD_TYPE :
				sb.append(GET_RESULT_METHOD_NAME);
				break;
			default :
				//CAN NOT HAPPEN
		}
		sb.append(nomService + "\n");
		sb.append("\t *\n");
		
		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				//Nothing to do
				break;
			default :
				for(int i=0 ; i<inputs.length ; i++) {
					sb.append("\t * @param " + inputs[i].name + "\n");
				}
		}

		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				sb.append("\t * @return boolean\n");
				break;
			case CONSUME_ASYNC_METHOD_TYPE :
				break;
			default :
				sb.append("\t * @return " + generatedOutput + "\n");
		}
		sb.append("\t * @throws ModelBusCommunicationException\n");
		sb.append("\t * @throws ServiceUnknownException\n");
		sb.append("\t * @throws NoToolAvailableException\n"); 
		sb.append("\t * @throws ModelTypeMismatchException\n");
		sb.append("\t * @throws Exception\n");
		
		sb.append("\t **/\n");
		sb.append("\tpublic ");
		
		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				sb.append("boolean");
				break;
			case CONSUME_ASYNC_METHOD_TYPE :
				sb.append("void");
				break;
			default :
				sb.append(generatedOutput);
		}
		
		sb.append(" ");
		
		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				sb.append(IS_RESULT_READY_METHOD_NAME);
				break;
			case CONSUME_ASYNC_METHOD_TYPE :
				sb.append(CONSUME_ASYNC_METHOD_NAME);
				break;
			case CONSUME_METHOD_TYPE :
				sb.append(CONSUME_METHOD_NAME);
				break;
			case GET_RESULT_METHOD_TYPE :
				sb.append(GET_RESULT_METHOD_NAME);
				break;
			default :
				//CAN NOT HAPPEN
		}
		
		sb.append(nomService);
		sb.append("(");
		
		switch(methodType) {
			case IS_RESULT_READY_METHOD_TYPE :
				//Nothng to do
				break;
			default :
				sb.append(generatedInput);
		}
		
		sb.append(") ");
		sb.append("throws ModelBusCommunicationException, ServiceUnknownException, " +
				  "NoToolAvailableException, ModelTypeMismatchException, " +
				  "Exception ");
		
		switch(mode) {
			case INTERFACE_MODE :
				// ;
				sb.append(";");
				break;
			case STUB_MODE :
				String serviceName = toolName + "." + modelingServiceInterfaceName + "." + nomService;
				
				// {
				sb.append("{\n");
				sb.append("\t\tString serviceName = \"" + serviceName + "\";\n");
				
				switch(methodType) {
					case IS_RESULT_READY_METHOD_TYPE :
						sb.append("\t\treturn consumer.isResultReady(serviceName);\n");
						break;
					case GET_RESULT_METHOD_TYPE :
						//Nothing to do
						break;
					default :
						sb.append("\t\tObject[] inputs = new Object[" + inputs.length + "];\n");
						for(int i = 0 ; i<inputs.length ; i++) {
							sb.append("\t\tinputs[" + i + "] = " + inputs[i].getName() + ";\n");
						}
					
						switch(methodType) {
							case CONSUME_ASYNC_METHOD_TYPE :
								sb.append("\t\tconsumer.consumeAsync(serviceName, inputs);\n");
								break;
							default :
								//Nothing to do
						}
				}
				
				switch(methodType) {
					case IS_RESULT_READY_METHOD_TYPE :
						//Nothing to do
						break;
					case CONSUME_ASYNC_METHOD_TYPE :
						//Nothing to do
						break;
					default :
						sb.append("\t\tObject[] outputs = consumer.");
						switch(methodType) {
							case CONSUME_METHOD_TYPE :
								sb.append("consume(serviceName, inputs);\n");
								break;
							case GET_RESULT_METHOD_TYPE :
								sb.append("getResult(serviceName);\n");
								break;
							default :
								//CAN NOT HAPPEN
						}
						if(outputs.length == 1) {
							if(outputs[0].getType().compareTo("void") != 0) {
								sb.append("\t\treturn (" + outputs[0].getType() + ")outputs[0];\n");
							}
						} else {
							sb.append("\t\t" + outputs[0].getClassName() + " outputHolder = new " + outputs[0].getClassName() + "();\n");
							for(int i=0 ; i<outputs.length ; i++) {
								sb.append("\t\toutputHolder." + outputs[i].getName() + " = (" + outputs[i].getType() + ")outputs[" + i + "];\n");
							}
							sb.append("\t\treturn outputHolder;\n");
						}
				}
				sb.append("\t}\n");
				break;
			default :
				//CAN NOT HAPPEND
		}
		sb.append("\n");
		return sb.toString();
	}
	
	/**
	 * 
	 * Generate all the four services
	 * 
	 * @param ms The ModelingService
	 * @param ToolName The tool name
	 * @param ModelingInterfaceName  The modeling interface name
	 * @return String The code of the services
	 */
	protected String generateService(ModelingService ms, String ToolName, String ModelingInterfaceName) {
		StringBuffer sb = new StringBuffer();
		
		sb.append(generateMethod(ms, ToolName, ModelingInterfaceName, CONSUME_METHOD_TYPE));
		sb.append(generateMethod(ms, ToolName, ModelingInterfaceName, CONSUME_ASYNC_METHOD_TYPE));
		sb.append(generateMethod(ms, ToolName, ModelingInterfaceName, IS_RESULT_READY_METHOD_TYPE));
		sb.append(generateMethod(ms, ToolName, ModelingInterfaceName, GET_RESULT_METHOD_TYPE));
		
		return sb.toString();
	}
	
	/**
	 * 
	 * Create the input holder (parameters)
	 * 
	 * @param ms The modeling service
	 * @return InputHolder[] The parameters
	 */
	protected InputHolder[] createInputHolders(ModelingService ms) {
		EList params = ms.eContents();
		Vector paramsIN = new Vector();
		
		for(int i=0 ; i<params.size() ; i++) {
			Parameter parameter = (Parameter)params.get(i);
			int direction = parameter.getDirection().getValue();
			if(direction == DirectionKind.IN || direction == DirectionKind.INOUT) {
				paramsIN.add(new InputHolder(parameter));
			}
		}
		InputHolder[] retour = new InputHolder[paramsIN.size()];
		retour = (InputHolder[])paramsIN.toArray(retour);
		return retour;
	}
	
	/**
	 * 
	 * Generate parameters
	 * 
	 * @param inputHolders The parameters 
	 * @return String The code of the parameters
	 */
	protected String generateInput(InputHolder[] inputHolders) {
		StringBuffer sb = new StringBuffer();
		
		for(int i=0 ; i<inputHolders.length ; i++) {
			sb.append(inputHolders[i].getType() + " " + inputHolders[i].getName());
			if(i != inputHolders.length - 1) {
				sb.append(", ");
			}
		}
		
		return sb.toString();
	}

	/**
	 * 
	 * Create the output holders
	 *  
	 * @param ms The modeling service
	 * @return OutputHolder[] The output holders
	 */
	protected OutputHolder[] createOutputHolders(ModelingService ms) {
		EList params = ms.eContents();
		Vector paramsOUT = new Vector();
		
		for(int i=0 ; i<params.size() ; i++){
			Parameter parameter = (Parameter)params.get(i);
			int direction = parameter.getDirection().getValue();
			if(direction == DirectionKind.OUT || direction == DirectionKind.INOUT) {
				paramsOUT.add(parameter);
			}
		}
		
		if(paramsOUT.size() == 0) {
			OutputHolder[] oh = new OutputHolder[1];
			oh[0] = new OutputHolder();
			return oh;
		}
		if(paramsOUT.size() == 1) {
			OutputHolder[] oh = new OutputHolder[1];
			oh[0] = new OutputHolder((Parameter)paramsOUT.get(0));
			return oh;
		}
		OutputHolder[] retour = new OutputHolder[paramsOUT.size()];
		String className = OUTPUT_HOLDER_NAME + ms.getName();
		OutputHolderClass classeInterne = new OutputHolderClass(className, paramsOUT);
		outputHolderClasses.put(className, classeInterne);
		for(int i=0 ; i<retour.length ; i++) {
			Parameter current = (Parameter)paramsOUT.get(i);
			retour[i] = new OutputHolder(className, current);
		}
		return retour;
	}
	
	/**
	 * 
	 * Generate the output holders
	 * 
	 * @param outputHolders The output holders 
	 * @return String The code of the output holders
	 */
	protected String generateOutput(OutputHolder[] outputHolders) {
		if(outputHolders.length == 1){
			return outputHolders[0].getType();
		}
		return outputHolders[0].getClassName();
	}

	/**
	 * 
	 * Generate the complete interface or stub
	 * 
	 * @param outputDirectory The output directory
	 * @param outputPackageName The package name
	 * @param mode The mode
	 * @throws BadModeException If the mode is not correct
	 * @throws IOException If an I/O error occurs
	 */
	protected void generateInternal(String outputDirectory,	String outputPackageName, int mode) throws BadModeException,
			IOException {
		for (int i = 0; i < modelLoader.getModelingServiceInterfaces().length; i++) {

			setMode(mode);
			
			String code = generateAdapterCode(modelLoader.getModelingServiceInterfaces()[i]);
			
			code = CVS + "\n" + code;
			
			code = LICENCE + "\n" + code;
			
			if(outputPackageName.compareTo("") != 0) {
				code = "\npackage " + outputPackageName + ";\n\n" + code;
			}

			File file = new File(outputDirectory + File.separator + adapterName
			                     + ".java");

			try {
				FileOutputStream fos = new FileOutputStream(file);
				fos.write(code.getBytes());
				fos.close();
			} catch (FileNotFoundException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * 
	 * Generate the complete adapter (stub + interface)
	 * 
	 * @param modelLoader The model loader
	 * @param outputDirectory The output directory
	 * @param outputPackageName The package name
	 * @throws IOException If an I/O error occurs
	 */
	public static void generateAdapter(DescriptionModelLoader modelLoader, String outputDirectory, String outputPackageName) throws IOException{
		Generator gen = new Generator(modelLoader);
		try {
			gen.generateInternal(outputDirectory, outputPackageName, INTERFACE_MODE);
			gen.generateInternal(outputDirectory, outputPackageName, STUB_MODE);
			gen.generateOutputHolders(outputDirectory, outputPackageName);
		} catch (BadModeException e) {
			//CAN NOT HAPPEN
		}		
	}
}