/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

package org.eclipse.mddi.modelbus.toolkit.adapter.validator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.mddi.modelbus.description.ModelBusPackaging;
import org.eclipse.mddi.modelbus.description.abstract_.Type;
import org.eclipse.mddi.modelbus.description.concrete.Tool;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.EException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.ETException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.MPException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.MSException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.MSIException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.MetaSException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.NameFormatException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.PException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.TException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.ToolException;
import org.eclipse.mddi.modelbus.toolkit.adapter.validator.exception.TypeFormatException;

/**
 * 
 * Check that description is correctly formed
 * 
 * @author Fateh Bekhouche, Fatima Fadil, Nicolas Garandeau, Nils Henner,
 *         Saoussen Kraiem, Remy-Christophe Schermesser, Marc Schwitzguebel
 */
public class Validation {

	/**
	 * 
	 * This method validates a description.
	 * 
	 * @param mp
	 * @throws MetaSException
	 * @throws MPException
	 * @throws MSIException
	 * @throws TException
	 * @throws EException
	 * @throws ETException
	 * @throws NameFormatException
	 * @throws MetaSException
	 * @throws MSException
	 * @throws PException
	 * @throws ToolException
	 */
	public static void globalValidation(ModelBusPackaging mp)
			throws MPException, MSIException, TException, EException,
			ETException, NameFormatException, MetaSException, MSException,
			PException, ToolException {
		EList mPContents = mp.getContent();
		int toolFound = 0;
		for (int i = 0; i < mPContents.size(); i++) {
			Object mPContent = mPContents.get(i);
			if (mPContent instanceof Tool) {
				toolFound++;
				ToolValidation.validateTool(mp);
			}
		}
		if (toolFound != 1) {
			throw new MPException(
					"ModelBusPackage :"
							+ mp.getName()
							+ " has more or less than one Tool, this ModelBusPackage" +
									" must have exactly one Tool");
		}
		globalValidationRec(mp);
		if (MPValidation.getToolsFound() != 1) {
			throw new MPException(
					"ModelBusPackage :"
							+ mp.getName()
							+ " has Tool(s) in its sons, only the first ModelBusPackage " +
									"must have exactly one Tool");
		}
		if (!MPValidation.isMsiFound()) {
			throw new MPException(
					"ModelBusPackage :"
							+ mp.getName()
							+ " has no ModelingServiceInterface, it must have at least one");
		}
	}

	/**
	 * 
	 * This method validates a description recursively when several
	 * ModelBusPackaging appear.
	 * 
	 * @param mp
	 * @throws MetaSException
	 * @throws MPException
	 * @throws MSIException
	 * @throws TException
	 * @throws EException
	 * @throws ETException
	 * @throws NameFormatException
	 * @throws MetaSException
	 * @throws MSException
	 * @throws PException
	 * @throws ToolException
	 */
	public static void globalValidationRec(ModelBusPackaging mp)
			throws MPException, MSIException, TException, EException,
			ETException, NameFormatException, MetaSException, MSException,
			PException, ToolException {
		MPValidation.validateMP(mp);
		MSIValidation.validateMSI(mp);
		EValidation.validationE(mp);
		ETValidation.validationET(mp);
		MetaSValidation.validateMetaS(mp);
		MSValidation.validationMS(mp);
		PValidation.validationParameter(mp);
		TValidation.validationT(mp);
		MSValidation.validationMS(mp);
		PValidation.validationParameter(mp);
	}

	/**
	 * 
	 * Check that the name attribute is correctly formed, throw
	 * NameFormatException otherwise
	 * 
	 * @param name
	 *            the name to check
	 * @param className
	 *            the class which contains the name to check
	 * @throws NameFormatException
	 *             exception thrown if the name is not correct
	 */
	public static void validateName(String name, String className)
			throws NameFormatException {

		final String[] javaReservedKeywords = { "abstract", "assert",
				"boolean", "break", "byte", "case", "catch", "char", "class",
				"const", "continue", "default", "do", "double", "else", "enum",
				"extends", "final", "finally", "float", "for", "goto", "if",
				"implements", "import", "instanceof", "int", "interface",
				"long", "native", "new", "package", "private", "protected",
				"public", "return", "short", "static", "strictfp", "super",
				"switch", "synchronized", "this", "throw", "throws",
				"transient", "try", "void", "volatile", "while" };

		// if name is missing
		if (name == null) {
			throw new NameFormatException("A " + className
					+ " has its name attribute missing");
		}// if name is empty
		else if (name.equals("")) {
			throw new NameFormatException("A " + className
					+ " has its name attribute empty");
		}// if name is not correct

		try {
			int intName = Integer.parseInt(name.substring(0, 1));// throw an
			// exception
			// if it's
			throw new NameFormatException(className + " name: " + name
					+ " begins with number " + intName);
		} catch (NumberFormatException e) {/* Ok */
			for (int i = 0; i < javaReservedKeywords.length; i++) {
				if (javaReservedKeywords[i].equals(name)) {
					throw new NameFormatException(
							"A "
									+ className
									+ " has its name which is a reserved keyword of Java");
				}
			}

			for (int i = 0; i < name.length(); i++) {
				if (!(name.charAt(i) == '_' // underscore
						|| name.charAt(i) == '$' // dollar
						|| (name.charAt(i) >= '0' && name.charAt(i) <= '9') // number
						|| (name.charAt(i) >= 'A' && name.charAt(i) <= 'Z') // capital
				// letter
				|| (name.charAt(i) >= 'a' && name.charAt(i) <= 'z'))) { // letter
					throw new NameFormatException(
							"A "
									+ className
									+ " has its name which is a word contening letters a-z A-Z, numbers 0-9, underscore or dollarsign");
				}
			}

		}
	}

	/**
	 * 
	 * Check that the type attribute is correctly formed, throw
	 * TypeFormatException otherwise
	 * 
	 * @param type
	 *            the type to check
	 * @param className
	 *            the class which contains the type to check
	 * @throws TypeFormatException
	 *             exception thrown if the type is not correct
	 */
	public static void validateType(Type type, String className)
			throws TypeFormatException {
		// if type missing
		if (type == null) {
			throw new TypeFormatException("A  " + className
					+ " has  its type attribute missing");
		}
		// if type is empty
		else if (type.getName().equals("")) {
			throw new TypeFormatException("A  " + className
					+ "  has  its type attribute missing");
		}
	}

}
