/*
 * Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.
 * All rights reserved.
 *
 * This software is published under the terms of the ModelBus Software License
 * in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * A copy of ModelBus Software License is provided with this distribution in
 * doc/LICENSE.txt file.
 */

package org.eclipse.mddi.modelbus.toolkit.adapter.generator;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Set;
import java.util.Vector;
import java.util.Map.Entry;

import org.eclipse.emf.common.util.EList;
import org.eclipse.mddi.modelbus.description.abstract_.DirectionKind;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingService;
import org.eclipse.mddi.modelbus.description.abstract_.ModelingServiceInterface;
import org.eclipse.mddi.modelbus.description.abstract_.Parameter;
import org.eclipse.mddi.modelbus.description.concrete.Tool;
import org.eclipse.mddi.modelbus.toolkit.adapter.generator.exception.BadModeException;

/**
 * 
 * StubGenerator of the interfaces and stubs
 * 
 * @author Fateh Bekhouche, Fatima Fadil, Nicolas Garandeau, Nils Henner,
 *         Saoussen Kraiem, Remy-Christophe Schermesser, Marc Schwitzguebel,
 *         Andrey Sadovykh (LIP6)
 *  
 */
public abstract class AbstractGenerator {

    /**
     * Constants for the name of the generated class, interface, method
     */

    static final String OUTPUT_HOLDER_PREFIX = "OutputHolder_";

    String FILE_POSTFIX = "";

    /**
     * Constant for the EPL licence
     */
    static final String LICENCE = "/*\n"
            + "* Copyright (c) 2002-2003 IST-2004-2006-511731 ModelWare - ModelBus.\n"
            + "* All rights reserved.\n"
            + "*\n"
            + "* This software is published under the terms of the ModelBus Software License\n"
            + "* in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even\n"
            + "* the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n"
            + "* A copy of ModelBus Software License is provided with this distribution in\n"
            + "* doc/LICENSE.txt file.\n" + "*/\n\n";

    /**
     * Constant for the CVS header
     */
    static final String CVS = "/*\n" + "* $RCSf" + "ile: $\n" + "* $Da"
            + "te: $\n" + "* $R" + "evision: $\n" + "* $A" + "uthor: $\n"
            + "*/\n\n";

    /**
     * The output holder classes
     */
    Hashtable outputHolderClasses;

    /**
     * The file name
     */
    String packageName;

    String outputDirectory;

    OutputStream fileOs;

    Tool toolDescription;

    /**
     * 
     * The default constructor
     * 
     * @param modelLoader
     */
    public AbstractGenerator(Tool t, OutputStream os) {
        super();
        outputHolderClasses = new Hashtable();
        this.toolDescription = t;
        this.fileOs = os;
    }

    /**
     * 
     * The default constructor
     * 
     * @param modelLoader
     */
    public AbstractGenerator(Tool t, String outputDirectory) {
        super();
        outputHolderClasses = new Hashtable();
        this.toolDescription = t;

        this.outputDirectory = outputDirectory;

    }

    public AbstractGenerator() {
        super();
    }
    /**
     * 
     * generateHeader
     * 
     * @param os
     * @throws IOException
     *  
     */
    protected void generateHeader() throws IOException {
        //add CVS header
        fileOs.write(CVS.getBytes());
        //add License Note
        fileOs.write(LICENCE.getBytes());

    }

    /**
     * 
     * generateFileComment
     * 
     * @param fileName
     * @param sb
     * @throws IOException
     *  
     */

    protected void generateFileComment() throws IOException {
        StringBuffer sb = new StringBuffer();
        sb.append("/**\n");
        sb.append(" * " + getFileName() + ".java\n");
        sb.append(" *\n");
        sb.append(" * @author auto-generated\n");
        sb.append(" *\n");
        sb.append(" * @version $R" + "evision: $ $D" + "ate: $\n");
        sb.append(" **/\n");
        fileOs.write(sb.toString().getBytes());
        fileOs.write(("\n").getBytes());
    }

    /**
     * 
     * Generate the adapter code
     * 
     * @param msi
     *            The ModelingServiceInterface to generate
     * @return String The code of this ModelingServiceInterface
     */
    protected void generateCode(ModelingServiceInterface msi)
            throws IOException {
        //add import declaration
        generateImport();

        generateFileComment();

        generateClassName();

        generateConstructor();

        EList services = msi.eContents();

        for (int i = 0; i < services.size(); i++) {
            ModelingService ms = (ModelingService) services.get(i);
            generateServiceMethods(ms, toolDescription.getName(), msi.getName());
            fileOs.write(("\n").getBytes());
        }

        generateAdditionalMethods(msi);
        fileOs.write(("}\n").getBytes());
    }

    /**
     * 
     * Generate the output holders
     * 
     * @param outputDirectory
     *            The output directory
     * @param outputPackageName
     *            The output package name
     * @throws IOException
     *             If an I/O error occurs
     */
    protected void generateOutputHolderFiles(String outputDirectory,
            String outputPackageName) throws IOException {
        Set outputHolderClassesSet = outputHolderClasses.entrySet();
        for (Iterator iter = outputHolderClassesSet.iterator(); iter.hasNext();) {
            StringBuffer sb = new StringBuffer();
            OutputHolderClass classe = (OutputHolderClass) ((Entry) iter.next())
                    .getValue();

            sb.append(LICENCE);
            if (outputPackageName.compareTo("") != 0) {
                sb.append("\npackage " + outputPackageName + ";\n\n");
            }

            sb.append("/**\n");
            sb.append(" * " + classe.getName() + ".java\n");
            sb.append(" *\n");
            sb.append(" * @author auto-generated\n");
            sb.append(" *\n");
            sb
                    .append(" * @version $Revision: 1.7 $ $Date: 2007/05/10 11:35:20 $\n");
            sb.append(" **/\n");
            // public class NAME {
            sb.append("public class ");
            sb.append(classe.getName());
            sb.append(" {\n\n");

            Vector params = classe.getParametres();

            for (int j = 0; j < params.size(); j++) {
                Parameter parameter = (Parameter) params.elementAt(j);
                String pName = parameter.getName();
                String pType = TypeConversion.generateType(parameter);

                sb.append("\t/**\n");
                sb.append("\t * " + pType + " " + pName + "\n");
                sb.append("\t **/\n");
                // public PTYPE PNAME;
                sb.append("\tpublic ");
                sb.append(pType);
                sb.append(" ");
                sb.append(pName);
                sb.append(";\n\n");
            }

            // }
            sb.append("}\n");

            File file = new File(outputDirectory + File.separator
                    + classe.getName() + ".java");

            try {
                FileOutputStream fos = new FileOutputStream(file);
                fos.write(sb.toString().getBytes());
                fos.close();
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 
     * Generate all the methods related to a service
     * 
     * @param ms
     *            The ModelingService
     * @param ToolName
     *            The tool name
     * @param ModelingInterfaceName
     *            The modeling interface name
     * @return String The code of the services
     * @throws IOException
     */
    protected void generateServiceMethods(ModelingService ms, String ToolName,
            String ModelingInterfaceName) throws IOException {
    }

    protected void generateAdditionalMethods(ModelingServiceInterface msi)
            throws IOException {
    }

    /**
     * 
     * Create the input holder (parameters)
     * 
     * @param ms
     *            The modeling service
     * @return InputHolder[] The parameters
     */
    protected InputHolder[] createInputHolders(ModelingService ms) {
        EList params = ms.eContents();
        Vector paramsIN = new Vector();

        for (int i = 0; i < params.size(); i++) {
            if (params.get(i) instanceof Parameter) {
                Parameter parameter = (Parameter) params.get(i);
                int direction = parameter.getDirection().getValue();
                if (direction == DirectionKind.IN
                        || direction == DirectionKind.INOUT) {
                    paramsIN.add(new InputHolder(parameter));
                }
            }
        }
        InputHolder[] retour = new InputHolder[paramsIN.size()];
        retour = (InputHolder[]) paramsIN.toArray(retour);
        return retour;
    }

    /**
     * 
     * Generate parameters
     * 
     * @param inputHolders
     *            The parameters
     * @return String The code of the parameters
     */
    protected String generateInputParameters(InputHolder[] inputHolders) {
        StringBuffer sb = new StringBuffer();

        for (int i = 0; i < inputHolders.length; i++) {
            sb.append(inputHolders[i].getType() + " "
                    + inputHolders[i].getName());
            if (i != inputHolders.length - 1) {
                sb.append(", ");
            }
        }

        return sb.toString();
    }

    /**
     * 
     * Create the output holders
     * 
     * @param ms
     *            The modeling service
     * @return OutputHolder[] The output holders
     */
    protected OutputHolder[] createOutputHolders(ModelingService ms) {
        EList params = ms.eContents();
        Vector paramsOUT = new Vector();

        for (int i = 0; i < params.size(); i++) {
            if (params.get(i) instanceof Parameter) {
                Parameter parameter = (Parameter) params.get(i);
                int direction = parameter.getDirection().getValue();
                DirectionKind dkind = parameter.getDirection();
                //System.out.println("Dkind name:"+ dkind.getName());
                
                if ((direction == DirectionKind.OUT)
                        || (direction == DirectionKind.INOUT)
                        || (dkind.getName().equalsIgnoreCase("return"))) {
                    paramsOUT.add(parameter);
                }
            }
        }

        if (paramsOUT.size() == 0) {
            OutputHolder[] oh = new OutputHolder[1];
            oh[0] = new OutputHolder();
            return oh;
        }
        if (paramsOUT.size() == 1) {
            OutputHolder[] oh = new OutputHolder[1];
            oh[0] = new OutputHolder((Parameter) paramsOUT.get(0));
            return oh;
        }
        OutputHolder[] retour = new OutputHolder[paramsOUT.size()];
        String className = OUTPUT_HOLDER_PREFIX + ms.getName();
        OutputHolderClass classeInterne = new OutputHolderClass(className,
                paramsOUT);
        outputHolderClasses.put(className, classeInterne);
        for (int i = 0; i < retour.length; i++) {
            Parameter current = (Parameter) paramsOUT.get(i);
            retour[i] = new OutputHolder(className, current);
        }
        return retour;
    }

    /**
     * 
     * Generate the output holders
     * 
     * @param outputHolders
     *            The output holders
     * @return String The code of the output holders
     */
    protected String generateOutputParameters(OutputHolder[] outputHolders) {
        if (outputHolders.length == 1) {
            return outputHolders[0].getType();
        }
        return outputHolders[0].getClassName();
    }

    /**
     * 
     * Generate the complete file
     * 
     * @param outputDirectory
     *            The output directory
     * @param outputPackageName
     *            The package name
     * @param mode
     *            The mode
     * @throws BadModeException
     *             If the mode is not correct
     * @throws IOException
     *             If an I/O error occurs
     */
    public void generateContent(String outputPackageName,
            ModelingServiceInterface msi) throws IOException {
        generateHeader();
        this.packageName = outputPackageName;
        generatePackageDeclaration();

        generateCode(msi);

        fileOs.close();

    }

    public void generateClassName() throws IOException {

    }

    public void generateConstructor() throws IOException {

    }

    public void generatePackageDeclaration() throws IOException {
        if (packageName.compareTo("") != 0) {
            fileOs.write(("\npackage " + packageName + ";\n\n").getBytes());
        }

    }

    /**
     * 
     * generateImport - it should be field for each kind of generator
     * 
     * @throws IOException
     *  
     */
    protected void generateImport() throws IOException {

    }

    /**
     * getFileName generates a FileName information
     * 
     * 
     *  
     */
    public String getFileName() {
        String name = toolDescription.getName();
        name = (name.substring(0, 1)).toUpperCase() + name.substring(1);
        return name + FILE_POSTFIX;

    }

    /**
     * 
     * getMethodComment generates a methods comment with exception description
     * 
     * @param methodPrefix
     * @param ms
     * @return
     *  
     */
    protected String getMethodComment(String methodPrefix, ModelingService ms) {
        String methodName = methodPrefix + ms.getName();
        OutputHolder[] outputs = createOutputHolders(ms);

        InputHolder[] inputs = createInputHolders(ms);
        String outputType = generateOutputParameters(outputs);

        //comment
        String comment;

        comment = "\n\t/**\n";
        comment += "\t *\n";
        comment += "\t * ";

        comment += methodName;
        comment += "\n\t *\n";
        for (int i = 0; i < inputs.length; i++) {
            comment += "\t * @param " + inputs[i].name + "\n";
        }
        comment += "\t * @return " + outputType + " - " + outputs[0].getName()
                + "\n";

        //

        return comment;
    }

    /**
     * @return Returns the fileOs.
     */
    protected OutputStream getFileOs() {
        if (fileOs == null) {
            fileOs = getOS(outputDirectory, getFileName());
        }

        return fileOs;
    }

    protected OutputStream getOS(String outputDirectory, String fileName) {
        File file = new File(outputDirectory + File.separator + fileName
                + ".java");

        try {
            return new FileOutputStream(file);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        return null;

    }

}