/*
 * Copyright (c) 2006 Eclipse.org
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Dmitry Stadnik - initial API and implementation
 */
package org.eclipse.gmf.internal.bridge.wizards.pages;

import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.ecore.provider.EcoreItemProviderAdapterFactory;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.gmf.internal.bridge.resolver.Resolution;
import org.eclipse.gmf.internal.bridge.resolver.ResolvedItem;
import org.eclipse.gmf.internal.bridge.ui.Plugin;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/**
 * @author dstadnik
 */
class DomainModelViewerFactory {

	public static TreeViewer createViewer(Tree tree) {
		TreeViewer viewer = new TreeViewer(tree);
		viewer.setContentProvider(new ResolverContentProvider());
		AdapterFactory adapterFactory = new EcoreItemProviderAdapterFactory();
		viewer.setLabelProvider(new ResolverLabelProvider(new AdapterFactoryLabelProvider(adapterFactory)));
		viewer.setColumnProperties(new String[] { "no", Resolution.NODE.getName(), Resolution.LINK.getName(), Resolution.LABEL.getName() }); //$NON-NLS-1$
		viewer.setCellEditors(new CellEditor[] { null, new CheckboxCellEditor(), new CheckboxCellEditor(), new CheckboxCellEditor() });
		viewer.setCellModifier(new ResolverCellModifier(viewer));
		return viewer;
	}

	private static class ResolverCellModifier implements ICellModifier {

		private TreeViewer viewer;

		public ResolverCellModifier(TreeViewer viewer) {
			this.viewer = viewer;
		}

		public Object getValue(Object element, String property) {
			ResolvedItem item = (ResolvedItem) element;
			return Boolean.valueOf(item.getResolution() == Resolution.getByName(property));
		}

		public boolean canModify(Object element, String property) {
			return true;
		}

		public void modify(Object element, String property, Object value) {
			ResolvedItem item = (ResolvedItem) ((TreeItem) element).getData();
			Resolution resolution = Resolution.getByName(property);
			if (!item.isPossibleResolution(resolution)) {
				return;
			}
			item.setResolution(((Boolean) value).booleanValue() ? resolution : null);
			viewer.update(item, new String[] { Resolution.NODE.getName(), Resolution.LINK.getName(), Resolution.LABEL.getName() });
		}
	}

	private static class ResolverContentProvider implements ITreeContentProvider {

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object getParent(Object element) {
			return ((ResolvedItem) element).getParent();
		}

		public Object[] getChildren(Object parentElement) {
			List children = ((ResolvedItem) parentElement).getChildren();
			return children.toArray();
		}

		public boolean hasChildren(Object element) {
			List children = ((ResolvedItem) element).getChildren();
			return !children.isEmpty();
		}

		public Object[] getElements(Object inputElement) {
			return getChildren(inputElement);
		}
	}

	private static class ResolverLabelProvider extends LabelProvider implements ITableLabelProvider {

		private ILabelProvider domainLabelProvider;

		public ResolverLabelProvider(ILabelProvider domainLabelProvider) {
			this.domainLabelProvider = domainLabelProvider;
		}

		public String getColumnText(Object element, int columnIndex) {
			if (columnIndex == 0) {
				Object domainRef = ((ResolvedItem) element).getDomainRef();
				return domainLabelProvider.getText(domainRef);
			} else {
				return null;
			}
		}

		public Image getColumnImage(Object element, int columnIndex) {
			if (columnIndex == 0) {
				Object domainRef = ((ResolvedItem) element).getDomainRef();
				return domainLabelProvider.getImage(domainRef);
			}
			ResolvedItem item = (ResolvedItem) element;
			Image checkedIcon = Plugin.getDefault().getImageRegistry().get(Plugin.CHECKED_ICON);
			Image uncheckedIcon = Plugin.getDefault().getImageRegistry().get(Plugin.UNCHECKED_ICON);
			if (columnIndex == 1) {
				if (!item.isPossibleResolution(Resolution.NODE)) {
					return null;
				}
				return Resolution.NODE == item.getResolution() ? checkedIcon : uncheckedIcon;
			} else if (columnIndex == 2) {
				if (!item.isPossibleResolution(Resolution.LINK)) {
					return null;
				}
				return Resolution.LINK == item.getResolution() ? checkedIcon : uncheckedIcon;
			} else if (columnIndex == 3) {
				if (!item.isPossibleResolution(Resolution.LABEL)) {
					return null;
				}
				return Resolution.LABEL == item.getResolution() ? checkedIcon : uncheckedIcon;
			}
			return null;
		}
	}
}
