/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Payneau (Mia-Software) - initial API and implementation
 *******************************************************************************/
package org.eclipse.gmt.modisco.workflow.ui;

import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.ILaunchGroup;
import org.eclipse.gmt.modisco.infra.common.core.logging.MoDiscoLogger;
import org.eclipse.gmt.modisco.workflow.Activator;
import org.eclipse.gmt.modisco.workflow.Constants;
import org.eclipse.gmt.modisco.workflow.controller.ControllerDriverPage;
import org.eclipse.gmt.modisco.workflow.ui.common.Common;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;

/**
 * This class defines a page of Wizard. This page shows a different user
 * interface according to driver type selected previously.
 * 
 * @author Nicolas Payneau
 */
public class DriverPage extends WizardPage {
	private Composite rootContainer;
	private ControllerDriverPage controllerDriverpage;
	private Constants mode;
	private ScrolledComposite scrollContainer;
	protected Set<String> fReservedNames = null;
	private ILaunchGroup group;
	private MoDiscoWorkLaunchConfigurationDialog dialog;

	/**
	 * Class constructor
	 * 
	 * @param pageName
	 *            Name of the wizard page
	 * @param mode
	 *            Opening mode of wizard page
	 */
	protected DriverPage(final String pageName, final Constants mode, final ILaunchGroup group) {
		super(pageName);
		this.setImageDescriptor(Activator
				.getImageDescriptor(Constants.pathBigImage));
		this.mode = mode;
		this.group = group;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets
	 * .Composite)
	 */
	public void createControl(final Composite parent) {
		this.scrollContainer = new ScrolledComposite(parent, SWT.H_SCROLL
				| SWT.V_SCROLL);
		this.scrollContainer.setExpandHorizontal(true);
		this.scrollContainer.setExpandVertical(true);

		this.rootContainer = new Composite(this.scrollContainer, SWT.NULL);
		GridLayout layout = new GridLayout();
		this.rootContainer.setLayout(layout);
		this.scrollContainer.setContent(this.rootContainer);

		this.dialog = new MoDiscoWorkLaunchConfigurationDialog(getShell(),
				this.group, this.rootContainer, this);
		setControl(this.scrollContainer);
		setControl(this.rootContainer);
		this.rootContainer.layout();
		this.scrollContainer.setMinSize(this.rootContainer.computeSize(
				SWT.DEFAULT, SWT.DEFAULT));
		this.scrollContainer.layout();

		if (this.mode.equals(Constants.editMode)) {
			this.controllerDriverpage
					.refreshDriverPage(this.controllerDriverpage
							.getTypeEdition());
			this.controllerDriverpage.getDriverPage().initializeFrom(
					this.controllerDriverpage.getConfigurationEdition());
			this.controllerDriverpage.getDriverPage().refresh();
		}
	}

	/**
	 * Refresh the display
	 */
	public void refresh() {
		for (Control ctrl : this.rootContainer.getChildren()) {
			setControl(ctrl);
		}
		this.rootContainer.layout();
		setControl(this.scrollContainer);
		setControl(this.rootContainer);
		this.dialog.updateMessage();
	}

	/**
	 * Allows to change the controller current
	 * 
	 * @param controllerDriverpageParameter
	 *            The controller managing the display
	 */
	public void setController(final ControllerDriverPage controllerDriverpageParameter) {
		controllerDriverpageParameter.setDriverPage(this);
		this.controllerDriverpage = controllerDriverpageParameter;
	}

	/**
	 * Allows to retrieve the name of transformation
	 * 
	 * @return the name of transformation
	 */
	public String getTransformationName() {
		return this.dialog.getTabGroupViewer().getWorkingCopy().getName();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	@Override
	public void setVisible(final boolean visible) {
		if (visible) {
			this.getShell().setSize(800, 650);
			this.getShell().setLocation(
					Common.getInstance().centerShell(Display.getDefault(),
							this.getShell()));
		}
		super.setVisible(visible);
	}

	/**
	 * Create the tabs (run configuration) corresponding at type specified
	 * 
	 * @param typeLaunch
	 *            The launcher type which we want to create
	 */
	public void createTab(final String typeLaunch) {
		ILaunchConfigurationType type = DebugPlugin.getDefault()
				.getLaunchManager().getLaunchConfigurationType(typeLaunch);
		ILaunchConfiguration launchConfiguration = null;
		try {
			launchConfiguration = type.newInstance(null, DebugPlugin
					.getDefault().getLaunchManager()
					.generateLaunchConfigurationName(
							Constants.configurationDefaultName));
		} catch (CoreException e) {
			MoDiscoLogger.logError(e, Activator.getDefault());
		}
		this.dialog.getTabGroupViewer().setInput(launchConfiguration);
	}

	/**
	 * Saved the current configuration
	 * 
	 * @return the configuration saved
	 */
	public ILaunchConfigurationWorkingCopy performApply() {
		this.dialog.getTabGroupViewer().handleApplyPressed();
		ILaunchConfiguration config = null;
		ILaunchConfigurationWorkingCopy wc = null;
		try {
			config = this.dialog.getTabGroupViewer().getWorkingCopy()
					.getParent().doSave();
			wc = config.getWorkingCopy();
		} catch (CoreException e) {
			MoDiscoLogger.logError(e, Activator.getDefault());
		}
		return wc;
	}

	/**
	 * Load a configuration specified
	 * 
	 * @param configurationEdition
	 *            A configuration of launcher
	 */
	public void initializeFrom(
			final ILaunchConfigurationWorkingCopy configurationEdition) {
		this.dialog.getTabGroupViewer().setInput(configurationEdition);
	}

	/**
	 * Returns the reserved name set (if there is one), <code>null</code>
	 * otherwise
	 * 
	 * @return the reserved name set or <code>null</code>
	 */
	public Set<String> getReservedNameSet() {
		return this.fReservedNames;
	}
}
