/**
 * *******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * 
 *     Fabien Giquel (Mia-Software) - initial API and implementation
 * *******************************************************************************
 *
 */
package org.eclipse.gmt.modisco.xml.discoverer.actions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.gmt.modisco.infra.discoverymanager.Discoverer;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameter;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterDirectionKind;
import org.eclipse.gmt.modisco.infra.discoverymanager.DiscoveryParameterImpl;
import org.eclipse.gmt.modisco.xml.discoverer.XmlActivator;
import org.eclipse.gmt.modisco.xml.internal.resource.GenericXMLHandler;
import org.eclipse.gmt.modisco.xml.resource.GenericXMLResourceFactoryImpl;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;

/**
 * Discover XML generic model action.
 */
public class XMLModelDiscoverer implements Discoverer {

	private static Resource.Factory modiscoResourceFactory = new GenericXMLResourceFactoryImpl();
	private static List<DiscoveryParameter> parametersKeys = null;
	private Resource resourceResult = null;

	/**
	 * A parameter key for indicating to work without user interaction.
	 */
	public static final DiscoveryParameter PARAMETER_SILENT_MODE = new DiscoveryParameterImpl(
			"OPTION_SILENT_MODE", DiscoveryParameterDirectionKind.in, //$NON-NLS-1$
			Boolean.class, false);

	/**
	 * A parameter key for indicating to ignore white spaces in text portions
	 * found.
	 */
	public static final DiscoveryParameter PARAMETER_IGNORE_WHITESPACES = new DiscoveryParameterImpl(
			GenericXMLHandler.OPTION_IGNORE_WHITESPACES,
			DiscoveryParameterDirectionKind.in, Boolean.class, false);

	/**
	 * A parameter key for indicating to minimize the memory size of obtained
	 * model. If set to true : comments are ignored, text portions with only
	 * indentation and line delimiters are ignored.
	 */
	public static final DiscoveryParameter PARAMETER_LIGHTWEIGHT_MODEL = new DiscoveryParameterImpl(
			GenericXMLHandler.OPTION_LIGHTWEIGHT_MODEL,
			DiscoveryParameterDirectionKind.in, Boolean.class, false);

	/**
	 * A parameter key for indicating to serialize as an xmi file the obtained
	 * model.
	 */
	public static final DiscoveryParameter PARAMETER_SERIALIZE_XMI = new DiscoveryParameterImpl(
			"OPTION_SERIALIZE_XMI", //$NON-NLS-1$
			DiscoveryParameterDirectionKind.in, Boolean.class, false);

	/**
	 * A parameter key for indicating an output target Resource instance.
	 */
	public static final DiscoveryParameter PARAMETER_TARGET_RESOURCE = new DiscoveryParameterImpl(
			"TARGET_RESOURCE", DiscoveryParameterDirectionKind.out, //$NON-NLS-1$
			Resource.class, false);

	public final boolean isApplicableTo(final Object source) {
		boolean result = false;
		if (source instanceof IFile && ((IFile) source).exists()) {
			try {
				IContentType xmlContentType = Platform.getContentTypeManager()
						.getContentType("org.eclipse.core.runtime.xml"); //$NON-NLS-1$
				if (((IFile) source).getContentDescription() != null) {
					IContentType candidateType = ((IFile) source)
							.getContentDescription().getContentType();
					if (candidateType != null) {
						result = candidateType.isKindOf(xmlContentType);
					} else {
						// without content information, propose xml discovering
						result = true;
					}
				} else {
					// without content information, propose xml discovering
					result = true;
				}
			} catch (CoreException e) {
				IStatus warning = new Status(IStatus.WARNING,
						XmlActivator.PLUGIN_ID,
						"Could not test xml nature for file " //$NON-NLS-1$
								+ source.toString(), e);
				XmlActivator.getDefault().getLog().log(warning);
			}
		}

		return result;
	}

	/**
	 * Discovers a generic XML model and open MoDisco editor or default editor.
	 * This discoverer does not create additional physical artefact.
	 * 
	 * @param source
	 *            a IFile instance pointing XML contents
	 * @param target
	 *            should be null here.
	 */
	public final Resource discoverElement(final Object source, final URI target) {
		Map<DiscoveryParameter, Object> parameters = new HashMap<DiscoveryParameter, Object>();
		parameters.put(XMLModelDiscoverer.PARAMETER_SILENT_MODE, false);
		parameters.put(XMLModelDiscoverer.PARAMETER_IGNORE_WHITESPACES, false);
		parameters.put(XMLModelDiscoverer.PARAMETER_LIGHTWEIGHT_MODEL, false);
		parameters.put(XMLModelDiscoverer.PARAMETER_SERIALIZE_XMI, false);
		discoverElement(source, parameters);
		return (Resource) parameters
				.get(XMLModelDiscoverer.PARAMETER_TARGET_RESOURCE);
	}

	/**
	 * Discovers a generic XML model.
	 * 
	 * @param source
	 *            can be a workspace IFile or a java.io.File instance pointing
	 *            XML contents
	 * @param parameters
	 *            <ul>
	 *            <li>PARAMETER_SILENT_MODE : IN parameter; if is not set or set
	 *            to <code>false</code>, the discovered model will be opened in
	 *            an editor.
	 *            <li>PARAMETER_IGNORE_WHITESPACES : see
	 *            {@link XMLModelDiscoverer#PARAMETER_IGNORE_WHITESPACES}
	 *            <li>PARAMETER_LIGHTWEIGHT_MODEL :see
	 *            {@link XMLModelDiscoverer#PARAMETER_LIGHTWEIGHT_MODEL}
	 *            <li>PARAMETER_SERIALIZE_XMI :see
	 *            {@link XMLModelDiscoverer#PARAMETER_SERIALIZE_XMI}
	 *            <li>PARAMETER_TARGET_RESOURCE : OUT parameter; the discovered
	 *            model is stored with this key.
	 *            </ul>
	 */
	public final void discoverElement(final Object source,
			final Map<DiscoveryParameter, Object> parameters) {
		boolean isSilent = false;
		final boolean serializeXMI;
		if (parameters != null
				&& parameters.get(XMLModelDiscoverer.PARAMETER_SILENT_MODE) != null) {
			isSilent = (Boolean) parameters
					.get(XMLModelDiscoverer.PARAMETER_SILENT_MODE);
		}
		if (parameters != null
				&& parameters.get(XMLModelDiscoverer.PARAMETER_SERIALIZE_XMI) != null) {
			serializeXMI = (Boolean) parameters
					.get(XMLModelDiscoverer.PARAMETER_SERIALIZE_XMI);
		} else {
			serializeXMI = false;
		}

		this.setResourceResult(null);

		Job job = new Job(Messages.XMLModelDiscoverer_4) {
			@Override
			protected IStatus run(final IProgressMonitor monitor) {
				IStatus result = null;
				monitor.beginTask(Messages.XMLModelDiscoverer_5,
						IProgressMonitor.UNKNOWN);
				try {
					String absolutePath = null;
					URI sourceURI = null;
					if (source instanceof IFile) {
						absolutePath = ((IFile) source).getFullPath()
								.toString();
						sourceURI = URI.createURI(absolutePath);
					} else if (source instanceof java.io.File) {
						absolutePath = ((java.io.File) source)
								.getAbsolutePath();
						sourceURI = URI.createFileURI(absolutePath);

					}

					Resource resource = XMLModelDiscoverer
							.getModiscoResourceFactory().createResource(
									sourceURI);
					resource.load(parameters);

					if (monitor.isCanceled()) {
						result = Status.CANCEL_STATUS;
						return result;
					}
					result = Status.OK_STATUS;
					XMLModelDiscoverer.this.setResourceResult(resource);
					if (serializeXMI) {
						XMLModelDiscoverer.this.saveResource(monitor,
								sourceURI, resource);
					}

				} catch (IOException e) {
					result = new Status(IStatus.ERROR, XmlActivator.PLUGIN_ID,
							"An error occured during model discovering from " //$NON-NLS-1$
									+ source.toString(), e);
					XmlActivator.getDefault().getLog().log(result);
				} finally {
					monitor.done();
				}

				return result;
			}

		};

		if (!isSilent) {
			scheduleEditorOpening(source, job);
		}

		job.schedule();
		try {
			job.join();
		} catch (InterruptedException e) {
			IStatus status = new Status(IStatus.ERROR, XmlActivator.PLUGIN_ID,
					e.getMessage(), e);
			XmlActivator.getDefault().getLog().log(status);
		}

		if (parameters != null) {
			parameters.put(XMLModelDiscoverer.PARAMETER_TARGET_RESOURCE, this
					.getResourceResult());
		}
	}

	private void scheduleEditorOpening(final Object source, final Job job) {
		job.addJobChangeListener(new JobChangeAdapter() {
			@Override
			public void done(final IJobChangeEvent event) {
				if (event.getJob().getResult().isOK()) {
					PlatformUI.getWorkbench().getDisplay().asyncExec(
							new Runnable() {
								public void run() {
									try {
										// for opening in MoDisco Browser,
										// temporarily force our factory in emf
										// registry
										// TODO find a way to give resource to
										// the editor (IResourceEditorInput?)
										String fileExtension = ((IFile) source)
												.getFullPath()
												.getFileExtension();
										Object factoryBak = Resource.Factory.Registry.INSTANCE
												.getExtensionToFactoryMap()
												.get(fileExtension);

										Resource.Factory.Registry.INSTANCE
												.getExtensionToFactoryMap()
												.put(
														fileExtension,
														XMLModelDiscoverer
																.getModiscoResourceFactory());

										IWorkbenchPage page = PlatformUI
												.getWorkbench()
												.getActiveWorkbenchWindow()
												.getActivePage();

										// We must close previous editor opened
										// on xml file
										IEditorPart existingEditorOnResource = page
												.findEditor(new FileEditorInput(
														(IFile) source));
										if (existingEditorOnResource != null) {
											page.closeEditor(
													existingEditorOnResource,
													true);
										}

										IEditorDescriptor editDesc = PlatformUI
												.getWorkbench()
												.getEditorRegistry()
												.findEditor(
														"org.eclipse.gmt.modisco.infra.browser.editorID"); //$NON-NLS-1$

										if (editDesc != null) {
											IDE.openEditor(page,
													(IFile) source, editDesc
															.getId());
										} else {
											IDE
													.openEditor(page,
															(IFile) source);
										}

										// Setting back EMF registry
										Resource.Factory.Registry.INSTANCE
												.getExtensionToFactoryMap()
												.put(fileExtension, factoryBak);
									} catch (PartInitException e) {
										IStatus status = new Status(
												IStatus.ERROR,
												XmlActivator.PLUGIN_ID, e
														.getMessage(), e);
										XmlActivator.getDefault().getLog().log(
												status);
									}
								}
							});
				}
			}
		});
	}

	public final List<DiscoveryParameter> getDiscovererParameters() {
		if (XMLModelDiscoverer.parametersKeys == null) {
			XMLModelDiscoverer.parametersKeys = new ArrayList<DiscoveryParameter>();
			XMLModelDiscoverer.parametersKeys
					.add(XMLModelDiscoverer.PARAMETER_SILENT_MODE);
			XMLModelDiscoverer.parametersKeys
					.add(XMLModelDiscoverer.PARAMETER_TARGET_RESOURCE);
			XMLModelDiscoverer.parametersKeys
					.add(XMLModelDiscoverer.PARAMETER_IGNORE_WHITESPACES);
			XMLModelDiscoverer.parametersKeys
					.add(XMLModelDiscoverer.PARAMETER_LIGHTWEIGHT_MODEL);
			XMLModelDiscoverer.parametersKeys
					.add(XMLModelDiscoverer.PARAMETER_SERIALIZE_XMI);
		}
		return XMLModelDiscoverer.parametersKeys;
	}

	protected void saveResource(final IProgressMonitor monitor,
			final URI sourceURI, final Resource resource) throws IOException {
		Map<String, Object> options = new HashMap<String, Object>();
		options.put(XMLResource.OPTION_ENCODING, "UTF-8"); //$NON-NLS-1$
		resource.setURI(sourceURI.appendFileExtension("xmi")); //$NON-NLS-1$
		monitor.subTask(Messages.XMLModelDiscoverer_1);
		resource.save(options);
	}

	public void setResourceResult(final Resource aResourceResult) {
		this.resourceResult = aResourceResult;
	}

	public Resource getResourceResult() {
		return this.resourceResult;
	}

	public static void setModiscoResourceFactory(
			final Resource.Factory aModiscoResourceFactory) {
		XMLModelDiscoverer.modiscoResourceFactory = aModiscoResourceFactory;
	}

	public static Resource.Factory getModiscoResourceFactory() {
		return XMLModelDiscoverer.modiscoResourceFactory;
	}

}
