/**
 * <copyright> 
 *
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: TypeResolverImpl.java,v 1.3 2007/01/25 18:34:33 cdamus Exp $
 */
package org.eclipse.emf.ocl.parser;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EParameter;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceFactoryImpl;
import org.eclipse.emf.ocl.expressions.CollectionKind;
import org.eclipse.emf.ocl.expressions.ExpressionsFactory;
import org.eclipse.emf.ocl.expressions.OCLExpression;
import org.eclipse.emf.ocl.expressions.Variable;
import org.eclipse.emf.ocl.types.CollectionType;
import org.eclipse.emf.ocl.types.MessageType;
import org.eclipse.emf.ocl.types.TupleType;
import org.eclipse.emf.ocl.types.TypeType;
import org.eclipse.emf.ocl.types.TypesFactory;
import org.eclipse.emf.ocl.types.impl.TypeTypeImpl;
import org.eclipse.emf.ocl.types.impl.TypeUtil;
import org.eclipse.emf.ocl.types.util.TypesSwitch;
import org.eclipse.emf.ocl.uml.TypedElement;
import org.eclipse.emf.ocl.utilities.PredefinedType;
import org.eclipse.emf.ocl.utilities.impl.TupleFactory;


/**
 * Default implementation of the {@link TypeResolver} interface, storing the types
 * that it generates in the a resource to support persistence of
 * {@link OCLExpression}s referencing these types.
 * <p>
 * It is recommended that clients extend this class to customize resolution of
 * types based on their models, rather than implementing the interface.  Simply
 * override the protected <code>createXyzPackage()</code> methods to determine
 * where the resolved types are stored and the <code>resolveXyzType()</code>
 * methods to create or find types are required.
 * </p>
 * 
 * @deprecated Use the {@link org.eclipse.ocl.AbstractTypeResolver} class,
 * instead, with the {@link org.eclipse.ocl.AbstractEnvironment}.
 * 
 * @author Christian W. Damus (cdamus)
 */
public class TypeResolverImpl
	implements TypeResolver {

	private final TypesSwitch resolveSwitch = new ResolveSwitch();
	
	private Resource resource;
	private EPackage collectionPackage;
	private EPackage tuplePackage;
	private EPackage typePackage;
	private EPackage messagePackage;
	private EPackage additionalFeaturesPackage;
	
	/**
	 * Initializes me.  I create my own resource for persistence of model-based
	 * types.
	 */
	public TypeResolverImpl() {
		super();
	}
	
	/**
	 * Initializes me with a resource in which I will persist the model-based
	 * types that I generate in my associated {@link Environment}.
	 * 
	 * @param resource my resource
	 */
	public TypeResolverImpl(Resource resource) {
		this.resource = resource;
	}
	
	// Documentation copied from the inherited specification
	public EClassifier resolve(EClassifier type) {
		return (type == null)? type : (EClassifier) resolveSwitch.doSwitch(type);
	}
	
	// Documentation copied from the inherited specification
	public Resource getResource() {
		if (resource == null) {
			resource = createResource();
		}
		
		return resource;
	}
	
	/**
	 * Creates the resource that persists my generated types.  Subclasses requiring
	 * persistence must override this default implementation, as it creates a
	 * resource that does not support persistence and does not have a useful URI.
	 * A subclass could even find some other resource, such as that which will store
	 * the expressions parsed in this environment.
	 * 
	 * @return the new resource
	 */
	protected Resource createResource() {
        Resource.Factory factory;
        Object maybeFactory = Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(
            "ecore"); //$NON-NLS-1$
        if (maybeFactory instanceof Resource.Factory) {
            factory = (Resource.Factory) maybeFactory;
        } else {
            maybeFactory = Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().get(
                Resource.Factory.Registry.DEFAULT_EXTENSION);
            
            if (maybeFactory instanceof Resource.Factory) {
                factory = (Resource.Factory) maybeFactory;
            } else {
                factory = new ResourceFactoryImpl();
            }
        }
        
        return factory.createResource(URI.createURI("ocl:///oclenv.ecore")); //$NON-NLS-1$
	}
	
	/**
	 * Obtains the package containing the collection types that I generate.
	 * 
	 * @return my collection type package
	 */
	public EPackage getCollectionPackage() {
		if (collectionPackage == null) {
			collectionPackage = createCollectionPackage();
		}
		
		return collectionPackage;
	}
	
	/**
	 * Creates the package containing the collection types that I generate.
	 * 
	 * @return the new collection type package
	 */
	protected EPackage createCollectionPackage() {
		EPackage result = EcoreFactory.eINSTANCE.createEPackage();
		
		result.setName("collections"); //$NON-NLS-1$
		getResource().getContents().add(result);
		
		return result;
	}
	
	// Documentation copied from the inherited specification
	public CollectionType resolveCollectionType(CollectionKind kind,
			EClassifier elementType) {
		CollectionType result = findCollectionType(kind, elementType);
		
		if (result == null) {
			result = createCollectionType(kind, elementType);
		}
		
		return result;
	}
	
	/**
	 * Creates a new collection type of the specified <code>kind</code> and element
	 * type, assuming that it does not already exist.
	 * 
	 * @param kind the kind of collection to create
	 * @param elementType the collection's element type
	 * 
	 * @return the new collection type
	 */
	protected CollectionType createCollectionType(CollectionKind kind, EClassifier elementType) {
		CollectionType result = TypesFactory.eINSTANCE.createCollectionType(kind, elementType);
		getCollectionPackage().getEClassifiers().add(result);
		return result;
	}
	
	/**
	 * Finds an existing collection type matching the specified <code>kind</code> and
	 * element type, if any has already been created.
	 * 
	 * @param kind the element kind to search for
	 * @param elementType the element type to search for
	 * 
	 * @return the existing collection type, or <code>null</code> if none found
	 */
	protected CollectionType findCollectionType(CollectionKind kind, EClassifier elementType) {
		for (Iterator iter = getCollectionPackage().getEClassifiers().iterator(); iter.hasNext();) {
			CollectionType type = (CollectionType) iter.next();
			
			if ((type.getKind() == kind) &&
					(TypeUtil.getRelationship(
						type.getElementType(),
						elementType) == PredefinedType.SAME_TYPE)) {
					return type;
			}
		}
		
		return null;
	}
	
	/**
	 * Obtains the package containing the tuple types that I generate.
	 * 
	 * @return my tuple type package
	 */
	public EPackage getTuplePackage() {
		if (tuplePackage == null) {
			tuplePackage = createTuplePackage();
		}
		
		return tuplePackage;
	}
	
	/**
	 * Creates the package containing the tuple types that I generate.
	 * 
	 * @return the new tuple type package
	 */
	protected EPackage createTuplePackage() {
		EPackage result = EcoreFactory.eINSTANCE.createEPackage();
		
		result.setName("tuples"); //$NON-NLS-1$
		result.setEFactoryInstance(new TupleFactory());
		getResource().getContents().add(result);
		
		return result;
	}

	// Documentation copied from the inherited specification
	public TupleType resolveTupleType(List parts) {
		TupleType result = findTupleType(parts);
		
		if (result == null) {
			result = createTupleType(parts);
		}
		
		return result;
	}
	
	/**
	 * Creates a new tuple type from the specified <code>parts</code>, assuming that
	 * it does not already exist.
	 * 
	 * @param parts the {@link TypedElement}s describing the tuple parts
	 * 
	 * @return the new tuple type
	 */
	protected TupleType createTupleType(List parts) {
		TupleType result = TypesFactory.eINSTANCE.createTupleType(parts);
		getTuplePackage().getEClassifiers().add(result);
		return result;
	}
	
	/**
	 * Finds an existing tuple type matching the specified <code>parts</code>, if any
	 * has already been created.
	 * 
	 * @param parts the {@link TypedElement}s describing the tuple parts
	 * 
	 * @return the existing tuple type, or <code>null</code> if none found
	 */
	protected TupleType findTupleType(List parts) {
		for (Iterator iter = getTuplePackage().getEClassifiers().iterator(); iter.hasNext();) {
			TupleType type = (TupleType) iter.next();
			
			if (type.getEStructuralFeatures().size() == parts.size()) {
				for (Iterator jter = parts.iterator(); jter.hasNext();) {
					TypedElement part = (TypedElement) jter.next();
					EStructuralFeature property = type.getEStructuralFeature(part.getName());
					
					if ((property == null) ||
							(TypeUtil.getRelationship(
								TypeUtil.getOCLType(property),
								part.getType()) != PredefinedType.SAME_TYPE)) {
						// this isn't the tuple type we're looking for
						break;
					}
				}
				
				// this must be the tuple type we're looking for
				return type;
			}
		}
		
		return null;
	}
	
	/**
	 * Obtains the package containing the type types that I generate.
	 * 
	 * @return my type type package
	 */
	public EPackage getTypePackage() {
		if (typePackage == null) {
			typePackage = createTypePackage();
		}
		
		return typePackage;
	}
	
	/**
	 * Creates the package containing the type types that I generate.
	 * 
	 * @return the new type type package
	 */
	protected EPackage createTypePackage() {
		EPackage result = EcoreFactory.eINSTANCE.createEPackage();
		
		result.setName("types"); //$NON-NLS-1$
		getResource().getContents().add(result);
		
		return result;
	}
	
	// Documentation copied from the inherited specification
	public TypeType resolveTypeType(EClassifier type) {
		TypeType result = findTypeType(type);
		
		if (result == null) {
			result = createTypeType(type);
		}
		
		return result;
	}
	
	/**
	 * Creates a new type type for the specified <code>type</code>,
	 * assuming that it does not already exist.
	 * 
	 * @param type the referenced model type
	 * 
	 * @return the new type type
	 */
	protected TypeType createTypeType(EClassifier type) {
		TypeType result = TypesFactory.eINSTANCE.createTypeType(type);
		getTypePackage().getEClassifiers().add(result);
		return result;
	}
	
	/**
	 * Finds an existing type type matching the specified <code>type</code>,
	 * if any has already been created.
	 * 
	 * @param type the referenced model type
	 * 
	 * @return the existing type type, or <code>null</code> if none found
	 */
	protected TypeType findTypeType(EClassifier type) {
		for (Iterator iter = getTypePackage().getEClassifiers().iterator(); iter.hasNext();) {
			TypeTypeImpl typeType = (TypeTypeImpl) iter.next();
			
			if (TypeUtil.getRelationship(
						typeType.getReferredType(),
						type) == PredefinedType.SAME_TYPE) {
					return typeType;
			}
		}
		
		return null;
	}
	
	/**
	 * Obtains the package containing the message types that I generate.
	 * 
	 * @return my message type package
	 */
	public EPackage getMessagePackage() {
		if (messagePackage == null) {
			messagePackage = createMessagePackage();
		}
		
		return messagePackage;
	}
	
	/**
	 * Creates the package containing the message types that I generate.
	 * 
	 * @return the new message type package
	 */
	protected EPackage createMessagePackage() {
		EPackage result = EcoreFactory.eINSTANCE.createEPackage();
		
		result.setName("messages"); //$NON-NLS-1$
		getResource().getContents().add(result);
		
		return result;
	}
	
	// Documentation copied from the inherited specification
	public MessageType resolveMessageType(EOperation operation) {
		MessageType result = findMessageType(operation);
		
		if (result == null) {
			result = createMessageType(operation);
		}
		
		return result;
	}
	
	// Documentation copied from the inherited specification
	public MessageType resolveMessageType(EClass signal) {
		MessageType result = findMessageType(signal);
		
		if (result == null) {
			result = createMessageType(signal);
		}
		
		return result;
	}
	
	/**
	 * Creates a new message type for the specified <code>element</code>,
	 * assuming that it does not already exist.
	 * 
	 * @param element the operation or signal referenced by the message type
	 * 
	 * @return the new message type
	 */
	protected MessageType createMessageType(ENamedElement element) {
		MessageType result = TypesFactory.eINSTANCE.createMessageType(element);
		getMessagePackage().getEClassifiers().add(result);
		return result;
	}
	
	/**
	 * Finds an existing message type matching the specified <code>element</code>,
	 * if any has already been created.
	 * 
	 * @param element the referenced model element
	 * 
	 * @return the existing message type, or <code>null</code> if none found
	 */
	protected MessageType findMessageType(ENamedElement element) {
		for (Iterator iter = getMessagePackage().getEClassifiers().iterator(); iter.hasNext();) {
			MessageType type = (MessageType) iter.next();
			
			if ((type.getReferredOperation() == element)
					|| (type.getReferredSignal() == element)) {
				
				return type;
			}
		}
		
		return null;
	}
	
	/**
	 * Obtains the package containing the additional operations and properties
	 * parsed in my environment.
	 * 
	 * @return my additional features package
	 */
	public EPackage getAdditionalFeaturesPackage() {
		if (additionalFeaturesPackage == null) {
			additionalFeaturesPackage = createAdditionalFeaturesPackage();
		}
		
		return additionalFeaturesPackage;
	}
	
	/**
	 * Creates the package containing the additional operations and properties
	 * parsed in my environment.
	 * 
	 * @return the new additional features package
	 */
	protected EPackage createAdditionalFeaturesPackage() {
		EPackage result = EcoreFactory.eINSTANCE.createEPackage();
		
		result.setName("additional"); //$NON-NLS-1$
		getResource().getContents().add(result);
		
		return result;
	}
	
	// Documentation copied from the inherited specification
	public EOperation resolveAdditionalOperation(EClassifier owner, EOperation operation) {
		EClass shadow = findShadowClass(owner);
		
		if (shadow == null) {
			shadow = createShadowClass(owner);
		}
		
		EOperation result = findMatchingOperation(shadow, operation);
		if (result == null) {
			result = operation;
			shadow.getEOperations().add(result);
		}
		
		return result;
	}
	
	/**
	 * Finds an operation already existing in the specified <code>shadow</code>
	 * class that matches the specified <code>operation</code> signature.
	 * 
	 * @param shadow the shadow class to search
	 * @param operation the operation to match
	 * 
	 * @return the matching operation, or <code>null</code> if not found
	 */
	protected EOperation findMatchingOperation(EClass shadow, EOperation operation) {
		for (Iterator iter = shadow.getEOperations().iterator(); iter.hasNext();) {
			EOperation next = (EOperation) iter.next();
			
			if ((next == operation)
					|| (next.getName().equals(operation.getName())
							&& matchParameters(next, operation))) {
				return next;
			}
		}
		
		return null;
	}
	
	private boolean matchParameters(EOperation a, EOperation b) {
		EList aparms = a.getEParameters();
		EList bparms = b.getEParameters();
		
		if (aparms.size() == bparms.size()) {
			int count = aparms.size();
			
			for (int i = 0; i < count; i++) {
				EParameter aparm = (EParameter) aparms.get(i);
				EParameter bparm = (EParameter) bparms.get(i);
				
				if (!aparm.getName().equals(bparm.getName())
						|| TypeUtil.getRelationship(aparm.getEType(), bparm.getEType())
							!= PredefinedType.SAME_TYPE) {
					
					return false;
				}
			}
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * Finds a property already existing in the specified <code>shadow</code>
	 * class that matches the specified <code>property</code> signature.
	 * 
	 * @param shadow the shadow class to search
	 * @param property the property to match
	 * 
	 * @return the matching operation, or <code>null</code> if not found
	 */
	protected EStructuralFeature findMatchingProperty(EClass shadow, EStructuralFeature property) {
		for (Iterator iter = shadow.getEStructuralFeatures().iterator(); iter.hasNext();) {
			EStructuralFeature next = (EStructuralFeature) iter.next();
			
			if ((next == property) || next.getName().equals(property.getName())) {
				return next;
			}
		}
		
		return null;
	}
	
	// Documentation copied from the inherited specification
	public EStructuralFeature resolveAdditionalProperty(EClassifier owner, EStructuralFeature property) {
		EClass shadow = findShadowClass(owner);
		
		if (shadow == null) {
			shadow = createShadowClass(owner);
		}
		
		EStructuralFeature result = findMatchingProperty(shadow, property);
		if (result == null) {
			result = property;
			shadow.getEStructuralFeatures().add(result);
		}
		
		return result;
	}
	
	/**
	 * Creates the shadow class to contain additional features defined for the
	 * specified OCL <code>type</code>.
	 * 
	 * @param type an OCL type
	 * 
	 * @return the class containing its additional features
	 */
	protected EClass createShadowClass(EClassifier type) {
		EClass result = EcoreFactory.eINSTANCE.createEClass();
		result.setName(type.getName() + "_Class"); //$NON-NLS-1$
		
		EAnnotation ann = EcoreFactory.eINSTANCE.createEAnnotation();
		ann.setSource("realOwner"); //$NON-NLS-1$
		ann.getReferences().add(type);
		result.getEAnnotations().add(ann);
		
		getAdditionalFeaturesPackage().getEClassifiers().add(result);
		
		return result;
	}
	
	/**
	 * Finds the shadow class to contain additional features defined for the
	 * specified OCL <code>type</code>, if it already exists.
	 * 
	 * @param type an OCL type
	 * 
	 * @return the class containing its additional features, or <code>null</code>
	 *      if not found
	 */
	protected EClass findShadowClass(EClassifier type) {
		for (Iterator iter = getAdditionalFeaturesPackage().getEClassifiers().iterator(); iter.hasNext();) {
			EClass next = (EClass) iter.next();
			
			EAnnotation ann = next.getEAnnotation("realOwner"); //$NON-NLS-1$
			if ((ann != null) && ann.getReferences().contains(type)) {
				return next;
			}
		}
		
		return null;
	}
	
	/**
	 * A type switch that resolves types against my resolver's environment.
	 * 
	 * @author Christian W. Damus (cdamus)
	 */
	private class ResolveSwitch extends TypesSwitch {
		public Object caseCollectionType(CollectionType object) {
			return resolveCollectionType(
				object.getKind(),
				resolve(object.getElementType()));
		}
		
		public Object caseTupleType(TupleType object) {
			return resolveTupleType(createTupleParts(
				object.getEStructuralFeatures()));
		}
		
		public Object caseTypeType(TypeType object) {
			return resolveTypeType(
				resolve(((TypeTypeImpl) object).getReferredType()));
		}
		
		public Object caseMessageType(MessageType object) {
			if (object.getReferredOperation() != null) {
				return resolveMessageType(object.getReferredOperation());
			} else if (object.getReferredSignal() != null) {
				return resolveMessageType(object.getReferredSignal());
			}
			
			return null;
		}
		
		public Object defaultCase(EObject object) {
			return object;
		}
		
		private List createTupleParts(Collection features) {
			List result = new java.util.ArrayList();
			
			for (Iterator iter = features.iterator(); iter.hasNext();) {
				EStructuralFeature next = (EStructuralFeature) iter.next();
				
				Variable v = ExpressionsFactory.eINSTANCE.createVariable();
				v.setName(next.getName());
				v.setType(resolve(next.getEType()));
				result.add(v);
			}
			
			return result;
		}
	}
}
