/*****************************************************************************
 * Copyright (c) 2009 CEA LIST.
 *
 *    
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *  Remi Schnekenburger (CEA LIST) Remi.Schnekenburger@cea.fr - Initial API and implementation
 *  Yann Tanguy (CEA LIST) yann.tanguy@cea.fr - customization for CollaborationUse
 *
 *****************************************************************************/
package org.eclipse.papyrus.parsers.modelgenerator;

import org.antlr.runtime.ANTLRStringStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.MismatchedTokenException;
import org.antlr.runtime.RecognitionException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.papyrus.parsers.PapyrusParsersPlugin;
import org.eclipse.papyrus.parsers.antlr.CollaborationUseLabelLexer;
import org.eclipse.papyrus.parsers.antlr.CollaborationUseLabelParser;
import org.eclipse.papyrus.parsers.util.SimpleStringErrorReporter;
import org.eclipse.papyrus.parsers.util.TypeRecognitionException;
import org.eclipse.uml2.uml.CollaborationUse;

public class CollaborationUseGenerator {

	/** The CollaborationUse to modify */
	private CollaborationUse collaborationUse;

	/** Error reporter for lexer and parser generated by antlr */
	private SimpleStringErrorReporter reporter;

	/**
	 * Default Constructor.
	 * 
	 * @param collaborationUse
	 *        the CollaborationUse to modify
	 */
	public CollaborationUseGenerator(CollaborationUse collaborationUse) {
		this.collaborationUse = collaborationUse;
		this.reporter = new SimpleStringErrorReporter();
	}

	/**
	 * Parse the label of the CollaborationUse and modify CollaborationUse attributes.
	 * 
	 * @param label
	 *        the label that defines the CollaborationUse
	 */
	public void parseAndModifyCollaborationUse(String label) {
		try {
			CollaborationUseLabelLexer lexer = new CollaborationUseLabelLexer(new ANTLRStringStream(label));
			CommonTokenStream tokens = new CommonTokenStream(lexer);
			CollaborationUseLabelParser parser = new CollaborationUseLabelParser(tokens, collaborationUse,
					new SimpleStringErrorReporter());
			parser.label();
		} catch (RecognitionException e) {
			PapyrusParsersPlugin.getDefault().getLog().log(
					new Status(IStatus.ERROR, PapyrusParsersPlugin.PLUGIN_ID, IStatus.OK,
					"could not parse the property label: " + label, e));
		} catch (RuntimeException e) {
			PapyrusParsersPlugin.getDefault().getLog().log(
					new Status(IStatus.ERROR, PapyrusParsersPlugin.PLUGIN_ID, IStatus.OK,
					"could not parse the property label: " + label, e));
		}
	}

	/**
	 * Parse the label of the CollaborationUse and validate it.
	 * 
	 * @param label
	 *        the label that defines the CollaborationUse
	 * 
	 * @return null if label is valid, else return the message that describes the error.
	 */
	public String parseAndValidateCollaborationUse(String label) {

		// To validate the message : parse it.
		// If no errors are detected, it is ok. If exceptions : not ok.
		// Returns the exception message
		CollaborationUseLabelLexer lexer = new CollaborationUseLabelLexer(new ANTLRStringStream(label));
		CommonTokenStream tokens = new CommonTokenStream(lexer);
		CollaborationUseLabelParser parser = new CollaborationUseLabelParser(tokens, collaborationUse, reporter);
		parser.setValidation(true);
		try {
			parser.label();
		} catch (MismatchedTokenException e) {
			PapyrusParsersPlugin.logError("context:" + parser.getContext());
		} catch (RecognitionException e) {
			reporter.setMessage(e.getLocalizedMessage());
		} catch (RuntimeException e) {
			reporter.setMessage(e.getLocalizedMessage());
		}
		return reporter.getMessage();
	}

	/**
	 * Parse the label of the CollaborationUse, and generates an exception if does not validate.
	 * 
	 * @param label
	 *        the label that defines the CollaborationUse
	 * 
	 * @return null if label is valid, else return the message that describes the error.
	 */
	public String parseUndefinedCollaborationUseType(String label) {

		// To validate the message : parse it.
		// If no errors are detected, it is ok. If exceptions are detected : not ok.
		// Returns the exception message
		CollaborationUseLabelLexer lexer = new CollaborationUseLabelLexer(new ANTLRStringStream(label));
		CommonTokenStream tokens = new CommonTokenStream(lexer);
		CollaborationUseLabelParser parser = new CollaborationUseLabelParser(tokens, collaborationUse,
				new SimpleStringErrorReporter());
		parser.setValidation(true);
		try {
			parser.label();
		} catch (RecognitionException re) {

		} catch (RuntimeException tse) {
			if(tse instanceof TypeRecognitionException)
				return ((TypeRecognitionException)tse).getTypeName();
		}
		return null;
	}
}
