/**
 * Copyright (c) 2014, 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.papyrus.infra.elementtypesconfigurations.ElementTypeConfiguration;
import org.eclipse.papyrus.infra.filters.CompoundFilter;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.filters.FiltersFactory;
import org.eclipse.papyrus.infra.filters.OperatorKind;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistedElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ModelingAssistantProvider;
import org.eclipse.papyrus.uml.filters.ProfileApplied;
import org.eclipse.papyrus.uml.filters.UMLFiltersFactory;
import org.eclipse.papyrus.uml.profile.assistants.generator.ModelingAssistantProviderRule;
import org.eclipse.papyrus.uml.profile.elementtypesconfigurations.generator.UML;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;

/**
 * Utility extensions for working with {@link Filter}s.
 */
@Singleton
@SuppressWarnings("all")
public class FiltersUtil {
  @Extension
  private static FiltersFactory filtersFactory = FiltersFactory.eINSTANCE;
  
  @Extension
  private static UMLFiltersFactory umlFiltersFactory = UMLFiltersFactory.eINSTANCE;
  
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;
  
  @Inject
  @Extension
  private UML _uML;
  
  @Inject
  @Extension
  private ModelingAssistantProviderRule _modelingAssistantProviderRule;
  
  public ElementTypeFilter toElementTypeFilter(final String typeID, final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(typeID, umlProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_toElementTypeFilter) {
      if (_createCache_toElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_toElementTypeFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_toElementTypeFilter.put(_cacheKey, _result);
    }
    _init_toElementTypeFilter(_result, typeID, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_toElementTypeFilter = CollectionLiterals.newHashMap();
  
  private void _init_toElementTypeFilter(final ElementTypeFilter it, final String typeID, final Profile umlProfile) {
    it.setElementTypeID(typeID);
    int _lastIndexOf = typeID.lastIndexOf(".");
    int _plus = (_lastIndexOf + 1);
    String _substring = typeID.substring(_plus);
    it.setName(_substring);
    Profile _rootProfile = this._uML.getRootProfile(umlProfile);
    ModelingAssistantProvider _modelingAssistantProvider = this._modelingAssistantProviderRule.toModelingAssistantProvider(_rootProfile);
    EList<Filter> _ownedFilters = _modelingAssistantProvider.getOwnedFilters();
    _ownedFilters.add(it);
  }
  
  public ElementTypeFilter toFilter(final IElementType elementType, final Profile umlProfile) {
    Profile _rootProfile = this._uML.getRootProfile(umlProfile);
    return this.internalToFilter(elementType, _rootProfile);
  }
  
  private ElementTypeFilter internalToFilter(final IElementType elementType, final Profile rootProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementType, rootProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_internalToFilter) {
      if (_createCache_internalToFilter.containsKey(_cacheKey)) {
        return _createCache_internalToFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_internalToFilter.put(_cacheKey, _result);
    }
    _init_internalToFilter(_result, elementType, rootProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_internalToFilter = CollectionLiterals.newHashMap();
  
  private void _init_internalToFilter(final ElementTypeFilter it, final IElementType elementType, final Profile rootProfile) {
    String _id = elementType.getId();
    it.setElementTypeID(_id);
    String _displayName = elementType.getDisplayName();
    it.setName(_displayName);
    ModelingAssistantProvider _modelingAssistantProvider = this._modelingAssistantProviderRule.toModelingAssistantProvider(rootProfile);
    EList<Filter> _ownedFilters = _modelingAssistantProvider.getOwnedFilters();
    _ownedFilters.add(it);
  }
  
  public ElementTypeFilter toFilter(final ElementTypeConfiguration elementType, final Profile umlProfile) {
    Profile _rootProfile = this._uML.getRootProfile(umlProfile);
    return this.internalToFilter(elementType, _rootProfile);
  }
  
  private ElementTypeFilter internalToFilter(final ElementTypeConfiguration elementType, final Profile rootProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementType, rootProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_internalToFilter_1) {
      if (_createCache_internalToFilter_1.containsKey(_cacheKey)) {
        return _createCache_internalToFilter_1.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_internalToFilter_1.put(_cacheKey, _result);
    }
    _init_internalToFilter_1(_result, elementType, rootProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_internalToFilter_1 = CollectionLiterals.newHashMap();
  
  private void _init_internalToFilter_1(final ElementTypeFilter it, final ElementTypeConfiguration elementType, final Profile rootProfile) {
    String _identifier = elementType.getIdentifier();
    it.setElementTypeID(_identifier);
    String _name = elementType.getName();
    it.setName(_name);
    ModelingAssistantProvider _modelingAssistantProvider = this._modelingAssistantProviderRule.toModelingAssistantProvider(rootProfile);
    EList<Filter> _ownedFilters = _modelingAssistantProvider.getOwnedFilters();
    _ownedFilters.add(it);
  }
  
  public CompoundFilter toFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final CompoundFilter _result;
    synchronized (_createCache_toFilter) {
      if (_createCache_toFilter.containsKey(_cacheKey)) {
        return _createCache_toFilter.get(_cacheKey);
      }
      CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
      _result = _createCompoundFilter;
      _createCache_toFilter.put(_cacheKey, _result);
    }
    _init_toFilter(_result, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, CompoundFilter> _createCache_toFilter = CollectionLiterals.newHashMap();
  
  private void _init_toFilter(final CompoundFilter it, final Profile umlProfile) {
    it.setOperator(OperatorKind.OR);
    String _qualifiedName = umlProfile.getQualifiedName();
    String _plus = ("pertains to Profile " + _qualifiedName);
    it.setName(_plus);
    EList<Filter> _ownedFilters = it.getOwnedFilters();
    ProfileApplied _appliedFilter = this.toAppliedFilter(umlProfile);
    AssistedElementTypeFilter _assistedElementTypeFilter = this.toAssistedElementTypeFilter(umlProfile);
    CollectionExtensions.<Filter>addAll(_ownedFilters, _appliedFilter, _assistedElementTypeFilter);
    Profile _rootProfile = this._uML.getRootProfile(umlProfile);
    ModelingAssistantProvider _modelingAssistantProvider = this._modelingAssistantProviderRule.toModelingAssistantProvider(_rootProfile);
    EList<Filter> _ownedFilters_1 = _modelingAssistantProvider.getOwnedFilters();
    _ownedFilters_1.add(it);
  }
  
  private ProfileApplied toAppliedFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ProfileApplied _result;
    synchronized (_createCache_toAppliedFilter) {
      if (_createCache_toAppliedFilter.containsKey(_cacheKey)) {
        return _createCache_toAppliedFilter.get(_cacheKey);
      }
      ProfileApplied _createProfileApplied = FiltersUtil.umlFiltersFactory.createProfileApplied();
      _result = _createProfileApplied;
      _createCache_toAppliedFilter.put(_cacheKey, _result);
    }
    _init_toAppliedFilter(_result, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ProfileApplied> _createCache_toAppliedFilter = CollectionLiterals.newHashMap();
  
  private void _init_toAppliedFilter(final ProfileApplied it, final Profile umlProfile) {
    URI _uRI = EcoreUtil.getURI(umlProfile);
    String _string = _uRI.toString();
    it.setProfileURI(_string);
    String _qualifiedName = umlProfile.getQualifiedName();
    it.setProfileQualifiedName(_qualifiedName);
    String _qualifiedName_1 = umlProfile.getQualifiedName();
    String _plus = (_qualifiedName_1 + " is applied in context");
    it.setName(_plus);
  }
  
  protected Filter _andProfileApplied(final Void filter, final Profile umlProfile) {
    return null;
  }
  
  protected Filter _andProfileApplied(final Filter filter, final Profile umlProfile) {
    CompoundFilter _filter = this.toFilter(umlProfile);
    return this.operator_and(_filter, filter);
  }
  
  private AssistedElementTypeFilter toAssistedElementTypeFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final AssistedElementTypeFilter _result;
    synchronized (_createCache_toAssistedElementTypeFilter) {
      if (_createCache_toAssistedElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_toAssistedElementTypeFilter.get(_cacheKey);
      }
      AssistedElementTypeFilter _createAssistedElementTypeFilter = FiltersUtil.assistantFactory.createAssistedElementTypeFilter();
      _result = _createAssistedElementTypeFilter;
      _createCache_toAssistedElementTypeFilter.put(_cacheKey, _result);
    }
    _init_toAssistedElementTypeFilter(_result, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, AssistedElementTypeFilter> _createCache_toAssistedElementTypeFilter = CollectionLiterals.newHashMap();
  
  private void _init_toAssistedElementTypeFilter(final AssistedElementTypeFilter it, final Profile umlProfile) {
  }
  
  protected boolean _isCompound(final Void filter) {
    return false;
  }
  
  protected boolean _isCompound(final Filter filter) {
    return false;
  }
  
  protected boolean _isCompound(final CompoundFilter filter) {
    return true;
  }
  
  protected Filter _reduce(final Void filter) {
    return null;
  }
  
  protected Filter _reduce(final Filter filter) {
    return filter;
  }
  
  protected Filter _reduce(final CompoundFilter filter) {
    Filter _switchResult = null;
    EList<Filter> _filters = filter.getFilters();
    int _size = _filters.size();
    switch (_size) {
      case 0:
        _switchResult = null;
        break;
      case 1:
        EList<Filter> _filters_1 = filter.getFilters();
        _switchResult = _filters_1.get(0);
        break;
      default:
        _switchResult = filter;
        break;
    }
    return _switchResult;
  }
  
  private boolean add(final CompoundFilter compound, final Filter other) {
    boolean _xifexpression = false;
    EObject _eContainer = other.eContainer();
    boolean _notEquals = (!Objects.equal(_eContainer, null));
    if (_notEquals) {
      EList<Filter> _filters = compound.getFilters();
      _xifexpression = _filters.add(other);
    } else {
      EList<Filter> _ownedFilters = compound.getOwnedFilters();
      _xifexpression = _ownedFilters.add(other);
    }
    return _xifexpression;
  }
  
  private Boolean addAll(final CompoundFilter compound, final CompoundFilter other) {
    boolean _xifexpression = false;
    EObject _eContainer = other.eContainer();
    boolean _notEquals = (!Objects.equal(_eContainer, null));
    if (_notEquals) {
      EList<Filter> _filters = other.getFilters();
      final Procedure1<Filter> _function = new Procedure1<Filter>() {
        public void apply(final Filter it) {
          FiltersUtil.this.add(compound, it);
        }
      };
      IterableExtensions.<Filter>forEach(_filters, _function);
    } else {
      boolean _xblockexpression = false;
      {
        EList<Filter> _ownedFilters = compound.getOwnedFilters();
        EList<Filter> _ownedFilters_1 = other.getOwnedFilters();
        _ownedFilters.addAll(_ownedFilters_1);
        EList<Filter> _filters_1 = compound.getFilters();
        EList<Filter> _filters_2 = other.getFilters();
        _xblockexpression = _filters_1.addAll(_filters_2);
      }
      _xifexpression = _xblockexpression;
    }
    return Boolean.valueOf(_xifexpression);
  }
  
  protected Filter _operator_or(final Void left, final Filter right) {
    return right;
  }
  
  protected Filter _operator_or(final Filter left, final Filter right) {
    return this.createOr(left, right);
  }
  
  private CompoundFilter createOr(final Filter left, final Filter right) {
    CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
    final Procedure1<CompoundFilter> _function = new Procedure1<CompoundFilter>() {
      public void apply(final CompoundFilter it) {
        it.setOperator(OperatorKind.OR);
        FiltersUtil.this.add(it, left);
        FiltersUtil.this.add(it, right);
      }
    };
    return ObjectExtensions.<CompoundFilter>operator_doubleArrow(_createCompoundFilter, _function);
  }
  
  protected Filter _operator_or(final CompoundFilter left, final Filter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            this.add(left, right);
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }
  
  protected Filter _operator_or(final Filter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = right.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            this.add(right, left);
            _xblockexpression = right;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }
  
  protected Filter _operator_or(final CompoundFilter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            OperatorKind _operator_1 = right.getOperator();
            if (_operator_1 != null) {
              switch (_operator_1) {
                case OR:
                  EObject _eContainer = right.eContainer();
                  boolean _equals = Objects.equal(_eContainer, null);
                  if (_equals) {
                    this.addAll(left, right);
                  } else {
                    this.add(left, right);
                  }
                  break;
                default:
                  this.add(left, right);
                  break;
              }
            } else {
              this.add(left, right);
            }
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }
  
  protected Filter _operator_and(final Void left, final Filter right) {
    return right;
  }
  
  protected Filter _operator_and(final Filter left, final Filter right) {
    return this.createAnd(left, right);
  }
  
  private CompoundFilter createAnd(final Filter left, final Filter right) {
    CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
    final Procedure1<CompoundFilter> _function = new Procedure1<CompoundFilter>() {
      public void apply(final CompoundFilter it) {
        it.setOperator(OperatorKind.AND);
        FiltersUtil.this.add(it, left);
        FiltersUtil.this.add(it, right);
      }
    };
    return ObjectExtensions.<CompoundFilter>operator_doubleArrow(_createCompoundFilter, _function);
  }
  
  protected Filter _operator_and(final CompoundFilter left, final Filter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            this.add(left, right);
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }
  
  protected Filter _operator_and(final Filter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = right.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            this.add(right, left);
            _xblockexpression = right;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }
  
  protected Filter _operator_and(final CompoundFilter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            OperatorKind _operator_1 = right.getOperator();
            if (_operator_1 != null) {
              switch (_operator_1) {
                case AND:
                  EObject _eContainer = right.eContainer();
                  boolean _equals = Objects.equal(_eContainer, null);
                  if (_equals) {
                    this.addAll(left, right);
                  } else {
                    this.add(left, right);
                  }
                  break;
                default:
                  this.add(left, right);
                  break;
              }
            } else {
              this.add(left, right);
            }
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }
  
  public Filter andProfileApplied(final Filter filter, final Profile umlProfile) {
    if (filter != null) {
      return _andProfileApplied(filter, umlProfile);
    } else if (filter == null) {
      return _andProfileApplied((Void)null, umlProfile);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(filter, umlProfile).toString());
    }
  }
  
  public boolean isCompound(final Filter filter) {
    if (filter instanceof CompoundFilter) {
      return _isCompound((CompoundFilter)filter);
    } else if (filter != null) {
      return _isCompound(filter);
    } else if (filter == null) {
      return _isCompound((Void)null);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(filter).toString());
    }
  }
  
  public Filter reduce(final Filter filter) {
    if (filter instanceof CompoundFilter) {
      return _reduce((CompoundFilter)filter);
    } else if (filter != null) {
      return _reduce(filter);
    } else if (filter == null) {
      return _reduce((Void)null);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(filter).toString());
    }
  }
  
  public Filter operator_or(final Filter left, final Filter right) {
    if (left instanceof CompoundFilter
         && right instanceof CompoundFilter) {
      return _operator_or((CompoundFilter)left, (CompoundFilter)right);
    } else if (left instanceof CompoundFilter
         && right != null) {
      return _operator_or((CompoundFilter)left, right);
    } else if (left != null
         && right instanceof CompoundFilter) {
      return _operator_or(left, (CompoundFilter)right);
    } else if (left != null
         && right != null) {
      return _operator_or(left, right);
    } else if (left == null
         && right != null) {
      return _operator_or((Void)null, right);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(left, right).toString());
    }
  }
  
  public Filter operator_and(final Filter left, final Filter right) {
    if (left instanceof CompoundFilter
         && right instanceof CompoundFilter) {
      return _operator_and((CompoundFilter)left, (CompoundFilter)right);
    } else if (left instanceof CompoundFilter
         && right != null) {
      return _operator_and((CompoundFilter)left, right);
    } else if (left != null
         && right instanceof CompoundFilter) {
      return _operator_and(left, (CompoundFilter)right);
    } else if (left != null
         && right != null) {
      return _operator_and(left, right);
    } else if (left == null
         && right != null) {
      return _operator_and((Void)null, right);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(left, right).toString());
    }
  }
}
