/**
 * Copyright (c) 2014, 2015 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Christian W. Damus - Initial API and implementation
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.function.Consumer;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ConnectionAssistant;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ModelingAssistantProvider;
import org.eclipse.papyrus.infra.gmfdiag.assistant.PopupAssistant;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.profile.assistants.generator.ConnectionAssistantRule;
import org.eclipse.papyrus.uml.profile.assistants.generator.FiltersUtil;
import org.eclipse.papyrus.uml.profile.assistants.generator.PopupAssistantRule;
import org.eclipse.papyrus.uml.profile.types.generator.Identifiers;
import org.eclipse.papyrus.uml.profile.types.generator.ImpliedExtension;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.papyrus.uml.profile.types.generator.UMLElementTypes;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Transformation rule for generating a {@link ModelingAssistantProvider} from a UML {@link Profile}.
 */
@Singleton
@SuppressWarnings("all")
public class ModelingAssistantProviderRule {
  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;
  
  @Inject
  @Extension
  private Identifiers _identifiers;
  
  @Inject
  @Extension
  private UML _uML;
  
  @Inject
  @Extension
  private UMLElementTypes _uMLElementTypes;
  
  @Inject
  @Extension
  private FiltersUtil _filtersUtil;
  
  @Inject
  @Extension
  private PopupAssistantRule _popupAssistantRule;
  
  @Inject
  @Extension
  private ConnectionAssistantRule _connectionAssistantRule;
  
  public ModelingAssistantProvider toModelingAssistantProvider(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ModelingAssistantProvider _result;
    synchronized (_createCache_toModelingAssistantProvider) {
      if (_createCache_toModelingAssistantProvider.containsKey(_cacheKey)) {
        return _createCache_toModelingAssistantProvider.get(_cacheKey);
      }
      ModelingAssistantProvider _createModelingAssistantProvider = ModelingAssistantProviderRule.assistantFactory.createModelingAssistantProvider();
      _result = _createModelingAssistantProvider;
      _createCache_toModelingAssistantProvider.put(_cacheKey, _result);
    }
    _init_toModelingAssistantProvider(_result, umlProfile);
    return _result;
  }
  
  private final HashMap<ArrayList<?>, ModelingAssistantProvider> _createCache_toModelingAssistantProvider = CollectionLiterals.newHashMap();
  
  private void _init_toModelingAssistantProvider(final ModelingAssistantProvider it, final Profile umlProfile) {
    this._identifiers.setIdentifierBase(umlProfile);
    String _name = umlProfile.getName();
    it.setName(_name);
    Iterable<ImpliedExtension> _allExtensions = this._uML.getAllExtensions(umlProfile);
    final Consumer<ImpliedExtension> _function = new Consumer<ImpliedExtension>() {
      public void accept(final ImpliedExtension ext) {
        org.eclipse.uml2.uml.Class _metaclass = ext.getMetaclass();
        Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes = ModelingAssistantProviderRule.this._uMLElementTypes.getDiagramSpecificElementTypes(_metaclass);
        final Function1<ElementTypeConfiguration, String> _function = new Function1<ElementTypeConfiguration, String>() {
          public String apply(final ElementTypeConfiguration it) {
            return ModelingAssistantProviderRule.this._identifiers.toElementTypeID(ext, it);
          }
        };
        Iterable<String> profileElementTypes = IterableExtensions.map(_diagramSpecificElementTypes, _function);
        final Consumer<String> _function_1 = new Consumer<String>() {
          public void accept(final String it) {
            ModelingAssistantProviderRule.this._filtersUtil.toElementTypeFilter(it, umlProfile);
          }
        };
        profileElementTypes.forEach(_function_1);
        org.eclipse.uml2.uml.Class _metaclass_1 = ext.getMetaclass();
        EClass _eClass = ModelingAssistantProviderRule.this._uMLElementTypes.getEClass(_metaclass_1);
        boolean _isRelationship = ModelingAssistantProviderRule.this._uML.isRelationship(_eClass);
        boolean _not = (!_isRelationship);
        if (_not) {
          EList<String> _elementTypeIDs = it.getElementTypeIDs();
          Iterables.<String>addAll(_elementTypeIDs, profileElementTypes);
          EList<PopupAssistant> _popupAssistants = it.getPopupAssistants();
          org.eclipse.uml2.uml.Class _metaclass_2 = ext.getMetaclass();
          Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes_1 = ModelingAssistantProviderRule.this._uMLElementTypes.getDiagramSpecificElementTypes(_metaclass_2);
          final Function1<ElementTypeConfiguration, PopupAssistant> _function_2 = new Function1<ElementTypeConfiguration, PopupAssistant>() {
            public PopupAssistant apply(final ElementTypeConfiguration it) {
              return ModelingAssistantProviderRule.this._popupAssistantRule.toPopupAssistant(ext, it);
            }
          };
          Iterable<PopupAssistant> _map = IterableExtensions.map(_diagramSpecificElementTypes_1, _function_2);
          Iterables.<PopupAssistant>addAll(_popupAssistants, _map);
        } else {
          EList<String> _relationshipTypeIDs = it.getRelationshipTypeIDs();
          Iterables.<String>addAll(_relationshipTypeIDs, profileElementTypes);
          EList<ConnectionAssistant> _connectionAssistants = it.getConnectionAssistants();
          org.eclipse.uml2.uml.Class _metaclass_3 = ext.getMetaclass();
          Iterable<? extends ElementTypeConfiguration> _diagramSpecificElementTypes_2 = ModelingAssistantProviderRule.this._uMLElementTypes.getDiagramSpecificElementTypes(_metaclass_3);
          final Function1<ElementTypeConfiguration, ConnectionAssistant> _function_3 = new Function1<ElementTypeConfiguration, ConnectionAssistant>() {
            public ConnectionAssistant apply(final ElementTypeConfiguration it) {
              return ModelingAssistantProviderRule.this._connectionAssistantRule.toConnectionAssistant(ext, it);
            }
          };
          Iterable<ConnectionAssistant> _map_1 = IterableExtensions.map(_diagramSpecificElementTypes_2, _function_3);
          Iterables.<ConnectionAssistant>addAll(_connectionAssistants, _map_1);
        }
      }
    };
    _allExtensions.forEach(_function);
  }
}
