package org.eclipse.gmt.modisco.common.editor.editors;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmt.modisco.common.editor.adapters.LinkItemProvider;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;

/**
 * Custom painter to add information on tree elements:
 * <ul>
 * <li>Draws a down-pointing arrow on ordered references when they are effectively displayed in
 * their original order, that is, when "sort instances" is disabled.
 * <li>Adds the order of elements in their collection corresponding to the link under which they are
 * displayed (this is impossible to do in the element's adapter)
 * </ul>
 */
public class CustomTreePainter implements PropertyChangeListener {

	// private Color arrowColor;
	private Font arrowFont;
	private final Tree tree;
	private int baseHeight;
	private int currentFontSize;
	private final EditorConfiguration editorConfiguration;

	public CustomTreePainter(Tree tree, EditorConfiguration editorConfiguration) {
		this.tree = tree;
		this.editorConfiguration = editorConfiguration;
		// this.arrowColor = tree.getDisplay().getSystemColor(SWT.COLOR_BLUE);
		this.baseHeight = JFaceResources.getDialogFont().getFontData()[0].getHeight();

		editorConfiguration.addPropertyChangeListener(this);
		configureFont(editorConfiguration.getFontSizeDelta());
		setupTreeCustomPaint(tree);
	}

	public void propertyChange(PropertyChangeEvent event) {
		if (EditorConfiguration.FONT_SIZE_DELTA_PROP.equals(event.getPropertyName())) {
			configureFont((Integer) event.getNewValue());
		}
	}

	private void configureFont(int fontDelta) {
		this.currentFontSize = this.baseHeight + fontDelta;
		if (this.arrowFont != null) {
			this.arrowFont.dispose();
		}
		this.arrowFont = new Font(this.tree.getDisplay(), "Wingdings", this.currentFontSize,
				SWT.NONE);
	}

	private boolean isOrderedReference(Object object) {
		// when instances are sorted, references are not ordered anymore
		if (this.editorConfiguration.isSortInstances())
			return false;

		if (object instanceof LinkItemProvider) {
			LinkItemProvider linkItemProvider = (LinkItemProvider) object;
			EReference reference = linkItemProvider.getReference();
			return reference.isMany() && reference.isOrdered();
		}
		return false;
	}

	private boolean isEnabled() {
		return this.editorConfiguration.isShowOrdering();
	}

	private void handleMeasureItem(Event event) {
		if (isEnabled()) {
			TreeItem item = (TreeItem) event.item;
			Object data = item.getData();
			if (isOrderedReference(data)) {
				event.width += CustomTreePainter.this.currentFontSize;
			} else if (data instanceof EObject) {
				int order = findElementOrder(item);
				if (order != -1) {
					event.width += event.gc.textExtent(orderStringFor(order)).x + 3;
				}
			}
		}
	}

	private void handlePaintItem(Event event) {
		if (isEnabled()) {
			TreeItem item = (TreeItem) event.item;
			Object data = item.getData();
			if (isOrderedReference(data)) {
				paintOrderArrow(event);
			} else if (data instanceof EObject) {
				int order = findElementOrder(item);
				if (order != -1) {
					paintElementOrder(order, event);
				}
			}
		}
	}

	private void paintOrderArrow(Event event) {
		event.gc.setFont(this.arrowFont);
		event.gc.fillRectangle(event.x + event.width, event.y, this.currentFontSize + 2,
				event.height);
		event.gc.drawText("\u00f2", event.x + event.width, event.y + event.height / 10);
	}

	/** Find the order of the model element corresponding to the given tree item */
	private int findElementOrder(TreeItem item) {
		TreeItem parentItem = item.getParentItem();
		if (parentItem != null) {
			Object parentData = parentItem.getData();
			if (parentData instanceof LinkItemProvider) {
				TreeItem grandParentItem = parentItem.getParentItem();
				if (grandParentItem != null) {
					Object grandParentData = grandParentItem.getData();
					if (grandParentData instanceof EObject) {
						return findElementOrder((EObject) item.getData(),
								(LinkItemProvider) parentData, (EObject) grandParentData);
					}
				}
			}
		}

		return -1;
	}

	/**
	 * Find the order of the given element in the given link from the other given element.
	 * 
	 * @param element
	 *            the element whose order must be painted
	 * @param parentLink
	 *            the link visually containing the element in the tree
	 * @param parentElement
	 *            the element which has the link <code>parentLink</code> to <code>element</code>
	 * @return the order of the element or <code>-1</code> if the feature is not ordered or it does
	 *         not belong to it.
	 */
	private int findElementOrder(EObject element, LinkItemProvider parentLink, EObject parentElement) {

		EReference reference = parentLink.getReference();
		if (reference.isMany() && reference.isOrdered()) {
			@SuppressWarnings("unchecked")
			EList<EObject> list = (EList<EObject>) parentElement.eGet(reference);

			Iterator<EObject> iterator = list.iterator();
			int i = 0;
			while (iterator.hasNext()) {
				EObject listElement = iterator.next();
				if (listElement == element)
					return i;
				i++;
			}
		}

		return -1;
	}

	private String orderStringFor(int order) {
		return "[" + order + "]";
	}

	private void paintElementOrder(int order, Event event) {
		String string = orderStringFor(order);
		Point textExtent = event.gc.textExtent(string);

		// draw the selection background
		event.gc.fillRectangle(event.x + event.width - 3, event.y, textExtent.x + 6, event.height);
		// center the text vertically
		int posY = event.y + (event.height - textExtent.y) / 2;
		event.gc.drawText(string, event.x + event.width, posY);
	}

	private void setupTreeCustomPaint(final Tree tree) {
		tree.addListener(SWT.MeasureItem, new Listener() {
			public void handleEvent(Event event) {
				handleMeasureItem(event);
			}

		});
		tree.addListener(SWT.PaintItem, new Listener() {
			public void handleEvent(Event event) {
				handlePaintItem(event);
			}
		});
	}

	public void dispose() {
		if (this.arrowFont != null) {
			this.arrowFont.dispose();
		}
	}
}
