/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors.table;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.gmt.modisco.common.editor.core.InstancesForMetaclass;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.util.EMFUtil;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPersistableElement;

public class TableEditorInput implements IEditorInput {

	/** The elements that are displayed in the table */
	private List<EObject> elements;
	/** Union of all the elements' attributes */
	private List<EAttribute> attributes;
	/** Union of all the elements' references */
	private List<EReference> references;

	/** The main editor's configuration */
	private final EditorConfiguration editorConfiguration;
	/** A textual description of what is shown in the editor */
	private final String description;
	/** All the metaclasses of the elements */
	private HashSet<EClass> metaclasses;

	/**
	 * Create an input for all the given model elements
	 * 
	 * @param editorConfiguration
	 *            the main editor configuration
	 * @param elements
	 *            the elements to show as rows in the table
	 * @param description
	 *            a descriptive String of what will be displayed in the table view editor
	 * */
	public TableEditorInput(EditorConfiguration editorConfiguration, List<EObject> elements,
			String description) {
		this.editorConfiguration = editorConfiguration;
		this.elements = elements;
		this.description = description;

		this.metaclasses = new HashSet<EClass>();
		for (EObject element : elements) {
			this.metaclasses.add(element.eClass());
		}

		// all the attributes of the metaclasses of the elements
		HashSet<EAttribute> attributeSet = new HashSet<EAttribute>();
		for (EClass eClass : this.metaclasses) {
			attributeSet.addAll(eClass.getEAllAttributes());
		}

		// all the references of the metaclasses of the elements
		HashSet<EReference> referenceSet = new HashSet<EReference>();
		for (EClass eClass : this.metaclasses) {
			referenceSet.addAll(eClass.getEAllReferences());
		}

		this.attributes = new ArrayList<EAttribute>(attributeSet);
		this.references = new ArrayList<EReference>(referenceSet);
	}

	/**
	 * Create an input for all elements of the given class
	 * 
	 * @param editorConfiguration
	 *            the main editor configuration
	 * @param eClass
	 *            the metaclass of the elements to show
	 * @param description
	 *            a descriptive String of what will be displayed in the table view editor
	 * */
	public TableEditorInput(EditorConfiguration editorConfiguration, EClass eClass,
			String description) {
		this.editorConfiguration = editorConfiguration;
		this.description = description;
		InstancesForMetaclass instances = this.editorConfiguration.getInstancesForMetaclasses()
				.getInstancesForMetaclass(EMFUtil.getMetaclassQualifiedName(eClass));
		this.elements = new ArrayList<EObject>(instances.getElements());
		this.attributes = new ArrayList<EAttribute>(eClass.getEAllAttributes());
		this.references = new ArrayList<EReference>(eClass.getEAllReferences());
	}

	public List<EObject> getElements() {
		return this.elements;
	}

	public List<EAttribute> getAttributes() {
		return this.attributes;
	}

	public List<EReference> getReferences() {
		return this.references;
	}

	public String getDescription() {
		return this.description;
	}

	public EditorConfiguration getEditorConfiguration() {
		return this.editorConfiguration;
	}

	public boolean exists() {
		return false;
	}

	public ImageDescriptor getImageDescriptor() {
		return null;
	}

	public String getName() {
		return "Table Viewer";
	}

	public IPersistableElement getPersistable() {
		return null;
	}

	public String getToolTipText() {
		return "Table Viewer";
	}

	@SuppressWarnings("unchecked")
	public Object getAdapter(Class adapter) {
		return null;
	}

	public EClass[] getMetaclasses() {
		return this.metaclasses.toArray(new EClass[this.metaclasses.size()]);
	}
}
