/*******************************************************************************
 * Copyright (c) 2009 Mia-Software.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Nicolas Bros (Mia-Software) - initial API and implementation
 *    
 *******************************************************************************/

package org.eclipse.gmt.modisco.common.editor.editors.table;

import java.util.List;

import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.ui.provider.ExtendedImageRegistry;
import org.eclipse.gmt.modisco.common.editor.editors.EditorConfiguration;
import org.eclipse.gmt.modisco.common.editor.editors.table.ColumnDescription.Type;
import org.eclipse.gmt.modisco.common.editor.util.Util;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;

public class TableEditorLabelProvider extends LabelProvider implements ITableLabelProvider {

	private final EditorConfiguration editorConfiguration;
	private final List<ColumnDescription> columns;

	public TableEditorLabelProvider(List<ColumnDescription> columns,
			EditorConfiguration editorConfiguration) {
		this.columns = columns;
		this.editorConfiguration = editorConfiguration;
	}

	public String getTextFor(Object element) {
		if (element == null)
			return "null";
		AdapterFactory adapterFactory = this.editorConfiguration.getAdapterFactory();
		IItemLabelProvider labelProvider = (IItemLabelProvider) adapterFactory.adapt(element,
				IItemLabelProvider.class);
		return labelProvider.getText(element);
	}

	public Image getImageFor(Object element) {
		if (element == null)
			return null;

		AdapterFactory adapterFactory = this.editorConfiguration.getAdapterFactory();
		IItemLabelProvider labelProvider = (IItemLabelProvider) adapterFactory.adapt(element,
				IItemLabelProvider.class);
		if (labelProvider != null)
			return ExtendedImageRegistry.getInstance().getImage(labelProvider.getImage(element));
		else
			return null;
	}

	public Image getColumnImage(Object element, int columnIndex) {
		if (columnIndex < this.columns.size()) {
			ColumnDescription columnDescription = this.columns.get(columnIndex);
			if (columnDescription.getType() == Type.DEFAULT_LABEL) {
				return getImageFor(element);
			} else if (columnDescription.getType() == Type.REFERENCE) {
				EReference reference = columnDescription.getReference();
				EObject eObject = (EObject) element;

				if (!eObject.eClass().getEAllReferences().contains(reference)) {
					return null;
				}

				Object value = eObject.eGet(reference);
				if (reference.isMany()) {
					@SuppressWarnings("unchecked")
					EList<EObject> list = (EList<EObject>) value;

					// when only 1 element, then display its icon directly
					if (list.size() == 1)
						return getImageFor(list.get(0));
					else
						return null;
				} else {
					if (value != null) {
						return getImageFor(value);
					}
				}
			} else if (columnDescription.getType() == Type.ECONTAINER) {
				return getImageFor(((EObject) element).eContainer());
			}
		}

		return null;
	}

	public String getColumnText(Object element, int columnIndex) {
		if (columnIndex >= this.columns.size())
			return "<index out of bounds>";

		EObject eObject = (EObject) element;
		ColumnDescription columnDescription = this.columns.get(columnIndex);
		switch (columnDescription.getType()) {
		case DEFAULT_LABEL: {
			return getTextFor(element);
		}
		case METACLASS_NAME: {
			EClass eClass = eObject.eClass();
			if (eClass != null)
				return eClass.getName();
			else
				return "<none>";
		}
		case ATTRIBUTE: {
			EAttribute attribute = columnDescription.getAttribute();

			if (!eObject.eClass().getEAllAttributes().contains(attribute)) {
				return "N/A";
			}

			Object attrValue = eObject.eGet(attribute);
			if (attrValue == null)
				return "null";
			else
				return Util.truncateBeforeNewline(attrValue.toString());
		}
		case REFERENCE: {
			EReference reference = columnDescription.getReference();

			if (!eObject.eClass().getEAllReferences().contains(reference)) {
				return "N/A";
			}

			Object refValue = eObject.eGet(reference);

			if (refValue == null)
				return "null";
			else if (reference.isMany()) {
				@SuppressWarnings("unchecked")
				EList<EObject> list = (EList<EObject>) refValue;

				// when only 1 element, then display it directly
				if (list.size() == 1) {
					return getTextFor(list.get(0));
				}

				return "(" + list.size() + ")";
			} else {
				return getTextFor(refValue);
			}
		}
		case ECONTAINER: {
			return getTextFor(((EObject) element).eContainer());
		}
		default:
			return "<unexpected column type>";
		}
	}
}
